/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/tk/control/checkgroup', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/forms',
    'io.ox/office/tk/control/buttongroup',
    'gettext!io.ox/office/tk/main'
], function (Utils, Forms, ButtonGroup, gt) {

    'use strict';

    var // label of the leading check box to toggle all check buttons
        SELECT_ALL_LABEL =
            //#. Label of a special check box on top of an entire check list, that checks/unchecks all entries in the list
            gt.pgettext('check-list', 'Select all');

    // class CheckGroup =======================================================

    /**
     * Creates a control that contains a set of option buttons, and allows to
     * select multiple buttons at the same time. The value of this control
     * group is an array containing the values of all selected option buttons.
     *
     * @constructor
     *
     * @extends ButtonGroup
     *
     * @param {Object} [initOptions]
     *  Optional parameters. Supports all options of the base class
     *  ButtonGroup. Additionally, the following options are supported:
     *  @param {Boolean} [initOptions.boxed=false]
     *      If set to true, the icons to visualize the unchecked and checked
     *      state will contain rectangular boxes. Otherwise, unchecked state is
     *      simply empty space, and checked state is a simple check mark.
     *  @param {Boolean} [initOptions.checkAll=false]
     *      If set to true, an additional special check box with the label
     *      'Select all' will be shown on top of all other check boxes.
     *      Clicking this check box will switch the state of all other check
     *      boxes to the new state of the 'Select all' check box.
     */
    function CheckGroup(initOptions) {

        var // self reference
            self = this,

            // comparator to select option buttons
            matcher = Utils.getFunctionOption(initOptions, 'matcher', _.isEqual),

            // icon set to be used for the check icons
            design = Utils.getBooleanOption(initOptions, 'boxed', false) ? 'boxed' : 'check',

            // the 'Select all' check box
            selectAllNode = null;

        // base constructor ---------------------------------------------------

        ButtonGroup.call(this, Utils.extendOptions(initOptions, { matcher: arrayMatcher }));

        // private methods ----------------------------------------------------

        /**
         * A matcher predicate that compares elements of a value array and the
         * values of the option buttons in this control, using the value
         * matcher passed to the constructor.
         *
         * @param {Any} valueArray
         *  Any value. If this parameter is an array, its elements will be
         *  matched against the passed button value. Otherwise, this method
         *  returns null.
         *
         * @param {Any} buttonValue
         *  Any button value to be matched against the array elements.
         *
         * @returns {Boolean}
         *  True, if the parameter 'valueArray' is an array, and the matcher
         *  passed to the constructor returns true for at least one of its
         *  elements and the button value. Otherwise, this method returns false
         *  (no array element matches, or the value passed to the 'valueArray'
         *  parameter is not an array).
         */
        function arrayMatcher(valueArray, buttonValue) {
            return _.isArray(valueArray) && _.any(valueArray, function (value) {
                return matcher(value, buttonValue) === true;
            });
        }

        /**
         * Returns the values of all passed button nodes as plain array.
         *
         * @param {jQuery} buttonNodes
         *  The button nodes whose values will be returned.
         *
         * @returns {Array}
         *  The values of all passed button nodes.
         */
        function getButtonValues(buttonNodes) {
            return _.map(buttonNodes.get(), Forms.getButtonValue);
        }

        /**
         * Checks all option buttons in this group that match one of the
         * elements in the passed array.
         *
         * @param {Array|Null} valueArray
         *  The values associated to all check boxes to be checked. If null,
         *  all check boxes will be unchecked.
         */
        function updateHandler(valueArray) {

            var // all existing option buttons
                optionButtons = self.getOptionButtons();

            // check all buttons matching the array elements
            if (_.isArray(valueArray)) {
                Forms.checkMatchingButtonNodes(optionButtons, valueArray, { matcher: arrayMatcher, design: design });
            } else {
                // uncheck all buttons for null value (TOOD: support ambiguous state?)
                Forms.checkButtonNodes(optionButtons, false, { design: design });
            }

            // update state of the 'Select all' check box
            if (selectAllNode) {
                var checkedCount = Forms.filterCheckedButtonNodes(optionButtons).length,
                    state = (checkedCount === 0) ? false : (checkedCount === optionButtons.length) ? true : null;
                Forms.checkButtonNodes(selectAllNode, state, { design: design, ambiguous: false });
            }
        }

        /**
         * Handles click events of regular check boxes in this group, and
         * toggles the clicked check box.
         */
        function buttonClickHandler(event) {

            var // current value of this control
                valueArray = self.getValue(),
                // the value of the clicked button
                buttonValue = Forms.getButtonValue(this),
                // all button elements for the new value
                optionButtons = null;

            // if the current value is not an array (ambiguous state), the new
            // value is the clicked button only; otherwise the value of the button
            // will be toggled in the current value array
            if (!_.isArray(valueArray)) {
                valueArray = [buttonValue];
            } else {
                optionButtons = Forms.filterCheckedButtonNodes(self.getOptionButtons());
                if (Forms.isCheckedButtonNode(event.currentTarget)) {
                    optionButtons = optionButtons.not(event.currentTarget);
                } else {
                    optionButtons = optionButtons.add(event.currentTarget);
                }
                valueArray = getButtonValues(optionButtons);
            }

            // commit the new value of this control group
            self.triggerChange(valueArray, { sourceEvent: event });
        }

        /**
         * Handles click events of the 'Select all' check box in this group,
         * and toggles all check boxes.
         */
        function selectAllClickHandler(event) {

            var // the new state for all check boxes
                state = !Forms.isCheckedButtonNode(selectAllNode),
                // the new value for this group
                valueArray = state ? getButtonValues(self.getOptionButtons()) : [];

            // commit the new value of this control group
            self.triggerChange(valueArray, { sourceEvent: event });
        }

        // initialization -----------------------------------------------------

        // add special marker classes used to adjust formatting
        this.getNode().addClass('check-group');

        // update handler for selected state
        this.registerUpdateHandler(updateHandler);

        // option button click handler (convert ENTER and SPACE keys to click events)
        Forms.setButtonKeyHandler(this.getNode());

        Forms.touchAwareListener({ node: this.getNode(), delegate: Forms.OPTION_BUTTON_SELECTOR }, buttonClickHandler);

        // add a 'Select all' check box on top
        if (Utils.getBooleanOption(initOptions, 'checkAll', false)) {

            // create and insert the 'Select all' check box
            selectAllNode = $(Forms.createButtonMarkup({ label: SELECT_ALL_LABEL })).addClass('select-all');
            Forms.checkButtonNodes(selectAllNode, false, { design: design });
            this.addChildNodes(selectAllNode);

            // handle clicks on the check box separately
            selectAllNode.on('click', selectAllClickHandler); //  TODO TOUCH??
        }

        // destroy all class members on destruction
        this.registerDestructor(function () {
            initOptions = self = matcher = selectAllNode = null;
        });

    } // class CheckGroup

    // exports ================================================================

    // derive this class from class ButtonGroup
    return ButtonGroup.extend({ constructor: CheckGroup });

});
