/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define('io.ox/office/textframework/model/hardbreakmixin', [
    'io.ox/office/drawinglayer/view/drawingframe',
    'io.ox/office/editframework/utils/attributeutils',
    'io.ox/office/textframework/utils/dom',
    'io.ox/office/textframework/utils/operations',
    'io.ox/office/textframework/utils/position'
], function (DrawingFrame, AttributeUtils, DOM, Operations, Position) {

    'use strict';

    // mix-in class HardBreakMixin ======================================

    /**
     * A mix-in class for the hard break handling in the document model.
     * And additionally for the manual page break handling.
     *
     * @constructor
     */
    function HardBreakMixin(app) {

        var // self reference for local functions
            self = this;

        // private methods ----------------------------------------------------

        /**
         * Inserts a hard break component into the document DOM.
         *
         * @param {Number[]} start
         *  The logical start position for the new tabulator.
         *
         * @param {Object} [attrs]
         *  Attributes to be applied at the new hard-break component, as map of
         *  attribute maps (name/value pairs), keyed by attribute family.
         *
         * @param {Boolean} [external]
         *  Will be set to true, if the invocation of this method originates
         *  from an external operation.
         *
         * @returns {Boolean}
         *  Whether the tabulator has been inserted successfully.
         */
        function implInsertHardBreak(start, type, attrs, target/*, external*/) {

            var // text span that will precede the hard-break node
                span = self.prepareTextSpanForInsertion(start, {}, target),
                // new text span for the hard-break node
                hardbreakSpan = null,
                // element from which we calculate pagebreaks
                currentElement = span ? span.parentNode : '',
                // the hard break type
                hardBreakType = type || 'textWrapping';  // defaulting to 'textWrapping'

            if (!span) { return false; }

            // expanding attrs to disable change tracking, if not explicitely set
            attrs = self.checkChangesMode(attrs);

            // split the text span to get initial character formatting for the hard-break
            hardbreakSpan = DOM.splitTextSpan(span, 0);

            // insert a hard-break container node before the addressed text node, move
            // the hard-break span element into the hard-break container node
            // Fix for 29265: Replacing '.empty()' of hardbreakSpan with '.contents().remove().end()'
            // to get rid of empty text node in span in IE.

            // TODO: implementing hard break type 'column'
            // -> fallback to text node with one space, so that counting is still valid.
            if ((hardBreakType === 'page') || (hardBreakType === 'column')) {
                hardbreakSpan.contents().remove().end().text('\xa0');  // non breakable space
                hardbreakSpan.addClass('ms-hardbreak-page');
                $(currentElement).addClass('manual-page-break');
            } else {
                hardbreakSpan.contents().remove().end().append($('<br>'));
            }

            DOM.createHardBreakNode(hardBreakType).append(hardbreakSpan).insertAfter(span);

            // apply the passed tab attributes
            if (_.isObject(attrs)) {
                self.getCharacterStyles().setElementAttributes(hardbreakSpan, attrs);
            }

            // validate paragraph, store new cursor position
            self.implParagraphChanged(span.parentNode);
            self.setLastOperationEnd(Position.increaseLastIndex(start));

            // don't call explicitly page break rendering, if target for header/footer comes with operation
            if (target) {
                self.setBlockOnInsertPageBreaks(true);
            }

            if ($(currentElement).data('lineBreaksData')) {
                $(currentElement).removeData('lineBreaksData'); //we changed paragraph layout, cached line breaks data needs to be invalidated
            }

            // call for debounced render of pagebreaks
            self.insertPageBreaks(currentElement, DrawingFrame.getClosestTextFrameDrawingNode(span.parentNode));
            if (!self.isPageBreakMode()) { app.getView().recalculateDocumentMargin(); }

            // immediately updating element formatting to underline the tab correctly
            // -> This is not necessary with localStorage and fastLoad during loading
            if (self.isImportFinished() && self.getChangeTrack().isActiveChangeTracking()) {
                self.getParagraphStyles().updateElementFormatting(span.parentNode);
            }

            return true;
        }

        // public methods -----------------------------------------------------

        /**
         * Inserting a hard line break, that leads to a new line inside a paragraph. This
         * function can be called from the side pane with a button or with 'Shift+Enter' from
         * within the document.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved if the dialog has been closed with
         *  the default action; or rejected, if the dialog has been canceled.
         *  If no dialog is shown, the promise is resolved immediately.
         */
        this.insertHardBreak = function () {

            var // the undo manager object
                undoManager = self.getUndoManager();

            // the undo manager returns the return value of the callback function
            return undoManager.enterUndoGroup(function () {

                var // the selection object
                    selection = self.getSelection();

                function doInsertHardBreak() {
                    var start = selection.getStartPosition(),
                        operation = { name: Operations.HARDBREAK_INSERT, start: start },
                        // target for operation - if exists, it's for ex. header or footer
                        target = self.getActiveTarget();

                    self.extendPropertiesWithTarget(operation, target);
                    self.doCheckImplicitParagraph(start);
                    // modifying the attributes, if changeTracking is activated
                    if (self.getChangeTrack().isActiveChangeTracking()) {
                        operation.attrs = operation.attrs || {};
                        operation.attrs.changes = { inserted: self.getChangeTrack().getChangeTrackInfo(), removed: null };
                    }
                    self.applyOperations(operation);
                }

                if (selection.hasRange()) {
                    return self.deleteSelected()
                    .done(function () {
                        doInsertHardBreak();
                        selection.setTextSelection(self.getLastOperationEnd()); // finally setting the cursor position
                    });
                }

                doInsertHardBreak();
                selection.setTextSelection(self.getLastOperationEnd()); // finally setting the cursor position
                return $.when();

            }, this);

        };

        /**
         * Inserting a maunal page break, that leads to a new page inside a document. This
         * function can be called from the toolbar with a button or with 'CTRL+Enter' from
         * within the document.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved if the dialog has been closed with
         *  the default action; or rejected, if the dialog has been canceled.
         *  If no dialog is shown, the promise is resolved immediately.
         */
        this.insertManualPageBreak = function () {

            var // the undo manager object
                undoManager = self.getUndoManager();

            if (self.isHeaderFooterEditState()) { return; }

            // the undo manager returns the return value of the callback function
            return undoManager.enterUndoGroup(function () {

                var // the selection object
                    selection = self.getSelection(),
                    // start position of selection
                    start = selection.getStartPosition(),
                    // copy of start position, prepared for modifying
                    positionOfCursor = _.clone(start),
                    // last index in positionOfCursor array, used for incrementing cursor position
                    lastIndex = positionOfCursor.length - 1,
                    // the page node
                    editdiv = self.getNode(),
                    // first level element, paragraph or table
                    firstLevelNode = Position.getContentNodeElement(editdiv, start.slice(0, 1)),
                    // object passed to generateOperation method
                    operationOptions,
                    // operations generator
                    generator = this.createOperationsGenerator();

                function doInsertManualPageBreak() {

                    var // the attributes for the paragraph between the tables
                        paraAttrs = {},
                        // the change track object
                        changeTrack = self.getChangeTrack();

                    // refreshing start position, that might be modified by deleteSelected()
                    start = selection.getStartPosition();

                    // different behavior for inserting page break inside paragraphs and tables
                    if (DOM.isParagraphNode(firstLevelNode)) {

                        // first split paragraph
                        generator.generateOperation(Operations.PARA_SPLIT, { start: start });
                        self.doCheckImplicitParagraph(start);
                        //update cursor position value after paragraph split
                        positionOfCursor[lastIndex - 1] += 1;
                        positionOfCursor[lastIndex] = 0;
                        // set paragraph attribute page break
                        operationOptions = { start: positionOfCursor.slice(0, -1), attrs: { paragraph: { pageBreakBefore: true } } };

                        if (changeTrack.isActiveChangeTracking()) {
                            operationOptions.attrs.changes = { inserted: changeTrack.getChangeTrackInfo(), removed: null };
                        }
                        generator.generateOperation(Operations.SET_ATTRIBUTES, operationOptions);

                    } else if (DOM.isTableNode(firstLevelNode)) {
                        var firstParagraphPosInFirstCell,
                            firstParagraphPosHelper = positionOfCursor.slice(0, 3);

                        // we always apply pageBreakBefore attribute to the first paragraph in table - because of compatibility with MS Word
                        firstParagraphPosHelper[2] = 0;
                        firstParagraphPosInFirstCell = Position.getFirstPositionInCurrentCell(editdiv, firstParagraphPosHelper);
                        // before applying paragraph attribute, we need to check if it's implicit paragraph, #34624
                        // #35328 - table is not yet split, and we dont check impl paragraph for cursor position, but for first par in row, where we apply pageBreakBefore attr
                        self.doCheckImplicitParagraph(firstParagraphPosInFirstCell);

                        // different behavior for odt and msword page breaks in table
                        if (app.isODF()) {
                            if (start[0] === 0 || (DOM.isManualPageBreakNode(firstLevelNode))) {
                                // if table is at first position in document, or there is already page break applied to table, ODF ignores operation
                                return;
                            }

                            // #35446 - odf applies pageBreakBefore attr always on first cell in table, and doesn't split table
                            firstParagraphPosInFirstCell = Position.getFirstPositionInParagraph(editdiv, positionOfCursor.slice(0, 1));
                            self.doCheckImplicitParagraph(firstParagraphPosInFirstCell);
                            // odf specific, set attr to table, too, #35585
                            operationOptions = { start: firstParagraphPosInFirstCell.slice(0, -1), attrs: { paragraph: { pageBreakBefore: true } } };
                            generator.generateOperation(Operations.SET_ATTRIBUTES, operationOptions);

                        } else {
                            // split table with page break, but only if it's not first row. Otherwise, just insert new paragraph before table
                            if (positionOfCursor[1] !== 0) {
                                generator.generateOperation(Operations.TABLE_SPLIT, { start: positionOfCursor.slice(0, 2) }); // passing row position of table closest to pagecontent node (top level table)

                                firstParagraphPosInFirstCell[0] += 1;
                                firstParagraphPosInFirstCell[1] = 0;

                                operationOptions = { start: firstParagraphPosInFirstCell.slice(0, -1), attrs: { paragraph: { pageBreakBefore: true } } };
                                generator.generateOperation(Operations.SET_ATTRIBUTES, operationOptions);

                                // Insert empty paragraph in between split tables.
                                // Adding change track information only to the inserted paragraph and only, if
                                // change tracking is active.
                                // -> the splitted table cannot be marked as inserted. Rejecting this change track
                                // requires automatic merge of the splitted table, if possible.
                                if (changeTrack.isActiveChangeTracking()) {
                                    paraAttrs.changes = { inserted: changeTrack.getChangeTrackInfo(), removed: null };
                                }

                                // update position of cursor
                                positionOfCursor[0] += 1;

                                // #36130 - avoid error in console: Selection.restoreBrowserSelection(): missing text selection range
                                selection.setTextSelection(Position.getFirstPositionInParagraph(editdiv, start.slice(0, 1)));

                                // insert empty paragraph in between split tables
                                generator.generateOperation(Operations.PARA_INSERT, { start: positionOfCursor.slice(0, 1), attrs: paraAttrs });

                                // update position of cursor
                                positionOfCursor[0] += 1;
                                positionOfCursor[1] = 0;

                            } else {
                                operationOptions = { start: firstParagraphPosInFirstCell.slice(0, -1), attrs: { paragraph: { pageBreakBefore: true } } };
                                generator.generateOperation(Operations.SET_ATTRIBUTES, operationOptions);

                                // if we apply page break to element that already has that property, set to paragraph that is inserted before it
                                if (DOM.isManualPageBreakNode(firstLevelNode)) {
                                    paraAttrs.paragraph = { pageBreakBefore: true };
                                }

                                // Insert empty paragraph in between split tables.
                                // Adding change track information only to the inserted paragraph and only, if
                                // change tracking is active.
                                // -> the splitted table cannot be marked as inserted. Rejecting this change track
                                // requires automatic merge of the splitted table, if possible.
                                if (changeTrack.isActiveChangeTracking()) {
                                    paraAttrs.changes = { inserted: changeTrack.getChangeTrackInfo(), removed: null };
                                }

                                // insert empty paragraph in between split tables
                                generator.generateOperation(Operations.PARA_INSERT, { start: positionOfCursor.slice(0, 1), attrs: paraAttrs });

                                // update position of cursor
                                positionOfCursor[0] += 1;

                                // #36130 - avoid error in console: Selection.restoreBrowserSelection(): missing text selection range
                                selection.setTextSelection(Position.getFirstPositionInParagraph(editdiv, positionOfCursor.slice(0, 1)));
                            }
                        }
                    }

                    self.applyOperations(generator);
                }

                if (selection.hasRange() && !DOM.isTableNode(firstLevelNode)) { // #35346 don't delete if table is selected
                    return self.deleteSelected()
                    .done(function () {
                        doInsertManualPageBreak();
                        selection.setTextSelection(positionOfCursor); // finally setting the cursor position
                    });
                }

                doInsertManualPageBreak();
                selection.setTextSelection(positionOfCursor); // finally setting the cursor position
                return $.when();

            }, this);

        };

        // operation handler --------------------------------------------------

        /**
         * The handler for the insertHardBreak operation.
         *
         * @param {Object} operation
         *  The operation object.
         *
         * @param {Boolean} external
         *  Whether this is an external operation.
         *
         * @returns {Boolean}
         *  Whether the hard break was inserted successfully.
         */
        this.insertHardBreakHandler = function (operation, external) {

            var // the undo manager
                undoManager = self.getUndoManager(),
                // the undo operation
                undoOperation = null;

            if (!implInsertHardBreak(operation.start, operation.type, operation.attrs, operation.target, external)) {
                return false;
            }

            if (undoManager.isUndoEnabled()) {
                undoOperation = { name: Operations.DELETE, start: operation.start };
                self.extendPropertiesWithTarget(undoOperation, operation.target);
                undoManager.addUndo(undoOperation, operation);
            }

            return true;
        };

        // initialization -----------------------------------------------------

        // destroy all class members on destruction
        this.registerDestructor(function () {
            self = null;
        });

    } // class HardBreakMixin

    // exports ================================================================

    return HardBreakMixin;

});
