/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 * @author Malte Timmermann <malte.timmermann@open-xchange.com>
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define('io.ox/office/text/app/application', [

    'io.ox/mail/compose/inline-images',

    'io.ox/office/tk/utils',

    'io.ox/office/baseframework/view/baselabels',
    'io.ox/office/baseframework/utils/errorcode',

    'io.ox/office/editframework/app/editapplication',
    'io.ox/office/editframework/utils/attributeutils',

    'io.ox/office/textframework/utils/dom',
    'io.ox/office/textframework/utils/export',
    'io.ox/office/textframework/utils/config',

    'io.ox/office/text/model/model',
    'io.ox/office/text/view/view',
    'io.ox/office/text/app/controller',

    'gettext!io.ox/office/text/main'

], function (Inline, Utils, BaseLabels, ErrorCode, EditApplication, AttributeUtils, DOM, Export, TextConfig, TextModel, TextView, TextController, gt) {

    'use strict';

    var
        global = window,
      //global = (Function('return this;')()), // implicit eval but sole bullet proof way of accessing `global` within strict mode.

        Blob   = global.Blob, // due to linter that does not acknowledge ...
        atob   = global.atob; // ... these objects as global ones.

    // private static
    function recursivelyReduceTextFragmentFromJQuerySiblings(node, collector) {
        var text = Utils.trimString([

                collector.text,
                Utils.trimAndCleanString(node.text()).split(/\s+/).join(' ')        // - due to firefox that, with `innerText` ...
              //Utils.trimAndCleanString(node[0].innerText).split(/\s+/).join(' ')  //   ... delivers [undefined] for empty text nodes.

            ].join(' ')),

            diff = (collector.maxLength - text.length);

        if (diff <= 0) {
            text = text.substring(0, (text.length + diff));

            if (diff < 0) {
                text = text.replace((/\s+[^\s]*$/), '');
            }
        } else if (node.next()[0]) {
            collector.text = text;

            text = recursivelyReduceTextFragmentFromJQuerySiblings(node.next(), collector);
        }
        return text;
    }

    function reduceTextFragmentFromDOM(node, maxLength) {
        return recursivelyReduceTextFragmentFromJQuerySiblings(node.children().eq(0), {
            text: '',
            maxLength: maxLength
        });
    }

    function predictivelyGetTitleTextFromDOM(node) {
        var getElementStyleId     = AttributeUtils.getElementStyleId,
            styleIdRegXFilterList = [

                (/^(?:Title|Heading[0-6]*)$/i),
                (/^(?:ResumeStyle|ResumeDescription)[0-6]*$/i),
                (/^Normal[0-6]*$/i)
            ],
            createStyleIdFilter = function (regX) {
                return function (node) {
                    return regX.test(getElementStyleId(node));
                };
            },
            regX,

            firstLevelNodeList = node.children().toArray().filter(function (node/*, idx list*/) {
                return node.classList.contains('p');
            }),
            nodeList = [],

            text = '';

        while ((regX = styleIdRegXFilterList.shift()) && !nodeList.length) {

            nodeList = firstLevelNodeList.filter(createStyleIdFilter(regX));
          //console.log('predictivelyGetTitleTextFromDOM :: nodeList.length : ', nodeList.length);
        }

        ((nodeList.length && nodeList) || firstLevelNodeList).some(function (node/*, idx, list*/) {
            text = Utils.trimString([

                text,
                Utils.trimAndCleanString($(node).text()).split(/\s+/).join(' ') // - due to firefox that, with `innerText` ...
              //Utils.trimAndCleanString(node.innerText).split(/\s+/).join(' ') //   ... delivers [undefined] for empty text nodes.

            ].join(' '));

            return (text !== '');       // targets the first non empty paragraph.
          //return (text.length >= 80); // targets all paragraphs till maxLength condition is fulfilled.
        });

        if (text.length > 80) {
            text = text.substring(0, 81).replace((/\s+[^\s]*$/), '');
        }
      //console.log('predictivelyGetTitleTextFromDOM :: before leeave - text : ', text);

        return text;
    }

    function extractTitleTextFromDOM(node) {
        var contentNode = DOM.getPageContentNode(node).eq(0),

            title = (
                predictivelyGetTitleTextFromDOM(contentNode) ||
                reduceTextFragmentFromDOM(contentNode, 80)
            );

        return title;
    }

    // class TextApplication ==================================================

    /**
     * The OX Text application.
     *
     * @constructor
     *
     * @extends EditApplication
     *
     * @param {Object} launchOptions
     *  All options passed to the core launcher (the ox.launch() method).
     */
    var TextApplication = EditApplication.extend({ constructor: function (launchOptions) {

        var // self reference
            self = this;

        // base constructor ---------------------------------------------------

        EditApplication.call(this, TextModel, TextView, TextController, launchOptions, {
            applyActionsDetached: true,
            postProcessHandler: postProcessDocument,
            postProcessHandlerStorage: postProcessHandlerStorage,
            fastEmptyLoadHandler: fastEmptyLoadHandler,
            optimizeOperationsHandler: optimizeOperations,
            mergeCachedOperationsHandler: mergeCachedOperations,
            prepareLoseEditRightsHandler: prepareLoseEditRights,
            operationFilter: operationFilter,
            importFailedHandler: importFailedHandler,
            prepareFlushHandler: prepareFlushDocument,
            sendActionsDelay: 1000,
            localStorageApp: true,
            requiredStorageVersion: 2,
            supportedStorageVersion: 2,
            supportsOnlineSync: true
        });

        // private methods ----------------------------------------------------

        /**
         * Post-processes the document contents and formatting, after all its
         * import operations have been applied.
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved when the
         *  document has been post-processed successfully, or rejected when an
         *  error has occurred.
         */
        function postProcessDocument() {
            return self.getModel().updateDocumentFormatting();
        }

        /**
         * Load performance: Post-processes the document contents and formatting,
         * after the document was loaded from local storage.
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved when the
         *  document has been post-processed successfully, or rejected when an
         *  error has occurred.
         */
        function postProcessHandlerStorage() {
            return self.getModel().updateDocumentFormattingStorage();
        }

        /**
         * Handler will be called by base class if importing the document
         * failed. The functions handles Text specific errors.
         */
        function importFailedHandler(response) {

            var // specific error code sent by the server
                error = new ErrorCode(response);

            switch (error.getCodeAsConstant()) {
                case 'LOADDOCUMENT_COMPLEXITY_TOO_HIGH_ERROR':
                    response.message = gt('This document exceeds the complexity limits.');
                    break;
            }
        }

        /**
         * Preprocessing before the document will be flushed for downloading,
         * printing, sending as mail, or closing.
         *
         * @param {String} reason
         *  Can be 'download', 'email', or 'quit'
         */
        function prepareFlushDocument(reason) {
            if (reason === 'download' || reason === 'email') {
                self.getModel().getFieldManager().prepareDateTimeFieldsDownload();
            }
        }

        /**
         * Optimizing the actions (operations) before sending them to the server.
         *
         * @param {Object[]} actionsBuffer
         *  An array with actions.
         *
         * @returns {Object[]}
         *  The optimized array with actions.
         */
        function optimizeOperations(actionsBuffer) {
            return self.getModel().optimizeActions(actionsBuffer);
        }

        /**
         * Merge cached operations buffer, with current action buffer, before registering.
         *
         * @param {Array} actionBuffer
         *  Current array of operations.
         *
         * @param {Array} cacheBuffer
         *  Cached array of operations, e.g operations for complex fields update
         *  on document open are cached until document is modified.
         *
         * @returns {Array}
         *
         */
        function mergeCachedOperations(actionBuffer, cacheBuffer) {
            return cacheBuffer.concat(actionBuffer);
        }

        /**
         * Trigger flushing of buffered operations on losing edit rights.
         */
        function prepareLoseEditRights() {
            self.getModel().trigger('cacheBuffer:flush');
        }

        /**
         * Load performance: Only selected operations need to be executed, if document
         * can be loaded from local storage.
         *
         * @param {Object[]} operation
         *  An operation object.
         *
         * @returns {Boolean}
         *  Whether the specified operation needs to be executed, although the document
         *  is loaded from the local storage.
         */
        function operationFilter(operation) {
            return self.getModel().operationFilter(operation);
        }

        /**
         * Handler for fast load of empty documents
         *
         * @param {String} markup
         *  The HTML mark-up to be shown as initial document contents.
         *
         * @param {Array} actions
         *  The operation actions to be applied to finalize the fast import.
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved when the
         *  actions have been applied.
         */
        function fastEmptyLoadHandler(markup, actions) {
            return self.getModel().fastEmptyLoadHandler(markup, actions);
        }

        /**
         * Deferred helper that sanitizes base64 encoded image urls
         * within the passed chunk of stringified HTML markup.
         *
         * Any matching image gets converted into a file and uploaded
         * via a special ox service. After uploading a physical URL gets
         * retrieved that replaces the former base64 encoded image source.
         *
         * @param htmlFragmentString
         *  Any chunk of stringified HTML markup.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved if the file upload could be
         *  managed successfully, or rejected otherwise.
         *  If resolved without failure it returns the promised
         *  and sanitized HTML fragment in its stringified form.
         */
        function sanitizeBase64ImageUrlsDeferred(htmlFragmentString) {
            var
                InlineApi     = Inline.api,

                $htmlFragment = $(['<div>', htmlFragmentString, '</div>'].join('')),
                imageList     = $htmlFragment.find('img'),

                promise = self.iterateArraySliced(imageList, function (img/*, idx, list*/) {
                    var src = img.src;

                  //if ((idx + 2) >= list.length) {     // forced rejection for development and testing purposes.
                  //    return $.Deferred().reject();
                  //}
                    if (/^data:/.test(src)) {
                        var byteString = atob(src.split(',')[1]),
                            mimeString = src.split(',')[0].split(':')[1].split(';')[0],
                            buffer = new ArrayBuffer(byteString.length),
                            intArray = new Uint8Array(buffer);

                        for (var i = 0; i < byteString.length; i++) {
                            intArray[i] = byteString.charCodeAt(i);
                        }
                        return InlineApi.inlineImage({ file: new Blob([intArray], { type: mimeString }) }).done(function success(response) {

                            img.src = InlineApi.getInsertedImageUrl(response);
                        });
                    }
                });

            return promise.then(function () {

                return $htmlFragment.html();

            }, function () {

                return { text: gt('Document images could not be processed. Send as mail is not possible.') };
            });
        }

        /**
         * Does select all of an current document's content,
         * converts that via clipboard functionality into a
         * stringified HTML markup and returns the latter.
         *
         * @attention
         *  This locally declared method will be registered as
         *  a converter to its base application. Nevertheless
         *  it always does work within the current document's
         *  `this` context.
         *
         * @returns {jQuery.Promise}
         *  If resolved without failure it returns the promised
         *  representative of the current model that features
         *  the stringified HTML markup equivalent of all of
         *  a document's `content` together with a `title`.
         */
        function createInlineHtmlDocument() {
            var htmlFragmentString,

                editor    = self.getModel(),
                selection = editor.getSelection(),

                rootNode  = selection.getRootNode(),
                titleText = extractTitleTextFromDOM(rootNode);

            selection.selectAll();
            htmlFragmentString = Export.getHTMLFromSelection(editor/*, { isEncodeBase64: false }*/);
            selection.resetSelection();

            return sanitizeBase64ImageUrlsDeferred(htmlFragmentString).then(function (htmlFragmentString) {
                return {
                    title:   titleText,
                    content: htmlFragmentString
                };
            });
        }

        // public methods -----------------------------------------------------

        /**
         * Text specific guard method that decides whether an application is able
         * of sending its document as mail - in this case as inline-html - or not.
         *
         * @returns {Boolean}
         * Returns `true` if the document's overall page count is in between 1 and
         * a maximum value that is maintained as office text property and provided
         * by `TextConfig.SEND_AS_HTML_MAX_PAGE_COUNT`. Otherwise returns `false`.
         */
        this.canSendMail = function () {
            var pageCount = self.getModel().getPageLayout().getNumberOfDocumentPages();

            return (
                (pageCount >= 1) &&
                (pageCount <= TextConfig.SEND_AS_HTML_MAX_PAGE_COUNT)
            );
        };

        // application runtime ------------------------------------------------

        // destroy all class members
        this.registerDestructor(function () {
            launchOptions = self = null;
        });

        // initialization -----------------------------------------------------

        /**
         *  @FIXME
         *
         *  this is the entry point (or hook) for making the new feature -
         *  'send document as inline html mail' - available to the application.
         *
         *  As soon as the next iteration step for this feature take place, please
         *  consider refactoring this feature by moving its entire tool set into
         *  an own model mixin ... maybe something like `sendAsMailMixin` that
         *  should reside in >> /ui/apps/io.ox/office/text/model/ << .
         */
        // register the 'inline-html-document' converter.
        this.registerConverter('inline-html-document', createInlineHtmlDocument);

    } }); // class TextApplication

    // static methods ---------------------------------------------------------

    /**
     * Replacement for the generic method EditApplication.createLauncher()
     * without parameters, to launch text applications.
     */
    TextApplication.createLauncher = function () {
        return EditApplication.createLauncher('io.ox/office/text', TextApplication, { icon: 'fa-file-text', trackingId: 'io.ox/text-editor' });
    };

    // exports ================================================================

    return TextApplication;

});
