/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/render/rendercache', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/object/triggerobject',
    'io.ox/office/tk/object/timermixin',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/utils/paneutils',
    'io.ox/office/spreadsheet/model/cellcollection',
    'io.ox/office/spreadsheet/view/render/renderutils',
    'io.ox/office/spreadsheet/view/render/autostylecache'
], function (Utils, TriggerObject, TimerMixin, SheetUtils, PaneUtils, CellCollection, RenderUtils, AutoStyleCache) {

    'use strict';

    var // convenience shortcuts
        Address = SheetUtils.Address,
        IntervalArray = SheetUtils.IntervalArray,
        RangeArray = SheetUtils.RangeArray,

        // line width for grid colors in the canvas
        GRID_LINE_WIDTH = RenderUtils.GRID_LINE_WIDTH;

    // class RenderCache ======================================================

    /**
     * A cache that stores settings for the visible areas in a specific sheet.
     *
     * Instances of this class trigger the following events:
     *  - 'update:matrix'
     *      After updating the existing cell elements in the rendering matrix
     *      due to cell changes from document operations or from view update
     *      notifications received from the server. Event listeners receive the
     *      following parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {RangeArray} dirtyRanges
     *          The addresses of the changed cell ranges to be rendered.
     *      (4) {Object} [options]
     *          Optional parameters:
     *          - {Boolean} [options.direct=false]
     *              Whether the event listeners should process the event
     *              immediately, i.e. render the changed cells directly.
     *
     * @constructor
     *
     * @extends TriggerObject
     * @extends TimerMixin
     *
     * @param {SpreadsheetView} docView
     *  The document view that contains this cache instance.
     */
    function RenderCache(docView) {

        var // self reference
            self = this,

            // the spreadsheet model and its containers
            docModel = docView.getDocModel(),
            numberFormatter = docModel.getNumberFormatter(),

            // the cache for all auto style descriptors
            styleCache = new AutoStyleCache(docView),

            // all cell elements of the rendering matrix, mapped by cell key
            cellMatrix = {},

            // column headers in the rendering matrix (as sorted array, and mapped by column index)
            colHeader = new RenderUtils.MatrixHeaderCache(true),

            // row headers in the rendering matrix (as sorted array, and mapped by column index)
            rowHeader = new RenderUtils.MatrixHeaderCache(false),

            // the current bounding ranges and rectangles, per pane position (missing entry: hidden)
            boundaryMap = {},

            // the unified bounding ranges covered by this cache
            boundRanges = new RangeArray(),

            // the total range covered by all bounding ranges
            totalRange = null,

            // all merged ranges (instances of MergeDescriptor) covering the bounding ranges, mapped by cell key
            mergedDataMap = {},

            // all pending cell ranges to be updated in the rendering matrix
            dirtyRanges = new RangeArray();

        // base constructors --------------------------------------------------

        TriggerObject.call(this);
        TimerMixin.call(this);

        // private methods ----------------------------------------------------

        /**
         * Inserts the passed merged range into the internal caches.
         */
        function insertMergedRange(mergedRange) {

            var // shrink to visible columns/rows in the sheet
                shrunkenRange = docView.getSheetModel().shrinkRangeToVisible(mergedRange);

            // skip invisible merged ranges (e.g. all columns hidden)
            if (!shrunkenRange) { return; }

            var // the descriptor object of the merged range
                mergedData = new RenderUtils.MergeDescriptor(mergedRange, shrunkenRange);

            // store the new descriptor of the merged range in the map
            mergedDataMap[mergedData.originKey] = mergedData;
            mergedData.element = cellMatrix[mergedData.shrunkenKey];
        }

        /**
         * Updates the cached merged range settings. Inserts new merged ranges
         * not yet cached, and removes old merged ranges not visible anymore.
         */
        var updateMergedRanges = RenderUtils.profileMethod('RenderCache.updateMergedRanges()', function () {

            var // addresses of entire visible rows (needed to support overlapping text)
                rowRanges = RangeArray.map(rowHeader.intervals, docModel.makeRowRange, docModel),
                // all merged ranges inside the bounding row intervals
                mergedRanges = docView.getMergeCollection().getMergedRanges(rowRanges),
                // descriptors of all known merged ranges
                oldMergedDataMap = _.clone(mergedDataMap);

            // create new descriptors, remove descriptors of existing merged ranges from
            // 'oldMergedDataMap' (remaining descriptors will be deleted in the caches)
            mergedRanges.forEach(function (mergedRange) {

                var // the cell key of the range origin, used as map key in the cache maps
                    originKey = mergedRange.start.key();

                if (originKey in mergedDataMap) {
                    delete oldMergedDataMap[originKey];
                } else {
                    insertMergedRange(mergedRange);
                }
            });

            // delete remaining descriptors from other caches
            _.each(oldMergedDataMap, function (mergedData, originKey) {
                delete mergedDataMap[originKey];
            });

            // delete all sorted arrays of merged ranges per row (will be updated on demand)
            rowHeader.resetMergedData();
        });

        /**
         * Returns the descriptors of all merged ranges covering the specified
         * row in the rendering matrix.
         *
         * @param {Number} row
         *  The zero-based row index.
         *
         * @returns {Array<MergeDescriptor>}
         *  An array with descriptors of all merged ranges covering the
         *  specified row.
         */
        function getMergedDataInRow(row) {

            var // the matrix header element of the passed row
                headerElement = rowHeader.map[row];

            // lazy update of the cached map of merged ranges by row
            if (!headerElement.mergedData) {

                // find all merged ranges covering the specified row
                headerElement.mergedData = _.filter(mergedDataMap, function (mergedData) {
                    return mergedData.mergedRange.containsRow(row);
                });

                // sort ranges by start column for binary search
                headerElement.mergedData = _.sortBy(headerElement.mergedData, function (mergedData) {
                    return mergedData.mergedRange.start[0];
                });
            }

            return headerElement.mergedData;
        }

        /**
         * Returns a descriptor for the merged range covering the cell with the
         * specified address.
         *
         * @param {Address} address
         *  The address of a cell.
         *
         * @returns {MergeDescriptor|Null}
         *  The descriptor of a merged range covering the specified cell, if
         *  available; otherwise null.
         */
        function findMergedData(address) {

            var // the merged ranges covering the row contained in the address
                mergedDataInRow = getMergedDataInRow(address[1]),
                // find the last range starting before or at the column index
                mergedData = Utils.findLast(mergedDataInRow, function (data) {
                    return data.mergedRange.start[0] <= address[0];
                }, { sorted: true });

            // return the merged range, if it really covers the passed address
            return (mergedData && (address[0] <= mergedData.mergedRange.end[0])) ? mergedData : null;
        }

        /**
         * Returns a descriptor for the nearest merged range in the same row,
         * but left of or right of the specified cell address.
         *
         * @param {Address} address
         *  The address of a cell.
         *
         * @param {Boolean} forward
         *  Specifies where to search for a merged range: If set to true, a
         *  merged range will be searched right of the cell, otherwise left of
         *  the cell.
         *
         * @returns {MergeDescriptor|Null}
         *  The descriptor of the nearest merged range next to the specified
         *  cell address, if available; otherwise null.
         */
        function findNextMergedData(address, forward) {

            var // the merged ranges covering the row contained in the address
                mergedDataInRow = getMergedDataInRow(address[1]);

            // binary search in the array of merged ranges according to direction
            return forward ?
                Utils.findFirst(mergedDataInRow, function (data) { return address[0] < data.mergedRange.start[0]; }, { sorted: true }) :
                Utils.findLast(mergedDataInRow, function (data) { return data.mergedRange.end[0] < address[0]; }, { sorted: true });
        }

        /**
         * Updates the matrix header elements according to the current bounding
         * intervals. Removes all header and cell elements that are not visible
         * anymore, and inserts new header and cell elements that have become
         * visible.
         *
         * @returns {RangeArray}
         *  The addresses of all cell ranges that have been added to the
         *  rendering matrix.
         */
        function buildMatrixElements(columns) {

            var // the model of the active sheet
                sheetModel = docView.getSheetModel(),
                // the column/row collection
                collection = columns ? sheetModel.getColCollection() : sheetModel.getRowCollection(),
                // the matrix header to be updated in this pass
                header1 = columns ? colHeader : rowHeader,
                // the matrix header in the opposite direction
                header2 = columns ? rowHeader : colHeader,
                // array index of the next header element not processed yet
                nextIndex = 0,
                // the property name for the next linked cell element
                nextProp = columns ? 'b' : 'r',
                // column/row indexes of all new header elements
                newIndexes = [];

            // unlinks and releases the passed header element, and all attached cell elements
            function releaseHeaderElement(headerElement) {

                var // the current matrix cell element
                    cellElement = headerElement.f;

                // delete all matrix cell elements attached to the header element
                while (cellElement) {
                    var nextElement = cellElement[nextProp];
                    delete cellMatrix[cellElement.key];
                    cellElement.unlink();
                    cellElement = nextElement;
                }

                // unlink and delete the header element
                delete header1.map[headerElement.index];
                headerElement.unlink();
            }

            // returns a cell element from the rendering matrix
            function getCellElement(colElement, rowElement) {
                return (colElement && rowElement) ? cellMatrix[Address.key(colElement.index, rowElement.index)] : null;
            }

            // process all visible columns/rows
            collection.iterateEntries(header1.intervals, function (collEntry, uniqueInterval, origInterval) {

                var // an existing or new header element in header1
                    element1 = null;

                // release all preceding existing header elements that are not visible anymore
                while ((element1 = header1.array[nextIndex]) && (element1.index < collEntry.index)) {
                    header1.array.splice(nextIndex, 1);
                    releaseHeaderElement(element1);
                }

                // insert a new header element if needed
                element1 = header1.array[nextIndex];
                if (!element1 || (collEntry.index < element1.index)) {

                    // create and store the new header element
                    element1 = new RenderUtils.MatrixHeaderElement(collEntry.index, columns, header1.array[nextIndex - 1], element1);
                    header1.map[element1.index] = element1;
                    header1.array.splice(nextIndex, 0, element1);
                    newIndexes.push(element1.index);

                    // create all cell elements in the matrix
                    header2.array.forEach(function (element2) {

                        var // the column and row header element, according to direction
                            colElement = columns ? element1 : element2,
                            rowElement = columns ? element2 : element1,
                            // the address of the new cell
                            address = new Address(colElement.index, rowElement.index),
                            // the adjacent cell elements
                            l = getCellElement(colElement.p, rowElement),
                            r = getCellElement(colElement.n, rowElement),
                            t = getCellElement(colElement, rowElement.p),
                            b = getCellElement(colElement, rowElement.n),
                            // the new matrix cell element
                            cellElement = new RenderUtils.MatrixCellElement(address, colElement, rowElement, l, r, t, b);

                        // insert the new cell element into the rendering matrix
                        cellMatrix[cellElement.key] = cellElement;
                    });
                }
                nextIndex += 1;

                // update contents and formatting of the header element
                element1.update(styleCache, collEntry);

                // update the leading/trailing flags specifying whether the header
                // element is located at the beginning or end of a bounding interval
                element1.leading = element1.index === origInterval.first;
                element1.trailing = element1.index === origInterval.last;
            });

            // release all trailing existing header elements that are not visible anymore
            header1.array.splice(nextIndex).forEach(releaseHeaderElement);

            // create and return the range addresses for all new columns/rows in the matrix
            return IntervalArray.mergeIndexes(newIndexes).map(function (interval) {
                return docModel.makeFullRange(interval, columns);
            });
        }

        /**
         * Visits all cells in the passed ranges, and returns the descriptors
         * of all covered merged ranges starting outside.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.columns=false]
         *      If set to true, the cells in the passed ranges will be visited
         *      in vertical orientation (all cells top-down in first column,
         *      then all cells top-down in second column, etc.). By default,
         *      the cells will be visited in horizontal orientation (all cells
         *      left-to-right in first row, then all cells left-to-right in
         *      second row, etc.).
         *  @param {Boolean} [options.first=false]
         *      If set to true, only the matrix elements of the first visible
         *      row (or the first visible column in vertical mode, see option
         *      'columns') of each range will be visited.
         *
         * @returns {Array<MergeDescriptor>}
         *  The descriptors of all merged ranges covering but not starting in
         *  the passed ranges.
         */
        function iterateCellsInRanges(ranges, callback, options) {

            var // whether to visit the first column/row only
                firstOnly = Utils.getBooleanOption(options, 'first', false),
                // iteration direction (true for columns instead of rows)
                columns = Utils.getBooleanOption(options, 'columns', false),
                // first visible cell of all visited merged ranges (true: top-left cell visited)
                visitedOriginsMap = {};

            function invokeCallback(element, range) {

                // first, invoke the callback function (internal update process adds merged range information)
                callback.call(self, element, range);

                // collect origin cells of all merged ranges covering the visited cells (value false
                // for cells covered by the merged range, value true for top-left visible cells)
                if (element.merged && !visitedOriginsMap[element.merged.originKey]) {
                    visitedOriginsMap[element.merged.originKey] = element.isMergedOrigin();
                }
            }

            // process all ranges separately
            RangeArray.forEach(ranges, function (range) {

                var // the first visible column header element
                    colElement = colHeader.findFirst(range.start[0], range.end[0]),
                    // the first visible row header element
                    rowElement = colElement ? rowHeader.findFirst(range.start[1], range.end[1]) : null,
                    // the first cell element in the current row
                    firstElement = rowElement ? cellMatrix[Address.key(colElement.index, rowElement.index)] : null;

                // do nothing, if the range is not visible at all
                if (!firstElement) { return; }

                var // last column/row index to be visited
                    lastCol = range.end[0],
                    lastRow = range.end[1],
                    // the current cell element in the current row
                    currElement = null;

                // visit all cells in the passed range (performance: implemented separate loops
                // for horizontal and vertical iteration orientation, to prevent fancy indirect
                // property access which takes a little more time)
                if (columns) {

                    // vertical orientation: column-by-column
                    while (firstElement && (firstElement.col.index <= lastCol)) {
                        currElement = firstElement;
                        while (currElement && (currElement.row.index <= lastRow)) {
                            invokeCallback(currElement, range);
                            currElement = currElement.b;
                        }
                        firstElement = firstOnly ? null : firstElement.r;
                    }

                } else {

                    // horizontal orientation: row-by-row
                    while (firstElement && (firstElement.row.index <= lastRow)) {
                        currElement = firstElement;
                        while (currElement && (currElement.col.index <= lastCol)) {
                            invokeCallback(currElement, range);
                            currElement = currElement.r;
                        }
                        firstElement = firstOnly ? null : firstElement.b;
                    }
                }
            });

            // collect and return descriptors of all merged ranges whose top-left cell has NOT been visited
            return _.reduce(visitedOriginsMap, function (outerMergedData, topLeftVisited, originKey) {
                if (!topLeftVisited) { outerMergedData.push(mergedDataMap[originKey]); }
                return outerMergedData;
            }, []);
        }

        /**
         * Registers the specified cell ranges as 'dirty', which will include
         * them in the next update cycle.
         *
         * @param {RangeArray|Range} ranges
         *  An array of cell range addresses, or a single cell range address.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.direct=false]
         *      Whether the elements in the rendering matrix should be updated
         *      immediately.
         */
        var registerDirtyRanges = (function () {

            // direct callback: register passed ranges
            function registerRanges(ranges) {
                RenderUtils.log('RenderCache.registerDirtyRanges(): (1) store dirty ranges');
                dirtyRanges.append(ranges);
            }

            // deferred callback: updates all dirty cell elements in the rendering matrix
            var updateMatrix = RenderUtils.profileMethod('RenderCache.registerDirtyRanges(): (2) update rendering matrix', function (options) {

                var // the model of the active sheet
                    sheetModel = docView.getSheetModel(),
                    // the cell collection of the active sheet
                    cellCollection = sheetModel.getCellCollection(),
                    // the ranges to be updated and rendered
                    updateRanges = new RangeArray(),
                    // all matrix cell elements with overflowing text, mapped by cell key, for second-step-update
                    overflowElementMap = {},
                    // all matrix cell elements whose overflowing neighbors need to be updated
                    neighborElementMap = {},
                    // descriptors for all visited merged ranges starting outside the update ranges
                    outerMergedData = null;

                // updates cell formatting and all text settings of a single matrix cell element
                function updateCellElement(element) {
                    var address = element.merged ? element.merged.mergedRange.start : element.address;
                    element.update(styleCache, numberFormatter, sheetModel, cellCollection.getExistingCellEntry(address));
                }

                // returns the pixel position of the next available adjacent merged range
                function findNextMergedRectangle(element, forward) {
                    var mergedData = findNextMergedData(element.address, forward);
                    return mergedData ? sheetModel.getRangeRectangle(mergedData.mergedRange) : null;
                }

                // finds the content cell next to the passed cell element
                function findNextContentElement(element, forward) {
                    var prop = forward ? 'r' : 'l';
                    for (element = element[prop]; element && element.blank; element = element[prop]) {}
                    return element ? element : null;
                }

                // finds the column header of a content cell next to the passed cell element
                function findNextContentCol(element, forward) {
                    element = findNextContentElement(element, forward);
                    return element ? element.col : null;
                }

                // finds and registers a matrix element with overflowing text next to the passed element
                function registerAdjacentOverflowCell(element, forward) {

                    // find next available content cell element
                    var nextElement = findNextContentElement(element, forward);

                    // no content cell in the matrix: search in cell collection, create a dummy cell element
                    if (!nextElement) {
                        var cellDesc = cellCollection.findNearestCell(element.address, forward ? 'right' : 'left', { type: 'content' });
                        // skip existing elements without overflowing text
                        if (!cellDesc || !CellCollection.isOverflowText(cellDesc)) { return; }
                        // create a dummy matrix element
                        nextElement = new RenderUtils.MatrixCellElement(cellDesc.address);
                        nextElement.update(styleCache, numberFormatter, sheetModel, cellDesc);
                    }

                    // skip existing elements without overflowing text
                    if (!nextElement.hasOverflowText(!forward)) { return; }

                    // check that no merged range is between overflow cell and current position
                    var mergedData = findNextMergedData(element.address, forward);
                    if (mergedData && (forward ? (mergedData.mergedRange.start[0] <= nextElement.address[0]) : (nextElement.address[0] <= mergedData.mergedRange.end[0]))) { return; }

                    // register matrix cell element for clipping update
                    var overflowEntry = overflowElementMap[nextElement.key];
                    if (!overflowEntry) {
                        overflowEntry = overflowElementMap[nextElement.key] = { element: nextElement };
                    }

                    // performance: store the original matrix element causing text clipping in the map,
                    // to prevent searching for that element later (used as clipping boundary)
                    if (!element.blank) {
                        overflowEntry[forward ? 'l' : 'r'] = element;
                    }
                }

                // adds a range address to the update ranges covering the surrounding cells of an overflowing text cell
                function addOverflowRange(address, overflowL, overflowR) {
                    var overflowRange = docModel.makeRowRange(address[1]);
                    if (!overflowL) { overflowRange.start[0] = address[0]; }
                    if (!overflowR) { overflowRange.end[0] = address[0]; }
                    updateRanges.push(overflowRange);
                }

                // reduce update ranges to bounding row intervals (but do not reduce to bounding
                // column intervals, original ranges are needed for updating overflowing text cells)
                dirtyRanges = dirtyRanges.merge();
                rowHeader.intervals.forEach(function (interval) {
                    updateRanges.append(dirtyRanges.intersect(docModel.makeRowRange(interval)));
                });

                // merge the ranges to be updated, and shrink them to visible areas
                updateRanges = sheetModel.mergeAndShrinkRanges(updateRanges);
                RenderUtils.log('update ranges: ' + updateRanges);

                // update all existing matrix elements
                RenderUtils.takeTime('update matrix elements', function () {
                    outerMergedData = iterateCellsInRanges(updateRanges, function (element) {

                        var // whether the cell element was blank before
                            oldBlank = element.blank,
                            // whether the cell element has contained overflowing text before
                            oldOverflowL = element.hasOverflowText(false),
                            oldOverflowR = element.hasOverflowText(true);

                        // find and store descriptor of a merged range covering the cell element
                        element.merged = mergedDataMap[element.key] || findMergedData(element.address);

                        // update cell formatting and all text settings
                        updateCellElement(element);

                        var // whether the cell element is blank now
                            newBlank = element.blank,
                            // whether the cell element contains overflowing text now
                            newOverflowL = element.hasOverflowText(false),
                            newOverflowR = element.hasOverflowText(true);

                        // register overflowing cell elements for update of clipping area
                        // in next step (after all matrix cell elements have been updated)
                        if (newOverflowL || newOverflowR) {
                            overflowElementMap[element.key] = { element: element };
                        }

                        // ensure to invalidate cell ranges that contained overflowing text before
                        if (oldOverflowL || oldOverflowR) {
                            addOverflowRange(element.address, oldOverflowL, oldOverflowR);
                        }

                        // update adjacent cell elements with text floating towards the updated ranges, if blank state changes
                        if (oldBlank !== newBlank) {
                            neighborElementMap[element.key] = element;
                        }
                    });
                });

                // update origin cells of merged ranges
                RenderUtils.takeTime('update origins of merged range', function () {

                    // bug 41135: create missing matrix elements for all merged range in the
                    // rendering matrix (not only the merged ranges covered by the dirty ranges)
                    _.each(mergedDataMap, function (mergedData) {
                        if (!mergedData.element) {
                            mergedData.element = new RenderUtils.MatrixCellElement(mergedData.mergedRange.start);
                            mergedData.element.merged = mergedData;
                            updateCellElement(mergedData.element);
                        }
                    });

                    // update origins of merged ranges, if contained in the original (not shrunken) dirty ranges
                    outerMergedData.forEach(function (mergedData) {
                        if (dirtyRanges.containsAddress(mergedData.mergedRange.start)) {
                            updateCellElement(mergedData.element);
                        }
                    });
                });

                // update clipping rectangles of cells with overflowing text (restrict to content cells and merged ranges)
                RenderUtils.takeTime('update text clipping rectangles', function () {

                    // find adjacent cells with overflowing text (left/right of changed cells)
                    _.each(neighborElementMap, function (element) {
                        registerAdjacentOverflowCell(element, false);
                        registerAdjacentOverflowCell(element, true);
                    });

                    // update clipping rectangles of all registered cells
                    _.each(overflowElementMap, function (overflowEntry) {

                        var // the matrix cell element to be updated
                            element = overflowEntry.element,
                            // the inner cell rectangle
                            innerRect = element.text.inner,
                            // the new clipping positions
                            clipL = innerRect.left - element.text.overflowL,
                            clipR = innerRect.left + innerRect.width + element.text.overflowR,
                            // whether left or right text overflows
                            overflowL = element.hasOverflowText(false),
                            overflowR = element.hasOverflowText(true),
                            // position of merged range next to the current cell element
                            mergedRect = null,
                            // column header of adjacent cell element with content value
                            contentCol = null;

                        // reduce left clipping position according to existing merged ranges and content cell
                        if (overflowL) {
                            if (overflowEntry.l) {
                                contentCol = overflowEntry.l.col;
                            } else {
                                mergedRect = findNextMergedRectangle(element, false);
                                contentCol = findNextContentCol(element, false);
                            }
                            if (mergedRect) { clipL = Math.max(clipL, mergedRect.left + mergedRect.width); }
                            if (contentCol) { clipL = Math.max(clipL, contentCol.offset + contentCol.size); }
                        }

                        // reduce right clipping position according to existing merged ranges and content cell
                        if (overflowR) {
                            if (overflowEntry.r) {
                                contentCol = overflowEntry.r.col;
                            } else {
                                mergedRect = findNextMergedRectangle(element, true);
                                contentCol = findNextContentCol(element, true);
                            }
                            if (mergedRect) { clipR = Math.min(clipR, mergedRect.left - GRID_LINE_WIDTH); }
                            if (contentCol) { clipR = Math.min(clipR, contentCol.offset - GRID_LINE_WIDTH); }
                        }

                        // set the new clip position
                        element.text.clip.left = clipL;
                        element.text.clip.width = clipR - clipL;

                        // add overflow range to update ranges for rendering
                        if (overflowL || overflowR) {
                            addOverflowRange(element.address, overflowL, overflowR);
                        }
                    });
                });

                // reset settings for dirty ranges for next update cycle
                dirtyRanges.clear();

                // notify rendering listeners
                if (!updateRanges.empty()) {
                    self.trigger('update:matrix', updateRanges, options);
                }
            });

            // create the debounced version of RenderCache.registerDirtyRanges(),
            // that also waits for document operations currently applied
            var registerDirtyRangesDebounced = docView.getApp().createDebouncedActionsMethodFor(self, registerRanges, updateMatrix);

            // the actual method RenderCache.registerDirtyRanges() to be returned from the local scope
            function registerDirtyRanges(ranges, options) {

                // refresh dirty ranges immediately depending on the passed option
                if (Utils.getBooleanOption(options, 'direct', false)) {
                    registerRanges(ranges);
                    updateMatrix({ direct: true });
                } else {
                    // default: debounced update for remote changes and every local changes
                    // which do not affect the optimal column width or row height
                    registerDirtyRangesDebounced(ranges);
                }
            }

            return registerDirtyRanges;
        }());

        /**
         * Updates all cells in this rendering cache that have been changed in
         * the active sheet (handler of the 'change:cells' event triggered by
         * the cell collection of the active sheet).
         */
        var changeCellsHandler = RenderUtils.profileMethod('RenderCache.changeCellsHandler()', function (event, ranges, options) {
            registerDirtyRanges(ranges, { direct: Utils.getBooleanOption(options, 'direct', false) });
        });

        /**
         * Inserts all merged ranges into this rendering cache that have been
         * inserted into the active sheet (handler of the 'insert:merged' event
         * triggered by the merged ranges collection of the active sheet).
         */
        var insertMergedHandler = RenderUtils.profileMethod('RenderCache.insertMergedHandler()', function (event, mergedRanges) {

            // filter all visible merged range overlapping with the visible rows
            mergedRanges = mergedRanges.filter(function (mergedRange) {
                return rowHeader.findFirst(mergedRange.start[1], mergedRange.end[1]);
            });

            // insert all merged ranges into the caches
            mergedRanges.forEach(insertMergedRange);

            // clear the sorted arrays of all covered row header elements (will be updated on demand)
            rowHeader.resetMergedData(mergedRanges);

            // update all matrix cells elements covered by the merged ranges
            registerDirtyRanges(mergedRanges);
        });

        /**
         * Removes all merged ranges from this rendering cache that have been
         * removed from the active sheet (handler of the 'delete:merged' event
         * triggered by the merged ranges collection of the active sheet).
         */
        var deleteMergedHandler = RenderUtils.profileMethod('RenderCache.deleteMergedHandler()', function (event, mergedRanges) {

            // delete all merged ranges from the cache
            mergedRanges.forEach(function (mergedRange) {
                delete mergedDataMap[mergedRange.start.key()];
            });

            // clear the sorted arrays of all covered row header elements (will be updated on demand)
            rowHeader.resetMergedData(mergedRanges);

            // update all matrix cells elements covered by the merged ranges
            registerDirtyRanges(mergedRanges);
        });

        /**
         * Updates this rendering cache, after the grid color of the active
         * sheet has been changed (cell borders with automatic line color are
         * using the current sheet grid color).
         */
        function changeSheetAttributesHandler(event, newAttributes, oldAttributes) {
            if (!_.isEqual(newAttributes.sheet.gridColor, oldAttributes.sheet.gridColor)) {
                styleCache.updateStyles();
                if (totalRange) { registerDirtyRanges(totalRange); }
            }
        }

        // public methods -----------------------------------------------------

        /**
         * Invokes the passed callback function for all visible columns or rows
         * covered by the passed interval.
         *
         * @param {Range} range
         *  The cell range whose visible columns or rows will be visited.
         *
         * @param {Boolean} columns
         *  Whether to visit the visible columns in the passed range (true), or
         *  the visible rows (false).
         *
         * @param {Function} callback
         *  The callback function invoked for every visible column or row.
         *  Receives the following parameters:
         *  (1) {MatrixHeaderElement} element
         *      The header element of the rendering matrix currently visited.
         *  Will be called in the context of this rendering cache instance.
         *
         * @returns {RenderCache}
         *  A reference to this instance.
         */
        this.iterateInterval = function (range, columns, callback) {
            var header = columns ? colHeader : rowHeader,
                interval = range.interval(columns);
            header.iterateElements(interval.first, interval.last, callback, this);
            return this;
        };

        /**
         * Visits all merged ranges overlapping with the rendering matrix.
         *
         * @param {Function} callback
         *  The callback function invoked for every merged range overlapping
         *  with the rendering matrix. Receives the following parameters:
         *  (1) {Range} mergedRange
         *      The original address of the merged range.
         *  (2) {Range} shrunkenRange
         *      The address of the merged range shrunken to the visible area
         *      (without leading and trailing hidden columns and rows).
         *  Will be called in the context of this rendering cache instance.
         *
         * @returns {RenderCache}
         *  A reference to this instance.
         */
        this.iterateMergedRanges = function (callback) {
            _.each(mergedDataMap, function (mergedData) {
                callback.call(this, mergedData.mergedRange, mergedData.shrunkenRange);
            }, this);
            return this;
        };

        /**
         * Invokes the passed callback function for all existing cell elements
         * in the rendering matrix covered by the passed cell range.
         *
         * @param {Range} range
         *  The address of a cell range whose matrix elements will be visited.
         *
         * @param {Function} callback
         *  The callback function invoked for every matrix cell element.
         *  Receives the following parameters:
         *  (1) {MatrixCellElement} element
         *      The cell element from the rendering matrix.
         *  Will be called in the context of this rendering cache instance.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.columns=false]
         *      If set to true, the cells in the passed range will be visited
         *      in vertical orientation (all cells top-down in first column,
         *      then all cells top-down in second column, etc.). By default,
         *      the cells will be visited in horizontal orientation (all cells
         *      left-to-right in first row, then all cells left-to-right in
         *      second row, etc.).
         *  @param {Boolean} [options.first=false]
         *      If set to true, only the matrix elements of the first visible
         *      row (or the first visible column in vertical mode, see option
         *      'columns') will be visited.
         *  @param {Boolean} [options.origins=false]
         *      If set to true, the origin cells of all merged ranges covering
         *      the passed range will be visited, regardless if they are inside
         *      the passed range; and the cells in the passed range that are
         *      covered by these merged ranges will not be visited.
         *
         * @returns {RenderCache}
         *  A reference to this instance.
         */
        this.iterateCells = function (range, callback, options) {

            var // whether to visit the origin cells of merged range sonly
                origins = Utils.getBooleanOption(options, 'origins', false),
                // merge descriptors for all merged ranges starting outside the range
                outerMergedData = null;

            // invoke passed callback function for the cells in all ranges
            outerMergedData = iterateCellsInRanges(range, function (element) {
                if (!origins || !element.isMergedCovered()) {
                    callback.call(self, element);
                }
            }, options);

            // visit origins of all merged ranges starting outside
            if (origins) {
                outerMergedData.forEach(function (mergedData) {
                    callback.call(self, mergedData.element);
                });
            }

            return this;
        };

        /**
         * Invokes the passed callback function for all existing cell elements
         * in the specified column of the rendering matrix.
         *
         * @param {Number} col
         *  The zero-based column index.
         *
         * @param {Function} callback
         *  The callback function invoked for every matrix cell element in the
         *  column. Receives the following parameters:
         *  (1) {MatrixCellElement} element
         *      The cell element from the rendering matrix.
         *  Will be called in the context of this rendering cache instance.
         *
         * @returns {RenderCache}
         *  A reference to this instance.
         */
        this.iterateCellsInCol = function (col, callback) {
            return this.iterateCells(docModel.makeColRange(col), callback);
        };

        /**
         * Invokes the passed callback function for all existing cell elements
         * in the specified row of the rendering matrix.
         *
         * @param {Number} row
         *  The zero-based row index.
         *
         * @param {Function} callback
         *  The callback function invoked for every matrix cell element in the
         *  column. Receives the following parameters:
         *  (1) {MatrixCellElement} element
         *      The cell element from the rendering matrix.
         *  Will be called in the context of this rendering cache instance.
         *
         * @returns {RenderCache}
         *  A reference to this instance.
         */
        this.iterateCellsInRow = function (row, callback) {
            return this.iterateCells(docModel.makeRowRange(row), callback);
        };

        /**
         * Invokes the passed callback function for all existing cell elements
         * with text contents in the rendering matrix covered by the passed
         * cell range.
         *
         * @param {Range} range
         *  The address of a cell range whose matrix elements will be visited.
         *
         * @param {Function} callback
         *  The callback function invoked for every matrix cell element.
         *  Receives the following parameters:
         *  (1) {MatrixCellElement} element
         *      The cell element from the rendering matrix. Each visited matrix
         *      element will contain a property 'text' referring to a valid
         *      text descriptor.
         *  Will be called in the context of this rendering cache instance.
         *
         * @returns {RenderCache}
         *  A reference to this instance.
         */
        this.iterateTextCells = function (range, callback) {

            function findNextOverflowElement(element, forward) {
                var prop = forward ? 'r' : 'l';
                for (element = element[prop]; element && element.blank && !element.isMerged(); element = element[prop]) {}
                return (element && !element.isMerged() && element.hasOverflowText(!forward)) ? element : null;
            }

            // visit all matrix cell elements in the range
            this.iterateCells(range, function (element) {

                // invoke the callback function for text elements
                if (element.text) {
                    callback.call(self, element);
                    return;
                }

                // do nothing else for non-blank cells without visible display text
                if (!element.blank) { return; }

                var nextElement = null;

                // blank cell in first column: invoke callback function for text floating from left
                if (!element.l || (element.l.address[0] < range.start[0])) {
                    if ((nextElement = findNextOverflowElement(element, false))) {
                        callback.call(self, nextElement);
                    }
                }

                // blank cell in last column: invoke callback function for text floating from right
                if (!element.r || (element.r.address[0] > range.end[0])) {
                    if ((nextElement = findNextOverflowElement(element, true))) {
                        callback.call(self, nextElement);
                    }
                }
            }, { origins: true });

            return this;
        };

        /**
         * Returns whether this cache currently contains pending dirty ranges
         * overlapping with the passed cell range.
         *
         * @param {Range} range
         *  The range to be compared with the pending dirty ranges.
         *
         * @returns {Boolean}
         *  Whether this cache currently contains pending dirty ranges
         *  overlapping with the passed cell range.
         */
        this.hasDirtyRanges = function (range) {
            return dirtyRanges.overlaps(range);
        };

        /**
         * Updates the layer ranges of this rendering cache according to the
         * passed bounding intervals, and returns descriptors for all changed
         * layer ranges.
         *
         * @param {Object} headerBoundaryMap
         *  All changed bounding intervals, as instances of the class
         *  HeaderBoundary, mapped by pane side identifiers. The boundary
         *  intervals of pane sides missing in this map will not be changed. If
         *  a boundary in this map exists and is invalid (all of its properties
         *  are null), the respective layer ranges will be removed from the
         *  rendering cache.
         *
         * @param {RangeArray} invalidRanges
         *  An array with the addresses of all dirty cell ranges in the sheet
         *  that must be recalculated and repainted (e.g. after one or more
         *  document operations such as inserting or deleting columns or rows).
         *
         * @returns {Object|Null}
         *  A map containing descriptors for all changed layer ranges, as
         *  instances of the class RangeBoundary, mapped by pane position
         *  identifier. The boundary ranges of pane positions missing in this
         *  map have not been changed. If a descriptor in this map exists and
         *  is invalid (all of its properties are null), the layer range has
         *  been removed from the rendering cache.
         */
        this.updateLayerRanges = RenderUtils.profileMethod('RenderCache.updateLayerRanges()', function (headerBoundaryMap, invalidRanges) {

            var // the model of the active sheet
                sheetModel = docView.getSheetModel(),
                // the column, row, and cell collection of the active sheet
                colCollection = sheetModel.getColCollection(),
                rowCollection = sheetModel.getRowCollection(),
                cellCollection = sheetModel.getCellCollection(),
                // flags for changed intervals (per pane side), and changed columns/rows
                changedSides = {},
                changedCols = false,
                changedRows = false,
                // the pane positions of all changed bounding ranges
                changedPanes = [],
                // all cell ranges to be refreshed in the rendering matrix
                refreshRanges = new RangeArray(),
                // the changed range boundaries returned to caller
                resultBoundaryMap = {};

            _.each(headerBoundaryMap, function (boundary, paneSide) {

                var // whether to change a column or row interval
                    columns = PaneUtils.isColumnSide(paneSide),
                    // the column/row collection
                    collection = columns ? colCollection : rowCollection,
                    // the header elements of the rendering matrix
                    header = columns ? colHeader : rowHeader,
                    // the passed interval, shrunken to visible columns/rows
                    shrunkenInterval = boundary.isValid() ? collection.shrinkIntervalToVisible(boundary.interval) : null;

                // store the new bounding intervals
                if (shrunkenInterval) {
                    boundary.interval = shrunkenInterval;
                    header.boundaries[paneSide] = boundary;
                } else if (paneSide in header.boundaries) {
                    delete header.boundaries[paneSide];
                } else {
                    return;
                }

                // set the correct change marker flags
                changedSides[paneSide] = true;
                if (columns) { changedCols = true; } else { changedRows = true; }
            });

            // nothing more to do, if no bounding intervals have changed at all
            if (!changedCols && !changedRows) { return null; }

            // calculate the unified bounding intervals (shrunken to visible parts)
            colHeader.intervals = colCollection.mergeAndShrinkIntervals(_.pluck(colHeader.boundaries, 'interval'));
            rowHeader.intervals = rowCollection.mergeAndShrinkIntervals(_.pluck(rowHeader.boundaries, 'interval'));

            // calculate the new bounding ranges for all combinations of column/row intervals
            _.each(PaneUtils.ALL_PANE_POSITIONS, function (panePos) {

                var // column/row side identifier
                    colPaneSide = PaneUtils.getColPaneSide(panePos),
                    rowPaneSide = PaneUtils.getRowPaneSide(panePos);

                // nothing to do, if neither column nor row interval has changed
                if (!changedSides[colPaneSide] && !changedSides[rowPaneSide]) { return; }

                var // current column/row interval and position
                    colBoundary = colHeader.boundaries[colPaneSide],
                    rowBoundary = rowHeader.boundaries[rowPaneSide];

                // update bounding range
                if (colBoundary && rowBoundary) {
                    boundaryMap[panePos] = new PaneUtils.RangeBoundary(colBoundary, rowBoundary);
                    changedPanes.push(panePos);
                } else {
                    delete boundaryMap[panePos];
                    changedPanes.push(panePos);
                }
            });

            // calculate unified bounding ranges, get total cell range
            boundRanges = new RangeArray(_.pluck(boundaryMap, 'range')).merge();
            totalRange = boundRanges.boundary();
            RenderUtils.log('bounding ranges: ' + boundRanges);

            // fetch missing cells from server (new cells will be notified with 'change:cells' event later)
            cellCollection.fetchMissingCellRanges(boundRanges, { visible: true, merged: true });

            // update settings in auto style cache
            styleCache.updateStyles();

            // update column and row headers in the rendering matrix (delete unused cell elements, insert new cell elements)
            RenderUtils.takeTime('create/delete matrix elements', function () {
                if (changedCols) { refreshRanges.append(buildMatrixElements(true)); }
                if (changedRows) { refreshRanges.append(buildMatrixElements(false)); }
            });

            // nothing more to update without existing bounding ranges
            if (totalRange) {

                // add dirty ranges passed to this method
                refreshRanges.append(invalidRanges);

                // reset cached merged ranges after document operations
                if (!invalidRanges.empty()) { mergedDataMap = {}; }

                // Refresh the cached merged ranges, if either the bounding row intervals have changed,
                // or if columns or rows have been changed in the document (especially the visibility
                // of columns or rows). There is no need to update them when the column intervals have
                // changed without any changes in the document (e.g. while scrolling horizontally),
                // because all merged ranges from the entire bounding rows will be cached.
                if (!invalidRanges.empty() || changedRows) { updateMergedRanges(); }

                // register the ranges to be refreshed (there may already be other pending dirty
                // ranges received from 'change:cells' events).
                registerDirtyRanges(refreshRanges);

            } else {
                // no more bounding ranges available: reset all pending dirty ranges immediately
                mergedDataMap = {};
                dirtyRanges.clear();
            }

            // fill the changed range boundaries into the result map returned to caller
            changedPanes.forEach(function (panePos) {
                var boundary = boundaryMap[panePos];
                resultBoundaryMap[panePos] = boundary ? boundary.clone() : new PaneUtils.RangeBoundary();
            });
            return resultBoundaryMap;
        });

        // initialization -----------------------------------------------------

        // update rendering matrix after any cells or merged ranges have changed in the active sheet
        this.listenTo(docView, 'change:cells', changeCellsHandler);
        this.listenTo(docView, 'insert:merged', insertMergedHandler);
        this.listenTo(docView, 'delete:merged', deleteMergedHandler);

        // update cell entries with borders after the grid color has changed
        this.listenTo(docView, 'change:sheet:attributes', changeSheetAttributesHandler);

        // destroy all class members on destruction
        this.registerDestructor(function () {
            styleCache.destroy();
            self = docView = docModel = numberFormatter = null;
            styleCache = cellMatrix = colHeader = rowHeader = boundaryMap = boundRanges = totalRange = null;
            mergedDataMap = dirtyRanges = null;
        });

    } // class RenderCache

    // exports ================================================================

    // derive this class from class TriggerObject
    return TriggerObject.extend({ constructor: RenderCache });

});
