/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/render/autostylecache', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/object/baseobject',
    'io.ox/office/tk/object/timermixin',
    'io.ox/office/editframework/utils/color',
    'io.ox/office/spreadsheet/utils/paneutils',
    'io.ox/office/spreadsheet/view/render/renderutils'
], function (Utils, BaseObject, TimerMixin, Color, PaneUtils, RenderUtils) {

    'use strict';

    var // convenience shortcuts
        StyleDescriptor = RenderUtils.StyleDescriptor;

    // class AutoStyleCache ===================================================

    /**
     * A cache that stores rendering information for all known auto styles in
     * the document.
     *
     * @constructor
     *
     * @extends BaseObject
     * @extends TimerMixin
     *
     * @param {SpreadsheetView} docView
     *  The document view that contains this cache instance.
     */
    function AutoStyleCache(docView) {

        var // self reference
            self = this,

            // the spreadsheet model
            docModel = docView.getDocModel(),

            // map of all style descriptor caches
            styleCacheMap = {},

            // the current style descriptor cache (according to grid color and zoom)
            styleCache = null,

            // the current grid line color
            gridColor = new Color('auto'),

            // the current zoom factor
            sheetZoom = 0;

        // base constructors --------------------------------------------------

        BaseObject.call(this);
        TimerMixin.call(this);

        // private methods ----------------------------------------------------

        /**
         * In debug mode, collects the use counts of multiple cache accesses,
         * and prints a message to the browser console.
         */
        var logUsage = RenderUtils.isLoggingActive() ? (function () {

            var counts = { total: 0, hit: 0, update: 0, miss: 0, uncached: 0 };

            function collect(type) {
                counts[type] += 1;
                counts.total += 1;
            }

            function format(key) {
                var count = counts[key];
                return (count > 0) ? (', ' + key + '=' + count + ' (' + Math.round(count / counts.total * 1000) / 10 + '%)') : '';
            }

            function flush() {
                RenderUtils.info('AutoStyleCache access statistics: total=' + counts.total + format('hit') + format('update') + format('miss') + format('uncached'));
                _.each(counts, function (count, key) { counts[key] = 0; });
            }

            return self.createDebouncedMethod(collect, flush);
        }()) : _.noop;

        /**
         * Updates the passed style descriptor, if its own age is less than the
         * age of the current style cache.
         *
         * @param {StyleDescriptor} styleDesc
         *  The style descriptor to be updated if needed.
         *
         * @returns {StyleDescriptor}
         *  The passed style descriptor, for convenience.
         */
        function updateStyleDescriptor(styleDesc, type) {
            if (styleDesc.age < styleCache.age) {
                styleDesc.update(docModel, gridColor, sheetZoom);
                styleDesc.age = styleCache.age;
                if (!type) { type = 'update'; }
            }
            logUsage(type || 'hit');
            return styleDesc;
        }

        /**
         * Creates a new style descriptor for the formatting attributes of the
         * specified column, row, or cell descriptor.
         *
         * @param {ColRowDescriptor|CellDescriptor} entryDesc
         *  The descriptor object of a column/row (as returned by the class
         *  ColRowCollection), or the descriptor of a single cell (as returned
         *  by the class CellCollection).
         *
         * @returns {StyleDescriptor}
         *  The new style descriptor for the passed column, row, or cell.
         */
        function createStyleDescriptor(entryDesc, type) {
            var styleDesc = new StyleDescriptor(entryDesc.attributes);
            styleDesc.age = 0;
            return updateStyleDescriptor(styleDesc, type || 'miss');
        }

        // public methods -----------------------------------------------------

        /**
         * Returns a style descriptor with rendering information for the
         * specified column, row, or cell.
         *
         * @param {ColRowDescriptor|CellDescriptor} entryDesc
         *  The descriptor object of a column/row (as returned by the class
         *  ColRowCollection), or the descriptor of a single cell (as returned
         *  by the class CellCollection). Expects the properties 'style' (the
         *  identifier of an auto style), 'explicit' (additional explicit
         *  formatting attributes to be merged over the auto style), and
         *  'attributes' (the effective merged attribute set).
         *
         * @returns {StyleDescriptor}
         *  The style descriptor for the specified column, row, or cell.
         */
        this.getStyle = function (entryDesc) {

            // always create a new style descriptor for explicit attributes, but do not store it in this cache
            if (!_.isEmpty(entryDesc.explicit)) {
                return createStyleDescriptor(entryDesc, 'uncached');
            }

            // style descriptor exists already in the cache, update it on demand
            var styleId = entryDesc.style || '';
            if (styleId in styleCache) {
                return updateStyleDescriptor(styleCache[styleId]);
            }

            // create a new style descriptor for an auto style, and store it in the cache
            return (styleCache[styleId] = createStyleDescriptor(entryDesc));
        };

        /**
         * Updates this cache according to the grid color and zoom index of the
         * active sheet in the spreadsheet document, and marks the cached style
         * descriptors to be dirty, if the grid color or zoom factor has really
         * been changed (the current grid color is used to render cell border
         * lines with automatic line color). When accessing a dirty style
         * descriptor the next time via the method AutoStyleCache.getStyle(),
         * its formatting settings will be updated automatically.
         *
         * @returns {AutoStyleCache}
         *  A reference to this instance.
         */
        this.updateStyles = function () {

            var // model of the active sheet in the document
                sheetModel = docView.getSheetModel(),
                // current grid color of the active sheet
                newGridColor = sheetModel.getGridColor(),
                // current grid color, as CSS color value
                oldCssColor = docModel.resolveColor(gridColor, 'line').css,
                // new grid color, as CSS color value
                newCssColor = docModel.resolveColor(newGridColor, 'line').css,
                // whether the grid color has been changed
                gridColorChanged = oldCssColor !== newCssColor,
                // current zoom factor of the active sheet
                newSheetZoom = sheetModel.getEffectiveZoom(),
                // whether the zoom factor has been changed
                sheetZoomChanged = sheetZoom !== newSheetZoom;

            // nothing to do, if neither effective CSS grid color nor zoom factor have changed
            if (!gridColorChanged && !sheetZoomChanged) { return this; }

            // store the new settings, mark the styles to be dirty, but do not touch the descriptors yet
            gridColor = newGridColor;
            sheetZoom = newSheetZoom;

            var // use permanent cache for simple grid colors
                permColorKey = /^#(00|FF)(00|FF)(00|FF)$/i.test(newCssColor) ? newCssColor.toUpperCase() : '',
                // use permanent cache for predefined zoom factors
                permZoomKey = _.contains(PaneUtils.ZOOM_FACTORS, sheetZoom) ? Math.round(sheetZoom * 100) : '',
                // key of a the current style descriptor cache
                cacheKey = permColorKey + ':' + permZoomKey;

            // create the style descriptor cache if missing
            styleCache = styleCacheMap[cacheKey] || (styleCacheMap[cacheKey] = { styles: {}, age: 1 });

            // update the age of non-permanent style caches, if grid color or zoom factor has changed
            if ((gridColorChanged && !permColorKey) || (sheetZoomChanged && !permZoomKey)) {
                RenderUtils.log('AutoStyleCache.updateStyles(): style cache "' + cacheKey + '" invalidated');
                styleCache.age += 1;
            } else {
                RenderUtils.log('AutoStyleCache.updateStyles(): style cache "' + cacheKey + '" activated');
            }

            return this;
        };

        // initialization -----------------------------------------------------

        // invalidate the cache after inserting new fonts (CSS font family chains may change)
        this.listenTo(docModel.getFontCollection(), 'triggered', function () {
            styleCacheMap = {};
            self.updateStyles();
        });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            self = docView = docModel = styleCacheMap = styleCache = null;
        });

    } // class AutoStyleCache

    // exports ================================================================

    // derive this class from class BaseObject
    return BaseObject.extend({ constructor: AutoStyleCache });

});
