/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/labels', [
    'io.ox/office/tk/utils',
    'io.ox/office/editframework/view/editlabels',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'gettext!io.ox/office/spreadsheet/main'
], function (Utils, EditLabels, SheetUtils, gt) {

    'use strict';

    var // convenience shortcuts
        Address = SheetUtils.Address;

    // static class SpreadsheetLabels =========================================

    /**
     * Provides generic caption texts for control groups, menu headers, and
     * other GUI elements for the Spreadsheet application in a single map for
     * convenience.
     *
     * @extends EditLabels
     */
    var SpreadsheetLabels = _.extend({}, EditLabels);

    // header labels ----------------------------------------------------------

    /**
     * Header label for column/row operations.
     *
     * @constant
     */
    SpreadsheetLabels.COL_ROW_HEADER_LABEL = /* menu title: insert/delete/resize rows and columns in the sheet */ gt.pgettext('menu-title', 'Rows/Columns');

    /**
     * Header label for data filter operations.
     *
     * @constant
     */
    SpreadsheetLabels.FILTER_HEADER_LABEL = /*#. Header label for data filter options */ gt.pgettext('menu-title', 'Filter');

    // control labels ---------------------------------------------------------

    /**
     * Label for an 'Insert sheet' button.
     *
     * @constant
     */
    SpreadsheetLabels.INSERT_SHEET_LABEL = gt('Insert sheet');

    /**
     * Label for a 'Rename sheet' button.
     *
     * @constant
     */
    SpreadsheetLabels.RENAME_SHEET_LABEL = gt('Rename sheet');

    /**
     * Label for a 'Copy sheet' button.
     *
     * @constant
     */
    SpreadsheetLabels.COPY_SHEET_LABEL = gt('Copy sheet');

    /**
     * Label for a 'Delete sheet' button.
     *
     * @constant
     */
    SpreadsheetLabels.DELETE_SHEET_LABEL = gt('Delete sheet');

    /**
     * Label for a 'Hide sheet' button.
     *
     * @constant
     */
    SpreadsheetLabels.HIDE_SHEET_LABEL = gt('Hide sheet');

    /**
     * Label for a 'Reorder sheets' button.
     *
     * @constant
     */
    SpreadsheetLabels.REORDER_SHEETS_LABEL = gt('Reorder sheets');

    /**
     * Label for a 'Show all hidden sheets' button.
     *
     * @constant
     */
    SpreadsheetLabels.SHOW_SHEETS_LABEL = gt('Show all hidden sheets');

    /**
     * Label for a line-break toggle button.
     *
     * @constant
     */
    SpreadsheetLabels.LINEBREAK_LABEL =
        /*#. check box: automatic word wrapping (multiple text lines) in spreadsheet cells */ gt('Automatic word wrap');

    /**
     * A special value for the 'Custom' entry in the 'Number Format Code'
     * drop-down list, which must be different from any valid number format.
     *
     * @constant
     */
    SpreadsheetLabels.CUSTOM_FORMAT_VALUE = '\x00';

    /**
     * A text label or tool tip for a 'Format Painter' control.
     *
     * @constant
     */
    SpreadsheetLabels.FORMAT_PAINTER_LABEL =
        //#. copy formatting from one location and apply it to other cells
        gt('Format painter');

    /**
     * An map containing GUI labels for all function categories.
     *
     * @constant
     */
    SpreadsheetLabels.FUNCTION_CATEGORY_LABELS = {
        complex: /*#. Category for spreadsheet functions: complex numbers (e.g. IMSUM, IMSIN, IMPOWER) */ gt.pgettext('function-category', 'Complex'),
        conversion: /*#. Category for spreadsheet functions: conversion (e.g. DEC2HEX, ROMAN, CONVERT) */ gt.pgettext('function-category', 'Conversion'),
        database: /*#. Category for spreadsheet functions: database-like (e.g. DSUM, DCOUNT) */ gt.pgettext('function-category', 'Database'),
        datetime: /*#. Category for spreadsheet functions: date/time calculation (e.g. TODAY, WEEKDAY) */ gt.pgettext('function-category', 'Date and time'),
        engineering: /*#. Category for spreadsheet functions: engineering (e.g. BESSELI, ERF) */ gt.pgettext('function-category', 'Engineering'),
        financial: /*#. Category for spreadsheet functions: financial functions (e.g. ACCRINT, COUPDAYS) */ gt.pgettext('function-category', 'Financial'),
        information: /*#. Category for spreadsheet functions: information about cells (e.g. ISNUMBER, ISTEXT) */ gt.pgettext('function-category', 'Information'),
        logical: /*#. Category for spreadsheet functions: logical functions (e.g. AND, OR, NOT, IF) */ gt.pgettext('function-category', 'Logical'),
        math: /*#. Category for spreadsheet functions: common mathematical functions (e.g. SUM, SIN, ROUND) */ gt.pgettext('function-category', 'Math'),
        matrix: /*#. Category for spreadsheet functions: matrix functions (e.g. MMULT, MINVERSE, MDETERM) */ gt.pgettext('function-category', 'Matrix'),
        reference: /*#. Category for spreadsheet functions: cell references (e.g. ADDRESS, HLOOKUP, COLUMN) */ gt.pgettext('function-category', 'Reference'),
        statistical: /*#. Category for spreadsheet functions: statistical functions (e.g. COUNT, STDEV, HYPGEOMDIST) */ gt.pgettext('function-category', 'Statistical'),
        text: /*#. Category for spreadsheet functions: text functions (e.g. LEN, MID, REPLACE) */ gt.pgettext('function-category', 'Text'),
        web: /*#. Category for spreadsheet functions: web functions (e.g. ENCODEURL, HYPERLINK, WEBSERVICE) */ gt.pgettext('function-category', 'Web')
    };

    /**
     * An map containing GUI labels for all predefined number format
     * categories.
     *
     * @constant
     */
    SpreadsheetLabels.FORMAT_CATEGORY_LABELS = {
        standard: /*#. number format category in spreadsheets: no special format, show numbers as they are */ gt.pgettext('number-format', 'General'),
        number: /*#. number format category in spreadsheets: numbers with specific count of decimal places */ gt.pgettext('number-format', 'Number'),
        currency: /*#. number format category in spreadsheets: number with currency symbol */ gt.pgettext('number-format', 'Currency'),
        date: /*#. number format category in spreadsheets: date formats */ gt.pgettext('number-format', 'Date'),
        time: /*#. number format category in spreadsheets: time formats */ gt.pgettext('number-format', 'Time'),
        datetime: /*#. number format category in spreadsheets: combined date/time formats */ gt.pgettext('number-format', 'Date and time'),
        percent: /*#. number format category in spreadsheets: numbers with percent sign */ gt.pgettext('number-format', 'Percentage'),
        text: /*#. number format category in spreadsheets: text only */ gt.pgettext('number-format', 'Text'),
        scientific: /*#. number format category in spreadsheets: scientific notation (e.g. 1.23E+10) */ gt.pgettext('number-format', 'Scientific'),
        fraction: /*#. number format category in spreadsheets: fractional numbers (e.g. 3 1/4) */ gt.pgettext('number-format', 'Fraction'),
        custom: /*#. number format category in spreadsheets: all user-defined number formats */ gt.pgettext('number-format', 'Custom')
    };

    // control options --------------------------------------------------------

    /**
     * Standard options for a line-break toggle button.
     *
     * @constant
     */
    SpreadsheetLabels.LINEBREAK_BUTTON_OPTIONS = {
        icon: 'docs-insert-linebreak',
        tooltip: SpreadsheetLabels.LINEBREAK_LABEL,
        toggle: true,
        dropDownVersion: { label: SpreadsheetLabels.LINEBREAK_LABEL }
    };

    /**
     * Standard options for a 'Format painter' button.
     *
     * @constant
     */
    SpreadsheetLabels.FORMAT_PAINTER_BUTTON_OPTIONS = {
        icon: 'docs-format-painter',
        tooltip: SpreadsheetLabels.FORMAT_PAINTER_LABEL,
        toggle: true,
        dropDownVersion: { label: SpreadsheetLabels.FORMAT_PAINTER_LABEL }
    };

    /**
     * Standard options for a 'Insert rows' button.
     *
     * @constant
     */
    SpreadsheetLabels.INSERT_ROWS_OPTIONS = {
        label: gt('Insert rows')
    };

    /**
     * Standard options for a 'Insert row' button.
     *
     * @constant
     */
    SpreadsheetLabels.INSERT_ROW_OPTIONS = {
        label: gt('Insert row')
    };

    /**
     * Standard options for a 'Delete rows' button.
     *
     * @constant
     */
    SpreadsheetLabels.DELETE_ROWS_OPTIONS = {
        label: gt('Delete rows')
    };

    /**
     * Standard options for a 'Delete row' button.
     *
     * @constant
     */
    SpreadsheetLabels.DELETE_ROW_OPTIONS = {
        label: gt('Delete row')
    };

    /**
     * Standard options for 'Insert columns'.
     *
     * @constant
     */
    SpreadsheetLabels.INSERT_COLUMNS_OPTIONS = {
        label: gt('Insert columns')
    };

    /**
     * Standard options for 'Insert column'.
     *
     * @constant
     */
    SpreadsheetLabels.INSERT_COLUMN_OPTIONS = {
        label: gt('Insert column')
    };

    /**
     * Standard options for 'Delete columns'.
     *
     * @constant
     */
    SpreadsheetLabels.DELETE_COLUMNS_OPTIONS = {
        label: gt('Delete columns')
    };

    /**
     * Standard options for 'Delete column'.
     *
     * @constant
     */
    SpreadsheetLabels.DELETE_COLUMN_OPTIONS = {
        label: gt('Delete column')
    };

    /**
     * Standard options for 'filter function'.
     *
     * @constant
     */
    SpreadsheetLabels.FILTER_OPTIONS = {
        label: /*#. Button label: filter a cell range */ gt.pgettext('filter', 'Filter'),
        tooltip: gt('Filter the selected cells'),
        toggle: true
    };

    /**
     * Standard options for 'sort function'.
     *
     * @constant
     */
    SpreadsheetLabels.SORT_OPTIONS = {
        label: /*#. sorting: label of the button opening a menu with custom sort options */ gt.pgettext('sort', 'Custom'),
        tooltip: /*#. tooltip for custom sort button */ gt.pgettext('sort', 'Settings for sorting')
    };

    /**
     * Standard options for 'clean function'.
     *
     * @constant
     */
    SpreadsheetLabels.CLEAN_OPTIONS = {
        label: gt('Clear contents')
        //TODO: missing tooltip
    };

    // insert options ---------------------------------------------------------

    /**
     * Standard options for 'Insert sum'.
     *
     * @constant
     */
    SpreadsheetLabels.INSERT_SUM_OPTIONS = {
        label: gt('Sum'),
        tooltip: /*#. automatically create a SUM function for selected cells */ gt('Sum automatically'),
        value: 'SUM'
    };

    /**
     * Standard options for 'Insert function'.
     *
     * @constant
     */
    SpreadsheetLabels.INSERT_FUNCTION_OPTIONS = {
        label: gt('Function'),
        tooltip: /*#. Insert a function at the current cursor location */ gt('Insert function at cursor location')
    };

    // style sets for list controls -------------------------------------------

    /**
     * Predefined border styles (line width, line style) supported by OOXML.
     * Intended to be used as parameter for the constructor of the class
     * BorderStylePicker.
     *
     * @constant
     */
    SpreadsheetLabels.BORDER_OOXML_PRESET_STYLES = [
        { value: 'single:hair',       label: /*#. border line style (width 0.5 pixels, solid) */ gt.pgettext('borders', 'Hair line'),                         icon: { style: 'solid',      width: 0 } },
        { value: 'single:thin',       label: /*#. border line style (width 1 pixel, solid) */ gt.pgettext('borders', 'Thin line'),                            icon: { style: 'solid',      width: 1 } },
        { value: 'dashed:thin',       label: /*#. border line style (width 1 pixel, dashed) */ gt.pgettext('borders', 'Dashed thin line'),                    icon: { style: 'dashed',     width: 1 } },
        { value: 'dotted:thin',       label: /*#. border line style (width 1 pixel, dotted) */ gt.pgettext('borders', 'Dotted thin line'),                    icon: { style: 'dotted',     width: 1 } },
        { value: 'dashDot:thin',      label: /*#. border line style (width 1 pixel, dash-dot) */ gt.pgettext('borders', 'Dot-and-dash thin line'),            icon: { style: 'dashDot',    width: 1 } },
        { value: 'dashDotDot:thin',   label: /*#. border line style (width 1 pixel, dash-dot-dot) */ gt.pgettext('borders', 'Two-dots-and-dash thin line'),   icon: { style: 'dashDotDot', width: 1 } },
        { value: 'single:medium',     label: /*#. border line style (width 2 pixels, solid) */ gt.pgettext('borders', 'Medium line'),                         icon: { style: 'solid',      width: 2 } },
        { value: 'dashed:medium',     label: /*#. border line style (width 2 pixels, dashed) */ gt.pgettext('borders', 'Dashed medium line'),                 icon: { style: 'dashed',     width: 2 } },
        { value: 'dashDot:medium',    label: /*#. border line style (width 2 pixel, dash-dot) */ gt.pgettext('borders', 'Dot-and-dash medium line'),          icon: { style: 'dashDot',    width: 2 } },
        { value: 'dashDotDot:medium', label: /*#. border line style (width 2 pixel, dash-dot-dot) */ gt.pgettext('borders', 'Two-dots-and-dash medium line'), icon: { style: 'dashDotDot', width: 2 } },
        { value: 'single:thick',      label: /*#. border line style (width 3 pixels, solid) */ gt.pgettext('borders', 'Thick line'),                          icon: { style: 'solid',      width: 3 } },
        { value: 'double:thick',      label: /*#. border line style (width 3 pixels, double line) */ gt.pgettext('borders', 'Double line'),                   icon: { style: 'solid',      width: 1, count: 2 } }
    ];

    /**
     * All available simple border line styles for ODF files. Intended to be
     * used as parameter for the constructor of the class BorderStylePicker.
     *
     * @constant
     */
    SpreadsheetLabels.BORDER_ODF_LINE_STYLES = [
        { value: 'single', label: /*#. Border line style: a solid single line */ gt.pgettext('borders', 'Single'),  icon: { style: 'solid', width: 1 } },
        { value: 'double', label: /*#. Border line style: a solid double line */ gt.pgettext('borders', 'Double'),  icon: { style: 'solid', width: 1, count: 2 } },
        { value: 'dotted', label: /*#. Border line style: a dotted single line */ gt.pgettext('borders', 'Dotted'), icon: { style: 'dotted', width: 1 } },
        { value: 'dashed', label: /*#. Border line style: a dashed single line */ gt.pgettext('borders', 'Dashed'), icon: { style: 'dashed', width: 1 } }
    ];

    // other spreadsheet specific labels --------------------------------------

    /**
     * Returns the name of the passed column together with the word 'Column',
     * e.g. 'Column A'.
     *
     * @param {Number} col
     *  The zero-based column index to be converted to a column label.
     */
    SpreadsheetLabels.getColLabel = function (col) {
        return (
            //#. UI label for a spreadsheet column, e.g. "Column A", "Column B", etc.
            //#. %1$s is the name of the column
            //#, c-format
            gt('Column %1$s', _.noI18n(Address.stringifyCol(col))));
    };

    /**
     * Returns the name of the passed row together with the word 'Row', e.g.
     * 'Row 1'.
     *
     * @param {Number} row
     *  The zero-based row index to be converted to a row label.
     */
    SpreadsheetLabels.getRowLabel = function (row) {
        return (
            //#. UI label for a spreadsheet row, e.g. "Row 1", "Row 2", etc.
            //#. %1$s is the name of the row
            //#, c-format
            gt('Row %1$s', _.noI18n(Address.stringifyRow(row))));
    };

    /**
     * Returns the localized name of the specified function category.
     *
     * @param {String} category
     *  The internal identifier of a function category.
     *
     * @returns {String}
     *  The localized name of the specified function category.
     */
    SpreadsheetLabels.getCategoryLabel = function (category) {

        // return labels for valid categories
        if (category in SpreadsheetLabels.FUNCTION_CATEGORY_LABELS) {
            return SpreadsheetLabels.FUNCTION_CATEGORY_LABELS[category];
        }

        // assert an unknown category, return it capitalized
        Utils.error('SpreadsheetLabels.getCategoryLabel(): unknown function category: "' + category + '"');
        return _.noI18n(Utils.capitalize(category));
    };

    // exports ================================================================

    return SpreadsheetLabels;

});
