/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Stefan Eckert <stefan.eckert@open-xchange.com>
 */

define('io.ox/office/spreadsheet/services/commandmixin', [
    'io.ox/office/spreadsheet/utils/rangearray'
], function (RangeArray) {

    'use strict';

    // mix-in class CommandMixin =============================================

    /**
     *
     * @constructor
     */
    function CommandMixin() {

        var self = this;
        var model = null;
        var view = null;

        // private methods ----------------------------------------------------

        function setAutoWidthForSheet(sheet) {
            var promise = $.Deferred();

            function handle() {
                view.off('change:layoutdata', handle);

                var currentSelection = view.getSelection();

                var allSelection = view.getSelection();
                allSelection.ranges = new RangeArray(view.getCellCollection().getUsedRange());
                allSelection.active = 0;

                view.setCellSelection(allSelection);
                view.setOptimalColumnWidth().then(function () {
                    view.setCellSelection(currentSelection);
                    promise.resolve();
                });
            }

            if (view.getActiveSheet() === sheet) {
                handle();
            } else {
                view.setActiveSheet(sheet);
                view.on('change:layoutdata', handle);
            }

            return promise;
        }

        function setAutoWidth() {
            var activeSheet = model.getActiveSheet();

            var promise = $.when();
            _.times(model.getSheetCount(), function (sheet) {
                promise = promise.then(function () {
                    return setAutoWidthForSheet(sheet);
                });
            });

            promise = promise.done(function () {
                view.setActiveSheet(activeSheet);
            });

            return promise;
        }

        // initialization -----------------------------------------------------

        self.waitForSuccess(self.getImportFinishPromise(), function () {

            model = self.getModel();
            view = self.getView();

            function init() {
                var allSteps = [];
                if (self.getLaunchOption('autoWidth')) {
                    allSteps.push(setAutoWidth());
                }

                if (allSteps.length) {
                    view.enterBusy({ delay: 0, immediate: true });
                    $.when.apply($, allSteps).done(function () {
                        view.leaveBusy();
                    });
                }
            }

            function firstStep() {
                view.off('change:usedarea', firstStep);
                self.executeDelayed(init, { delay: 1000 });
            }

            view.on('change:usedarea', firstStep);
        });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            self = model = view = null;
        });

    } // class CommandMixin

    // exports ================================================================

    return CommandMixin;

});
