/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/namecollection', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/locale/localedata',
    'io.ox/office/baseframework/model/modelobject',
    'io.ox/office/spreadsheet/utils/operations',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/model/formula/tokenarray'
], function (Utils, LocaleData, ModelObject, Operations, SheetUtils, TokenArray) {

    'use strict';

    // convenience shortcuts
    var Address = SheetUtils.Address,
        Range = SheetUtils.Range;

    // class NameModel ========================================================

    /**
     * Stores settings for a single defined name in a specific sheet, or for a
     * global defined name in the document.
     *
     * @constructor
     *
     * @extends ModelObject
     *
     * @param {SpreadsheetModel|SheetModel} parentModel
     *  The spreadsheet document model (for globally defined names), or the
     *  sheet model (for sheet-local names) containing this instance.
     *
     * @param {String} label
     *  The exact label of this defined name, with correct character case as
     *  used in formula expressions.
     *
     * @param {String} initGrammar
     *  The identifier of the formula grammar to be used to parse the passed
     *  formula expression. See constructor of the class Tokenizer for details.
     *
     * @param {String} initFormula
     *  The formula expression bound to the defined name.
     *
     * @param {Object} [initOptions]
     *  Optional parameters:
     *  @param {String} [initOptions.refAddress]
     *      The address of the reference cell used to interpret relative cell
     *      references in the formula expression. The reference cell must be a
     *      formula expression by itself that results in a single cell address
     *      with sheet name.
     *  @param {Boolean} [initOptions.hidden=false]
     *      Whether the defined name should be hidden in the user interface.
     */
    var NameModel = ModelObject.extend({ constructor: function (parentModel, label, initGrammar, initFormula, initOptions) {

        var // the document model
            docModel = parentModel.getDocModel(),

            // the sheet model of sheet-local names
            sheetModel = (parentModel === docModel) ? null : parentModel,

            // the token array representing the definition of this name
            tokenArray = new TokenArray(parentModel, { type: 'name' }),

            // whether the defined name is hidden in the GUI
            hidden = Utils.getBooleanOption(initOptions, 'hidden', false);

        // base constructor ---------------------------------------------------

        ModelObject.call(this, docModel);

        // protected methods --------------------------------------------------

        /**
         * Creates and returns a cloned instance of this name for the specified
         * sheet.
         *
         * @internal
         *  Used by the class NameCollection during clone construction. DO NOT
         *  CALL from external code!
         *
         * @param {SpreadsheetModel|SheetModel} targetModel
         *  The model instance that will own the clone returned by this method.
         *
         * @returns {NameModel}
         *  A clone of this name model, initialized for ownership by the passed
         *  sheet model.
         */
        this.clone = function (targetModel) {

            var // construct a new name model
                nameModel = new NameModel(targetModel, label, 'op', tokenArray.getFormula('op'), initOptions),
                // the sheet index of this name model
                oldSheet = sheetModel ? sheetModel.getIndex() : null,
                // the sheet index of the new cloned name model
                newSheet = (targetModel !== docModel) ? targetModel.getIndex() : null;

            // relocate sheet references, if old and new parent models are sheet models
            if (_.isNumber(oldSheet) && _.isNumber(newSheet)) {
                nameModel.getTokenArray().relocateSheet(oldSheet, newSheet);
            }

            return nameModel;
        };

        // public methods -----------------------------------------------------

        /**
         * Returns the exact label of this name model, with correct character
         * case as used in formula expressions.
         *
         * @returns {String}
         *  The exact label of this name model.
         */
        this.getLabel = function () {
            return label;
        };

        /**
         * Changes the label of this name model.
         *
         * @param {String} newLabel
         *  The new label to be set for this name model.
         *
         * @returns {NameModel}
         *  A reference to this instance.
         */
        this.setLabel = function (newLabel) {
            label = newLabel;
            return this;
        };

        /**
         * Returns the token array representing the formula definition of this
         * name model.
         *
         * @returns {TokenArray}
         *  The token array containing the formula definition.
         */
        this.getTokenArray = function () {
            return tokenArray;
        };

        /**
         * Returns the formula expression of this name model, relative to the
         * passed target address.
         *
         * @param {String} grammar
         *  The identifier of the formula grammar to be used to generate the
         *  formula expression. See class GrammarConfig for details.
         *
         * @param {Address} targetAddress
         *  The target address to relocate the formula expression to.
         *
         * @returns {String}
         *  The relocated formula expression of this name model.
         */
        this.getFormula = function (grammar, targetAddress) {
            return tokenArray.getFormula(grammar, { targetAddress: targetAddress });
        };

        /**
         * Changes the formula definition of this name model.
         *
         * @param {String} grammar
         *  The identifier of the formula grammar to be used to parse the
         *  passed formula expression. See class GrammarConfig for details.
         *
         * @param {String} formula
         *  The new formula expression bound to the name model.
         *
         * @param {String} refAddressFormula
         *  The address of the reference cell used to interpret relative cell
         *  references in the formula expression. The reference cell must be a
         *  formula expression by itself that results in a single cell address
         *  with sheet name.
         *
         * @returns {NameModel}
         *  A reference to this instance.
         */
        this.setFormula = function (grammar, formula, refAddressFormula) {

            // parse the formula expression
            tokenArray.parseFormula(grammar, formula);

            // relocate formula according to reference address (used by ODF only, OOXML always uses A1)
            if (refAddressFormula) {

                // parse the reference address expression passed to the constructor
                var refTokenArray = new TokenArray(docModel, { temp: true });
                refTokenArray.parseFormula(grammar, refAddressFormula);

                // try to convert to a single reference address (do not pass a reference
                // sheet index, sheet name is expected to exist in the formula expression)
                var ranges = refTokenArray.resolveRangeList();
                refTokenArray.destroy();

                // if the expression resolves to a single cell address, use it to relocate the formula
                if ((ranges.length === 1) && ranges.first().single()) {
                    tokenArray.relocateRanges(ranges.first().start, Address.A1);
                }
            }

            return this;
        };

        /**
         * Returns whether the defined name should be hidden in the user
         * interface.
         *
         * @returns {Boolean}
         *  Whether the defined name should be hidden in the user interface.
         */
        this.isHidden = function () {
            return hidden;
        };

        // initialization -----------------------------------------------------

        // parse the initial formula definition
        this.setFormula(initGrammar, initFormula, Utils.getStringOption(initOptions, 'refAddress'));

        // destroy all class members on destruction
        this.registerDestructor(function () {
            tokenArray.destroy();
            parentModel = docModel = sheetModel = tokenArray = null;
        });

    } }); // class NameModel

    // class NameCollection ===================================================

    /**
     * Collects all global defined names of the spreadsheet document, or the
     * defined names contained in a single sheet of the document.
     *
     * Triggers the following events:
     * - 'insert:name'
     *      After a new defined name has been inserted into the collection.
     *      Event handlers receive the label of the new defined name.
     * - 'change:name'
     *      After the label, or the formula definition of a defined name has
     *      been modified. Event handlers receive the (new) label of the
     *      changed defined name.
     * - 'delete:name'
     *      After a defined name has been deleted from the collection. Event
     *      handlers receive the label of the deleted defined name.
     *
     * @constructor
     *
     * @extends ModelObject
     *
     * @param {SpreadsheetModel|SheetModel} parentModel
     *  The spreadsheet document model (for globally defined names), or the
     *  sheet model (for sheet-local names) containing this instance.
     */
    function NameCollection(parentModel) {

        var // the document model
            docModel = parentModel.getDocModel(),

            // the models of all defined names, mapped by upper-case name
            nameModels = {};

        // base constructor ---------------------------------------------------

        ModelObject.call(this, parentModel.getDocModel());

        // private methods ----------------------------------------------------

        /**
         * Returns a descriptor for an existing name model addressed by the
         * passed operation context.
         *
         * @param {SheetOperationContext} context
         *  A wrapper representing a document operation for defined names.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.insert=false]
         *      If set to true, the name model addressed by the operation must
         *      not exist in this name collection. By default, the name model
         *      must exist in the collection.
         *
         * @returns {Object}
         *  A descriptor for the name model addressed by the passed operation,
         *  with the following properties:
         *  - {String} label
         *      The original label as contained in the passed operation.
         *  - {String} key
         *      The (upper-case) label used as map key for the name model.
         *  - {NameModel|Null} model
         *      The name model addressed by the operation; or null, if the name
         *      does not exist (see option 'insert' above).
         *
         * @throws {OperationException}
         *  If the operation does not address an existing name model, or if it
         *  addresses an existing name model that must not exist (see option
         *  'insert' above).
         */
        function getModelData(context, options) {

            var // get the label from the operation
                label = context.getStr('exprName'),
                // the upper-case label is used as map key
                nameKey = label.toUpperCase(),
                // get the name model
                nameModel = nameModels[nameKey] || null;

            // check existence/absence of the name model
            if (Utils.getBooleanOption(options, 'insert', false)) {
                context.ensure(!nameModel, 'defined name exists');
            } else {
                context.ensure(nameModel, 'missing defined name');
            }

            return { label: label, key: nameKey, model: nameModel };
        }

        /**
         * Validates the passed label for a defined name. Returns a rejected
         * promise if the label is not valid.
         *
         * @param {String} label
         *  The label for a defined name to be checked.
         *
         * @returns {jQuery.Promise|Null}
         *  A rejected promise with a specific error code, if the label is not
         *  valid (see method GrammarConfig.validateNameLabel() for a list of
         *  error codes); or the value null, if the passed label is valid.
         */
        function validateNameLabel(label) {
            var labelError = docModel.getGrammarConfig('ui').validateNameLabel(docModel, label);
            return labelError ? SheetUtils.makeRejected(labelError) : null;
        }

        /**
         * Creates the operation properties for the passed formula expression,
         * intended to be used in the document operations 'insertName', and
         * 'changeName'. The formula expression will be checked for syntax and
         * semantic errors, and it will be relocated for usage in document
         * operations.
         *
         * @param {Object} formulaDesc
         *  The descriptor of the formula expression. See public method
         *  NameCollection.insertName() for details about the properties.
         *
         * @returns {Object|Null}
         *  On success, an object with the property 'formula' containing the
         *  relocated formula expression, the property 'parse' with the current
         *  locale code, and an optional property 'ref' with the passed
         *  reference address (as formula expression) for ODF documents.
         */
        function getFormulaProperties(formulaDesc) {

            var // a temporary token array for the passed formula, and for the reference address (TODO: sheet-local names)
                tokenArray = new TokenArray(parentModel, { type: 'name', temp: true }),
                // whether the current file format is OOXML
                ooxml = docModel.getApp().isOOXML(),
                // the resulting operation properties
                properties = null;

            // parse the formula; bug 40293: add sheet name to references without sheet (OOXML only)
            tokenArray.parseFormula(formulaDesc.grammar, formulaDesc.formula, ooxml ? { extendSheet: formulaDesc.refSheet } : null);

            // interpret the formula to find syntax errors or semantic errors
            var formulaResult = tokenArray.interpretFormula('any');
            if (formulaResult.type !== 'error') {

                // TODO (bug 40438): Currently, the CalcEngine EXPECTS translated formula expressions, even
                // without 'parse' property. For now, always pass translated formulas and the parse property.

                // OOXML: relocate formula expression from reference cell to A1; ODF: pass reference cell with operation
                if (ooxml) {
                    properties = { formula: tokenArray.getFormula('ui', { refAddress: formulaDesc.refAddress, targetAdddress: Address.A1 }) };
                } else {
                    properties = { formula: tokenArray.getFormula('ui') };
                    // store reference cell as formula expression with sheet name
                    tokenArray.clear().appendRange(new Range(formulaDesc.refAddress), { sheet: formulaDesc.refSheet, abs: true });
                    properties.ref = tokenArray.getFormula('op');
                }

                // TODO (bug 40438): remove the parse property
                properties.parse = LocaleData.LOCALE;
            }

            tokenArray.destroy();
            return properties;
        }

        // protected methods --------------------------------------------------

        /**
         * Returns the internal contents of this collection, needed for cloning
         * into another collection.
         *
         * @internal
         *  Used during clone construction. DO NOT CALL from external code!
         *
         * @returns {Object}
         *  The internal contents of this collection.
         */
        this.getCloneData = function () {
            return { nameModels: nameModels };
        };

        /**
         * Clones all contents from the passed collection into this collection.
         *
         * @internal
         *  Used during clone construction. DO NOT CALL from external code!
         *
         * @param {NameCollection} collection
         *  The source collection whose contents will be cloned into this
         *  collection.
         *
         * @returns {NameCollection}
         *  A reference to this instance.
         */
        this.cloneFrom = function (collection) {

            // the internal contents of the source collection
            var cloneData = collection.getCloneData();

            // clone the name models of the source collection
            _.each(cloneData.nameModels, function (sourceModel, nameKey) {
                nameModels[nameKey] = sourceModel.clone(parentModel);
            });

            return this;
        };

        /**
         * Callback handler for the document operation 'insertName'. Creates
         * and stores a new defined name, and triggers an 'insert:name' event.
         *
         * @param {SheetOperationContext} context
         *  A wrapper representing the 'insertName' document operation.
         */
        this.applyInsertOperation = function (context) {

            // get name and map key from operation (name must not exist, throws on error)
            var modelData = getModelData(context, { insert: true });

            // create a new name model (formula expression may be empty)
            nameModels[modelData.key] = new NameModel(parentModel, modelData.label, 'op', context.getStr('formula', true), {
                refAddress: context.getOptStr('ref'),
                hidden: context.getOptBool('hidden')
            });

            // notify listeners
            this.trigger('insert:name', modelData.key);
        };

        /**
         * Callback handler for the document operation 'deleteName'. Deletes an
         * existing defined name from this collection, and triggers a
         * 'delete:name' event.
         *
         * @param {SheetOperationContext} context
         *  A wrapper representing the 'deleteName' document operation.
         */
        this.applyDeleteOperation = function (context) {

            // resolve the name model addressed by the operation (throws on error)
            var modelData = getModelData(context);

            // delete the name model and notify listeners
            modelData.model.destroy();
            delete nameModels[modelData.key];
            this.trigger('delete:name', modelData.key);
        };

        /**
         * Callback handler for the document operation 'changeName'. Changes
         * the label or formula definition of an existing defined name, and
         * triggers a 'change:name' event.
         *
         * @param {SheetOperationContext} context
         *  A wrapper representing the 'changeName' document operation.
         */
        this.applyChangeOperation = function (context) {

            // resolve the name model addressed by the operation (throws on error)
            var modelData = getModelData(context);

            // set a new name for the model if specified
            if (context.has('newName')) {

                var // get new label (will be checked to be non-empty)
                    newLabel = context.getStr('newName'),
                    // the upper-case name is used as map key
                    newKey = newLabel.toUpperCase();

                // check that the passed new name does not exist yet (but: allow to change case of the name)
                context.ensure((modelData.key === newKey) || !(newKey in nameModels), 'name exists');

                // set the new name, and remap the model in the collection
                modelData.model.setLabel(newLabel);
                delete nameModels[modelData.key];
                nameModels[newKey] = modelData.model;
                modelData.key = newKey;
            }

            // set a new formula expression for the model if specified
            if (context.has('formula')) {
                modelData.model.setFormula('op', context.getStr('formula', true), context.getOptStr('ref'));
            }

            // notify listeners
            this.trigger('change:name', modelData.key);
        };

        // public methods -----------------------------------------------------

        /**
         * Returns the models of all defined names contained in this collection
         * as an array.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.skipHidden=false]
         *      If set to true, hidden defined names will not be returned.
         *  @param {Boolean} [options.sort=false]
         *      If set to true, the defined names will be sorted by label.
         *
         * @returns {Array<NameModel>}
         *  The models of all defined names in this collection.
         */
        this.getNameModels = function (options) {

            // whether to skip hidden style sheets
            var skipHidden = Utils.getBooleanOption(options, 'skipHidden', false);

            // get all name models in an array
            var resultModels = _.filter(nameModels, function (nameModel) {
                return !skipHidden || !nameModel.isHidden();
            });

            // sort the array if specified
            if (Utils.getBooleanOption(options, 'sort', false)) {
                resultModels.sort(function (nameModel1, nameModel2) {
                    return nameModel1.getLabel().localeCompare(nameModel2.getLabel());
                });
            }

            return resultModels;
        };

        /**
         * Returns whether this collection contains a defined name with the
         * passed label.
         *
         * @param {String} label
         *  The (case-insensitive) label of a defined name.
         *
         * @returns {Boolean}
         *  Whether this collection contains a defined name with the passed
         *  label.
         */
        this.hasNameModel = function (label) {
            return label.toUpperCase() in nameModels;
        };

        /**
         * Returns the model of the specified defined name.
         *
         * @param {String} label
         *  The (case-insensitive) label of the defined name to be returned.
         *
         * @returns {NameModel|Null}
         *  The model of the specified defined name, if existing; otherwise
         *  null.
         */
        this.getNameModel = function (label) {
            var nameModel = nameModels[label.toUpperCase()];
            return nameModel ? nameModel : null;
        };

        // operation generators -----------------------------------------------

        /**
         * Creates and applies an 'insertName' document operation to insert a
         * new defined name into this name collection.
         *
         * @param {String} label
         *  The exact label for the new defined name.
         *
         * @param {Object} formulaDesc
         *  The descriptor of the formula expression to be bound to the new
         *  defined name. Must contain the following properties:
         *  @param {String} formulaDesc.grammar
         *      The identifier of the formula grammar to be used to parse the
         *      formula expression. See class GrammarConfig for details.
         *  @param {String} formulaDesc.formula
         *      The formula expression to be bound to the new defined name.
         *  @param {Number} formulaDesc.refSheet
         *      The index of the reference sheet the formula is related to.
         *  @param {Address} formulaDesc.refAddress
         *      The address of the reference cell used to interpret relative
         *      cell references in the formula expression.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected with an object
         *  with 'cause' property set to one of the following error codes:
         *  - 'name:used': A defined name with the passed label exists already.
         *  - 'name:empty': The specified label is empty.
         *  - 'name:invalid': The specified label contains invalid characters.
         *  - 'name:address': The specified label would be valid, but conflicts
         *      with the representation of a relative cell reference in A1
         *      notation, or a cell reference in R1C1 notation (either English,
         *      e.g. 'R1C1', or according to current UI language, e.g. 'Z1S1'
         *      in German).
         *  - 'formula:invalid': The passed formula expression is invalid.
         *  - 'operation': Internal error while applying the operation.
         */
        this.insertName = function (label, formulaDesc) {

            // check that the defined name does not exist yet
            if (this.hasNameModel(label)) { return SheetUtils.makeRejected('name:used'); }

            // the label must be valid
            var labelError = validateNameLabel(label);
            if (labelError) { return labelError; }

            // parse and validate the formula expression
            var properties = getFormulaProperties(formulaDesc);
            if (!properties) { return SheetUtils.makeRejected('formula:invalid'); }

            // generate and apply the 'insertName' operation (if the parent model is a
            // sheet model, it will automatically add the property 'sheet' to the operation)
            properties.exprName = label;
            return parentModel.createAndApplyOperations(function (generator) {
                generator.generateOperation(Operations.INSERT_NAME, properties);
            });
        };

        /**
         * Creates and applies a 'changeName' document operation to change the
         * label or formula definition of an existing defined name.
         *
         * @param {String} label
         *  The label of the defined name to be changed.
         *
         * @param {String|Null} newLabel
         *  The new label for the defined name. If set to null, the label of
         *  the defined name will not be changed.
         *
         * @param {Object|Null} formulaDesc
         *  The descriptor of the formula expression to be bound to the defined
         *  name. If omitted, the formula expression of the defined name will
         *  not be changed (an invalid formula expression in the defined name
         *  will be retained). See method NameCollection.insertName() for
         *  details about the expected properties.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected with an object
         *  with 'cause' property set to one of the following error codes:
         *  - 'name:used': A defined name with the new label exists already.
         *  - 'name:empty': The new label is empty.
         *  - 'name:invalid': The new label contains invalid characters.
         *  - 'name:address': The new label would be valid, but conflicts with
         *      the representation of a relative cell reference in A1 notation,
         *      or a cell reference in R1C1 notation (either English, e.g.
         *      'R1C1', or according to current UI language, e.g. 'Z1S1' in
         *      German).
         *  - 'formula:invalid': The passed formula expression is invalid.
         *  - 'operation': Internal error while applying the operation.
         */
        this.changeName = function (label, newLabel, formulaDesc) {

            // check that the name exists
            if (!this.hasNameModel(label)) { return $.Deferred().reject(); }

            // the properties for the 'changeName' operation
            var properties = {};

            // check the passed new label (ignore unchanged existing invalid labels, e.g. from import filter)
            if (_.isString(newLabel) && (label !== newLabel)) {

                // new label must not exist yet in the document (but allow to change case of an existing label)
                if ((label.toUpperCase() !== newLabel.toUpperCase()) && this.hasNameModel(newLabel)) {
                    return SheetUtils.makeRejected('name:used');
                }

                // the label must be valid
                var labelError = validateNameLabel(newLabel);
                if (labelError) { return labelError; }

                // add the new label to the operation properties
                properties.newName = newLabel;
            }

            // parse and validate the formula expression
            if (_.isObject(formulaDesc)) {
                _.extend(properties, getFormulaProperties(formulaDesc));
                if (!('formula' in properties)) { return SheetUtils.makeRejected('formula:invalid'); }
            }

            // nothing to do without any changed properties
            if (_.isEmpty(properties)) { return $.when(); }

            // generate and apply the 'changeName' operation (if the parent model is a
            // sheet model, it will automatically add the property 'sheet' to the operation)
            properties.exprName = label;
            return parentModel.createAndApplyOperations(function (generator) {
                generator.generateOperation(Operations.CHANGE_NAME, properties);
            });
        };

        /**
         * Creates and applies a 'deleteName' document operation to delete a
         * defined name from this name collection.
         *
         * @param {String} label
         *  The label of the defined name to be deleted.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected with an object
         *  with 'cause' property set to one of the following error codes:
         *  - 'operation': Internal error while applying the operation.
         */
        this.deleteName = function (label) {

            // check that the name exists
            if (!this.hasNameModel(label)) { return $.Deferred().reject(); }

            // generate and apply the 'deleteName' operation (if the parent model is a
            // sheet model, it will automatically add the property 'sheet' to the operation)
            return parentModel.createAndApplyOperations(function (generator) {
                generator.generateOperation(Operations.DELETE_NAME, { exprName: label });
            });
        };
        // initialization -----------------------------------------------------

        // destroy all class members on destruction
        this.registerDestructor(function () {
            _.invoke(nameModels, 'destroy');
            parentModel = docModel = nameModels = null;
        });

    } // class NameCollection

    // exports ================================================================

    // derive this class from class ModelObject
    return ModelObject.extend({ constructor: NameCollection });

});
