/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/formula/tokens', [
    'io.ox/office/tk/utils',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/model/formula/formulautils',
    'io.ox/office/spreadsheet/model/formula/cellref'
], function (Utils, SheetUtils, FormulaUtils, CellRef) {

    'use strict';

    // convenience shortcuts
    var Address = SheetUtils.Address,
        Range = SheetUtils.Range,
        Range3D = SheetUtils.Range3D;

    // private global functions ===============================================

    /**
     * Returns whether the passed value is an existing sheet reference that
     * represents a reference error.
     */
    function isSheetError(sheetRef) {
        return _.isObject(sheetRef) && !sheetRef.valid();
    }

    /**
     * Returns whether the passed sheet reference may be affected by a renamed
     * sheet (always true for relative sheet references).
     */
    function isBoundSheet(sheetRef, sheet) {
        return _.isObject(sheetRef) && _.isNumber(sheetRef.sheet) && (!sheetRef.abs || (sheetRef.sheet === sheet));
    }

    // class BaseToken ========================================================

    /**
     * Base class for all formula tokens used in the tokenizer.
     *
     * @constructor
     *
     * @param {String} type
     *  The token type identifier.
     *
     * @param {Function} generator
     *  A callback function that implements converting the current value of
     *  this token to its text representation according to a specific formula
     *  grammar. Receives all parameters passed to the public method
     *  BaseToken.getText(), and must return a string. Will be invoked in the
     *  context of this token instance.
     *
     * @param {Function} toString
     *  A callback function that implements converting the current value of
     *  this token to its text representation for debug logging. Does not
     *  receive any parameter, and must return a string
     */
    var BaseToken = _.makeExtendable(function (type) {

        // private properties
        this._type = type;

    }); // class BaseToken

    // public methods ---------------------------------------------------------

    /**
     * Returns the type identifier of this token.
     *
     * @returns {String}
     *  The type identifier of this token.
     */
    BaseToken.prototype.getType = function () {
        return this._type;
    };

    /**
     * Returns whether the type of this token matches the passed type.
     *
     * @param {String} test
     *  A type identifier that must match the type of this token exactly.
     *
     * @returns {Boolean}
     *  Whether the type of this token matches the passed type.
     */
    BaseToken.prototype.isType = function (test) {
        return test === this._type;
    };

    /**
     * Returns whether the type identifier of this token matches the passed
     * regular expression.
     *
     * @param {RegExp} regExp
     *  A regular expression that will be tested against the type of this
     *  token.
     *
     * @returns {Boolean}
     *  Whether the type of this token matches the regular expression.
     */
    BaseToken.prototype.matchesType = function (regExp) {
        return regExp.test(this._type);
    };

    /**
     * All sub classes MUST overwrite this dummy method to generate the text
     * representation of this token, according to the passed grammar
     * configuration.
     *
     * @param {GrammarConfig} config
     *  The configuration object containing grammar-specific settings for
     *  formula operators.
     *
     * @param {Object} [options]
     *  Optional parameters for relocation of range addresses.
     *
     * @returns {String}
     *  The text representation of this token.
     */
    BaseToken.prototype.getText = function (/*config, options*/) {
        Utils.error('BaseToken.getText(): missing implementation');
        return '';
    };

    /**
     * All sub classes MUST overwrite this dummy method to generate a text
     * description of the contents of this token for debugging (!) purposes.
     *
     * @returns {String}
     *  The debug text representation of the contents of this token.
     */
    BaseToken.prototype.debugText = function () {
        Utils.error('BaseToken.debugText(): missing implementation');
        return '';
    };

    /**
     * Generates a text description of this token for debugging (!) purposes.
     *
     * @returns {String}
     *  The complete debug text representation of this token.
     */
    BaseToken.prototype.toString = function () {
        var txt = this.debugText();
        return this._type + (txt ? ('[' + txt + ']') : '');
    };

    // class FixedToken =======================================================

    /**
     * A formula token of an arbitrary type with a fixed string representation
     * independent from a formula grammar.
     *
     * @constructor
     *
     * @extends BaseToken
     *
     * @param {String} type
     *  The token type identifier.
     *
     * @param {String} text
     *  The fixed text representation of the token.
     */
    var FixedToken = BaseToken.extend({ constructor: function (type, text) {

        // base constructor
        BaseToken.call(this, type);

        // private properties
        this._text = text;

    } }); // class FixedToken

    // public methods ---------------------------------------------------------

    /**
     * Returns the current text of this token.
     *
     * @returns {String}
     *  The current text of this token.
     */
    FixedToken.prototype.getValue = function () {
        return this._text;
    };

    /**
     * Replaces the text of this token.
     *
     * @param {String} newText
     *  The new text to be inserted into this token.
     *
     * @returns {Boolean}
     *  Whether the text has actually changed.
     */
    FixedToken.prototype.setValue = function (newText) {
        if (this._text === newText) { return false; }
        this._text = newText;
        return true;
    };

    /**
     * Appends more text to this token.
     *
     * @param {String} newText
     *  The new text to be appended to this token.
     *
     * @returns {Boolean}
     *  Whether the token has actually changed (the passed text was not empty).
     */
    FixedToken.prototype.appendValue = function (newText) {
        if (newText.length === 0) { return false; }
        this._text += newText;
        return true;
    };

    /**
     * Generates the text representation of this token.
     */
    FixedToken.prototype.getText = function () {
        return this._text;
    };

    /**
     * Generates a text description of this token for debugging (!) purposes.
     */
    FixedToken.prototype.debugText = function () {
        return this._text;
    };

    // class OperatorToken ====================================================

    /**
     * A formula token that represents a unary or binary operator in a formula.
     *
     * @constructor
     *
     * @extends BaseToken
     *
     * @param {String} value
     *  The grammar-independent operator identifier.
     */
    var OperatorToken = BaseToken.extend({ constructor: function (value) {

        // base constructor
        BaseToken.call(this, 'op');

        // private properties
        this._value = value;

    } }); // class OperatorToken

    // public methods ---------------------------------------------------------

    /**
     * Returns the grammar-independent identifier of the operator.
     *
     * @returns {String}
     *  The grammar-independent identifier of the operator.
     */
    OperatorToken.prototype.getValue = function () {
        return this._value;
    };

    /**
     * Generates the text representation of this token.
     */
    OperatorToken.prototype.getText = function (config) {
        return config.getOperatorName(this._value) || '?';
    };

    /**
     * Generates a text description of this token for debugging (!) purposes.
     */
    OperatorToken.prototype.debugText = function () {
        return this._value;
    };

    // class SeparatorToken ===================================================

    /**
     * A formula token that represents the range list operator, or a separator
     * in a function parameter list.
     *
     * @constructor
     *
     * @extends BaseToken
     */
    var SeparatorToken = BaseToken.extend({ constructor: function () {

        // base constructor
        BaseToken.call(this, 'sep');

    } }); // class SeparatorToken

    // public methods ---------------------------------------------------------

    /**
     * Generates the text representation of this token.
     */
    SeparatorToken.prototype.getText = function (config) { return config.SEP; };

    /**
     * Generates a text description of this token for debugging (!) purposes.
     */
    SeparatorToken.prototype.debugText = _.constant('');

    // class ParenthesisToken =================================================

    /**
     * A formula token that represents an opening or a closing parenthesis.
     *
     * @constructor
     *
     * @extends FixedToken
     *
     * @param {Boolean} open
     *  Whether this token represents an opening parenthesis (true), or a
     *  closing parenthesis (false).
     */
    var ParenthesisToken = FixedToken.extend({ constructor: function (open) {

        // base constructor
        FixedToken.call(this, open ? 'open' : 'close', open ? '(' : ')');

    } }); // class ParenthesisToken

    /**
     * Generates a text description of this token for debugging (!) purposes.
     */
    ParenthesisToken.prototype.debugText = _.constant('');

    // class MatrixDelimiterToken =============================================

    /**
     * A formula token that represents a delimiter in a matrix literal: an
     * opening or a closing parenthesis, a row separator, or a column
     * separator.
     *
     * @constructor
     *
     * @extends BaseToken
     *
     * @param {Boolean} type
     *  The type of the matrix delimiter. Must be one of 'mat_open',
     *  'mat_close', 'mat_row', or 'mat_col'.
     */
    var MatrixDelimiterToken = FixedToken.extend({ constructor: function (type) {

        // base constructor
        BaseToken.call(this, type);

    } }); // class MatrixDelimiterToken

    // public methods ---------------------------------------------------------

    /**
     * Generates the text representation of this token.
     */
    MatrixDelimiterToken.prototype.getText = function (config) {
        switch (this.getType()) {
            case 'mat_open': return config.MAT_OPEN;
            case 'mat_close': return config.MAT_CLOSE;
            case 'mat_row': return config.MAT_ROW;
            case 'mat_col': return config.MAT_COL;
        }
        Utils.error('MatrixDelimiterToken.getText(): invalid matrix delimiter type');
        return '';
    };

    /**
     * Generates a text description of this token for debugging (!) purposes.
     */
    MatrixDelimiterToken.prototype.debugText = _.constant('');

    // class LiteralToken =====================================================

    /**
     * A formula token that represents a literal value (a number, a string, a
     * boolean value, or an error code) in a formula.
     *
     * @constructor
     *
     * @extends BaseToken
     *
     * @param {Number|String|Boolean|ErrorCode|Null} value
     *  The value for this literal token. The special value null is reserved
     *  for empty parameter in a function call, e.g. in the formula =SUM(1,,2).
     */
    var LiteralToken = BaseToken.extend({ constructor: function (value) {

        // base constructor
        BaseToken.call(this, 'lit');

        // private properties
        this._value = value;

    } }); // class LiteralToken

    // public methods ---------------------------------------------------------

    /**
     * Returns the current value of this literal token.
     *
     * @returns {Number|String|Boolean|ErrorCode|Null}
     *  The current value of this literal token.
     */
    LiteralToken.prototype.getValue = function () {
        return this._value;
    };

    /**
     * Changes the current value of this token.
     *
     * @param {Number|String|Boolean|ErrorCode|Null} newValue
     *  The new value for this token.
     *
     * @returns {Boolean}
     *  Whether the value of the token has actually been changed.
     */
    LiteralToken.prototype.setValue = function (newValue) {
        if (this._value === newValue) { return false; }
        this._value = newValue;
        return true;
    };

    /**
     * Generates the text representation of this token.
     */
    LiteralToken.prototype.getText = function (config) {
        return config.formatScalar(this._value);
    };

    /**
     * Generates the text representation for debug logging.
     */
    LiteralToken.prototype.debugText = function () {
        return FormulaUtils.valueToString(this._value);
    };

    // class MatrixToken ======================================================

    /**
     * A formula token that represents a constant matrix literal in a formula.
     *
     * @constructor
     *
     * @extends BaseToken
     *
     * @param {Matrix} matrix
     *  The matrix literal represented by the token.
     */
    var MatrixToken = BaseToken.extend({ constructor: function (matrix) {

        // base constructor
        BaseToken.call(this, 'mat');

        // private properties
        this._matrix = matrix;

    } }); // class MatrixToken

    // public methods ---------------------------------------------------------

    /**
     * Returns the matrix literal contained in this token.
     *
     * @returns {Matrix}
     *  The matrix literal of this token.
     */
    MatrixToken.prototype.getMatrix = function () {
        return this._matrix;
    };

    /**
     * Generates the text representation of this token.
     */
    MatrixToken.prototype.getText = function (config) {
        return config.MAT_OPEN + this._matrix.array.map(function (elems) {
            return elems.map(function (elem) {
                return config.formatScalar(elem);
            }).join(config.MAT_COL);
        }).join(config.MAT_ROW) + config.MAT_CLOSE;
    };

    /**
     * Generates the text representation for debug logging.
     */
    MatrixToken.prototype.debugText = function () {
        return this._matrix.toString();
    };

    // class FunctionToken ====================================================

    /**
     * A formula token that represents a built-in function in a formula.
     *
     * @constructor
     *
     * @extends BaseToken
     *
     * @param {String} funcKey
     *  The unique resource key of the built-in sheet function.
     */
    var FunctionToken = BaseToken.extend({ constructor: function (funcKey) {

        // base constructor
        BaseToken.call(this, 'func');

        // private properties
        this._value = funcKey;

    } }); // class FunctionToken

    // public methods ---------------------------------------------------------

    /**
     * Returns the native name of the function, as passed to the constructor.
     *
     * @returns {String}
     *  The native name of the function.
     */
    FunctionToken.prototype.getValue = function () {
        return this._value;
    };

    /**
     * Generates the text representation of this token.
     */
    FunctionToken.prototype.getText = function (config) {
        return config.getFunctionName(this._value) || this._value.toUpperCase();
    };

    /**
     * Returns a text description of this token for debug logging.
     */
    FunctionToken.prototype.debugText = function () {
        return this._value;
    };

    // class SheetRefToken ====================================================

    /**
     * Base class for formula tokens containing one or two sheet references.
     *
     * @constructor
     *
     * @extends BaseToken
     *
     * @param {Boolean} type
     *  The type of the token.
     *
     * @param {SpreadsheetModel} docModel
     *  The spreadsheet document model. Needed to resolve sheet names.
     *
     * @param {SheetRef|Null} sheet1Ref
     *  The first sheet reference structure (e.g. 'Sheet1' in the expression
     *  Sheet1:Sheet2!A1). If null or undefined, the token represents a local
     *  cell range reference into the passed sheet (e.g. A1:B2).
     *
     * @param {SheetRef|Null} sheet2Ref
     *  The second sheet reference structure (e.g. 'Sheet2' in the expression
     *  Sheet1:Sheet2!A1). If null or undefined, the token represents a single
     *  sheet reference (e.g. Sheet1!A1).
     */
    var SheetRefToken = BaseToken.extend({ constructor: function (type, docModel, sheet1Ref, sheet2Ref) {

        // base constructor
        BaseToken.call(this, type);

        // private properties
        this._docModel = docModel;
        this._sheet1Ref = sheet1Ref;
        this._sheet2Ref = sheet2Ref;

    } }); // class SheetRefToken

    // protected methods ------------------------------------------------------

    /**
     * Returns the sheet range referred by this token.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {Number} [options.refSheet]
     *      The index of the reference sheet used to resolve references without
     *      sheets. If omitted, and this token does not contain its own sheet
     *      reference, this method will return null.
     *
     * @returns {Object|Null}
     *  The range of sheets referred by this token, in the properties 'sheet1'
     *  and 'sheet2'. If the token does not contain a valid sheet range
     *  (according to the passed reference sheet), null will be returned.
     */
    SheetRefToken.prototype._resolveSheetRange = function (options) {

        var sheet1 = this._sheet1Ref ? this._sheet1Ref.sheet : Utils.getIntegerOption(options, 'refSheet', null),
            sheet2 = this._sheet2Ref ? this._sheet2Ref.sheet : sheet1;

        return (_.isNumber(sheet1) && (sheet1 >= 0) && _.isNumber(sheet2) && (sheet2 >= 0)) ?
            { sheet1: Math.min(sheet1, sheet2), sheet2: Math.max(sheet1, sheet2) } : null;
    };

    // public methods ---------------------------------------------------------

    /**
     * Returns whether this token contains a sheet reference, regardless if it
     * is valid.
     *
     * @returns {Boolean}
     *  Whether this token contains a sheet reference.
     */
    SheetRefToken.prototype.hasSheetRef = function () {
        return _.isObject(this._sheet1Ref);
    };

    /**
     * Returns whether a sheet reference of this token is invalid (it exists,
     * AND it points to a non-existing sheet).
     *
     * @returns {Boolean}
     *  Whether a sheet reference of this token is invalid.
     */
    SheetRefToken.prototype.hasSheetError = function () {
        return isSheetError(this._sheet1Ref) || isSheetError(this._sheet2Ref);
    };

    /**
     * Refreshes the sheet references after a sheet has been inserted, deleted,
     * or moved in the document.
     *
     * @returns {Boolean}
     *  Whether the token has been changed.
     */
    SheetRefToken.prototype.transformSheet = function (toSheet, fromSheet) {
        var changed1 = _.isObject(this._sheet1Ref) && this._sheet1Ref.transform(toSheet, fromSheet),
            changed2 = _.isObject(this._sheet2Ref) && this._sheet2Ref.transform(toSheet, fromSheet);
        return changed1 || changed2;
    };

    /**
     * Changes all sheet references containing the specified old sheet index to
     * the new sheet index. Used for example when copying existing sheets, and
     * adjusting all references and defined names that point to the original
     * sheet.
     *
     * @param {Number} oldSheet
     *  The zero-based index of the original sheet.
     *
     * @param {Number} newSheet
     *  The zero-based index of the target sheet.
     *
     * @returns {Boolean}
     *  Whether the token has been changed.
     */
    SheetRefToken.prototype.relocateSheet = function (oldSheet, newSheet) {
        var changed1 = _.isObject(this._sheet1Ref) && this._sheet1Ref.relocate(oldSheet, newSheet),
            changed2 = _.isObject(this._sheet2Ref) && this._sheet2Ref.relocate(oldSheet, newSheet);
        return changed1 || changed2;
    };

    /**
     * Refreshes the text representation after a sheet has been renamed in the
     * document.
     *
     * @param {Number} sheet
     *  The zero-based index of the renamed sheet.
     *
     * @returns {Boolean}
     *  Whether the token has been changed.
     */
    SheetRefToken.prototype.renameSheet = function (sheet) {
        // the token does not contain the sheet name anymore (no need to modify something),
        // but it still returns true if it is actually affected (for event handlers)
        return isBoundSheet(this._sheet1Ref, sheet) || isBoundSheet(this._sheet2Ref, sheet);
    };

    /**
     * Returns a text description of the sheet names for debug logging.
     */
    SheetRefToken.prototype.debugSheetPrefix = function () {

        // convert first sheet name
        if (!this._sheet1Ref) { return ''; }
        var sheetName = this._sheet1Ref.toString();

        // convert second sheet name
        if (this._sheet2Ref && !this._sheet1Ref.equals(this._sheet2Ref)) {
            sheetName += ':' + this._sheet2Ref.toString();
        }

        return sheetName + '!';
    };

    // class ReferenceToken ===================================================

    /**
     * A formula token that represents a cell range reference in a formula.
     *
     * @constructor
     *
     * @extends SheetRefToken
     *
     * @param {SpreadsheetModel} docModel
     *  The spreadsheet document model containing this token instance.
     *
     * @param {CellRef|Null} cell1Ref
     *  The first cell reference structure (e.g. the cell address 'A1' in the
     *  formula A1:B2). If null or omitted, the token will represent a
     *  reference error (e.g. Sheet1!#REF!, after deleting the referenced
     *  column or row).
     *
     * @param {CellRef|Null} cell2Ref
     *  The second cell reference structure (e.g. the cell address 'B2' in the
     *  formula A1:B2). If null or omitted, the token will represent a single
     *  cell (e.g. Sheet1!A1).
     *
     * @param {SheetRef|Null} sheet1Ref
     *  The first sheet reference structure (e.g. 'Sheet1' in the formula
     *  Sheet1:Sheet2!A1). If null or undefined, the token represents a local
     *  cell range reference into the passed sheet (e.g. A1:B2).
     *
     * @param {SheetRef|Null} sheet2Ref
     *  The second sheet reference structure (e.g. 'Sheet2' in the formula
     *  Sheet1:Sheet2!A1). If null or undefined, the token represents a single
     *  sheet reference (e.g. Sheet1!A1).
     */
    var ReferenceToken = SheetRefToken.extend({ constructor: function (docModel, cell1Ref, cell2Ref, sheet1Ref, sheet2Ref) {

        // base constructor
        SheetRefToken.call(this, 'ref', docModel, sheet1Ref, sheet2Ref);

        // private properties
        this._cell1Ref = cell1Ref;
        this._cell2Ref = cell2Ref;

    } }); // class ReferenceToken

    // private methods --------------------------------------------------------

    /**
     * Returns copies of the own cell reference objects that have been
     * relocated according to the passed settings.
     *
     * @returns {Object}
     *  The resulting cell references, in the properties 'r1' and 'r2'.
     */
    ReferenceToken.prototype._resolveCellRefs = function (options) {

        var // reference and target address for relocation
            refAddress = Utils.getOption(options, 'refAddress', Address.A1),
            targetAddress = Utils.getOption(options, 'targetAddress', Address.A1);

        // performance: return original cell references, if nothing to relocate
        if (refAddress.equals(targetAddress)) { return { r1: this._cell1Ref, r2: this._cell2Ref }; }

        var // the result object returned by this method: start with clones of the cell references
            cellRefs = {
                r1: this._cell1Ref ? this._cell1Ref.clone() : null,
                r2: this._cell2Ref ? this._cell2Ref.clone() : null
            },
            // whether to wrap the references at sheet borders
            wrapReferences = Utils.getBooleanOption(options, 'wrapReferences', false);

        // if relocating first reference is not successful, remove it from the result
        if (cellRefs.r1 && !cellRefs.r1.relocate(this._docModel, refAddress, targetAddress, wrapReferences)) {
            cellRefs.r1 = null;
        }

        // if relocating second reference is not successful, remove both cell references from the result
        if (cellRefs.r1 && cellRefs.r2 && !cellRefs.r2.relocate(this._docModel, refAddress, targetAddress, wrapReferences)) {
            cellRefs.r1 = cellRefs.r2 = null;
        }

        return cellRefs;
    };

    // public methods ---------------------------------------------------------

    /**
     * Returns the address of the cell range represented by this token,
     * including sheet indexes, or null if the range address is invalid.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {Number} [options.refSheet]
     *      The index of the reference sheet used to resolve missing sheet
     *      references. If omitted, and this token does not contain its own
     *      sheet reference, this method will return null.
     *  @param {Address} [options.refAddress]
     *      The address of the original reference cell this token is related
     *      to. If omitted, cell A1 will be used instead. If this address is
     *      different to the address resulting from the option 'targetAddress',
     *      the returned range will be relocated according to these two
     *      addresses.
     *  @param {Address} [options.targetAddress]
     *      The address of the target cell for relocation. If omitted, cell A1
     *      will be used instead. If this address is different to the address
     *      resulting from the option 'refAddress', the returned range will be
     *      relocated according to these two addresses.
     *  @param {Boolean} [options.wrapReferences=false]
     *      If set to true, and if the relocated range is located outside the
     *      sheet, the range will be wrapped at the sheet borders.
     *
     * @returns {Range3D|Null}
     *  The address of the cell range referenced by this reference token; or
     *  null, if the token does not refer to a valid range.
     */
    ReferenceToken.prototype.getRange3D = function (options) {

        // resolve sheet references, return null on error
        var sheetRange = this._resolveSheetRange(options);
        if (!sheetRange) { return null; }

        // return immediately, if cell1Ref cannot be relocated
        var cellRefs = this._resolveCellRefs(options);
        if (!cellRefs.r1) { return null; }

        // create an intermediate range with ordered column/row indexes
        var range = Range.createFromAddresses(cellRefs.r1.toAddress(), cellRefs.r2 && cellRefs.r2.toAddress());
        return Range3D.createFromRange(range, sheetRange.sheet1, sheetRange.sheet2);
    };

    /**
     * Changes the current cell range of this token, keeps all absolute flags
     * and sheet references intact.
     *
     * @returns {Boolean}
     *  Whether the token has been changed.
     */
    ReferenceToken.prototype.setRange = function (range) {

        // whether the token changes (either start or end address)
        var changed = false;

        // put a clone of the start address into cell1Ref
        if (this._cell1Ref && this._cell1Ref.toAddress().differs(range.start)) {
            this._cell1Ref.col = range.start[0];
            this._cell1Ref.row = range.start[1];
            changed = true;
        } else if (!this._cell1Ref) {
            this._cell1Ref = CellRef.create(range.start, true, true);
            this._cell2Ref = null;
            changed = true;
        }

        // create missing cell2Ref (unless the passed range is a single cell),
        // start as a clone of cell1Ref to get the correct absolute flags
        if (!this._cell2Ref && !range.single()) {
            this._cell2Ref = this._cell1Ref.clone();
            changed = true;
        }

        // put a clone of the end address into cell2Ref
        if (this._cell2Ref && this._cell2Ref.toAddress().differs(range.end)) {
            this._cell2Ref.col = range.end[0];
            this._cell2Ref.row = range.end[1];
            // reset cell2Ref if address and absolute flags are equal to cell1Ref
            if (this._cell1Ref.equals(this._cell2Ref)) { this._cell2Ref = null; }
            changed = true;
        }

        return changed;
    };

    /**
     * Transforms this reference token after columns or rows have been inserted
     * into or deleted from a sheet in the document.
     *
     * @returns {Boolean}
     *  Whether the token has been changed.
     */
    ReferenceToken.prototype.transformRange = function (sheet, interval, insert, columns, refSheet) {

        // the original range, with sheet indexes
        var oldRange = this.getRange3D({ refSheet: refSheet });

        // nothing to do, if this token represents a reference error, or points
        // to another sheet or a sheet range, or sheet reference is invalid
        if (!oldRange || !oldRange.isSheet(sheet)) { return false; }

        // the transformed range address (transformRange() returns a 2D range)
        var newRange = this._docModel.transformRange(oldRange, interval, insert, columns);

        // nothing to do, if the range does not change (compare only the columns/rows, not the sheets)
        if (newRange && newRange.equals(oldRange)) { return false; }

        // check that the range has been transformed successfully
        if (newRange) {
            this.setRange(newRange);
        } else {
            this._cell1Ref = this._cell2Ref = null;
        }
        return true;
    };

    /**
     * Relocates this reference token to a new cell position.
     *
     * @returns {Boolean}
     *  Whether the token has been changed.
     */
    ReferenceToken.prototype.relocateRange = function (refAddress, targetAddress, wrapReferences) {

        // do not touch tokens representing a reference error
        if (!this._cell1Ref || refAddress.equals(targetAddress)) { return false; }

        // relocate first cell reference, bail out immediately on error
        if (!this._cell1Ref.relocate(this._docModel, refAddress, targetAddress, wrapReferences)) {
            this._cell1Ref = this._cell2Ref = null;
            return true;
        }

        // relocate second cell reference if existing
        if (this._cell2Ref && !this._cell2Ref.relocate(this._docModel, refAddress, targetAddress, wrapReferences)) {
            this._cell1Ref = this._cell2Ref = null;
        }
        return true;
    };

    /**
     * Generates the text representation of this token.
     *
     * @param {GrammarConfig} config
     *  The configuration object containing grammar-specific settings for
     *  formula operators.
     *
     * @param {Object} [options]
     *  Optional parameters for relocation of range addresses. See method
     *  ReferenceToken.getRange3D() for details.
     */
    ReferenceToken.prototype.getText = function (config, options) {
        var cellRefs = this._resolveCellRefs(options);
        return config.formatReference(this._docModel, this._sheet1Ref, this._sheet2Ref, cellRefs.r1, cellRefs.r2);
    };

    /**
     * Returns a text description of this token for debug logging.
     */
    ReferenceToken.prototype.debugText = function () {
        var result = this.debugSheetPrefix();
        result += this._cell1Ref ? this._cell1Ref.toString() : '#REF';
        result += (this._cell1Ref && this._cell2Ref) ? (':' + this._cell2Ref.toString()) : '';
        return result;
    };

    // class NameToken ========================================================

    /**
     * A formula token that represents a defined name in a formula.
     *
     * @constructor
     *
     * @extends SheetRefToken
     *
     * @param {SpreadsheetModel} docModel
     *  The spreadsheet document model containing this token instance.
     *
     * @param {String} value
     *  The label of the defined name.
     *
     * @param {SheetRef|Null} sheetRef
     *  The sheet reference structure (e.g. the particle 'Sheet1' in the
     *  formula Sheet1!my_name). If null or undefined, the token represents a
     *  local name in the passed sheet, or a global name.
     */
    var NameToken = SheetRefToken.extend({ constructor: function (docModel, value, sheetRef) {

        // base constructor
        SheetRefToken.call(this, 'name', docModel, sheetRef, null);

        // private properties
        this._value = value;

    } }); // class NameToken

    // private methods --------------------------------------------------------

    /**
     * Returns the model of a defined name from the specified sheet.
     */
    NameToken.prototype._getNameModelFromSheet = function (sheet) {
        var sheetModel = _.isNumber(sheet) ? this._docModel.getSheetModel(sheet) : null;
        return sheetModel ? sheetModel.getNameCollection().getNameModel(this._value) : null;
    };

    // public methods ---------------------------------------------------------

    /**
     * Returns the label of the defined name, as passed to the constructor.
     *
     * @returns {String}
     *  The label of the defined name.
     */
    NameToken.prototype.getValue = function () {
        return this._value;
    };

    /**
     * Returns the model of the defined name referred by this token.
     *
     * @param {Number|Null} refSheet
     *  The index of the reference sheet used to resolve sheet-local names
     *  without sheet reference. If set to null, and this token does not
     *  contain its own sheet reference, this method looks for globally defined
     *  names only.
     *
     * @returns {NameModel|Null}
     *  The model of the defined name, if existing; otherwise null.
     */
    NameToken.prototype.resolveNameModel = function (refSheet) {

        // sheet reference exists (e.g.: Sheet2!name): resolve from specified sheet, do not try global names
        if (this._sheet1Ref) { return this._getNameModelFromSheet(this._sheet1Ref.sheet); }

        // no sheet reference: try sheet-local names first, then global names
        return this._getNameModelFromSheet(refSheet) || this._docModel.getNameCollection().getNameModel(this._value);
    };

    /**
     * Generates the text representation of this token.
     */
    NameToken.prototype.getText = function (config) {
        return config.formatName(this._docModel, this._sheet1Ref, this._value);
    };

    /**
     * Returns a text description of this token for debug logging.
     */
    NameToken.prototype.debugText = function () {
        return this.debugSheetPrefix() + this._value;
    };

    // class MacroToken =======================================================

    /**
     * A formula token that represents a macro function call in a formula.
     *
     * @constructor
     *
     * @extends SheetRefToken
     *
     * @param {SpreadsheetModel} docModel
     *  The spreadsheet document model containing this token instance.
     *
     * @param {String} value
     *  The name of the macro function.
     *
     * @param {SheetRef|Null} sheetRef
     *  The sheet reference structure (e.g. the particle 'Module1' in the
     *  formula Module1!myFunc() of a macro call). If null or undefined, the
     *  token represents a globally available macro function.
     */
    var MacroToken = SheetRefToken.extend({ constructor: function (docModel, value, sheetRef) {

        // base constructor
        SheetRefToken.call(this, 'macro', docModel, sheetRef, null);

        // private properties
        this._value = value;

    } }); // class MacroToken

    // public methods ---------------------------------------------------------

    /**
     * Returns the name of the macro function, as passed to the constructor.
     *
     * @returns {String}
     *  The name of the macro function.
     */
    MacroToken.prototype.getValue = function () {
        return this._value;
    };

    /**
     * Generates the text representation of this token.
     */
    MacroToken.prototype.getText = function (config) {
        return config.formatName(this._docModel, this._sheet1Ref, this._value);
    };

    /**
     * Returns a text description of this token for debug logging.
     */
    MacroToken.prototype.debugText = function () {
        return this.debugSheetPrefix() + this._value;
    };

    // static class Tokens ====================================================

    var Tokens = {
        FixedToken: FixedToken,
        OperatorToken: OperatorToken,
        SeparatorToken: SeparatorToken,
        ParenthesisToken: ParenthesisToken,
        MatrixDelimiterToken: MatrixDelimiterToken,
        LiteralToken: LiteralToken,
        MatrixToken: MatrixToken,
        FunctionToken: FunctionToken,
        ReferenceToken: ReferenceToken,
        NameToken: NameToken,
        MacroToken: MacroToken
    };

    // exports ================================================================

    return Tokens;

});
