/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/formula/interpreter', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/object/baseobject',
    'io.ox/office/baseframework/model/modelobject',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/model/formula/formulautils',
    'io.ox/office/spreadsheet/model/formula/complex',
    'io.ox/office/spreadsheet/model/formula/matrix',
    'io.ox/office/spreadsheet/model/formula/operand',
    'io.ox/office/spreadsheet/model/formula/formulacontext'
], function (Utils, BaseObject, ModelObject, SheetUtils, FormulaUtils, Complex, Matrix, Operand, FormulaContext) {

    'use strict';

    // convenience shortcuts
    var ErrorCode = SheetUtils.ErrorCode,
        Address = SheetUtils.Address,
        Range3D = SheetUtils.Range3D,
        Range3DArray = SheetUtils.Range3DArray;

    // maximum time available for evaluating a single formula, in milliseconds
    var MAX_EVAL_TIME = 1000;

    // private global functions ===============================================

    /**
     * Throws the first error code found in the passed matrix. If no error code
     * has been found, the passed matrix will be returned.
     *
     * @param {Matrix} matrix
     *  A matrix containing any scalar values used in formulas.
     *
     * @returns {Matrix}
     *  The passed matrix, if it does not contain an error code.
     *
     * @throws {ErrorCode}
     *  The first error code contained in the passed matrix.
     */
    function checkMatrixForErrorCode(matrix) {
        matrix.forEach(FormulaUtils.throwErrorCode);
        return matrix;
    }

    /**
     * Checks that all elements in the passed matrix are passing the specified
     * truth test. If any matrix element fails, the #VALUE! error code will be
     * thrown. If any matrix element is an error code, it will be thrown
     * instead. If all matrix elements are valid, the passed matrix will be
     * returned.
     *
     * @param {Matrix} matrix
     *  A matrix containing any scalar values used in formulas.
     *
     * @param {Function} callback
     *  The predicate callback function invoked for every matrix element.
     *
     * @returns {Matrix}
     *  The passed matrix, if all its elements are valid according to the test.
     *
     * @throws {ErrorCode}
     *  The first error code contained in the passed matrix, or the #VALUE!
     *  error code, if a matrix element does not pass the truth test.
     */
    function checkMatrixForDataType(matrix, callback) {
        matrix.forEach(function (elem) {
            FormulaUtils.throwErrorCode(elem);
            if (!callback(elem)) { throw ErrorCode.VALUE; }
        });
        return matrix;
    }

    /**
     * Checks that the all cell range addresses in the passed array refer to
     * the same single sheet.
     *
     * @returns {Range3DArray}
     *  The range array passed to this function, if it is valid.
     *
     * @throws {ErrorCode}
     *  - The #NULL! error code, if the range array is empty.
     *  - The #VALUE! error code, if the ranges refer to different sheets.
     */
    function checkSingleSheetRanges(ranges) {
        if (ranges.empty()) { throw ErrorCode.NULL; }
        if (!ranges.singleSheet()) { throw ErrorCode.VALUE; }
        return ranges;
    }

    // class InterpreterInstance ==============================================

    /**
     * Resolves a compiled formula token array to the result of the formula.
     * This implementation has been moved outside the Interpreter class to be
     * able to recursively resolve multiple formulas (e.g. defined names).
     *
     * @constructor
     *
     * @extends BaseObject
     *
     * @param {SpreadsheetModel} docModel
     *  The spreadsheet document model containing this instance.
     *
     * @param {Array} compilerTokens
     *  An array of token descriptors, in prefix notation, as returned by the
     *  method Compiler.compileTokens().
     *
     * @param {Object} [initOptions]
     *  Optional parameters. Supports all options also supported by the method
     *  Interpreter.interpretTokens().
     */
    var InterpreterInstance = BaseObject.extend({ constructor: function (docModel, compilerTokens, initOptions) {

        var // the descriptors of all supported operators
            operatorCollection = docModel.getFormulaResource().getOperatorCollection(),

            // the descriptors of all supported operators
            functionCollection = docModel.getFormulaResource().getFunctionCollection(),

            // the reference sheet
            refSheet = Utils.getIntegerOption(initOptions, 'refSheet', null),

            // the target reference address
            targetAddress = Utils.getOption(initOptions, 'targetAddress', Address.A1),

            // the calling context for the operator resolver callback function
            context = new FormulaContext(docModel, refSheet, targetAddress),

            // current array index into the compiler token array
            tokenIndex = 0,

            // start time of formula evaluation
            t0 = 0;

        // base constructor ---------------------------------------------------

        BaseObject.call(this);

        // private methods ----------------------------------------------------

        /**
         * Resolves the passed reference token to its result.
         *
         * @param {ReferenceToken} refToken
         *  The formula token representing a cell range reference. If the token
         *  cannot be resolved to a cell range address, this method returns the
         *  #REF! error code.
         *
         * @returns {Range3DArray|ErrorCode}
         *  The cell range addresses contained in the reference, or a #REF!
         *  error code.
         */
        function resolveReference(refToken) {
            var range = refToken.getRange3D(initOptions);
            return range ? new Range3DArray(range) : ErrorCode.REF;
        }

        /**
         * Resolves the passed name token to its result.
         *
         * @param {String} contextType
         *  The context type that influences how to resolve values, matrixes,
         *  and references. See methods FormulaInterpreter.interpretTokens()
         *  for details.
         *
         * @param {NameToken} nameToken
         *  The formula token representing a defined name. If the name does not
         *  exist in the document, this method returns the #NAME? error code.
         *
         * @returns {Any}
         *  The result value of the defined name (may be values, matrixes, or
         *  references).
         */
        function resolveDefinedName(contextType, nameToken) {

            // try to resolve the model of a defined name from the name token
            var nameModel = nameToken.resolveNameModel(refSheet);

            // unknown names result in the #NAME? error code
            if (!nameModel) { return ErrorCode.NAME; }

            // resolve the formula in the defined name (defined names are always relative to cell A1)
            var result = nameModel.getTokenArray().interpretFormula(contextType, { refSheet: refSheet, refAddress: Address.A1, targetAddress: targetAddress });

            // return UNSUPPORTED_ERROR on any error (TODO: evaluate error/warning code?)
            return (result.type === 'result') ? result.value : FormulaUtils.UNSUPPORTED_ERROR;
        }

        /**
         * Checks that the passed return type, and the context type of the
         * formula or subexpression match. Throws a 'reference' error, if the
         * context type is 'ref', and the passed type is different from 'ref'
         * or 'any'. Excel rejects such formulas, and even announces a broken
         * file format if such formulas are written into the file. Examples of
         * such invalid formulas are:
         *  =A1:1 (combined number with range operator)
         *  =OFFSET(1,1,1) (passed number to a reference-only parameter)
         *
         * It is possible to use operands and functions with return type
         * 'any' in reference context, e.g.
         *  =A1:name (return type of defined names is 'any')
         *  =A1:IF(TRUE,A1,B1) (return type of IF is 'any').
         * If the defined name or function returns a non-reference value, the
         * formula simply results in the #VALUE! error.
         *
         * @param {String} type
         *  The return type of an operator or function, or the type of a single
         *  operand to be checked. Must be one of:
         *  - 'val': A scalar value (numbers, dates, strings, boolean values,
         *      complex numbers, or error codes).
         *  - 'mat': Constant matrixes, or functions returning matrixes.
         *  - 'ref': Cell references (constant references, reference operators,
         *      functions returning a reference, or the #REF! error code).
         *  - 'any': Any of the above (defined names, a few functions returning
         *      any data type).
         *
         * @param {String} contextType
         *  The context type that influences how to resolve values, matrixes,
         *  and references. See method FormulaInterpreter.interpretTokens() for
         *  details. If the context type is 'ref', only the return types 'ref'
         *  and 'any' will be accepted. All other return types will cause to
         *  throw an internal 'reference' exception that marks the formula
         *  structure to be ill-formed (see description above).
         *
         * @throws {String}
         *  The special error code 'reference', if the passed return type is
         *  not supported in reference context.
         */
        function checkContextType(type, contextType) {
            // context type 'ref': accept 'ref' and 'any' only
            if ((contextType === 'ref') && (type !== 'ref') && (type !== 'any')) {
                throw 'reference';
            }
        }

        /**
         * Returns whether the passed value is supported by the operand type
         * 'val', i.e. whether it is a scalar value.
         */
        function isValType(value) {
            return _.isNumber(value) || _.isString(value) || _.isBoolean(value) || (value instanceof Date) || (value instanceof Complex) || (value instanceof ErrorCode);
        }

        /**
         * Returns whether the passed value is supported by the operand type
         * 'mat', i.e. whether it is a matrix.
         */
        function isMatType(value) {
            return value instanceof Matrix;
        }

        /**
         * Returns whether the passed value is supported by the operand type
         * 'ref', i.e. whether it is a single cell range address, or an array
         * of cell range addresses.
         */
        function isRefType(value) {
            return (value instanceof Range3D) || (value instanceof Range3DArray);
        }

        /**
         * Returns whether the passed value has the specified base type.
         */
        function isValueOfType(value, type) {
            switch (type) {
                case 'val':
                    return isValType(value);
                case 'mat':
                    return isMatType(value);
                case 'ref':
                    return isRefType(value);
                case 'any':
                    return isValType(value) || isMatType(value) || isRefType(value) || (value instanceof Operand);
            }
            FormulaUtils.throwInternal('isValueOfType(): unknown type identifier: "' + type + '"');
        }

        /**
         * Creates an operand object for the passed parser token.
         *
         * @param {Object} compilerToken
         *  A compiler token representing an operand (scalar values, constant
         *  matrixes, cell references, defined names).
         *
         * @param {String} contextType
         *  The context type that influences how to resolve values, matrixes,
         *  and references. See methods FormulaInterpreter.interpretTokens()
         *  for details.
         *
         * @returns {Operand}
         *  An operand instance representing the passed token.
         *
         * @throws {ErrorCode}
         *  The special error code UNSUPPORTED_ERROR, if the type of the passed
         *  token is not supported.
         */
        function makeTokenOperand(compilerToken, contextType) {

            var // the parser token representing the operand value
                token = compilerToken.token,
                // the token type
                type = null,
                // the value to be inserted into the operand
                value = null;

            FormulaUtils.log('> resolving operand token: ' + token);

            switch (token.getType()) {
                case 'lit': // scalar literals
                    value = token.getValue();
                    type = (value === ErrorCode.REF) ? 'ref' : 'val';
                    break;
                case 'mat': // matrix literals
                    value = token.getMatrix();
                    type = 'mat';
                    break;
                case 'ref': // cell range references
                    value = resolveReference(token);
                    type = 'ref';
                    break;
                case 'name': // defined names
                    value = resolveDefinedName(contextType, token);
                    type = 'any';
                    break;
                default:
                    FormulaUtils.throwInternal('InterpreterInstance.makeTokenOperand(): unknown token type: "' + token.getType() + '"');
            }

            // check context type (will throw 'reference' error)
            checkContextType(type, contextType);
            return new Operand(context, value);
        }

        /**
         * Converts the passed operand according to the passed parameter type
         * from the operator/function signature.
         */
        function resolveParam(operand, sig, row, col) {
            switch (sig) {
                case 'any':
                    return operand;
                case 'val':
                    return FormulaUtils.throwErrorCode(operand.getValue(row, col));
                case 'val:num':
                    return context.convertToNumber(operand.getValue(row, col));
                case 'val:int':
                    return context.convertToNumber(operand.getValue(row, col), true);
                case 'val:date':
                    return context.convertToDate(operand.getValue(row, col));
                case 'val:day':
                    return context.convertToDate(operand.getValue(row, col), true);
                case 'val:str':
                    return context.convertToString(operand.getValue(row, col));
                case 'val:bool':
                    return context.convertToBoolean(operand.getValue(row, col));
                case 'val:comp':
                    return context.convertToComplex(operand.getValue(row, col));
                case 'val:any':
                    return operand.getValue(row, col);
                case 'mat':
                    return checkMatrixForErrorCode(operand.getMatrix());
                case 'mat:num':
                    // no conversion from other data types in matrix parameters!
                    return checkMatrixForDataType(operand.getMatrix(), _.isNumber);
                case 'mat:str':
                    // no conversion from other data types in matrix parameters!
                    return checkMatrixForDataType(operand.getMatrix(), _.isString);
                case 'mat:bool':
                    // no conversion from other data types in matrix parameters!
                    return checkMatrixForDataType(operand.getMatrix(), _.isBoolean);
                case 'mat:any':
                    return operand.getMatrix();
                case 'ref':
                    return operand.getRanges();
                case 'ref:sheet':
                    return checkSingleSheetRanges(operand.getRanges());
                case 'ref:single':
                    return context.convertToRange(operand.getRanges());
                case 'ref:multi':
                    return context.convertToRange(operand.getRanges(), { multiSheet: true });
            }
            FormulaUtils.throwInternal('InterpreterInstance.resolveParam(): unknown parameter signature type: "' + sig + '"');
        }

        /**
         * Processes the passed operator or function call. Calculates the
         * required number of operands from the following compiler tokens, and
         * calculates the result of the operator or function.
         *
         * @param {Object} compilerToken
         *  A compiler token representing an operator or function.
         *
         * @param {String} contextType
         *  The context type that influences how to resolve values, matrixes,
         *  and references. See methods FormulaInterpreter.interpretTokens()
         *  for details.
         *
         * @returns {Operand}
         *  An operand instance representing the result of the operator or
         *  function call.
         *
         * @throws {String}
         *  The error code 'missing', if there are not enough tokens available
         *  in the compiled token array.
         */
        function processOperator(compilerToken, contextType) {

            var // the parser token representing the operator or function
                token = compilerToken.token,
                // the operator/function specification
                descriptor = null,
                // the expanded type signature
                signature = null,
                // the operands required by the operator or function
                operands = null,
                // matrix size for repeated processing of value operators
                repeatRows = 0,
                repeatCols = 0,
                // the resulting operand
                result = null;

            FormulaUtils.log('> processing operator ' + token + ' with ' + compilerToken.params + ' parameters');

            // returns the type of the specified parameter from the signature
            function getParamType(index) {
                if (descriptor.signature.length <= index) {
                    var repeat = _.isNumber(descriptor.repeatCount) ? descriptor.repeatCount : 1;
                    index = descriptor.signature.length - repeat + ((index - descriptor.signature.length) % repeat);
                }
                return descriptor.signature[index];
            }

            // resolves the operands according to the type signature
            function resolveOperator(row, col) {

                var // the parameter values to be passed to the operator resolver callback
                    params = null,
                    // the resulting operand
                    result = null;

                try {
                    // resolves the operand values according to the signature
                    params = operands.map(function (operand, index) {
                        return resolveParam(operand, signature[index].paramType, row, col);
                    });

                    // invoke the resolver callback function of the operator
                    FormulaUtils.logTokens('\xa0 resolving with parameters', params);
                    result = descriptor.resolve.apply(context, params);

                    // compare result with the specified return type
                    if (!isValueOfType(result, descriptor.type)) {
                        FormulaUtils.throwInternal('InterpreterInstance.resolveOperator(): ' + descriptor.name + ': operator result type does not match operator return type');
                    }

                } catch (error) {
                    if (error instanceof ErrorCode) {
                        // operator may throw error codes
                        result = error;
                    } else {
                        // re-throw other error codes, or internal JavaScript exceptions
                        throw error;
                    }
                }

                // check elapsed time, return with UNSUPPORTED_ERROR if formula is too complex
                // (but not if the formula logger is active, to allow debugging the interpreter)
                if (!FormulaUtils.isLoggingActive() && (_.now() - t0 > MAX_EVAL_TIME)) {
                    result = FormulaUtils.UNSUPPORTED_ERROR;
                }

                return result;
            }

            // resolve token to operator or function descriptor, return #NAME? for unknown function (external functions, macro calls)
            descriptor = token.isType('op') ? operatorCollection.get(token.getValue()) : token.isType('func') ? functionCollection.get(token.getValue()) : null;
            if (!descriptor) { return new Operand(context, ErrorCode.NAME); }

            // check context type (will throw 'reference' error)
            checkContextType(descriptor.type, contextType);

            // get minimum number of parameters
            var minParams = descriptor.minParams;

            // get maximum number of parameters, and length of repeated sequences
            var maxParams = 0, repeatParams = 0;
            if ('maxParams' in descriptor) {
                maxParams = descriptor.maxParams;
                repeatParams = 1;
            } else {
                repeatParams = _.isNumber(descriptor.repeatParams) ? descriptor.repeatParams : 1;
                maxParams = minParams + Utils.roundDown(FormulaUtils.MAX_PARAM_COUNT - minParams, repeatParams);
            }

            // check operand count
            if (compilerToken.params < minParams) { throw 'missing'; }
            if (compilerToken.params > maxParams) { throw 'unexpected'; }

            // repeated parameter sequences must be complete
            if (((compilerToken.params - minParams) % repeatParams) !== 0) { throw 'missing'; }

            // check missing implementation
            if (!_.isFunction(descriptor.resolve)) {
                FormulaUtils.warn('InterpreterInstance.processOperator(): unsupported operator or function "' + token + '"');
                return new Operand(context, FormulaUtils.UNSUPPORTED_ERROR);
            }

            // build the complete type signature
            signature = _.times(compilerToken.params, function (index) {

                // get operand type from signature in the descriptor
                var paramType = getParamType(index);
                // the base parameter type used as context type for the parameter
                var baseType = paramType.split(':')[0];

                // return type signature entry
                return { paramType: paramType, baseType: baseType };
            });

            // build the array of operands
            operands = signature.map(function (typeData) {

                // the resulting context type for the operand
                var opContextType = typeData.baseType;

                // if context type for this operator is matrix, and the current parameter is of type value,
                // and this operator returns a value, pass the matrix context type through to the operand
                // (resolve nested cell references to matrixes instead of single values)
                if ((contextType === 'mat') && (typeData.baseType === 'val') && (descriptor.type === 'val')) {
                    opContextType = 'mat';
                }

                return getNextOperand(opContextType);
            });

            // special handling for all operands of operators returning values
            if (descriptor.type === 'val') {
                operands.forEach(function (operand, index) {

                    // convert or preprocess operands, if operator expects a value type
                    if (signature[index].baseType === 'val') {

                        // If outer operator context is matrix, convert cell references to matrixes.
                        // Example: In the formula =MMULT(A1:B2;ABS(A1:B2)), the function MMULT passes
                        // context type matrix to its operands which causes ABS to resolve the cell
                        // reference as matrix instead of value.
                        if ((contextType === 'mat') && operand.isReference()) {
                            operand = operands[index] = new Operand(context, operand.getMatrix());
                        }

                        // Check for matrixes in value operands. Resolve operator repeatedly to a matrix,
                        // if it returns values. Example: The plus operator in the formula =SUM({1;2|3;4}+1)
                        // will convert the number 1 to the matrix {1;1|1;1}.
                        if (operand.isMatrix()) {
                            var matrix = operand.getMatrix();
                            repeatRows = Math.max(repeatRows, matrix.rows());
                            repeatCols = Math.max(repeatCols, matrix.cols());
                        }
                    }
                });
            }

            // extend the calling context for the operator resolver with the operands
            context.registerOperands(operands);

            // resolve as matrix, if any parameter with value signature type is a matrix
            if (repeatRows * repeatCols > 0) {

                // restrict to supported matrix size
                if (!FormulaUtils.isValidMatrixSize(repeatRows, repeatCols)) {
                    return new Operand(context, FormulaUtils.UNSUPPORTED_ERROR);
                }

                // build a matrix as result value
                FormulaUtils.info('\xa0 process as ' + repeatRows + 'x' + repeatCols + ' matrix');
                result = Matrix.generate(repeatRows, repeatCols, resolveOperator);

            } else {
                result = resolveOperator();
            }

            // return result as operand instance
            return (result instanceof Operand) ? result : new Operand(context, result);
        }

        /**
         * Returns the result of the next compiler token. If it is an operand,
         * it will be returned; if it is an operator or function call, its
         * result will be calculated and returned. If the operands of the
         * operator or functions are also operators or functions, their results
         * will be calculated recursively.
         *
         * @param {String} contextType
         *  The context type that influences how to resolve values, matrixes,
         *  and references. See methods FormulaInterpreter.interpretTokens()
         *  for details.
         *
         * @returns {Operand}
         *  The result of the next compiler token.
         */
        function getNextOperand(contextType) {

            var // the compiler token top be processed
                compilerToken = compilerTokens[tokenIndex],
                // the resulting operand
                result = null;

            // double-check the token (compiler should not create invalid token arrays)
            if (!compilerToken) {
                FormulaUtils.throwInternal('InterpreterInstance.getNextOperand(): missing compiler token');
            }

            // move to next unprocessed token
            tokenIndex += 1;

            // process token by type
            switch (compilerToken.type) {
                case 'val':
                    // return operands directly
                    result = makeTokenOperand(compilerToken, contextType);
                    break;
                case 'op':
                    // process operators recursively
                    result = processOperator(compilerToken, contextType);
                    break;
                default:
                    FormulaUtils.throwInternal('InterpreterInstance.getNextOperand(): unknown token type: "' + compilerToken.type + '"');
            }

            FormulaUtils.log('< result of ' + compilerToken + ': ' + result);
            return result;
        }

        // public methods -----------------------------------------------------

        /**
         * Calculates the result of the formula represented by the token array
         * passed in the constructor.
         *
         * @param {String} contextType
         *  The context type that influences how to resolve values, matrixes,
         *  and references. See method FormulaInterpreter.interpretTokens()
         *  for details.
         *
         * @returns {Operand}
         *  The result of the formula, as unresolved operand object.
         */
        this.getResult = function (contextType) {
            t0 = _.now();
            return getNextOperand(contextType);
        };

        // initialization -----------------------------------------------------

        this.registerDestructor(function () {
            docModel = compilerTokens = initOptions = context = null;
        });

    } }); // class InterpreterInstance

    // class Interpreter ======================================================

    /**
     * Calculates the result for an array of token descriptors that has been
     * compiled to prefix notation (a.k.a. Polish Notation). See class Compiler
     * for details about compiled token arrays.
     *
     * @constructor
     *
     * @extends ModelObject
     *
     * @param {SpreadsheetModel} docModel
     *  The spreadsheet document model containing this instance.
     */
    function Interpreter(docModel) {

        // base constructor ---------------------------------------------------

        ModelObject.call(this, docModel);

        // public methods -----------------------------------------------------

        /**
         * Calculates the result of the formula represented by the passed
         * compiled token array.
         *
         * @param {String} contextType
         *  The context type that influences how to resolve values, matrixes,
         *  and references. The following context types are supported:
         *  - 'val': A single value is expected, e.g. in simple cell formulas,
         *      or operators and functions working with simple values (plus
         *      operator, ABS).
         *  - 'mat': A matrix of values is expected, e.g. in matrix formulas,
         *      or functions working on entire matrixes (MMULT).
         *  - 'ref': An unresolved cell reference is expected, e.g. reference
         *      operators (list, intersection, range), or functions calculating
         *      with range addresses instead of the cell contents (OFFSET).
         *  - 'any': Accepts any of the types mentioned above, with minimum
         *      conversion, e.g. the result of defined names, functions passing
         *      one of their original parameters through (IF, CHOOSE), or
         *      functions that iterate over available values in matrixes and
         *      cell references (SUM, PRODUCT).
         *
         * @param {Array} compilerTokens
         *  An array of token descriptors, in prefix notation, as returned by
         *  the method Compiler.compileTokens().
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Number} [options.refSheet]
         *      The index of the reference sheet that will be used to resolve
         *      reference tokens and defined names without explicit sheet
         *      reference. If omitted, reference tokens without sheet indexes
         *      will result in #REF! errors.
         *  @param {Address} [options.refAddress]
         *      The source reference address used to relocate reference tokens
         *      with relative column/row components. If omitted, cell A1 will
         *      be used as source reference cell.
         *  @param {Address} [options.targetAddress]
         *      The target reference address used to relocate reference tokens
         *      with relative column/row components. If omitted, cell A1 will
         *      be used as target reference cell.
         *  @param {Boolean} [options.wrapReferences=false]
         *      If set to true, relocated ranges that are located outside the
         *      sheet will be wrapped at the sheet borders.
         *
         * @returns {Object}
         *  The result descriptor of the formula interpreter, with the
         *  following properties:
         *  - {String} type
         *      The result type:
         *      - 'result': A valid result has been calculated. The property
         *          'value' contains the formula result.
         *      - 'warn': The formula result has not been calculated, but the
         *          formula structure is valid. The property 'value' specifies
         *          the warning: 'circular' for circular references, or
         *          'unsupported', if an unsupported feature was found in the
         *          formula, or 'internal', if any internal error has occurred.
         *      - 'error': The formula structure is invalid. The property
         *          'value' contains an identifier for the reason: 'missing',
         *          if something is missing, e.g. a function was called with
         *          less arguments than required; or 'unexpected' if something
         *          was not expected, e.g. a function was called with too many
         *          arguments.
         *  - {Any} value
         *      The result of the formula, or an error code, if calculation has
         *      failed (see property 'type' above).
         */
        this.interpretTokens = function (contextType, compilerTokens, options) {

            var // the actual interpreter instance, needed for recursive interpretation of e.g. defined names
                instance = new InterpreterInstance(docModel, compilerTokens, options),
                // the resulting operand
                result = null;

            try {

                // calculate the formula, pull formula result operand
                FormulaUtils.takeTime('Interpreter.interpretTokens(): contextType=' + contextType, function () {
                    result = instance.getResult(contextType);
                });

                // 'any' context: resolve to actual type of the resulting operand
                contextType = (contextType === 'any') ? result.getType() : contextType;

                // resolve result operand to result value according to context type
                switch (contextType) {
                    case 'val':
                        result = result.getValue();
                        // in value context, empty values (reference to empty cell) become zero
                        if (_.isNull(result)) { result = 0; }
                        break;
                    case 'mat':
                        result = result.getMatrix();
                        break;
                    case 'ref':
                        result = result.getRanges();
                        break;
                    default:
                        FormulaUtils.throwInternal('FormulaInterpreter.interpretTokens(): unknown context type: "' + contextType + '"');
                }

                // throw error code (special handling for internal errors in catch clause)
                FormulaUtils.throwErrorCode(result);

                // convert to valid result object
                result = { type: 'result', value: result };

            } catch (error) {

                // operator may throw error codes
                if (error instanceof ErrorCode) {

                    // 'unsupported' state, circular references, internal errors: return empty result with warning property
                    result = error.internal ?
                        { type: 'warn', value: error.key } :
                        { type: 'result', value: error };

                // other internal errors (must be strings): invalid formula structure
                } else if (_.isString(error)) {
                    FormulaUtils.warn('\xa0 error: ' + error);
                    result = { type: 'error', value: error };

                // re-throw internal JS exceptions
                } else {
                    throw error;
                }

            } finally {
                instance.destroy();
            }

            FormulaUtils.info('result: type=' + result.type + ', value=' + FormulaUtils.valueToString(result.value));
            return result;
        };

        // initialization -----------------------------------------------------

        // destroy all class members on destruction
        this.registerDestructor(function () {
            docModel = null;
        });

    } // class Interpreter

    // exports ================================================================

    // derive this class from class ModelObject
    return ModelObject.extend({ constructor: Interpreter });

});
