/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/formula/impl/referencefuncs', [
    'io.ox/office/tk/utils',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/model/formula/formulautils',
    'io.ox/office/spreadsheet/model/formula/cellref',
    'io.ox/office/spreadsheet/model/formula/sheetref'
], function (Utils, SheetUtils, FormulaUtils, CellRef, SheetRef) {

    'use strict';

    /**************************************************************************
     *
     * This module implements all spreadsheet functions dealing with cell range
     * references used as values (cell address arithmetics), and functions that
     * look up specific values in a range of cells.
     *
     * See the README file in this directory for a detailed documentation about
     * the format of function descriptor objects.
     *
     * In spreadsheet documents, cell range references can be given literally,
     * or can be the result of functions or operators. The client-side formula
     * engine provides the function signature type 'ref', and the class
     * Reference whose instances represent a single cell range address, or a
     * list of several cell range addresses (e.g. as result of the reference
     * list operator).
     *
     *************************************************************************/

    // convenience shortcuts
    var ErrorCode = SheetUtils.ErrorCode;

    // private global functions ===============================================

    /**
     * @returns {Number} order
     *  1  is ascending
     *  -1 is descending
     *  0  is no order
     */
    function getOrderType(vector) {
        var len = vector.length;
        if (len < 2) { return 1; }

        var order = FormulaUtils.compareValues(vector[0], vector[1]);
        for (var i = 2; i < len; i++) {
            var subOrder = FormulaUtils.compareValues(vector[i - 1], vector[i]);
            if (subOrder !== order) { return 0; }
        }
        return order * -1;
    }

    function searchInVector(lookup, searchVector, resultVector, inexact) {
        if (resultVector.length < searchVector.length) {
            Utils.warn('searchInVector() resultVector is longer than searchVector. this must be handled by spreadsheet correctly, because excel can access data outside the resultVector, too!!!');
            //TODO: FIXME: searchInVector() resultVector is longer than searchVector. this must be handled by spreadsheet correctly, because excel can access data outside the resultVector, too!!!
            throw ErrorCode.REF;
        }

        var index = searchIndexInVector.call(this, lookup, searchVector, inexact);
        return resultVector[index];
    }

    function searchIndexInVector(lookup, searchVector, inexact) {

        function getTypeId(value) {
            return _.isNumber(value) ? 1 : _.isString(value) ? 2 : _.isBoolean(value) ? 3 : (value instanceof ErrorCode) ? 4 : 0;
        }

        if (_.isUndefined(inexact)) { inexact = true; }
        if (_.isUndefined(lookup) || _.isNull(lookup)) { throw ErrorCode.VALUE; }

        var regex = null;
        if (!inexact && _.isString(lookup)) { regex = this.convertPatternToRegExp(lookup, true); }

        var lookupType = getTypeId(lookup);

        for (var i = 0; i < searchVector.length; i++) {
            var currentEntry = searchVector[i];

            var currentType = getTypeId(currentEntry);
            if (currentType !== lookupType) {
                continue;
            }

            var compare = FormulaUtils.compareValues(currentEntry, lookup);
            if (compare === 0) {
                return i;
            } else if (inexact && compare > 0) {
                if (i < 1) { throw ErrorCode.NA; }
                return i - 1;
            }

            if (regex && regex.test(currentEntry)) {
                return i;
            }
        }
        if (inexact) {
            var lastIndex = searchVector.length - 1;
            var last = searchVector[lastIndex];
            if (getTypeId(last) === lookupType && FormulaUtils.compareValues(last, lookup) < 0) { return lastIndex; }
        }
        throw ErrorCode.NA;
    }

    // exports ================================================================

    return {

        ADDRESS: {
            category: 'reference',
            minParams: 2,
            maxParams: 5,
            type: 'val',
            signature: 'val:int val:int val:int val:bool val:str',
            resolve: function (row, col, absMode, a1Style, sheetName) {

                var // the configuration of the UI formula grammar
                    config = this.getGrammarConfig(),
                    // the resulting address string
                    result = '';

                // default for missing or empty absolute mode (always defaults to 1)
                if (this.isMissingOrEmptyOperand(2)) {
                    absMode = 1;
                } else if ((absMode < 1) || (absMode > 4)) {
                    throw ErrorCode.VALUE;
                }

                // default for missing or empty A1 style (always defaults to TRUE)
                if (this.isMissingOrEmptyOperand(3)) {
                    a1Style = true;
                }

                // initialize and validate the cell reference structure
                var cellRef = new CellRef(0, 0, (absMode % 2) > 0, absMode <= 2);
                cellRef.col = (a1Style || cellRef.absCol) ? (col - 1) : col;
                cellRef.row = (a1Style || cellRef.absRow) ? (row - 1) : row;
                this.checkCellRef(cellRef, !a1Style);

                // start with encoded sheet name (enclosed in apostrophes if needed)
                // (not for missing or empty parameter, but for explicit empty string)
                if (_.isString(sheetName)) {

                    // enclose complex names in apostrophes
                    if ((sheetName.length > 0) && !config.isSimpleSheetName(this.getDocModel(), sheetName, { range: true, external: true })) {
                        sheetName = SheetRef.encodeComplexSheetName(sheetName);
                    }

                    // use sheet name with an exclamation mark as separator
                    result = sheetName + '!';
                }

                // append the range address in A1 or R1C1 style
                return result + config.formatReference(this.getDocModel(), null, null, cellRef, null, !a1Style);
            }
        },

        AREAS: {
            category: 'reference',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'ref',
            resolve: function (ranges) { return ranges.length; }
        },

        CHOOSE: {
            category: 'reference',
            minParams: 2,
            type: 'any',
            signature: 'val:int any',
            resolve: function (index) {
                return this.getOperand(index - 1);
            }
        },

        COLUMN: {
            category: 'reference',
            minParams: 0,
            maxParams: 1,
            type: 'val',
            signature: 'ref:single',
            resolve: function (range) {
                // use reference address, if parameter is missing
                var address = range ? range.start : this.getRefAddress();
                // function returns one-based column index
                return address[0] + 1;
            }
        },

        COLUMNS: {
            category: 'reference',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'any',
            resolve: function (operand) {
                switch (operand.getType()) {
                    case 'val':
                        var value = operand.getValue();
                        return (value instanceof ErrorCode) ? value : 1;
                    case 'mat':
                        return operand.getMatrix().cols();
                    case 'ref':
                        return this.convertToRange(operand.getRanges()).cols();
                    default:
                        FormulaUtils.throwInternal('ReferenceFuncs.COLUMNS.resolve(): unknown operand type');
                }
            }
        },

        FORMULATEXT: {
            category: 'reference',
            name: { ooxml: '_xlfn.FORMULATEXT', odf: 'FORMULA' },
            ceName: { odf: 'FORMULA' },
            minParams: 1,
            maxParams: 1,
            type: 'val',
            volatile: true/*,
            signature: 'ref:single',
            resolve: function (range) {

                // use the first cell in the range (TODO: matrix context)
                var address = range.start;

                // getCellFormula() returns null for value cells
                var text = this.getCellFormula(range.sheet1, address);

                if (_.isString(text)) { return text; }

                throw ErrorCode.NA;
            }*/
        },

        GETPIVOTDATA: {
            category: 'reference',
            hidden: true,
            minParams: 2,
            repeatParams: 2,
            type: 'val'
        },

        HLOOKUP: {
            category: 'reference',
            minParams: 3,
            maxParams: 4,
            type: 'any',
            signature: 'val mat:any val:int val:bool',
            resolve: function (lookup, matrix, rowIndexParam, inexact) {
                var rowIndex = rowIndexParam - 1;
                var rows = matrix.rows();

                if (rowIndex < 0) { throw ErrorCode.VALUE; }
                if (rowIndex >= rows) { throw ErrorCode.REF; }

                return searchInVector.call(this, lookup, matrix.getRow(0), matrix.getRow(rowIndex), inexact);
            }
        },

        INDEX: {
            category: 'reference',
            minParams: 2,
            maxParams: 4,
            type: 'any'
        },

        INDIRECT: {
            category: 'reference',
            minParams: 1,
            maxParams: 2,
            type: 'ref',
            volatile: true
        },

        LOOKUP: {
            category: 'reference',
            minParams: 2,
            maxParams: 3,
            type: 'any',
            signature: 'val mat:any mat:any',
            resolve: function (lookup, searchMat, resultMat) {
                var searchVector = null;
                var resultVector = null;

                if (_.isUndefined(resultMat)) {
                    if (searchMat.rows() > searchMat.cols()) {
                        searchVector = searchMat.getCol(0);
                        resultVector = searchMat.getCol(searchMat.cols() - 1);
                    } else {
                        searchVector = searchMat.getRow(0);
                        resultVector = searchMat.getRow(searchMat.rows() - 1);
                    }
                } else {
                    if (Math.min(resultMat.rows(), resultMat.cols()) !== 1) { throw ErrorCode.NA; }

                    if (searchMat.rows() > searchMat.cols()) {
                        searchVector = searchMat.getCol(0);
                    } else {
                        searchVector = searchMat.getRow(0);
                    }

                    if (resultMat.rows() > resultMat.cols()) {
                        resultVector = resultMat.getCol(resultMat.cols() - 1);
                    } else {
                        resultVector = resultMat.getRow(resultMat.rows() - 1);
                    }
                }
                return searchInVector.call(this, lookup, searchVector, resultVector);
            }
        },

        MATCH: {
            category: 'reference',
            minParams: 2,
            maxParams: 3,
            type: 'val',
            signature: 'val mat:any val:int',
            resolve: function (lookup, matrix, order) {

                if (Math.min(matrix.rows(), matrix.cols()) !== 1) { throw ErrorCode.NA; }

                var vector = null;
                if (matrix.rows() > matrix.cols()) {
                    vector = matrix.getCol(0);
                } else {
                    vector = matrix.getRow(0);
                }

                if (_.isUndefined(order)) { order = 1; }

                if (order !== 0 && order !== getOrderType.call(this, vector)) { throw ErrorCode.NA; }

                if (order === -1) { vector.reverse(); }

                return searchIndexInVector.call(this, lookup, vector, order !== 0) + 1;
            }
        },

        OFFSET: {
            category: 'reference',
            minParams: 3,
            maxParams: 5,
            type: 'ref',
            volatile: true,
            signature: 'ref val:int val:int val:int val:int',
            resolve: function (ranges, rows, cols, height, width) {

                // reference must contain a single range address
                var range = this.convertToRange(ranges, { valueError: true });

                // apply offset
                range.start[0] += cols;
                range.start[1] += rows;
                range.end[0] += cols;
                range.end[1] += rows;

                // modify size (but not if 'height' or 'width' parameters exist but are empty)
                if (!this.isMissingOrEmptyOperand(4)) { range.end[0] = range.start[0] + width - 1; }
                if (!this.isMissingOrEmptyOperand(3)) { range.end[1] = range.start[1] + height - 1; }

                // check that the range does not left the sheet boundaries
                if (this.getDocModel().isValidRange(range)) { return range; }
                throw ErrorCode.REF;
            }
        },

        ROW: {
            category: 'reference',
            minParams: 0,
            maxParams: 1,
            type: 'val',
            signature: 'ref:single',
            resolve: function (range) {
                // use reference address, if parameter is missing
                var address = range ? range.start : this.getRefAddress();
                // function returns one-based row index
                return address[1] + 1;
            }
        },

        ROWS: {
            category: 'reference',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'any',
            resolve: function (operand) {
                switch (operand.getType()) {
                    case 'val':
                        var value = operand.getValue();
                        return (value instanceof ErrorCode) ? value : 1;
                    case 'mat':
                        return operand.getMatrix().rows();
                    case 'ref':
                        return this.convertToRange(operand.getRanges()).rows();
                    default:
                        FormulaUtils.throwInternal('ReferenceFuncs.ROWS.resolve(): unknown operand type');
                }
            }
        },

        SHEET: {
            category: 'reference',
            name: { ooxml: '_xlfn.SHEET' },
            minParams: 0,
            maxParams: 1,
            type: 'val',
            volatile: true,
            signature: 'any', // string (sheet name) or reference
            resolve: function (operand) {
                // use reference sheet, if parameter is missing
                if (_.isUndefined(operand)) { return this.getRefSheet() + 1; }
                // use sheet index of a reference
                if (operand.isReference()) { return this.convertToRange(operand.getRanges()).sheet1 + 1; }
                // convert values to strings (sheet name may look like a number or Boolean)
                if (operand.isValue()) {
                    var sheet = this.getDocModel().getSheetIndex(this.convertToString(operand.getValue()));
                    return (sheet < 0) ? ErrorCode.NA : (sheet + 1);
                }
                // always fail for matrix operands, regardless of context
                return ErrorCode.NA;
            }
        },

        SHEETS: {
            category: 'reference',
            name: { ooxml: '_xlfn.SHEETS' },
            minParams: 0,
            maxParams: 1,
            type: 'val',
            volatile: true,
            signature: 'ref:multi',
            resolve: function (range) {
                // return total number of sheets in document, if parameter is missing
                return range ? range.sheets() : this.getDocModel().getSheetCount();
            }
        },

        VLOOKUP: {
            category: 'reference',
            minParams: 3,
            maxParams: 4,
            type: 'any',
            signature: 'val mat:any val:int val:bool',
            resolve: function (lookup, matrix, colIndexParam, inexact) {
                var colIndex = colIndexParam - 1;
                var cols = matrix.cols();

                if (colIndex < 0) { throw ErrorCode.VALUE; }
                if (colIndex >= cols) { throw ErrorCode.REF; }

                return searchInVector.call(this, lookup, matrix.getCol(0), matrix.getCol(colIndex), inexact);
            }
        }
    };

});
