/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/formula/impl/logicalfuncs', [
    'io.ox/office/spreadsheet/utils/errorcode'
], function (ErrorCode) {

    'use strict';

    /**************************************************************************
     *
     * This module implements all spreadsheet functions dealing with Boolean
     * values.
     *
     * See the README file in this directory for a detailed documentation about
     * the format of function descriptor objects.
     *
     *************************************************************************/

    // standard options for logical parameter iterators (method FormulaContext.iterateBooleans())
    var SKIP_MAT_SKIP_REF = { // id: CSS5
        valMode: 'convert', // value operands: convert strings and numbers to Booleans
        matMode: 'skip', // matrix operands: skip all strings
        refMode: 'skip', // reference operands: skip all strings
        emptyParam: true, // empty parameters count as FALSE
        complexRef: true // accept multi-range and multi-sheet references
    };

    // private global functions ===============================================

    /**
     * Returns the logical AND of the passed Booleans (used for callbacks).
     */
    function logicalAnd(bool1, bool2) {
        return bool1 && bool2;
    }

    /**
     * Returns the logical OR of the passed Booleans (used for callbacks).
     */
    function logicalOr(bool1, bool2) {
        return bool1 || bool2;
    }

    /**
     * Returns the logical XOR of the passed Booleans (used for callbacks).
     */
    function logicalXor(bool1, bool2) {
        return bool1 !== bool2;
    }

    /**
     * Creates and returns a resolver function for spreadsheet functions that
     * reduces all Boolean values (and other values convertible to Boolean
     * values) of all function parameters to the result of the function. See
     * method FormulaContext.aggregateBooleans() for more details about the
     * parameters.
     *
     * @returns {Function}
     *  The resulting function implementation to be assigned to the 'resolve'
     *  property of a function descriptor. The spreadsheet function will pass
     *  all Boolean values of all its operands to the aggregation callback
     *  function. If no Boolean values have been found at all, the spreadsheet
     *  function will result in the #VALUE! error code, instead of the initial
     *  result.
     */
    function implementLogicalAggregation(initial, aggregate, iteratorOptions) {

        // no Booleans available results in the #VALUE! error code (not the initial value)
        function finalize(result, count) {
            if (count === 0) { throw ErrorCode.VALUE; }
            return result;
        }

        // create and return the resolver function to be assigned to the 'resolve' property of a function descriptor
        return function logicalAggregator() {
            return this.aggregateBooleans(arguments, initial, aggregate, finalize, iteratorOptions);
        };
    }

    // exports ================================================================

    return {

        AND: {
            category: 'logical',
            minParams: 1,
            type: 'val',
            signature: 'any',
            // empty parameters count as FALSE: AND(TRUE,) = FALSE
            resolve: implementLogicalAggregation(true, logicalAnd, SKIP_MAT_SKIP_REF)
        },

        FALSE: {
            category: 'logical',
            minParams: 0,
            maxParams: 0,
            type: 'val',
            resolve: _.constant(false)
        },

        IF: {
            category: 'logical',
            minParams: 2,
            maxParams: 3,
            type: 'any',
            signature: 'val:bool any any',
            resolve: function (cond, operand1, operand2) {
                return cond ? (_.isUndefined(operand1) ? true : operand1) : (_.isUndefined(operand2) ? false : operand2);
            }
        },

        IFERROR: {
            category: 'logical',
            ceName: { odf: null }, // TODO: CalcEngine should allow to use this function in ODF
            minParams: 2,
            maxParams: 2,
            type: 'any',
            signature: 'val:any val:any', // always return values (in difference to IF)
            resolve: function (value1, value2) {
                return (value1 instanceof ErrorCode) ? value2 : value1;
            }
        },

        IFNA: {
            category: 'logical',
            name: { ooxml: '_xlfn.ISNA' },
            ceName: { odf: null }, // TODO: CalcEngine should allow to use this function in ODF
            minParams: 2,
            maxParams: 2,
            type: 'any',
            signature: 'val:any val:any', // always return values (in difference to IF)
            resolve: function (value1, value2) {
                return (value1 === ErrorCode.NA) ? value2 : value1;
            }
        },

        NOT: {
            category: 'logical',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:bool',
            resolve: function (value) { return !value; }
        },

        OR: {
            category: 'logical',
            minParams: 1,
            type: 'val',
            signature: 'any',
            // empty parameters count as FALSE (but do not skip to catch empty-only parameters)
            resolve: implementLogicalAggregation(false, logicalOr, SKIP_MAT_SKIP_REF)
        },

        TRUE: {
            category: 'logical',
            minParams: 0,
            maxParams: 0,
            type: 'val',
            resolve: _.constant(true)
        },

        XOR: {
            category: 'logical',
            name: { ooxml: '_xlfn.XOR' },
            minParams: 1,
            type: 'val',
            signature: 'any',
            // empty parameters count as FALSE (but do not skip to catch empty-only parameters)
            resolve: implementLogicalAggregation(false, logicalXor, SKIP_MAT_SKIP_REF)
        }
    };

});
