/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Oliver Specht <oliver.specht@open-xchange.com>
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/formula/impl/datetimefuncs', [
    'io.ox/office/tk/utils',
    'io.ox/office/spreadsheet/utils/errorcode',
    'io.ox/office/spreadsheet/model/formula/formulautils'
], function (Utils, ErrorCode, FormulaUtils) {

    'use strict';

    /**************************************************************************
     *
     * This module implements all spreadsheet functions dealing with date and
     * time.
     *
     * See the README file in this directory for a detailed documentation about
     * the format of function descriptor objects.
     *
     * In spreadsheet documents, dates and times are represented by
     * floating-point numbers (integral part is number of days, counted from a
     * specific 'null date', fractional part is the time of a day). The
     * client-side formula engine provides the function signature type
     * 'val:date', and uses the JavaScript class Date containing the UTC date
     * and time received from spreadsheet cells, or other functions.
     *
     *************************************************************************/

    // number of hours in a day
    var HOURS_PER_DAY = 24;

    // number of minutes in a day
    var MINUTES_PER_DAY = HOURS_PER_DAY * 60;

    // number of seconds in a day
    var SECONDS_PER_DAY = MINUTES_PER_DAY * 60;

    // number of milliseconds in a day
    var MILLISECONDS_PER_DAY = SECONDS_PER_DAY * 1000;

    // number of milliseconds in a week
    var MILLISECONDS_PER_WEEK = MILLISECONDS_PER_DAY * 7;

    // standard options for date parameter iterators (method FormulaContext.iterateNumbers())

    var RCONVERT_ALL_SKIP_EMPTY = { // id: RRR0
        valMode: 'rconvert', // value operands: convert strings to numbers (but not booleans)
        matMode: 'rconvert', // matrix operands: convert strings to numbers (but not booleans)
        refMode: 'rconvert', // reference operands: convert strings to numbers (but not booleans)
        emptyParam: false, // skip empty parameter
        complexRef: false, // do not accept multi-range and multi-sheet references
        convertToDate: true, // convert numbers to dates
        floor: true // remove time components from all dates
    };

    // private global functions ===============================================

    /**
     * Returns an object containing the date components of the passed date
     * value as single numeric properties.
     *
     * @param {Date} date
     *  The date to be converted, as UTC or local date value.
     *
     * @param {Boolean} [local=false]
     *  Whether to extract the UTC date components (false, default), or the
     *  local date components (true).
     *
     * @returns {Object}
     *  The date components, in the properties 'Y' (full year), 'M' (zero-based
     *  month), and 'D' (one-based day in month).
     */
    function getDateComponents(date, local) {
        return {
            Y: local ? date.getFullYear() : date.getUTCFullYear(),
            M: local ? date.getMonth() : date.getUTCMonth(),
            D: local ? date.getDate() : date.getUTCDate()
        };
    }

    /**
     * Returns an object containing the date and time components of the passed
     * date value as single numeric properties.
     *
     * @param {Date} date
     *  The date to be converted, as UTC or local date value.
     *
     * @param {Boolean} [local=false]
     *  Whether to extract the UTC date/time components (false, default), or
     *  the local date/time components (true).
     *
     * @returns {Object}
     *  The date and time components, in the properties 'Y' (full year), 'M'
     *  (zero-based month), 'D' (one-based day in month), 'h' (hours), 'm'
     *  (minutes), 's' (seconds), and 'ms' (milliseconds).
     */
    function getDateTimeComponents(date, local) {
        var comps = getDateComponents(date, local);
        comps.h = local ? date.getHours() : date.getUTCHours();
        comps.m = local ? date.getMinutes() : date.getUTCMinutes();
        comps.s = local ? date.getSeconds() : date.getUTCSeconds();
        comps.ms = local ? date.getMilliseconds() : date.getUTCMilliseconds();
        return comps;
    }

    /**
     * Creates a UTC date value from the specified date and time components.
     *
     * @param {Object} comps
     *  The date/time components, in the properties 'Y', 'M', 'D', 'h', 'm',
     *  's', and 'ms', as returned by the function getDateTimeComponents().
     *
     * @returns {Date}
     *  The UTC date value representing the specified date and time.
     */
    function makeDateTime(comps) {
        return new Date(Date.UTC(comps.Y, comps.M, comps.D, comps.h, comps.m, comps.s, comps.ms));
    }

    /**
     * Creates a UTC date value from the specified date components.
     *
     * @param {Object} comps
     *  The date components, in the properties 'Y', 'M', and 'D', as returned
     *  by the function getDateComponents().
     *
     * @returns {Date}
     *  The UTC date value representing the specified date. The time will be
     *  set to midnight.
     */
    function makeDate(comps) {
        return new Date(Date.UTC(comps.Y, comps.M, comps.D, 0, 0, 0, 0));
    }

    /**
     * Returns whether the passed number is a leap year.
     *
     * @param {Number} year
     *  The year to be checked.
     *
     * @returns {Boolean}
     *  Whether the passed number is a leap year.
     */
    function isLeapYear(year) {
        return (((year % 4) === 0) && ((year % 100) !== 0)) || ((year % 400) === 0);
    }

    /**
     * Returns the number of days in the specified year.
     *
     * @param {Number} year
     *  The full year.
     *
     * @returns {Number}
     *  The number of days in the specified year.
     */
    function getDaysInYear(year) {
        return isLeapYear(year) ? 366 : 365;
    }

    /**
     * Returns the number of leap years contained in the passed year interval.
     *
     * @param {Number} firstYear
     *  The first year in the year interval (inclusive).
     *
     * @param {Number} lastYear
     *  The last year in the year interval (inclusive). MUST be equal to or
     *  greater than the value passed in firstYear.
     *
     * @returns {Number}
     *  The number of leap years contained in the passed year interval.
     */
    function getLeapYears(firstYear, lastYear) {

        // returns the number of years divisible by the specified value
        function getDivisibleCount(divisor) {
            return Math.floor(lastYear / divisor) - Math.floor((firstYear - 1) / divisor);
        }

        // add the number of years divisible by 4, subtract the number of years divisible by 100, add the number of years divisible by 400
        return getDivisibleCount(4) - getDivisibleCount(100) + getDivisibleCount(400);
    }

    /**
     * Returns the number of days in the month February of the specified year.
     *
     * @param {Number} year
     *  The year.
     *
     * @returns {Number}
     *  The number of days in February of the specified year.
     */
    function getDaysInFebruary(year) {
        return isLeapYear(year) ? 29 : 28;
    }

    /**
     * Returns the number of days in the specified month.
     *
     * @param {Number} year
     *  The year of the month.
     *
     * @param {Number} month
     *  The month. MUST be a zero-based integer (0 to 11).
     *
     * @returns {Number}
     *  The number of days in the specified month.
     */
    var getDaysInMonth = (function () {

        var // number of days per month (regular years)
            DAYS_PER_MONTH = [31, 0, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31];

        return function getDaysInMonth(year, month) {
            // special handling for February in leap years
            return (month === 1) ? getDaysInFebruary(year) : DAYS_PER_MONTH[month];
        };
    }());

    /**
     * Adds the specified number of days to the passed date, and returns the
     * resulting date.
     *
     * @param {Date} date
     *  The original date, as UTC date value.
     *
     * @param {Number} days
     *  The number of days to add. If negative, a date in the past will be
     *  returned accordingly. The number will be truncated to an integer
     *  (always towards zero).
     *
     * @returns {Date}
     *  The new date, as UTC date value.
     */
    function addDaysToDate(date, days) {
        return new Date(date.getTime() + FormulaUtils.trunc(days) * MILLISECONDS_PER_DAY);
    }

    /**
     * Adds the specified number of months to the passed date, and returns the
     * resulting date. Implementation helper for the spreadsheet functions
     * EDATE and EOMONTH.
     *
     * @param {Date} date
     *  The original date, as UTC date value.
     *
     * @param {Number} months
     *  The number of months to add. If negative, a date in the past will be
     *  returned accordingly. The number will be truncated to an integer
     *  (always towards zero).
     *
     * @param {Boolean} end
     *  If set to true, the returned date will be set to the last day of the
     *  resulting month. Otherwise, the day of the passed original date will be
     *  used (unless the resulting month is shorter than the passed day, in
     *  this case, the last valid day of the new month will be used).
     *
     * @returns {Date}
     *  The new date, as UTC date value.
     */
    function addMonthsToDate(date, months, end) {

        var // extract the date components from the passed date
            comps = getDateTimeComponents(date);

        // calculate the new month and year
        comps.M += FormulaUtils.trunc(months);
        comps.Y += Math.floor(comps.M / 12);
        comps.M = Utils.mod(comps.M, 12);

        // validate the day (new month may have less days than the day passed in the original date)
        comps.D = end ? getDaysInMonth(comps.Y, comps.M) : Math.min(comps.D, getDaysInMonth(comps.Y, comps.M));

        // build and return the new date (interpreter will throw #NUM! error, if new year is invalid)
        return makeDateTime(comps);
    }

    /**
     * Returns the number of entire days between the passed start and end date,
     * ignoring the time contained in the passed date objects.
     *
     * @param {Date} startDate
     *  The start date, as UTC date value.
     *
     * @param {Date} endDate
     *  The end date, as UTC date value.
     *
     * @returns {Number}
     *  The number of days between start date and end date (negative, if the
     *  start date is behind the end date).
     */
    function getDaysDiff(startDate, endDate) {
        // ignore time for each date separately (may otherwise distort the result by one day)
        return Math.floor(endDate.getTime() / MILLISECONDS_PER_DAY) - Math.floor(startDate.getTime() / MILLISECONDS_PER_DAY);
    }

    /**
     * Returns the number of days between the passed start and end date for a
     * specific 30/360 day count convention (based on a year with 360 days, and
     * all months with 30 days each).
     *
     * @param {Date} startDate
     *  The start date, as UTC date value.
     *
     * @param {Date} endDate
     *  The end date, as UTC date value.
     *
     * @param {String} method
     *  The day count convention. Must be one of the following values:
     *  - 'eur': The 30/360 European method.
     *  - 'us:days': The 30/360 US method as used by the function DAYS360.
     *  - 'us:frac': The 30/360 US method as used by the function YEARFRAC.
     *
     * @returns {Number}
     *  The number of days between start date and end date (negative, if the
     *  start date is behind the end date).
     */
    function getDays360(startDate, endDate, method) {

        // returns whether the passed date is the last day of February
        function isLastDayOfFeb(comps) {
            return (comps.M === 1) && (comps.D === getDaysInFebruary(comps.Y));
        }

        var // extract original year/month/day of the passed start date
            startComps = getDateComponents(startDate),
            // extract original year/month/day of the passed end date
            endComps = getDateComponents(endDate),
            // the adjusted day of the start and end date
            startDay = 0, endDay = 0;

        switch (method) {

            // The 30/360 European method, as used by the functions DAYS360 and YEARFRAC
            case 'eur':

                // if start date is day 31 of the month, set it to day 30
                startDay = Math.min(startComps.D, 30);
                // if end date is day 31 of the month, set it to day 30
                endDay = Math.min(endComps.D, 30);
                break;

            // The 30/360 US method, as used by the function DAYS360
            case 'us:days':

                // if start date is the last day of its month, set it to day 30 (also for February)
                startDay = isLastDayOfFeb(startComps) ? 30 : Math.min(startComps.D, 30);
                // if end date is day 31 of its month, and adjusted start day is 30,
                // set end day it to day 30 (28th/29th of February remains as is)
                endDay = (startDay === 30) ? Math.min(endComps.D, 30) : endComps.D;
                break;

            // The 30/360 US method, as used by the function YEARFRAC (slightly different than DAYS360, of course)
            case 'us:frac':

                var // whether start date is the last day in February
                    startLastFeb = isLastDayOfFeb(startComps),
                    // whether end date is the last day in February
                    endLastFeb = isLastDayOfFeb(endComps);

                // if start date is the last day of its month, set it to day 30 (also for February)
                startDay = startLastFeb ? 30 : Math.min(startComps.D, 30);
                // if (start date and end date are last day of February), OR if (original start day
                // is 30 or 31, and original end day is 31), set end date to day 30
                endDay = ((startLastFeb && endLastFeb) || ((startComps.D >= 30) && (endComps.D === 31))) ? 30 : endComps.D;
                break;

            default:
                FormulaUtils.throwInternal('DateTimeFuncs.getDays360(): unknown day convention code: ' + method);
        }

        // the resulting number of days, based on a year with equally-sized months of 30 days each
        return (endComps.Y - startComps.Y) * 360 + (endComps.M - startComps.M) * 30 + endDay - startDay;
    }

    /**
     * Implementation helper for the function YEARFRAC for the actual/actual
     * date mode.
     *
     * @param {Date} startDate
     *  The start date, as UTC date value.
     *
     * @param {Date} endDate
     *  The end date, as UTC date value. MUST be equal to or larger than the
     *  date passed to startDate.
     *
     * @returns {Number}
     *  The number of years between start date and end date, as floating-point
     *  number, according to the actual/actual day count convention as used by
     *  the function YEARFRAC.
     */
    function getYearFracActAct(startDate, endDate) {

        // returns whether first day/month is equal to or before second day/month (assuming same year)
        function lessOrEqDate(month1, day1, month2, day2) {
            return (month1 < month2) || ((month1 === month2) && (day1 <= day2));
        }

        var // exact number of days
            days = getDaysDiff(startDate, endDate),
            // extract original year/month/day of the passed start date
            startComps = getDateComponents(startDate),
            // extract original year/month/day of the passed end date
            endComps = getDateComponents(endDate),
            // effective number of days per year
            daysPerYear = 365;

        // special behavior if end date is at most one year behind start date (regardless of actual day and month)
        if ((startComps.Y === endComps.Y) || ((startComps.Y + 1 === endComps.Y) && lessOrEqDate(endComps.M, endComps.D, startComps.M, startComps.D))) {

            // YEARFRAC uses 366 days per year, if (start date is located in a leap year, and is
            // before Mar-01), OR if (end date is located in a leap year, and is after Feb-28)
            if ((isLeapYear(startComps.Y) && (startComps.M <= 1)) || (isLeapYear(endComps.Y) && lessOrEqDate(1, 29, endComps.M, endComps.D))) {
                daysPerYear += 1;
            }

        // otherwise: end date is more than one year after start date (including e.g. 2011-02-28 to 2012-02-29)
        } else {

            // include the number of leap years contained in the year interval as fractional part
            daysPerYear += getLeapYears(startComps.Y, endComps.Y) / (endComps.Y - startComps.Y + 1);
        }

        // divide exact number of days by the effective number of days per year
        return days / daysPerYear;
    }

    /**
     * Implementation helper for the sheet functions WEEKNUM and ISOWEEKNUM.
     *
     * @param {Date} date
     *  The date the week number will be calculated for, as UTC date value.
     *
     * @param {Number} firstDayOfWeek
     *  Determines the start day of the week: 0 for Sunday, 1 for Monday, ...,
     *  or 6 for Saturday.
     *
     * @param {Boolean} iso8601
     *  Whether to use ISO 8601 mode (week of January 1st is the first week, if
     *  at least 4 days are in the new year), or US mode (weeks are counted
     *  strictly inside the year, first and last week in the year may consist
     *  of less than 7 days).
     *
     * @returns {Number}
     *  The resulting week number for the passed date.
     *
     * @throws {ErrorCode}
     *  The #NUM! error code, if the date mode is not valid.
     */
    function getWeekNum(date, firstDayOfWeek, iso8601) {

        // returns the date of the first day of the first week, according to specified first day in week
        function getNullDateOfYear(year) {

            var // base date (that is always in the first week of the year; ISO: Jan-04, otherwise: Jan-01)
                baseDate = makeDate({ Y: year, M: 0, D: iso8601 ? 4 : 1 }),
                // week day of the base date
                baseDay = baseDate.getUTCDay();

            // first day of first week in the year, according to date mode
            return addDaysToDate(baseDate, -Utils.mod(baseDay - firstDayOfWeek, 7));
        }

        var // the year of the passed date
            currYear = date.getUTCFullYear(),
            // first day of first week in the year, according to date mode
            nullDate = getNullDateOfYear(currYear);

        // adjustment for ISO mode: passed date may be counted for the previous year
        // (for Jan-01 to Jan-03), or for the next year (for Dec-29 to Dec-31)
        if (iso8601) {
            if (date < nullDate) {
                // passed date is counted for previous year (e.g. Jan-01-2000 is in week 52 of 1999)
                nullDate = getNullDateOfYear(currYear - 1);
            } else {
                var nextNullDate = getNullDateOfYear(currYear + 1);
                if (nextNullDate <= date) {
                    // passed date is counted for next year (e.g. Dec-31-2001 is in week 1 of 2002)
                    nullDate = nextNullDate;
                }
            }
        }

        return Math.floor((date - nullDate) / MILLISECONDS_PER_WEEK) + 1;
    }

    // date diff helpers ----------------------------------------------------

    function getWeekDay(date) {
        return (date.getUTCDay() + 6) % 7;
    }

    function getWeeksDiff(startDate, endDate) {
        return Math.floor(getDaysDiff(startDate, endDate) / 7);
    }

    function getWeekStart(date) {
        return addDaysToDate(date, -getWeekDay(date));
    }

    function getMonthsDiff(startDate, endDate) {
        var diff = (12 * (endDate.getUTCFullYear() - startDate.getUTCFullYear())) + (endDate.getUTCMonth() - startDate.getUTCMonth());
        if (diff > 0 && endDate.getUTCDate() < startDate.getUTCDate()) {
            diff -= 1;
        }
        return diff;
    }

    function getYearStart(date) {
        return new Date(Date.UTC(date.getUTCFullYear(), 0, 0));
    }

    function getYearsDiff(startDate, endDate) {
        return Math.floor(getMonthsDiff(startDate, endDate) / 12);
    }

    function getMonthStart(date) {
        return new Date(Date.UTC(date.getUTCFullYear(), date.getUTCMonth(), 0));
    }

    function getDiff(options) {
        var startDate = options.startDate;
        var endDate = options.endDate;
        var mode = options.mode;
        var startFn = options.startFn;
        var diffFn = options.diffFn;

        var prefix = 1;

        // order start and end date
        if (endDate < startDate) {
            var tmpDate = startDate;
            startDate = endDate;
            endDate = tmpDate;
            prefix = -1;
        }

        if (mode === 1) {
            return prefix * diffFn(startFn(startDate), startFn(endDate));
        } else {
            return prefix * diffFn(startDate, endDate);
        }
    }

    // workday helpers ----------------------------------------------------

    var WEEKEND_CODES = { 1: /* default */ '0000011', 2: '1000001', 3: '1100000', 4: '0110000', 5: '0011000', 6: '0001100', 7: '0000110', 11: '0000001', 12: '1000000', 13: '0100000', 14: '0010000', 15: '0001000', 16: '0000100', 17: '0000010' };

    function getWeekendFromParam(weekend) {
        if (_.isNull(weekend) || _.isUndefined(weekend)) {
            weekend = WEEKEND_CODES['1'];
        } else if (_.isBoolean(weekend)) {
            if (weekend) {
                weekend = WEEKEND_CODES['1'];
            } else {
                throw ErrorCode.NUM;
            }
        } else if (_.isNumber(weekend)) {
            weekend = WEEKEND_CODES[weekend];
            if (_.isUndefined(weekend)) {
                throw ErrorCode.NUM;
            }
        }
        if (_.isUndefined(weekend)) {
            throw ErrorCode.VALUE;
        } else if (weekend.length === 7 && weekend.match(/0|1/g).length === 7) {
            //weekend code
        } else {
            throw ErrorCode.VALUE;
        }
        return weekend;
    }

    function getWorkDaysPerWeek(weekendCode) {
        return (weekendCode.match(/0/g) || '').length;
    }

    function getWorkPieces(startDate, endDate, holidays) {
        //FIXME: parse to full days

        var pieces = [];
        holidays = _.filter(holidays, function (holiday) {
            return holiday >= startDate && holiday <= endDate;
        });

        holidays.sort();

        var lastWorkDay = startDate;
        for (var i = 0; i < holidays.length; i++) {
            var holiday = holidays[i];
            if (lastWorkDay && getDaysDiff(lastWorkDay, holiday) > 0) {
                pieces.push({ from: lastWorkDay, to: addDaysToDate(holiday, -1) });
                lastWorkDay = addDaysToDate(holiday, 1);
            } else {
                lastWorkDay = null;
            }
        }
        if (!lastWorkDay) {
            lastWorkDay = addDaysToDate(_.last(holidays), 1);
        }
        if (getDaysDiff(lastWorkDay, endDate) >= 0) {
            pieces.push({ from: lastWorkDay, to: endDate });
        }
        //console.warn('getWorkPieces', pieces);
        return pieces;
    }

    function getWorkDays(startDate, endDate, weekend, workDaysPerWeek) {
        var startWeekDay = getWeekDay(startDate);
        var endWeekDay = getWeekDay(endDate);

        var startMonday = getWeekStart(startDate);
        var endMonday = getWeekStart(endDate);

        var workDays = 0;
        var i = 0;

        if (getDaysDiff(startMonday, endMonday) === 0) {
            //same week and also one timeslot
            for (i = startWeekDay; i <= endWeekDay; i++) {
                if (weekend[i] === '0') {
                    workDays++;
                }
            }
        } else {
            //first week slot
            for (i = startWeekDay; i <= 6; i++) {
                if (weekend[i] === '0') {
                    workDays++;
                }
            }

            //last week slot
            for (i = 0; i <= endWeekDay; i++) {
                if (weekend[i] === '0') {
                    workDays++;
                }
            }

            //all weeks between
            var endSunday = addDaysToDate(endMonday, -1);
            var weeksDiff = getWeeksDiff(startMonday, endSunday);
            workDays += weeksDiff * workDaysPerWeek;

        }
        return workDays;
    }

    function findNextWorkday(startDate, endDate, weekend, workDaysPerWeek, prefix, daysCounter) {
        //console.warn('findNextWorkday', startDate, endDate);
        if (prefix < 0) {
            var tmpDate = startDate;
            startDate = endDate;
            endDate = tmpDate;
        }

        //---- begin of perf optimization ---------

        var sliceWeeks = Math.floor(getDaysDiff(startDate, endDate) / 7) * prefix;
        var restWeeks = Math.floor(daysCounter.value / workDaysPerWeek);

        var weeks = Math.abs(sliceWeeks) < Math.abs(restWeeks) ? sliceWeeks : restWeeks;

        if (weeks < -1 || weeks > 1) {

            //console.warn('weeks before', startDate, endDate, 'dayCount', daysCounter.value);

            startDate = addDaysToDate(startDate, (weeks - prefix) * 7);
            daysCounter.value -= (weeks - prefix) * workDaysPerWeek;

            //console.warn('weeks after', startDate, endDate, 'dayCount', daysCounter.value);
        }

        //---- end of perf optimization ---------

        var dayCount = getDaysDiff(startDate, endDate) * prefix;

        //console.warn('dayCount', startDate, endDate, 'dayCount', dayCount);

        for (var i = 0; i <= dayCount; i++) {
            var currentDate = addDaysToDate(startDate, i * prefix);
            var currentWeekDay = getWeekDay(currentDate);
            if (weekend[currentWeekDay] === '0') {
                daysCounter.value -= prefix;
                if (daysCounter.value === 0) {
                    return currentDate;
                }
            }
        }

        //console.warn('no found', startDate, endDate, daysCounter.value);
    }

    function networkDaysIntl(startDate, endDate, weekendParam, holidaysParam) {
        var weekend = getWeekendFromParam(weekendParam);
        var workDaysPerWeek = getWorkDaysPerWeek(weekend);
        var holidays = this.getNumbersAsArray(holidaysParam, RCONVERT_ALL_SKIP_EMPTY);

        if (workDaysPerWeek === 0) {
            return 0;
        }
        var prefix = 1;

        // order start and end date
        if (endDate < startDate) {
            var tmpDate = startDate;
            startDate = endDate;
            endDate = tmpDate;
            prefix = -1;
        }

        var workDays = 0;

        if (holidays.length) {
            var workPieces = getWorkPieces(startDate, endDate, holidays);
            _.each(workPieces, function (workPiece) {
                workDays += getWorkDays(workPiece.from, workPiece.to, weekend, workDaysPerWeek);
            });
        } else {
            workDays += getWorkDays(startDate, endDate, weekend, workDaysPerWeek);
        }
        return workDays * prefix;
    }

    function workDayIntl(startDate, days, weekendParam, holidaysParam) {

        var weekend = getWeekendFromParam(weekendParam);
        var workDaysPerWeek = getWorkDaysPerWeek(weekend);
        var holidays = this.getNumbersAsArray(holidaysParam, RCONVERT_ALL_SKIP_EMPTY);

        //console.warn('workDayIntl', startDate, days, weekend, holidays);

        if (workDaysPerWeek === 0) {
            throw ErrorCode.VALUE;
        }
        if (days === 0) {
            return startDate;
        }
        var prefix = 1;
        var endDate = null;
        var daysCounter = {
            value: days
        };

        var holidayCount = holidays.length;
        var maxDays = Math.ceil((Math.abs(days) / workDaysPerWeek) * 7) + holidayCount;

        //the safe way
        maxDays *= 2;

        // order start and end date
        if (days < 0) {
            prefix = -1;
            endDate = startDate;
            startDate = addDaysToDate(startDate, maxDays * prefix);
            //first day is always ignored!!!
            endDate = addDaysToDate(endDate, prefix);
        } else {
            endDate = addDaysToDate(startDate, maxDays * prefix);
            //first day is always ignored!!!
            startDate = addDaysToDate(startDate, prefix);
        }

        var workPieces = null;
        if (holidays) {
            workPieces = getWorkPieces(startDate, endDate, holidays);
            if (prefix < 0) {
                workPieces.reverse();
            }
        } else {
            workPieces = [{ from: startDate, to: endDate }];
        }

        for (var i = 0; i < workPieces.length; i++) {
            var workPiece = workPieces[i];
            var result = findNextWorkday(workPiece.from, workPiece.to, weekend, workDaysPerWeek, prefix, daysCounter);
            if (result) {
                return result;
            }
        }
        throw ErrorCode.REF;
    }

    // exports ================================================================

    return {

        DATE: {
            category: 'datetime',
            minParams: 3,
            maxParams: 3,
            type: 'val',
            signature: 'val:int val:int val:int',
            resolve: function (year, month, day) {
                // Excel adds 1900 to all years less than 1900; e.g. 1899 becomes 3799 (TODO: different behavior for ODF?)
                if (year < 1900) { year += 1900; }
                // create a UTC date object (Date object expects zero-based month)
                return makeDate({ Y: year, M: month - 1, D: day });
            }
        },

        DATEDIF: {
            category: 'datetime',
            ceName: null,
            hidden: true,
            minParams: 3,
            maxParams: 3,
            type: 'val'
        },

        DATEVALUE: {
            category: 'datetime',
            minParams: 1,
            maxParams: 1,
            type: 'val'
        },

        DAY: {
            category: 'datetime',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:date',
            resolve: function (date) { return date.getUTCDate(); }
        },

        DAYS: {
            category: 'datetime',
            name: { ooxml: '_xlfn.DAYS' },
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: 'val:date val:date',
            resolve: function (endDate, startDate) {
                // DAYS() expects end date before start date
                return getDaysDiff(startDate, endDate);
            }
        },

        DAYS360: {
            category: 'datetime',
            minParams: 2,
            maxParams: 3,
            type: 'val',
            signature: 'val:date val:date val:bool',
            resolve: function (startDate, endDate, euroMode) {
                // euroMode missing or FALSE: use 30/360 US mode
                return getDays360(startDate, endDate, euroMode ? 'eur' : 'us:days');
            }
        },

        DAYSINMONTH: {
            category: 'datetime',
            name: { ooxml: null, odf: 'ORG.OPENOFFICE.DAYSINMONTH' },
            ceName: { ooxml: null },
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:date',
            resolve: function (date) {
                return getDaysInMonth(date.getUTCFullYear(), date.getUTCMonth());
            }
        },

        DAYSINYEAR: {
            category: 'datetime',
            name: { ooxml: null, odf: 'ORG.OPENOFFICE.DAYSINYEAR' },
            ceName: { ooxml: null },
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:date',
            resolve: function (date) {
                return getDaysInYear(date.getUTCFullYear());
            }
        },

        EASTERSUNDAY: {
            category: 'datetime',
            // AOO 4.4 writes 'EASTERSUNDAY' (https://bz.apache.org/ooo/show_bug.cgi?id=126519)
            name: { ooxml: null, odf: ['ORG.OPENOFFICE.EASTERSUNDAY', 'EASTERSUNDAY'] },
            ceName: { ooxml: null },
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:int',
            resolve: function (year) {

                // adjust two-digit year, check validity (TODO: use file configuration for threshold)
                if ((year >= 0) && (year <= 29)) {
                    year += 2000;
                } else if ((year >= 30) && (year <= 99)) {
                    year += 1900;
                }

                // check validity (TODO: correct error code?)
                if ((year < 1583) || (year > 9999)) { throw ErrorCode.NUM; }

                // implementation of the Gauss algorithm, adapted from OOo source code (sc/source/core/tool/interpr2.cxx)
                var n = year % 19,
                    b = Math.floor(year / 100),
                    c = year % 100,
                    d = Math.floor(b / 4),
                    e = b % 4,
                    f = Math.floor((b + 8) / 25),
                    g = Math.floor((b - f + 1) / 3),
                    h = (19 * n + b - d - g + 15) % 30,
                    i = Math.floor(c / 4),
                    k = c % 4,
                    l = (32 + 2 * e + 2 * i - h - k) % 7,
                    m = Math.floor((n + 11 * h + 22 * l) / 451),
                    o = h + l - 7 * m + 114;

                return makeDate({ Y: year, M: Math.floor(o / 31) - 1, D: o % 31 + 1 });
            }
        },

        EDATE: {
            category: 'datetime',
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: 'val:date val:num',
            resolve: function (date, months) {
                // 2nd parameter will be truncated (negative toward zero)
                // set day in result to the day of the passed original date
                return addMonthsToDate(date, months, false);
            }
        },

        EOMONTH: {
            category: 'datetime',
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: 'val:date val:num',
            resolve: function (date, months) {
                // 2nd parameter will be truncated (negative toward zero)
                // set day in result to the last day in the resulting month
                return addMonthsToDate(date, months, true);
            }
        },

        HOUR: {
            category: 'datetime',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:date',
            resolve: function (date) { return date.getUTCHours(); }
        },

        ISLEAPYEAR: {
            category: 'datetime',
            name: { ooxml: null, odf: 'ORG.OPENOFFICE.ISLEAPYEAR' },
            ceName: { ooxml: null },
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:date',
            resolve: function (date) {
                return isLeapYear(date.getUTCFullYear());
            }
        },

        ISOWEEKNUM: {
            // called WEEKNUM in the UI of AOO/LO
            category: 'datetime',
            name: { ooxml: '_xlfn.ISOWEEKNUM' },
            ceName: { odf: 'WEEKNUM' },
            minParams: { ooxml: 1, odf: 2 },
            maxParams: { ooxml: 1, odf: 2 },
            type: 'val',
            signature: 'val:date val:int',
            resolve: function (date, mode) {
                // ODF: date mode 1 for Sunday, every other value for Monday; OOXML: always Monday
                return getWeekNum(date, (mode === 1) ? 0 : 1, true);
            }
        },

        MINUTE: {
            category: 'datetime',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:date',
            resolve: function (date) { return date.getUTCMinutes(); }
        },

        MONTH: {
            category: 'datetime',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:date',
            resolve: function (date) { return date.getUTCMonth() + 1; } // JS Date class returns zero-based month
        },

        MONTHS: {
            category: 'datetime',
            name: { ooxml: null, odf: 'ORG.OPENOFFICE.MONTHS' },
            ceName: { ooxml: null },
            minParams: 3,
            maxParams: 3,
            type: 'val',
            signature: 'val:date val:date val:int',
            resolve: function (startDate, endDate, mode) {
                return getDiff({
                    startDate: startDate,
                    endDate: endDate,
                    mode: mode,
                    startFn: getMonthStart,
                    diffFn: getMonthsDiff
                });
            }
        },

        NETWORKDAYS: {
            category: 'datetime',
            minParams: 2,
            maxParams: 3,
            type: 'val',
            signature: 'val:day val:day any',
            resolve: function (startDate, endDate, holidays) {
                return networkDaysIntl.call(this, startDate, endDate, 1, holidays);
            }
        },

        'NETWORKDAYS.INTL': {
            category: 'datetime',
            name: { odf: 'COM.MICROSOFT.NETWORKDAYS.INTL' },
            ceName: null,
            minParams: 2,
            maxParams: 4,
            type: 'val',
            signature: 'val:day val:day val any',
            resolve: function (startDate, endDate, weekend, holidays) {
                return networkDaysIntl.call(this, startDate, endDate, weekend, holidays);
            }
        },

        NOW: {
            category: 'datetime',
            minParams: 0,
            maxParams: 0,
            type: 'val',
            volatile: true,
            resolve: function () {
                // create a UTC date object representing the current local date/time
                return makeDateTime(getDateTimeComponents(new Date(), true));
            }
        },

        SECOND: {
            category: 'datetime',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:date',
            resolve: function (date) { return date.getUTCSeconds(); }
        },

        TIME: {
            category: 'datetime',
            minParams: 3,
            maxParams: 3,
            type: 'val',
            signature: 'val:int val:int val:int',
            resolve: function (hour, minute, second) {

                var // the time value, as fraction of a day
                    time = (hour / HOURS_PER_DAY + minute / MINUTES_PER_DAY + second / SECONDS_PER_DAY) % 1;

                // must not result in a negative value
                if (time < 0) { throw ErrorCode.NUM; }

                // convert to a date object, using the document's current null date
                return this.convertToDate(time);
            }
        },

        TIMEVALUE: {
            category: 'datetime',
            minParams: 1,
            maxParams: 1,
            type: 'val'
        },

        TODAY: {
            category: 'datetime',
            minParams: 0,
            maxParams: 0,
            type: 'val',
            volatile: true,
            resolve: function () {
                // create a UTC date object representing the current local date (but not the time)
                return makeDate(getDateComponents(new Date(), true));
            }
        },

        WEEKDAY: {
            category: 'datetime',
            minParams: 1,
            maxParams: 2,
            type: 'val',
            signature: 'val:date val:int',
            resolve: function (date, mode) {

                var // weekday index, from 0 to 6, starting at Sunday
                    weekDay = date.getUTCDay();

                // missing parameter defaults to 1 (but: empty parameter defaults to 0)
                switch (_.isUndefined(mode) ? 1 : mode) {
                    case 1:
                        return weekDay + 1; // 1 to 7, starting at Sunday
                    case 2:
                        return (weekDay === 0) ? 7 : weekDay; // 1 to 7, starting at Monday
                    case 3:
                        return (weekDay + 6) % 7; // 0 to 6, starting at Monday
                }

                // other modes result in #NUM! error
                throw ErrorCode.NUM;
            }
        },

        WEEKNUM: {
            // called WEEKNUM_ADD in the UI of AOO/LO
            category: 'datetime',
            ceName: { odf: null }, // { odf: 'WEEKNUM_ADD' } TODO: CalcEngine should allow to use this function in ODF
            minParams: 1,
            maxParams: 2,
            type: 'val',
            signature: 'val:date val:int',
            resolve: (function () {

                var // maps date modes to first day in week (zero-based, starting from Sunday)
                    FIRST_WEEKDAY = { 1: 0, 2: 1, 11: 1, 12: 2, 13: 3, 14: 4, 15: 5, 16: 6, 17: 0, 21: 1 };

                return function (date, mode) {

                    var // first day in the week, according to passed date mode
                        firstDayOfWeek = FIRST_WEEKDAY[_.isNumber(mode) ? mode : 1];

                    // bail out if passed date mode is invalid
                    if (!_.isNumber(firstDayOfWeek)) { throw ErrorCode.NUM; }

                    return getWeekNum(date, firstDayOfWeek, mode === 21);
                };
            }())
        },

        WEEKS: {
            category: 'datetime',
            name: { ooxml: null, odf: 'ORG.OPENOFFICE.WEEKS' },
            ceName: { ooxml: null },
            minParams: 3,
            maxParams: 3,
            type: 'val',
            signature: 'val:date val:date val:int',
            resolve: function (startDate, endDate, mode) {
                return getDiff({
                    startDate: startDate,
                    endDate: endDate,
                    mode: mode,
                    startFn: getWeekStart,
                    diffFn: getWeeksDiff
                });
            }
        },

        WEEKSINYEAR: {
            category: 'datetime',
            name: { ooxml: null, odf: 'ORG.OPENOFFICE.WEEKSINYEAR' },
            ceName: { ooxml: null },
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:date',
            resolve: function (date) {
                // the 28th of December has always maximum week number in ISO8601 mode
                return getWeekNum(makeDate({ Y: date.getUTCFullYear(), M: 11, D: 28 }), 1, true);
            }
        },

        WORKDAY: {
            category: 'datetime',
            minParams: 2,
            maxParams: 3,
            type: 'val',
            signature: 'val:day val:int val',
            resolve: function (startDate, days, holidays) {
                return workDayIntl.call(this, startDate, days, undefined, holidays);
            }
        },

        'WORKDAY.INTL': {
            category: 'datetime',
            name: { odf: 'COM.MICROSOFT.WORKDAY.INTL' },
            ceName: null,
            minParams: 2,
            maxParams: 4,
            type: 'val',
            signature: 'val:day val:int val:int val',
            resolve: function (startDate, days, weekend, holidays) {
                return workDayIntl.call(this, startDate, days, weekend, holidays);
            }
        },

        YEAR: {
            category: 'datetime',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:date',
            resolve: function (date) { return date.getUTCFullYear(); }
        },

        YEARFRAC: {
            category: 'datetime',
            minParams: 2,
            maxParams: 3,
            type: 'val',
            signature: 'val:date val:date val:int',
            resolve: function (startDate, endDate, mode) {

                // default to mode 0 (30/360 US)
                if (!_.isNumber(mode)) { mode = 0; }

                // order start and end date
                if (endDate < startDate) {
                    var tmpDate = startDate;
                    startDate = endDate;
                    endDate = tmpDate;
                }

                // return fraction according to passed mode
                switch (mode) {
                    case 0: // 30/360 US mode, special handling for last days in February
                        return getDays360(startDate, endDate, 'us:frac') / 360;
                    case 1: // actual/actual mode
                        return getYearFracActAct(startDate, endDate);
                    case 2: // actual/360 mode
                        return getDaysDiff(startDate, endDate) / 360;
                    case 3: // actual/365 mode
                        return getDaysDiff(startDate, endDate) / 365;
                    case 4: // 30/360 European mode
                        return getDays360(startDate, endDate, 'eur') / 360;
                }

                throw ErrorCode.NUM;
            }
        },

        YEARS: {
            category: 'datetime',
            name: { ooxml: null, odf: 'ORG.OPENOFFICE.YEARS' },
            ceName: { ooxml: null },
            minParams: 3,
            maxParams: 3,
            type: 'val',
            signature: 'val:date val:date val:int',
            resolve: function (startDate, endDate, mode) {
                return getDiff({
                    startDate: startDate,
                    endDate: endDate,
                    mode: mode,
                    startFn: getYearStart,
                    diffFn: getYearsDiff
                });
            }
        }
    };

});
