/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/formula/formulacontext', [
    'io.ox/office/tk/utils',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/model/formula/formulautils',
    'io.ox/office/spreadsheet/model/formula/complex',
    'io.ox/office/spreadsheet/model/formula/matrix',
    'io.ox/office/spreadsheet/model/formula/operand'
], function (Utils, SheetUtils, FormulaUtils, Complex, Matrix, Operand) {

    'use strict';

    // convenience shortcuts
    var ErrorCode = SheetUtils.ErrorCode,
        Address = SheetUtils.Address,
        Range3D = SheetUtils.Range3D,
        Range3DArray = SheetUtils.Range3DArray;

    // class FormulaContext ===================================================

    /**
     * An instance of this class serves as calling context object for the
     * implementations of operators and functions, so that the symbol 'this'
     * inside these implementations provides various useful helper methods.
     *
     * @constructor
     *
     * @param {SpreadsheetModel} docModel
     *  The spreadsheet document model containing this instance.
     *
     * @param {Number|Null} refSheet
     *  The index of the sheet the interpreted formula is related to. Used to
     *  resolve reference tokens without sheet reference.
     *
     * @param {Address} targetAddress
     *  The address of the target reference cell in the specified reference
     *  sheet the formula is related to.
     */
    function FormulaContext(docModel, refSheet, targetAddress) {

        var // self reference
            self = this,

            // the application instance
            app = docModel.getApp(),

            // the file format identifier for different behavior of functions and operators
            fileFormat = app.getFileFormat(),

            // the grammar configuration for the UI language
            grammarConfig = docModel.getGrammarConfig('ui'),

            // the number formatter
            numberFormatter = docModel.getNumberFormatter(),

            // maximum length of a string, maximum valid string index (one-based)
            MAX_STRING_LEN = (fileFormat === 'odf') ? 65535 : 32767,

            // options for the method FormulaUtils.compareValues(), according to file format
            COMPARE_OPTIONS = { withCase: fileFormat === 'odf', convertNull: true },

            // the array of operands (instance of class Operand) of the current function/operator
            currOperands = null;

        // private methods ----------------------------------------------------

        /**
         * Throws a CIRCULAR_ERROR error code, if the passed cell range address
         * contains the reference cell of this formula context.
         *
         * @param {Range3D} range
         *  The cell range address to be checked.
         *
         * @returns {Range3D}
         *  The passed cell range address, if it does not contain the reference
         *  cell.
         *
         * @throws {ErrorCode}
         *  The CIRCULAR_ERROR error code, if the passed cell range address
         *  contains the reference cell of this formula context.
         */
        function checkCircularReference(range) {
            if (_.isNumber(refSheet) && range.containsSheet(refSheet) && range.containsAddress(targetAddress)) {
                throw FormulaUtils.CIRCULAR_ERROR;
            }
            return range;
        }

        /**
         * Returns the type of the passed operand.
         */
        function getOperandType(operand) {
            if (operand instanceof Operand) { return operand.getType(); }
            if (operand instanceof Matrix) { return 'mat'; }
            if ((operand instanceof Range3D) || (operand instanceof Range3DArray)) { return 'ref'; }
            return 'val';
        }

        /**
         * Visits all single values contained in the passed operand, matrix, or
         * reference (invokes the callback function with the result of the
         * value converter).
         */
        function iterateOperandValues(operand, converter, callback, options) {

            var // whether to visit the empty function parameter
                emptyParam = Utils.getBooleanOption(options, 'emptyParam', false),
                // whether to visit empty cells in references
                emptyCell = Utils.getBooleanOption(options, 'emptyCell', false),
                // whether to pass error codes to the iterator
                acceptErrors = Utils.getBooleanOption(options, 'acceptErrors', false),
                // whether to visit cells in references in order
                ordered = Utils.getBooleanOption(options, 'ordered', false),
                // performance: do not visit more than a specific number of cells in cell references
                count = 0;

            // invokes the callback function for a single value
            function invokeCallback(value, col, row) {
                // throw error codes, unless error codes will be accepted
                if (!acceptErrors) { FormulaUtils.throwErrorCode(value); }
                // convert the passed value (may throw an error code)
                value = converter.call(self, value);
                // check maximum number of visited values (for performance)
                count += 1;
                if (count > FormulaUtils.MAX_CELL_ITERATION_COUNT) { throw FormulaUtils.UNSUPPORTED_ERROR; }
                // invoke the callback function
                callback.call(self, value, col, row);
            }

            // visits a single value operand (skip empty parameters unless specified otherwise)
            function iterateValue(value) {
                if (emptyParam || !_.isNull(value)) {
                    invokeCallback(value, 0, 0);
                }
            }

            // visits all elements of a matrix
            function iterateMatrix(matrix) {
                matrix.forEach(function (element, row, col) {
                    invokeCallback(element, col, row);
                });
            }

            // visits all elements of a plain array (simulate a column vector), intended use: test code
            function iterateArray(array) {
                array.forEach(function (element, index) {
                    invokeCallback(element, 0, index);
                });
            }

            // visits all filled cells in the passed cell range address
            function iterateRange(range) {

                var // start position of the iteration range
                    col = range.start[0], row = range.start[1];

                // loop through all sheets referred by the rage
                Utils.iterateRange(range.sheet1, range.sheet2 + 1, function (sheet) {

                    var // the sheet model containing the cell
                        sheetModel = docModel.getSheetModel(sheet);

                    // safety check: #REF! error if passed sheet index is invalid
                    if (!sheetModel) { throw ErrorCode.REF; }

                    // iterate all content cells in the cell collection (skip empty cells)
                    sheetModel.getCellCollection().iterateCellsInRanges(range.toRange(), function (cellData) {
                        invokeCallback(cellData.result, cellData.address[0] - col, cellData.address[1] - row);
                    }, { type: emptyCell ? 'all' : 'content', hidden: 'all', ordered: ordered });
                });
            }

            // visit all filled cells in a cell reference
            function iterateRanges(ranges) {
                if (Utils.getBooleanOption(options, 'complexRef', false)) {
                    Range3DArray.forEach(ranges, function (range) {
                        checkCircularReference(range);
                        iterateRange(range);
                    });
                } else {
                    iterateRange(self.convertToValueRange(ranges));
                }
            }

            // passed operand can also be a matrix or a reference
            if (operand instanceof Matrix) {
                iterateMatrix(operand);
            } else if ((operand instanceof Range3D) || (operand instanceof Range3DArray)) {
                iterateRanges(operand);
            } else if (operand instanceof Operand) {
                if (operand.isMatrix()) {
                    iterateMatrix(operand.getMatrix());
                } else if (operand.isReference()) {
                    iterateRanges(operand.getRanges());
                } else {
                    iterateValue(operand.getValue());
                }
            } else if (_.isArray(operand)) {
                iterateArray(operand);
            } else if (!_.isUndefined(operand)) {
                iterateValue(operand);
            }
        }

        // protected methods --------------------------------------------------

        /**
         * Registers the operands of the next function or operator to be
         * resolved.
         *
         * @attention
         *  This method will be invoked by the formula interpreter to be able
         *  to reuse this context instance for all functions and operators
         *  contained in a complete formula.
         *
         * @param {Array<Operand>} operands
         *  The array of operands for the next function or operator.
         *
         * @returns {FormulaContext}
         *  A reference to this instance.
         */
        this.registerOperands = function (operands) {
            currOperands = operands;
            return this;
        };

        // public methods -----------------------------------------------------

        /**
         * Returns the document model containing the interpreted formula.
         *
         * @returns {SpreadsheetModel}
         *  The document model containing the interpreted formula.
         */
        this.getDocModel = function () {
            return docModel;
        };

        /**
         * Returns the file format identifier of the edited document.
         *
         * @returns {String}
         *  The file format identifier of the edited document.
         */
        this.getFileFormat = function () {
            return fileFormat;
        };

        /**
         * Returns the configuration for the UI formula grammar from the parent
         * model, needed for specific conversion tasks.
         *
         * @returns {GrammarConfig}
         *  The configuration for the UI formula grammar.
         */
        this.getGrammarConfig = function () {
            return grammarConfig;
        };

        /**
         * Returns the number formatter of the document model.
         *
         * @returns {NumberFormatter}
         *  The number formatter of the document model.
         */
        this.getNumberFormatter = function () {
            return numberFormatter;
        };

        /**
         * Returns the number of existing operands passed to the operator or
         * function.
         *
         * @returns {Number}
         *  The number of existing operands.
         */
        this.getOperandCount = function () {
            return currOperands.length;
        };

        /**
         * Returns the specified operand passed to the operator or function.
         *
         * @param {Number} index
         *  The zero-based index of an operand.
         *
         * @returns {Operand}
         *  The specified operand, if existing.
         *
         * @throws {ErrorCode}
         *  The #VALUE! error code, if the operand does not exist.
         */
        this.getOperand = function (index) {
            if ((index >= 0) && (index < currOperands.length)) { return currOperands[index]; }
            throw ErrorCode.VALUE;
        };

        /**
         * Returns whether the specified operand is missing (the optional
         * parameters at the end of a function; e.g. the third parameter in the
         * formula =SUM(1,2) is missing).
         *
         * @param {Number} index
         *  The zero-based index of an operand.
         *
         * @returns {Boolean}
         *  Whether the specified operand is missing. Returns false for
         *  existing but empty operands, e.g. in the formula =SUM(1,,2).
         */
        this.isMissingOperand = function (index) {
            return index >= currOperands.length;
        };

        /**
         * Returns whether the specified operand exists, but is empty (e.g. the
         * second operand in the formula =SUM(1,,2) is empty).
         *
         * @param {Number} index
         *  The zero-based index of an operand.
         *
         * @returns {Boolean}
         *  Whether the specified operand exists but is empty. Returns false
         *  for missing operands (optional parameters at the end of the
         *  function). Returns false for references to empty cells too (the
         *  operand is a reference, and not empty).
         */
        this.isEmptyOperand = function (index) {
            var operand = currOperands[index];
            return _.isObject(operand) && operand.isEmpty();
        };

        /**
         * Returns whether the specified operand is missing (the optional
         * parameters at the end of a function; e.g. the third parameter in the
         * formula =SUM(1,2) is missing), or empty (e.g. the second operand in
         * the formula =SUM(1,,2) is empty).
         *
         * @param {Number} index
         *  The zero-based index of an operand.
         *
         * @returns {Boolean}
         *  Whether the specified operand is missing or empty.
         */
        this.isMissingOrEmptyOperand = function (index) {
            return this.isMissingOperand(index) || this.isEmptyOperand(index);
        };

        /**
         * Returns the specified operands passed to the operator or function as
         * array.
         *
         * @param {Number} index
         *  The zero-based index of the first operand to be returned.
         *
         * @returns {Array<Operand>}
         *  The specified operand and all its successors, if existing.
         *
         * @throws {ErrorCode}
         *  The #VALUE! error code, if the specified operand does not exist.
         */
        this.getOperands = function (index) {
            if ((index >= 0) && (index < currOperands.length)) { return currOperands.slice(index); }
            throw ErrorCode.VALUE;
        };

        /**
         * Visits the existing operands of the processed operator or function.
         *
         * @param {Number} start
         *  Specifies the zero-based index of the first operand to be visited.
         *
         * @param {Function} callback
         *  The callback function that will be invoked for each operand.
         *  Receives the following parameters:
         *  (1) {Operand} operand
         *      The original operand instance.
         *  (2) {Number} index
         *      The zero-based index of the operand.
         *  The callback function will be invoked in the context of this
         *  instance.
         *
         * @returns {OperandsMixin}
         *  A reference to this instance.
         */
        this.iterateOperands = function (start, callback) {
            currOperands.slice(start).forEach(function (operand, index) {
                callback.call(this, operand, start + index);
            }, this);
            return this;
        };

        /**
         * Compares the passed scalar values according to the current file
         * format. For ODF documents, the character case of strings matters.
         * Provided to help implementing the comparison operators.
         *
         * @param {Number|String|Boolean|Null} value1
         *  The first value to be compared.
         *
         * @param {Number|String|Boolean|Null} value2
         *  The second value to be compared.
         *
         * @returns {Number}
         *  - A negative value, if value1 is less than value2.
         *  - A positive value, if value1 is greater than value2.
         *  - Zero, if both values are of the same type and are equal.
         */
        this.compareValues = function (value1, value2) {
            return FormulaUtils.compareValues(value1, value2, COMPARE_OPTIONS);
        };

        /**
         * Converts the passed scalar value to be stored in an operand. A date
         * will be converted to a floating-point number, if possible. If the
         * value is INF or NaN, the #NUM! error code will be returned instead.
         * If the number is too close to zero (using a denormalized binary
         * representation internally), zero will be returned. If a string is
         * too long, a #VALUE! error code will be returned. Complex numbers
         * will be converted to their string representation, unless one of the
         * coefficients is INF or NaN which will result in the #NUM! error too.
         *
         * @param {Any} value
         *  The value to be validated.
         *
         * @returns {Any}
         *  The validated value, ready to be inserted into a formula operand.
         */
        this.validateValue = function (value) {

            // convert date object to floating-point number
            if (value instanceof Date) {
                value = numberFormatter.convertDateToNumber(value);
                return _.isNumber(value) ? value : ErrorCode.NUM;
            }

            // convert complex numbers to strings
            if (value instanceof Complex) {
                // convert INF or NaN to #NUM! error
                if (!_.isFinite(value.real) || !_.isFinite(value.imag)) { return ErrorCode.NUM; }
                // convert very small (denormalized) numbers to zero
                if (Math.abs(value.real) < Utils.MIN_NUMBER) { value.real = 0; }
                if (Math.abs(value.imag) < Utils.MIN_NUMBER) { value.imag = 0; }
                // convert to string
                value = numberFormatter.convertComplexToString(value, SheetUtils.MAX_LENGTH_STANDARD_FORMULA);
            }

            // validate floating-point numbers
            if (_.isNumber(value)) {
                // convert INF or NaN to #NUM! error
                if (!_.isFinite(value)) { return ErrorCode.NUM; }
                // convert very small (denormalized) numbers to zero
                return (Math.abs(value) < Utils.MIN_NUMBER) ? 0 : value;
            }

            // validate strings
            if (_.isString(value)) {
                // check maximum length of string (depends on file format)
                return (value.length <= MAX_STRING_LEN) ? value : ErrorCode.VALUE;
            }

            // no conversion for other values
            return value;
        };

        /**
         * Checks the passed string length used in string functions. Throws the
         * #VALUE! error code, if the index is less than zero, or larger than
         * the maximum length of string results supported by the formula
         * engine.
         *
         * @param {Number} length
         *  The string length to be checked.
         *
         * @throws {ErrorCode}
         *  The #VALUE! error code, if the passed string length is invalid.
         */
        this.checkStringLength = function (length) {
            if ((length < 0) || (length > MAX_STRING_LEN)) { throw ErrorCode.VALUE; }
        };

        /**
         * Checks the passed character index used in string functions. Throws
         * the #VALUE! error code, if the index is less than one, or larger
         * than the maximum length of string results supported by the formula
         * engine.
         *
         * @param {Number} index
         *  The character index to be checked.
         *
         * @throws {ErrorCode}
         *  The #VALUE! error code, if the passed character index is invalid.
         */
        this.checkStringIndex = function (index) {
            if ((index < 1) || (index > MAX_STRING_LEN)) { throw ErrorCode.VALUE; }
        };

        /**
         * Checks the passed complex numbers for their imaginary unit. Throws
         * the #VALUE! error code, if the imaginary units of the numbers are
         * different.
         *
         * @param {Complex} complex1
         *  The first complex number to be checked.
         *
         * @param {Complex} complex2
         *  The second complex number to be checked.
         *
         * @throws {ErrorCode}
         *  The #VALUE! error code, if the imaginary units of the passed
         *  complex numbers are different.
         */
        this.checkComplexUnits = function (complex1, complex2) {
            if (_.isString(complex1.unit) && _.isString(complex2.unit) && (complex1.unit !== complex2.unit)) {
                throw ErrorCode.VALUE;
            }
        };

        /**
         * Returns whether the passed cell reference structure is valid.
         *
         * @param {CellRef} cellRef
         *  The cell reference structure to be checked.
         *
         * @param {Boolean} rcStyle
         *  Whether to check the cell reference according to A1 reference style
         *  (false), or according to R1C1 reference style (true). In A1 style,
         *  the address must point to an existing address regardless of the
         *  absolute flags. In R1C1 style, a relative column or row index may
         *  be negative, but its absolute value must be a valid index.
         *
         * @throws {ErrorCode}
         *  The #VALUE! error code, if the passed cell reference is not valid.
         */
        this.checkCellRef = function (cellRef, rcStyle) {

            // column/row index may be negative for relative index in R1C1 style
            var col = (rcStyle && !cellRef.absCol) ? Math.abs(cellRef.col) : cellRef.col;
            var row = (rcStyle && !cellRef.absRow) ? Math.abs(cellRef.row) : cellRef.row;

            // check the adjusted cell address
            if (!docModel.isValidAddress(new Address(col, row))) { throw ErrorCode.VALUE; }
        };

        /**
         * Returns whether the passed sheet index is equal to the reference
         * sheet the interpreted formula is located in.
         *
         * @param {Number} sheet
         *  A zero-based sheet index to be tested against the reference sheet.
         *
         * @returns {Boolean}
         *  Whether the passed sheet index is equal to the reference sheet.
         */
        this.isRefSheet = function (sheet) {
            return _.isNumber(refSheet) && (refSheet >= 0) && (refSheet === sheet);
        };

        /**
         * Returns the index of the reference sheet the interpreted formula is
         * located in.
         *
         * @returns {Number}
         *  The zero-based index of the reference sheet.
         *
         * @throws {ErrorCode}
         *  The #REF! error code, if the formula is being interpreted without a
         *  reference sheet.
         */
        this.getRefSheet = function () {
            if (!_.isNumber(refSheet)) { throw ErrorCode.REF; }
            return refSheet;
        };

        /**
         * Returns whether the passed cell address is equal to the address of
         * the target reference cell the interpreted formula is located in.
         *
         * @param {Address} address
         *  A cell address to be tested against the target reference address.
         *
         * @returns {Boolean}
         *  Whether the passed cell address is equal to the address of the
         *  target reference cell.
         */
        this.isRefAddress = function (address) {
            return targetAddress.equals(address);
        };

        /**
         * Returns the address of the target reference cell the interpreted
         * formula is located in.
         *
         * @returns {Address}
         *  The address of the target reference cell.
         */
        this.getRefAddress = function () {
            return targetAddress.clone();
        };

        /**
         * Returns the current result value of the specified cell.
         *
         * @param {Number} sheet
         *  The zero-based index of the sheet.
         *
         * @param {Address} address
         *  The address of the cell in the specified sheet.
         *
         * @returns {Number|String|Boolean|ErrorCode|Null}
         *  The current value of the specified cell.
         *
         * @throws {ErrorCode}
         *  The #REF! error code, if the passed sheet index is invalid.
         */
        this.getCellValue = function (sheet, address) {

            // the sheet model containing the cell
            var sheetModel = docModel.getSheetModel(sheet);

            // safety check: #REF! error if passed sheet index is invalid
            if (!sheetModel) { throw ErrorCode.REF; }

            // return typed value from cell collection
            return sheetModel.getCellCollection().getCellResult(address);
        };

        /**
         * Returns the formula string of the specified cell.
         *
         * @param {Number} sheet
         *  The zero-based index of the sheet.
         *
         * @param {Address} address
         *  The address of the cell in the specified sheet.
         *
         * @returns {String|Null}
         *  The current formula string of the specified cell; or null, if the
         *  cell does not contain a formula.
         *
         * @throws {ErrorCode}
         *  The #REF! error code, if the passed sheet index is invalid.
         */
        this.getCellFormula = function (sheet, address) {

            // the sheet model containing the cell
            var sheetModel = docModel.getSheetModel(sheet);

            // safety check: #REF! error if passed sheet index is invalid
            if (!sheetModel) { throw ErrorCode.REF; }

            // get formula string from cell collection
            return sheetModel.getCellCollection().getCellFormula(address);
        };

        /**
         * Tries to convert the passed scalar value to a floating-point number.
         *
         * @param {Any} value
         *  A scalar value to be converted to a number. Dates will be converted
         *  to numbers. Strings that represent a valid number (according to the
         *  current GUI language) will be converted to the number. The boolean
         *  value FALSE will be converted to 0, the boolean value TRUE will be
         *  converted to 1. The special value null (representing an empty cell)
         *  will be converted to 0.
         *
         * @param {Boolean} [floor=false]
         *  If set to true, the number will be rounded down to the next integer
         *  (negative numbers will be rounded down too, i.e. away from zero!).
         *
         * @returns {Number}
         *  The floating-point number, if available.
         *
         * @throws {ErrorCode}
         *  If the passed value is an error code, it will be thrown. If the
         *  value cannot be converted to a floating-point number, a #VALUE!
         *  error will be thrown.
         */
        this.convertToNumber = function (value, floor) {

            // immediately throw error codes
            FormulaUtils.throwErrorCode(value);

            // convert strings and booleans to numbers
            if (_.isString(value)) {
                var parseResult = numberFormatter.parseFormattedNumber(value);
                value = parseResult ? parseResult.number : Number.NaN;
            } else if (_.isBoolean(value)) {
                value = value ? 1 : 0;
            } else if (value instanceof Date) {
                // value may become null which will cause the #VALUE! error below
                value = numberFormatter.convertDateToNumber(value);
            } else if (_.isNull(value)) {
                value = 0;
            }

            // the resulting value must be a finite floating-point number
            if (_.isNumber(value) && isFinite(value)) { return floor ? Math.floor(value) : value; }
            throw ErrorCode.VALUE;
        };

        /**
         * Tries to convert the passed scalar value to a date.
         *
         * @param {Any} value
         *  A scalar value to be converted to a number. Strings that represent
         *  a valid number (according to the current GUI language) will be
         *  converted to a date representing that number. The boolean value
         *  FALSE will be converted to the null date of this formatter, the
         *  boolean value TRUE will be converted to the day following the null
         *  date. The special value null (representing an empty cell) will be
         *  converted to 0.
         *
         * @param {Boolean} [floor=false]
         *  If set to true, the time components of the resulting date will be
         *  removed (the time will be set to midnight).
         *
         * @returns {Date}
         *  The UTC date, if available.
         *
         * @throws {ErrorCode}
         *  If the passed value is an error code, it will be thrown. If the
         *  value cannot be converted to a date value, a #VALUE! error will be
         *  thrown.
         */
        this.convertToDate = function (value, floor) {

            // immediately throw error codes
            FormulaUtils.throwErrorCode(value);

            if (!(value instanceof Date)) {
                // convert the value to a number (failing conversion will throw)
                value = this.convertToNumber(value, floor);
                // convert the number to a date object
                value = numberFormatter.convertNumberToDate(value);
            }

            // the resulting value must be a Date object
            if (value instanceof Date) { return value; }
            throw ErrorCode.VALUE;
        };

        /**
         * Tries to convert the passed scalar value to a string.
         *
         * @param {Any} value
         *  A scalar value to be converted to a string. Numbers and dates will
         *  be converted to decimal or scientific notation (according to the
         *  current GUI language), boolean values will be converted to their
         *  localized text representation (formulas will always be calculated
         *  using the current GUI language). The special value null
         *  (representing an empty cell) will be converted to the empty string.
         *
         * @returns {String}
         *  The string representation of the value, if available.
         *
         * @throws {ErrorCode}
         *  If the passed value is an error code, it will be thrown. All other
         *  values will be converted to strings.
         */
        this.convertToString = function (value) {

            // immediately throw error codes
            FormulaUtils.throwErrorCode(value);

            // treat date objects as plain unformatted numbers
            if (value instanceof Date) {
                value = this.convertToNumber(value);
            }

            // convert numbers and booleans to strings
            if (_.isNumber(value) && isFinite(value)) {
                value = numberFormatter.formatStandardNumber(value, SheetUtils.MAX_LENGTH_STANDARD_FORMULA);
            } else if (_.isBoolean(value)) {
                value = grammarConfig.getBooleanName(value); // always localized
            } else if (_.isNull(value)) {
                value = '';
            }

            // the resulting value must be a string
            if (_.isString(value)) { return value; }
            throw ErrorCode.VALUE;
        };

        /**
         * Tries to convert the passed scalar value to a boolean value.
         *
         * @param {Any} value
         *  A scalar value to be converted to a boolean. Floating-point numbers
         *  will be converted to TRUE if not zero, otherwise FALSE. The null
         *  date will be converted to FALSE, all other dates will be converted
         *  to TRUE. Strings containing the exact translated name of the TRUE
         *  or FALSE values (case-insensitive) will be converted to the
         *  respective boolean value (formulas will always be calculated using
         *  the current GUI language). The special value null (representing an
         *  empty cell) will be converted to FALSE.
         *
         * @returns {Boolean}
         *  The boolean value, if available.
         *
         * @throws {ErrorCode}
         *  If the passed value is an error code, it will be thrown. If the
         *  value cannot be converted to a boolean, a #VALUE! error will be
         *  thrown.
         */
        this.convertToBoolean = function (value) {

            // immediately throw error codes
            FormulaUtils.throwErrorCode(value);

            // treat date objects as plain unformatted numbers
            if (value instanceof Date) {
                value = this.convertToNumber(value);
            }

            // convert numbers and strings to booleans
            if (_.isNumber(value) && isFinite(value)) {
                value = Math.abs(value) >= Utils.MIN_NUMBER;
            } else if (_.isString(value)) {
                // always localized, returns null on failure
                value = grammarConfig.getBooleanValue(value);
            } else if (_.isNull(value)) {
                value = false;
            }

            // the resulting value must be a boolean
            if (_.isBoolean(value)) { return value; }
            throw ErrorCode.VALUE;
        };

        /**
         * Tries to convert the passed scalar value to a complex number.
         *
         * @param {Any} value
         *  A scalar value to be converted to a complex number. Floating-point
         *  numbers will be converted to complex numbers with the imaginary
         *  coefficient set to zero. Strings must represent a valid complex
         *  number (optional leading signed real coefficient, optional trailing
         *  signed imaginary coefficient, followed by the lower-case character
         *  'i' or 'j'). The coefficients must containing the decimal separator
         *  of the current GUI language. The special value null (representing
         *  an empty cell) will be converted to the complex number 0.
         *
         * @returns {Complex}
         *  The complex number, if available.
         *
         * @throws {ErrorCode}
         *  If the passed value is an error code, it will be thrown. Boolean
         *  values will cause to throw a #VALUE! error. If the string cannot be
         *  converted to a complex number, a #NUM! error will be thrown (not
         *  the #VALUE! error code as thrown by the other conversion methods).
         */
        this.convertToComplex = function (value) {

            // immediately throw error codes
            FormulaUtils.throwErrorCode(value);

            // boolean values result in #VALUE! (but not invalid strings)
            if (_.isBoolean(value)) { throw ErrorCode.VALUE; }

            // treat date objects as plain unformatted numbers
            if (value instanceof Date) {
                value = this.convertToNumber(value);
            }

            // convert numbers and strings to complex numbers
            if (_.isNumber(value) && _.isFinite(value)) {
                value = new Complex(value, 0);
            } else if (_.isString(value)) {
                value = numberFormatter.convertStringToComplex(value);
            } else if (_.isNull(value)) {
                // empty cells are treated as complex number 0 (in difference to empty strings)
                value = new Complex(0, 0);
            }

            // the resulting value must be a complex number
            if (value instanceof Complex) { return value; }
            // invalid strings will result in #NUM! instead of #VALUE!
            throw ErrorCode.NUM;
        };

        /**
         * Extracts a single cell range from the passed array of cell range
         * addresses. The range array must contain exactly one cell range
         * address. Unless specified via an option, the cell range must refer
         * to a single sheet.
         *
         * @param {Range3DArray|Range3D} ranges
         *  An array of cell range addresses, or a single cell range address.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.multiSheet=false]
         *      If set to true, the cell range address may refer to multiple
         *      sheets.
         *  @param {Boolean} [options.valueError=false]
         *      If set to true, a #VALUE! error code (instead of a #REF! error
         *      code) will be thrown, if the passed range array contains more
         *      than one cell range address.
         *
         * @returns {Range3D}
         *  The cell range address, if available.
         *
         * @throws {ErrorCode}
         *  - The #NULL! error code, if the range array is empty.
         *  - The #REF! or #VALUE! error code (see option 'valueError'), if the
         *      range array contains more than one element.
         *  - The #VALUE! error code, if the option 'multiSheet' has not been
         *      set, and a single cell range address is available but refers to
         *      multiple sheets.
         */
        this.convertToRange = function (ranges, options) {

            // must not be empty
            if (ranges.empty()) { throw ErrorCode.NULL; }

            // must be a single range in a single sheet
            if (ranges.length > 1) {
                throw Utils.getBooleanOption(options, 'valueError', false) ? ErrorCode.VALUE : ErrorCode.REF;
            }

            // must point to a single sheet, unless multiple sheets are allowed (always throw the #VALUE! error code)
            if (!Utils.getBooleanOption(options, 'multiSheet', false) && !ranges[0].singleSheet()) {
                throw ErrorCode.VALUE;
            }

            return ranges[0];
        };

        /**
         * Extracts a single cell range from the passed array of cell range
         * addresses, intended to be resolved to the cell contents in that
         * range. The range array must contain exactly one cell range address
         * pointing to a single sheet, and the reference cell of this formula
         * context must not be part of that range.
         *
         * @param {Range3DArray|Range3D} ranges
         *  An array of cell range addresses, or a single cell range address.
         *
         * @returns {Range3D}
         *  The resulting cell range address, if available. The sheet indexes
         *  in that range will be equal.
         *
         * @throws {ErrorCode}
         *  - The #NULL! error code, if the range array is empty.
         *  - The #VALUE! error code, if the range array contains more than one
         *      element, or if the only range refers to multiple sheets.
         *  - The CIRCULAR_ERROR error code, if a single range is available,
         *      but the reference cell is part of that range (a circular
         *      reference).
         */
        this.convertToValueRange = function (ranges) {
            // resolve to a single range (throw #VALUE! on multiple ranges), detect circular references
            return checkCircularReference(this.convertToRange(ranges, { valueError: true }));
        };

        /**
         * Visits all values contained in the passed operand. Provides a single
         * value of a scalar operand, all elements of a matrix operand, and the
         * cell contents referred by a reference operand.
         *
         * @param {Any} operand
         *  The operand (scalar value of any supported data type, including
         *  null and error codes; or instances of the classes Matrix, Range3D,
         *  or Range3DArray; or a plain JS array of scalar values) whose
         *  contents will be visited.
         *
         * @param {Function} callback
         *  The callback function invoked for each value contained in the
         *  operand. Will be called in the context of this instance. Receives
         *  the value as first parameter.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.emptyParam=false]
         *      If set to true, and the operand represents the empty function
         *      parameter, the callback function will be invoked with the value
         *      null. By default, empty parameters will be skipped.
         *  @param {Boolean} [options.acceptErrors=false]
         *      If set to true, the callback function will be invoked for error
         *      codes too. By default, error codes will be thrown immediately.
         *  @param {Boolean} [options.complexRef=false]
         *      If set to true, a reference operand may be complex. Complex
         *      references may contain multiple cell ranges, and each of the
         *      cell ranges may refer to multiple sheets. By default, a
         *      reference must refer to a single range in a single sheet.
         *  @param {Boolean} [options.ordered=false]
         *      If set to true, the cells in each cell range will be visited in
         *      order (row-by-row from top to bottom, and left-to-right inside
         *      each of the rows).
         *
         * @returns {FormulaContext}
         *  A reference to this instance.
         *
         * @throws {ErrorCode}
         *  Any error code contained in the passed operand.
         */
        this.iterateValues = function (operand, callback, options) {
            iterateOperandValues(operand, _.identity, callback, options);
            return this;
        };

        /**
         * Visits all floating-point numbers contained in the passed operand.
         * Provides the value of a scalar operand, all elements of a matrix
         * operand, and the cell contents referred by a reference operand. In
         * difference to simple value parameters where strings and boolean
         * values contained in matrixes and cell ranges will always be
         * converted to numbers if possible, this method implements different
         * conversion strategies as required by the respective functions.
         *
         * @param {Any} operand
         *  The operand (scalar value of any supported data type, including
         *  null and error codes; or instances of the classes Matrix, Range3D,
         *  or Range3DArray; or a plain JS array of scalar values) whose
         *  contents will be visited.
         *
         * @param {Function} callback
         *  The callback function invoked for each floating-point number
         *  contained in the operand. Will be called in the context of this
         *  instance. Receives the number as first parameter.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {String} [options.valMode='convert']
         *      Conversion strategy for operands of type value:
         *      - 'convert': Strings will be converted to numbers, if possible.
         *          Boolean values will be converted to numbers. Strings that
         *          represent a boolean value result in a #VALUE! error code
         *          though.
         *      - 'rconvert' (restricted conversion): Strings will be converted
         *          to numbers, if possible. Boolean values and strings that
         *          represent a boolean value result in a #VALUE! error code
         *          though.
         *      - 'exact': Accepts numbers only. All strings and boolean values
         *          result in a #VALUE! error code.
         *      - 'skip': All strings and all boolean values will be skipped
         *          silently, also strings that could be converted to numbers.
         *      - 'zero': All strings will be treated as the number zero, also
         *          strings that could be converted to numbers. Boolean values
         *          will be converted to numbers.
         *  @param {String} [options.matMode='convert']
         *      Conversion strategy for the elements in matrixes. See option
         *      'valMode' for details.
         *  @param {String} [options.refMode='convert']
         *      Conversion strategy for the contents of cells referenced by the
         *      operand. See option 'valMode' for details.
         *  @param {Boolean} [options.emptyParam=false]
         *      If set to true, and the operand represents the empty function
         *      parameter, the callback function will be invoked with the value
         *      zero. By default, empty parameters will be skipped.
         *  @param {Boolean} [options.complexRef=false]
         *      If set to true, a reference operand may be complex. Complex
         *      references may contain multiple cell ranges, and each of the
         *      cell ranges may refer to multiple sheets. By default, a
         *      reference must refer to a single range in a single sheet.
         *  @param {Boolean} [options.ordered=false]
         *      If set to true, the cells in each cell range will be visited in
         *      order (row-by-row from top to bottom, and left-to-right inside
         *      each of the rows).
         *  @param {Boolean} [options.floor=false]
         *      If set to true, the numbers will be rounded down to the next
         *      integer (negative numbers will be rounded down too, i.e. away
         *      from zero!).
         *  @param {Boolean} [options.convertToDate=false]
         *      If set to true, the numbers will be converted to date objects.
         *      If conversion of a value to a date fails, the #VALUE! error
         *      code will be thrown. If the option 'floor' has been set to
         *      true, the time components of the resulting dates will be
         *      removed (the time will be set to midnight).
         *
         * @returns {FormulaContext}
         *  A reference to this instance.
         *
         * @throws {ErrorCode}
         *  Any error code contained in the passed operand, or thrown while
         *  converting values to numbers.
         */
        this.iterateNumbers = function (operand, callback, options) {

            // conversion strategy
            var convertMode = Utils.getStringOption(options, getOperandType(operand) + 'Mode', 'convert');
            // whether to round numbers down to integers
            var floorMode = Utils.getBooleanOption(options, 'floor', false);
            // date mode (convert all numbers to date values)
            var dateMode = Utils.getBooleanOption(options, 'convertToDate', false);

            // process all values in the operand
            iterateOperandValues(operand, function (value) {

                // convert current value according to conversion strategy
                switch (convertMode) {
                    case 'convert':
                        // convert value to number (may still fail for strings)
                        value = this.convertToNumber(value, floorMode);
                        break;
                    case 'rconvert':
                        // throw #VALUE! error code on booleans
                        if (_.isBoolean(value)) { throw ErrorCode.VALUE; }
                        // convert value to number (may still fail for strings)
                        value = this.convertToNumber(value, floorMode);
                        break;
                    case 'exact':
                        // throw #VALUE! error code on booleans and strings
                        if (_.isString(value) || _.isBoolean(value)) { throw ErrorCode.VALUE; }
                        // convert dates to numbers
                        value = this.convertToNumber(value, floorMode);
                        break;
                    case 'skip':
                        // skip strings and boolean values
                        if (!_.isNumber(value)) { value = null; } else if (floorMode) { value = Math.floor(value); }
                        break;
                    case 'zero':
                        // replace all strings with zero
                        value = _.isString(value) ? 0 : this.convertToNumber(value, floorMode);
                        break;
                    default:
                        FormulaUtils.throwInternal('FormulaContext.iterateNumbers(): unknown conversion mode: "' + convertMode + '"');
                }

                // convert numbers to dates if specified
                return (dateMode && _.isNumber(value)) ? this.convertToDate(value) : value;

            }, callback, Utils.extendOptions(options, { acceptErrors: false }));

            return this;
        };

        /**
         * Visits all boolean values contained in the passed operand. Provides
         * the value of a scalar operand, all elements of a matrix operand, and
         * the cell contents referred by a reference operand. In difference to
         * simple value parameters where numbers and strings contained in
         * matrixes and cell ranges will always be converted to boolean values
         * if possible, this method implements different conversion strategies
         * as required by the respective functions.
         *
         * @param {Any} operand
         *  The operand (scalar value of any supported data type, including
         *  null and error codes; or instances of the classes Matrix, Range3D,
         *  or Range3DArray; or a plain JS array of scalar values) whose
         *  contents will be visited.
         *
         * @param {Function} callback
         *  The callback function invoked for each boolean value contained in
         *  the operand. Will be called in the context of this instance.
         *  Receives the boolean value as first parameter.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {String} [options.valMode='convert']
         *      Conversion strategy for operands of type value:
         *      - 'convert': Strings will be converted to boolean values, if
         *          possible. Numbers will be converted to boolean values.
         *          Strings that represent a number result in a #VALUE! error
         *          code though.
         *      - 'skip': All strings will be skipped silently, also strings
         *          that could be converted to boolean values. Numbers will be
         *          converted though.
         *  @param {String} [options.matMode='convert']
         *      Conversion strategy for the elements in matrixes. See option
         *      'valMode' for details.
         *  @param {String} [options.refMode='convert']
         *      Conversion strategy for the contents of cells referenced by the
         *      operand. See option 'valMode' for details.
         *  @param {Boolean} [options.emptyParam=false]
         *      If set to true, and the operand represents the empty function
         *      parameter, the callback function will be invoked with the value
         *      FALSE. By default, empty parameters will be skipped.
         *  @param {Boolean} [options.complexRef=false]
         *      If set to true, a reference operand may be complex. Complex
         *      references may contain multiple cell ranges, and each of the
         *      cell ranges may refer to multiple sheets. By default, a
         *      reference must refer to a single range in a single sheet.
         *  @param {Boolean} [options.ordered=false]
         *      If set to true, the cells in each cell range will be visited in
         *      order (row-by-row from top to bottom, and left-to-right inside
         *      each of the rows).
         *
         * @returns {FormulaContext}
         *  A reference to this instance.
         *
         * @throws {ErrorCode}
         *  Any error code contained in the passed operand, or thrown while
         *  converting values to booleans.
         */
        this.iterateBooleans = function (operand, callback, options) {

            // conversion strategy
            var convertMode = Utils.getStringOption(options, getOperandType(operand) + 'Mode', 'convert');

            iterateOperandValues(operand, function (value) {
                switch (convertMode) {
                    case 'convert':
                        // convert value to boolean (may still fail for strings)
                        return this.convertToBoolean(value);
                    case 'skip':
                        // skip strings (but not numbers)
                        return _.isString(value) ? null : this.convertToBoolean(value);
                    default:
                        FormulaUtils.throwInternal('FormulaContext.iterateBooleans(): unknown conversion mode: "' + convertMode + '"');
                }
            }, callback, Utils.extendOptions(options, { acceptErrors: false }));

            return this;
        };

        /**
         * Returns all scalar values contained in the passed operand as a plain
         * array. Internally, method FormulaContext.iterateValues() will be
         * used to collect the values.
         *
         * @param {Any} operand
         *  The operand (scalar value of any supported data type, including
         *  null and error codes; or instances of the classes Matrix, Range3D,
         *  or Range3DArray; or a plain JS array of scalar values) whose
         *  contents will be visited.
         *
         * @param {Object} [options]
         *  Optional parameters. See method FormulaContext.iterateValues() for
         *  details.
         *
         * @returns {Array<Any>}
         *  The scalar values contained in the passed operand.
         *
         * @throws {ErrorCode}
         *  Any error code contained in the passed operand (unless error codes
         *  will be accepted as values according to the passed options).
         */
        this.getValuesAsArray = function (operand, options) {
            var values = [];
            this.iterateValues(operand, function (value) { values.push(value); }, options);
            return values;
        };

        /**
         * Returns all floating-point numbers contained in the passed operand
         * as a plain array. Internally, method FormulaContext.iterateNumbers()
         * will be used to collect the numbers.
         *
         * @param {Any} operand
         *  The operand (scalar value of any supported data type, including
         *  null and error codes; or instances of the classes Matrix, Range3D,
         *  or Range3DArray; or a plain JS array of scalar values) whose
         *  contents will be visited.
         *
         * @param {Object} [options]
         *  Optional parameters. See method FormulaContext.iterateNumbers() for
         *  details.
         *
         * @returns {Array<Number>}
         *  The floating-point numbers contained in the passed operand.
         *
         * @throws {ErrorCode}
         *  Any error code contained in the passed operand, or thrown while
         *  converting values to numbers.
         */
        this.getNumbersAsArray = function (operand, options) {
            var numbers = [];
            this.iterateNumbers(operand, function (number) { numbers.push(number); }, options);
            return numbers;
        };

        /**
         * Reduces all numbers (and other values convertible to numbers) of all
         * function parameters to the result of a spreadsheet function.
         *
         * @param {Array<Any>|Arguments} operands
         *  The operands containing numbers to be aggregated. The elements of
         *  the array can be instances of the classes Operand, Matrix, Range3D,
         *  Range3DArray, or may be any scalar value used in formulas.
         *
         * @param {Number} initial
         *  The initial intermediate result, passed to the first invocation of
         *  the aggregation callback function.
         *
         * @param {Function} aggregate
         *  The aggregation callback function. Receives the following
         *  parameters:
         *  (1) {Number} result
         *      The current intermediate result. On first invocation, this is
         *      the number passed in the 'initial' parameter.
         *  (2) {Number} number
         *      The new number to be combined with the intermediate result.
         *  Must return the new intermediate result which will be passed to the
         *  next invocation of this callback function, or may throw an error
         *  code. Will be called in the context of this instance.
         *
         * @param {Function} finalize
         *  A callback function invoked at the end to convert the intermediate
         *  result (returned by the last invocation of the aggregation callback
         *  function) to the actual function result. Receives the following
         *  parameters:
         *  (1) {Number} result
         *      The last intermediate result to be converted to the result of
         *      the implemented function.
         *  (2) {Number} count
         *      The count of all visited numbers.
         *  Must return the final numeric result of the spreadsheet function,
         *  or an error code object. Alternatively, may throw an error code.
         *  Will be called in the context of this instance.
         *
         * @param {Object} iteratorOptions
         *  Parameters passed to the number iterator used internally (see
         *  method FormulaContext.iterateNumbers() for details).
         *
         * @returns {Number}
         *  The result of the spreadsheet function.
         *
         * @throws {ErrorCode}
         *  An error code if calculating the function result has failed.
         */
        this.aggregateNumbers = function (operands, initial, aggregate, finalize, iteratorOptions) {

            var // the intermediate result
                result = initial,
                // the count of all visited numbers in all operands
                count = 0;

            // process all function parameters (may be an instance of Arguments)
            _.each(operands, function (operand) {
                this.iterateNumbers(operand, function (number) {
                    result = aggregate.call(this, result, number);
                    count += 1;
                }, iteratorOptions);
            }, this);

            // resolve the final result, throw error codes
            result = FormulaUtils.throwErrorCode(finalize.call(this, result, count));
            if (_.isNumber(result)) { return result; }
            FormulaUtils.throwInternal('FormulaContext.aggregateNumbers(): invalid aggregation result');
        };

        /**
         * Reduces all boolean values (and other values convertible to boolean
         * values) of all function parameters to the result of a spreadsheet
         * function.
         *
         * @param {Array<Any>|Arguments} operands
         *  The operands containing boolean values to be aggregated. The
         *  elements of the array can be instances of the classes Operand,
         *  Matrix, Range3D, Range3DArray, or may be any scalar value used in
         *  formulas.
         *
         * @param {Boolean} initial
         *  The initial intermediate result, passed to the first invocation of
         *  the aggregation callback function.
         *
         * @param {Function} aggregate
         *  The aggregation callback function. Receives the following
         *  parameters:
         *  (1) {Boolean} result
         *      The current intermediate result. On first invocation, this is
         *      the Boolean value passed in the 'initial' parameter.
         *  (2) {Boolean} value
         *      The new boolean value to be combined with the intermediate
         *      result.
         *  Must return the new intermediate result which will be passed to the
         *  next invocation of this callback function, or may throw an error
         *  code. Will be called in the context of this instance.
         *
         * @param {Function} finalize
         *  A callback function invoked at the end to convert the intermediate
         *  result (returned by the last invocation of the aggregation callback
         *  function) to the actual function result. Receives the following
         *  parameters:
         *  (1) {Boolean} result
         *      The last intermediate result to be converted to the result of
         *      the implemented function.
         *  (2) {Number} count
         *      The count of all visited boolean values.
         *  Must return the final boolean result of the spreadsheet function,
         *  or an error code object. Alternatively, may throw an error code.
         *  Will be called in the context of this instance.
         *
         * @param {Object} iteratorOptions
         *  Parameters passed to the boolean value iterator used internally
         *  (see method FormulaContext.iterateBooleans() for details).
         *
         * @returns {Boolean}
         *  The result of the spreadsheet function.
         *
         * @throws {ErrorCode}
         *  An error code if calculating the function result has failed.
         */
        this.aggregateBooleans = function (operands, initial, aggregate, finalize, iteratorOptions) {

            var // the intermediate result
                result = initial,
                // the count of all visited Boolean values in all operands
                count = 0;

            // process all function parameters (may be an instance of Arguments)
            _.each(operands, function (operand) {
                this.iterateBooleans(operand, function (value) {
                    result = aggregate.call(this, result, value);
                    count += 1;
                }, iteratorOptions);
            }, this);

            // resolve the final result, throw error codes
            result = FormulaUtils.throwErrorCode(finalize.call(this, result, count));
            if (_.isBoolean(result)) { return result; }
            FormulaUtils.throwInternal('FormulaContext.aggregateBooleans(): invalid aggregation result');
        };

        /**
         * Converts the passed simple filter pattern to a case-insensitive
         * regular expression object.
         *
         * @param {String} pattern
         *  The simple filter pattern. An asterisk character in the string
         *  matches any sequence of characters (also the empty sequence), and a
         *  question mark matches an arbitrary single character. If an asterisk
         *  or question mark is preceded by a tilde character, they lose their
         *  special meaning and will be matched literally.
         *
         * @param {Boolean} [complete=false]
         *  If set to true, the regular expression created by this method will
         *  only match complete strings. By default, any substring of a string
         *  will be matched.
         *
         * @returns {RegExp}
         *  A case-insensitive regular expression that represents the passed
         *  simple filter pattern.
         */
        this.convertPatternToRegExp = function (pattern, complete) {

            // convert passed search text to regular expression pattern matching the text literally
            pattern = _.escapeRegExp(pattern)
                // replace all occurrences of asterisk (without leading tilde) to '.*' RE pattern (e.g. 'a\*b' => 'a.*b')
                .replace(/(^|[^~])\\\*/g, '$1.*')
                // replace all occurrences of question mark (without leading tilde) to '.' RE pattern (e.g. 'a\?b' => 'a.b')
                .replace(/(^|[^~])\\\?/g, '$1.')
                // remove tilde characters before (already escaped) asterisks and question marks (e.g. 'a~\*b' => 'a\*b')
                .replace(/~(\\[*?])/g, '$1');

            // optionally, let the regular expression only match complete strings
            if (complete) { pattern = '^' + pattern + '$'; }

            // create the case-insensitive regular expression
            return new RegExp(pattern, 'i');
        };

        /**
         * Creates a filter matcher predicate function for the specified filter
         * criterion.
         *
         * @param {Any} criterion
         *  A scalar value representing the filter criterion. Numbers, boolean
         *  values, and error codes will match the corresponding values, and
         *  will match strings that can be converted to these values. The value
         *  null represents a reference to an empty cell, and will match the
         *  number zero only (but not the empty string, and not empty cells).
         *  Strings may contain leading comparison operators, or may specify
         *  filter patterns with asterisk characters or question marks.
         *  Otherwise, string criteria will be converted to numbers, boolean
         *  values, or error codes, if possible, and will match the
         *  corresponding values.
         *
         * @returns {Function}
         *  A predicate function that takes a single parameter (a scalar value
         *  of any type that can occur in a formula), and returns a boolean
         *  whether the passed value matches the filter criterion.
         */
        this.createFilterMatcher = function (criterion) {

            var // the number formatter of the document
                numberFormatter = this.getNumberFormatter(),
                // the filter operator (empty string for criterion without operator)
                op = null,
                // the filter operand
                comp = null,
                // whether to test for equality (needs some special handling)
                equality = null;

            // parse the filter operator and the filter operand from a string criterion
            if (_.isString(criterion)) {
                var matches = /^(<=|>=|<>|<|>|=)?(.*)$/.exec(criterion);
                op = _.isString(matches[1]) ? matches[1] : '';
                // leading apostrophe in operand counts as regular character
                comp = numberFormatter.parseResultValue(matches[2], { keepApos: true });
            } else {
                // all other data types cause filtering by equality
                op = '';
                comp = criterion;
            }

            // testing for equality converts strings in source data to other data types
            equality = (op === '') || (op === '=');

            // empty filter operand (empty parameter, or reference to empty cell) matches zero
            // numbers (and strings that can be parsed to zero number) in source data, but not
            // empty strings, FALSE, or empty cells
            if (_.isNull(comp)) {
                return function zeroMatcher(value) {
                    return (_.isString(value) ? numberFormatter.parseResultValue(value) : value) === 0;
                };
            }

            // special handling for the empty string as filter operand
            if (comp === '') {
                switch (op) {

                    // empty string without filter operator matches empty strings AND empty cells
                    case '':
                        return function emptyMatcher(value) { return (value === '') || _.isNull(value); };

                    // single equality operator without operand matches empty cells, but not empty strings
                    case '=':
                        return _.isNull;

                    // single inequality operator without operand matches anything but empty cells
                    case '<>':
                        return function notEmptyMatcher(value) { return !_.isNull(value); };

                    // all other comparison operators without operand do not match anything
                    default:
                        return _.constant(false);
                }
            }

            // equality or inequality with pattern matching
            if (_.isString(comp) && /[*?]/.test(comp) && (equality || (op === '<>'))) {

                var // the regular expression for the pattern (match complete strings only)
                    regExp = this.convertPatternToRegExp(comp, true);

                // create the pattern matcher predicate function (pattern filters work on strings only)
                return function patternMatcher(value) {
                    return equality === (_.isString(value) && regExp.test(value));
                };
            }

            // strings are always matched case-insensitively
            if (_.isString(comp)) { comp = comp.toUpperCase(); }

            // create the comparison matcher predicate function (pattern filters work on strings only)
            return function comparisonMatcher(value) {

                var // the relation between the value and the filter operand (signed integer)
                    rel = Number.NaN;

                // equality operator (but not inequality operator) converts strings in source data to other data types
                if (equality && _.isString(value)) {
                    value = numberFormatter.parseResultValue(value);
                }

                // compare the passed value with the filter operand (NaN means incompatible types)
                if (_.isNumber(comp) && _.isNumber(value)) {
                    rel = FormulaUtils.compareNumbers(value, comp);
                } else if (_.isString(comp) && _.isString(value)) {
                    rel = FormulaUtils.compareStrings(value, comp);
                } else if (_.isBoolean(comp) && _.isBoolean(value)) {
                    rel = FormulaUtils.compareBooleans(value, comp);
                } else if ((comp instanceof ErrorCode) && (value instanceof ErrorCode)) {
                    rel = FormulaUtils.compareErrorCodes(value, comp);
                }

                // return whether the value matches, according to the filter operand
                switch (op) {
                    case '':
                    case '=':
                        return isFinite(rel) && (rel === 0);
                    case '<>':
                        return !isFinite(rel) || (rel !== 0); // incompatible types match the inequality operator
                    case '<':
                        return isFinite(rel) && (rel < 0);
                    case '<=':
                        return isFinite(rel) && (rel <= 0);
                    case '>':
                        return isFinite(rel) && (rel > 0);
                    case '>=':
                        return isFinite(rel) && (rel >= 0);
                }

                FormulaUtils.throwInternal('FormulaContext.comparisonMatcher(): invalid operator');
            };
        };

        /**
         * Reduces all values that match a specific filter criterion to the
         * result of a spreadsheet function.
         *
         * @param {Range3D|Range3DArray} sourceRanges
         *  The addresses of the cell ranges containing the source data to be
         *  filtered for.
         *
         * @param {Any} criterion
         *  A scalar value representing the filter criterion. See method
         *  FormulaContext.createFilterMatcher() method for more details.
         *
         * @param {Range3D|Range3DArray|Null} dataRanges
         *  The addresses of the data ranges that will be aggregated to the
         *  result of the spreadsheet function. If set to null, the values in
         *  the filtered source ranges will be used.
         *
         * @param {Number} initial
         *  The initial intermediate result, passed to the first invocation of
         *  the aggregation callback function.
         *
         * @param {Function} aggregate
         *  The aggregation callback function. Receives the following
         *  parameters:
         *  (1) {Number} result
         *      The current intermediate result. On first invocation, this is
         *      the number passed in the 'initial' parameter.
         *  (2) {Number} number
         *      The new number to be combined with the intermediate result.
         *  Must return the new intermediate result which will be passed to the
         *  next invocation of this callback function, or may throw an error
         *  code. Will be called in the context of this instance.
         *
         * @param {Function} finalize
         *  A callback function invoked at the end to convert the intermediate
         *  result (returned by the last invocation of the aggregation callback
         *  function) to the actual function result. Receives the following
         *  parameters:
         *  (1) {Number} result
         *      The last intermediate result to be converted to the result of
         *      the implemented function.
         *  (2) {Number} numberCount
         *      The count of all numbers the aggregation callback function has
         *      been invoked for.
         *  (3) {Number} matchingCount
         *      The count of all matching values in the source range (also
         *      non-numeric matching cells).
         *  Must return the final numeric result of the spreadsheet function,
         *  or an error code. Alternatively, may throw an error code. Will be
         *  called in the context of this instance.
         *
         * @returns {Number}
         *  The result of the spreadsheet function.
         *
         * @throws {ErrorCode}
         *  An error code if calculating the function result has failed.
         */
        this.aggregateFiltered = function (sourceRanges, criterion, dataRanges, initial, aggregate, finalize) {

            var // value matcher predicate function for the source data
                matcher = this.createFilterMatcher(criterion),
                // extract the address of the data range (throws on error)
                dataRange = dataRanges ? this.convertToValueRange(dataRanges) : null,
                // the intermediate result
                result = initial,
                // the count of all numbers
                numberCount = 0,
                // the count of all matching values in the source range
                matchingCount = 0;

            // process all values in the passed operand (also empty cells in cell references)
            this.iterateValues(sourceRanges, function (value, col, row) {

                // skip non-matching cells completely
                if (!matcher(value)) { return; }

                var // the target address in the data range (invalid addresses will be ignored silently)
                    address = dataRange ? new Address(dataRange.start[0] + col, dataRange.start[1] + row) : null,
                    // pick the value from the data cell to be aggregated, or use the source value
                    dataValue = address ? this.getCellValue(dataRange.sheet1, address) : value;

                // aggregate numbers only, without automatic conversion of strings
                if (_.isNumber(dataValue)) {
                    result = aggregate.call(this, result, dataValue);
                    numberCount += 1;
                }

                // count all matching cells regardless of their type, or the type of the corresponding data cell
                matchingCount += 1;

            }, {
                emptyParam: true, // empty parameters count as zero
                emptyCell: true, // empty cells in references are matched by the empty string
                acceptErrors: true, // error codes can be matched by the filter criterion
                complexRef: false // do not accept multi-range and multi-sheet references
            });

            // resolve the final result
            return finalize.call(this, result, numberCount, matchingCount);
        };

    } // class FormulaContext

    // exports ================================================================

    return FormulaContext;

});
