/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/formula/cellref', [
    'io.ox/office/spreadsheet/utils/sheetutils'
], function (SheetUtils) {

    'use strict';

    // convenience shortcuts
    var Address = SheetUtils.Address;

    // class CellRef ==========================================================

    /**
     * A structure representing the address of a single cell in a reference
     * token (e.g. the particles '$A$1' or '$B$2' in Sheet1!$A$1:$B$2).
     *
     * @constructor
     *
     * @property {Number} col
     *  The column index. May be negative for a relative column reference in
     *  R1C1 notation.
     *
     * @property {Number} row
     *  The row index. May be negative for a relative row reference in R1C1
     *  notation.
     *
     * @property {Boolean} absCol
     *  Whether the column reference is absolute (as in $C2).
     *
     * @property {Boolean} absRow
     *  Whether the row reference is absolute (as in C$2).
     */
    function CellRef(col, row, absCol, absRow) {

        this.col = col;
        this.row = row;
        this.absCol = absCol;
        this.absRow = absRow;

    } // class CellRef

    // static methods ---------------------------------------------------------

    /**
     * Creates a cell reference structure for the passed cell address.
     *
     * @param {Address} address
     *  The cell address.
     *
     * @property {Boolean} absCol
     *  Whether the column reference is absolute (as in $C2).
     *
     * @property {Boolean} absRow
     *  Whether the row reference is absolute (as in C$2).
     */
    CellRef.create = function (address, absCol, absRow) {
        return new CellRef(address[0], address[1], absCol, absRow);
    };

    // public methods ---------------------------------------------------------

    /**
     * Creates a deep clone of this cell reference structure.
     *
     * @returns {CellRef}
     *  A deep clone of this cell reference structure.
     */
    CellRef.prototype.clone = function () {
        return new CellRef(this.col, this.row, this.absCol, this.absRow);
    };

    /**
     * Returns whether this cell reference and the passed cell reference are
     * equal.
     *
     * @param {CellRef} cellRef
     *  The other cell reference to be compared with this cell reference.
     *
     * @returns {Boolean}
     *  Whether this cell reference and the passed cell reference are equal.
     */
    CellRef.prototype.equals = function (cellRef) {
        return (this.col === cellRef.col) && (this.row === cellRef.row) && (this.absCol === cellRef.absCol) && (this.absRow === cellRef.absRow);
    };

    /**
     * Returns the cell address represented by this cell reference.
     *
     * @returns {Address}
     *  The cell address represented by this cell reference.
     */
    CellRef.prototype.toAddress = function () {
        return new Address(this.col, this.row);
    };

    /**
     * Relocates the relative column/row components in this cell reference
     * in-place.
     *
     * @param {SpreadsheetDocument} docModel
     *  The spreadsheet document used to validate the column and row indexes.
     *
     * @param {Address} refAddress
     *  The source reference address of the formula containing the token.
     *
     * @param {Address} targetAddress
     *  The target reference address for relocation.
     *
     * @param {Boolean} wrapReferences
     *  Whether to wrap the index at the sheet boundaries.
     *
     * @returns {Boolean}
     *  Whether relocation was successful, and the address of this reference
     *  is still valid.
     */
    CellRef.prototype.relocate = function (docModel, refAddress, targetAddress, wrapReferences) {

        // returns a relocated column/row index
        function relocateIndex(index, refIndex, targetIndex, maxIndex) {
            index += (targetIndex - refIndex);
            return wrapReferences ?
                ((index < 0) ? (index + maxIndex + 1) : (index > maxIndex) ? (index - maxIndex - 1) : index) :
                (((index < 0) || (index > maxIndex)) ? -1 : index);
        }

        // relocate relative column
        if (!this.absCol) {
            this.col = relocateIndex(this.col, refAddress[0], targetAddress[0], docModel.getMaxCol());
            if (this.col < 0) { return false; }
        }

        // relocate relative row
        if (!this.absRow) {
            this.row = relocateIndex(this.row, refAddress[1], targetAddress[1], docModel.getMaxRow());
            if (this.row < 0) { return false; }
        }

        return true;
    };

    /**
     * Returns the complete name of the column part of this cell reference.
     *
     * @returns {String}
     *  The complete name of the column part of this cell reference.
     */
    CellRef.prototype.colText = function () {
        return (this.absCol ? '$' : '') + Address.stringifyCol(this.col);
    };

    /**
     * Returns the complete name of the row part of this cell reference.
     *
     * @returns {String}
     *  The complete name of the row part of this cell reference.
     */
    CellRef.prototype.rowText = function () {
        return (this.absRow ? '$' : '') + Address.stringifyRow(this.row);
    };

    /**
     * Returns the complete name of this cell reference.
     *
     * @returns {String}
     *  The complete name of this cell reference.
     */
    CellRef.prototype.refText = function () {
        return this.colText() + this.rowText();
    };

    /**
     * Returns the complete name of the column part of this cell reference, for
     * R1C1 notation.
     *
     * @param {String} prefixChars
     *  The prefix characters (two characters) used for R1C1 notation.
     *
     * @returns {String}
     *  The complete name of the column part of this cell reference, for R1C1
     *  notation.
     */
    CellRef.prototype.colTextRC = function (prefixChars) {
        return prefixChars[1] + (this.absCol ? (this.col + 1) : (this.col !== 0) ? ('[' + this.col + ']') : '');
    };

    /**
     * Returns the complete name of the row part of this cell reference, for
     * R1C1 notation.
     *
     * @param {String} prefixChars
     *  The prefix characters (two characters) used for R1C1 notation.
     *
     * @returns {String}
     *  The complete name of the row part of this cell reference, for R1C1
     *  notation.
     */
    CellRef.prototype.rowTextRC = function (prefixChars) {
        return prefixChars[0] + (this.absRow ? (this.row + 1) : (this.row !== 0) ? ('[' + this.row + ']') : '');
    };

    /**
     * Returns the complete name of this cell reference, for R1C1 notation.
     *
     * @param {String} prefixChars
     *  The prefix characters (two characters) used for R1C1 notation.
     *
     * @returns {String}
     *  The complete name of this cell reference.
     */
    CellRef.prototype.refTextRC = function (prefixChars) {
        return this.rowTextRC(prefixChars) + this.colTextRC(prefixChars);
    };

    /**
     * Returns a text description of this cell reference for debug logging.
     */
    CellRef.prototype.toString = CellRef.prototype.refText;

    // exports ================================================================

    return CellRef;

});
