/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/app/application', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/locale/localedata',
    'io.ox/office/baseframework/utils/errorcode',
    'io.ox/office/baseframework/utils/errorcontext',
    'io.ox/office/editframework/app/editapplication',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/model/model',
    'io.ox/office/spreadsheet/view/view',
    'io.ox/office/spreadsheet/app/controller',
    'io.ox/office/spreadsheet/services/commandmixin',
    'gettext!io.ox/office/spreadsheet/main'
], function (Utils, LocaleData, ErrorCode, ErrorContext, EditApplication, SheetUtils, SpreadsheetModel, SpreadsheetView, SpreadsheetController, CommandMixin, gt) {

    'use strict';

    var // convenience shortcuts
        RangeArray = SheetUtils.RangeArray,
        Range3DArray = SheetUtils.Range3DArray;

    // class SpreadsheetApplication ===========================================

    /**
     * The OX Spreadsheet application.
     *
     * Triggers the following additional events:
     * - 'docs:update:cells'
     *      After the application has received a 'docs:update' event which
     *      contains the contents and/or addresses of changed cells in the
     *      document. Event handlers receive the following parameter:
     *      (1) {Object} changedData
     *          A map of change descriptors for each changed sheet, mapped by
     *          the zero-based sheet index. Each change descriptor object
     *          contains the following properties:
     *          - {Number} sheet
     *              The zero-based sheet index. Equal to the map key of this
     *              object, but as number for convenience.
     *          - {Array} [rangeContents]
     *              An array with cell range descriptors. Each descriptor
     *              object contains the same properties as also sent in the
     *              response data of view update requests. If the entire array
     *              has been omitted, no cells have been changed in the sheet,
     *              or no explicit cell content data is available yet (see
     *              property 'dirtyRanges').
     *          - {RangeArray} [contentRanges]
     *              An array with the merged cell range addresses covered by
     *              the elements of the array property 'rangeContents'.
     *          - {RangeArray} [dirtyRanges]
     *              An array with cell range addresses of additional cells that
     *              are not included in the 'rangeContents' property, but whose
     *              contents have been changed in the sheet. These ranges need
     *              to be updated with further server requests. If omitted, the
     *              sheet does not contain additional dirty cell ranges.
     *          - {RangeArray} [changedRanges]
     *              An array containing the merged cell range addresses from
     *              the properties 'rangeContents' and 'dirtyRanges'. Will
     *              exist, if either of these properties exists too.
     *          - {Number} [usedCols]
     *              The number of used columns in the sheet. If omitted, the
     *              number of used columns has not changed.
     *          - {Number} [usedRows]
     *              The number of used rows in the sheet. If omitted, the
     *              number of used rows has not changed.
     *      (2) {Range3DArray} allChangedRanges
     *          An array containing the addresses of all changed cell ranges in
     *          all sheets, including the sheet indexes. This array is the
     *          concatenation of all cell ranges contained in the properties
     *          'changedRanges' for all sheets in the 'changedData' event
     *          parameter. The properties 'sheet1' and 'sheet2' in each cell
     *          range address will be equal (each range refers to a single
     *          sheet only), but they may differ in different ranges.
     *      (3) {String} [errorCode]
     *          A specific error code associated to the changed cells. The
     *          following error codes are supported:
     *          - 'circular': The changed cells contain at least one formula
     *              cell with a circular reference.
     *
     * @constructor
     *
     * @extends EditApplication
     *
     * @param {Object} launchOptions
     *  All options passed to the core launcher (the ox.launch() method).
     */
    var SpreadsheetApplication = EditApplication.extend({ constructor: function (launchOptions) {

        var // self reference
            self = this,

            // the spreadsheet document model and view
            docModel = null,
            docView = null;

        // base constructor ---------------------------------------------------

        EditApplication.call(this, SpreadsheetModel, SpreadsheetView, SpreadsheetController, launchOptions, {
            newDocumentParams: { initial_sheetname: SheetUtils.getSheetName(0) },
            preProcessHandler: preProcessDocument,
            postProcessHandler: postProcessDocument,
            previewHandler: previewHandler,
            importFailedHandler: importFailedHandler,
            prepareFlushHandler: prepareFlushDocument,
            prepareLoseEditRightsHandler: prepareLoseEditRights,
            prepareRenameHandler: prepareRenameHandler,
            realTimeDelay: 10,
            supportsOnlineSync: true
        });

        // private methods ----------------------------------------------------

        /**
         * Preparation of the document, before its operations will be applied.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the document has been prepared
         *  successfully; or rejected on any error.
         */
        function preProcessDocument() {

            // initialize formula resource data, fail with an appropriate error message
            return docModel.initFormulaResource({ remote: true }).then(null, function () {
                Utils.error('SpreadsheetModel.initFormulaResource(): could not download resource data from server');
                return { headline: gt('Server Error'), message: gt('The engine used for calculating formulas is not available.') };
            });
        }

        /**
         * Post-processing of the document, after all its import operations
         * have been applied successfully.
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved when the
         *  document has been post-processed successfully; or rejected when the
         *  document is invalid, or an error has occurred.
         */
        function postProcessDocument() {

            // document must contain at least one sheet (returning a rejected Deferred
            // object will cause invocation of the prepareInvalidDocument() method)
            if (docModel.getSheetCount() === 0) { return $.Deferred().reject(); }

            // create all built-in cell styles not imported from the file
            docModel.getCellStyles().createMissingStyles();

            // nothing to wait for
            return $.when();
        }

        /**
         * Shows an early preview of the document. Activates the sheet that is
         * specified in the passed preview data, and starts querying cell
         * contents for the active sheet.
         *
         * @param {Object} previewData
         *  The preview data containing the index of the active sheet.
         *
         * @returns {Boolean}
         *  Whether a sheet has been activated successfully.
         */
        function previewHandler(previewData) {
            var activeSheet = Utils.getIntegerOption(previewData, 'activeSheet', 0);
            return self.getView().activatePreviewSheet(activeSheet);
        }

        /**
         * Handler will be called by base class if importing the document
         * failed. The functions handles Spreadsheet specific errors.
         */
        function importFailedHandler(response) {

            var // specific error code sent by the server
                error = new ErrorCode(response);

            switch (error.getCodeAsConstant()) {
                case 'LOADDOCUMENT_COMPLEXITY_TOO_HIGH_ERROR':
                    self.setInternalError(error, ErrorContext.SPREADSHEET, null, { showMessage: false });
                    break;
                case 'LOADDOCUMENT_COMPLEXITY_TOO_HIGH_SHEET_COUNT_ERROR':
                    self.setInternalError(error, ErrorContext.SPREADSHEET, null, { showMessage: false });
                    break;
            }
        }

        /**
         * Preprocessing before the document will be flushed for downloading,
         * printing, sending as mail, or closing.
         */
        function prepareFlushDocument() {
            // commit current text of the cell in-place edit mode (without validation)
            docView.leaveCellEditMode('auto');
            // send all changed view settings before flushing editable document
            docView.sendChangedViewAttributes();
        }

        /**
         * Preparations before the edit mode will be switched off.
         */
        function prepareLoseEditRights() {
            // commit current text of the cell in-place edit mode (without validation)
            docView.leaveCellEditMode('auto');
        }

        /**
         * Preparations before the rename requests will be sent to the
         * backend. Must call sendChangedViewAttributes to make sure that
         * view settings are not sent while closing the document.
         */
        function prepareRenameHandler() {
            docView.sendChangedViewAttributes();
        }

        /**
         * Handles 'docs:update' notifications. If the message data contains
         * information about changed cells in the document, this method
         * triggers a 'docs:update:cells' event with preprocessed user data.
         *
         * @param {Object} data
         *  The message data of the 'docs:update' application event.
         */
        function updateNotificationHandler(data) {

            var // complete changed data
                changedData = Utils.getObjectOption(data, 'changed'),
                // the locale used to format the cell contents
                locale = Utils.getStringOption(data, 'locale'),
                // a special error code associated to the changed cells
                errorCode = Utils.getStringOption(changedData, 'error'),
                // all changed cell information for the own 'docs:update:cells' event
                eventData = null,
                // all changed range addresses, with sheet index
                allChangedRanges = null;

            // update message may not contain any data about changed cells
            if (!_.isObject(changedData)) { return; }
            Utils.info('SpreadsheetApplication.updateNotificationHandler(): notification received:', changedData);

            // extract the map of changed data per sheet
            changedData = Utils.getObjectOption(changedData, 'sheets');
            if (!changedData) {
                Utils.warn('SpreadsheetApplication.updateNotificationHandler(): no sheets defined');
                return;
            }

            // process all changed sheets
            eventData = {};
            allChangedRanges = new Range3DArray();
            _.each(changedData, function (changedSheetData, sheetKey) {

                // validate sheet index
                var sheet = parseInt(sheetKey, 10);
                if (!isFinite(sheet) || (sheet < 0) || (sheet >= docModel.getSheetCount())) {
                    Utils.warn('SpreadsheetApplication.updateNotificationHandler(): invalid sheet index: "' + sheetKey + '"');
                    return;
                }

                // ensure that the map value is an object
                if (!_.isObject(changedSheetData)) {
                    Utils.warn('SpreadsheetApplication.updateNotificationHandler(): unexpected data for sheet ' + sheet + ':', changedSheetData);
                    return;
                }

                var contentRanges = new RangeArray(),
                    dirtyRanges = docModel.createRangeArray(changedSheetData.cells) || new RangeArray(),
                    colIntervals = docModel.createIntervalArray(changedSheetData.columns, true),
                    rowIntervals = docModel.createIntervalArray(changedSheetData.rows, false),
                    jsonContents = Utils.getArrayOption(changedSheetData, 'contents', []),
                    usedCols = Utils.getIntegerOption(changedSheetData, 'usedCols'),
                    usedRows = Utils.getIntegerOption(changedSheetData, 'usedRows'),
                    eventSheetData = {};

                // add column/row intervals to dirty ranges (TODO: still necessary?)
                if (colIntervals) { dirtyRanges.append(colIntervals.map(docModel.makeColRange, docModel)); }
                if (rowIntervals) { dirtyRanges.append(rowIntervals.map(docModel.makeRowRange, docModel)); }

                // create real cell range addresses from the JSON content data, filter out invalid ranges
                jsonContents = jsonContents.filter(function (jsonData) {
                    var range = docModel.createRange(jsonData);
                    if (range) { contentRanges.push(range); return true; }
                });

                // bug 32624: use content data only if formatted with the own locale; otherwise store as 'dirty ranges'
                if (jsonContents.length > 0) {
                    if (locale === LocaleData.LOCALE) {
                        contentRanges = contentRanges.merge();
                        dirtyRanges = dirtyRanges.difference(contentRanges);
                        eventSheetData.rangeContents = jsonContents;
                        eventSheetData.contentRanges = contentRanges;
                    } else {
                        dirtyRanges.append(contentRanges);
                        contentRanges.clear();
                    }
                }

                // insert merged dirty ranges into the event data
                if (!dirtyRanges.empty()) {
                    dirtyRanges = dirtyRanges.merge();
                    eventSheetData.dirtyRanges = dirtyRanges.clone();
                }

                // insert number of used columns/rows
                if (_.isNumber(usedCols)) { eventSheetData.usedCols = usedCols; }
                if (_.isNumber(usedRows)) { eventSheetData.usedRows = usedRows; }

                // insert sheet entry into the own event data
                if (!_.isEmpty(eventSheetData)) {
                    eventSheetData.sheet = sheet;
                    eventData[sheetKey] = eventSheetData;
                }

                // crate array of all changed ranges in the sheet (dirty ranges, and content ranges)
                if (!contentRanges.empty()) { dirtyRanges = dirtyRanges.append(contentRanges).merge(); }
                if (!dirtyRanges.empty()) {
                    eventSheetData.changedRanges = dirtyRanges;
                    // extend array of all changed ranges in the entire document (with sheet indexes)
                    allChangedRanges.append(Range3DArray.createFromRanges(dirtyRanges, sheet));
                }
            });

            // trigger the event, if the event data object contains any useful information
            if (!_.isEmpty(eventData)) {
                self.trigger('docs:update:cells', eventData, allChangedRanges, errorCode);
            }
        }

        // initialization -----------------------------------------------------

        // add mix-in classes (need a complete application instance)
        CommandMixin.call(this);

        // initialization of class members
        this.onInit(function () {
            docModel = self.getModel();
            docView = self.getView();
        });

        // application notifies changed contents/results of cells create an own 'docs:update:cells' event
        this.on('docs:update', updateNotificationHandler);

        // destroy all class members
        this.registerDestructor(function () {
            launchOptions = self = docModel = docView = null;
        });

    } }); // class SpreadsheetApplication

    // static methods ---------------------------------------------------------

    /**
     * Replacement for the generic method EditApplication.createLauncher()
     * without parameters, to launch spreadsheet applications.
     */
    SpreadsheetApplication.createLauncher = function () {
        return EditApplication.createLauncher('io.ox/office/spreadsheet', SpreadsheetApplication, { icon: 'fa-table', trackingId: 'io.ox/spreadsheet-editor' });
    };

    // exports ================================================================

    return SpreadsheetApplication;

});
