/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Jonas Regier <jonas.regier@open-xchange.com>
 */

define('io.ox/office/presentation/view/slidepane', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/keycodes',
    'io.ox/office/tk/forms',
    'io.ox/office/baseframework/view/pane',
    'io.ox/office/presentation/view/slidepanecontextmenu'
], function (Utils, KeyCodes, Forms, Pane, SlidePaneContextMenu) {

    'use strict';

    // class SlidePane =======================================================

    /**
     * The side pane in presentation to select slides.
     *
     * @constructor
     *
     * @extends Pane
     *
     * @param {PresentationView} docView
     *  The presentation view containing this instance.
     */
    function SlidePane(docView) {

        var debugSlidePane = false; // only for debug

        var // self reference
            self = this,
            // the application instance
            app = docView.getApp(),
            // app Model
            appModel = app.getModel(),
            // the slide-pane-container node
            slidePaneContainer = $('<div class ="slide-pane-container" tabindex="1" > </div>'),
            // context menu for the slide pane
            contextMenu = null;

        // base constructor ---------------------------------------------------

        Pane.call(this, docView, { position: 'left', classes: 'slide-pane standard-design' });

        // private methods ----------------------------------------------------

        // ------ render functions ------

        /**
         * Render the Slidepane-
         *
         * @param {Boolean} isMasterView
         *  A flag if the normal or master/slide view is active.
         */
        function renderSlidePane(isMasterView) {

            // clear the DOM for new rendering
            slidePaneContainer.empty();

            // get slides from the model
            var slides = isMasterView ? appModel.getMasterSlideOrder() : appModel.getStandardSlideOrder();

            // render the slides and insert the slide id as an attribute
            _.each(slides, function (val, i) {
                var index = i + 1;
                slidePaneContainer.append('<div class ="slide-container" id = ' + val + '> <div class = "leftSide"><span>' + index + '</span> </div> <div class = "middle"> <div class="slide"> </div> </div>');
            });

            if (debugSlidePane) { Utils.log('::renderSlidePane:::', 'SLIDES', slides); }
        }

        /**
         * Select the slide in the GUI.
         *
         * @param {String} id
         *  The slide id.
         */
        function guiSelectSlide(id) {
            // unselect all slides
            slidePaneContainer.children('.selected').removeClass('selected');

            // selected the given slide
            slidePaneContainer.children('div[id = "' + id + '"]').addClass('selected');

            // scroll to the current selected slide in the slidepane
            Utils.scrollToChildNode(slidePaneContainer, slidePaneContainer.find('div[id = "' + id + '"]'));

            if (debugSlidePane) { Utils.log('::guiSelectSlide::', id); }
        }

        // ------ handler functions ------

        function setActiveSlideforClickedTarget(event) {

            // get the 'clicked' slide-container
            var selected = $(event.target).closest('.slide-container');

            // on COMPACT_DEVICES the context menu is invoked by a tap on the selected slide
            if (Utils.COMPACT_DEVICE) {
                // the focus should be on the slidepane when the slide pane is touched
                slidePaneContainer.focus();
                // if the touched target is allready selected, invoke the context menu
                if (selected.hasClass('selected')) {
                    slidePaneContainer.trigger(new $.Event('documents:contextmenu', { sourceEvent: event, stickToSlide: true }));
                }
            }

            // set the 'clicked' activeSlide in the model
            if (selected.attr('id') !== undefined) {  // is undefined when clicking on the margin from the slide-container
                docView.executeControllerItem('slidePaneSetActiveSlide', selected.attr('id'), { preserveFocus: true });
            }

            if (debugSlidePane) { Utils.log('::setActiveSlideforClickedTarget:: target ID', selected.attr('id')); }
        }

        function keyHandler(event) {
            // need to be prevented, otherwise the focus may be lost on up/down
            if (KeyCodes.matchKeyCode(event, 'UP_ARROW'))   { docView.executeControllerItem('slidePaneSlideUp', '', { preserveFocus: true }); event.preventDefault(); }
            if (KeyCodes.matchKeyCode(event, 'DOWN_ARROW')) { docView.executeControllerItem('slidePaneSlideDown', '', { preserveFocus: true }); event.preventDefault(); }
            if (KeyCodes.matchKeyCode(event, 'HOME'))       { docView.executeControllerItem('slidePaneShowFirstSlide', '', { preserveFocus: true }); event.preventDefault(); }
            if (KeyCodes.matchKeyCode(event, 'END'))        { docView.executeControllerItem('slidePaneShowLastSlide', '', { preserveFocus: true }); event.preventDefault(); }
        }

        /**
         * Observe the activeSlide in the model.
         *
         * @param {Event} event
         *  The source event.
         *
         * @param {Object} viewCollector
         *  Object with information about the currently active and used slides.
         */
        function observeActiveSlide(event, viewCollector) {

            // update the slidePane GUI with the currently active slide
            guiSelectSlide(viewCollector.activeSlideId);

            if (debugSlidePane) { Utils.log('::observeActiveSlide:: ', viewCollector); }
        }

        /**
         * Render the slidePane again when the activeView is changed or a slide is inserted/removed.
         *
         * @param {Event} event
         *  The source event.
         *
         * @param {Object} initOptions
         *  Optional parameters:
         *
         *  @param {Boolean} [initOptions.isMasterView=false]
         *   A flag if the normal or master/slide view is active.
         *
         */
        function updateSlidePaneView(event, initOptions) {
            var isMasterView = Utils.getBooleanOption(initOptions, 'isMasterView', false);
            renderSlidePane(isMasterView);

            if (debugSlidePane) { Utils.log('::updateSlidePaneView:: ', initOptions); }
        }

        /**
         * Initialization from the SlidePane. The activeSlide and activeView is set directly by the event.
         *
         * @param {Event} event
         *  The source event.
         *
         * @param {Object} initOptions
         *  Optional parameters:
         *
         *  @param {Boolean} [initOptions.activeSlideId]
         *   The currently active slide, at the moment the SlidePane was invoked by the source event.
         *
         *  @param {Boolean} [initOptions.isMasterView=false]
         *   A flag if the normal or master/slide view is active, at the moment the SlidePane was invoked by the source event.
         *
         */
        function initSlidePane(event, initOptions) {
            //ini = true;
            var // the activeSlide id
                activeSlideId = Utils.getStringOption(initOptions, 'activeSlideId'),
                // if normal oder master/slide view is active
                isMasterView = Utils.getBooleanOption(initOptions, 'isMasterView', false);

            initListener();

            renderSlidePane(isMasterView);

            // since there is no change:slide' event after the initialization, the selection must be set here initially.
            guiSelectSlide(activeSlideId);

            contextMenu = new SlidePaneContextMenu(docView, slidePaneContainer);

            if (debugSlidePane) { Utils.log('::initSlidePane::', initOptions, isMasterView, appModel.getActiveSlideId()); }
        }

        /**
         * Initialization from event listener after initialization.
         */
        function initListener() {

            // slide change
            self.listenTo(appModel, 'change:slide', observeActiveSlide);

            // activeView change (between normal and master/layout)
            self.listenTo(appModel, 'change:activeView', updateSlidePaneView);

            // update the view after insert/remove slide
            self.listenTo(appModel, 'inserted:slide', updateSlidePaneView);
            self.listenTo(appModel, 'removed:slide', updateSlidePaneView);

            // handle key events on the slidepane (when focused)
            slidePaneContainer.on('keydown', keyHandler);

            // mousemove have to be preventDefault() or the focus can be lost on some mouse interactions on the pane
            slidePaneContainer.on('mousemove', function (e) { e.preventDefault(); });

            // when a slide is clicked with the right mouse button...
            slidePaneContainer.on('contextmenu', function (e) {
                // not on compact devices
                if (!Utils.COMPACT_DEVICE) {
                    slidePaneContainer.focus();
                    setActiveSlideforClickedTarget(e);
                }
            });

            // handle clicks on the slide in the slidePane
            Forms.touchAwareListener({ node: slidePaneContainer }, setActiveSlideforClickedTarget);

        }

        /**
         * Getting the slidePaneContainer.
         *
         * @returns {jQuery}
         *   The slidePaneContainer as a jQuery object.
         */
        this.getSlidePaneContainer = function () {
            return slidePaneContainer;
        };

        // initialization -----------------------------------------------------

        // add the main HTML container to the slidePane
        self.getNode().append(slidePaneContainer);

        // init the slidepane
        self.listenTo(appModel, 'slideModel:init', initSlidePane);

        // destroy class members on destruction
        this.registerDestructor(function () {
            if (contextMenu) { contextMenu.destroy(); }
            self = app = docView = appModel = slidePaneContainer = contextMenu = null;
        });

    } // class StatusPane

    // exports ================================================================

    // derive this class from class ToolPane
    return Pane.extend({ constructor: SlidePane });

});
