/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define('io.ox/office/presentation/model/updatelistsmixin', [
    'io.ox/office/tk/io',
    'io.ox/office/textframework/utils/textutils',
    'io.ox/office/textframework/utils/dom'
], function (IO, Utils, DOM) {

    'use strict';

    // mix-in class UpdateListsMixin ======================================

    /**
     * A mix-in class for the document model providing the handler functions
     * that keep lists up-to-date.
     *
     * @constructor
     *
     * @param {Application} app
     *  The application containing this instance.
     */
    function UpdateListsMixin(app) {

        var // self reference for local functions
            self = this,
            // a collector for all paragraph nodes that need to be updated
            paragraphs = $(),
            // whether only the current paragraph needs to be updated
            onlySingleParagraph = false,
            // the debounced list handler function
            updateListsDebounced = self.createDebouncedMethod(registerListUpdate, updateList, { delay: 10, infoString: 'Presentation: updateLists' });

        // private methods ----------------------------------------------------

        // direct callback: called every time when implParagraphChanged() has been called
        // Option 'singleParagraph': If set to true, only the one specified paragraph is updated. Otherwise
        // all paragraphs in the drawing that contains the specified paragraph are updated. 'singleParagraph'
        // is useful, if a bullet type is modified or some character attributes in a paragraph are modified.
        function registerListUpdate(paragraph, options) {

            // whether only the specified paragraph needs to be updated
            onlySingleParagraph = Utils.getBooleanOption(options, 'singleParagraph', false);

            // store the new paragraph in the collection (jQuery keeps the collection unique)
            if (paragraph) {
                paragraphs = paragraphs.add(paragraph);
            }
        }

        // deferred callback
        function updateList() {
            var allParagraps = onlySingleParagraph ? paragraphs : paragraphs.parent().children(DOM.PARAGRAPH_NODE_SELECTOR);
            updateLists(null, allParagraps);
            paragraphs = $();
            onlySingleParagraph = false;
        }

        /**
         * Updates all paragraphs that are part of any bullet or numbering
         * list or update  all specified paragraphs.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.async=false]
         *      If set to true, all lists are updated asynchronously. This
         *      should happen only once when the document is loaded.
         *
         * @param {jQuery} [paragraphs]
         *  Optional set of paragraphs, that need to be updated. If not specified
         *  all paragraphs in the document need to be updated. This happens after
         *  the document was loaded successfully. In this case the option 'async'
         *  needs to be set to true.
         *
         * @returns {jQuery.Promise}
         *  A promise that is resolved, when all paragraphs are updated.
         */
        function updateLists(options, paragraphs) {

            var // an array collecting all numbers for numbered list items
                listItemCounter = [],
                // paragraph index in a list
                listParagraphIndex = [],
                // list of all paragraphs in the document
                paragraphNodes = null,
                // the deferred, whose promise will be returned
                def = null,
                // Performance: If only paragraphs with numbered lists were splitted in the registration phase,
                // it is sufficient to update all paragraphs starting from the one paragraph that was marked
                // with 'splitInNumberedList'.
                splitInNumberedList = false, // TODO updateListsDebouncedOptions.splitInNumberedList,
                // whether the splitted paragraph in the numbered list was found
                splittedNumberedListFound = false,
                // whether DOM manipulations can be suppressed. This is the case for numbered
                suppressDomManipulation = false;

            function updateListInParagraph(para) {

                var // the attributes at the paragraph
                    elementAttributes = self.getParagraphStyles().getElementAttributes(para),
                    // the paragraph attributes at the paragraph
                    paraAttributes = elementAttributes.paragraph,
                    // the paragraph level
                    listLevel = paraAttributes.level,
                    // an existing list label at the paragraph
                    oldLabel = null,
                    // whether tab stops need to be recalculated at the paragraph
                    updateParaTabstops = false,
                    // during loading the document, the counter has to be resetted for every first paragraph in text frame
                    resetListItemCounter = !paragraphs && $(para).prevAll(DOM.PARAGRAPH_NODE_SELECTOR).length === 0,
                    // for some characters and fonts it is necessary to replace the values by more reliable values
                    replacementChar = null,
                    // the character and font name that are used for displaying
                    displayChar = null, displayFontName = null,
                    // the numbering type
                    numType = null,
                    // whether the paragraph has a list label
                    noListLabel = true,
                    // an url for a list item bitmap
                    imageUrl = null,
                    // the id of the numbered list
                    listCounterId =  null,
                    // the index of the sub levels
                    subLevelIdx = 0,
                    // a text string
                    textChar = '',
                    // the list label node
                    numberingElement = null,
                    // the first text span in the paragraph
                    span = null,
                    // the character attributes of the text span following the list label node
                    followingTextAttributes = null,
                    // the list span inside the list label node
                    listSpans = null;

                // helper function to count the number for the list items in numbered lists
                function countListLevel() {

                    if (!listItemCounter[listCounterId] || resetListItemCounter) {
                        listItemCounter[listCounterId] = [0, 0, 0, 0, 0, 0, 0, 0, 0, 0];
                        listParagraphIndex[listCounterId] = 0;
                    }

                    if (!noListLabel) {
                        listItemCounter[listCounterId][listLevel]++;
                        listParagraphIndex[listCounterId]++;
                    }

                    if (paraAttributes.listStartValue >= 0) {
                        listItemCounter[listCounterId][listLevel] = paraAttributes.listStartValue;
                    }

                    // TODO: reset sub-levels depending on their 'levelRestartValue' attribute
                    subLevelIdx = listLevel + 1;

                    for (; subLevelIdx < 10; subLevelIdx++) {
                        listItemCounter[listCounterId][subLevelIdx] = 0;
                    }

                    // fix level counts of non-existing upper levels
                    subLevelIdx = listLevel - 1;

                    for (; subLevelIdx >= 0; subLevelIdx--) {
                        if (listItemCounter[listCounterId][subLevelIdx] === 0) {
                            listItemCounter[listCounterId][subLevelIdx] = 1;
                        }
                    }

                }

                // Updating paragraphs, that are no longer part of lists (for example after 'Backspace')
                if ($(para).data('removeLabel')) {
                    $(para).removeData('removeLabel');
                }

                if ($(para).data('updateList')) {
                    $(para).removeData('updateList');
                }

                if ($(para).data('splitInNumberedList')) {  // might be set at more than one paragraph
                    splittedNumberedListFound = true;
                    $(para).removeData('splitInNumberedList');
                }

                // Performance III: No DOM manipulation, if the split happened in a numbered list and the marked
                // paragraph (marked with 'splitInNumberedList') was not found yet. Only for following paragraphs
                // the DOM manipulations are necessary. But the number counting is necessary for all paragraphs.
                suppressDomManipulation = (splitInNumberedList && !splittedNumberedListFound);

                if (!suppressDomManipulation) {
                    oldLabel = $(para).children(DOM.LIST_LABEL_NODE_SELECTOR);
                    updateParaTabstops = oldLabel.length > 0;
                    oldLabel.remove();
                }

                if (_.isNumber(listLevel) && listLevel >= 0) {

                    noListLabel = paraAttributes.bullet.type === 'none';
                    imageUrl = paraAttributes.bullet.type === 'bitmap' ? paraAttributes.bullet.imageUrl : null;

                    // preparing counting of list item numbers
                    if (paraAttributes.bullet.type === 'numbering') {
                        listCounterId = paraAttributes.bullet.numType;  // different counter for every different numbering type
                    } else {
                        resetListItemCounter = true; // restart counting after every non-numbering paragraph (TODO: Only in same level?)
                    }

                    if (listLevel !== -1 && listLevel < 10) {

                        if (paraAttributes.bullet.type === 'numbering') { countListLevel(); }

                        // Performance III: After updating the listItemCounter and listParagraphIndex, the following
                        // DOM manipulations can be ignored, if 'suppressDomManipulation' is set to true. This is the
                        // case for numbered lists in which the added paragraph marked with 'splitInNumberedList' was
                        // not found yet.
                        if (suppressDomManipulation) { return; }

                        updateParaTabstops = true;

                        if (paraAttributes.bullet.type === 'character') {

                            // check if a replacement is required for displaying the character correctly
                            replacementChar = self.getUnicodeReplacementForChar(paraAttributes.bullet.character, paraAttributes.bulletFont.name);

                            if (_.isObject(replacementChar)) {
                                displayChar = replacementChar.levelText;
                                displayFontName = replacementChar.fontFamily;
                            } else {
                                displayChar = paraAttributes.bullet.character;
                                displayFontName = paraAttributes.bulletFont.name ? paraAttributes.bulletFont.name : null;
                            }

                            textChar = paraAttributes.bullet.character ? displayChar : ''; // setting the character that is used inside the list label

                        } else if (paraAttributes.bullet.type === 'numbering') {
                            numType = paraAttributes.bullet.numType || 'arabicPeriod';
                            textChar = self.createNumberedListItem(listItemCounter[listCounterId][listLevel], numType);
                            displayFontName = paraAttributes.bulletFont.name ? paraAttributes.bulletFont.name : null;
                        }

                        numberingElement = DOM.createListLabelNode(noListLabel ? '' : textChar);
                        span = DOM.findFirstPortionSpan(para);
                        followingTextAttributes = self.getCharacterStyles().getElementAttributes(span).character;
                        listSpans = numberingElement.children('span');

                        if (imageUrl) {

                            var absUrl = app.getServerModuleUrl(IO.FILTER_MODULE_NAME, { action: 'getfile', get_filename: imageUrl }),
                                imageWidth = followingTextAttributes.fontSize, // TODO
                                image = null;

                            image = $('<div>', { contenteditable: false })
                            .addClass('drawing')
                            .data('url', imageUrl)
                            .append($('<div>').addClass('content')
                                .append($('<img>', { src: absUrl }).css('width', imageWidth + 'pt'))
                            );

                            self.getCharacterStyles().updateElementLineHeight(image, paraAttributes.lineHeight, followingTextAttributes);
                            $(image).css('height', followingTextAttributes.fontSize + 'pt');
                            $(image).css('width', followingTextAttributes.fontSize + 'pt');
                            numberingElement.prepend(image);

                        } else if (!noListLabel) {

                            if (paraAttributes.bulletSize.type === 'followText') {
                                listSpans.css('font-size', followingTextAttributes.fontSize + 'pt');
                            } else if (paraAttributes.bulletSize.type === 'percent') {
                                listSpans.css('font-size', Utils.round((paraAttributes.bulletSize.size / 100) * followingTextAttributes.fontSize, 1) + 'pt');
                            } else {
                                listSpans.css('font-size', paraAttributes.bulletSize.size + 'pt'); // TODO: Is the size given in 'pt' ?
                            }

                            if (paraAttributes.bulletFont.followText) {
                                listSpans.css('font-family', self.getCssFontFamily(followingTextAttributes.fontName));
                            } else {
                                if (displayFontName) { listSpans.css('font-family', self.getCssFontFamily(displayFontName)); }
                            }

                            if (paraAttributes.bulletColor.followText) {
                                listSpans.css('color', self.getCssColor(followingTextAttributes.color, 'text'));
                            } else {
                                listSpans.css('color', self.getCssColor(paraAttributes.bulletColor.color, 'text'));
                            }

                            // in numbering also bold and italic are adapted from the following text span
                            if (paraAttributes.bullet.type === 'numbering' && paraAttributes.bulletFont.followText) {
                                listSpans.css('font-weight', $(span).css('font-weight'));
                                listSpans.css('font-style', $(span).css('font-style'));
                            }
                        }

                        // self.getCharacterStyles().updateElementLineHeight(numberingElement, paraAttributes.lineHeight, elementAttributes.character);

                        // setting distance between margin, numbering element and text (effects only the numbering element!)
                        if (!noListLabel && paraAttributes.indentFirstLine < 0) {
                            numberingElement.css('min-width', (-paraAttributes.indentFirstLine / 100) + 'mm');
                        }

                        if (-paraAttributes.indentFirstLine > paraAttributes.indentLeft) {
                            numberingElement.css('margin-left', (-paraAttributes.indentLeft - paraAttributes.indentFirstLine) / 100 + 'mm');
                        }

                        $(para).prepend(numberingElement);
                    }
                }

                if (updateParaTabstops) {
                    self.getParagraphStyles().updateTabStops(para);
                }
            }

            // receiving list of all document paragraphs
            paragraphNodes = paragraphs ? paragraphs : self.getCurrentRootNode().find(DOM.PARAGRAPH_NODE_SELECTOR);

            if (Utils.getBooleanOption(options, 'async', false)) {
                def = self.iterateArraySliced(paragraphNodes, updateListInParagraph, { delay: 'immediate', infoString: 'Text: updateListInParagraph' });
            } else {
                paragraphNodes.each(function () { updateListInParagraph(this); });
                def = $.when();
            }

            return def.promise();
        }

        // public methods -----------------------------------------------------

        /**
         * Getting the debounced handler function that is used for updating the lists
         * in the document. This function must be used instead of 'updateLists', after
         * the document was loaded successfully.
         *
         * @returns {Function}
         *  The debounced handler function for updating the lists.
         */
        this.getDebouncedUpdateListsHandler = function () {
            return updateListsDebounced;
        };

        /**
         * Getting the not debounced handler function that is used for updating
         * the lists synchronously. This updateLists should only be called from
         * 'updateDocumentFormatting' during the loading phase. After the document
         * was loaded successfully, 'updateListsDebounced' need to be used.
         *
         * @returns {Function}
         *  The handler function for updating the lists.
         */
        this.getUpdateListsHandler = function () {
            return updateLists;
        };

        // initialization -----------------------------------------------------

        // destroy all class members on destruction
        this.registerDestructor(function () {
            self = null;
        });

    } // class UpdateListsMixin

    // constants --------------------------------------------------------------

    // exports ================================================================

    return UpdateListsMixin;

});
