/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define('io.ox/office/presentation/model/updatedocumentmixin', [
    'io.ox/office/drawinglayer/view/drawingframe',
    'io.ox/office/textframework/utils/config',
    'io.ox/office/textframework/utils/dom',
    'io.ox/office/textframework/utils/textutils'
], function (DrawingFrame, Config, DOM, Utils) {

    'use strict';

    // mix-in class UpdateDocumentMixin ======================================

    /**
     * A mix-in class for the 'updateDocumentFormatting' function.
     *
     * @constructor
     *
     * @param {EditApplication} app
     *  The application instance.
     */
    function UpdateDocumentMixin(app) {

        var // self reference for local functions
            self = this;

        // private methods ----------------------------------------------------

        // public methods -----------------------------------------------------

        /**
         * Updates the formatting of all elements, after the import operations
         * of the document have been applied.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the formatting has been
         *  updated successfully, or rejected when an error has occurred.
         */
        this.updateDocumentFormatting = function () {

            var // the result deferred object
                def = $.Deferred(),
                // the page node
                pagediv = self.getNode(),
                // the content node of the page
                pageContentNode = DOM.getPageContentNode(pagediv),
                // all top-level paragraphs, all tables (also embedded), and all drawings
                // formattingNodes = pageContentNode.find(DOM.PARAGRAPH_NODE_SELECTOR + ', ' + DrawingFrame.NODE_SELECTOR + ', ' + DOM.TABLE_NODE_SELECTOR),
                formattingNodes = pageContentNode.find('> ' + DOM.PARAGRAPH_NODE_SELECTOR + ', ' + DrawingFrame.NODE_SELECTOR + ', ' + DOM.TABLE_NODE_SELECTOR),
                // number of progress ticks for tables (1 tick per paragraph/drawing)
                TABLE_PROGRESS_TICKS = 20,
                // total number of progress ticks
                totalProgressTicks = formattingNodes.length + (TABLE_PROGRESS_TICKS - 1) * formattingNodes.filter(DOM.TABLE_NODE_SELECTOR).length,
                // current progress for formatting process
                currentProgressTicks = 0,
                // update progress bar at most 250 times
                progressBlockSize = totalProgressTicks / 250,
                // index of progress block from last progress update
                lastProgressBlock = -1;

            // adding also the nodes inside master slide layer and layout slide layer
            formattingNodes = formattingNodes.add(self.getOrCreateLayoutSlideLayerNode().children().find('> ' + DOM.PARAGRAPH_NODE_SELECTOR + ', ' + DrawingFrame.NODE_SELECTOR + ', ' + DOM.TABLE_NODE_SELECTOR));
            formattingNodes = formattingNodes.add(self.getOrCreateMasterSlideLayerNode().children().find('> ' + DOM.PARAGRAPH_NODE_SELECTOR + ', ' + DrawingFrame.NODE_SELECTOR + ', ' + DOM.TABLE_NODE_SELECTOR));

            // first 90% for formatting (last 10% for updating lists)
            function updateFormatProgress(progressTicks) {
                var progressBlock = Math.floor(progressTicks / progressBlockSize);
                if (lastProgressBlock < progressBlock) {
                    def.notify(0.9 * progressTicks / totalProgressTicks);
                    lastProgressBlock = progressBlock;
                }
            }

            // updates the formatting of the passed element, returns a promise
            function updateElementFormatting(element) {

                var // the Promise for asynchronous element formatting
                    promise = null,
                    // whether the element is a direct child of the page content node
                    topLevel = !element.parentNode;

                // convert element to jQuery object
                element = $(element);

                // insert the detached element into the page content node
                // (performance optimization for large documents, see below)
                if (topLevel) { pageContentNode.append(element); }

                // determine the type of the passed element
                if (DOM.isSlideNode(element)) {
                    self.getSlideStyles().updateElementFormatting(element);
                    updateFormatProgress(currentProgressTicks += 1);

                } else if (DOM.isParagraphNode(element)) {
                    // validate DOM contents AND update formatting of the paragraph
                    self.validateParagraphNode(element);
                    self.getParagraphStyles().updateElementFormatting(element);
                    updateFormatProgress(currentProgressTicks += 1);

                } else if (DrawingFrame.isDrawingFrame(element)) {

                    self.getDrawingStyles().updateElementFormatting(element);
                    updateFormatProgress(currentProgressTicks += 1);

                    // also updating all paragraphs inside text frames (not searching twice for the drawings)
                    self.implParagraphChangedSync($(element).find(DOM.PARAGRAPH_NODE_SELECTOR));  // TODO: Is this necessary?

                    // trigger update of parent paragraph
                    self.trigger('paragraphUpdate:after', $(element).closest(DOM.PARAGRAPH_NODE_SELECTOR));  // TODO: Is this necessary?

                } else if (DOM.isTableNode(element) && !DOM.isExceededSizeTableNode(element)) {

                    // Bug 28409: Validate DOM contents of embedded paragraphs without
                    // formatting (done by the following table formatting). Finds and
                    // processes all cell paragraphs contained in a top-level table
                    // (also from embedded tables).
                    if (topLevel) {
                        element.find(DOM.CELLCONTENT_NODE_SELECTOR + ' > ' + DOM.PARAGRAPH_NODE_SELECTOR).each(function () {
                            self.validateParagraphNode(this);
                        });
                    }
                    // update table formatting asynchronous to prevent browser alerts
                    promise = self.getTableStyles().updateElementFormatting(element, { async: true })
                        .progress(function (localProgress) {
                            updateFormatProgress(currentProgressTicks + TABLE_PROGRESS_TICKS * localProgress);
                        })
                        .always(function () {
                            currentProgressTicks += TABLE_PROGRESS_TICKS;
                        });
                }

                return promise;
            }

            // updates the formatting of all lists, forwards the progress, returns a promise
            function updateListFormatting() {
                return self.updateLists({ async: true }).progress(function (progress) {
                    def.notify(0.9 + 0.1 * progress);  // last 10% for updating lists
                });
            }

            // dumps profiling information to browser console
            function dumpProfilingInformation() {
                var tableCount,
                    cellCount,
                    paragraphCount,
                    spanCount,
                    drawingCount;

                function dumpUpdateFormattingCount(styleSheets, elementCount, elementName) {
                    var updateCount = styleSheets.DBG_COUNT || 0,
                        perElementCount = (elementCount === 0) ? 0 : Utils.round(updateCount / elementCount, 0.01);
                    if ((elementCount > 0) || (updateCount > 0)) {
                        Utils.info('Editor.updateDocumentFormatting(): ' + elementCount + ' ' + elementName + ' updated ' + updateCount + ' times (' + perElementCount + ' per element)');
                    }
                }

                function dumpParagraphCount(methodName, debugCount) {
                    var updateCount = debugCount || 0,
                        perElementCount = (paragraphCount === 0) ? 0 : Utils.round(updateCount / paragraphCount, 0.01);
                    Utils.info('Editor.' + methodName + '(): called ' + updateCount + ' times (' + perElementCount + ' per paragraph)');
                }

                if (!Config.DEBUG) { return $.noop; }

                tableCount = pagediv.find(DOM.TABLE_NODE_SELECTOR).length;
                cellCount = pagediv.find('td').length;
                paragraphCount = pagediv.find(DOM.PARAGRAPH_NODE_SELECTOR).length;
                spanCount = pagediv.find('span').filter(function () { return DOM.isPortionSpan(this) || DOM.isTextComponentNode(this.parentNode); }).length;
                drawingCount = pagediv.find(DrawingFrame.NODE_SELECTOR).length;

                dumpUpdateFormattingCount(self.getCharacterStyles(), spanCount, 'text spans');
                dumpUpdateFormattingCount(self.getParagraphStyles(), paragraphCount, 'paragraphs');
                dumpUpdateFormattingCount(self.getTableCellStyles(), cellCount, 'table cells');
                dumpUpdateFormattingCount(self.getTableStyles(), tableCount, 'tables');
                dumpUpdateFormattingCount(self.getDrawingStyles(), drawingCount, 'drawing objects');
                dumpParagraphCount('validateParagraphNode', self.getValidateParagraphNode().DBG_COUNT);
            }

            // resetting an existing selection, after all operations are applied
            self.getSelection().resetSelection();
            // lastOperationEnd = null; // -> trying, not to use this

            // receiving list of all change track authors of the document
            app.addAuthors(self.getDocumentAttributes().changeTrackAuthors);

            // detach all child nodes of the page content node, this improves
            // performance in very large documents notably
            formattingNodes.filter(function () { return this.parentNode === pageContentNode[0]; }).detach();

            // first, update the root page node
            self.getPageStyles().updateElementFormatting(pagediv);

            this.iterateArraySliced(formattingNodes, updateElementFormatting, { delay: 'immediate', infoString: 'Presentation: updateElementFormatting' })
                // .then(updateDrawingGroups)
                .then(updateListFormatting)
                .done(function () { dumpProfilingInformation(); def.resolve(); })
                .fail(function () { def.reject(); });

            return def.promise();
        };

        // initialization -----------------------------------------------------

        // destroy all class members on destruction
        this.registerDestructor(function () {
            self = null;
        });

    } // class UpdateDocumentMixin

    // exports ================================================================

    return UpdateDocumentMixin;

});
