/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define('io.ox/office/presentation/model/model', [
    'io.ox/office/tk/utils',
    'io.ox/office/presentation/model/listhandlermixin',
    'io.ox/office/presentation/model/modelattributesmixin',
    'io.ox/office/presentation/model/slideoperationmixin',
    'io.ox/office/presentation/model/objectoperationmixin',
    'io.ox/office/presentation/model/pagehandlermixin',
    'io.ox/office/presentation/model/updatedocumentmixin',
    'io.ox/office/presentation/model/updatelistsmixin',
    'io.ox/office/presentation/components/drawing/drawingresize',
    'io.ox/office/textframework/model/editor',
    'io.ox/office/editframework/utils/attributeutils',
    'io.ox/office/editframework/utils/border',
    'io.ox/office/editframework/utils/color',
    'io.ox/office/textframework/utils/dom',
    'io.ox/office/textframework/utils/position',
    'io.ox/office/textframework/components/hyperlink/hyperlink',
    'io.ox/office/textframework/components/table/tableresize',
    'io.ox/office/textframework/utils/config',
    'io.ox/office/presentation/utils/operations',
    'io.ox/office/presentation/utils/presentationutils',
    'io.ox/office/drawinglayer/view/drawingframe',
    'io.ox/office/drawinglayer/view/imageutil'
], function (Utils, ListHandlerMixin, ModelAttributesMixin, SlideOperationMixin, ObjectOperationMixin, PageHandlerMixin, UpdateDocumentMixin, UpdateListsMixin, DrawingResize, Editor, AttributeUtils, Border, Color, DOM, Position, Hyperlink, TableResize, Config, Operations, PresentationUtils, DrawingFrame, Image) {

    'use strict';

    // class PresentationModel ================================================

    /**
     * Represents the document model of a presentation application.
     *
     * Triggers the events supported by the base class EditModel, and the
     * following additional events:
     * - 'inserted:slide': A new slide was inserted.
     * - 'removed:slide': A slide was removed.
     * - 'change:activeView': The active view was modified.
     * - 'change:slide': The active slide was modified.
     * - 'change:slidelayout': A layout or master slide was modified.
     * - 'slideModel:init': All slide related model values are ready to use (e.g. for initializing the slide pane).
     *
     * @constructor
     *
     * @extends Editor
     * @extends ListHandlerMixin
     * @extends ModelAttributesMixin
     * @extends PageOperationMixin
     * @extends ObjectOperationMixin
     * @extends PageHandlerMixin
     * @extends UpdateDocumentMixin
     * @extends UpdateListsMixin
     *
     * @param {PresentationApplication} app
     *  The application containing this document model.
     */
    function PresentationModel(app) {

        var // self reference for local functions
            self = this,
            // the root element for the document contents
            pagediv = null,
            // the logical selection, synchronizes with browser DOM selection
            selection = null,
            // a counter for the 'standard' slides, used to generate unique keys for the slides (only used locally)
            mainSlideCounter = 1,
            // shortcuts for style sheet containers
            characterStyles = null, paragraphStyles = null, tableStyles = null, tableCellStyles = null, tableRowStyles = null, drawingStyles = null, slideStyles = null, pageStyles = null,
            // the model for all 'standard' slides. Those slides, that are not master or layout slides.
            allStandardSlides = {},
            // the model for the layout slides, an object with target string as key and target node as value
            allLayoutSlides = {},
            // the model for the master slides, an object with target string as key and target node as value
            allMasterSlides = {},
            // the model for standard/layout slides -> the key is the id of the standard slide, the value the key of the layout slide
            layoutConnection = {},
            // the model for layout/master slides -> the key is the id of the layout slide, the value the key of the master slide
            layoutMasterConnection = {},
            // a collector that combines the id of a slide with the type as key value pair
            idTypeConnection = {},
            // container with the slide IDs in the correct order (model)
            slideOrder = [],
            // container with the slide IDs of master and layout slides in the correct order (model)
            masterSlideOrder = [],
            // the id of the currently active slide (set in setActiveSlideId)
            activeSlideId = '',
            // the id of the active slide that was used before switching to master/layout view
            lastActiveStandardSlideId = '',
            // whether the master and layout slides are visible (true) or the 'standard' slides are visible (model) (set in setActiveView)
            isMasterView = false,
            // the layer node for all layout slide nodes
            layoutSlideLayerNode = null,
            // the layer node for all master slide nodes
            masterSlideLayerNode = null,
            // a selection range in a master or layout slide
            masterLayoutSelection = null;

        // base constructor ---------------------------------------------------

        Editor.call(this, app, { useSlideMode: true });
        SlideOperationMixin.call(this);
        ObjectOperationMixin.call(this);
        ListHandlerMixin.call(this);
        ModelAttributesMixin.call(this);
        PageHandlerMixin.call(this, app);
        UpdateDocumentMixin.call(this, app);
        UpdateListsMixin.call(this, app);

        // private methods ----------------------------------------------------

        /**
         * Registering one layout or master slide in the slide model.
         *
         * @param {jQuery} slide
         *      The jQuerified slide that will be saved inside the slide collector. The key
         *      inside this collector is the id, the value the jQuerified slide node.
         *
         * @param {String} id
         *      The id of the slide. For master and layout slides, this id is sent from the
         *      server. For 'standard' slides a locally generated id is used.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {String} [options.type='']
         *      The type of the slide. This is required to distinguish master slides, layout
         *      slide and 'standard' slides. The latter are the default.
         *  @param {String} [options.target='']
         *      The target of the slide that will be registered. This is especially important
         *      for the connection from layout slides to master slides.
         *  @param {Number} [options.index=-1]
         *      The zero-based index describing the position of the slide inside the currently
         *      active view.
         *
         * @returns {Number}
         *  The number of slides in the standard or master/layout model.
         */
        function addIntoSlideModel(slide, id, options) {

            var // the type of the slide
                type = Utils.getStringOption(options, 'type', ''),
                // the target of the slide
                target = null,
                // the order index at which the slide shall be inserted
                index = -1,
                // whether the slide is a standard slide
                isStandardSlide = false,
                // whether the model is modified
                insertedSlide = false;

            if (type === PresentationModel.TYPE_SLIDE) {

                isStandardSlide = true;
                insertedSlide = true;

                // adding the slide into the model
                allStandardSlides[id] = slide;

                // handling the slide order
                index = Utils.getNumberOption(options, 'index', -1);

                if (index > -1) {
                    slideOrder.splice(index, 0, id);
                } else {
                    slideOrder.push(id); // if index not specified, append slide to the end
                }

                // reading the required options
                target = Utils.getStringOption(options, 'target', '');

                // saving connection between standard slide and its layout slide
                if (target) { layoutConnection[id] = target; }

            } else if (type === PresentationModel.TYPE_MASTER_SLIDE) {

                insertedSlide = true;

                // saving the jQuerified slide in the model
                allMasterSlides[id] = slide;

                // handling the slide order (appending at the end)
                masterSlideOrder.push(id);

            } else if (type === PresentationModel.TYPE_LAYOUT_SLIDE) {

                insertedSlide = true;

                // saving the jQuerified slide in the model
                allLayoutSlides[id] = slide;

                // handling the slide order (appending at the end)
                masterSlideOrder.push(id); // if index not specified, append slide to the end

                // reading the required options
                target = Utils.getStringOption(options, 'target', '');

                // saving connection between layout slide and its master slide
                if (target) { layoutMasterConnection[id] = target; }

            } else {
                Utils.log('Warning: Unknown slide type: ' + type);
            }

            // registering the type of the slide for every id
            idTypeConnection[id] = type;

            // informing the listeners, that a new slide was inserted (model was updated)
            if (insertedSlide) { self.trigger('inserted:slide', { id: id, isMasterView: isMasterView }); }

            return isStandardSlide ? slideOrder.length : masterSlideOrder.length;
        }

        /**
         * Removing a slide specified by its ID from the model.
         * Before calling this function it must be checked, if the specified slide is the
         * only slide in its view (self.isOnlySlideInView(id)) or if it is a used
         * master or layout slide (self.isUsedMasterOrLayoutSlide(id)).
         *
         * @param {String} id
         *      The id of the slide. For master and layout slides, this id is sent from the
         *      server. For 'standard' slides a locally generated id is used.
         *
         * @returns {Number}
         *  The number of slides in the standard or master/layout model.
         */
        function removeFromSlideModel(id) {

            // Required checks:
            // self.isUsedMasterOrLayoutSlide(id) -> not removing master/layout slides that are referenced
            // self.isOnlySlideInView(id)         -> not removing the last slide in its view

            var // whether the slide is a standard slide
                isStandardSlide = false,
                // whether the model is modified
                removedSlide = false;

            if (self.isStandardSlideId(id)) {

                isStandardSlide = true;
                removedSlide = true;

                // removing the slide from the model
                delete allStandardSlides[id];

                // removing the connection between standard slide and its layout slide
                delete layoutConnection[id];

                // removing the id from the slide order array
                slideOrder = _.without(slideOrder, id);

            } else if (self.isLayoutSlideId(id)) {

                removedSlide = true;

                // removing the slide from the model
                delete allLayoutSlides[id];

                // removing the connection between layout slide and its masterlayout slide
                delete layoutMasterConnection[id];

                // removing the id from the master slide order array
                masterSlideOrder = _.without(masterSlideOrder, id);

            } else if (self.isMasterSlideId(id)) {

                removedSlide = true;

                // removing the slide from the model
                delete allMasterSlides[id];

                // removing the id from the master slide order array
                masterSlideOrder = _.without(masterSlideOrder, id);
            }

            // removing registration of type for the slide
            delete idTypeConnection[id];

            // informing the listeners, that a slide was removed (model was updated)
            if (removedSlide) { self.trigger('removed:slide', { id: id, isMasterView: isMasterView }); }

            return isStandardSlide ? slideOrder.length : masterSlideOrder.length;
        }

        /**
         * Setting the id of the active slide (model).
         *
         * @param {String} id
         *  The zero-based index of the active slide inside the standard or  master/layout view.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.forceUpdate=false]
         *   Whether a view update is required, even if the index of the slide did not change.
         *
         * @returns {Boolean}
         *  Whether another slide was activated.
         */
        function setActiveSlideId(id, options) {

            var // whether the slide was modified
                modified = activeSlideId !== id,
                // whether an update is forced
                forceUpdate = Utils.getBooleanOption(options, 'forceUpdate', false),
                // whether another view was activated
                activated = false,
                // the id of the currently active slide
                oldId = activeSlideId,
                // collector for all modified slides
                viewCollector = {},
                // the logical position of the active slide
                slidePos = null;

            if (isValidSlideId(id)) {

                // modifying the model
                activeSlideId = id;

                if (modified || forceUpdate) {

                    // setting all IDs that need to be activated
                    if (self.isStandardSlideId(id)) {
                        viewCollector.newSlideId = id;
                        viewCollector.newLayoutId = self.getLayoutSlideId(viewCollector.newSlideId);
                        viewCollector.newMasterId = self.getMasterSlideId(viewCollector.newLayoutId);
                    } else if (self.isLayoutSlideId(id)) {
                        viewCollector.newLayoutId = id;
                        viewCollector.newMasterId = self.getMasterSlideId(viewCollector.newLayoutId);
                    } else if (self.isMasterSlideId(id)) {
                        viewCollector.newMasterId = id;
                    }

                    // setting all IDs that need to be deactivated
                    if (self.isStandardSlideId(oldId)) {
                        viewCollector.oldSlideId = oldId;
                        viewCollector.oldLayoutId = self.getLayoutSlideId(viewCollector.oldSlideId);
                        viewCollector.oldMasterId = self.getMasterSlideId(viewCollector.oldLayoutId);
                    } else if (self.isLayoutSlideId(oldId)) {
                        viewCollector.oldLayoutId = oldId;
                        viewCollector.oldMasterId = self.getMasterSlideId(viewCollector.oldLayoutId);
                    } else if (self.isMasterSlideId(oldId)) {
                        viewCollector.oldMasterId = oldId;
                    }

                    // add the currently active slide to the viewCollector
                    viewCollector.activeSlideId = self.getActiveSlideId();

                    // informing the listeners, that the slide need to be updated (model was updated)
                    self.trigger('change:slide', viewCollector);

                    // clearing the selection of the selected drawing
                    if (self.getSelection().getSelectedDrawing().length > 0) {
                        DrawingFrame.clearSelection(self.getSelection().getSelectedDrawing());
                    }

                    // updating the value for the active target (TODO)
                    self.setActiveTarget(self.isLayoutOrMasterId(activeSlideId) ? activeSlideId : '');
                    self.getSelection().setNewRootNode(self.getRootNode(activeSlideId));
                    // selecting the first element on the slide
                    slidePos = self.getActiveSlidePosition();
                    self.getSelection().setTextSelection(Position.appendNewIndex(slidePos, 0), Position.appendNewIndex(slidePos, 1));

                    // a new slide was activated
                    activated = true;
                }
            }

            return activated;
        }

        /**
         * Setting the active view (model) and trigger a change:activeView event to notify observers.
         * The active view can be the view of the standard slides or the view of the master/layout slides.
         *
         * @param {Boolean} isMaster
         *  Whether the view of the master/layout slides shall be activated.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.forceUpdate=false]
         *   Whether a view update is required, even if the saved value does not change.
         *
         * @returns {Boolean}
         *  Whether another slide view was activated.
         */
        function setActiveView(isMaster, options) {

            var // whether an update is forced
                forceUpdate = Utils.getBooleanOption(options, 'forceUpdate', false),
                // whether the slide view was modified
                modified = (isMaster && !isMasterView) || (!isMaster && isMasterView) || forceUpdate;

            // updating the model
            if (modified) {
                isMasterView = isMaster;
                // informing the listeners, that the activeView is changed (model was updated)
                self.trigger('change:activeView', { isMasterView: isMasterView });
            }

            return modified;
        }

        /**
         * Provides a unique ID for the standard slides. This ID is used locally only. The master
         * and layout slides get their IDs from the server. The local ID for the standard slides
         * is used for the model only.
         *
         * @returns {String}
         *  A unique id.
         */
        function getNextSlideID() {
            return 'slide_' + mainSlideCounter++;
        }

        /**
         * Getting the index of the active slide in the current container
         *
         * @returns {Number}
         *  The index of the active slide in the current container. If the
         *  active slide id could not be found in the array, '-1' is returned.
         */
        function getActiveSlideIndex() {

            var // the container that contains the order of the slides
                orderContainer = isMasterView ? masterSlideOrder : slideOrder;

            return _.indexOf(orderContainer, activeSlideId);
        }

        /**
         * Function that is executed after the 'import success' event of the application.
         */
        function importSuccessHandler() {

            var // the collection of all slides
                allSlides = self.getAllSlides();

            // no functionality of page layout required
            self.deletePageLayout();
            // no functionality of drawing layer required
            self.disableDrawingLayer();
            // deleting place holder for header and footer
            DOM.getMarginalTemplateNode(pagediv).remove();

            // making all slides invisible
            _.invoke(allSlides, 'addClass', 'invisibleslide');

            // activating the first slide
            setActiveSlideId(self.getIdOfFirstSlideOfActiveView(), { forceUpdate: true });

            // setting the handler function for updating lists after the document is imported successfully
            self.setUpdateListsDebounced(self.getDebouncedUpdateListsHandler());

            // trigger an event to initialize the SlidePane
            self.trigger('slideModel:init', { isMasterView: isMasterView, activeSlideId: self.getActiveSlideId() });

            // register the listener for 'change:slidelayout'.
            // -> this update shall not be triggered during document load.
            self.listenTo(self, 'change:slidelayout', updateDrawingLayoutFormatting);

            // register the listener for 'change:activeView'.
            // -> this update shall not be triggered during document load.
            self.listenTo(self, 'change:activeView', updateActiveViewHandler);
        }

        /**
         * The handler function for the event 'change:defaults'. Within the object
         * changedAttributes the default text list styles are defined, that will be
         * used in non-placeholder drawings, like text frames or shapes.
         *
         * @param {jQuery.Event} event
         *  The 'change:defaults' event.
         *
         * @param {Object} newAttributes
         *  An object with the new attributes (will be ignored).
         *
         * @param {Object} oldAttributes
         *  An object with the old attributes (will be ignored).
         *
         * @param {Object} changedAttributes
         *  An object with the changed attributes.
         */
        function registerDefaultTextListStyles(event, newAttributes, oldAttributes, changedAttributes) {

            if (changedAttributes && changedAttributes.defaultTextListStyles) {
                self.setDefaultTextListStyles(changedAttributes.defaultTextListStyles);
            }
        }

        /**
         * Receiving the layer node for the layout slides. If it does not exist, it is created.
         *
         * @returns {jQuery}
         *  The layout slide layer node
         */
        function getOrCreateLayoutSlideLayerNode() {

            // is there already a layout slide layer node
            if (layoutSlideLayerNode) { return layoutSlideLayerNode; }

            // creating the new node
            layoutSlideLayerNode = $('<div>').addClass(DOM.LAYOUTSLIDELAYER_CLASS);

            // inserting the new node before the page content node
            layoutSlideLayerNode.insertBefore(DOM.getPageContentNode(pagediv));

            return layoutSlideLayerNode;
        }

        /**
         * Receiving the layer node for the master slides. If it does not exist, it is created.
         *
         * @returns {jQuery}
         *  The master slide layer node
         */
        function getOrCreateMasterSlideLayerNode() {

            // is there already a layout slide layer node
            if (masterSlideLayerNode) { return masterSlideLayerNode; }

            masterSlideLayerNode = $('<div>').addClass(DOM.MASTERSLIDELAYER_CLASS);

            // inserting the new node before the page content node
            masterSlideLayerNode.insertBefore(DOM.getPageContentNode(pagediv));

            return masterSlideLayerNode;
        }

        /**
         * Adding a new master or layout slide into the layer node.
         *
         * @param {jQuery} layerNode
         *  The layout layer node or master layer node, to which the new slide will be
         *  appended.
         *
         * @param {String} id
         *  The id string of the layout or master slide.
         *
         * @returns {jQuery}
         *  The new created slide node inside the layout layer node or master layer node.
         */
        function appendSlideToLayerNode(layerNode, id) {

            var // the new slide node
                slide = DOM.createSlideNode(),
                // the slide container (used for positioning)
                slideContainer = $('<div>').addClass(DOM.SLIDECONTAINER_CLASS).append(slide);

            // assigning the id of the slide also to the content node (the slide's parent)
            DOM.setTargetContainerId(slideContainer, id);

            // appending the new slide
            layerNode.append(slideContainer);

            // insert required helper nodes
            self.validateParagraphNode(slide);

            return slide;
        }

        /**
         * Provides the standard slide node corresponding to the specified id string.
         *
         * @param {String} id
         *  The slide id.
         *
         * @returns {jQuery|null}
         *  The standard slide node. Or null, if no standard slide node with the specified id exists.
         */
        function getStandardSlide(id) {
            return allStandardSlides[id] || null;
        }

        /**
         * Provides the layout slide node in the layout slide layer corresponding to the specified
         * id string.
         *
         * @param {String} id
         *  The slide id.
         *
         * @returns {jQuery|null}
         *  The slide node inside the layout layer node. Or null, if no layout slide node with the
         *  specified id exists.
         */
        function getLayoutSlide(id) {
            return allLayoutSlides[id] || null;
        }

        /**
         * Provides the master slide node in the master slide layer corresponding to the specified
         * id string.
         *
         * @param {String} id
         *  The id string.
         *
         * @returns {jQuery|null}
         *  The slide node inside the master layer node. Or null, if no master slide node with the
         *  specified id exists.
         */
        function getMasterSlide(id) {
            return allMasterSlides[id] || null;
        }

        /**
         * Getting the last top level position inside the slide, that has the current
         * selection.
         *
         * @returns {Array<Number>}
         *  The next available position at the root of the slide.
         */
        function getNextAvailablePositionInActiveSlide() {

            var // the active slide
                slide = self.getSlideById(activeSlideId),
                // the position of the active slide
                pos = self.getActiveSlidePosition();

            pos.push(slide.children(DrawingFrame.NODE_SELECTOR).length); // only drawings are relevant children

            return pos;
        }

        /**
         * Provides the content node of the layout slide node or master slide node in the layout slide
         * layer or master slide layer corresponding to the specified target string. This node can be
         * used for calculations of logical positions (not the slide node itself).
         *
         * @param {String} id
         *  The slide id.
         *
         * @returns {jQuery|null}
         *  The slide content node inside the layout or master slide layer node. This content node is
         *  the parent of the slide node. Or null, if no slide with the specified target exists.
         */
        function getLayoutOrMasterSlideContentNode(id) {

            var // the slide node, that is the child of the content node
                slide = self.getLayoutOrMasterSlide(id);

            return slide ? slide.parent() : null;
        }

        /**
         * Checking, whether a specified id is a valid and registered slide id.
         *
         * @param {String} id
         *  The slide id to be checked.
         *
         * @returns {Boolean}
         *  Whether the specified id is a valid value for a slide id.
         */
        function isValidSlideId(id) {
            return _.isString(idTypeConnection[id]);
        }

        /**
         * Forcing the update of the slides, that have a dependency to a specified target. The
         * target must be defined inside the data object. The event 'change:slidelayout' is
         * triggered to inform all listeners.
         *
         * @param {Object} data
         *  This object is generated inside the function 'savePlaceHolderAttributes' in the
         *  drawing styles. The object's keys are: 'target', 'placeHolderType', 'placeHolderIndex'
         *  and 'attrs'.
         */
        function triggerLayoutChangeEvent(data) {
            self.trigger('change:slidelayout', data);
        }

        /**
         * This function is called, if a node received new attributes. If this node is inside
         * a layout or master slide, it is necessary to update the collector for the attributes
         * of the drawings that are located inside a layout or master slide (the model is the object
         * 'placeHolderCollector'.
         *
         * Because this function is triggered after (!) the drawing received the new attributes,
         * these new attributes are already assigned to the drawing and saved in the data object
         * at the drawing node.
         *
         * Additionally all slides, that have the affected layout or master slide as target,
         * need to be repainted (or invalidated?). Therefore this function itself triggers the
         * event 'change:slidelayout'.
         *
         * @param {HTMLElement|jQuery} element
         *  The element node, that received the new attributes.
         *
         * @param {String} target
         *  The active target for the specified element.
         *
         * @param {String} family
         *  The family of the attribute set.
         *
         * @param {Object} attributes
         *  A map with formatting attribute values, mapped by the attribute
         *  names, and by attribute family names.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {String} [options.immediateUpdate=true]
         *      Whether slides that are dependent from the layout or master slide change
         *      shall be updated immediately. Default is the immediate update.
         *  @param {String} [options.saveOldAttrs=false]
         *      Whether the old attributes shall be saved and returned in the return object.
         *
         * @returns {Object|Null}
         *  If the attributes are registered, an object with the complete registration
         *  information is returned. The object's keys are: 'target', 'placeHolderType',
         *  'placeHolderIndex', 'attrs' and 'oldAttrs'. If the attributes were not registered,
         *  because target, type or index are not defined, null is returned.
         */
        function registerPlaceHolderAttributes(element, target, family, attrs, options) {

            var // the drawing node containing the specified element
                drawing = null,
                // a paragraph node
                paragraph = null,
                // the explicit attributes at the drawing
                drawingAttrs = null,
                // the paragraph level
                paraLevel = 0,
                // the key in the list styles corresponding to the paragraph level
                paraKey = null,
                // an object containing the information about the saved values
                registrationObject = null,
                // whether drawing attributes shall be registered
                isDrawing = (family === 'drawing'),
                // whether an update of dependent slide shall be triggered immediately
                immediateUpdate = Utils.getBooleanOption(options, 'immediateUpdate', true),
                // whether an update of dependent slide shall be triggered immediately
                saveOldAttrs = Utils.getBooleanOption(options, 'saveOldAttrs', false);

            if (target && self.isLayoutOrMasterId(target)) {

                drawing = isDrawing ? $(element) : $(element).closest('div.drawing');

                if (drawing.length > 0 && PresentationUtils.isPlaceHolderDrawing(drawing)) {

                    if (isDrawing) {
                        drawingAttrs = attrs;
                    } else {
                        paragraph = (family === 'paragraph') ? $(element) : $(element).closest(DOM.PARAGRAPH_NODE_SELECTOR);
                        paraLevel = paragraphStyles.getParagraphLevel(paragraph);
                        paraKey = paragraphStyles.generateKeyFromLevel(paraLevel);

                        drawingAttrs = { listStyle: {} };
                        drawingAttrs.listStyle[paraKey] = attrs;
                    }

                    registrationObject = self.getDrawingStyles().savePlaceHolderAttributes(target, drawingAttrs, drawing, { saveOldAttrs: saveOldAttrs });

                    if (immediateUpdate && registrationObject) {

                        // informing the listeners that place holder drawing styles were modified and that
                        // all affected drawings (that use the modified place holder type and index at a slide
                        // with the specified target) need to be updated.
                        // Therefore the registrationObject is 'complete', with type and index containing at
                        // least the default values.
                        triggerLayoutChangeEvent(registrationObject);
                    }
                }
            }

            return registrationObject;
        }

        /**
         * Inserting an image as slide background. This function generates the required
         * 'setAttributes' operation.
         *
         * @param {Object} imageHolder
         *  The object containing the url for the image.
         */
        function insertImageForSlideBackground(imageHolder) {

            var // the operation attributes
                attrs = { fill: { type: 'bitmap', color: null } },
                // the new operation
                newOperation = { name: Operations.SET_ATTRIBUTES, start: self.getActiveSlidePosition(), attrs: attrs };

            if (imageHolder.url) {
                attrs.fill.imageUrl = imageHolder.url;
            } else {
                attrs.fill.imageUrl = imageHolder;
            }

            self.extendPropertiesWithTarget(newOperation, self.getActiveTarget());
            self.applyOperations(newOperation);
        }

        // private listener functions -----------------------------------------

        /**
         * Listener function for the event 'change:slidelayout'. This function is called after modifications
         * of layout or master slide. Then it is necessary to update all slides, that use the modified slide
         * as target (or whose target uses the modified slide as target).
         *
         * Info: This function is critical for the performance. It might be necessary to update
         *       many elements. Therefore this should be done asynchronous (TODO).
         *
         * -> TODO: Shifting this into another class?
         *
         * @param {jQuery.Event} event
         *  The 'drawingUpdate:after' event
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {String} [options.target='']
         *   The target of the layout or master slide that was modified
         *  @param {String} [options.placeHolderType='']
         *   The place holder type that was modified.
         *  @param {Number} [options.placeHolderIndex=null]
         *   The place holder type that was modified.
         *  @param {Object} [options.attrs=null]
         *   The set of modified attributes.
         */
        function updateDrawingLayoutFormatting(event, options) {

            var // the target of the modified master or layout slide
                target = Utils.getStringOption(options, 'target', ''),
                // a container with the IDs of all slides affected by the modification of the master/layout slide
                allSlideIDs = self.getAllSlideIDsWithSpecificTargetAncestor(target),
                // the type of place holder that was modified
                type = allSlideIDs.length > 0 ? Utils.getStringOption(options, 'placeHolderType', '') : null,
                // the type of place holder that was modified
                index = allSlideIDs.length > 0 ? Utils.getNumberOption(options, 'placeHolderIndex', null) : null,
                // the changed attributes
                attrs = allSlideIDs.length > 0 ? Utils.getObjectOption(options, 'attrs', null) : null;

            if (target && type && _.isNumber(index) && attrs) {

                _.each(allSlideIDs, function (id) {

                    var // a slide node
                        slide = self.getSlideById(id),
                        // the selector specific to the place holder type and index. This is used to find the drawing
                        // that need an update of formatting (the model is already updated).
                        selector = (type === 'body') ? 'div[placeholderindex=' + index + ']' : 'div[placeholdertype=' + type + ']';

                    // updating all drawings, that have the specified place holder type
                    // -> using options.baseAttributes for the place holder attributes from layout slide and master slide
                    _.each(slide.children(selector), function (drawing) {

                        // updating the drawing itself, if required
                        if (attrs.drawing) { self.getDrawingStyles().updateElementFormatting($(drawing)); }

                        // updating the paragraphs, if required
                        if (attrs.listStyle) {
                            _.each($(drawing).find(DOM.PARAGRAPH_NODE_SELECTOR), function (para) {
                                // updating all paragraphs, that have a modified attribute set
                                if (paragraphStyles.generateKeyFromLevel(paragraphStyles.getParagraphLevel(para)) in attrs.listStyle) {
                                    paragraphStyles.updateElementFormatting($(para));
                                    self.updateListsDebounced($(para), { singleParagraph: true }); // only list update for the specified paragraph
                                }
                            });
                        }
                    });
                });
            }

        }

        /**
         * Listener function for the event 'change:activeView'. This function is called when the active view
         * (model) is changed, or to put it simply, when the view is switched between the 'normal' slides and
         * the master/layout slides. After that it is necessary to activate the appropriate slide in the view.
         *
         * @param {jQuery.Event} event
         *  The 'change:activeView' event
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.isMasterView=false]
         *   Whether the masterView was activated.
         */
        function updateActiveViewHandler(event, options) {

            var // the id of the slide that will be activated
                newSlideId = null,
                // whether the master/layout view will be activated or not
                showMaster = Utils.getBooleanOption(options, 'isMasterView', false);

            // activating a slide in the view
            if (showMaster) {
                lastActiveStandardSlideId = activeSlideId;
                newSlideId = self.getLayoutOrMasterSlideId(activeSlideId);
            } else {
                newSlideId = lastActiveStandardSlideId;
            }

            setActiveSlideId(newSlideId || self.getIdOfFirstSlideOfActiveView(), { forceUpdate: true });
        }

//        /**
//         * Updating the top value of a drawing node. This is necessary, if a drawing node has its vertical
//         * text content aligned to bottom or centered and if this drawing has 'autoResizeHeight' set to true.
//         * In that case, the drawing needs to grow in top direction.
//         * -> Problem: Can this happen without an operation?
//         *
//         * @param {jQuery.Event} event
//         *  The 'drawingUpdate:after' event.
//         *
//         * @param {jQuery} node
//         *  The drawing node.
//         */
//        function updateVerticalDrawingPosition(event, node) {
//
//            var // the text frame node inside the drawing node
//                textFrame = null,
//                // the vertical text aligment property
//                verticalAlign = null,
//                // the height saved at the node
//                savedHeight = null,
//                // the new height of the drawing node
//                newHeight = null,
//                // the difference of old and new height
//                difference = null,
//                // the new value for the top offset
//                newTopOffset = null;
//
//            if (DrawingFrame.isAutoResizeHeightDrawingFrame(node)) {
//
//                // the text frame node inside the drawing node
//                textFrame = DrawingFrame.getTextFrameNode(node);
//                // the vertical text alignment property
//                verticalAlign = (textFrame && textFrame.length > 0) ? textFrame.attr(DrawingFrame.VERTICAL_ALIGNMENT_ATTRIBUTE) : '';
//
//                if (verticalAlign && (verticalAlign === 'bottom' || verticalAlign === 'centered')) {
//
//                    if (!_.isString(node.attr('verticalTop'))) { saveVerticalPositionAttributes(null, node); }
//
//                    savedHeight = Utils.getElementAttributeAsInteger(node, 'verticalHeight', 0);
//                    newHeight = node.height();
//
//                    if (savedHeight !== newHeight) {
//                        difference = savedHeight - newHeight;
//                        if (verticalAlign === 'centered') { difference = Utils.round(difference / 2, 1); }
//                        newTopOffset = Utils.getElementAttributeAsInteger(node, 'verticalTop', 0) + difference;
//                        node.css('top', newTopOffset + 'px');
//                        node.attr({ verticalTop: newTopOffset, verticalHeight: newHeight });
//                    }
//                }
//            }
//        }
//
//        /**
//         * Saving the vertical position and height of the text frame. This values can be used
//         * to calculate a possible vertical movement 'upwards' of the text frame. This is only
//         * necessary, if the vertical text alignment is set to 'bottom' or 'centered'.
//         * This values need to be set for example from 'handleAdditionalTextFrameSelection'
//         * -> never overwrite existing values
//         *
//         * @param {jQuery.Event} event
//         *  The 'drawingUpdate:after' event.
//         *
//         * @param {jQuery} node
//         *  The drawing node.
//         */
//        function saveVerticalPositionAttributes(event, node) {
//            if (DrawingFrame.isAutoResizeHeightDrawingFrame(node) && !_.isNumber(node.attr('verticalTop'))) {
//                node.attr({ verticalTop: Utils.convertCssLength($(node).css('top'), 'px', 1), verticalHeight: node.height() });
//            }
//        }

        // public methods -----------------------------------------------------

        /**
         * Public helper function: Adding a slide in the model.
         * See description at private function 'addIntoSlideModel'.
         */
        this.addIntoSlideModel = function (slide, id, options) {
            return addIntoSlideModel(slide, id, options);
        };

        /**
         * Public helper function: Removing a slide from the model.
         * See description at private function 'removeFromSlideModel'.
         */
        this.removeFromSlideModel = function (id) {
            return removeFromSlideModel(id);
        };

        /**
         * Public helper function: Provides a unique ID for the standard slides.
         * See description at private function 'getNextSlideID'.
         */
        this.getNextSlideID = function () {
            return getNextSlideID();
        };

        /**
         * Public helper function: Receiving the layer node for the layout slides.
         * If it does not exist, it is created.
         * See description at private function 'getOrCreateLayoutSlideLayerNode'.
         */
        this.getOrCreateLayoutSlideLayerNode = function () {
            return getOrCreateLayoutSlideLayerNode();
        };

        /**
         * Public helper function: Receiving the layer node for the master slides.
         * If it does not exist, it is created.
         * See description at private function 'getOrCreateMasterSlideLayerNode'.
         */
        this.getOrCreateMasterSlideLayerNode = function () {
            return getOrCreateMasterSlideLayerNode();
        };

        /**
         * Public helper function: Adding a new master or layout slide into the layer node.
         * See description at private function 'appendSlideToLayerNode'.
         */
        this.appendSlideToLayerNode = function (layerNode, id) {
            return appendSlideToLayerNode(layerNode, id);
        };

        /**
         * Public helper function: Setting the id of the active slide.
         * See description at private function 'setActiveSlideId'.
         */
        this.setActiveSlideId = function (id, options) {
            return setActiveSlideId(id, options);
        };

        /**
         * Public helper function: Getting the last top level position inside the slide,
         * that has the current selection.
         * See description at private function 'getNextAvailablePositionInActiveSlide'.
         */
        this.getNextAvailablePositionInActiveSlide = function () {
            return getNextAvailablePositionInActiveSlide();
        };

        /**
         * Getting the id of the first slide in the currently active view.
         *
         * @returns {String}
         *  The id of the first slide of the currently active view.
         */
        this.getIdOfFirstSlideOfActiveView = function () {
            return self.getIdOfSlideOfActiveViewAtIndex(0);
        };

        /**
         * Getting the id of the last slide in the currently active view.
         *
         * @returns {String}
         *  The id of the last slide of the currently active view.
         */
        this.getIdOfLastSlideOfActiveView = function () {
            return isMasterView ? _.last(masterSlideOrder) : _.last(slideOrder);
        };

        /**
         * Getting the id of the specified slide in the currently active view. The
         * specification is done by the index of the slide (which is its logical
         * position).
         *
         * @param {Number} index
         *  The zero-based index of the slide inside the currently active view.
         *
         * @returns {String}
         *  The id of the first slide of the currently active view.
         */
        this.getIdOfSlideOfActiveViewAtIndex = function (index) {
            return isMasterView ? masterSlideOrder[index] : slideOrder[index];
        };

        /**
         * Getting the key, that is used to mark standard slides.
         *
         * @returns {String}
         *  The key, that is used to mark standard slides.
         */
        this.getStandardSlideType = function () {
            return PresentationModel.TYPE_SLIDE;
        };

        /**
         * Getting the key, that is used to mark master slides.
         *
         * @returns {String}
         *  The key, that is used to mark master slides.
         */
        this.getMasterSlideType = function () {
            return PresentationModel.TYPE_MASTER_SLIDE;
        };

        /**
         * Getting the key, that is used to mark layout slides.
         *
         * @returns {String}
         *  The key, that is used to mark layout slides.
         */
        this.getLayoutSlideType = function () {
            return PresentationModel.TYPE_LAYOUT_SLIDE;
        };

        /**
         * Getting an object that contains the default drawing attributes
         * for new inserted drawings.
         *
         * @returns {Object}
         *  An object containing the default drawing attributes for new inserted
         *  drawings.
         */
        this.getInsertDrawingAttibutes = function () {
            return PresentationModel.INSERT_DRAWING_ATTRIBUTES;
        };

        /**
         * Getting the registered slide styles.
         *
         * @returns {Object}
         *  The container with the registered slide styles.
         */
        this.getSlideStyles = function () {
            return slideStyles;
        };

        /**
         * Getting the id of the currently activated slide.
         *
         * @returns {String}
         *  The id of the currently active slide.
         */
        this.getActiveSlideId = function () {
            return activeSlideId;
        };

        /**
         * Getting the logical position of the active slide
         *
         * @returns {Array<Number>}
         *  The logical position of the active slide in its content container.
         */
        this.getActiveSlidePosition = function () {
            return isMasterView ? [0] : [getActiveSlideIndex()];
        };

        /**
         * Check, whether the master and layout slides are visible.
         *
         * @returns {Boolean}
         *  Whether the master and layout slides are visible.
         */
        this.isMasterView = function () {
            return isMasterView;
        };

        /**
         * Receiving the type of the slide for a specified id.
         *
         * @param {String} id
         *  The slide id.
         *
         * @returns {String|Null}
         *  The type of the slide. Or null, if no type could be determined.
         */
        this.getSlideType = function (id) {
            return idTypeConnection[id] || null;
        };

        /**
         * Check, whether the specified id string is a valid id of a standard slide.
         *
         * @param {String} id
         *  The id string.
         *
         * @returns {Boolean}
         *  Whether the specified id string is a valid id for a standard slide.
         */
        this.isStandardSlideId = function (id) {
            return allStandardSlides[id] !== undefined;
        };

        /**
         * Check, whether the specified id string is a valid id of a layout slide.
         *
         * @param {String} id
         *  The id string.
         *
         * @returns {Boolean}
         *  Whether the specified id string is a valid id for a layout slide.
         */
        this.isLayoutSlideId = function (id) {
            return allLayoutSlides[id] !== undefined;
        };

        /**
         * Check, whether the specified id string is a valid id for a master slide.
         *
         * @param {String} id
         *  The id string.
         *
         * @returns {Boolean}
         *  Whether the specified id string is a valid id for a master slide.
         */
        this.isMasterSlideId = function (id) {
            return allMasterSlides[id] !== undefined;
        };

        /**
         * Check, whether the specified id string is a valid id of a layout or master slide.
         *
         * @param {String} id
         *  The id string.
         *
         * @returns {Boolean}
         *  Whether the specified id string is a valid id for a layout or master slide node.
         */
        this.isLayoutOrMasterId = function (id) {
            return self.isLayoutSlideId(id) || self.isMasterSlideId(id);
        };

        /**
         * Provides the layout slide node or master slide node in the layout slide layer or master slide
         * layer corresponding to the specified target string.
         *
         * @param {String} id
         *  The slide id.
         *
         * @returns {jQuery|null}
         *  The slide node inside the layout or master slide layer node. Or null, if no node with the
         *  specified target exists.
         */
        this.getLayoutOrMasterSlide = function (id) {
            return getLayoutSlide(id) || getMasterSlide(id) || null;
        };

        /**
         * Provides a jQuerified slide node specified by its id. Or null, if no slide is registered with
         * this id.
         *
         * @param {String} id
         *  The slide id.
         *
         * @returns {jQuery|null}
         *  The slide node specified by the id. Or null, if no node with the specified id exists.
         */
        this.getSlideById = function (id) {
            return getStandardSlide(id) || getLayoutSlide(id) || getMasterSlide(id) || null;
        };

        /**
         * Receiving all slides of the currently active view. This can be the standard view or the
         * master/layout view.
         *
         * @returns {Object|null}
         *  An object containing the IDs as key and the jQuerified slides as value for all slides
         *  of the currently active view.
         */
        this.getAllSlidesOfActiveView = function () {
            return isMasterView ? Utils.extendOptions(allLayoutSlides, allMasterSlides) : allStandardSlides;
        };

        /**
         * Receiving all slides from all views.
         *
         * @returns {Object|null}
         *  An object containing the IDs as key and the jQuerified slides as value for all slides.
         */
        this.getAllSlides = function () {
            return Utils.extendOptions(allStandardSlides, allLayoutSlides, allMasterSlides);
        };

        /**
         * Activating a slide with a specified id.
         *
         * @param {String} id
         *  The id of the slide, that shall be activated.
         */
        this.changeToSlide = function (id) {
            if (id !== activeSlideId) { setActiveSlideId(id); }
        };

        /**
         * Check, if the active slide is the first slide inside the standard or the master/layout view.
         *
         * @returns {Boolean}
         *  Whether the active slide is the first slide inside the standard or the master/layout view.
         */
        this.isFirstSlideActive = function () {
            return (isMasterView ? _.first(masterSlideOrder) : _.first(slideOrder)) === activeSlideId;
        };

        /**
         * Check, if the active slide is the last slide inside the standard or the master/layout view.
         *
         * @returns {Boolean}
         *  Whether the active slide is the last slide inside the standard or the master/layout view.
         */
        this.isLastSlideActive = function () {
            return (isMasterView ? _.last(masterSlideOrder) : _.last(slideOrder)) === activeSlideId;
        };

        /**
         * Check, whether a slide specified by its id, is the only slide in its view. This can be the view
         * for the standard slides or the view for the master/layout slides.
         *
         * @param {String} [id]
         *  The id of the slide. If not specified, the active slide id is used.
         *
         * @returns {Boolean}
         *  Whether the specified slide is the only slide in its view.
         */
        this.isOnlySlideInView = function (id) {

            var // whether the specified slide is the only slide in the view (standard view or master/layout view)
                isOnlySlide = false,
                // the id of the slide
                localId = id || activeSlideId;

            if (self.isStandardSlideId(localId)) {
                isOnlySlide = (slideOrder.length === 1);
            } else if (self.isLayoutOrMasterId(localId)) {
                isOnlySlide = (masterSlideOrder.length === 1);
            }

            return isOnlySlide;
        };

        /**
         * Check, whether a layout slide specified by its id is used as target by another slide. If this
         * is the case, the slide cannot be removed.
         *
         * @param {String} id
         *  The id of the slide.
         *
         * @returns {Boolean}
         *  Whether the specified slide is used as target by another slide.
         */
        this.isUsedLayoutSlide = function (id) {
            return _.contains(_.values(layoutConnection), id);
        };

        /**
         * Check, whether a master slide specified by its id is used as target by another slide. If this
         * is the case, the slide cannot be removed.
         *
         * @param {String} id
         *  The id of the slide.
         *
         * @returns {Boolean}
         *  Whether the specified slide is used as target by another slide.
         */
        this.isUsedMasterSlide = function (id) {
            return _.contains(_.values(layoutMasterConnection), id);
        };

        /**
         * Check, whether a layout or master slide specified by its id is used as target by another slide.
         * If this is the case, the slide cannot be removed.
         *
         * @param {String} id
         *  The id of the slide.
         *
         * @returns {Boolean}
         *  Whether the specified slide is used as target by another slide.
         */
        this.isUsedMasterOrLayoutSlide = function (id) {
            return self.isUsedLayoutSlide(id) || self.isUsedMasterSlide(id);
        };

        /**
         * Receiving the layout slide id for a slide specified by its id.
         *
         * @param {String} id
         *  The id of the slide, whose layout slide id shall be determined.
         *
         * @returns {String|Null}
         *  The id of the layout slide that belongs to the specified slide. If no layout slide exists,
         *  null is returned.
         */
        this.getLayoutSlideId = function (id) {
            return layoutConnection[id] || null;
        };

        /**
         * Receiving the master slide id for a layout slide specified by its id.
         *
         * @param {String} id
         *  The id of the layout slide, whose master slide id shall be determined.
         *
         * @returns {String|Null}
         *  The id of the master slide that belongs to the specified layout slide. If no master slide
         *  exists, null is returned.
         */
        this.getMasterSlideId = function (id) {
            return layoutMasterConnection[id] || null;
        };

        /**
         * Receiving the layout slide id or master slide id for a slide specified by its id.
         *
         * @param {String} id
         *  The id of the slide, whose layout slide id or master slide id shall be determined.
         *
         * @returns {String|Null}
         *  The id of the layout or master slide that belongs to the specified slide. If no
         *  layout or master slide exists, null is returned.
         */
        this.getLayoutOrMasterSlideId = function (id) {
            return self.getLayoutSlideId(id) || self.getMasterSlideId(id);
        };

        /**
         * Receiving the number of all normal (standard) slides.
         *
         * @returns {Number}
         *  The number of all normal (standard) slides
         */
        this.getStandardSlideCount = function () {
            return slideOrder.length;
        };

        /**
         * Receiving the sorted array of normal slide IDs.
         *
         * @returns {String[]}
         *  The sorted array of normal slide IDs.
         */
        this.getStandardSlideOrder = function () {
            return slideOrder;
        };

        /**
         * Receiving the sorted array of master and layout slide IDs.
         *
         * @returns {String[]}
         *  The sorted array of master and layout slide IDs.
         */
        this.getMasterSlideOrder = function () {
            return masterSlideOrder;
        };

        /**
         * Receiving the number of master and layout slides.
         *
         * @returns {Number}
         *  The number of all master and layout slides
         */
        this.getMasterSlideCount = function () {
            return masterSlideOrder.length;
        };

        /**
         * Receiving a list of all slide IDs of those slides, that have as target the specified target.
         * The target can be the id of the corresponding layout slide or master slide.
         *
         * @param {String} target
         *  The id of the target slide. The target slide is a layout slide or a master slide.
         *
         * @returns {String[]}
         *  An array containing the IDs of those slides, that have the specified target.
         */
        this.getAllSlideIDsWithSpecificTargetParent = function (target) {

            var // a collector for all affected slide IDs
                allSlideIDs = [];

            if (self.isLayoutSlideId(target)) {
                _.each(_.keys(layoutConnection), function (key) {
                    if (layoutConnection[key] === target) { allSlideIDs.push(key); }
                });
            } else if (self.isMasterSlideId(target)) {
                _.each(_.keys(layoutMasterConnection), function (key) {
                    if (layoutMasterConnection[key] === target) { allSlideIDs.push(key); }
                });
            }

            return allSlideIDs;
        };

        /**
         * Receiving a list of all slide IDs of those slides, that have as target the specified target
         * or whose target slide has the specified target.
         *
         * The target can be the id of the corresponding layout slide or master slide.
         *
         * @param {String} target
         *  The id of the target slide. The target slide is a layout slide or a master slide.
         *
         * @returns {String[]}
         *  An array containing the IDs of those slides, that have the specified target or whose
         *  target slide have the specified target.
         */
        this.getAllSlideIDsWithSpecificTargetAncestor = function (target) {

            var // a collector for all affected slide IDs
                allSlideIDs = self.getAllSlideIDsWithSpecificTargetParent(target);

            // additionally search all 'standard' slides affected by the changes of the layout slides
            if (self.isMasterSlideId(target)) {
                _.each(_.clone(allSlideIDs), function (layoutID) {
                    allSlideIDs = allSlideIDs.concat(self.getAllSlideIDsWithSpecificTargetParent(layoutID));
                });
            }

            return allSlideIDs;
        };

        /**
         * Switching the view for displaying the 'normal' slides or the master and
         * layout slides.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.showMaster=false]
         *      Whether the normal slides shall be displayed or the master and layout slides.
         *      Default are the normal slides.
         */
        this.selectSlideView = function (options) {

            var // whether the master/layout view will be activated or not
                showMaster = Utils.getBooleanOption(options, 'showMaster', false);

            // setting value in the model
            setActiveView(showMaster);
        };

        /**
         * Check, if a specified slide or the currently active slide displays content of layout or master slide.
         *
         * @param {String} [id]
         *  The id of a slide. If not specified, the id of the currently active slide is used.
         *
         * @returns {Boolean}
         *  Whether the specified or the active slide has the attribute 'followmastershapes' set or not.
         */
        this.isFollowMasterShapeSlide = function (id) {
            return (id || activeSlideId) ? (self.getSlideById(id ? id : activeSlideId).attr('followmastershapes') !== 'false') : true;
        };

        /**
         * Generating an operation for the slide attribute 'followMasterShapes'.
         *
         * @param {Boolean} value
         *  The value for the attribute 'followMasterShapes'.
         */
        this.setFollowMasterShape = function (value) {

            var // the operations generator
                generator = self.createOperationsGenerator(),
                // the options for the setAttributes operation
                operationOptions = { start: self.getActiveSlidePosition(), attrs: { slide: { followMasterShapes: value } } };

            // generate the 'setAttributes' operation
            generator.generateOperation(Operations.SET_ATTRIBUTES, operationOptions);

            // apply all collected operations
            this.applyOperations(generator);
        };

        /**
         * Check, whether the active slide or a slide specified by an ID has a background set.
         *
         * @param {String} [id]
         *  The id of a slide. If not specified, the id of the currently active slide is used.
         *
         * @returns {Boolean}
         *  Whether the specified or the active slide has a defined background.
         */
        this.isSlideWithBackground = function (id) {

            var // the slide node
                slide = self.getSlideById(id || activeSlideId),
                // the slide attributes
                slideAttrs = slide ? AttributeUtils.getExplicitAttributes(slide)  : null;

            return slideAttrs && slideAttrs.fill && slideAttrs.fill.type && slideAttrs.fill.type !== 'none';
        };

        /**
         * Deleting the slide background.
         */
        this.resetBackground = function () {

            var // the operations generator
                generator = self.createOperationsGenerator(),
                // the options for the setAttributes operation
                operationOptions = { start: self.getActiveSlidePosition(), attrs: slideStyles.getEmptySlideBackgroundAttributes() };

            // generate the 'setAttributes' operation
            generator.generateOperation(Operations.SET_ATTRIBUTES, operationOptions);

            // apply all collected operations
            this.applyOperations(generator);
        };

        /**
         * Setting the background color of the active slide.
         *
         * @param {Object} color
         *  The color object.
         */
        this.setBackgroundColor = function (color) {

            var // the operations generator
                generator = self.createOperationsGenerator(),
                // the options for the setAttributes operation
                operationOptions = { start: self.getActiveSlidePosition(), attrs: { fill: { type: 'solid', color: color, imageUrl: null } } };

            // generate the 'setAttributes' operation
            generator.generateOperation(Operations.SET_ATTRIBUTES, operationOptions);

            // apply all collected operations
            this.applyOperations(generator);
        };

        /**
         * Shows an insert image dialog from drive, local or URL dialog.
         *
         * @param {String} dialogType
         *  The dialog type. Types: 'drive', 'local' or 'url'. Default is 'drive'.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved or rejected after the dialog has
         *  been closed and the image has been inserted into the document.
         */
        this.showInsertImageDialogForSlideBackground = function (dialogType) {
            return Image.showInsertImageDialogByType(app, dialogType).then(function (imageFragment) {
                insertImageForSlideBackground(imageFragment);
            });
        };

        /**
         * Receiving the (explicit) slide attributes of a slide specified by its ID
         * or of the active slide.
         * This is a simplified version of model.getAttributes(), that handles only
         * explicit attributes in the moment. It might be necessary to improve this
         * handling in the future.
         *
         * @param {String} [id]
         *  An optional slide id. If not specified, the active slide is used.
         */
        this.getSlideAttributes = function (id) {

            var // the slide specified by the id
                slide = self.getSlideById(id || activeSlideId);

            return slide ? AttributeUtils.getExplicitAttributes(slide) : null;
        };

        /**
         * Getting the vertical text alignment of a currently selected text frame node.
         *
         * @returns {String}
         *  The vertical text alignment mode of the text inside a selected text frame node. If it cannot
         *  be determined, the default value 'top' is returned.
         */
        this.getVerticalAlignmentMode = function () {

            var // the selected text frame node (also finding text frames inside groups)
                drawingFrame = selection.getAnyTextFrameDrawing({ forceTextFrame: true }),
                // the text frame node inside the shape
                textFrame = DrawingFrame.getTextFrameNode(drawingFrame);

            return (textFrame && textFrame.length > 0 && textFrame.attr(DrawingFrame.VERTICAL_ALIGNMENT_ATTRIBUTE)) || 'top';
        };

        /**
         * Setting the vertical text alignment mode inside a text frame.
         *
         * @param {String} mode
         *  The vertical text alignment mode of the text inside a selected text frame node.
         */
        this.setVerticalAlignmentMode = function (state) {

            var // the operations generator
                generator = self.createOperationsGenerator(),
                // the options for the setAttributes operation
                operationOptions = {},
                // a selected text frame node or the text frame containing the selection
                textFrame = selection.getAnyTextFrameDrawing({ forceTextFrame: true });

            // collecting the attributes for the operation
            operationOptions.attrs =  {};
            operationOptions.attrs.shape = { anchor: state };

            if (selection.isAdditionalTextframeSelection()) {
                operationOptions.start = Position.getOxoPosition(self.getCurrentRootNode(), textFrame, 0);
            } else {
                operationOptions.start = selection.getStartPosition();
            }

            // generate the 'setAttributes' operation
            generator.generateOperation(Operations.SET_ATTRIBUTES, operationOptions);

            // apply all collected operations
            self.applyOperations(generator);
        };

        /**
         * Getter method for root container node with passed target, or if target is not passed, default page node.
         *
         * @param {String} [target]
         *  ID of root container node
         *
         * @returns {jQuery}
         *  The root node specified by the target string, or the page node, if no target was specified.
         */
        this.getRootNode = function (target) {
            return (target && self.isLayoutOrMasterId(target)) ? getLayoutOrMasterSlideContentNode(target) : pagediv;
        };

        /**
         * Getter method for currently active root container node.
         *
         * @param {String} [target]
         *  ID of root container node
         *
         * @returns {jQuery}
         *  The root node specified by the target string
         */
        this.getCurrentRootNode = function (target) {

            // target for operation - if exists
            var currTarget = target || self.getActiveTarget();

            // checking, if this is a target from a master or layout slide
            if (currTarget && self.isLayoutOrMasterId(currTarget)) { return getLayoutOrMasterSlideContentNode(currTarget); }

            // checking, if the target is a target for a comment. Then this is handled by the comment layer.
            if (currTarget && self.getCommentLayer().getCommentRootNode(currTarget)) { return self.getCommentLayer().getCommentRootNode(currTarget); }

            // returning the page node, if no other root node was found before
            return pagediv;
        };

        /**
         * Changing the active slide, if possible
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.next=true]
         *   Whether the next or the previous slide shall be selected.
         */
        this.changeToNeighbourSlide = function (options) {

            var // whether the next slide shall be made visible
                next = Utils.getBooleanOption(options, 'next', true),
                // the container that contains the order of the slides
                orderContainer = isMasterView ? masterSlideOrder : slideOrder,
                // the index of the currently active slide inside the ordered container
                index = _.indexOf(orderContainer, activeSlideId),
                // the index of the slide that shall be activated
                newIndex = -1;

            if (next) {
                if (index < (isMasterView ? masterSlideOrder.length : slideOrder.length)) { newIndex = index + 1; }
            } else {
                if (index > 0) { newIndex = index - 1; }
            }

            if (newIndex > -1) { self.changeToSlide(self.getIdOfSlideOfActiveViewAtIndex(newIndex)); }
        };

        /**
         * Modifying the selection in that way, that only complete paragraphs are selected. The start
         * position has to be set to the beginning of its current paragraph and the end position has
         * to be set to the end of the current paragraph.
         *
         * This function is currently used for generating setAttribute operations, if the selection
         * is inside a layout or master slide. In this case
         *
         */
        this.setMasterLayoutSelection = function () {

            var // the logical start position
                startPos = _.clone(selection.getStartPosition()),
                // the logical end position
                endPos = _.clone(selection.getEndPosition());

            // checking if the selection is inside a place holder drawing
            if (selection.isAdditionalTextframeSelection() && PresentationUtils.isPlaceHolderDrawing(selection.getSelectedTextFrameDrawing())) {

                // saving the current selection
                masterLayoutSelection = [_.clone(startPos), _.clone(endPos)];

                // expanding the selection
                startPos[startPos.length - 1] = 0;
                endPos[endPos.length - 1] = Position.getParagraphLength(self.getCurrentRootNode(), _.initial(endPos));
                selection.setTextSelection(startPos, endPos);
            }

        };

        /**
         * After a setAttributes operation on a master or layout slide, it is necessary to update
         * the styles defined at the drawing/slide.
         * After this 'styling' update, all slides that used this style need to be updated.
         *
         * @param {HTMLElement|jQuery} element
         *  The element node, that received the new attributes.
         *
         * @param {String} target
         *  The active target for the specified element.
         *
         * @param {String} family
         *  The family of the attribute set.
         *
         * @param {Object} attributes
         *  A map with formatting attribute values, mapped by the attribute
         *  names, and by attribute family names.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {String} [options.immediateUpdate=true]
         *      Whether slides that are dependent from the layout or master slide change
         *      shall be updated immediately. Default is the immediate update.
         *  @param {String} [options.saveOldAttrs=false]
         *      Whether the old attributes shall be saved and returned in the return object.
         *
         * @returns {Object|Null}
         *  If the attributes are registered, an object with the complete registration
         *  information is returned. The object's keys are: 'target', 'placeHolderType',
         *  'placeHolderIndex', 'attrs' and 'oldAttrs'. If the attributes were not registered,
         *  because target, type or index are not defined, null is returned.
         */
        this.updateMasterLayoutStyles = function (element, target, family, attributes, options) {

            // restoring a previous selection (if user has edit privileges)
            if (masterLayoutSelection) {
                if (target && self.isLayoutOrMasterId(target) && self.getEditMode()) { selection.setTextSelection.apply(selection, masterLayoutSelection); }
                masterLayoutSelection = null;
            }

            return registerPlaceHolderAttributes(element, target, family, attributes, options);
        };

        /**
         * Merging the place holder attributes into a specified attribute object.
         *
         * @param {HTMLElement|jQuery} element
         *  The element node, that received the new attributes.
         *
         * @param {String} family
         *  The family of the attribute set.
         *
         * @param {Object} attrs
         *  A map with formatting attribute values, mapped by the attribute
         *  names, and by attribute family names.
         *
         * @param {Object} placeHolderAttrs
         *  A map with formatting attribute values, mapped by the attribute
         *  names, and by attribute family names that is saved in the place holder model.
         *
         * @returns {Object}
         *  An object with the merged attributes of the specified attributes
         *  and the specified place holder attributes.
         */
        this.mergePlaceHolderAttributes = function (element, family, attrs, placeHolderAttrs) {

            var // a paragraph node
                paragraph = null,
                // the paragraph level
                paraLevel = 0,
                // the key in the list styles corresponding to the paragraph level
                paraKey = null,
                // whether drawing attributes shall be registered
                isDrawing = (family === 'drawing'),
                // the merged attributes object
                mergedAttrs = attrs;

            if (isDrawing) {
                if (placeHolderAttrs[family]) {
                    mergedAttrs = self.extendAttributes(placeHolderAttrs, attrs);
                }
            } else {
                if (placeHolderAttrs.listStyle) {
                    paragraph = (family === 'paragraph') ? $(element) : $(element).closest(DOM.PARAGRAPH_NODE_SELECTOR);
                    paraLevel = paragraphStyles.getParagraphLevel(paragraph);
                    paraKey = paragraphStyles.generateKeyFromLevel(paraLevel);

                    if (placeHolderAttrs.listStyle[paraKey] && placeHolderAttrs.listStyle[paraKey][family]) {
                        mergedAttrs = self.extendAttributes(placeHolderAttrs.listStyle[paraKey], attrs);
                    }
                }
            }

            return mergedAttrs;
        };

        /**
         * Public function to force an update of slides after a change of a master or layout
         * slide.
         *
         * @param {Object} data
         *  This object is generated inside the function 'savePlaceHolderAttributes' in the
         *  drawing styles. The object's keys are: 'target', 'placeHolderType', 'placeHolderIndex'
         *  and 'attrs'.
         */
        this.forceMasterLayoutUpdate = function (data) {
            triggerLayoutChangeEvent(data);
        };

        /**
         * Returning the ratio of width divided by height of the page.
         *
         * @returns {Number}
         *  The ratio of width divided by height of the page. If it cannot be determined, 1 is returned.
         */
        this.getSlideRatio = function () {

            var // the attributes of the page
                pageAttributes = pageStyles.getElementAttributes(self.getNode()).page;

            return pageAttributes && pageAttributes.width && pageAttributes.height ? Utils.round(pageAttributes.width / pageAttributes.height, 0.01) : 1;
        };

        /**
         * Check, whether at a specified position the list auto detection shall be executed. In
         * the text application there are not limitations yet.
         *
         * @returns {Boolean}
         *  Whether the list auto detection can be executed in the specified paragraph.
         */
        this.isForcedHardBreakPosition = function (paragraph) {
            return PresentationUtils.isTitlePlaceHolderDrawing($(paragraph).closest('div.drawing'));
        };

        // registering the handler functions for the operations ---------------

        this.registerOperationHandler(Operations.MASTER_SLIDE_INSERT, self.insertMasterSlideHandler);
        this.registerOperationHandler(Operations.LAYOUT_SLIDE_INSERT, self.insertLayoutSlideHandler);
        this.registerOperationHandler(Operations.SLIDE_INSERT, self.insertSlideHandler);
        this.registerOperationHandler(Operations.DRAWING_INSERT, self.insertDrawingHandler);
        this.registerOperationHandler(Operations.TABLE_INSERT, self.insertTableHandler);

        // initialization -----------------------------------------------------

        pagediv = self.getNode();
        selection = self.getSelection();

        // setting style collectors
        characterStyles = self.getStyleCollection('character');
        paragraphStyles = self.getStyleCollection('paragraph');
        tableStyles = self.getStyleCollection('table');
        tableRowStyles = self.getStyleCollection('row');
        tableCellStyles = self.getStyleCollection('cell');
        drawingStyles = self.getStyleCollection('drawing');
        slideStyles = self.getStyleCollection('slide');
        pageStyles = self.getStyleCollection('page');

        // the editor.js has an own drawing style object with own format handler -> that need to be overwritten
        self.setCharacterStyles(characterStyles);
        self.setParagraphStyles(paragraphStyles);
        self.setSlideStyles(slideStyles);
        self.setDrawingStyles(drawingStyles);
        self.setTableStyles(tableStyles);
        self.setTableRowStyles(tableRowStyles);
        self.setTableCellStyles(tableCellStyles);
        self.setPageStyles(pageStyles);

        // setting the handler function for moving and resizing drawings
        self.getSelection().setDrawingResizeHandler(DrawingResize.drawDrawingSelection);

        // setting the handler function for updating lists (this is used during 'updateDocumentFormatting')
        self.setUpdateLists(self.getUpdateListsHandler());

        self.waitForImportSuccess(importSuccessHandler);

//        // register the listener for 'textframeHeight:update'.
//        self.listenTo(self, 'textframeHeight:update', updateVerticalDrawingPosition);
//
//        // register the listener for 'addtionalTextFrame:selected' of the selection
//        self.listenTo(selection, 'addtionalTextFrame:selected', saveVerticalPositionAttributes);

        // registering the default list styles
        self.listenTo(self, 'change:defaults', registerDefaultTextListStyles);

        // registering process mouse down handler at the page
        pagediv.on({ 'mousedown touchstart': self.getPageProcessMouseDownHandler() });
        if (self.getListenerList()) { self.getListenerList()['mousedown touchstart'] = self.getPageProcessMouseDownHandler(); }

    } // class PresentationModel

    // constants --------------------------------------------------------------

    /**
     * The attributes that are used for new inserted drawings.
     */
    PresentationModel.INSERT_DRAWING_ATTRIBUTES = {
        left: 5000,
        top: 3000
    };

    /**
     * The type used to recognize master slides.
     */
    PresentationModel.TYPE_MASTER_SLIDE = 'masterSlide';

    /**
     * The type used to recognize layout slides.
     */
    PresentationModel.TYPE_LAYOUT_SLIDE = 'layoutSlide';

    /**
     * The type used to recognize 'standard' slides.
     */
    PresentationModel.TYPE_SLIDE = 'standardSlide';

    // exports ================================================================

    // derive this class from class EditModel
    return Editor.extend({ constructor: PresentationModel });

});
