/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define('io.ox/office/presentation/format/drawingstyles', [
    'io.ox/office/presentation/utils/presentationutils',
    'io.ox/office/tk/utils',
    'io.ox/office/editframework/utils/attributeutils',
    'io.ox/office/editframework/utils/border',
    'io.ox/office/drawinglayer/model/drawingstylecollection',
    'io.ox/office/drawinglayer/view/drawingframe',
    'io.ox/office/textframework/utils/dom'
], function (PresentationUtils, Utils, AttributeUtils, Border, DrawingStyleCollection, DrawingFrame, DOM) {

    'use strict';

    var // definitions for presentation-specific drawing attributes
        DEFINITIONS = {

            /**
             * Margin from top border of the drawing to text contents, in 1/100
             * of millimeters.
             */
            marginTop: { def: 0 },

            /**
             * Margin from bottom border of the drawing to text contents, in
             * 1/100 of millimeters.
             */
            marginBottom: { def: 0 },

            /**
             * Margin from left border of the drawing to text contents, in 1/100
             * of millimeters.
             */
            marginLeft: { def: 0 },

            /**
             * Margin from right border of the drawing to text contents, in
             * 1/100 of millimeters.
             */
            marginRight: { def: 0 },

            /**
             * Style, width and color of the left drawing border.
             */
            borderLeft: {
                def: Border.NONE,
                format: function (element, border) {
                    element.css('border-left', this.getCssBorder(border));
                }
            },

            /**
             * Style, width and color of the right drawing border.
             */
            borderRight: {
                def: Border.NONE,
                format: function (element, border) {
                    element.css('border-right', this.getCssBorder(border));
                }
            },

            /**
             * Style, width and color of the top drawing border.
             */
            borderTop: {
                def: Border.NONE,
                format: function (element, border) {
                    element.css('border-top', this.getCssBorder(border));
                }
            },

            /**
             * Style, width and color of the bottom drawing border.
             */
            borderBottom: {
                def: Border.NONE,
                format: function (element, border) {
                    element.css('border-bottom', this.getCssBorder(border));
                }
            },

            // object anchor --------------------------------------------------

            /**
             * The distance of the drawing to the left slide border, as number
             * in 1/100 of millimeters.
             */
            left: { def: 0 },

            /**
             * The distance of the drawing to the top slide border, as number
             * in 1/100 of millimeters.
             */
            top: { def: 0 },

            /**
             * Width of the drawing, as number in 1/100 of millimeters. The
             * value 0 will be interpreted as 100% width of the parent element.
             */
            width: { def: 0 },

            /**
             * Height of the drawing, as number in 1/100 of millimeters.
             */
            height: { def: 0 }

        };

    // class PresentationDrawingStyles ================================================

    /**
     * Contains the style sheets for drawing formatting attributes. The CSS
     * formatting will be read from and written to drawing elements of any type.
     *
     * @constructor
     *
     * @extends DrawingStyleCollection
     *
     * @param {TextModel} docModel
     *  The text document model containing instance.
     */
    function PresentationDrawingStyles(docModel) {

        var // self reference
            self = this,
            // collector for the place holder attributes in layout or master slides
            placeHolderCollector = {};

        // base constructor ---------------------------------------------------

        DrawingStyleCollection.call(this, docModel, {
            families: 'presentation line fill image shape geometry changes',
            formatHandler: updateDrawingFormatting,
            layoutHandler: addLayoutAttributes
        });

        // private methods ----------------------------------------------------

        /**
         * Setting the width of the drawing.
         *
         * @param {jQuery} drawing
         *  The drawing node as jQuery object.
         *
         * @param {Object} attributes
         *  A map with the merged drawing attributes, as name/value pairs.
         */
        function setDrawingWidth(drawing, attributes) {

            if (!DrawingFrame.isGroupedDrawingFrame(drawing)) {
                if (attributes.drawing.width === 0) {
                    drawing.width('100%');
                } else {
                    drawing.width(Utils.convertHmmToLength(attributes.drawing.width, 'px', 1));
                }
            }
        }

        /**
         * Setting the height of the drawing. This needs to be done, because it might be
         * dependent from other attribute values.
         *
         * @param {jQuery} drawing
         *  The drawing node as jQuery object.
         *
         * @param {Object} attributes
         *  A map with the merged drawing attributes, as name/value pairs.
         */
        function setDrawingHeight(drawing, attributes) {

            if (!(DrawingFrame.isGroupedDrawingFrame(drawing) || attributes.shape.autoResizeHeight)) {
                drawing.height(Utils.convertHmmToLength(attributes.drawing.height, 'px', 1));
            } else if (attributes.shape.autoResizeHeight) {
                // -> no setting of height for grouped drawings and for shapes with property autoResizeHeight
                drawing.height(''); // if 'autoResizeHeight' is set, an explicit height must be removed
            }
        }

        /**
         * Helper function to update attributes of children of drawings. This is only used for text frames yet.
         *
         * @param {jQuery|Node} drawing
         *  The drawing node as jQuery object, whose children need an update of element formatting.
         */
        function handleChildAttributes(drawing) {

            var // the explicit drawing attributes
                explicitDrawingAttrs = AttributeUtils.getExplicitAttributes(drawing),
                // the paragraph style sheet container
                paragraphStyles = null,
                // the container for the paragraphs and tables inside the drawing
                textFrameNode = null;

            // checking the fill color (task 36385)
            if (explicitDrawingAttrs && explicitDrawingAttrs.fill) {

                paragraphStyles = docModel.getStyleCollection('paragraph');
                textFrameNode = DrawingFrame.getTextFrameNode(drawing);

                Utils.iterateSelectedDescendantNodes(textFrameNode, DOM.PARAGRAPH_NODE_SELECTOR, function (paragraph) {
                    paragraphStyles.updateElementFormatting(paragraph);
                }, undefined, { children: true });
            }
        }

        /**
         * Handling drawing that are absolute positioned. This are all drawings that are not in-line
         * with text. This is especially important, if the drawing anchor was modified. In this case
         * some clean up at the node is required.
         *
         * @param {jQuery} drawing
         *  The drawing node as jQuery object.
         *
         * @param {Object} mergedAttributes
         *  A map with the merged drawing attributes, as name/value pairs.
         *
         * @param {Number} borderWidth
         *  The width of the drawing of the border in 1/100 mm.
         *
         * @param {Number} halfBorderWidth
         *  The half width of the drawings border in 1/100 mm.
         *
         * @param {Number} zoomFactor
         *  The current zoom factor in percent.
         */
        function assignDrawingAttributes(drawing, mergedAttributes) {

            var // the drawing attributes of the passed attribute map
                drawingAttributes = mergedAttributes.drawing,
//                // offset from top of slide element, in 1/100 mm
//                topOffset = drawingAttributes.top,
//                // margins to be applied at the drawing
//                topMargin = 0, bottomMargin = 0, leftMargin = 0, rightMargin = 0,
                // the type of place holder for presentation drawings
                placeHolderType = mergedAttributes.presentation.phType,
                // the index of place holder for presentation drawings
                placeHolderIndex = mergedAttributes.presentation.phIndex;

            // calculate drawing margins
//            topMargin = Utils.minMax(drawingAttributes.marginTop, 0, topOffset > 0 ? topOffset : 0);
//            bottomMargin = Math.max(drawingAttributes.marginBottom, 0);
//            leftMargin = Math.max(drawingAttributes.marginLeft, 0);
//            rightMargin = Math.max(drawingAttributes.marginRight, 0);

            // drawing is absolutely positioned
            drawing.addClass('absolute');

            // marking presentation placeholder drawings
            drawing.attr('placeholdertype', placeHolderType ? placeHolderType : null); // toggle attribute
            drawing.attr('placeholderindex', _.isNumber(placeHolderIndex) ? placeHolderIndex : null); // toggle attribute

            // apply CSS formatting to drawing node
            // -> adding the half border width to the margin (border is drawn with a canvas)
            drawing.css({
                left: Utils.convertHmmToLength((drawingAttributes.left - drawingAttributes.marginLeft), 'px', 1),
                top: Utils.convertHmmToLength((drawingAttributes.top - drawingAttributes.marginTop), 'px', 1)
            });
        }

        /**
         * Adding attributes of master and layout slide into the merge attributes. This has to happen
         * before the explicit attributes are set into the specified attribute set.
         *
         * @param {jQuery} element
         *  The node whose attributes have been changed, as jQuery object.
         *
         * @param {Object} mergedAttributes
         *  A map of attribute maps (name/value pairs), keyed by attribute family, containing the
         *  effective attribute values merged from style sheets, but without explicit attributes.
         */
        function addLayoutAttributes(element, attributes) {

            var // the attribute set of the place holder elements in layout and/or master slide
                placeHolderAttributes = null,
                // the explicit attributes at the specified element
                elemAttrs = AttributeUtils.getExplicitAttributes(element),
                // the IDs of document slide, layout slide and master slide
                id = null, layoutId = null, masterId = null,
                // an object containing the place holder type and index as property
                keySet = null;

            if (PresentationUtils.isPlaceHolderAttributeSet(elemAttrs)) {

                id = DOM.getTargetContainerId(element.parent()); // checking only direct children of slide
                keySet = self.getValidTypeIndexSet(elemAttrs);

                if (keySet) {
                    if (docModel.isStandardSlideId(id)) {
                        layoutId = docModel.getLayoutSlideId(id);
                        masterId = docModel.getMasterSlideId(layoutId);
                        placeHolderAttributes = getMergedPlaceHolderAttributes(layoutId, masterId, keySet.type, keySet.index);

                    } else if (docModel.isLayoutSlideId(id)) {
                        masterId = docModel.getMasterSlideId(id);
                        placeHolderAttributes = getMergedPlaceHolderAttributes(null, masterId, keySet.type, keySet.index);
                    }
                }
            }

            // merging the new place holder attributes into the specified attribute set
            if (placeHolderAttributes) { docModel.extendAttributes(attributes || {}, placeHolderAttributes); }
        }

        /**
         * Will be called for every drawing node whose attributes have been changed. Repositions
         * and reformats the drawing according to the passed attributes.
         *
         * @param {jQuery} drawing
         *  The drawing node whose attributes have been changed, as jQuery object.
         *
         * @param {Object} mergedAttributes
         *  A map of attribute maps (name/value pairs), keyed by attribute family, containing the
         *  effective attribute values merged from style sheets and explicit attributes.
         */
        function updateDrawingFormatting(drawing, mergedAttributes) {

            // nothing to do, if paragraph is a group content object
            // -> in this case the group is responsible for the correct attributes
            if (DrawingFrame.isGroupContentNode(drawing.parent())) {
                drawing.addClass('absolute grouped');  // absolute positioned drawing inside its group!
                // set generic formatting to the drawing frame
                DrawingFrame.updateFormatting(docModel.getApp(), drawing, mergedAttributes);
                // simply do nothing, let the group make further attribute settings
                return;
            }

            // setting the drawing height
            setDrawingHeight(drawing, mergedAttributes);

            // setting the drawing width
            setDrawingWidth(drawing, mergedAttributes);

            // setting the drawing attributes except width and height (those are already set)
            assignDrawingAttributes(drawing, mergedAttributes);

            // set generic formatting to the drawing frame
            DrawingFrame.updateFormatting(docModel.getApp(), drawing, mergedAttributes);

            // update formatting of all paragraphs inside the text frame content node (36385)
            if (DrawingFrame.isTextFrameShapeDrawingFrame(drawing)) { handleChildAttributes(drawing); }
        }

        /**
         * Saving the specified drawing attributes in the model container.
         *
         * This function can be called directly from insertDrawing operation. In that
         * case the attributes already contain the full set of properties from the
         * families 'drawing' and 'presentation'.
         *
         * @param {String} target
         *  The id of the layout or master slide, that defines the attributes for
         *  the presentation place holder type.
         *
         * @param {String} type
         *  The place holder type.
         *
         * @param {Number} index
         *  The place holder index.
         *
         * @param {Object} attrs
         *  The attributes object, that contains the attributes for the specified
         *  target and place holder type. This are the explicit attributes set at
         *  the drawing.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {String} [options.saveOldAttrs=false]
         *      Whether the old attributes shall be saved and returned in the return object.
         *
         * @returns {Object|Null}
         *  If the attributes are registered, an object with the complete registration
         *  information is returned. The object's keys are: 'target', 'placeHolderType',
         *  'placeHolderIndex', 'attrs' and 'oldAttrs'. If the attributes were not registered,
         *  because target, type or index are not defined, null is returned.
         */
        function savePlaceHolderAttributes(target, type, index, attrs, options) {

            var // whether list style handling is required
                listStyle = attrs && attrs.listStyle,
                // the level specified in the list styles
                level = null,
                // a helper object for merging list style attributes
                drawingAttrs = null,
                // the old place holder attributes
                oldAttrs = null,
                // whether an update of dependent slide shall be triggered immediately
                saveOldAttrs = Utils.getBooleanOption(options, 'saveOldAttrs', false);

            if (!(target && type && _.isNumber(index))) { return null; }

            placeHolderCollector[target] = placeHolderCollector[target] || {};
            placeHolderCollector[target][type] = placeHolderCollector[target][type] || {};
            placeHolderCollector[target][type][index] = placeHolderCollector[target][type][index] || {};

            // optionally saving a deep copy of the old value in the return object for later usage
            if (saveOldAttrs) { oldAttrs = _.copy(placeHolderCollector[target][type][index], true); }

            // merging the attributes, that are not list styles ('autoclear' is required for undo, so that attributes can be removed)
            docModel.extendAttributes(placeHolderCollector[target][type][index], attrs, { autoClear: true });

            // special handling for list style attribute at drawings
            if (listStyle) {
                for (level in listStyle) {
                    drawingAttrs = placeHolderCollector[target][type][index];
                    drawingAttrs.listStyle = drawingAttrs.listStyle || {};
                    drawingAttrs.listStyle[level] = drawingAttrs.listStyle[level] || {};
                    docModel.extendAttributes(drawingAttrs.listStyle[level], listStyle[level], { autoClear: true });
                }
            }

            // returning an object with the complete registration information
            return { target: target, placeHolderType: type, placeHolderIndex: index, attrs: attrs, oldAttrs: oldAttrs };
        }

        /**
         * Getting the merged place holder attributes from master slide and layout slide.
         *
         * @param {String} [layoutId]
         *  The id of the layout slide.
         *
         * @param {String} [masterId]
         *  The id of the master slide.
         *
         * @param {String} placeHoderType
         *  The drawing place holder type.
         *
         * @param {Number} placeHoderIndex
         *  The drawing place holder index.
         *
         * @returns {Object}
         *  An object containing the attributes from the layout slide and the master slide. If attributes are defined
         *  at both slide, the layout slide overwrites the values from the master slide.
         */
        function getMergedPlaceHolderAttributes(layoutId, masterId, placeHolderType, placeHolderIndex) {

            var // the layout slide attributes
                layoutAttrs = layoutId ? self.getPlaceHolderAttributes(layoutId, placeHolderType, placeHolderIndex) : null,
                // the master slide attributes
                masterAttrs = masterId ? self.getPlaceHolderAttributes(masterId, placeHolderType, placeHolderIndex) : null,
                // the attributes of layout and master slide
                attributes = null;

            // merging the collected attributes from master and layout slides and drawings (never overwrite attribute set)
            if (masterAttrs) { attributes = docModel.extendAttributes({}, masterAttrs); }
            if (layoutAttrs) { attributes = attributes ? docModel.extendAttributes(attributes, layoutAttrs) : docModel.extendAttributes({}, layoutAttrs); }

            return attributes;
        }

        // public methods -----------------------------------------------------

        /**
         * Saving the drawing attributes in a container, if the drawing is a place holder
         * element.
         *
         * @param {String} target
         *  The id of the layout or master slide, that defines the attributes for
         *  the presentation place holder type.
         *
         * @param {Object} attrs
         *  The attributes object, that contains the attributes for the specified
         *  target and place holder type. This are the explicit attributes set at
         *  the drawing.
         *
         * @param {jQuery|Node} drawing
         *  The drawing node, that gets the specified attributes assigned.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {String} [options.saveOldAttrs=false]
         *      Whether the old attributes shall be saved and returned in the return object.
         *
         * @returns {Object|Null}
         *  If the attributes are registered, an object with the complete registration
         *  information is returned. The object's keys are: 'target', 'placeHolderType',
         *  'placeHolderIndex', 'attrs' and 'oldAttrs'. If the attributes were not registered,
         *  because target is not set (no master or layout slide) or the attributes
         *  contain no place holder information (no place holder drawing), null is
         *  returned.
         */
        this.savePlaceHolderAttributes = function (target, attrs, drawing, options) {

            var // an object containing the place holder type and index as property
                keySet = null;

            if (!target || !PresentationUtils.isPlaceHolderDrawing(drawing)) { return null; }

            keySet = self.getValidTypeIndexSet(AttributeUtils.getExplicitAttributes(drawing));

            // target must be defined and the attributes must have phType or phIndex specified
            // -> this is a placeholder drawing in master or layout slide -> the attributes are saved in the model.
            return savePlaceHolderAttributes(target, keySet.type, keySet.index, attrs, options);
        };

        /**
         * Receiving the attributes for a drawing with place holder attribute. In this case the attributes from the
         * parent layers can be used. If the drawing is inside a 'standard' slide, the attributes might be defined
         * in the layout or the master slide. If the drawing is inside a layout slide, the missing attributes might
         * be defined in the master slide.
         *
         * @param {String} target
         *  ID of target container node.
         *
         * @param {String} type
         *  The place holder type.
         *
         * @param {Number} index
         *  The place holder index.
         *
         * @returns {Object|Null}
         *  The attributes defined for the specified target and place holder type.
         */
        this.getPlaceHolderAttributes = function (target, type, index) {
            return (placeHolderCollector && placeHolderCollector[target] && placeHolderCollector[target][type] && placeHolderCollector[target][type][index]) ? _.clone(placeHolderCollector[target][type][index]) : null;
        };

        /**
         * Receiving all place holder attributes that are stored for a specific master or layout node.
         *
         * @param {String} target
         *  ID of target container node.
         *
         * @returns {Object|Null}
         *  An object containing all place holder attributes defined for the specified layout/master
         *  slide. Or null, if this slide contains no place holder attributes.
         */
        this.getAllPlaceHolderAttributesForId = function (target) {
            return placeHolderCollector[target] || null;
        };

        /**
         * Receiving a valid set for the properties 'type' and 'index', so that the model can be accessed.
         * Typically the specified attributes parameter is the set of explicit drawing attributes.
         *
         * @param {Object} attrs
         *  The attributes object. This are typically the explicit attributes set at a drawing.
         *
         * @returns {Object|Null}
         *  An object containing the properties 'type' and 'index'. 'type' is a string, that describes
         *  the place holder type, 'index' is a number describing the index specified for the given
         *  place holder type.
         *  If the specified attribute does not contain a 'presentation' property or inside this
         *  'presentation' property neither 'phType' nor 'phIndex' are defined, null is returned.
         */
        this.getValidTypeIndexSet = function (attrs) {

            var // the place holder type
                type = null,
                // the place holder index
                index = null;

            // it is necessary that at least one of phType or phIndex is defined in the attributes
            if (!attrs || !attrs.presentation || !('phType' in attrs.presentation || 'phIndex' in attrs.presentation)) { return null; }

            type = Utils.getStringOption(attrs.presentation, 'phType', PresentationUtils.getDefaultPlaceHolderType());
            index = Utils.getIntegerOption(attrs.presentation, 'phIndex', PresentationUtils.getDefaultPlaceHolderIndex());

            return { type: type, index: index };
        };

        // initialization -----------------------------------------------------

        // register the attribute definitions for the style family
        docModel.registerAttributeDefinitions('drawing', DEFINITIONS);

        // destroy all class members on destruction
        this.registerDestructor(function () {
            docModel = null;
        });

    } // class PresentationDrawingStyles

    // constants --------------------------------------------------------------

    // exports ================================================================

    // derive this class from class DrawingStyleCollection
    return DrawingStyleCollection.extend({ constructor: PresentationDrawingStyles });

});
