/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Edy Haryono <edy.haryono@open-xchange.com>
 */

/* eslint no-new: 0 */

define('io.ox/office/portal/action/templateactions', [
    'io.ox/core/extensions',
    'io.ox/core/extPatterns/links',
    'io.ox/core/extPatterns/actions',
    'io.ox/office/portal/portalutils',
    'io.ox/office/tk/utils',
    'io.ox/office/tk/io',
    'io.ox/office/tk/keycodes',
    'io.ox/office/tk/utils/driveutils',
    'io.ox/office/portal/view/contextmenu',
    'gettext!io.ox/office/portal/main'
], function (Ext, Links, CoreActions, PortalUtils, Utils, IO, KeyCodes, DriveUtils, ContextMenu, gt) {

    'use strict';

    /**
     * @param {JQuery} image node
     *
     * @param {String} src
     *
     */
    var loadImageInQueue = (function () {
        var queue = [];
        var max = 2;

        function err(error) {
            var img = $(this);
            Utils.error(new Date().toLocaleTimeString(), 'error loading preview for "' + img.attr('name') + '"', error);
            img.data('state', 'done');
            img.off('load', loaded);
            img.off('error', err);
            checkQueue();
        }

        function loaded() {
            var img = $(this);
            img.off('load', loaded);
            img.off('error', err);
            img.data('src', null);
            img.data('state', 'done');

            checkQueue();
        }

        function checkQueue() {
            var currentLoading = max;
            var toRemove = [];
            _.each(queue, function inLoop(entry) {
                var state = entry.data('state');
                if (state === 'loading') {
                    currentLoading--;
                } else if (state === 'done') {
                    toRemove.push(entry);
                }
            });
            while (toRemove.length > 0) {
                var index = queue.indexOf(toRemove.shift());
                queue.splice(index, 1);
            }
            if (currentLoading > 0) {
                for (var i = 0; i < queue.length; i++) {
                    var entry = queue[i];
                    if (entry.data('state') === 'inQueue') {
                        currentLoading--;

                        bindListener(entry);

                        if (currentLoading <= 0) {
                            break;
                        }
                    }
                }
            }
        }

        function bindListener(img) {
            img.on('load', loaded);
            img.on('error', err);
            img.attr('src', img.data('src'));
            img.data('state', 'loading');
        }

        return function loadImageInQueueInner(img, src) {
            img.data('src', src);
            img.data('state', 'inQueue');
            queue.push(img);

            checkQueue();
        };
    }());

    function TemplateExtensions() {

        var // extension point id
            templateExtension = 'io.ox/portal/office/templates',
            // template thumbnail size
            thumbnailSize = null;

        /*
         * Function returning the URL to the global templates using
         * the file descriptor for the creation of the URL.
         *
         * @param {Object} data
         *  The file descriptor object.
         *
         * @param {Object} [options]
         *  Additional optional options.
         *  @param {Boolean} [options.isGlobal=true]
         *      If set to true, the url for global templates is created,
         *      otherwise for user templates.
         *
         * @returns {String}
         *  The url string for the template.
         */
        var createTemplateURL = function (data, options) {

            // Thumbnails have a different size for presentation and text/spreadsheet.
            // In presentation the standard is 16:9 (180px : 100px). When the ratio is higher, the thumbnail
            // width is 180px but the hight is smaller. When the ratio is smaller, the thumbnail
            // height is 100px, but the width is rendered smaller. In text/spreadsheet we just a have a width of
            // 120px (legacy behaviour).
            if (data.type === 'presentation') {
                thumbnailSize = '&height=' + 100 + '&width=' + 180;
            } else {
                thumbnailSize = '&width=' + 120;
            }

            var // whether the template files are local or global
                isGlobal = PortalUtils.getBooleanOption(options, 'isGlobal', true),
                // the constructed url
                url = isGlobal ? ox.abs + ox.apiRoot + '/oxodocumentfilter?action=templatepreview&id=' + data.id +
                    '&folder=' + data.folder_id + '&format=preview_image&delivery=view&scaleType=contains' :
                    DriveUtils.getFileUrl(data, 'preview');

            url += thumbnailSize + '&dummy=' + data.last_modified;
            return url;
        };

        /*
         * Helper function to generate the draw-functions for the
         * different extension points for creating new documents from
         * global or user templates.
         * The templates are displayed differently in the presentation and
         * text/spreadsheet portal. In the presentation portal the templates are
         * rendered with their aspect ratio, in text/spreadsheet a predefined
         * container is used.
         *
         * @param {itemClass} String
         *  A class name specific for the application and for global or
         *  user templates.
         *
         * @param {Object} [options]
         *  Additional optional options.
         *  @param {Boolean} [options.isGlobal=true]
         *      If set to true, the url for global templates is created,
         *      otherwise for user templates.
         *
         * @returns {Function}
         *  The draw function for the extension.
         */
        var getDrawFunction = function (itemClass, options) {

            // baton : a template file descriptor
            return function (baton) {

                // This block sets the data to get a different layout
                // in the presentation and text/spreadsheet portal.
                var // appBaseName for the currently used portal
                    appBaseName = baton.data.type,
                    fullFileName = baton.data['com.openexchange.file.sanitizedFilename'] || baton.data.filename,
                    // set classes according to the current app portal to get the right css layout
                    templateItemClass = 'template-item ' + itemClass + ' ' + appBaseName,
                    templatePreviewClass = getTemplatePreviewClass(appBaseName),
                    // the element which should be highlighted, it has to be different in presentation or text/spreadsheet
                    highLightElement = null;

                var // the template container
                    templateItem = $('<a>').addClass(templateItemClass)
                    .attr({
                        'data-toggle': 'tooltip',
                        'data-animation': 'false',
                        'data-delay': '{ "show": 1400 }',
                        title: _.noI18n(fullFileName),
                        tabindex: 0
                    }),
                    templatePreview = $('<div>').addClass(templatePreviewClass),
                    showName = _.noI18n(PortalUtils.removeFileExtension(fullFileName)),
                    templatePreviewIMG = $('<img class = "templateIMG" name = "' + showName + '"</img>'),
                    templateName = $('<p>').addClass('template-name').text(showName);

                loadImageInQueue(templatePreviewIMG, createTemplateURL(baton.data, options));

                // attach the preview img to the template
                templatePreview.append(templatePreviewIMG);

                // set the element which should be highlighted
                highLightElement = (appBaseName === 'presentation') ? templatePreviewIMG : templatePreview;

                // add new preview to the template container
                templateItem.append(templatePreview, templateName);

                // Get file descriptor of the newly created file as result, and open it with an office app.
                templateItem.click(function () {

                    // only invoke the action when no context menu is open
                    // use case: the user can click everywere on the screen to close the contextmenu
                    if (!ContextMenu.isOpen()) {

                        // provide feedback to the user which element will be opened
                        PortalUtils.markElement(highLightElement);

                        CoreActions.invoke(templateExtension + '/action/open', this, baton.data);
                    }
                });

                templateItem.on('keydown', { grid: true }, PortalUtils.keydownHandler);

                // handle context menu
                templateItem.on('contextmenu', function (e) {
                    contextMenuHandler(e, templateItem, baton, highLightElement);
                });

                // prevent focus change on click due to mousedown
                templateItem.on('mousedown', function (e) {
                    e.preventDefault();
                });

                // iOS: check if there is a long tap to invoke a 'contextmenu' event
                if (Utils.IOS) {
                    templateItem.on('touchstart', function (event) {
                        PortalUtils.iosContextEventHandler(event, templateItem);
                    });
                }

                this.append(templateItem);
            };
        };

        /*
         * A Handler function to generate the the context menu for templates in
         * the text/spreadsheet portal.
         *
         * @param {Event} event
         *  The event which triggered the context menu.
         *  The event contains the position were the context menu is opened.
         *
         * @param {JQuery} link
         *  The DOM element the context menu is opened for.
         *
         * @param {Object} baton
         *  The baton for the element the context menu is opened for.
         *
         * @param {jQuery} highLight
         *  The selected element which should be highlighted in the GUI
         *  (note: different for presentation and test/spreadsheet).
         */
        function contextMenuHandler(event, link, baton, highLight) {

            // global flag that a context menu is open
            ContextMenu.setOpen(true);
            // mark the selected element
            PortalUtils.markElement(highLight);

            // create the context menu
            var cmenu = new ContextMenu(event);

            // init the GUI - case: system template (note: it's called global in the baton)
            if (baton.data.source === 'global') {
                cmenu.createSectionNode('heading', { label: gt('System template') }); //heading
                getNewFromTemplateButton(cmenu, this, baton, 'heading');
            }

            // init the GUI - case: global template set by the admin
            if (baton.data.source === 'admin') {
                cmenu.createSectionNode('heading', { label: gt('Global template') }); //heading
                getNewFromTemplateButton(cmenu, this, baton, 'heading');
                getEditTemplateButton(cmenu, this, baton, 'heading');
                getshowTemplateInDriveButton(cmenu, baton, 'heading');
            }

            // init the GUI - case: user template
            if (baton.data.source === 'user') {
                cmenu.createSectionNode('heading', { label: gt('User template') }); //heading
                getNewFromTemplateButton(cmenu, this, baton, 'heading');
                getEditTemplateButton(cmenu, this, baton, 'heading');
                getshowTemplateInDriveButton(cmenu, baton, 'heading');

            }

            // Create the "Edit template" button in the context menu with
            // eventListener and the invoked action on execution.
            function getEditTemplateButton(cmenu, self, baton, section) {
                var editTemplateButton = cmenu.createItemNode('edittemplate', { section: section, content: gt('Edit template') });

                editTemplateButton.on('click', function () {
                    cmenu.hide();  //hide the context menu after a button was clicked
                    CoreActions.invoke(PortalUtils.getAppBaseName(baton.data.filename) + '-edit-template', self, baton.data);
                });

                return editTemplateButton;
            }

            // Create the "New from template" button in the context menu with
            // eventListener and the invoked action on execution.
            function getNewFromTemplateButton(cmenu, self, baton, section) {
                var newFromTemplateButton = cmenu.createItemNode('newfromtemplate', { section: section, content: gt('New from template') });

                newFromTemplateButton.on('click', function () {
                    cmenu.hide();  //hide the context menu after a button was clicked
                    CoreActions.invoke(templateExtension + '/action/open', self, baton.data);
                });
                return newFromTemplateButton;
            }

            // Create the "Show in drive' button in the context menu with
            // eventListener and the invoked action on execution.
            function getshowTemplateInDriveButton(cmenu, baton, section) {
                var showTemplateInDriveButton = cmenu.createItemNode('showindrive', { section: section, content: gt('Show in drive') });

                showTemplateInDriveButton.on('click', function (evt) {
                    evt.preventDefault(); // ?? TODO
                    cmenu.hide();  //hide the context menu after a button was clicked

                    PortalUtils.showInDrive(baton.data, true);
                });
                return showTemplateInDriveButton;
            }

            // show the context menu
            cmenu.show();

            // remove the highlight and destory the context menu
            cmenu.on('popup:hide', function () {
                PortalUtils.removeMark(highLight);

                // After a click event the context menu is closed by the basemenu globalClickHandler().
                // When a context menu is open, clicking on an element should not invoke it's attached action, but just close the context menu.
                // As 'closing' and 'invocation' due to a click event happens at the same time, ContextMenu.setOpen(false) must be called delayed.
                window.setTimeout(function () { ContextMenu.setOpen(false); }, 300);

                // destroy the context menu
                cmenu = null;
            });
        }

        /*
         * Helper function to generate the draw-functions for the
         * different extension points for creating new documents from
         * blank templates.
         * The templates are displayed differently in presentation and text/spreadsheet portal.
         * In presentation the newBlank template is in 16:9 and for text/spreadsheet it's 4:3.
         *
         * @param {fileName} String
         *  The localized name for the empty template visible in the
         *  application.
         *
         * @param {invokeId} String
         *  The unique ID used to invoke the application with a click on the
         *  blank template.
         *
         * @returns {Function}
         *  The draw function for the blank template extension.
         */
        var getBlankDrawFunction = function (fileName, invokeId) {

            return function (baton) {

                // This block sets the data to get a different layout
                // in the presentation and text/spreadsheet portal.
                var // appBaseName for the currently used portal
                    appBaseName = baton.appBaseName,
                    // set classes according to the current app portal to get the right css layout
                    templateClass = 'new-blank ' + getTemplatePreviewClass(appBaseName),
                    templateItemClass = 'template-item template-blank ' + appBaseName;

                var // the template container
                    templateItem = $('<a>').addClass(templateItemClass)
                    .attr({
                        'data-toggle': 'tooltip',
                        'data-animation': 'false',
                        'data-delay': '{ "show": 1400 }',
                        title: _.noI18n(fileName),
                        tabindex: 0
                    }),
                    templatePreview = $('<div>').addClass(templateClass)
                        .append($('<i>').addClass('fa fa-plus')),
                    templateName = $('<p>').addClass('template-name').text(_.noI18n(fileName));

                // add new blank template to the template container
                templateItem.append(templatePreview, templateName);

                templateItem.click(function () {

                    // only invoke the action when no context menu is open
                    // use case: the user can click everywhere on the screen to close the contextmenu
                    if (!ContextMenu.isOpen()) {

                        // provide feedback to the user which element will be opened
                        PortalUtils.markElement(templatePreview);

                        CoreActions.invoke(invokeId, this, baton);
                    }
                });

                templateItem.on('keydown', PortalUtils.keydownHandler);

                // prevent native context menu for the getBlank template and the focus change on click due to mousedown
                templateItem.on('mousedown contextmenu', function (e) {
                    e.preventDefault();
                });

                // iOS: check if there is a long tap to invoke a 'contextmenu' event
                // it's also needed here to get the same interaction behaviour on touchend
                // for the newBlank template
                if (Utils.IOS) {
                    templateItem.on('touchstart', function (event) {
                        PortalUtils.iosContextEventHandler(event, templateItem);
                    });
                }

                this.append(templateItem);
            };

        };

        // helper function to get the right class name according to the used app portal
        // for the 'blank' and 'normal' template
        function getTemplatePreviewClass(appBaseName) {
            return (appBaseName === 'presentation') ? 'template-preview-presentation' : 'template-preview';
        }

        // create extension points
        this.text = Ext.point(templateExtension + '/text');
        this.textuser = Ext.point(templateExtension + '/text/templateuser');
        this.textglobal = Ext.point(templateExtension + '/text/templateglobal');
        this.spreadsheet = Ext.point(templateExtension + '/spreadsheet');
        this.spreadsheetuser = Ext.point(templateExtension + '/spreadsheet/templateuser');
        this.spreadsheetglobal = Ext.point(templateExtension + '/spreadsheet/templateglobal');
        this.presentation = Ext.point(templateExtension + '/presentation');
        this.presentationuser = Ext.point(templateExtension + '/presentation/templateuser');
        this.presentationglobal = Ext.point(templateExtension + '/presentation/templateglobal');

        // action to create a blank text document
        new Links.Action(templateExtension + '/action/new/text', {
            action: function () {
                PortalUtils.trackEvent({ moduleName: 'io.ox/office/text', target: 'templates', action: 'newblank' });
                return ox.launch('io.ox/office/text/main', PortalUtils.getNewLaunchOptions());
            }
        });

        // action to create a blank spreadsheet document
        new Links.Action(templateExtension + '/action/new/spreadsheet', {
            action: function () {
                PortalUtils.trackEvent({ moduleName: 'io.ox/office/spreadsheet', target: 'templates', action: 'newblank' });
                return ox.launch('io.ox/office/spreadsheet/main', PortalUtils.getNewLaunchOptions());
            }
        });

        // action to create a blank presentation document
        new Links.Action(templateExtension + '/action/new/presentation', {
            action: function () {
                PortalUtils.trackEvent({ moduleName: 'io.ox/office/presentation', target: 'templates', action: 'newblank' });
                return ox.launch('io.ox/office/presentation/main', PortalUtils.getNewLaunchOptions());
            }
        });

        // action to create from an existing template
        new Links.Action(templateExtension + '/action/open', {
            action: function (baton) {
                var loadDeferred = baton.data.loadDeferred;

                if (loadDeferred && loadDeferred.state() === 'pending') {
                    PortalUtils.warn('cant run convert document, because old call is still pending');
                    return;
                }

                var appModuleName = PortalUtils.getModuleNameForFile(baton.data.filename);
                loadDeferred = ox.launch(appModuleName + '/main', { action: 'convert', template: true, target_folder_id: DriveUtils.getStandardDocumentsFolderId(), templateFile: baton.data, target_filename: gt('unnamed') });
                PortalUtils.trackEvent({ moduleName: appModuleName, target: 'templates', action: 'newfromtemplate' });
                baton.data.loadDeferred = loadDeferred;
                return loadDeferred;
            }
        });

        // define extension for creating text document with a blank template
        this.text.extend({
            id: 'text_template_blank',
            draw: getBlankDrawFunction(gt('Blank text document'), templateExtension + '/action/new/text')
        });

        // define extension for creating text documents with global templates
        this.textglobal.extend({
            id: 'text_global_template',
            draw: getDrawFunction('template-text-global', { isGlobal: true })
        });

        // define extension for creating text documents with user templates
        this.textuser.extend({
            id: 'text_user_template',
            draw: getDrawFunction('template-text-local', { isGlobal: false })
        });

        // define extension for creating spreadsheet with a blank template
        this.spreadsheet.extend({
            id: 'spreadsheet_template_blank',
            draw: getBlankDrawFunction(gt('Blank spreadsheet'), templateExtension + '/action/new/spreadsheet')
        });

        // define extension for creating spreadsheet with global templates
        this.spreadsheetglobal.extend({
            id: 'spreadsheet_global_template',
            draw: getDrawFunction('template-spreadsheet-global', { isGlobal: true })
        });

        // define extension for creating spreadsheet with user templates
        this.spreadsheetuser.extend({
            id: 'spreadsheet_user_template',
            draw: getDrawFunction('template-spreadsheet-local', { isGlobal: false })
        });

        // define extension for creating presentation with a blank template
        this.presentation.extend({
            id: 'presentation_template_blank',
            draw: getBlankDrawFunction(gt('Blank presentation'), templateExtension + '/action/new/presentation')
        });

        // define extension for creating presentation with global templates
        this.presentationglobal.extend({
            id: 'presentation_global_template',
            draw: getDrawFunction('template-presentation-global', { isGlobal: true })
        });

        // define extension for creating presentation with user templates
        this.presentationuser.extend({
            id: 'presentation_user_template',
            draw: getDrawFunction('template-presentation-local', { isGlobal: false })
        });

    } // class TemplateExtensions

    return new TemplateExtensions();

});
