/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/editframework/model/fontcollection', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/render/font',
    'io.ox/office/baseframework/model/modelobject'
], function (Utils, Font, ModelObject) {

    'use strict';

    var // all known fonts, mapped by font name
        PREDEFINED_FONTS = {

            Helvetica:              { cssDefault: 'sans-serif', fontProps: { pitch: 'variable', altNames: ['Arial'] } },
            Arial:                  { cssDefault: 'sans-serif', fontProps: { pitch: 'variable', altNames: ['Helvetica'] } },
            'Arial Black':          { cssDefault: 'sans-serif', fontProps: { pitch: 'variable', altNames: ['Arial'] }, hidden: true },
            Verdana:                { cssDefault: 'sans-serif', fontProps: { pitch: 'variable', altNames: ['Arial'] } },
            Tahoma:                 { cssDefault: 'sans-serif', fontProps: { pitch: 'variable', altNames: ['Arial'] } },
            Calibri:                { cssDefault: 'sans-serif', fontProps: { pitch: 'variable', altNames: ['Carlito', 'Arial'] } },
            Carlito:                { cssDefault: 'sans-serif', fontProps: { pitch: 'variable', altNames: ['Calibri', 'Arial'] }, hidden: true },
            Albany:                 { cssDefault: 'sans-serif', fontProps: { pitch: 'variable', altNames: ['Arial'] }, hidden: true },
            Impact:                 { cssDefault: 'sans-serif', fontProps: { pitch: 'variable', altNames: ['Arial'] } },
            'Helvetica Neue':       { cssDefault: 'sans-serif', fontProps: { pitch: 'variable', altNames: ['Helvetica'] }, hidden: true },
            'Open Sans':            { cssDefault: 'sans-serif', fontProps: { pitch: 'variable', altNames: ['Helvetica Neue'] }, hidden: true },
            'Open Sans Light':      { cssDefault: 'sans-serif', fontProps: { pitch: 'variable', altNames: ['Open Sans'] }, hidden: true },
            'Open Sans Semibold':   { cssDefault: 'sans-serif', fontProps: { pitch: 'variable', altNames: ['Open Sans'] }, hidden: true },
            'Open Sans Extrabold':  { cssDefault: 'sans-serif', fontProps: { pitch: 'variable', altNames: ['Open Sans Semibold'] }, hidden: true },

            Times:                  { cssDefault: 'serif', fontProps: { pitch: 'variable', altNames: ['Times New Roman'] }, hidden: true },
            'Times New Roman':      { cssDefault: 'serif', fontProps: { pitch: 'variable', altNames: ['Times'] } },
            Georgia:                { cssDefault: 'serif', fontProps: { pitch: 'variable', altNames: ['Times New Roman'] } },
            Palatino:               { cssDefault: 'serif', fontProps: { pitch: 'variable', altNames: ['Times New Roman'] } },
            Cambria:                { cssDefault: 'serif', fontProps: { pitch: 'variable', altNames: ['Caladea', 'Times New Roman'] } },
            Caladea:                { cssDefault: 'serif', fontProps: { pitch: 'variable', altNames: ['Cambria', 'Times New Roman'] }, hidden: true },
            Thorndale:              { cssDefault: 'serif', fontProps: { pitch: 'variable', altNames: ['Times New Roman'] }, hidden: true },
            'Book Antiqua':         { cssDefault: 'serif', fontProps: { pitch: 'variable', altNames: ['Palatino'] } },

            Courier:                { cssDefault: 'monospace', fontProps: { pitch: 'fixed', altNames: ['Courier New'] }, hidden: true },
            'Courier New':          { cssDefault: 'monospace', fontProps: { pitch: 'fixed', altNames: ['Courier'] } },
            'Andale Mono':          { cssDefault: 'monospace', fontProps: { pitch: 'fixed', altNames: ['Courier New'] } },
            Consolas:               { cssDefault: 'monospace', fontProps: { pitch: 'fixed', altNames: ['Courier New'] } },
            Cumberland:             { cssDefault: 'monospace', fontProps: { pitch: 'fixed', altNames: ['Courier New'] }, hidden: true }
        },

        // a regular expression that matches valid font names
        RE_VALID_FONTNAME = new RegExp('^[^<>[\\]()"\'&/\\\\]+$');

    // class FontCollection ===================================================

    /**
     * Contains the definitions of all known fonts in a document.
     *
     * Instances of this class trigger the following events:
     * - 'insert:font'
     *      After a new font definition has been inserted into this collection,
     *      or an existing font definition has been changed. Event handlers
     *      receive the name of the changed font.
     *
     * @constructor
     *
     * @extends ModelObject
     *
     * @param {EditModel} docModel
     *  The document model containing this instance.
     */
    function FontCollection(docModel) {

        var // all registered fonts, mapped by capitalized font name
            fonts = {};

        // base constructor ---------------------------------------------------

        ModelObject.call(this, docModel);

        // private methods ----------------------------------------------------

        /**
         * Removes the calculated CSS font families from all registered fonts.
         */
        function deleteAllCssFontFamilies() {
            _.each(fonts, function (font) { delete font.cssFontFamily; });
        }

        /**
         * Calculates and returns the CSS font family attribute containing the
         * specified font name and all alternative font names.
         */
        function calculateCssFontFamily(fontName) {

            var // the resulting collection of CSS font names
                cssFontNames = [fontName],
                // the last found CSS default font family
                lastCssDefault = null;

            // collects all alternative font names of the fonts in the passed array
            function collectAltNames(fontNames) {

                var // alternative font names of the passed fonts
                    altFontNames = [];

                // collect all alternative font names
                _.each(fontNames, function (fontName2) {
                    if ((fontName2 in fonts) && _.isArray(fonts[fontName2].fontProps.altNames)) {
                        altFontNames = altFontNames.concat(fonts[fontName2].fontProps.altNames);
                        if (_.isString(fonts[fontName2].cssDefault)) {
                            lastCssDefault = fonts[fontName2].cssDefault;
                        }
                    }
                });

                // unify the new alternative font names and remove font names
                // already collected in cssFontNames (prevent cyclic references)
                altFontNames = _.chain(altFontNames).unique().difference(cssFontNames).value();

                // insert the new alternative font names into result list, and
                // call recursively until no new font names have been found
                if (altFontNames.length > 0) {
                    cssFontNames = cssFontNames.concat(altFontNames);
                    collectAltNames(altFontNames);
                }
            }

            // collect alternative font names by starting with passed font name
            collectAltNames([fontName]);

            // finally, add the CSS default family
            if (_.isString(lastCssDefault)) {
                cssFontNames.push(lastCssDefault);
            }

            // remove invalid/dangerous characters
            cssFontNames = _.filter(cssFontNames, _.bind(RE_VALID_FONTNAME.test, RE_VALID_FONTNAME));

            // enclose all fonts with white-space into quote characters
            _.each(cssFontNames, function (cssFontName, index) {
                if (/\s/.test(cssFontName)) {
                    cssFontNames[index] = '\'' + cssFontName.replace(/\s+/g, ' ') + '\'';
                }
            });

            // return the CSS font-family attribute value
            return cssFontNames.join(',');
        }

        // public methods -----------------------------------------------------

        /**
         * Inserts a new font definition into this container. An existing font
         * definition with the specified name will be replaced.
         *
         * @param {String} fontName
         *  The name of of the new font.
         *
         * @param {Object} fontProps
         *  The definition properties of the font.
         *
         * @returns {FontCollection}
         *  A reference to this instance.
         */
        this.insertFont = function (fontName, fontProps) {

            // fonts are keyed by capitalized names
            fontName = Utils.capitalizeWords(fontName);

            // store passed font attributes
            fontProps = _.copy(fontProps, true);
            fonts[fontName] = { fontProps: fontProps, hidden: false };

            // validate alternative names
            if (_.isArray(fontProps.altNames)) {
                fontProps.altNames = _.chain(fontProps.altNames).without('').map(Utils.capitalizeWords).value();
            } else {
                fontProps.altNames = [];
            }

            // add predefined alternative names
            if ((fontName in PREDEFINED_FONTS) && _.isArray(PREDEFINED_FONTS[fontName].fontProps.altNames)) {
                fontProps.altNames = fontProps.altNames.concat(PREDEFINED_FONTS[fontName].fontProps.altNames);
            }

            // unify alternative names, remove own font name
            fontProps.altNames = _.chain(fontProps.altNames).unique().without(fontName).value();

            // add default CSS font family if known
            if (fontName in PREDEFINED_FONTS) {
                fonts[fontName].cssDefault = PREDEFINED_FONTS[fontName].cssDefault;
            }

            // remove calculated CSS font families of all fonts, notify listeners
            deleteAllCssFontFamilies();
            this.trigger('insert:font', fontName);

            return this;
        };

        /**
         * Returns the names of all predefined and registered fonts. Predefined
         * fonts with the hidden flag that have not been registered will not be
         * included.
         *
         * @returns {Array<String>}
         *  The names of all predefined and registered fonts, as array.
         */
        this.getFontNames = function () {
            var fontNames = [];
            _.each(fonts, function (fontEntry, fontName) {
                if (!fontEntry.hidden) { fontNames.push(fontName); }
            });
            return fontNames;
        };

        /**
         * Returns the value of the CSS 'font-family' attribute containing the
         * specified font name and all alternative font names.
         *
         * @param {String} fontName
         *  The name of of the font (case-insensitive).
         *
         * @returns {String}
         *  The value of the CSS 'font-family' attribute containing the
         *  specified font name and all alternative font names.
         */
        this.getCssFontFamily = function (fontName) {

            // fonts are keyed by capitalized names
            fontName = Utils.capitalizeWords(fontName);

            // default for unregistered fonts: just the passed font name
            if (!(fontName in fonts)) {
                return fontName;
            }

            // calculate and cache the CSS font family
            if (!_.isString(fonts[fontName].cssFontFamily)) {
                fonts[fontName].cssFontFamily = calculateCssFontFamily(fontName);
            }

            return fonts[fontName].cssFontFamily;
        };

        /**
         * Calculates the scaled font size for the passed font size and zoom
         * factor.
         *
         * @param {Number} fontSize
         *  The original font size, in points.
         *
         * @param {Number} [zoom=1]
         *  The zoom factor. The value 1 represents the normal zoom (of 100%).
         *
         * @returns {Number}
         *  The scaled font size, in points, rounded to 1/10 of points.
         */
        this.getScaledFontSize = function (fontSize, zoom) {
            if (typeof zoom === 'number') { fontSize *= zoom; }
            return Utils.round(fontSize, 0.1);
        };

        /**
         * Converts the passed character formatting attributes to a font
         * descriptor, according to the font registered in this collection, and
         * the passed zoom factor.
         *
         * @param {Object} charAttributes
         *  The character formatting attributes, as used in operations, to be
         *  converted to the CSS font style, with the following properties:
         *  @param {String} charAttributes.fontName
         *      The original font name (case-insensitive).
         *  @param {Number} charAttributes.fontSize
         *      The original font size, in points.
         *  @param {Boolean} [charAttributes.bold=false]
         *      Whether the text will be rendered in bold characters.
         *  @param {Boolean} [charAttributes.italic=false]
         *      Whether the text will be rendered in italic characters.
         *
         * @param {Number} [zoom=1]
         *  The zoom factor used to scale the font size. The value 1 represents
         *  the normal zoom (of 100%).
         *
         * @returns {Font}
         *  The resulting font. The property 'family' includes all known
         *  fall-back fonts for the font. The property 'size' contains the font
         *  size as number (scaled according to the zoom factor if specified).
         */
        this.getFont = function (charAttributes, zoom) {
            return new Font(
                this.getCssFontFamily(charAttributes.fontName),
                this.getScaledFontSize(charAttributes.fontSize, zoom),
                charAttributes.bold,
                charAttributes.italic
            );
        };

        /**
         * Calculates the normal line height for the passed character
         * attributes, as reported by the current browser. Note that this value
         * may vary for the same font settings in different browsers.
         *
         * @param {Object} charAttributes
         *  The character formatting attributes, as used in operations. See
         *  method FontCollection.getFont() for details.
         *
         * @param {Number} [zoom=1]
         *  The zoom factor used to scale the font size. The value 1 represents
         *  the normal zoom (of 100%).
         *
         * @returns {Number}
         * The normal line height for the specified font, in pixels (with a
         * precision of 1/10 pixel for browsers using fractional line heights
         * internally).
         */
        this.getNormalLineHeight = function (charAttributes, zoom) {
            return this.getFont(charAttributes, zoom).getNormalLineHeight();
        };

        /**
         * Calculates the position of the font base line for the passed
         * character attributes and line height, as reported by the current
         * browser. Note that this value may vary for the same font settings in
         * different browsers.
         *
         * @param {Object} charAttributes
         *  The character formatting attributes, as used in operations. See
         *  method FontCollection.getFont() for details.
         *
         * @param {Number} [lineHeight]
         *  The line height to be used to determine the position of the font
         *  base line. If omitted, the normal line height, as reported by the
         *  method FontCollection.getNormalLineHeight(), will be used.
         *
         * @param {Number} [zoom=1]
         *  The zoom factor used to scale the font size. The value 1 represents
         *  the normal zoom (of 100%).
         *
         * @returns {Number}
         *  The position of the font base line for the passed character
         *  attributes and line height, in pixels, relative to the top border
         *  of the text line.
         */
        this.getBaseLineOffset = function (charAttributes, lineHeight, zoom) {
            return this.getFont(charAttributes, zoom).getBaseLineOffset(lineHeight);
        };

        /**
         * Returns the effective width of the passed text, in pixels.
         *
         * @param {Object} charAttributes
         *  The character formatting attributes, as used in operations. See
         *  method FontCollection.getFont() for details.
         *
         * @param {String} text
         *  The text whose width will be calculated. If the text consists of
         *  one character only, the calculated text width will be cached
         *  internally for subsequent invocations with the same text and
         *  character attributes.
         *
         * @param {Number} [zoom=1]
         *  The zoom factor used to scale the font size. The value 1 represents
         *  the normal zoom (of 100%).
         *
         * @returns {Number}
         *  The effective width of the passed text, in pixels.
         */
        this.getTextWidth = function (charAttributes, text, zoom) {
            // empty string: always zero (do not create a font)
            return (text.length === 0) ? 0 : this.getFont(charAttributes, zoom).getTextWidth(text);
        };

        /**
         * Returns measures for the digits '0' to '9'.
         *
         * @param {Object} charAttributes
         *  The character formatting attributes, as used in operations. See
         *  method FontCollection.getFont() for details.
         *
         * @param {Number} [zoom=1]
         *  The zoom factor used to scale the font size. The value 1 represents
         *  the normal zoom (of 100%).
         *
         * @returns {Array<Number>}
         *  An array with the widths of the digits '0' to '9', in pixels, and
         *  additional properties. See method Font.getDigitWidths() for more
         *  details.
         */
        this.getDigitWidths = function (charAttributes, zoom) {
            return this.getFont(charAttributes, zoom).getDigitWidths();
        };

        /**
         * Splits the passed string to text lines that all fit into the passed
         * width, using the specified font settings.
         *
         * @param {Object} charAttributes
         *  The character formatting attributes, as used in operations. See
         *  method FontCollection.getFont() for details.
         *
         * @param {String} text
         *  The text that will be split into single text lines.
         *
         * @param {Number} width
         *  The maximum width of the text lines, in pixels.
         *
         * @param {Number} [zoom=1]
         *  The zoom factor used to scale the font size. The value 1 represents
         *  the normal zoom (of 100%).
         *
         * @returns {Array}
         *  An array of strings containing the single text lines.
         */
        this.getTextLines = function (charAttributes, text, width, zoom) {
            return this.getFont(charAttributes, zoom).getTextLines(text, width);
        };

        // initialization -----------------------------------------------------

        // start with all predefined fonts
        fonts = _.copy(PREDEFINED_FONTS, true);

        // destroy all class members on destruction
        this.registerDestructor(function () {
            docModel = fonts = null;
        });

    } // class FontCollection

    // static methods ---------------------------------------------------------

    /**
     * Returns the first font name specified in the passed font family string
     * received from a CSS font-family attribute. The CSS font family may
     * consist of a comma separated list of font names, where the font names
     * may be enclosed in quote characters.
     *
     * @param {String} cssFontFamily
     *  The value of a CSS font-family attribute.
     *
     * @returns {String}
     *  The first font name from the passed string, without quote characters.
     */
    FontCollection.getFirstFontName = function (cssFontFamily) {

        var // extract first font name from comma separated list
            fontName = $.trim(cssFontFamily.split(',')[0]);

        // Remove leading and trailing quote characters (be tolerant and allow
        // mixed quotes such as "Font' or even a missing quote character on one
        // side of the string). After that, trim again and return.
        return Utils.capitalizeWords($.trim(fontName.match(/^["']?(.*?)["']?$/)[1]).toLowerCase());
    };

    // exports ================================================================

    // derive this class from class ModelObject
    return ModelObject.extend({ constructor: FontCollection });

});
