/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/editframework/model/editmodel', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/utils/logger',
    'io.ox/office/baseframework/model/basemodel',
    'io.ox/office/editframework/utils/operations',
    'io.ox/office/editframework/utils/operationcontext',
    'io.ox/office/editframework/model/undomanager',
    'io.ox/office/editframework/model/operationsgenerator',
    'io.ox/office/editframework/model/modelattributesmixin'
], function (Utils, Logger, BaseModel, Operations, OperationContext, UndoManager, OperationsGenerator, ModelAttributesMixin) {

    'use strict';

    var // operations logger
        logger = new Logger({ enable: 'office:log-ops', prefix: 'OP' }),

        // operation filter predicate that accepts all operations
        acceptFilter = _.constant(true),

        // operation filter predicate that rejects all operations
        rejectFilter = _.constant(false);

    // class EditModel ========================================================

    /**
     * The base class for editable document models. Adds generic support for
     * operations and undo action management.
     *
     * Triggers the following events:
     * - 'operations:before'
     *      Before an array of operations will be applied via the method
     *      EditModel.applyOperations(). The event handler receives the
     *      operations array, and the external state passed to the method. If a
     *      single operation has been passed to that method, it will be
     *      converted to an array with one element before calling the event
     *      handler.
     * - 'operations:after'
     *      After an array of operations have been applied via the method
     *      EditModel.applyOperations(), regardless if successfully or not. The
     *      event handler receives the external state passed to the method
     *      EditModel.applyOperations().
     * - 'operations:success'
     *      Directly after the event 'operations:after', if the operations have
     *      been applied successfully. The event handler receives the
     *      operations array, and the external state passed to the method
     *      EditModel.applyOperations(). If a single operation has been passed
     *      to that method, it will be converted to an array with one element
     *      before calling the event handler.
     * - 'operations:error'
     *      Directly after the event 'operations:after', if applying the
     *      operations has failed. The event handler receives the external
     *      state passed to the method EditModel.applyOperations().
     * - 'change:editmode'
     *      When the edit mode of the document model has been changed. Event
     *      handlers receive the new state of the edit mode.
     * - 'change:attributes': After the document attributes have been changed
     *      via the method 'EditModel.setDocumentAttributes()'. Event handlers
     *      receive the following parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {Object} newAttributes
     *          The new (complete) map of document attributes.
     *      (3) {Object} oldAttributes
     *          The old (complete) map of document attributes.
     *      (4) {Object} changedAttributes
     *          A map with the document attributes that have really changed.
     * - 'change:defaults': After the default values of formatting attributes
     *      have been changed. Event handlers receive the following parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {Object} newAttributeSet
     *          A complete attributes set with all current default values.
     *      (3) {Object} oldAttributeSet
     *          A complete attributes set with all previous default values.
     *      (4) {Object} changedAttributeSet
     *          An incomplete attributes set with all default values that have
     *          really changed.
     *
     * @constructor
     *
     * @extends BaseModel
     * @extends ModelAttributesMixin
     *
     * @param {EditApplication} app
     *  The application containing this document model instance.
     *
     * @param {UndoManager} undoManager
     *  The undo manager used by the document model to store undo and redo
     *  actions.
     *
     * @param {Object} [initOptions]
     *  Optional parameters that will be passed to the constructor of the undo
     *  manager (see class UndoManager for details). Additionally, the
     *  following options are supported:
     *  @param {Function} [initOptions.generatorClass=OperationsGenerator]
     *      The constructor function of the operations generator used by this
     *      document model. MUST be a sub class of class OperationsGenerator.
     *  @param {Function} [initOptions.contextClass=OperationContext]
     *      The constructor function of the operation context class used to
     *      wrap JSON operation objects, passed to operation handler callback
     *      functions. MUST be a sub class of class OperationContext.
     *  @param {Function} [initOptions.operationsFinalizer]
     *      A callback function that will be invoked before applying operations
     *      and sending them to the server. Receives an array of JSON operation
     *      objects, and MUST return an array of processed operations (the
     *      return value may be the same array modified in-place). Will be
     *      called in the context of this document model.
     */
    function EditModel(app, initOptions) {

        var // the constructor of the operations generator class used by this document
            GeneratorClass = Utils.getFunctionOption(initOptions, 'generatorClass', OperationsGenerator),

            // the constructor of the operations generator class used by this document
            ContextClass = Utils.getFunctionOption(initOptions, 'contextClass', OperationContext),

            // self reference for local functions
            self = this,

            // Deferred object for current asynchronous actions, will remain in
            // resolved/rejected state as long as no new actions will be applied
            actionsDef = $.when(),

            // callback to preprocess operations before applying them
            operationsFinalizer = Utils.getFunctionOption(initOptions, 'operationsFinalizer'),

            // maps all operation names to operation handler functions
            operationHandlers = {},

            // whether operations are currently applied (prevent recursive calls)
            processingOperations = false,

            // edit mode, received dynamically from server (one editing user at a time)
            editMode = null,

            // the total number of operations applied successfully
            totalOperations = 0,

            // a counter for the operations, that adds all single operations, taking care of merged
            // operations. So this number must be equal on every client, independent from merge
            // processes on client or server side.
            operationStateNumber = -1,

            // collecting the last two operations in cache
            operationsCache = { previous: null, current: null },

            // the undo manager of this document
            undoManager = null;

        // base constructors --------------------------------------------------

        BaseModel.call(this, app);
        ModelAttributesMixin.call(this);

        // private methods ----------------------------------------------------

        /**
         * Switches this document model into 'action processing mode', and
         * invokes the passed callback function. As long as processing mode is
         * active, the method EditModel.getActionsPromise() returns a pending
         * promise. The callback function may return a promise to control the
         * duration of the action processing mode. This method MUST NOT be
         * called recursively! The methods EditModel.applyOperations() and
         * EditModel.applyActions() must not be used (these methods will enter
         * the action processing mode by themselves).
         *
         * @param {Function} callback
         *  The callback function that will be invoked with activated action
         *  processing mode. Must return either a boolean value that represents
         *  the result of the callback function, or a promise to extend the
         *  processing mode until the promise has been resolved or rejected.
         *  Will be called in the context of this instance.
         *
         * @param {Boolean} [async=false]
         *  If set to true, the return value of this method will always be a
         *  promise. Internal errors will be represented by a rejected promise
         *  with an object with property 'cause'.
         *
         * @returns {Boolean|jQuery.Promise}
         *  The return value of the callback function; or false on an internal
         *  error.
         */
        function enterProcessActionsMode(callback, async) {

            var // the result of the callback function
                result = null;

            // prevent recursive invocations
            if (actionsDef.state() === 'pending') {
                Utils.error('EditModel.enterProcessActionsMode(): recursive call');
                return async ? $.Deferred().reject({ cause: 'recursive' }) : false;
            }

            // create a new deferred object which remains in pending state as long
            // as the callback function runs (and the promise it returns is pending)
            actionsDef = $.Deferred().always(function () {
                var state = actionsDef ? actionsDef.state().replace(/ed$/, 'ing') : 'destructed';
                logger.log('EditModel.enterProcessActionsMode(): ' + state + ' promise...');
            });

            // invoke the callback function
            result = callback.call(self);

            // resolve the actions promise
            if (_.isObject(result) && _.isFunction(result.promise)) {
                result.progress(_.bind(actionsDef.notify, actionsDef))
                    .done(_.bind(actionsDef.resolve, actionsDef))
                    .fail(_.bind(actionsDef.reject, actionsDef));
                return result;
            }

            // reject on any value but true
            if (result === true) { actionsDef.resolve(); } else { actionsDef.reject(); }
            return async ? actionsDef.promise() : result;
        }

        /**
         * Converts the passed operations input parameter of various methods to
         * a plain array of JSON operations.
         *
         * @param {Object|Array|OperationsGenerator} operations
         *  A single JSON operation, or an array of JSON operations, or an
         *  operations generator instance with operations.
         *
         * @returns {Array}
         *  An array of JSON operations.
         */
        function getOperationsArray(operations) {
            return (operations instanceof OperationsGenerator) ? operations.getOperations() : _.getArray(operations);
        }

        /**
         * Extracts and returns the operation filter callback function from the
         * passed options.
         */
        function getFilterCallback(options) {
            var filterFunc = Utils.getFunctionOption(options, 'filter');
            return _.isFunction(filterFunc) ? filterFunc : Utils.getBooleanOption(options, 'filter', true) ? acceptFilter : rejectFilter;
        }

        /**
         * Called before an operations array will be applied at the document
         * model. Triggers an 'operations:before' event.
         *
         * @param {Array} operations
         *  The JSON operations to be applied. Will be passed to the listeners
         *  of the 'operations:before' event.
         *
         * @param {Boolean} external
         *  Whether the operations have been received from the server. Will be
         *  passed to the listeners of the 'operations:before' event.
         *
         * @param {Boolean} shallow
         *  Whether the operation will be applied in shallow mode, i.e. without
         *  checking and/or updating the internal operation state number. Will
         *  be passed to the listeners of the 'operations:before' event.
         *
         * @returns {Boolean}
         *  False if this method has been called recursively, i.e. the model
         *  already processes operations. This is considered a fatal error.
         */
        function beginProcessOperations(operations, external, shallow) {

            if (processingOperations) {
                Utils.error('EditModel.beginProcessOperations(): recursive call - currently processing operations.');
                return false;
            }

            if (app.isInternalError()) {
                Utils.error('EditModel.beginProcessOperations(): illegal call - application is in error state.');
                return false;
            }

            self.trigger('operations:before', operations, external, shallow);
            processingOperations = true;
            return true;
        }

        /**
         * Called after an operations array has been applied at the document
         * model. Triggers an 'operations:after' event, and one of the
         * 'operations:success' or 'operations:error' events.
         *
         * @param {Boolean} success
         *  Whether applying the operations has succeeded.
         *
         * @param {Array} operations
         *  The operations that have been applied. Will be passed to the
         *  listeners of the operations events.
         *
         * @param {Boolean} external
         *  Whether the operations have been received from the server. Will be
         *  passed to the listeners of the operations events.
         *
         * @param {Boolean} shallow
         *  Whether the operation have been applied in shallow mode, i.e.
         *  without checking and/or updating the internal operation state
         *  number. Will be passed to the listeners of the operations events.
         */
        function endProcessOperations(success, operations, external, shallow) {
            processingOperations = false;
            self.trigger('operations:after', operations, external, shallow);
            if (success) {
                self.trigger('operations:success', operations, external, shallow);
            } else {
                self.setEditMode(false);
                self.trigger('operations:error', operations, external, shallow);
            }
        }

        /**
         * Executes the handler function for the passed operation.
         *
         * @param {Object} operation
         *  The JSON operation object to be applied at this document model.
         *
         * @param {Boolean} external
         *  Whether the operation has been received from the server.
         *
         * @param {Boolean} shallow
         *  Whether to apply the operation in shallow mode, i.e. without
         *  checking and/or updating the internal operation state number.
         *
         * @param {Function} filter
         *  A predicate function that specifies whether to invoke the operation
         *  callback handler.
         *
         * @returns {Boolean}
         *  Whether the operation has been applied successfully.
         */
        function applyOperation(operation, external, shallow, filter) {

            function failAndReturn() {
                var message = 'EditModel.applyOperation(): ' + _.printf.apply(_, arguments);
                Utils.error(message);
                app.sendLogMessage(message);
                return false;
            }

            // log operation
            logger.log('i=' + totalOperations + ', op=', operation);

            // check operation and operation name
            if (!_.isObject(operation)) {
                return failAndReturn('Expecting operation object.');
            }

            // operation handler may throw an exception
            try {

                var // the context passed to the operation handler
                    context = new ContextClass(self, operation, external),
                    // the callback handler for the operation
                    handler = operationHandlers[context.getStr('name')],
                    // the operation scope
                    scope = context.getOptEnum('scope', /^(all|filter)$/, 'all');

                // check existence of operation handler (always, also if operation will be filtered below)
                context.ensure(handler, 'unknown operation');

                // Check the correct operation state number of an external operation. The OSN must be
                // identical on every client. The OSN is sent from server, when the document is loaded.
                if (external && !shallow && context.has('osn')) {

                    // the OSN carried in the operation
                    var opOSN = context.getInt('osn');
                    context.ensure(opOSN >= 0, 'invalid OSN: %d', opOSN);

                    // During loading the document, the initial OSN is sent by the server.
                    // If we don't have a valid value use that OSN.
                    if (operationStateNumber === -1) {
                        operationStateNumber = opOSN;
                    }

                    // the current OSN must be equal to the OSN contained in the external operation
                    if (operationStateNumber !== opOSN) {
                        return failAndReturn('Wrong operation state number. current=%d operation=%d', operationStateNumber, operation.osn);
                    }
                }

                // in shallow mode, do not touch the internal OSN, but check that the operation does not carry an OSN
                if (shallow) {
                    context.ensure(external, 'unexpected shallow mode for internal operation');
                    context.ensure(!context.has('osn'), 'unexpected OSN in shallow operation');
                }

                // insert current OSN and OPL into internal operations
                if (!external) {
                    operation.osn = operationStateNumber;
                    operation.opl = 1;
                }

                // update the current OSN of the document, unless the operations are shallow
                if (!shallow && (operationStateNumber >= 0)) {
                    operationStateNumber += context.getOptInt('opl', 1);
                }

                // cache the last two operations
                operationsCache.previous = operationsCache.current;
                operationsCache.current = operation;

                // invoke the operation handler (unless the operation scope is 'filter' for operations intended
                // to be executed in the filter component only, or the filter callback function returns false)
                if ((scope !== 'filter') && (filter.call(self, operation) === true)) {
                    handler.call(self, context);
                }

            } catch (ex) {
                return failAndReturn('Operation failed. name="%s" msg="%s" osn=%d source=%s op=%s', operation.name, Utils.getStringOption(ex, 'message', ''), totalOperations + 1, external ? 'ext' : 'int', JSON.stringify(operation));
            }

            totalOperations += 1;
            return true;
        }

        /**
         * Executes the handler functions for all passed operations. If the
         * undo manager is active, all applied operations will be embedded into
         * a single undo group action.
         *
         * @param {Object|Array|OperationsGenerator} operations
         *  A single JSON operation, or an array of JSON operations, or an
         *  operations generator instance with operations to be applied at this
         *  document model. Note that an operations generator passed to this
         *  method will NOT be cleared after its operations have been sent.
         *
         * @param {Boolean} external
         *  Whether the operation has been received from the server.
         *
         * @param {Boolean} shallow
         *  Whether to apply the operation in shallow mode, i.e. without
         *  checking and/or updating the internal operation state number.
         *
         * @param {Function} filter
         *  A predicate function that specifies whether to invoke the operation
         *  callback handler.
         *
         * @returns {Boolean}
         *  Whether applying all operations was successful.
         */
        function applyOperationsSync(operations, external, shallow, filter) {

            var // the boolean result for synchronous mode
                success = false;

            // convert input parameter to an array of JSON operations
            operations = getOperationsArray(operations);

            // if the operation's origin is not an external client and
            // we don't have edit rights then we skip the operations silently.
            if ((!external && !editMode) || (operations.length === 0)) {
                return true;
            }

            // allow subclasses to modify locally generated operations
            if (!external && _.isFunction(operationsFinalizer)) {
                operationsFinalizer.call(self, operations);
            }

            // enter operations mode (prevent recursive calls)
            if (beginProcessOperations(operations, external, shallow)) {
                // apply all operations at once, group into an undo action
                undoManager.enterUndoGroup(function () {
                    success = _.all(operations, function (operation) {
                        return applyOperation(operation, external, shallow, filter);
                    });
                });
            }

            // post-processing (trigger end events, leave operations mode),
            // also if beginProcessOperations() has failed (notify the error)
            endProcessOperations(success, operations, external, shallow);
            return success;
        }

        /**
         * Executes the handler functions for all passed operations in an
         * asynchronous background loop. If the undo manager is active, all
         * applied operations will be embedded into a single undo group action.
         *
         * @param {Object|Array|OperationsGenerator} operations
         *  A single JSON operation, or an array of JSON operations, or an
         *  operations generator instance with operations to be applied at this
         *  document model. Note that an operations generator passed to this
         *  method will NOT be cleared after its operations have been sent.
         *
         * @param {Boolean} external
         *  Whether the operation has been received from the server.
         *
         * @param {Boolean} shallow
         *  Whether to apply the operation in shallow mode, i.e. without
         *  checking and/or updating the internal operation state number.
         *
         * @param {Function} filter
         *  A predicate function that specifies whether to invoke the operation
         *  callback handler.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved after all operations have been
         *  applied successfully, or rejected immediately after applying an
         *  operation has failed. The promise regularly sends progress updates
         *  while applying the operations.
         */
        function applyOperationsAsync(operations, external, shallow, filter) {

            var // the resulting Promise
                promise = null,
                // the global index of the first applied operation
                startIndex = totalOperations;

            // convert input parameter to an array of JSON operations
            operations = getOperationsArray(operations);

            // immediate success on empty operations array
            if (operations.length === 0) { return $.when(); }

            // enter operations mode (prevent recursive calls)
            if (beginProcessOperations(operations, external, shallow)) {

                // apply all operations in a background task, group into an undo action
                promise = undoManager.enterUndoGroup(function () {

                    // return the promise to enterUndoGroup() to defer the open undo group
                    return self.iterateArraySliced(operations, function (operation) {

                        // if the operation's origin is not an external client and
                        // we don't have edit rights then we skip the operations silently
                        if (!external && !editMode) {
                            promise.abort();
                            return Utils.BREAK;
                        }

                        // break the loop and reject the promise if an operation has failed
                        if (!applyOperation(operation, external, shallow, filter)) {
                            return $.Deferred().reject({ cause: 'operation' });
                        }
                    }, { delay: 'immediate' });
                });

            } else {
                // currently applying operations (called recursively)
                promise = $.Deferred().reject({ cause: 'recursive' });
            }

            // all passed operations applied successfully: trigger end events, leave operations mode
            self.waitForSuccess(promise, function () {
                endProcessOperations(true, operations, external, shallow);
            });

            // operations failed: trigger end events, leave operations mode
            self.waitForFailure(promise, function (result) {
                if (result === 'abort') {
                    // manually aborted: notify applied operations only
                    endProcessOperations(true, operations.slice(0, totalOperations - startIndex), external, shallow);
                } else {
                    // operations failed: notify all operations
                    endProcessOperations(false, operations, external, shallow);
                }
            });

            return promise;
        }

        // protected methods --------------------------------------------------

        /**
         * Registers a callback function that will be executed when a document
         * operation with the specified name will be applied.
         *
         * @param {String} name
         *  The name of the document operation, as contained in the property
         *  'name' of an operation JSON object, to be handled by the passed
         *  callback function.
         *
         * @param {Function} handler
         *  The callback function that will be invoked for every operation with
         *  the specified name. Receives the following parameters:
         *  (1) {OperationContext} context
         *      A wrapper for the JSON operation object, representing the
         *      operation to be applied for this document. The exact type of
         *      this parameter depends on the class passed to the constructor
         *      parameter 'contextClass' of this document model.
         *  Will be called in the context of this document model instance. If
         *  the operation handler throws an exception, further processing of
         *  other operations will be stopped immediately, and the event
         *  'operations:error' will be triggered. The return value of the
         *  callback function will be ignored.
         *
         * @returns {EditModel}
         *  A reference to this instance.
         */
        this.registerContextOperationHandler = function (name, handler) {
            operationHandlers[name] = handler;
            return this;
        };

        /**
         * Registers a callback function that will be executed when a document
         * operation with the specified name will be applied.
         *
         * @deprecated
         *  Use method EditModel.registerContextOperationHandler() instead.
         *
         * @param {String} name
         *  The name of the document operation, as contained in the property
         *  'name' of an operation JSON object, to be handled by the passed
         *  callback function.
         *
         * @param {Function} handler
         *  The callback function that will be invoked for every operation with
         *  the specified name. Receives the following parameters:
         *  (1) {Object} operation
         *      The operation to be applied for this document, as JSON object.
         *  (2) {Boolean} external
         *      The external state flag that has been passed e.g. to the method
         *      EditModel.applyActions().
         *  Will be called in the context of this document model instance. If
         *  the operation handler returns the Boolean value false, or throws an
         *  exception, further processing of other operations will be stopped
         *  immediately, and the event 'operations:error' will be triggered.
         *  Every other return value (also undefined) indicates that the
         *  operation has been applied successfully.
         *
         * @returns {EditModel}
         *  A reference to this instance.
         */
        this.registerOperationHandler = function (name, handler) {
            return this.registerContextOperationHandler(name, function (context) {
                // old operation handlers may return false for failed operation
                var result = handler.call(this, context.operation, context.external);
                context.ensure(result !== false, 'operation handler failed');
            });
        };

        // public methods -----------------------------------------------------

        /**
         * Returns whether the 'action processing mode' of this document model
         * is currently active (whether this model currently generates,
         * applies, or sends operation actions, either synchronously or
         * asynchronously).
         *
         * @returns {Boolean}
         *  Whether the 'action processing mode' is currently active.
         */
        this.isProcessingActions = function () {
            return actionsDef.state() === 'pending';
        };

        /**
         * Returns a promise representing the state of the 'action processing
         * mode'. If in pending state, this model currently generates, applies,
         * or sends operation actions (either synchronously or asynchronously).
         * If in resolved state, the last operations have been applied
         * successfully. If in rejected state, the last operations have not
         * been applied successfully.
         *
         * @attention
         *  The promise MUST NOT be cached externally. Its solely purpose is to
         *  register done/fail callbacks from operation handlers or other code
         *  depending on operations. Whenever new operations will be applied, a
         *  new promise will be created representing the new operations (the
         *  old promises returned by this method before will remain in
         *  resolved/rejected state forever).
         *
         * @returns {jQuery.Promise}
         *  A promise representing the state of the action processing mode.
         */
        this.getActionsPromise = function () {
            return actionsDef.promise();
        };

        /**
         * Returns whether operations are currently processed.
         *
         * @returns {Boolean}
         *  Whether the model processes operations.
         */
        this.isProcessingOperations = function () {
            return processingOperations;
        };

        /**
         * Sends the passed operations to the server without applying them
         * locally. In special situations, a client may choose to apply the
         * operations manually before they will be sent to the server. In this
         * case, the registered operation handlers must not be called.
         *
         * @param {Object|Array|OperationsGenerator} operations
         *  A single JSON operation, or an array of JSON operations, or an
         *  operations generator instance with operations to be sent to the
         *  server. Note that an operations generator passed to this method
         *  will NOT be cleared after its operations have been sent.
         *
         * @returns {Boolean}
         *  Whether registering all operations for sending was successful.
         */
        this.sendOperations = logger.profileMethod('EditModel.sendOperations()', function (operations) {
            return applyOperationsSync(operations, false, false, rejectFilter);
        });

        /**
         * Executes the handler functions for all passed operations. If the
         * undo manager is active, all applied operations will be embedded into
         * a single undo group action.
         *
         * @param {Object|Array|OperationsGenerator} operations
         *  A single JSON operation, or an array of JSON operations, or an
         *  operations generator instance with operations to be applied at this
         *  document model. Note that an operations generator passed to this
         *  method will NOT be cleared after its operations have been applied.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.async=false]
         *      If set to true, the operations will be applied asynchronously
         *      in a browser timeout loop. The promise returned by this method
         *      will be resolved or rejected after applying all operations, and
         *      will be notified about the progress of the operation.
         *  @param {Function|Boolean} [options.filter]
         *      A predicate function that will be invoked for each operation
         *      contained in the passed actions. Receives the operation as
         *      first parameter, and must return a Boolean specifying whether
         *      to invoke the operation handler. Can be set to the Boolean
         *      value false to prevent applying any operation locally (only
         *      send them to the server).
         *
         * @returns {Boolean|jQuery.Promise}
         *  A boolean value specifying whether applying all operations was
         *  successful in synchronous mode (option 'async' has not been set),
         *  or a promise for asynchronous mode. The promise will be resolved
         *  after all operations have been applied successfully, or rejected
         *  immediately after applying an operation has failed. The caller has
         *  to wait for the promise before new operations can be applied. The
         *  promise regularly sends progress updates while applying the
         *  operations. When rejected, the promise will pass a result object
         *  with the following properties:
         *  - {String} result.cause
         *      The error code. Will be set to 'operation', if an operation
         *      callback handler has failed; or to 'recursive', if this method
         *      has been invoked recursively.
         */
        this.applyOperations = logger.profileMethod('EditModel.applyOperations()', function (operations, options) {

            // apply the operations synchronously, but enter action processing mode anyway,
            // operation handler may register themselves at the actions promise
            return enterProcessActionsMode(function () {

                var // whether to execute asynchronously
                    async = Utils.getBooleanOption(options, 'async', false),
                    // the private method to be called
                    applyOperationsMethod = async ? applyOperationsAsync : applyOperationsSync;

                // apply the passed operations
                return applyOperationsMethod(operations, false, false, getFilterCallback(options));
            });
        });

        /**
         * Executes the handler functions for all operations contained in the
         * passed actions. The operations will always be applied in an
         * asynchronous background loop. The operations of each action will be
         * enclosed into distinct undo groups automatically.
         *
         * @param {Object|Array} actions
         *  A single document action, or an array with document actions, to be
         *  applied at this document model. A document action is a JSON object
         *  with an optional array property 'operations' containing JSON
         *  operations as elements.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.external=false]
         *      If set to true, the actions have been received from an external
         *      sender. Otherwise, the actions have been generated by the
         *      document model itself.
         *  @param {Boolean} [options.shallow=false]
         *      If set to true, the current internal operation state number
         *      will not be modified. The operations in the actions are
         *      expected to not specify own operation state numbers.
         *  @param {Function} [options.filter]
         *      A predicate function that will be invoked for each operation
         *      contained in the passed actions. Receives the operation as
         *      first parameter, and must return a Boolean specifying whether
         *      to invoke the operation handler. Can be set to the Boolean
         *      value false to prevent applying any operation locally (only
         *      send them to the server).
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved after all actions have been applied
         *  successfully, or rejected immediately after applying an action has
         *  failed. The caller has to wait for the promise before new actions
         *  can be applied. The promise regularly sends progress updates while
         *  applying the actions. It will contain an additional method abort()
         *  that can be used to stop applying the operations at any time. When
         *  rejected, the promise will pass a result object with the following
         *  properties:
         *  - {String} result.cause
         *      The error code. Will be set to 'operation', if an operation
         *      callback handler has failed; or to 'recursive', if this method
         *      has been invoked recursively.
         */
        this.applyActions = logger.profileAsyncMethod('EditModel.applyActions()', function (actions, options) {

            var // total number of operations
                operationCount = 0;

            // convert parameter to an array, filter for actions containing
            // at least one operation, get total number of operations
            actions = _.getArray(actions).filter(function (action) {
                var count = _.isArray(action.operations) ? action.operations.length : 0;
                operationCount += count;
                return count > 0;
            });

            // no operations: early exit
            if (operationCount === 0) { return $.when(); }

            // apply in action processing mode
            return enterProcessActionsMode(function () {

                var // whether the actions have been received from the server
                    external = Utils.getBooleanOption(options, 'external', false),
                    // whether to stick to the current internal OSN
                    shallow = Utils.getBooleanOption(options, 'shallow', false),
                    // invocation filter callback
                    filter = getFilterCallback(options),
                    // resulting deferred object returned by this method
                    resultDef = $.Deferred(),
                    // start index of first operations in the current action (for progress)
                    startIndex = 0,
                    // the promise of the inner loop processing operations of an action
                    innerPromise = null;

                // apply the actions in a browser timeout loop
                var timer = self.iterateArraySliced(actions, function (action) {

                    // apply the operations of the current action
                    innerPromise = applyOperationsAsync(action.operations, external, shallow, filter);

                    // calculate and notify the total progress of all actions
                    innerPromise.progress(function (progress) {
                        var currIndex = startIndex + action.operations.length * progress;
                        resultDef.notify(currIndex / operationCount);
                    });

                    // update the start index for the next action
                    innerPromise.done(function () { startIndex += action.operations.length; });

                    // clean up after finishing the loop
                    return innerPromise.always(function () { innerPromise = null; });

                }, { delay: 'immediate' });

                // forward the result of the timer to the own deferred object
                timer.done(function () { resultDef.notify(1).resolve(); }).fail(_.bind(resultDef.reject, resultDef));

                // create an abortable promise for the deferred object
                return self.createAbortablePromise(resultDef, function () {
                    // abort the outer loop processing the actions
                    timer.abort();
                    // abort the inner loop processing the operations of an action
                    if (innerPromise) { innerPromise.abort(); }
                });
            }, true);
        });

        /**
         * Executes the operation handler callback function for the passed JSON
         * operation locally and silently. No events will be generated at all
         * (the operation will not be sent to the server, the internal OSN will
         * not be changed, and it will not be processed in any other way).
         *
         * @param {Object} operation
         *  The JSON operation object of the document operation to be applied
         *  locally and silently.
         *
         * @returns {Boolean}
         *  Whether the passed operation has been applied successfully.
         */
        this.invokeOperationHandler = function (operation) {

            // callback handler throws exceptions to indicate operation errors
            try {

                var // the operation context passed to the callback
                    context = new ContextClass(this, operation, false),
                    // the callback handler for the operation
                    handler = operationHandlers[context.getStr('name')];

                // check existence of callback function , and invoke it
                context.ensure(handler, 'unknown operation');
                handler.call(this, context);
                return true;

            } catch (ex) {
                Utils.exception(ex);
                return false;
            }
        };

        /**
         * Creates and returns an initialized operations generator instance (an
         * instance of the class specified in the constructor parameter
         * OperationsGeneratorClass).
         *
         * @returns {OperationsGenerator}
         *  A new operations generator instance.
         */
        this.createOperationsGenerator = function () {
            return new GeneratorClass(this);
        };

        /**
         * Creates a new operations generator, invokes the callback function,
         * and then applies all operations contained in the generator, and
         * sends them to the server.
         *
         * @param {Function} callback
         *  The callback function. Receives a new operations generator as first
         *  parameter. May return a promise to defer applying and sending the
         *  operations until the promise has been resolved. Will be called in
         *  the context of this instance.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Function} [options.filter]
         *      A predicate function that will be invoked for each operation
         *      contained in the passed actions. Receives the operation as
         *      first parameter, and must return a Boolean specifying whether
         *      to invoke the operation handler. Can be set to the Boolean
         *      value false to prevent applying any operation locally (only
         *      send them to the server).
         *  @param {OperationsGeneratoe} [options.generator]
         *      A custom operations generator instance that will be passed to
         *      the callback function. If omitted, a new default operations
         *      generator will be created (see method
         *      EditModel.createOperationsGenerator() for details).
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved after the operations have been
         *  applied and sent successfully (with the result of the callback
         *  function); or rejected, if the callback has returned a rejected
         *  promise (with the result of that promise), or if applying the
         *  operations has failed (with the object {cause:'operation'}).
         */
        this.createAndApplyOperations = logger.profileAsyncMethod('EditModel.createAndApplyOperations()', function (callback, options) {

            // simulate actions processing mode for deferred callbacks
            return enterProcessActionsMode(function () {

                var // the generator passed to the callback function
                    generator = Utils.getObjectOption(options, 'generator'),
                    // the promise returned by this method
                    promise = null;

                // create default operations generator if none has been passed to this method
                if (!(generator instanceof OperationsGenerator)) { generator = self.createOperationsGenerator(); }

                // invoke the callback function
                promise = $.when(callback.call(self, generator));

                // apply and send the operations after the callback has finished
                promise = promise.then(function (result) {
                    var success = applyOperationsSync(generator, false, false, getFilterCallback(options));
                    return success ? result : $.Deferred().reject({ cause: 'operation' });
                });

                return promise;
            }, true);
        });

        /**
         * Returns the total number of operations applied successfully.
         */
        this.getOperationsCount = function () {
            return totalOperations;
        };

        /**
         * Returns the operation state number of the document.
         */
        this.getOperationStateNumber = function () {
            return operationStateNumber;
        };

        /**
         * Sets the operation state number of the document. Used when empty doc is fast displayed
         * @param {Number} osn - new osn number
         */
        this.setOperationStateNumber = function (osn) {
            operationStateNumber = osn;
            this.trigger('change:osn', osn); // update osn value in operations pane
        };

        /**
         * Returns the operation cache with the last two operations.
         *
         * @returns {Object}
         *  An object containing the last two operations in the properties
         *  last and current.
         */
        this.getOperationsCache = function () {
            return operationsCache;
        };

        /**
         * Returns the previous operation saved in the operation cache.
         * This is required for performance reasons. So no copy is returned
         * but the original operation object.
         *
         * @returns {Object}
         *  An object containing the previous operation.
         */
        this.getPreviousOperation = function () {
            return operationsCache.previous;
        };

        /**
         * Returns whether the document can be modified by the current user.
         *
         * @returns {Boolean}
         *  Whether the document can be modified.
         */
        this.getEditMode = function () {
            return editMode;
        };

        /**
         * Enables or disables the edit mode of this document model.
         *
         * @param {Boolean} state
         *  The new edit mode state. If set to true, the user can modify the
         *  document, otherwise the user can view the document contents, but
         *  cannot edit them.
         *
         * @returns {EditModel}
         *  A reference to this instance.
         */
        this.setEditMode = function (state) {
            if (editMode !== state) {
                editMode = state;
                // notify state change to listeners
                this.trigger('change:editmode', editMode);
            }
            return this;
        };

        /**
         * Returns the undo manager of this document containing the undo action
         * stack and the redo action stack.
         *
         * @returns {UndoManager}
         *  The undo manager of this document.
         */
        this.getUndoManager = function () {
            return undoManager;
        };

        // operation handlers -------------------------------------------------

        this.registerContextOperationHandler(Operations.SET_DOCUMENT_ATTRIBUTES, function (context) {
            // setting document/default attributes is not undoable
            this.setDocumentAttributes(context.getObj('attrs'));
        });

        this.registerContextOperationHandler(Operations.INSERT_THEME, function (context) {
            this.getThemeCollection().insertTheme(context.getStr('themeName'), context.getObj('attrs'));
        });

        this.registerContextOperationHandler(Operations.INSERT_FONT_DESCRIPTION, function (context) {
            this.getFontCollection().insertFont(context.getStr('fontName'), context.getObj('attrs'));
        });

        this.registerContextOperationHandler(Operations.INSERT_STYLESHEET, function (context) {

            var // the style family name
                styleFamily = context.getStr('type'),
                // target style sheet collection
                styleCollection = this.getStyleCollection(styleFamily),
                // the unique identifier of the style sheet
                styleId = context.getStr('styleId'),
                // the readable name of the style sheet
                styleName = context.getOptStr('styleName', styleId, true); // TODO: do not allow empty string, but will be sent for auto styles currently

            // check that the style family is valid (style sheet collection exists)
            context.ensure(styleCollection, 'invalid style family \'%s\'', styleFamily);

            // create undo operation (but only for custom style sheets, not for lateral style sheets (40878))
            if (undoManager.isLocalUndo() && context.getOptBool('custom')) {
                undoManager.addUndo({ name: Operations.DELETE_STYLESHEET, type: styleFamily, styleId: styleId }, context.operation);
            }

            return styleCollection.insertStyleSheet(styleId, styleName, context.getOptStr('parent'), context.getOptObj('attrs'), {
                auto: context.getOptBool('auto'),
                hidden: context.getOptBool('hidden'),
                priority: context.getOptInt('uiPriority'),
                defStyle: context.getOptBool('default'),
                custom: context.getOptBool('custom')
            });
        });

        this.registerContextOperationHandler(Operations.DELETE_STYLESHEET, function (context) {

            var // the style family name
                styleFamily = context.getStr('type'),
                // target style sheet collection
                styleCollection = this.getStyleCollection(styleFamily),
                // the unique identifier of the style sheet
                styleId = context.getStr('styleId'),
                // undo operation
                undo = null;

            // check that the style family is valid (style sheet collection exists)
            context.ensure(styleCollection, 'invalid style family \'%s\'', styleFamily);

            // create undo operation
            if (undoManager.isLocalUndo()) {
                undo = {
                    name: Operations.INSERT_STYLESHEET,
                    type: styleFamily,
                    styleId: styleId,
                    styleName: styleCollection.getName(styleId),
                    parentId: styleCollection.getParentId(styleId),
                    attrs: styleCollection.getStyleSheetAttributeMap(styleId),
                    hidden: styleCollection.isHidden(styleId),
                    uiPriority: styleCollection.getUIPriority(styleId),
                    custom: styleCollection.isCustom(styleId)
                };
                undoManager.addUndo(undo, context.operation);
            }

            return styleCollection.deleteStyleSheet(styleId);
        });

        this.registerContextOperationHandler(Operations.CHANGE_STYLESHEET, function (context) {

            var // the style family name
                styleFamily = context.getStr('type'),
                // target style sheet collection
                styleCollection = this.getStyleCollection(styleFamily),
                // the unique identifier of the style sheet
                styleId = context.getStr('styleId'),
                // the readable name of the style sheet
                styleName = context.has('styleName') ? context.getStr('styleName') : null,
                // the new parent of the style sheet
                parentId = context.has('parent') ? context.getStr('parent') : null,
                // the new attribute set for the style sheet
                attributes = context.getOptObj('attrs'),
                // undo operation
                undo = null;

            // check that the style family is valid (style sheet collection exists)
            context.ensure(styleCollection, 'invalid style family \'%s\'', styleFamily);

            // create undo operation
            if (undoManager.isLocalUndo()) {
                undo = { name: Operations.CHANGE_STYLESHEET, type: styleFamily, styleId: styleId };
                if (styleName) {
                    undo.styleName = styleCollection.getName(styleId);
                }
                if (attributes) {
                    undo.attrs = styleCollection.getStyleSheetAttributeMap(styleId);
                }
                undoManager.addUndo(undo, context.operation);
            }

            return styleCollection.changeStyleSheet(styleId, styleName, parentId, attributes);
        });

        // register handler for the operation 'noOp': does nothing, used for updating OSN etc.
        this.registerContextOperationHandler(Operations.NOOP, _.noop);

        // initialization -----------------------------------------------------

        // create the undo manager
        undoManager = new UndoManager(this, initOptions);

        // destroy all class members on destruction
        this.registerDestructor(function () {
            undoManager.destroy();
            self = initOptions = actionsDef = null;
            operationsFinalizer = operationHandlers = null;
            undoManager = GeneratorClass = null;
        });

    } // class EditModel

    // exports ================================================================

    // derive this class from class BaseModel
    return BaseModel.extend({ constructor: EditModel });

});
