/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Stefan Eckert <stefan.eckert@open-xchange.com>
 */

define('io.ox/office/drawinglayer/view/chartformatter', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/object/baseobject',
    'io.ox/office/drawinglayer/view/chartstyleutil',
    'gettext!io.ox/office/drawinglayer/main'
], function (Utils, BaseObject, ChartStyleUtil, gt) {

    'use strict';

    //'io.ox/office/spreadsheet/model/numberformatter'
    // null date (corresponding to cell value zero) TODO: use null date from document settings

    // class ChartFormatter ===================================================

    function ChartFormatter(docModel, chartModel) {

        // base constructor ---------------------------------------------------

        BaseObject.call(this);

        // private methods ----------------------------------------------------

        function newInfoHolder(type, label) {
            return {
                max: Number.NEGATIVE_INFINITY,
                date: false,
                type: type,
                label: label,
                unclean: false,
                exp: 0,
                labelLength: 0
            };
        }

        function handleData(info, source, dataPoint, nullAsZero) {
            var realNr = source.result;
            var display = source.display;
            var format = source.format;

            if (format && !info.format) {
                if (format.isAnyDateTime()) {
                    info.date = format;
                }
                info.format = format;
            }

            var label = null;
            var res = null;
            if (info.date) {
                res = realNr;
                info.max = Math.max(info.max, res);
                label = display;

            } else if (realNr === undefined || realNr === null || isNaN(realNr)) {
                info.unclean = true;
                res = nullAsZero ? 0 : undefined;
                label = display;
            } else if (info.format) {
                info.max = Math.max(info.max, realNr);
                res = realNr;
                label = docModel.getNumberFormatter().formatValue(realNr, info.format);
            } else {
                info.max = Math.max(info.max, realNr);
                res = realNr;
                label = docModel.getNumberFormatter().formatStandardNumber(realNr, 12);
            }

            dataPoint[info.label] = Utils.escapeHTML(label);
            dataPoint[info.type] = res;
        }

        function calcMaxExp(info) {
            if (isFinite(info.max)) {
                var norm = Utils.normalizeNumber(info.max);
                if (Math.abs(norm.exp) < 6) {
                    info.exp = 0;
                } else {
                    info.exp = norm.exp;
                }
            } else {
                info.exp = 0;
            }
        }

        function handleMantissa(info, dataPoint) {
            if (info.exp) {
                dataPoint[info.type] = Utils.mant(dataPoint[info.type], info.exp);
            }
        }

        function handleLabelLength(info, dataPoint) {
            var label = dataPoint[info.label];
            if (label) {
                info.labelLength = Math.max(info.labelLength, label.length);
            }
        }

        // public methods -----------------------------------------------------

        this.format = function () {
            var data = chartModel.getModelData();
            var attrs = chartModel.getMergedAttributes();
            var chartType = attrs.chart.type;
            var xyChart = chartModel.isXYType() || chartType.indexOf('area') === 0;
            var nullAsZero = chartType.indexOf('pie') === -1 && chartType.indexOf('donut') === -1;
            var dateChart = false;

            var x = newInfoHolder('x', 'label');
            var y = newInfoHolder('y', 'name');

            data.linearX = false;

            _.each(data.series, function (dataSeries, seriesIndex) {
                dataSeries.seriesIndex = seriesIndex;
                if (dataSeries.nameHolder) {
                    dataSeries.name = dataSeries.nameHolder.display;
                } else {
                    dataSeries.name =
                    //#. A data series in a chart object (a sequence of multiple data points with a title)
                    //#. %1$d is the numeric index of the series (e.g. "Series 1", "Series 2", etc.)
                    //#. This label should match the default title for data series in other spreadsheet
                    //#. applications (OpenOffice/LibreOffice Calc, Microsoft Excel).
                    //#, c-format
                    gt('Series %1$d', _.noI18n(seriesIndex + 1));
                }

                var dataPoints = dataSeries.dataPoints;
                _.each(dataPoints, function (dataPoint, n) {
                    dataPoint.x = null;
                    dataPoint.y = null;
                    dataPoint.z = null;

                    handleData(y, dataPoint.valueSource, dataPoint, nullAsZero);

                    if (!n && !seriesIndex && dataPoint.nameSource) {
                        //first of first
                        if (attrs.chart.stacking === 'clustered' || attrs.chart.stacking === 'standard') {
                            var format = dataPoint.nameSource ? dataPoint.nameSource.format : null;
                            if (format && format.isAnyDateTime()) {
                                dateChart = true;
                            }
                        }
                    }

                    if (dateChart) {
                        handleData(x, dataPoint.nameSource, dataPoint, nullAsZero);
                    } else if (xyChart && dataPoint.nameSource) {
                        handleData(x, dataPoint.nameSource, dataPoint, nullAsZero);
                    } else {
                        dataPoint[x.type] = n + 1;
                        data.linearX = true;
                    }

                    //ugly part against double formatting!!!
                    if (dataPoint.nameSource) {
                        dataPoint[x.label] = Utils.escapeHTML(dataPoint.nameSource.display);
                    } else {
                        dataPoint[x.label] = Utils.escapeHTML(String(dataPoint[x.type]));
                    }

                    handleLabelLength(x, dataPoint);
                    handleLabelLength(y, dataPoint);

                    if (dataPoint.sizeSource) {
                        dataPoint.z = Math.abs(dataPoint.sizeSource.result);

                        //fix for funny bug in canvasjs
                        dataPoint.markerBorderColor = 'transparent';
                        dataPoint.markerBorderThickness = 1;

                    } else {
                        dataPoint.z = null;
                    }
                    dataPoint.legendText = dataPoint.label;
                });

                dataSeries.toolTipContent = '{label}: {name}';

                if (xyChart && x.unclean) {
                    _.each(dataPoints, function (dataPoint, n) {
                        dataPoint.x = n + 1;
                    });
                }
            });

            if (xyChart) {
                calcMaxExp(x);
            }
            calcMaxExp(y);

            chartModel.getAxisModel('x').setFormat(x);
            chartModel.getAxisModel('y').setFormat(y);

            var thresholdY = 0.005;

            _.each(data.series, function (dataSeries) {
                _.each(dataSeries.dataPoints, function (dataPoint) {
                    if (!xyChart) {
                        //workaround that too small values stay visible
                        var rel = dataPoint.y / y.max;
                        if (Math.abs(rel) < thresholdY) {
                            if (rel < 0) {
                                dataPoint.y = -thresholdY * y.max;
                            } else {
                                dataPoint.y = thresholdY * y.max;
                            }
                        }
                    }

                    handleMantissa(y, dataPoint);
                    handleMantissa(x, dataPoint);
                });
            });

            data.data = data.series.slice(0);
            var mustReverse = attrs.chart.type.indexOf('bar') === 0 && (attrs.chart.stacking === 'clustered' || attrs.chart.stacking === 'standard');
            if (mustReverse) {
                data.data.reverse();
            }
        };

        this.update = function (seriesIndex, values, titles, names, bubbles) {
            var data = chartModel.getModelData();

            var dataSeries = data.series[seriesIndex];

            // build series title
            dataSeries.nameHolder = titles[0];
            if (!seriesIndex) {
                if (dataSeries.nameHolder) {
                    chartModel.setMainName(dataSeries.nameHolder.display);
                } else {
                    chartModel.setMainName(null);
                }
            }
            dataSeries.dataPoints = [];
            _.each(values, function (valueCell, n) {
                var dataPoint = {};
                dataSeries.dataPoints.push(dataPoint);
                dataPoint.nameSource = names[n];    //x
                dataPoint.valueSource = valueCell;  //y
                dataPoint.sizeSource = bubbles[n];  //z
                dataPoint.color = null;
                dataPoint.markerColor = null;
            });
        };

        // initialization -----------------------------------------------------

        this.registerDestructor(function () {
            docModel = chartModel = null;
        });

    } // class ChartFormatter

    // exports ================================================================

    return BaseObject.extend({ constructor: ChartFormatter });

});
