/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Edy Haryono <edy.haryono@open-xchange.com>
 */
define('io.ox/office/baseframework/view/control/userpicture', [
    'io.ox/contacts/api',
    'io.ox/office/tk/utils',
    'io.ox/office/tk/forms',
    'io.ox/office/tk/control/group'
], function (ContactsAPI, Utils, Forms, Group) {

    'use strict';

    // class UserPicture ======================================================

    /**
     * A form control that displays the photo uploaded by a specific user.
     *
     * @constructor
     *
     * @extends Group
     *
     * @param {Object|Null} userData
     *  The initial user information. If specified, must contain the string
     *  properties 'userId' and 'userName'.
     *
     * @param {Object} [initOptions]
     *  Optional parameters. Supports all options of the base class Group.
     *  Additionally, the following options are supported:
     *  @param {Number} [initOptions.size=48]
     *      The size of the picture element, in pixels.
     */
    function UserPicture(userData, initOptions) {

        var // the size of the image container
            nodeSize = Utils.getIntegerOption(initOptions, 'size', 48),

            // size of the photo bitmap (double on retina displays)
            bitmapSize = nodeSize * (Utils.RETINA ? 2 : 1),

            // the element containing the photo (as CSS background)
            pictureNode = $('<div class="photo-container" style="min-width:' + nodeSize + 'px;height:' + nodeSize + 'px;background-size:' + nodeSize + 'px;">');

        // base constructors --------------------------------------------------

        Group.call(this, initOptions);

        // private methods ----------------------------------------------------

        /**
         * Loads the photo of the specified user, according to the value of the
         * control.
         */
        function updateHandler(currentData) {

            var // unique identifier of the user
                userId = Utils.getIntegerOption(currentData, 'userId'),
                // display name of the user
                userName = _.noI18n(Utils.getStringOption(currentData, 'userName', ''));

            ContactsAPI.pictureHalo(pictureNode, _.isNumber(userId) ? { internal_userid: userId } : {}, { width: bitmapSize, height: bitmapSize, scaleType: 'cover' });
            Forms.setToolTip(pictureNode, { tooltip: userName });
        }

        // initialization -----------------------------------------------------

        this.getNode().addClass('user-picture');

        // insert the image container into the mark-up of this group
        this.addChildNodes(pictureNode);

        // update handler for user photo
        this.registerUpdateHandler(updateHandler);

        // initialize value of this control
        this.setValue(userData).refresh();

        // destroy class members
        this.registerDestructor(function () {
            userData = initOptions = pictureNode = null;
        });

    } // class UserPicture

    // exports ================================================================

    // derive this class from class Group
    return Group.extend({ constructor: UserPicture });

});
