/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/baseframework/app/appobjectmixin', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/io'
], function (Utils, IO) {

    'use strict';

    // mix-in class AppObjectMixin ============================================

    /**
     * A mix-in class intended to add convenience methods to arbitrary classes
     * for interacting with an application instance. Provides methods to send
     * server requests, and to defer code execution according to the state of
     * the document import process. All deferred code (server request handlers,
     * import callbacks) will be aborted automatically when destroying this
     * instance.
     *
     * @attention
     *  The extended object (bound to the symbol this) MUST be an instance of
     *  the class BaseObject.
     *
     * @constructor
     *
     * @param {BaseApplication} app
     *  The application this instance is bound to.
     */
    function AppObjectMixin(app) {

        var // all pending server requests currently running
            pendingRequests = null,

            // a promise representing whether importing the document has started
            importStartPromise = app.getImportStartPromise(),

            // a promise representing whether importing the document is finished
            importFinishPromise = app.getImportFinishPromise();

        // private methods ----------------------------------------------------

        /**
         * Aborts all pending server requests.
         */
        function abortPendingRequests() {
            // Each aborted request removes itself from the array, but a single abort() call
            // inside the while-loop MAY abort other dependent requests as well, therefore
            // a while loop will be used that checks the array length in each iteration.
            if (pendingRequests) {
                while (pendingRequests.length > 0) {
                    pendingRequests[0].abort();
                }
            }
        }

        // public methods -----------------------------------------------------

        /**
         * Sends a request to the server and returns a promise waiting for the
         * response. The unique identifier of the application will be added to
         * the request parameters automatically. See method IO.sendRequest()
         * for further details.
         *
         * @param {String} module
         *  The name of the server module that will receive the request.
         *
         * @param {Object} [params]
         *  Parameters that will be inserted into the request URL (method GET),
         *  or into the request body (method POST).
         *
         * @param {Object} [options]
         *  Optional parameters. See method IO.sendRequest() for details.
         *
         * @returns {jQuery.Promise}
         *  The abortable promise representing the server request. See method
         *  IO.sendRequest() for details.
         */
        this.sendRequest = function (module, params, options) {

            // add the required application UID and document type to the request parameters
            params = _.extend({}, params, { uid: app.get('uniqueID'), app: app.getDocumentType() });

            // send the request (returns an abortable promise)
            var request = IO.sendRequest(module, params, options);

            // store the request internally for automatic abort on destruction
            (pendingRequests || (pendingRequests = [])).push(request);

            // remove the request from the storage after it is finished
            request.always(function () {
                var index = pendingRequests.indexOf(request);
                if (index >= 0) { pendingRequests.splice(index, 1); }
            });

            return request;
        };

        /**
         * Sends a request to the server and returns a promise waiting for the
         * response. This method automatically adds the unique identifier of
         * the application, and the parameters of the file currently opened.
         * See method IO.sendRequest() for further details.
         *
         * @param {String} module
         *  The name of the server module that will receive the request.
         *
         * @param {Object} [params]
         *  Parameters that will be inserted into the request URL (method GET),
         *  or into the request body (method POST).
         *
         * @param {Object} [options]
         *  Optional parameters. See method IO.sendRequest() for details.
         *
         * @returns {jQuery.Promise}
         *  The abortable promise representing the server request. Will be
         *  rejected immediately, if the application is not connected to a
         *  document file. See method IO.sendRequest() for details.
         */
        this.sendFileRequest = function (module, params, options) {

            // reject immediately if no file is present
            if (!app.hasFileDescriptor()) {
                return $.Deferred().reject();
            }

            // extend parameters with file settings
            params = _.extend({}, params, app.getFileParameters());

            // send the request
            return this.sendRequest(module, params, options);
        };

        /**
         * Sends a request to the document filter server module to determine
         * if certain server features are available.
         *
         * @param {String|Array<String>} features
         *  The names of all required server features. Can be a simple string
         *  for a single server feature, or an array of strings for multiple
         *  server features.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved if ALL passed server features are
         *  available; or that will be rejected if at least one server feature
         *  is missing. The rejected promise will provide an array containing
         *  all missing feature names.
         */
        this.requireServerFeatures = function (features) {

            // send the feature request
            var promise = this.sendRequest(IO.FILTER_MODULE_NAME, { action: 'getfeatures' });

            // convert parameter to an array
            features = _.getArray(features);

            // evaluate the server response
            return promise.then(function (data) {

                // get the features from the result (fail immediately if missing or an empty string)
                var availableFeatures = Utils.getStringOption(data, 'features');
                if (!availableFeatures) { return $.Deferred().reject(features); }

                // the response is a comma-separated list, convert to an array
                availableFeatures = availableFeatures.split(/,/);

                // reduce the feature names passed to this method, result contains missing features
                var missingFeatures = _.difference(features, availableFeatures);
                if (missingFeatures.length > 0) { return $.Deferred().reject(missingFeatures); }
            });
        };

        /**
         * Return whether importing the document has started, or is already
         * finished.
         *
         * @returns {Boolean}
         *  Whether importing the document has started. Will be false before
         *  importing the document; and will be true after import has started,
         *  or even finished.
         */
        this.isImportStarted = function () {
            return importStartPromise.state() !== 'pending';
        };

        /**
         * Return whether importing the document was successful.
         *
         * @returns {Boolean}
         *  Whether importing the document has been completed successfully.
         *  Will be false before or while importing the document, or after
         *  import has failed; and will be true after import has succeeded.
         */
        this.isImportSucceeded = function () {
            return importFinishPromise.state() === 'resolved';
        };

        /**
         * Return whether importing the document has failed.
         *
         * @returns {Boolean}
         *  Whether importing the document has failed. Will be false before or
         *  while importing the document, or after import has finished
         *  successfully; and will be true after import has failed.
         */
        this.isImportFailed = function () {
            return importFinishPromise.state() === 'rejected';
        };

        /**
         * Return whether importing the document is completed, regardless
         * whether it was successful.
         *
         * @returns {Boolean}
         *  Whether importing the document is completed. Will be false before
         *  or while importing the document; and will be true after import has
         *  succeeded or failed.
         */
        this.isImportFinished = function () {
            return importFinishPromise.state() !== 'pending';
        };

        /**
         * Invokes the passed callback function before the application passed
         * to the constructor starts importing the document. If this method is
         * used after importing the document has started, the passed callback
         * function will be invoked immediately (unless initialization of the
         * application has failed).
         *
         * @param {Function} callback
         *  The callback function invoked before the application starts
         *  importing the document. Receives the following parameters:
         *  (1) {Boolean} alreadyStarted
         *      Whether the import was already started while invoking this
         *      method (i.e. this callback is being invoked immediately).
         *
         * @param {Object} [context]
         *  The context bound to the invoked callback function.
         *
         * @returns {AppObjectMixin}
         *  A reference to this instance.
         */
        this.waitForImportStart = function (callback, context) {
            callback = callback.bind(context, this.isImportStarted());
            return this.waitForSuccess(importStartPromise, callback);
        };

        /**
         * Invokes the passed callback function as soon as the application
         * passed to the constructor has finished importing the document
         * successfully. If this method is used after importing the document,
         * the passed callback function will be invoked immediately (unless
         * import has failed).
         *
         * @param {Function} callback
         *  The callback function invoked after the application has finished
         *  importing the document successfully. Receives the following
         *  parameters:
         *  (1) {Boolean} alreadyFinished
         *      Whether the import was already finished while invoking this
         *      method (i.e. this callback is being invoked immediately).
         *
         * @param {Object} [context]
         *  The context bound to the invoked callback function.
         *
         * @returns {AppObjectMixin}
         *  A reference to this instance.
         */
        this.waitForImportSuccess = function (callback, context) {
            callback = callback.bind(context, this.isImportFinished());
            return this.waitForSuccess(importFinishPromise, callback);
        };

        /**
         * Invokes the passed callback function as soon as the application
         * passed to the constructor has failed importing the document. If this
         * method is used after importing the document, the passed callback
         * function will be invoked immediately (unless import has succeeded).
         *
         * @param {Function} callback
         *  The callback function invoked after the application has failed
         *  importing the document. Receives the following parameters:
         *  (1) {Boolean} alreadyFinished
         *      Whether the import was already finished while invoking this
         *      method (i.e. this callback is being invoked immediately).
         *
         * @param {Object} [context]
         *  The context bound to the invoked callback function.
         *
         * @returns {AppObjectMixin}
         *  A reference to this instance.
         */
        this.waitForImportFailure = function (callback, context) {
            callback = callback.bind(context, this.isImportFinished());
            return this.waitForFailure(importFinishPromise, callback);
        };

        /**
         * Invokes the passed callback function as soon as the application
         * passed to the constructor has finished importing the document,
         * regardless whether the import was successful. If this method is used
         * after importing the document, the passed callback function will be
         * invoked immediately.
         *
         * @param {Function} callback
         *  The callback function invoked after the application has finished
         *  importing the document. Receives the following parameters:
         *  (1) {Boolean} alreadyFinished
         *      Whether the import was already finished while invoking this
         *      method (i.e. this callback is being invoked immediately).
         *
         * @param {Object} [context]
         *  The context bound to the invoked callback function.
         *
         * @returns {AppObjectMixin}
         *  A reference to this instance.
         */
        this.waitForImport = function (callback, context) {
            callback = callback.bind(context, this.isImportFinished());
            return this.waitForAny(importFinishPromise, callback);
        };

        // initialization -----------------------------------------------------

        // abort all running server requests, before invoking the application quit handlers
        this.listenTo(app, 'docs:beforequit', abortPendingRequests);

        // abort all running server requests on destruction of this instance
        this.registerDestructor(function () {
            abortPendingRequests();
            pendingRequests = importStartPromise = importFinishPromise = null;
        });

    } // class AppObjectMixin

    // exports ================================================================

    return AppObjectMixin;

});
