/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define([
    'io.ox/office/tk/render/font',
    'io.ox/office/tk/locale/formatter'
], function (Font, Formatter) {

    'use strict';

    // class Formatter ========================================================

    describe('Toolkit class Formatter', function () {

        it('should exist', function () {
            expect(Formatter).to.be.a('function');
        });

        // private helpers ----------------------------------------------------

        var formatter = null;

        // constructor --------------------------------------------------------

        describe('constructor', function () {
            it('creates a new instance', function () {
                formatter = new Formatter();
                expect(formatter).to.be.an('object');
            });
        });

        // public methods -----------------------------------------------------

        describe('method "configure"', function () {
            it('should exist', function () {
                expect(formatter).to.respondTo('configure');
            });
        });

        describe('method "isValidDate"', function () {
            it('should exist', function () {
                expect(formatter).to.respondTo('isValidDate');
            });
        });

        describe('method "convertNumberToDate"', function () {
            it('should exist', function () {
                expect(formatter).to.respondTo('convertNumberToDate');
            });
        });

        describe('method "convertDateToNumber"', function () {
            it('should exist', function () {
                expect(formatter).to.respondTo('convertDateToNumber');
            });
        });

        describe('method "parseLeadingDate"', function () {
            it('should exist', function () {
                expect(formatter).to.respondTo('parseLeadingDate');
            });
        });

        describe('method "parseLeadingTime"', function () {
            it('should exist', function () {
                expect(formatter).to.respondTo('parseLeadingTime');
            });
        });

        describe('method "parseFormattedNumber"', function () {
            it('should exist', function () {
                expect(formatter).to.respondTo('parseFormattedNumber');
            });
        });

        describe('method "formatStandardNumber"', function () {
            it('should exist', function () {
                expect(formatter).to.respondTo('formatStandardNumber');
            });
            it('should format numbers to decimal notation', function () {
                expect(formatter.formatStandardNumber(0, 7)).to.equal('0');
                expect(formatter.formatStandardNumber(1, 7)).to.equal('1');
                expect(formatter.formatStandardNumber(-1, 7)).to.equal('-1');
                expect(formatter.formatStandardNumber(12, 7)).to.equal('12');
                expect(formatter.formatStandardNumber(-12, 7)).to.equal('-12');
                expect(formatter.formatStandardNumber(123, 7)).to.equal('123');
                expect(formatter.formatStandardNumber(-123, 7)).to.equal('-123');
                expect(formatter.formatStandardNumber(1230, 7)).to.equal('1230');
                expect(formatter.formatStandardNumber(-1230, 7)).to.equal('-1230');
                expect(formatter.formatStandardNumber(12300, 7)).to.equal('12300');
                expect(formatter.formatStandardNumber(-12300, 7)).to.equal('-12300');
                expect(formatter.formatStandardNumber(123000, 7)).to.equal('123000');
                expect(formatter.formatStandardNumber(-123000, 7)).to.equal('-123000');
                expect(formatter.formatStandardNumber(1230000, 7)).to.equal('1230000');
                expect(formatter.formatStandardNumber(-1230000, 7)).to.equal('-1230000');
                expect(formatter.formatStandardNumber(1e10, 20)).to.equal('10000000000');
                expect(formatter.formatStandardNumber(-1e10, 20)).to.equal('-10000000000');
                expect(formatter.formatStandardNumber(9.87654321e307, 310)).to.match(/^9876543210{299}$/);
                expect(formatter.formatStandardNumber(-9.87654321e307, 310)).to.match(/^-9876543210{299}$/);
                expect(formatter.formatStandardNumber(12345.6, 7)).to.equal('12345,6');
                expect(formatter.formatStandardNumber(-12345.6, 7)).to.equal('-12345,6');
                expect(formatter.formatStandardNumber(1234.56, 7)).to.equal('1234,56');
                expect(formatter.formatStandardNumber(-1234.56, 7)).to.equal('-1234,56');
                expect(formatter.formatStandardNumber(123.456, 7)).to.equal('123,456');
                expect(formatter.formatStandardNumber(-123.456, 7)).to.equal('-123,456');
                expect(formatter.formatStandardNumber(12.3456, 7)).to.equal('12,3456');
                expect(formatter.formatStandardNumber(-12.3456, 7)).to.equal('-12,3456');
                expect(formatter.formatStandardNumber(1.23456, 7)).to.equal('1,23456');
                expect(formatter.formatStandardNumber(-1.23456, 7)).to.equal('-1,23456');
                expect(formatter.formatStandardNumber(0.123, 7)).to.equal('0,123');
                expect(formatter.formatStandardNumber(-0.123, 7)).to.equal('-0,123');
                expect(formatter.formatStandardNumber(0.0123, 7)).to.equal('0,0123');
                expect(formatter.formatStandardNumber(-0.0123, 7)).to.equal('-0,0123');
                expect(formatter.formatStandardNumber(0.00123, 7)).to.equal('0,00123');
                expect(formatter.formatStandardNumber(-0.00123, 7)).to.equal('-0,00123');
            });
            it('should round fractional part in decimal notation', function () {
                expect(formatter.formatStandardNumber(0.12345, 6)).to.equal('0,1235');
                expect(formatter.formatStandardNumber(-0.12345, 6)).to.equal('-0,1235');
                expect(formatter.formatStandardNumber(0.12345, 5)).to.equal('0,123');
                expect(formatter.formatStandardNumber(-0.12345, 5)).to.equal('-0,123');
                expect(formatter.formatStandardNumber(0.12345, 4)).to.equal('0,12');
                expect(formatter.formatStandardNumber(-0.12345, 4)).to.equal('-0,12');
                expect(formatter.formatStandardNumber(0.12345, 3)).to.equal('0,1');
                expect(formatter.formatStandardNumber(-0.12345, 3)).to.equal('-0,1');
                expect(formatter.formatStandardNumber(0.12345, 2)).to.equal('0');
                expect(formatter.formatStandardNumber(-0.12345, 2)).to.equal('-0');
                expect(formatter.formatStandardNumber(0.12345, 1)).to.equal('0');
                expect(formatter.formatStandardNumber(-0.12345, 1)).to.equal('-0');
                expect(formatter.formatStandardNumber(123.456, 6)).to.equal('123,46');
                expect(formatter.formatStandardNumber(-123.456, 6)).to.equal('-123,46');
                expect(formatter.formatStandardNumber(123.456, 5)).to.equal('123,5');
                expect(formatter.formatStandardNumber(-123.456, 5)).to.equal('-123,5');
                expect(formatter.formatStandardNumber(1234.56, 5)).to.equal('1235');
                expect(formatter.formatStandardNumber(-1234.56, 5)).to.equal('-1235');
                expect(formatter.formatStandardNumber(1234.56, 4)).to.equal('1235');
                expect(formatter.formatStandardNumber(-1234.56, 4)).to.equal('-1235');
                expect(formatter.formatStandardNumber(9999.99, 6)).to.equal('10000');
                expect(formatter.formatStandardNumber(-9999.99, 6)).to.equal('-10000');
                expect(formatter.formatStandardNumber(8.9999, 1)).to.equal('9');
                expect(formatter.formatStandardNumber(-8.9999, 1)).to.equal('-9');
                expect(formatter.formatStandardNumber(0.9999, 1)).to.equal('1');
                expect(formatter.formatStandardNumber(-0.9999, 1)).to.equal('-1');
                expect(formatter.formatStandardNumber(0.09999, 1)).to.equal('0');
                expect(formatter.formatStandardNumber(-0.09999, 1)).to.equal('-0');
            });
            it('should format numbers to scientific notation', function () {
                expect(formatter.formatStandardNumber(3450000000, 10)).to.equal('3450000000');
                expect(formatter.formatStandardNumber(-3450000000, 10)).to.equal('-3450000000');
                expect(formatter.formatStandardNumber(3450000000, 9)).to.equal('3,45E+09');
                expect(formatter.formatStandardNumber(-3450000000, 9)).to.equal('-3,45E+09');
                expect(formatter.formatStandardNumber(3450000000, 8)).to.equal('3,45E+09');
                expect(formatter.formatStandardNumber(-3450000000, 8)).to.equal('-3,45E+09');
                expect(formatter.formatStandardNumber(3450000000, 7)).to.equal('3,5E+09');
                expect(formatter.formatStandardNumber(-3450000000, 7)).to.equal('-3,5E+09');
                expect(formatter.formatStandardNumber(3450000000, 6)).to.equal('3E+09');
                expect(formatter.formatStandardNumber(-3450000000, 6)).to.equal('-3E+09');
                expect(formatter.formatStandardNumber(3450000000, 5)).to.equal('3E+09');
                expect(formatter.formatStandardNumber(-3450000000, 5)).to.equal('-3E+09');
                expect(formatter.formatStandardNumber(9876000000, 5)).to.equal('1E+10');
                expect(formatter.formatStandardNumber(-9876000000, 5)).to.equal('-1E+10');
                expect(formatter.formatStandardNumber(0.00003456, 10)).to.equal('0,00003456');
                expect(formatter.formatStandardNumber(-0.00003456, 10)).to.equal('-0,00003456');
                expect(formatter.formatStandardNumber(0.00003456, 9)).to.equal('3,456E-05');
                expect(formatter.formatStandardNumber(-0.00003456, 9)).to.equal('-3,456E-05');
                expect(formatter.formatStandardNumber(0.00003456, 8)).to.equal('3,46E-05');
                expect(formatter.formatStandardNumber(-0.00003456, 8)).to.equal('-3,46E-05');
                expect(formatter.formatStandardNumber(0.00003456, 7)).to.equal('3,5E-05');
                expect(formatter.formatStandardNumber(-0.00003456, 7)).to.equal('-3,5E-05');
                expect(formatter.formatStandardNumber(0.00003456, 6)).to.equal('3E-05');
                expect(formatter.formatStandardNumber(-0.00003456, 6)).to.equal('-3E-05');
                expect(formatter.formatStandardNumber(0.00003456, 5)).to.equal('3E-05');
                expect(formatter.formatStandardNumber(-0.00003456, 5)).to.equal('-3E-05');
                expect(formatter.formatStandardNumber(0.00003456, 4)).to.equal('0');
                expect(formatter.formatStandardNumber(-0.00003456, 4)).to.equal('-0');
                expect(formatter.formatStandardNumber(0.00003456, 1)).to.equal('0');
                expect(formatter.formatStandardNumber(-0.00003456, 1)).to.equal('-0');
                expect(formatter.formatStandardNumber(1e-100, 5)).to.equal('0');
                expect(formatter.formatStandardNumber(-1e-100, 5)).to.equal('-0');
            });
            it('should be able to round up to an exponent with more digits', function () {
                expect(formatter.formatStandardNumber(9.99e99, 8)).to.equal('9,99E+99');
                expect(formatter.formatStandardNumber(9.99e99, 7)).to.equal('1E+100');
            });
            it('should prefer decimal notation over scientific notation', function () {
                expect(formatter.formatStandardNumber(0.001, 5)).to.equal('0,001');
                expect(formatter.formatStandardNumber(10000, 5)).to.equal('10000');
            });
            it('should fail for large integral parts', function () {
                expect(formatter.formatStandardNumber(12345, 4)).to.equal(null);
                expect(formatter.formatStandardNumber(1234, 3)).to.equal(null);
                expect(formatter.formatStandardNumber(123, 2)).to.equal(null);
                expect(formatter.formatStandardNumber(12, 1)).to.equal(null);
                expect(formatter.formatStandardNumber(999.5, 3)).to.equal(null);
                expect(formatter.formatStandardNumber(99.5, 2)).to.equal(null);
                expect(formatter.formatStandardNumber(9.5, 1)).to.equal(null);
                expect(formatter.formatStandardNumber(9.49999999999, 1)).to.equal('9');
            });
            it('should format decimal notation for all powers of 10', function () {
                for (var i = 1; i <= 307; i += 1) {
                    expect(formatter.formatStandardNumber(Number('1e' + i), 310)).to.match(new RegExp('^10{' + i + '}$'));
                    expect(formatter.formatStandardNumber(Number('1e-' + i), 310)).to.match(new RegExp('^0,0{' + (i - 1) + '}1$'));
                }
            });
            it('should format scientific notation for all powers of 10', function () {
                for (var i = 21; i <= 307; i += 1) {
                    expect(formatter.formatStandardNumber(Number('1e' + i), 21)).to.match(new RegExp('^1E\\+' + i + '$'));
                    expect(formatter.formatStandardNumber(Number('1e-' + i), 21)).to.match(new RegExp('^1E-' + i + '$'));
                }
            });
            it('should format with a custom exponent length', function () {
                expect(formatter.formatStandardNumber(1e9, 8, { expLength: 1 })).to.equal('1E+9');
                expect(formatter.formatStandardNumber(1e9, 8, { expLength: 2 })).to.equal('1E+09');
                expect(formatter.formatStandardNumber(1e9, 8, { expLength: 3 })).to.equal('1E+009');
                expect(formatter.formatStandardNumber(1e9, 8, { expLength: 4 })).to.equal('1E+0009');
                expect(formatter.formatStandardNumber(1e19, 8, { expLength: 1 })).to.equal('1E+19');
                expect(formatter.formatStandardNumber(1e19, 8, { expLength: 2 })).to.equal('1E+19');
                expect(formatter.formatStandardNumber(1e19, 8, { expLength: 3 })).to.equal('1E+019');
                expect(formatter.formatStandardNumber(1e19, 8, { expLength: 4 })).to.equal('1E+0019');
                expect(formatter.formatStandardNumber(1e199, 8, { expLength: 1 })).to.equal('1E+199');
                expect(formatter.formatStandardNumber(1e199, 8, { expLength: 2 })).to.equal('1E+199');
                expect(formatter.formatStandardNumber(1e199, 8, { expLength: 3 })).to.equal('1E+199');
                expect(formatter.formatStandardNumber(1e199, 8, { expLength: 4 })).to.equal('1E+0199');
            });
            it('should use scientific notation for specific edge cases', function () {
                expect(formatter.formatStandardNumber(5e-100, 6)).to.equal('5E-100');
                expect(formatter.formatStandardNumber(5e-100, 5)).to.equal('1E-99');
                expect(formatter.formatStandardNumber(5e-10, 5)).to.equal('5E-10');
                expect(formatter.formatStandardNumber(5e-10, 4)).to.equal('0');
                expect(formatter.formatStandardNumber(5e-10, 4, { expLength: 1 })).to.equal('1E-9');
                expect(formatter.formatStandardNumber(5e-10, 3, { expLength: 1 })).to.equal('0');
            });
        });

        describe('method "formatStandardNumberToWidth"', function () {
            it('should exist', function () {
                expect(formatter).to.respondTo('formatStandardNumberToWidth');
            });
            var font = new Font('times', 10, false, false);
            it('should format positive numbers in decimal notation and round to pixel width', function () {
                var w1 = font.getTextWidth('0,12345'), w2 = font.getTextWidth('0,1235'), w3 = font.getTextWidth('0,123'),
                    w4 = font.getTextWidth('0,12'), w5 = font.getTextWidth('0,1'), w6 = font.getTextWidth('0'),
                    n = 0.12345;
                expect(formatter.formatStandardNumberToWidth(0, 10, font, w1)).to.deep.equal({ text: '0', scientific: false, width: w6 });
                expect(formatter.formatStandardNumberToWidth(n, 10, font, w1)).to.deep.equal({ text: '0,12345', scientific: false, width: w1 });
                expect(formatter.formatStandardNumberToWidth(n, 10, font, w1 - 0.1)).to.deep.equal({ text: '0,1235', scientific: false, width: w2 });
                expect(formatter.formatStandardNumberToWidth(n, 10, font, w2)).to.have.a.property('text', '0,1235');
                expect(formatter.formatStandardNumberToWidth(n, 10, font, w2 - 0.1)).to.have.a.property('text', '0,123');
                expect(formatter.formatStandardNumberToWidth(n, 10, font, w3)).to.have.a.property('text', '0,123');
                expect(formatter.formatStandardNumberToWidth(n, 10, font, w3 - 0.1)).to.have.a.property('text', '0,12');
                expect(formatter.formatStandardNumberToWidth(n, 10, font, w4)).to.have.a.property('text', '0,12');
                expect(formatter.formatStandardNumberToWidth(n, 10, font, w4 - 0.1)).to.have.a.property('text', '0,1');
                expect(formatter.formatStandardNumberToWidth(n, 10, font, w5)).to.have.a.property('text', '0,1');
                expect(formatter.formatStandardNumberToWidth(n, 10, font, w5 - 0.1)).to.have.a.property('text', '0');
                expect(formatter.formatStandardNumberToWidth(n, 10, font, w6)).to.have.a.property('text', '0');
                expect(formatter.formatStandardNumberToWidth(n, 10, font, w6 - 0.1)).to.equal(null);
                expect(formatter.formatStandardNumberToWidth(n, 5, font, w1)).to.have.a.property('text', '0,123');
                expect(formatter.formatStandardNumberToWidth(n, 5, font, w2)).to.have.a.property('text', '0,123');
                expect(formatter.formatStandardNumberToWidth(n, 5, font, w3)).to.have.a.property('text', '0,123');
                expect(formatter.formatStandardNumberToWidth(n, 5, font, w4)).to.have.a.property('text', '0,12');
                expect(formatter.formatStandardNumberToWidth(n, 5, font, w5)).to.have.a.property('text', '0,1');
                expect(formatter.formatStandardNumberToWidth(n, 5, font, w6)).to.have.a.property('text', '0');
            });
            it('should format negative numbers in decimal notation and round to pixel width', function () {
                var w1 = font.getTextWidth('-0,12345'), w2 = font.getTextWidth('-0,1235'), w3 = font.getTextWidth('-0,123'),
                    w4 = font.getTextWidth('-0,12'), w5 = font.getTextWidth('-0,1'), w6 = font.getTextWidth('-0'),
                    n = -0.12345;
                expect(formatter.formatStandardNumberToWidth(n, 10, font, w1)).to.deep.equal({ text: '-0,12345', scientific: false, width: w1 });
                expect(formatter.formatStandardNumberToWidth(n, 10, font, w1 - 0.1)).to.deep.equal({ text: '-0,1235', scientific: false, width: w2 });
                expect(formatter.formatStandardNumberToWidth(n, 10, font, w2)).to.have.a.property('text', '-0,1235');
                expect(formatter.formatStandardNumberToWidth(n, 10, font, w2 - 0.1)).to.have.a.property('text', '-0,123');
                expect(formatter.formatStandardNumberToWidth(n, 10, font, w3)).to.have.a.property('text', '-0,123');
                expect(formatter.formatStandardNumberToWidth(n, 10, font, w3 - 0.1)).to.have.a.property('text', '-0,12');
                expect(formatter.formatStandardNumberToWidth(n, 10, font, w4)).to.have.a.property('text', '-0,12');
                expect(formatter.formatStandardNumberToWidth(n, 10, font, w4 - 0.1)).to.have.a.property('text', '-0,1');
                expect(formatter.formatStandardNumberToWidth(n, 10, font, w5)).to.have.a.property('text', '-0,1');
                expect(formatter.formatStandardNumberToWidth(n, 10, font, w5 - 0.1)).to.have.a.property('text', '-0');
                expect(formatter.formatStandardNumberToWidth(n, 10, font, w6)).to.have.a.property('text', '-0');
                expect(formatter.formatStandardNumberToWidth(n, 10, font, w6 - 0.1)).to.equal(null);
                expect(formatter.formatStandardNumberToWidth(n, 5, font, w1)).to.have.a.property('text', '-0,123');
                expect(formatter.formatStandardNumberToWidth(n, 5, font, w2)).to.have.a.property('text', '-0,123');
                expect(formatter.formatStandardNumberToWidth(n, 5, font, w3)).to.have.a.property('text', '-0,123');
                expect(formatter.formatStandardNumberToWidth(n, 5, font, w4)).to.have.a.property('text', '-0,12');
                expect(formatter.formatStandardNumberToWidth(n, 5, font, w5)).to.have.a.property('text', '-0,1');
                expect(formatter.formatStandardNumberToWidth(n, 5, font, w6)).to.have.a.property('text', '-0');
            });
            it('should format positive numbers in scientific notation and round to pixel width', function () {
                var w1 = font.getTextWidth('0,00000345'), w2 = font.getTextWidth('3,45E-06'),
                    w3 = font.getTextWidth('3,5E-06'), w4 = font.getTextWidth('3E-06'), w5 = font.getTextWidth('0'),
                    n = 0.00000345;
                expect(formatter.formatStandardNumberToWidth(n, 10, font, w1)).to.deep.equal({ text: '0,00000345', scientific: false, width: w1 });
                expect(formatter.formatStandardNumberToWidth(n, 10, font, w1 - 0.1)).to.deep.equal({ text: '3,45E-06', scientific: true, width: w2 });
                expect(formatter.formatStandardNumberToWidth(n, 10, font, w2)).to.have.a.property('text', '3,45E-06');
                expect(formatter.formatStandardNumberToWidth(n, 10, font, w2 - 0.1)).to.have.a.property('text', '3,5E-06');
                expect(formatter.formatStandardNumberToWidth(n, 10, font, w3)).to.have.a.property('text', '3,5E-06');
                expect(formatter.formatStandardNumberToWidth(n, 10, font, w3 - 0.1)).to.have.a.property('text', '3E-06');
                expect(formatter.formatStandardNumberToWidth(n, 10, font, w4)).to.have.a.property('text', '3E-06');
                expect(formatter.formatStandardNumberToWidth(n, 10, font, w4 - 0.1)).to.deep.equal({ text: '0', scientific: false, width: w5 });
                expect(formatter.formatStandardNumberToWidth(n, 10, font, w5)).to.have.a.property('text', '0');
                expect(formatter.formatStandardNumberToWidth(n, 10, font, w5 - 0.1)).to.equal(null);
            });
            it('should format negative numbers in scientific notation and round to pixel width', function () {
                var w1 = font.getTextWidth('-0,00000345'), w2 = font.getTextWidth('-3,45E-06'),
                    w3 = font.getTextWidth('-3,5E-06'), w4 = font.getTextWidth('-3E-06'), w5 = font.getTextWidth('-0'),
                    n = -0.00000345;
                expect(formatter.formatStandardNumberToWidth(n, 10, font, w1)).to.have.a.property('text', '-0,00000345');
                expect(formatter.formatStandardNumberToWidth(n, 10, font, w1 - 0.1)).to.have.a.property('text', '-3,45E-06');
                expect(formatter.formatStandardNumberToWidth(n, 10, font, w2)).to.have.a.property('text', '-3,45E-06');
                expect(formatter.formatStandardNumberToWidth(n, 10, font, w2 - 0.1)).to.have.a.property('text', '-3,5E-06');
                expect(formatter.formatStandardNumberToWidth(n, 10, font, w3)).to.have.a.property('text', '-3,5E-06');
                expect(formatter.formatStandardNumberToWidth(n, 10, font, w3 - 0.1)).to.have.a.property('text', '-3E-06');
                expect(formatter.formatStandardNumberToWidth(n, 10, font, w4)).to.have.a.property('text', '-3E-06');
                expect(formatter.formatStandardNumberToWidth(n, 10, font, w4 - 0.1)).to.have.a.property('text', '-0');
                expect(formatter.formatStandardNumberToWidth(n, 10, font, w5)).to.have.a.property('text', '-0');
                expect(formatter.formatStandardNumberToWidth(n, 10, font, w5 - 0.1)).to.equal(null);
            });
            it('should prefer scientific notation for large numbers not fitting into the pixel width', function () {
                var w1 = font.getTextWidth('10000000000'), w2 = font.getTextWidth('1E+10');
                expect(formatter.formatStandardNumberToWidth(1e10, 11, font, w1)).to.have.a.property('text', '10000000000');
                expect(formatter.formatStandardNumberToWidth(1e10, 11, font, w1 - 0.1)).to.have.a.property('text', '1E+10');
                expect(formatter.formatStandardNumberToWidth(1e10, 11, font, w2)).to.have.a.property('text', '1E+10');
                expect(formatter.formatStandardNumberToWidth(1e10, 11, font, w2 - 0.1)).to.equal(null);
            });
            it('should use scientific notation for specific edge cases', function () {
                expect(formatter.formatStandardNumberToWidth(5e-100, 11, font, font.getTextWidth('5E-100'))).to.have.a.property('text', '5E-100');
                expect(formatter.formatStandardNumberToWidth(5e-100, 11, font, font.getTextWidth('1E-99'))).to.have.a.property('text', '1E-99');
                expect(formatter.formatStandardNumberToWidth(5e-100, 11, font, font.getTextWidth('1E-99') - 0.1)).to.have.a.property('text', '0');
                expect(formatter.formatStandardNumberToWidth(5e-10, 11, font, font.getTextWidth('1E-9'), { expLength: 1 })).to.have.a.property('text', '1E-9');
                expect(formatter.formatStandardNumberToWidth(5e-10, 11, font, font.getTextWidth('1E-9'))).to.have.a.property('text', '0');
            });
        });

        describe('method "formatValue"', function () {
            it('should exist', function () {
                expect(formatter).to.respondTo('formatValue');
            });
        });
    });

    // ========================================================================
});
