/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define([
    'io.ox/office/tk/keycodes'
], function (KeyCodes) {

    'use strict';

    // static class KeyCodes ==================================================

    describe('Toolkit module KeyCodes', function () {

        it('should exist', function () {
            expect(KeyCodes).to.be.an('object');
        });

        // private helpers ----------------------------------------------------

        function checkKeyCodes(keyCodes) {
            keyCodes.split(' ').forEach(function (key) {
                expect(KeyCodes).to.have.a.property(key);
                expect(KeyCodes[key]).to.be.a('number');
            });
        }

        // constants ----------------------------------------------------------

        describe('constants for control keys', function () {
            it('should exist', function () {
                checkKeyCodes('BACKSPACE TAB ENTER BREAK ESCAPE SPACE PAGE_UP PAGE_DOWN END HOME LEFT_ARROW UP_ARROW RIGHT_ARROW DOWN_ARROW PRINT INSERT DELETE');
            });
        });

        describe('constants for modifier keys', function () {
            it('should exist', function () {
                checkKeyCodes('SHIFT CONTROL ALT CAPS_LOCK LEFT_WINDOWS LEFT_COMMAND RIGHT_WINDOWS SELECT RIGHT_COMMAND NUM_LOCK SCROLL_LOCK');
            });
        });

        describe('constants for alpha-numeric keys', function () {
            it('should exist', function () {
                checkKeyCodes('0 1 2 3 4 5 6 7 8 9 A B C D E F G H I J K L M N O P Q R S T U V W X Y Z');
            });
        });

        describe('constants for numeric pad keys', function () {
            it('should exist', function () {
                checkKeyCodes('NUM_0 NUM_1 NUM_2 NUM_3 NUM_4 NUM_5 NUM_6 NUM_7 NUM_8 NUM_9 NUM_MULTIPLY NUM_PLUS NUM_MINUS NUM_POINT NUM_DIVIDE');
            });
        });

        describe('constants for function keys', function () {
            it('should exist', function () {
                checkKeyCodes('F1 F2 F3 F4 F5 F6 F7 F8 F9 F10 F11 F12');
            });
        });

        describe('constants for special Mozilla key codes', function () {
            it('should exist', function () {
                checkKeyCodes('MOZ_SEMICOLON MOZ_OPEN_ANGLE MOZ_EQUAL_SIGN MOZ_COMMAND');
            });
        });

        describe('constants for IME key codes', function () {
            it('should exist', function () {
                checkKeyCodes('IME_INPUT');
            });
        });

        // static methods -----------------------------------------------------

        describe('method "hasModifierKeys"', function () {
            it('should exist', function () {
                expect(KeyCodes).itself.to.respondTo('hasModifierKeys');
            });
            it('should return the correct state', function () {
                expect(KeyCodes.hasModifierKeys({})).to.equal(false);
                expect(KeyCodes.hasModifierKeys({ shiftKey: false })).to.equal(false);
                expect(KeyCodes.hasModifierKeys({ altKey: false })).to.equal(false);
                expect(KeyCodes.hasModifierKeys({ ctrlKey: false })).to.equal(false);
                expect(KeyCodes.hasModifierKeys({ metaKey: false })).to.equal(false);
                expect(KeyCodes.hasModifierKeys({ shiftKey: true })).to.equal(true);
                expect(KeyCodes.hasModifierKeys({ altKey: true })).to.equal(true);
                expect(KeyCodes.hasModifierKeys({ ctrlKey: true })).to.equal(true);
                expect(KeyCodes.hasModifierKeys({ metaKey: true })).to.equal(true);
                expect(KeyCodes.hasModifierKeys({ shiftKey: false, altKey: false, ctrlKey: false, metaKey: false })).to.equal(false);
                expect(KeyCodes.hasModifierKeys({ shiftKey: true, altKey: false, ctrlKey: false, metaKey: false })).to.equal(true);
                expect(KeyCodes.hasModifierKeys({ shiftKey: false, altKey: true, ctrlKey: false, metaKey: false })).to.equal(true);
                expect(KeyCodes.hasModifierKeys({ shiftKey: false, altKey: false, ctrlKey: true, metaKey: false })).to.equal(true);
                expect(KeyCodes.hasModifierKeys({ shiftKey: false, altKey: false, ctrlKey: false, metaKey: true })).to.equal(true);
                expect(KeyCodes.hasModifierKeys({ shiftKey: true, altKey: true, ctrlKey: true, metaKey: true })).to.equal(true);
            });
        });

        describe('method "matchModifierKeys"', function () {
            it('should exist', function () {
                expect(KeyCodes).itself.to.respondTo('matchModifierKeys');
            });
            it('should detect SHIFT key', function () {
                expect(KeyCodes.matchModifierKeys({})).to.equal(true);
                expect(KeyCodes.matchModifierKeys({ shiftKey: false })).to.equal(true);
                expect(KeyCodes.matchModifierKeys({ shiftKey: true })).to.equal(false);
                expect(KeyCodes.matchModifierKeys({},                  { shift: false })).to.equal(true);
                expect(KeyCodes.matchModifierKeys({ shiftKey: false }, { shift: false })).to.equal(true);
                expect(KeyCodes.matchModifierKeys({ shiftKey: true },  { shift: false })).to.equal(false);
                expect(KeyCodes.matchModifierKeys({},                  { shift: true })).to.equal(false);
                expect(KeyCodes.matchModifierKeys({ shiftKey: false }, { shift: true })).to.equal(false);
                expect(KeyCodes.matchModifierKeys({ shiftKey: true },  { shift: true })).to.equal(true);
                expect(KeyCodes.matchModifierKeys({},                  { shift: null })).to.equal(true);
                expect(KeyCodes.matchModifierKeys({ shiftKey: false }, { shift: null })).to.equal(true);
                expect(KeyCodes.matchModifierKeys({ shiftKey: true },  { shift: null })).to.equal(true);
            });
            it('should detect ALT key', function () {
                expect(KeyCodes.matchModifierKeys({})).to.equal(true);
                expect(KeyCodes.matchModifierKeys({ altKey: false })).to.equal(true);
                expect(KeyCodes.matchModifierKeys({ altKey: true })).to.equal(false);
                expect(KeyCodes.matchModifierKeys({},                { alt: false })).to.equal(true);
                expect(KeyCodes.matchModifierKeys({ altKey: false }, { alt: false })).to.equal(true);
                expect(KeyCodes.matchModifierKeys({ altKey: true },  { alt: false })).to.equal(false);
                expect(KeyCodes.matchModifierKeys({},                { alt: true })).to.equal(false);
                expect(KeyCodes.matchModifierKeys({ altKey: false }, { alt: true })).to.equal(false);
                expect(KeyCodes.matchModifierKeys({ altKey: true },  { alt: true })).to.equal(true);
                expect(KeyCodes.matchModifierKeys({},                { alt: null })).to.equal(true);
                expect(KeyCodes.matchModifierKeys({ altKey: false }, { alt: null })).to.equal(true);
                expect(KeyCodes.matchModifierKeys({ altKey: true },  { alt: null })).to.equal(true);
            });
            it('should detect CTRL key', function () {
                expect(KeyCodes.matchModifierKeys({})).to.equal(true);
                expect(KeyCodes.matchModifierKeys({ ctrlKey: false })).to.equal(true);
                expect(KeyCodes.matchModifierKeys({ ctrlKey: true })).to.equal(false);
                expect(KeyCodes.matchModifierKeys({},                 { ctrl: false })).to.equal(true);
                expect(KeyCodes.matchModifierKeys({ ctrlKey: false }, { ctrl: false })).to.equal(true);
                expect(KeyCodes.matchModifierKeys({ ctrlKey: true },  { ctrl: false })).to.equal(false);
                expect(KeyCodes.matchModifierKeys({},                 { ctrl: true })).to.equal(false);
                expect(KeyCodes.matchModifierKeys({ ctrlKey: false }, { ctrl: true })).to.equal(false);
                expect(KeyCodes.matchModifierKeys({ ctrlKey: true },  { ctrl: true })).to.equal(true);
                expect(KeyCodes.matchModifierKeys({},                 { ctrl: null })).to.equal(true);
                expect(KeyCodes.matchModifierKeys({ ctrlKey: false }, { ctrl: null })).to.equal(true);
                expect(KeyCodes.matchModifierKeys({ ctrlKey: true },  { ctrl: null })).to.equal(true);
            });
            it('should detect META key', function () {
                expect(KeyCodes.matchModifierKeys({})).to.equal(true);
                expect(KeyCodes.matchModifierKeys({ metaKey: false })).to.equal(true);
                expect(KeyCodes.matchModifierKeys({ metaKey: true })).to.equal(false);
                expect(KeyCodes.matchModifierKeys({},                 { meta: false })).to.equal(true);
                expect(KeyCodes.matchModifierKeys({ metaKey: false }, { meta: false })).to.equal(true);
                expect(KeyCodes.matchModifierKeys({ metaKey: true },  { meta: false })).to.equal(false);
                expect(KeyCodes.matchModifierKeys({},                 { meta: true })).to.equal(false);
                expect(KeyCodes.matchModifierKeys({ metaKey: false }, { meta: true })).to.equal(false);
                expect(KeyCodes.matchModifierKeys({ metaKey: true },  { meta: true })).to.equal(true);
                expect(KeyCodes.matchModifierKeys({},                 { meta: null })).to.equal(true);
                expect(KeyCodes.matchModifierKeys({ metaKey: false }, { meta: null })).to.equal(true);
                expect(KeyCodes.matchModifierKeys({ metaKey: true },  { meta: null })).to.equal(true);
            });
            it('should detect ALT or META key', function () {
                expect(KeyCodes.matchModifierKeys({}, { altOrMeta: true })).to.equal(false);
                expect(KeyCodes.matchModifierKeys({ shiftKey: true }, { altOrMeta: true })).to.equal(false);
                expect(KeyCodes.matchModifierKeys({ altKey: true }, { altOrMeta: true })).to.equal(true);
                expect(KeyCodes.matchModifierKeys({ ctrlKey: true }, { altOrMeta: true })).to.equal(false);
                expect(KeyCodes.matchModifierKeys({ metaKey: true }, { altOrMeta: true })).to.equal(true);
                expect(KeyCodes.matchModifierKeys({ altKey: true, metaKey: true }, { altOrMeta: true })).to.equal(false);
            });
            it('should detect CTRL or META key', function () {
                expect(KeyCodes.matchModifierKeys({}, { ctrlOrMeta: true })).to.equal(false);
                expect(KeyCodes.matchModifierKeys({ shiftKey: true }, { ctrlOrMeta: true })).to.equal(false);
                expect(KeyCodes.matchModifierKeys({ altKey: true }, { ctrlOrMeta: true })).to.equal(false);
                expect(KeyCodes.matchModifierKeys({ ctrlKey: true }, { ctrlOrMeta: true })).to.equal(true);
                expect(KeyCodes.matchModifierKeys({ metaKey: true }, { ctrlOrMeta: true })).to.equal(true);
                expect(KeyCodes.matchModifierKeys({ ctrlKey: true, metaKey: true }, { ctrlOrMeta: true })).to.equal(false);
            });
        });

        describe('method "matchKeyCode"', function () {
            it('should exist', function () {
                expect(KeyCodes).itself.to.respondTo('matchKeyCode');
            });
            it('should match key code as number', function () {
                expect(KeyCodes.matchKeyCode({ keyCode: 9 }, 9)).to.equal(true);
                expect(KeyCodes.matchKeyCode({ keyCode: 57 }, 57)).to.equal(true);
                expect(KeyCodes.matchKeyCode({ keyCode: 9 }, 57)).to.equal(false);
            });
            it('should match key code as string', function () {
                expect(KeyCodes.matchKeyCode({ keyCode: 9 }, 'TAB')).to.equal(true);
                expect(KeyCodes.matchKeyCode({ keyCode: 57 }, '9')).to.equal(true);
                expect(KeyCodes.matchKeyCode({ keyCode: 9 }, '9')).to.equal(false);
            });
            it('should match modifier keys', function () {
                expect(KeyCodes.matchKeyCode({ keyCode: 9 }, 'TAB', { shift: true })).to.equal(false);
                expect(KeyCodes.matchKeyCode({ keyCode: 9, shiftKey: true }, 'TAB', { shift: true })).to.equal(true);
                expect(KeyCodes.matchKeyCode({ keyCode: 9 }, 'TAB', { shift: null })).to.equal(true);
                expect(KeyCodes.matchKeyCode({ keyCode: 9, shiftKey: true }, 'TAB', { shift: null })).to.equal(true);
                expect(KeyCodes.matchKeyCode({ keyCode: 9 }, 'TAB', { alt: true })).to.equal(false);
                expect(KeyCodes.matchKeyCode({ keyCode: 9, altKey: true }, 'TAB', { alt: true })).to.equal(true);
                expect(KeyCodes.matchKeyCode({ keyCode: 9 }, 'TAB', { alt: null })).to.equal(true);
                expect(KeyCodes.matchKeyCode({ keyCode: 9, altKey: true }, 'TAB', { alt: null })).to.equal(true);
                expect(KeyCodes.matchKeyCode({ keyCode: 9 }, 'TAB', { ctrl: true })).to.equal(false);
                expect(KeyCodes.matchKeyCode({ keyCode: 9, ctrlKey: true }, 'TAB', { ctrl: true })).to.equal(true);
                expect(KeyCodes.matchKeyCode({ keyCode: 9 }, 'TAB', { ctrl: null })).to.equal(true);
                expect(KeyCodes.matchKeyCode({ keyCode: 9, ctrlKey: true }, 'TAB', { ctrl: null })).to.equal(true);
                expect(KeyCodes.matchKeyCode({ keyCode: 9 }, 'TAB', { meta: true })).to.equal(false);
                expect(KeyCodes.matchKeyCode({ keyCode: 9, metaKey: true }, 'TAB', { meta: true })).to.equal(true);
                expect(KeyCodes.matchKeyCode({ keyCode: 9 }, 'TAB', { meta: null })).to.equal(true);
                expect(KeyCodes.matchKeyCode({ keyCode: 9, metaKey: true }, 'TAB', { meta: null })).to.equal(true);
            });
        });
    });

    // ========================================================================
});
