/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 * @author Michael Nimz <michael.nimz@open-xchange.com>
 */

define([
    'io.ox/office/spreadsheet/utils/errorcode',
    'io.ox/office/spreadsheet/model/formula/formulautils',
    'io.ox/office/spreadsheet/model/formula/formulacontext',
    'io.ox/office/spreadsheet/model/formula/matrix',
    'io.ox/office/spreadsheet/model/formula/impl/matrixfuncs'
], function (ErrorCode, FormulaUtils, FormulaContext, Matrix, MatrixFuncs) {

    'use strict';

    // initialize the formula context
    var context = null;
    before(function (done) {
        ox.test.spreadsheet.createApp('ooxml').done(function (app) {
            context = new FormulaContext(app.getModel());
            done();
        });
    });

    // module MatrixFuncs =====================================================

    describe('Spreadsheet module MatrixFuncs', function () {

        it('should exist', function () {
            expect(MatrixFuncs).to.be.an('object');
        });

        // private helpers ----------------------------------------------------

        function m(values) { return new Matrix(values); }

        function matrixMatcher(expected) {
            expected = m(expected).array;
            return function (result) {
                return _.isEqual(expected, result.array);
            };
        }

        // function implementations -------------------------------------------

        describe('function "MDETERM"', function () {
            var MDETERM = MatrixFuncs.MDETERM;
            it('should exist', function () {
                expect(MDETERM).to.be.an('object');
            });
            it('should be implemented', function () {
                expect(MDETERM).itself.to.respondTo('resolve');
            });
            it('should return the matrix determinant', function () {
                expect(MDETERM.resolve.call(context, m([[1]]))).to.equal(1);
                expect(MDETERM.resolve.call(context, m([[3, 6], [1, 1]]))).to.equal(-3);
                expect(MDETERM.resolve.call(context, m([[0, 1, 2], [3, 4, 5], [6, 7, 8]]))).to.equal(0);
                expect(MDETERM.resolve.call(context, m([[3, 6, 1], [1, 1, 0], [3, 10, 2]]))).to.equal(1);
                expect(MDETERM.resolve.call(context, m([[-4, -2.5, 3], [5, 6, 4], [9, 10, -9]]))).to.equal(161.5);
                expect(MDETERM.resolve.call(context, m([[1, 3, 8, 5], [1, 3, 6, 1], [1, 1, 1, 0], [7, 3, 10, 2]]))).to.equal(88);
                expect(MDETERM.resolve.call(context, m([[1, 3, 8, 5, 1], [1, 3, 6, 1, 1], [1, 1, 1, 0, 1], [7, 3, 10, 2, 1], [1, 1, 1, 1, 1]]))).to.equal(-24);

                expect(MDETERM.resolve.bind(context, m([[1, 1], [1, 1], [1, 1]]))).to['throw'](ErrorCode).that.equals(ErrorCode.NUM);
            });
        });

        describe('function "MINVERSE"', function () {
            var MINVERSE = MatrixFuncs.MINVERSE;
            it('should exist', function () {
                expect(MINVERSE).to.be.an('object');
            });
            it('should be implemented', function () {
                expect(MINVERSE).itself.to.respondTo('resolve');
            });

            it('should return the inverse matrix', function () {

                expect(MINVERSE.resolve.call(context, m([[1, 0, 0], [0, 1, 0], [0, 0, 1]]))).to.satisfy(matrixMatcher([[1, 0, 0], [0, 1, 0], [0, 0, 1]]));
                expect(MINVERSE.resolve.call(context, m([[1, 2, 1], [3, 4, -1], [0, 2, 0]]))).to.satisfy(matrixMatcher([[0.25, 0.25, -0.75], [0, 0, 0.5], [0.75, -0.25, -0.25]]));

                expect(MINVERSE.resolve.bind(context, m([[1, 1], [1, 1], [1, 1]]))).to['throw'](ErrorCode).that.equals(ErrorCode.VALUE);
                expect(MINVERSE.resolve.bind(context, m([[1, 0, 0], [0, 1, 0], [0, 0, 0]]))).to['throw'](ErrorCode).that.equals(ErrorCode.NUM);
            });
        });

        describe('function "MMULT"', function () {
            var MMULT = MatrixFuncs.MMULT;
            it('should exist', function () {
                expect(MMULT).to.be.an('object');
            });
            it('should be implemented', function () {
                expect(MMULT).itself.to.respondTo('resolve');
            });
            it('should return the matrix product', function () {
                expect(MMULT.resolve.call(context, m([[2]]), m([[3]]))).to.satisfy(matrixMatcher([[6]]));
                expect(MMULT.resolve.call(context, m([[2]]), m([[3, 4]]))).to.satisfy(matrixMatcher([[6, 8]]));
                expect(MMULT.resolve.call(context, m([[2], [3]]), m([[4]]))).to.satisfy(matrixMatcher([[8], [12]]));
                expect(MMULT.resolve.call(context, m([[2, 3]]), m([[4], [5]]))).to.satisfy(matrixMatcher([[23]]));
                expect(MMULT.resolve.call(context, m([[2], [3]]), m([[4, 5]]))).to.satisfy(matrixMatcher([[8, 10], [12, 15]]));
                expect(MMULT.resolve.call(context, m([[2, 3], [4, 5]]), m([[6, 7], [8, 9]]))).to.satisfy(matrixMatcher([[36, 41], [64, 73]]));
            });
            it('should throw the #VALUE! error, if matrix sizes do not match', function () {
                expect(MMULT.resolve.bind(null, m([[2]]), m([[3], [4]]))).to['throw'](ErrorCode).that.equals(ErrorCode.VALUE);
                expect(MMULT.resolve.bind(null, m([[2, 3]]), m([[4, 5]]))).to['throw'](ErrorCode).that.equals(ErrorCode.VALUE);
            });
        });

        describe('function "MUNIT"', function () {
            var MUNIT = MatrixFuncs.MUNIT;
            it('should exist', function () {
                expect(MUNIT).to.be.an('object');
            });
            it('should be implemented', function () {
                expect(MUNIT).itself.to.respondTo('resolve');
            });
            it('should return the unit matrix', function () {
                expect(MUNIT.resolve.call(context, 1)).to.satisfy(matrixMatcher([[1]]));
                expect(MUNIT.resolve.call(context, 2)).to.satisfy(matrixMatcher([[1, 0], [0, 1]]));
                expect(MUNIT.resolve.call(context, 3)).to.satisfy(matrixMatcher([[1, 0, 0], [0, 1, 0], [0, 0, 1]]));
                expect(MUNIT.resolve.call(context, 4)).to.satisfy(matrixMatcher([[1, 0, 0, 0], [0, 1, 0, 0], [0, 0, 1, 0], [0, 0, 0, 1]]));
            });
            it('should throw the #VALUE! error, if passed dimension is too small', function () {
                expect(MUNIT.resolve.bind(null, 0)).to['throw'](ErrorCode).that.equals(ErrorCode.VALUE);
                expect(MUNIT.resolve.bind(null, -1)).to['throw'](ErrorCode).that.equals(ErrorCode.VALUE);
            });
            it('should throw the UNSUPPORTED error, if passed dimension is too large', function () {
                var maxSize = Math.max(FormulaUtils.MAX_MATRIX_ROW_COUNT, FormulaUtils.MAX_MATRIX_COL_COUNT);
                expect(MUNIT.resolve.bind(null, maxSize + 1)).to['throw'](ErrorCode).that.equals(FormulaUtils.UNSUPPORTED_ERROR);
            });
        });

        describe('function "SUMX2MY2"', function () {
            var SUMX2MY2 = MatrixFuncs.SUMX2MY2;
            it('should exist', function () {
                expect(SUMX2MY2).to.be.an('object');
            });
            it('should be implemented', function () {
                expect(SUMX2MY2).itself.to.respondTo('resolve');
            });

            it('should return the sum of the difference of squares of corresponding values in two matrices', function () {
                expect(SUMX2MY2.resolve.call(context, m([[2, 3, 9, 1, 8, 7, 5]]), m([[6, 5, 11, 7, 5, 4, 4]]))).to.equal(-55);

                expect(SUMX2MY2.resolve.call(context, m([['hallo', 2, 3, 9, 1, 8, 7]]), m([['echo', 6, 5, 11, 7, 5, 4]]))).to.equal(-64);

                expect(SUMX2MY2.resolve.call(context, m([[2, 'h', 9, 1, 8, 7, 5]]), m([[6, 5, 11, 7, 5, 'e', 4]]))).to.equal(-72);

                expect(SUMX2MY2.resolve.bind(context, m([[1, 1], [1, 1]]), m([[1, 1, 1], [1, 1, 1]]))).to['throw'](ErrorCode).that.equals(ErrorCode.NA);
            });
        });

        describe('function "SUMX2PY2"', function () {
            var SUMX2PY2 = MatrixFuncs.SUMX2PY2;
            it('should exist', function () {
                expect(SUMX2PY2).to.be.an('object');
            });
            it('should be implemented', function () {
                expect(SUMX2PY2).itself.to.respondTo('resolve');
            });

            it('should return the sum of the addend of squares of corresponding values in two matrices', function () {
                expect(SUMX2PY2.resolve.call(context, m([[2, 3, 9, 1, 8, 7, 5]]), m([[6, 5, 11, 7, 5, 4, 4]]))).to.equal(521);

                expect(SUMX2PY2.resolve.bind(context, m([[1, 1], [1, 1]]), m([[1, 1, 1], [1, 1, 1]]))).to['throw'](ErrorCode).that.equals(ErrorCode.NA);
            });
        });

        describe('function "SUMXMY2"', function () {
            var SUMXMY2 = MatrixFuncs.SUMXMY2;
            it('should exist', function () {
                expect(SUMXMY2).to.be.an('object');
            });
            it('should be implemented', function () {
                expect(SUMXMY2).itself.to.respondTo('resolve');
            });

            it('should return the sum of the difference of value in first matrix and squares second matrix', function () {
                expect(SUMXMY2.resolve.call(context, m([[2, 3, 9, 1, 8, 7, 5]]), m([[6, 5, 11, 7, 5, 4, 4]]))).to.equal(79);

                expect(SUMXMY2.resolve.bind(context, m([[1, 1], [1, 1]]), m([[1, 1, 1], [1, 1, 1]]))).to['throw'](ErrorCode).that.equals(ErrorCode.NA);
            });
        });

        describe('function "TRANSPOSE"', function () {
            var TRANSPOSE = MatrixFuncs.TRANSPOSE;
            it('should exist', function () {
                expect(TRANSPOSE).to.be.an('object');
            });
            it('should be implemented', function () {
                expect(TRANSPOSE).itself.to.respondTo('resolve');
            });
            it('should return the transposed matrix', function () {
                expect(TRANSPOSE.resolve.call(context, m([[1]]))).to.satisfy(matrixMatcher([[1]]));
                expect(TRANSPOSE.resolve.call(context, m([[1, 2]]))).to.satisfy(matrixMatcher([[1], [2]]));
                expect(TRANSPOSE.resolve.call(context, m([[1], [2]]))).to.satisfy(matrixMatcher([[1, 2]]));
                expect(TRANSPOSE.resolve.call(context, m([[1, 2], [3, 4]]))).to.satisfy(matrixMatcher([[1, 3], [2, 4]]));
                expect(TRANSPOSE.resolve.call(context, m([[1, 2, 3], [4, 5, 6]]))).to.satisfy(matrixMatcher([[1, 4], [2, 5], [3, 6]]));
            });
        });
    });

    // ========================================================================
});
