/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define([
    'io.ox/office/spreadsheet/view/labels',
    'io.ox/office/spreadsheet/model/formula/impl/operators',
    'io.ox/office/spreadsheet/model/formula/impl/complexfuncs',
    'io.ox/office/spreadsheet/model/formula/impl/conversionfuncs',
    'io.ox/office/spreadsheet/model/formula/impl/databasefuncs',
    'io.ox/office/spreadsheet/model/formula/impl/datetimefuncs',
    'io.ox/office/spreadsheet/model/formula/impl/engineeringfuncs',
    'io.ox/office/spreadsheet/model/formula/impl/financialfuncs',
    'io.ox/office/spreadsheet/model/formula/impl/informationfuncs',
    'io.ox/office/spreadsheet/model/formula/impl/logicalfuncs',
    'io.ox/office/spreadsheet/model/formula/impl/mathfuncs',
    'io.ox/office/spreadsheet/model/formula/impl/matrixfuncs',
    'io.ox/office/spreadsheet/model/formula/impl/referencefuncs',
    'io.ox/office/spreadsheet/model/formula/impl/statisticalfuncs',
    'io.ox/office/spreadsheet/model/formula/impl/textfuncs',
    'io.ox/office/spreadsheet/model/formula/impl/webfuncs'
], function (Labels, Operators) {

    'use strict';

    var functionModules = _.toArray(arguments).slice(2);
    var Functions = _.extend.apply(_, [{}].concat(functionModules));

    // private helpers ========================================================

    // identifiers of all supported file formats
    var FILE_FORMATS = ['ooxml', 'odf'];

    // regular expressions for valid property values
    var FILE_FORMAT_RE = new RegExp('^(' + FILE_FORMATS.join('|') + ')$');
    var CATEGORY_PATTERN = '(' + _.keys(Labels.FUNCTION_CATEGORY_LABELS).join('|') + ')';
    var CATEGORY_RE = new RegExp('^' + CATEGORY_PATTERN + '( ' + CATEGORY_PATTERN + ')*$');
    var SIG_PARAM_PATTERN = '(val(:(num|int|date|day|str|bool|comp|any))?|mat(:(num|str|bool|any))?|ref(:(sheet|single|multi))?|any)';
    var SIGNATURE_RE = new RegExp('^' + SIG_PARAM_PATTERN + '( ' + SIG_PARAM_PATTERN + ')*$');

    function isObject(value) {
        return _.isObject(value) && !_.isArray(value) && !_.isFunction(value);
    }

    function testPropertyNames(desc, re) {
        _.each(desc, function (value, key) {
            expect(key).to.match(re);
            if (isObject(value)) {
                _.each(value, function (v, format) {
                    expect(format).to.match(FILE_FORMAT_RE);
                });
            }
        });
    }

    function resolveDescriptor(desc, format) {
        var result = {};
        _.each(desc, function (value, key) {
            if (isObject(value)) {
                if (format in value) {
                    result[key] = value[format];
                }
            } else {
                result[key] = value;
            }
        });
        return result;
    }

    function testOperatorNameProperty(desc, key, propName, outArray) {
        var names = desc[propName];
        if (_.isString(names)) { names = [names]; }
        expect(names).to.be.an('array');
        expect(names.length).to.be.above(0);
        names.forEach(function (name) {
            expect(name).to.match(/^[^A-Z0-9._]{1,2}$/i);
        });
        outArray = outArray.concat(names);
    }

    function testFunctionNameProperty(desc, key, propName, outArray) {
        if (propName in desc) {
            var names = desc[propName];
            if (_.isNull(names)) { return; }
            if (_.isString(names)) { names = [names]; }
            expect(names).to.be.an('array');
            expect(names.length).to.be.above(0);
            names.forEach(function (name) {
                expect(name).to.match(/^(_xlfn\.)?[A-Z][A-Z0-9]*([._][A-Z0-9]+)*$/);
            });
            outArray = outArray.concat(names);
        } else {
            outArray.push(key);
        }
    }

    // tests ==================================================================

    describe('Spreadsheet operator descriptor module', function () {
        it('should exist', function () {
            expect(Operators).to.be.an('object');
        });
        it('should be a map of descriptors', function () {
            _.each(Operators, function (descriptor) {
                expect(descriptor).to.be.an('object');
            });
        });
    });

    describe('Spreadsheet function descriptor modules', function () {
        it('should exist', function () {
            expect(functionModules).to.be.an('array');
        });
        it('should be maps of descriptors', function () {
            functionModules.forEach(function (descriptors) {
                _.each(descriptors, function (descriptor) {
                    expect(descriptor).to.be.an('object');
                });
            });
        });
    });

    describe('Spreadsheet operator descriptors', function () {
        var keys = _.keys(Operators);
        it('should have valid resource keys', function () {
            keys.forEach(function (key) {
                expect(key).to.match(/^[a-z]+$/);
            });
        });
        it('should have unique resource keys', function () {
            expect(keys.length).to.equal(_.unique(keys).length);
        });
    });

    describe('Spreadsheet function descriptors', function () {
        var keys = _.flatten(_.map(functionModules, _.keys), true);
        it('should have valid resource keys', function () {
            keys.forEach(function (key) {
                expect(key).to.match(/^[A-Z][A-Z0-9]*(\.[A-Z0-9]+)*$/);
            });
        });
        it('should have unique resource keys', function () {
            expect(keys.length).to.equal(_.unique(keys).length);
        });
    });

    FILE_FORMATS.forEach(function (format) {
        var names = [], ceNames = [];
        _.each(Operators, function (descriptor, key) {
            describe('Spreadsheet operator descriptor "' + key + '"', function () {
                it('should be an object', function () {
                    expect(descriptor).to.be.an('object');
                });
                it('should not contain unknown properties', function () {
                    testPropertyNames(descriptor, /^(name|ceName|(min|max)Params|type|signature|resolve)$/);
                });
                it('should contain correct properties', function () {
                    var desc = resolveDescriptor(descriptor, format);
                    testOperatorNameProperty(desc, key, 'name', names);
                    testOperatorNameProperty(desc, key, 'ceName', ceNames);
                    expect(desc).to.have.a.property('minParams').that.is.a('number').within(1, 2);
                    expect(desc).to.have.a.property('maxParams').that.is.a('number').within(desc.minParams, 2);
                    expect(desc).to.have.a.property('type').that.match(/^(val|ref)$/);
                    if ('signature' in desc) { expect(desc.signature).to.match(SIGNATURE_RE); }
                    if ('resolve' in desc) { expect(_.isFunction(desc.resolve) || _.isString(desc.resolve)).to.equal(true); }
                    if (_.isString(desc.resolve)) {
                        expect(desc.resolve).to.not.equal(key);
                        expect(Operators).to.contain.a.property(desc.resolve);
                    }
                });
            });
        });
        describe('Spreadsheet operator descriptor', function () {
            it('names should be unique', function () {
                expect(names.length).to.equal(_.unique(names).length);
                expect(ceNames.length).to.equal(_.unique(ceNames).length);
            });
        });
    });

    FILE_FORMATS.forEach(function (format) {
        var names = [], ceNames = [];
        _.each(Functions, function (descriptor, key) {
            describe('Spreadsheet function descriptor "' + key + '"', function () {
                it('should be an object', function () {
                    expect(descriptor).to.be.an('object');
                });
                it('should not contain unknown properties', function () {
                    testPropertyNames(descriptor, /^(category|name|ceName|hidden|(min|max|repeat)Params|type|volatile|signature|resolve)$/);
                });
                it('should contain correct properties', function () {
                    var desc = resolveDescriptor(descriptor, format);
                    expect(desc).to.have.a.property('category').that.match(CATEGORY_RE);
                    testFunctionNameProperty(desc, key, 'name', names);
                    testFunctionNameProperty(desc, key, 'ceName', ceNames);
                    if ('hidden' in desc) { expect(desc.hidden).to.be.a('boolean'); }
                    expect(desc).to.have.a.property('minParams').that.is.a('number').at.least(0);
                    if ('maxParams' in desc) { expect(desc.maxParams).to.be.a('number').at.least(desc.minParams); }
                    if ('repeatParams' in desc) { expect(desc.repeatParams).to.be.a('number').at.least(1); }
                    expect(!('maxParams' in desc) || !('repeatParams' in desc)).to.equal(true);
                    expect(desc).to.have.a.property('type').that.match(/^(val|mat|ref|any)$/);
                    if ('volatile' in desc) { expect(desc.volatile).to.be.a('boolean'); }
                    if ('signature' in desc) { expect(desc.signature).to.match(SIGNATURE_RE); }
                    if ('resolve' in desc) {
                        expect(_.isFunction(desc.resolve) || _.isString(desc.resolve)).to.equal(true);
                        if (desc.maxParams !== 0) { expect(desc).to.have.a.property('signature'); }
                    }
                    if (_.isString(desc.resolve)) {
                        expect(desc.resolve).to.not.equal(key);
                        expect(Functions).to.contain.a.property(desc.resolve);
                    }
                });
            });
        });
        describe('Spreadsheet function descriptor', function () {
            it('names should be unique', function () {
                expect(names.length).to.equal(_.unique(names).length);
                expect(ceNames.length).to.equal(_.unique(ceNames).length);
            });
        });
    });

    // ========================================================================
});
