/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Michael Nimz <michael.nimz@open-xchange.com>
   @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define([
    'io.ox/office/presentation/model/model',
    'io.ox/office/textframework/utils/dom'
], function (PresentationModel, DOM) {

    'use strict';

    // class PresentationModel ================================================

    describe('Presentation class PresentationModel', function () {

        // private helpers ----------------------------------------------------

        var model = null,
            view = null,
            slideDefaultName = 'slide_1',
            layoutId_1 = 'layout1',
            layoutId_2 = 'layout2',
            masterId_1 = 'master1',
            backgroundColor = { type: 'rgb', value: 'FFCC00' },
            secondDrawingText = 'Hello Second Drawing',
            pasteText = '123',
            clipboardOperations = [{ name: 'insertText', start: [0, 0], text: pasteText }],
            pasteDrawingText1 = '123Hello Second Drawing',
            pasteDrawingText2 = '123Hel123lo Second Drawing',

            // the operations to be applied by the document model
            OPERATIONS = [
                { name: 'setDocumentAttributes', attrs: { page: { width: 33866, height: 19050, orientation: 'landscape' } } },
                { name: 'insertMasterSlide', id: masterId_1 },
                { name: 'insertLayoutSlide', id: layoutId_1, target: masterId_1 },
                { name: 'insertLayoutSlide', id: layoutId_2, target: masterId_1 },
                { name: 'insertSlide', start: [0], target: layoutId_1, attrs: { fill: { type: 'solid', color: { type: 'rgb', value: 'FF0000' } } } },
                { name: 'insertDrawing', start: [0, 0], type: 'shape', attrs: { presentation: { phType: 'ctrTitle' }, drawing: { name: 'Titel 1' } } },
                { name: 'insertParagraph', start: [0, 0, 0] },
                { name: 'insertText', text: 'Hello World', start: [0, 0, 0, 0] },
                { name: 'insertDrawing', start: [0, 1], type: 'shape', attrs: { presentation: { phType: 'subTitle', phIndex: 1 }, drawing: { name: 'Untertitel 2' } } },
                { name: 'insertParagraph', start: [0, 1, 0] },
                { name: 'insertText', text: secondDrawingText, start: [0, 1, 0, 0] }
            ];

        before(function (done) {
            ox.test.presentation.createApp('ooxml', OPERATIONS).done(function (app) {
                app.isOperationsBlockActive = function () {};
                model = app.getModel();
                model.getClipboardOperations = function () { return clipboardOperations; };
                model.hasInternalClipboard = function () { return true; };
                model.checkSetClipboardPasteInProgress = function () { return false; };
                model.leaveAsyncBusy = function () {};
                view = app.getView();
                view.clearVisibleDrawingAnchor = function () {};
                view.isSearchActive = function () {};
                view.enterBusy = function () {};
                done();
            });
        });

        // existence check ----------------------------------------------------

        it('should exist', function () {
            expect(PresentationModel).to.be.a('function');
        });

        // public methods -----------------------------------------------------

        describe('method "isStandardSlideId"', function () {
            it('should exist', function () {
                expect(model).to.respondTo('isStandardSlideId');
            });
            it('should return whether the specified ID is the ID of a normal (standard) slide', function () {
                expect(model.isStandardSlideId(layoutId_1)).to.equal(false);
                expect(model.isStandardSlideId(masterId_1)).to.equal(false);
                expect(model.isStandardSlideId(slideDefaultName)).to.equal(true);
            });
        });

        describe('method "isLayoutSlideId"', function () {
            it('should exist', function () {
                expect(model).to.respondTo('isLayoutSlideId');
            });
            it('should return whether the specified ID is the ID of a layout slide', function () {
                expect(model.isLayoutSlideId(layoutId_1)).to.equal(true);
                expect(model.isLayoutSlideId(masterId_1)).to.equal(false);
                expect(model.isLayoutSlideId(slideDefaultName)).to.equal(false);
            });
        });

        describe('method "isMasterSlideId"', function () {
            it('should exist', function () {
                expect(model).to.respondTo('isMasterSlideId');
            });
            it('should return whether the specified ID is the ID of a master slide', function () {
                expect(model.isMasterSlideId(layoutId_1)).to.equal(false);
                expect(model.isMasterSlideId(masterId_1)).to.equal(true);
                expect(model.isMasterSlideId(slideDefaultName)).to.equal(false);
            });
        });

        describe('method "getStandardSlideCount"', function () {
            it('should exist', function () {
                expect(model).to.respondTo('getStandardSlideCount');
            });
            it('should return the number of normal (standard) slides', function () {
                expect(model.getStandardSlideCount()).to.equal(1);
            });
        });

        describe('method "getMasterSlideCount"', function () {
            it('should exist', function () {
                expect(model).to.respondTo('getMasterSlideCount');
            });
            it('should return the number of master and layout slides', function () {
                expect(model.getMasterSlideCount()).to.equal(3);
            });
        });

        describe('method "isMasterView"', function () {
            it('should exist', function () {
                expect(model).to.respondTo('isMasterView');
            });
            it('should return whether the master/layout view is activated', function () {
                expect(model.isMasterView()).to.equal(false);
            });
        });

        describe('method "getActiveSlideId"', function () {
            it('should exist', function () {
                expect(model).to.respondTo('getActiveSlideId');
            });
            it('should return the ID of the active slide', function () {
                expect(model.getActiveSlideId()).to.equal(slideDefaultName);
            });
        });

        describe('method "getSlideById"', function () {
            it('should exist', function () {
                expect(model).to.respondTo('getSlideById');
            });
            it('should return a slide', function () {
                expect(model.getSlideById('notExistent')).to.equal(null);
                expect(model.getSlideById(slideDefaultName)).to.be.an['instanceof']($);
                expect(model.getSlideById(slideDefaultName)).to.have.property('hasClass');
                expect(model.getSlideById(slideDefaultName).hasClass('slide')).to.equal(true);
            });

        });

        describe('method "getActiveSlidePosition"', function () {
            it('should exist', function () {
                expect(model).to.respondTo('getActiveSlidePosition');
            });
            it('should return the logical position of the active slide', function () {
                expect(model.getActiveSlidePosition()).to.be.an('array');
                expect(model.getActiveSlidePosition().length).to.equal(1);
                expect(model.getActiveSlidePosition()[0]).to.equal(0);
            });
        });

        describe('method "resetBackground"', function () {
            it('should exist', function () {
                expect(model).to.respondTo('resetBackground');
            });
            it('should remove the slide background', function () {
                expect(model.getSlideById(slideDefaultName).data('attributes').fill.type).to.equal('solid');
                expect(model.getSlideById(slideDefaultName).css('background-color')).to.be.a('string').and.not.to.be.empty;
                model.resetBackground();
                expect(model.getSlideById(slideDefaultName).data('attributes').fill).to.equal(undefined);
                expect(model.getSlideById(slideDefaultName).css('background-color')).to.equal('');
            });
        });

        describe('method "setBackgroundColor"', function () {
            it('should exist', function () {
                expect(model).to.respondTo('setBackgroundColor');
            });
            it('should set the slide background color', function () {
                expect(model.getSlideById(slideDefaultName).data('attributes').fill).to.equal(undefined);
                expect(model.getSlideById(slideDefaultName).css('background-color')).to.equal('');
                model.setBackgroundColor(backgroundColor);
                expect(model.getSlideById(slideDefaultName).data('attributes').fill.type).to.equal('solid');
                expect(model.getSlideById(slideDefaultName).css('background-color')).to.be.a('string').and.not.to.be.empty;
            });
        });

        describe('method "getSlideAttributes"', function () {
            it('should exist', function () {
                expect(model).to.respondTo('getSlideAttributes');
            });
            it('should get the slide attributes without specifying an ID', function () {
                expect(model.getSlideAttributes().fill.type).to.equal('solid');
            });
            it('should get the slide attributes with specifying an ID', function () {
                expect(model.getSlideAttributes(slideDefaultName).fill.type).to.equal('solid');
            });
        });

        describe('method "getVerticalAlignmentMode"', function () {
            it('should exist', function () {
                expect(model).to.respondTo('getVerticalAlignmentMode');
            });
            it('should get the vertical alignment mode of the text inside the selected drawing', function () {
                model.getSelection().setTextSelection([0, 1], [0, 2]);
                expect(model.getSelection().getSelectedDrawing().length).to.equal(1);
                expect(model.getSelection().getSelectedDrawing().text()).to.equal(secondDrawingText);
                expect(_.last(model.getSelection().getStartPosition())).equal(1);
                expect(_.last(model.getSelection().getEndPosition())).equal(2);
                expect(model.getVerticalAlignmentMode()).to.equal('top');
            });
        });

        describe('method "setVerticalAlignmentMode"', function () {
            it('should exist', function () {
                expect(model).to.respondTo('setVerticalAlignmentMode');
            });
            it('should set the vertical alignment mode of the text inside the selected drawing', function () {
                model.getSelection().setTextSelection([0, 1], [0, 2]);
                expect(model.getSelection().getSelectedDrawing().text()).to.equal(secondDrawingText);
                expect(_.last(model.getSelection().getStartPosition())).equal(1);
                expect(_.last(model.getSelection().getEndPosition())).equal(2);
                model.setVerticalAlignmentMode('centered');
                expect(model.getVerticalAlignmentMode()).to.equal('centered');
                model.setVerticalAlignmentMode('bottom');
                expect(model.getVerticalAlignmentMode()).to.equal('bottom');
                model.setVerticalAlignmentMode('top');
                expect(model.getVerticalAlignmentMode()).to.equal('top');
            });
        });

        describe('method "pasteInternalClipboard"', function () {
            var drawing = null;
            it('should exist', function () {
                expect(model).to.respondTo('pasteInternalClipboard');
            });
            it('should paste some text into the text frame that is selected', function () {
                model.getSelection().setTextSelection([0, 1], [0, 2]);
                drawing = model.getSelection().getSelectedDrawing();
                expect(drawing.find(DOM.PARAGRAPH_NODE_SELECTOR).first().text()).to.equal(secondDrawingText);
                model.pasteInternalClipboard();
                model.getSelection().setTextSelection([0, 1], [0, 2]);
                expect(drawing.find(DOM.PARAGRAPH_NODE_SELECTOR).first().text()).to.equal(pasteDrawingText1);
            });
            it('should paste some text into the text frame that contains the selection', function () {
                model.getSelection().setTextSelection([0, 1, 0, 6]);
                model.pasteInternalClipboard();
                expect(drawing.find(DOM.PARAGRAPH_NODE_SELECTOR).first().text()).to.equal(pasteDrawingText2);
            });
            it('should paste some text into a new created text frame, if the browser selection is in the clipboard', function () {
                var followingDrawings = drawing.nextAll('div.drawing').length;
                expect(drawing.nextAll('div.drawing').length).to.equal(followingDrawings);
                expect(model.getSelection().getClipboardNode().css('display')).to.equal('none');
                model.getSelection().setTextSelection([0, 0]);
                expect(model.getSelection().getClipboardNode().css('display')).to.equal('block');
                model.pasteInternalClipboard();
                expect(drawing.nextAll('div.drawing').length).to.equal(followingDrawings + 1);
                expect(drawing.nextAll('div.drawing').text()).to.equal(pasteText);
            });
        });

    });
});
