/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/tk/control/group', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/keycodes',
    'io.ox/office/tk/forms',
    'io.ox/office/tk/object/triggerobject',
    'io.ox/office/tk/object/timermixin'
], function (Utils, KeyCodes, Forms, TriggerObject, TimerMixin) {

    'use strict';

    var // all existing groups and their focusable DOM nodes (stored globally  to be able
        // to use a single 'change:focus' listener instead of one listener per group instance)
        focusableNodesMap = {},

        // the unique IDs of the group instances currently focused (as stack, groups may be embedded)
        focusStack = [];

    // global focus handling ==================================================

    function registerFocusGroup(group) {
        focusableNodesMap[group.getUid()] = { group: group, nodes: group.getNode() };
    }

    function registerFocusableNodes(group, nodes) {
        var entry = focusableNodesMap[group.getUid()];
        entry.nodes = entry.nodes.add(nodes);
    }

    function unregisterFocusGroup(group) {
        delete focusableNodesMap[group.getUid()];
        focusStack = _.without(focusStack, group.getUid());
    }

    Utils.on('change:focus', function (event, focusNode) {

        // update CSS focus class, trigger event
        function changeFocusState(group, focused) {
            var type = focused ? 'group:focus' : 'group:blur';
            group.getNode().toggleClass(Forms.FOCUSED_CLASS, focused);
            group.trigger(type);
        }

        // returns whether the focus node is inside one of the focusable nodes
        function containsFocusNode(uid) {
            var focusableNodes = focusableNodesMap[uid].nodes;
            return (focusableNodes.filter(focusNode).length > 0) || (focusableNodes.has(focusNode).length > 0);
        }

        // remove all groups from the stack that are not focused anymore
        while ((focusStack.length > 0) && (!focusNode || !containsFocusNode(_.last(focusStack)))) {
            var uid = focusStack.pop();
            // check that group has not been destroyed already
            if (uid in focusableNodesMap) {
                changeFocusState(focusableNodesMap[uid].group, false);
            }
        }

        // push all groups onto the stack that contain the focus node (and are not yet in the stack)
        if (focusNode) {
            _.chain(focusableNodesMap).keys().difference(focusStack).each(function (uid) {
                if (containsFocusNode(uid)) {
                    focusStack.push(uid);
                    changeFocusState(focusableNodesMap[uid].group, true);
                }
            });
        }
    });

    // class Group ============================================================

    /**
     * Creates a container element used to hold a control. All controls shown
     * in view components must be inserted into such group containers. This is
     * the base class for specialized groups and does not add any specific
     * functionality to the inserted controls.
     *
     * Instances of this class trigger the following events:
     * - 'group:change'
     *      If the control has been activated in a special way depending on the
     *      type of the control group. The event handler receives the new value
     *      of the activated control.
     * - 'group:cancel'
     *      When the focus needs to be returned to the application (e.g. when
     *      the ESCAPE key is pressed, or when a click on a drop-down button
     *      closes the opened drop-down menu).
     * - 'group:beforeshow'
     *      Before the control will be shown (while it is currently hidden).
     * - 'group:show'
     *      After the control has been shown (and it was hidden before).
     * - 'group:beforehide'
     *      Before the control will be hidden (while it is currently visible).
     * - 'group:hide'
     *      After the control has been hidden (and it was visible before).
     * - 'group:enable'
     *      After the control has been enabled or disabled. The event handler
     *      receives the new state.
     * - 'group:resize'
     *      After the size of the root node has been changed. The event handler
     *      receives the new width and height of the root node, in pixels.
     * - 'group:focus'
     *      After the group has been focused, by initially focusing any of its
     *      focusable child nodes. As long as the focus remains inside the
     *      group (even if the focus moves to another DOM node in the group),
     *      no further 'group:focus' event will be triggered.
     * - 'group:blur'
     *      After the group has lost the browser focus, after focusing any
     *      other DOM node outside the group. As long as the focus remains
     *      inside the group (even if the focus moves to another DOM node in
     *      the group), the 'group:blur' event will not be triggered.
     *
     * @constructor
     *
     * @extends TriggerObject
     * @extends TimerMixin
     *
     * @param {Object} [initOptions]
     *  Optional parameters:
     *  @param {String} [initOptions.classes]
     *      Additional CSS classes that will be set at the root DOM node of
     *      this instance.
     *  @param {String} [initOptions.role]
     *      An ARIA role attribute that will be set at the root DOM node of
     *      this instance.
     */
    function Group(initOptions) {

        var // self reference
            self = this,

            // create the group container element
            groupNode = $('<div>').addClass('group'),

            // the last cached size of the root node, used to detect layout changes
            nodeSize = null,

            // the current value of this control group (must be undefined initially, NOT null)
            groupValue,

            // update handler functions
            updateHandlers = [],

            // special options for a drop-down version of this group
            dropDownVersion = Utils.getOption(initOptions, 'dropDownVersion');

        // base constructors --------------------------------------------------

        TriggerObject.call(this);
        TimerMixin.call(this);

        // private methods ----------------------------------------------------

        /**
         * Reads the current outer size of the root node of this group, and
         * triggers a 'group:resize' event if the group is visible and its size
         * has changed. Does nothing if the group is currently hidden.
         */
        function updateNodeSize() {

            // do nothing if the group is not visible, or is inside a hidden container node
            if (!Forms.isDeeplyVisibleNode(groupNode)) { return; }

            // read current node size, notify listeners if size has changed
            var newSize = { width: groupNode.outerWidth(), height: groupNode.outerHeight() };
            if (!_.isEqual(nodeSize, newSize)) {
                nodeSize = newSize;
                self.trigger('group:resize', nodeSize.width, nodeSize.height);
            }
        }

        /**
         * Updates the current value of this group, and invokes the registered
         * update handlers which will update the visual appearance of this
         * group.
         */
        function updateValue(newValue) {

            var // the old value of this group (for event handlers)
                oldValue = groupValue;

            // set new value, invoke the update handlers which refresh the appearance of the group
            groupValue = newValue;
            _.each(updateHandlers, function (updateHandler) {
                updateHandler.call(self, newValue, oldValue);
            });

            // update group size
            self.layout();
        }

        /**
         * Returns an object with focus options for the passed source event,
         * intended to be inserted into a 'group:change' or 'group:cancel'
         * event triggered by this control group.
         *
         * @param {Object} [options]
         *  Optional parameters. See method Group.triggerChange() for details.
         *
         * @returns {Object}
         *  An options object with the following properties:
         *  - {HTMLElement} sourceNode
         *      The target node of the passed event. Defaults to the own root
         *      node, if no source event has been passed.
         *  - {String} sourceType
         *      An abstract type identifier for the source event: 'click' for
         *      regular mouse click events; 'keyboard' for keyboard events
         *      (including click events originating from SPACE or RETURN keys),
         *      or 'custom' for all other events.
         *  - {Boolean} preserveFocus
         *      Whether the passed source event leads to preserving the current
         *      browser focus after performing the action associated to this
         *      control group (usually, after SPACE or TAB keys).
         */
        function getEventOptions(options) {

            var // resulting properties for the event object
                eventOptions = {
                    sourceNode: groupNode[0],
                    sourceType: 'custom',
                    preserveFocus: false
                },
                // the source event object
                sourceEvent = Utils.getOption(options, 'sourceEvent');

            // special handling for specific events
            if (_.isObject(sourceEvent)) {

                // use target of the passed source event as source node
                eventOptions.sourceNode = sourceEvent.currentTarget;

                // handle specific event types
                switch (sourceEvent.type) {
                    case 'tap':
                    case 'click':
                        // Forms.setButtonKeyHandler() triggers 'click' events with 'keyCode' field
                        eventOptions.sourceType = ('keyCode' in sourceEvent) ? 'keyboard' : 'click';
                        // click events from keyboard: keep focus on button after SPACE or TAB key (see Forms.setButtonKeyHandler() method)
                        eventOptions.preserveFocus = (sourceEvent.keyCode === KeyCodes.SPACE) || (sourceEvent.keyCode === KeyCodes.TAB);

                        eventOptions.sourceEventType = sourceEvent.type;
                        break;
                    case 'keydown':
                    case 'keypress':
                    case 'keyup':
                        eventOptions.sourceType = 'keyboard';
                        break;
                }
            }

            // override the 'preserveFocus' option
            eventOptions.preserveFocus = Utils.getBooleanOption(options, 'preserveFocus', eventOptions.preserveFocus);

            return eventOptions;
        }

        // protected methods --------------------------------------------------

        /**
         * Registers the passed update handler function. These handlers will be
         * called from the method Group.setValue() in order of their
         * registration.
         *
         * @attention
         *  Intended to be used by the internal implementations of sub class
         *  constructors. DO NOT CALL from external code!
         *
         * @param {Function} updateHandler
         *  The update handler function. Will be called in the context of this
         *  group. Receives the following parameters:
         *  (1) {Any} newValue
         *      The new value passed to the method Group.setValue().
         *  (2) {Any} oldValue
         *      The old value of the group.
         *
         * @returns {Group}
         *  A reference to this group.
         */
        this.registerUpdateHandler = function (updateHandler) {
            updateHandlers.push(updateHandler);
            return this;
        };

        /**
         * Registers one or more DOM nodes located outside the root node of
         * this group instance, which contain other control nodes that will be
         * included into the focus handling and generation of 'group:focus' and
         * 'group:blur' events. While the browser focus is located inside the
         * passed nodes, this group will not trigger a 'group:blur' event.
         *
         * @attention
         *  Intended to be used by the internal implementations of sub class
         *  constructors. DO NOT CALL from external code!
         *
         * @param {HTMLElement|jQuery} nodes
         *  The DOM node(s) to be added to the internal focus handling.
         *
         * @returns {Group}
         *  A reference to this instance.
         */
        this.registerFocusableNodes = function (nodes) {
            registerFocusableNodes(this, nodes);
            return this;
        };

        // methods ------------------------------------------------------------

        this.clone = function (additionalOptions) {
            return new this.constructor(Utils.extendOptions(initOptions, additionalOptions));
        };

        /**
         * Returns the DOM container element for this group as jQuery object.
         */
        this.getNode = function () {
            return groupNode;
        };

        /**
         * Returns the options that have been passed to the constructor.
         *
         * @returns {Object}
         *  The options passed to the constructor; or an empty object.
         */
        this.getOptions = function () {
            return initOptions || {};
        };

        /**
         * Triggers a 'group:resize' event to all listeners, if the size of the
         * group has been changed due to changed contents.
         *
         * @returns {Group}
         *  A reference to this instance.
         */
        this.layout = function () {
            // notify listeners if the group size has changed
            updateNodeSize();
            return this;
        };

        /**
         * Inserts the passed HTML mark-up into this group (all existing child
         * nodes will be removed). Triggers a 'group:layout' event notifying
         * all listeners about the changed layout of this group.
         *
         * @param {String} markup
         *  The HTML mark-up to be inserted into this group, as string.
         *
         * @returns {Group}
         *  A reference to this group.
         */
        this.setChildMarkup = function (markup) {
            groupNode.html(markup);
            return this.layout();
        };

        /**
         * Inserts the passed DOM elements into this group, and triggers a
         * 'group:layout' event notifying all listeners about the changed
         * layout of this group.
         *
         * @param {HTMLElement|jQuery|String} [...]
         *  The DOM nodes to be inserted into this group. Can be an arbitrary
         *  number of HTML elements, jQuery collections, and/or HTML mark-up as
         *  string.
         *
         * @returns {Group}
         *  A reference to this group.
         */
        this.addChildNodes = function () {
            groupNode.append.apply(groupNode, arguments);
            return this.layout();
        };

        /**
         * Returns whether this group is currently in focused state, either
         * because the browser focus is located inside the group itself, or it
         * is located in any of the additional DOM nodes that have been
         * registered with the method 'Group.registerFocusableNodes()'.
         *
         * @returns {Boolean}
         *  Whether the group currently owns the browser focus.
         */
        this.hasFocus = function () {
            return _.indexOf(focusStack, this.getUid()) >= 0;
        };

        /**
         * Sets the focus to the first control in this group, unless it is
         * already focused.
         *
         * @returns {Group}
         *  A reference to this group.
         */
        this.grabFocus = function () {
            if (!this.hasFocus()) { Forms.grabFocus(groupNode); }
            return this;
        };

        /**
         * Returns whether this control group is configured to be visible via
         * the methods Group.show(), Group.hide(), or Group.toggle().
         */
        this.isVisible = function () {
            return Forms.isVisibleNode(groupNode);
        };

        /**
         * Displays this control group, if it is currently hidden.
         *
         * @returns {Group}
         *  A reference to this group.
         */
        this.show = function () {
            return this.toggle(true);
        };

        /**
         * Hides this control group, if it is currently visible.
         *
         * @returns {Group}
         *  A reference to this group.
         */
        this.hide = function () {
            return this.toggle(false);
        };

        /**
         * Toggles the visibility of this control group.
         *
         * @param {Boolean} [state]
         *  If specified, shows or hides the groups depending on the boolean
         *  value. If omitted, toggles the current visibility of the group.
         *
         * @returns {Group}
         *  A reference to this group.
         */
        this.toggle = function (state) {

            var // whether to show the group instance
                visible = (state === true) || ((state !== false) && this.isVisible());

            if (this.isVisible() !== visible) {
                this.trigger(visible ? 'group:beforeshow' : 'group:beforehide');
                Forms.showNodes(groupNode, visible);
                this.trigger(visible ? 'group:show' : 'group:hide');
                updateNodeSize();
            }
            return this;
        };

        /**
         * Returns whether this control group is enabled.
         */
        this.isEnabled = function () {
            return Forms.isEnabledNode(groupNode);
        };

        /**
         * Enables or disables this control group.
         *
         * @param {Boolean} [state=true]
         *  If omitted or set to true, the group will be enabled. Otherwise,
         *  the group will be disabled.
         *
         * @returns {Group}
         *  A reference to this group.
         */
        this.enable = function (state) {

            var // whether to enable the group instance
                enabled = _.isUndefined(state) || (state === true);

            // enable/disable the entire group node with all its descendants
            if (this.isEnabled() !== enabled) {
                Forms.enableNodes(groupNode, enabled);
                var buttonNodes = groupNode.find(Forms.BUTTON_SELECTOR);
                if (state) { buttonNodes.removeAttr('aria-disabled'); } else { buttonNodes.attr('aria-disabled', true); }
                this.trigger('group:enable', enabled);
            }
            return this;
        };

        /**
         * Returns the current value of this control group.
         *
         * @returns {Any}
         *  The current value of this control group.
         */
        this.getValue = function () {
            return groupValue;
        };

        /**
         * Updates the controls in this group with the specified value, by
         * calling the registered update handlers.
         *
         * @param {Any} value
         *  The new value to be displayed in the controls of this group.
         *
         * @returns {Group}
         *  A reference to this group.
         */
        this.setValue = function (value) {
            updateValue(value);
            return this;
        };

        /**
         * Debounced call of the Group.setValue() method with the current value
         * of this group. Can be used to refresh the appearance of the group
         * after changing its content.
         *
         * @returns {Group}
         *  A reference to this group.
         */
        this.refresh = this.createDebouncedMethod(Utils.NOOP, function () {
            updateValue(groupValue);
        });

        /**
         * Triggers a 'group:change' event at this control group instance, and
         * inserts special focus options into the event object. If the control
         * group is disabled, a 'group:cancel' event will be triggered instead.
         *
         * @param {Any} value
         *  The new value for this group to be set and notified.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {jQuery.Event} [options.sourceEvent]
         *      The source event that has caused the new event. This source
         *      event may influence the focus behavior, (e.g.: browser focus is
         *      kept on a button, if the SPACE key has been used).
         *  @param {Boolean} [options.preserveFocus]
         *      If specified, forces to preserve the browser focus regardless
         *      of the passed source event.
         *
         * @returns {Group}
         *  A reference to this group.
         */
        this.triggerChange = function (value, options) {
            if (this.isEnabled() && !_.isNull(value)) {
                this.setValue(value);
                this.trigger(new $.Event('group:change', getEventOptions(options)), value);
            } else {
                this.triggerCancel(options);
            }
            return this;
        };

        /**
         * Triggers a 'group:cancel' event at this Group instance, and inserts
         * special focus options into the event object, according to the passed
         * source event.
         *
         * @param {Object} [options]
         *  Optional parameters. See method Group.triggerChange() for details.
         *
         * @returns {Group}
         *  A reference to this group.
         */
        this.triggerCancel = function (options) {
            this.trigger(new $.Event('group:cancel', getEventOptions(options)));
            return this;
        };

        /**
         * Needs to be overwritten by controls, if there were some special
         * small version options.
         */
        this.activateSmallVersion = Utils.NOOP;
        this.deactivateSmallVersion = Utils.NOOP;

        /**
         * Shows or hides this control group according to the passed state and
         * the drop-down configuration passed to the constructor.
         *
         * @param {Boolean} shrunkenMode
         *  Whether the container is in shrunken mode. This control group will
         *  only be updated, if the drop-down configuration passed to the
         *  constructor contains a 'visible' option. The control will be shown,
         *  if the flag 'shrunkenMode' equals the 'visible' option, otherwise
         *  it will be hidden.
         *
         * @returns {Group}
         *  A reference to this group.
         */
        this.toggleDropDownMode = function (shrunkenMode) {
            if (_.isObject(dropDownVersion) && _.isBoolean(dropDownVersion.visible)) {
                this.toggle(shrunkenMode === dropDownVersion.visible);
            }
            return this;
        };

        // initialization -----------------------------------------------------

        // register for global focus handling
        registerFocusGroup(this);

        // formatting and tool tip
        groupNode.addClass(Utils.getStringOption(initOptions, 'classes', ''));

        // ARIA role
        groupNode.attr('role', Utils.getStringOption(initOptions, 'role', null, true));

        // hide groups only shown in shrunken mode
        this.toggleDropDownMode(false);

        // log focus and layout events
        Utils.FocusLogger.logEvent(this, 'group:focus group:blur');
        Utils.LayoutLogger.logEvent(this, 'group:resize', 'width', 'height');
        Utils.LayoutLogger.logEvent(this, 'group:beforeshow group:show group:beforehide group:hide');

        // destroy all class members on destruction
        this.registerDestructor(function () {
            unregisterFocusGroup(this);
            groupNode.remove();
            self = initOptions = groupNode = updateHandlers = null;
        });

    } // class Group

    // exports ================================================================

    // derive this class from class TriggerObject
    return TriggerObject.extend({ constructor: Group });

});
