/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/tk/config', [
    'settings!io.ox/office'
], function (settings) {

    'use strict';

    var // the configuration items of the OX Documents modules
        documentsConfig = settings.get('module', {});

    // static class Config ====================================================

    /**
     * Wraps the configuration of the OX Documents module.
     */
    var Config = {};

    // methods ----------------------------------------------------------------

    /**
     * Returns the value of a specific configuration property, or the complete
     * set of all available configuration properties.
     *
     * @param {String} [key]
     *  If specified, the key of a specific configuration property. If omitted,
     *  this method will return the entire configuration as a map.
     *
     * @param {Any} [def]
     *  A default value, if the configuration property specified with the 'key'
     *  parameter is not set.
     *
     * @returns {Any}
     *  The value of a specific configuration property, or the complete set of
     *  configuration properties of the OX Documents module, according to the
     *  'key' parameter.
     */
    Config.get = function (key, def) {
        return _.isString(key) ? ((key in documentsConfig) ? documentsConfig[key] : def) : documentsConfig;
    };

    /**
     * Sets a value of a specific configuration property.
     *
     * @param {String} [key]
     *  The key of a specific configuration property. If the key is missing in
     *  the configuration, it will be added.
     *
     * @param {Any} [value]
     *  A value to be set in association with the provided key.
     */
    Config.set = function (key, value) {
        if (_.isString(key)) {
            documentsConfig[key] = value;
            settings.set('module', documentsConfig).save();
        }
    };

    /**
     * Returns the state of a specific Boolean configuration property.
     *
     * @param {String} key
     *  The key of a specific Boolean configuration property.
     *
     * @param {Boolean} [def=false]
     *  The default state, if the configuration property specified with the
     *  'key' parameter is not set.
     *
     * @returns {Boolean}
     *  The state of the specified configuration property. If the property does
     *  not exist, this method will return the default value.
     */
    Config.getFlag = function (key, def) {
        return Config.get(key, def === true) === true;
    };

    /**
     * Returns the state of the specified Boolean flag in the anchor part of
     * the browser URL.
     *
     * @param {String} flag
     *  The name of the URL anchor flag.
     *
     * @param {Boolean} [def=false]
     *  The default state, if the URL does not contain the specified option.
     *
     * @returns {Boolean}
     *  The state of the specified flag in the anchor part of the URL.
     */
    Config.getUrlFlag = function (flag, def) {
        var value = _.url.hash(flag);
        return (/^(1|true)$/i).test(value) || ((/^(0|false)$/i).test(value) ? false : (def === true));
    };

    /**
     * Returns the state of the specified Boolean flag in the anchor part of
     * the browser URL, but only if the application runs in debug mode (see
     * constant Config.DEBUG for details). Without debug mode, this method
     * simply returns the passed default value.
     *
     * @param {String} flag
     *  The name of the URL anchor flag.
     *
     * @param {Boolean} [def=false]
     *  The default state, if the URL does not contain the specified option, or
     *  if the application is not in debug mode.
     *
     * @returns {Boolean}
     *  The state of the specified flag in the anchor part of the URL.
     */
    Config.getDebugUrlFlag = function (flag, def) {
        return Config.DEBUG ? Config.getUrlFlag(flag, def) : (def === true);
    };

    /**
     * Returns whether a specific debug mode is enabled in all OX Documents
     * applications. The debug mode will be enabled, if the passed server-side
     * configuration property is set to true, and the passed URL hash option is
     * NOT explicitly set to false; or if the server-side configuration
     * property 'debugondemand' is set to true, and the passed URL hash option
     * is set to true.
     *
     * @param {String} configFlag
     *  The name of the server-side configuration item.
     *
     * @param {String} urlFlag
     *  The name of the URL hash option.
     *
     * @returns {Boolean}
     *  Whether the specified debug mode is enabled.
     */
    Config.getDebugState = function (configFlag, urlFlag) {
        return Config.getFlag(configFlag) ? Config.getUrlFlag(urlFlag, true) : (Config.getFlag('debugondemand') && Config.getUrlFlag(urlFlag));
    };

    // constants --------------------------------------------------------------

    /**
     * The full identifier of the current locale, with leading lower-case
     * language identifier, and trailing upper-case country identifier,
     * separated by an underscore character, e.g. 'en_US'.
     *
     * @constant
     */
    Config.LOCALE = ox.language;

    /**
     * The lower-case language identifier of the current locale, e.g. 'en'.
     *
     * @constant
     */
    Config.LANGUAGE = ox.language.split('_')[0];

    /**
     * The upper-case country identifier of the current locale, e.g. 'US'.
     *
     * @constant
     */
    Config.COUNTRY = ox.language.split('_')[1];

    /**
     * A Boolean flag specifying whether the debug mode is enabled in all OX
     * Documents applications. Debug mode will be enabled, if the server-side
     * configuration property 'debugavailable' is set to true, and the URL hash
     * option 'office:enable-debug' is NOT explicitly set to false; or if the
     * server-side configuration property 'debugondemand' is set to true, and
     * the URL hash option 'office:enable-debug' is set to true.
     *
     * @constant
     */
    Config.DEBUG = Config.getDebugState('debugavailable', 'office:enable-debug');

    // exports ================================================================

    return Config;

});
