/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Stefan Eckert <stefan.eckert@open-xchange.com>
 */

define('io.ox/office/textframework/selection/androidselection', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/keycodes',
    'io.ox/office/tk/forms',
    'io.ox/office/tk/render/font',
    'io.ox/office/baseframework/app/appobjectmixin',
    'io.ox/office/textframework/utils/operations',
    'io.ox/office/textframework/utils/position',
    'io.ox/office/textframework/utils/dom'
], function (Utils, KeyCodes, Forms, Font, AppObjectMixin, Operations, Position, DOM) {

    'use strict';

    var NBSP = '\xa0';
    var DUMMYLEN = 50;
    var CURSOR = 'androidcursor';
    var IMEMODE = 'androidime';
    var BLINK = 'androidblink';
    var SELECTOR = '.' + CURSOR + ', .' + IMEMODE;
    var CLASSES = CURSOR + ' ' + IMEMODE;

    //first letter is workaround for Camel-Case-Behavior on some devices (Bug 40066)
    var INITTEXT = 'G' + Utils.repeatString(' ', DUMMYLEN - 1);

    var ENDCHARS = Position.getWordBoundEndChars();

    var MONO_FONT = new Font('monospace', 20); // size in points
    var LETTER_WIDTH = MONO_FONT.getTextWidth('A');

    // mix-in class AndroidSelection ==========================================

    /**
     *
     * @constructor
     *
     * @extends AppObjectMixin
     */
    function AndroidSelection(app, rootNode) {

        var self = this;
        var blinkDeferred = null;
        var oldPara = null;
        var oldStart = null;
        var oldEnd = null;

        var wordFrom = null;
        var wordTo = null;
        var word = null;

        var div = $('<div>');
        var textarea = null;
        var singleArray = [0];

        var restoreFN = null;
        var lastCursorIndex = -5000;

        // base constructor ---------------------------------------------------

        AppObjectMixin.call(this, app);

        // private methods ----------------------------------------------------

        function isSoftKeyboardOpen() {
            if (Utils.isSoftKeyboardOpen()) {
                //TODO: better check focus in textarea!!!
                return (Utils.getInnerWindowHeight() / Utils.getScreenHeight()) < 0.7;
            }
        }

        function splitTextSpan(para, pos) {
            singleArray[0] = _.last(pos);

            if (singleArray[0] === 0) {
                var firstSpan = para.children().first();
                if (!firstSpan.is('span') || !firstSpan[0].innerHTML.length) {
                    return firstSpan[0];
                } else {
                    var newSpan = firstSpan.clone(true);
                    fillTextInSpan(newSpan[0], '');
                    para.prepend(newSpan);
                    return newSpan[0];
                }
            }

            var domPos = Position.getDOMPosition(para[0], singleArray);
            if (!domPos) {
                Utils.error('no result', para.children().length, pos, para[0]);
                return null;
            }

            if (domPos.node.nodeValue.length === domPos.offset) {
                return domPos.node.parentElement;
            } else {
                return DOM.splitTextSpan(domPos.node.parentElement, domPos.offset)[0];
            }
        }

        function fillTextInSpan(span, text) {
            span.innerHTML = '';
            span.appendChild(document.createTextNode(text));
        }

        function getWordBoundary(para, pos, beginning, dontAddSpace) {
            var len = pos.length;
            singleArray[0] = pos[len - 1];

            var result = _.clone(pos);
            result[len - 1] = Position.getWordBoundary(para[0], singleArray, beginning, dontAddSpace)[0];
            return result;
        }

        function changeSelection(event, selectionInfo) {
            var start = selectionInfo.start;
            var end = selectionInfo.end;

            var spans = null;
            var node = null;

            if (oldPara) {
                oldPara.removeClass(BLINK);
                spans = oldPara.find(SELECTOR);
                spans.removeClass(CLASSES);
                /*
                spans.each(function () {
                    TextUtils.mergeSiblingTextSpans(this, true);
                });
                */
                oldPara = null;
            }

            var visibility = 'hidden';

            if (app.getModel().getEditMode()) {

                var selType = window.getSelection().type;
                if (!selType || selType === 'None') {
                    selType = isSoftKeyboardOpen();
                }

                if (_.isEqual(start, end) && selType) {

                    oldPara = $(Position.getDOMPosition(app.getModel().getCurrentRootNode(), _.initial(start)).node);
                    oldPara.addClass(BLINK);

                    wordFrom = getWordBoundary(oldPara, start, true);
                    wordTo = getWordBoundary(oldPara, start, false, true);
                    if (_.isEqual(wordFrom, wordTo) ||  _.last(wordFrom) >= _.last(wordTo)) {
                        start[start.length - 1] -= 1;
                        wordFrom = getWordBoundary(oldPara, start, true);
                        wordTo = getWordBoundary(oldPara, start, false, true);
                        start[start.length - 1] += 1;
                    }

                    if (_.last(wordFrom) < _.last(wordTo)) {
                        splitTextSpan(oldPara, wordFrom);
                        splitTextSpan(oldPara, wordTo);

                        word = '';

                        singleArray[0] = _.last(wordFrom);
                        var spanFrom = Position.getDOMPosition(oldPara[0], singleArray, true).node;

                        var len = 0;
                        var maxLen = _.last(wordTo) - _.last(wordFrom);

                        while (len < maxLen && spanFrom) {
                            word += spanFrom.innerHTML;
                            len = word.length;
                            spanFrom.classList.add(IMEMODE);
                            spanFrom = spanFrom.nextSibling;
                        }
                    } else {
                        wordFrom = _.clone(start);
                        wordTo =  _.clone(start);
                        word = '';
                    }

                    oldPara.find('.' + CURSOR).removeClass(CURSOR);
                    node = splitTextSpan(oldPara, start);
                    node = $(node);
                    node.addClass(CURSOR);
                    oldStart = start;
                    oldEnd = end;

                    selectTextArea();
                    handleTextAreaPos();
                    visibility = 'visible';
                }
            }
            div.css('visibility', visibility);
        }

        function handleTextAreaPos() {
            var parent = div.parent();
            if (parent && parent.length) {

                var zoom = app.getView().getZoomFactor();
                var pos = Position.getPixelPositionFromDomPoint(rootNode,
                        [_.last(oldStart)], zoom, oldPara);
                var multi = zoom / 100;
                div.css({
                    left: pos.left * multi - 5,
                    top: pos.top * multi + pos.height * multi - 30
                });
            }
        }

        function isPassThrough(event, press) {
            if (event.ctrlKey || event.altKey || event.metaKey) { return true; }

            if (press) {
                return event.keyCode > 0  && event.keyCode !== KeyCodes.IME_INPUT;
            } else if (event.keyCode === KeyCodes.TAB) {
                event.preventDefault();
                return true;
            } else if (event.shiftKey && DOM.isCursorKey(event.keyCode)) {
                restoreFN();
                event.preventDefault();
                return true;
            } else if (event.keyCode === KeyCodes.BACKSPACE) {
                event.preventDefault();
                return true;
            } else if (DOM.isCursorKey(event.keyCode) || event.keyCode === KeyCodes.ENTER || event.keyCode === KeyCodes.DELETE) {
                return true;
            } else if (event.keyCode > 0  && event.keyCode !== KeyCodes.IME_INPUT) {
                return true;
            } else {
                return false;
            }
        }

        function passThrough(event) {
            if (isPassThrough(event, true)) {
                handlePassThrough(event);
            }
        }

        function handlePassThrough(event) {
            var docModel = app.getModel();

            if (!event.shiftKey && (event.keyCode === KeyCodes.UP_ARROW || event.keyCode === KeyCodes.DOWN_ARROW)) {
                event.preventDefault();
                if (event.type === 'keydown' && oldPara) {
                    var offset = event.keyCode === KeyCodes.UP_ARROW ? -1 : 1;

                    var cursorSpan = oldPara.find('.' + CURSOR);
                    var w = $(window);

                    var height = cursorSpan.height();
                    offset *= height * 1.5;

                    var pos = cursorSpan.offset();

                    var top = pos.top - w.scrollTop();
                    var left = pos.left + cursorSpan.width() - w.scrollLeft();

                    var newOxo = Position.getOxoPositionFromPixelPosition(rootNode, left, top + height / 2 + offset);

                    if (!newOxo) {
                        if (oldStart.length > 2) {
                            return;
                        }
                        newOxo = { start: [] };
                        if (event.keyCode === KeyCodes.UP_ARROW) {
                            newOxo.start[0] = Math.max(0, oldStart[0] - 1);
                        } else {
                            newOxo.start[0] = oldStart[0] + 1;
                        }
                        newOxo.start.push(0);
                    }

                    self.setTextSelection(newOxo.start);
                }
            } else if (!event.shiftKey && (event.keyCode === KeyCodes.LEFT_ARROW || event.keyCode === KeyCodes.RIGHT_ARROW)) {
                docModel.handleAndroidTyping();
                event.preventDefault();
                event.preventDefault = $.noop;
                rootNode.trigger(event);
            } else {
                docModel.handleAndroidTyping();
                event.preventDefault = $.noop;
                rootNode.trigger(event);
            }
            return true;
        }

        function simulateEvent(keyCode) {
            handlePassThrough(new $.Event('keydown', { keyCode: keyCode }));
            handlePassThrough(new $.Event('keypress', { keyCode: keyCode }));
            handlePassThrough(new $.Event('keyup', { keyCode: keyCode }));
        }

        function changedText(event) {
            var docModel = app.getModel();
            var cursorIndex = getCursorIndex();

            if (isPassThrough(event)) {
                handlePassThrough(event);
                lastCursorIndex = cursorIndex;
            } else {
                var text = getText();

                if (cursorIndex < 0 || (cursorIndex + 1 === lastCursorIndex && text.length <= 1)) {
                    //backspace!
                    if (event.type === 'input') {
                        simulateEvent(KeyCodes.BACKSPACE);
                    }
                    lastCursorIndex = cursorIndex;
                    return;
                }

                lastCursorIndex = cursorIndex;

                docModel.handleAndroidTyping();
                if (text.length > 0) {

                    var preAttrs = docModel.getPreselectedAttributes() || undefined;
                    var attrs = preAttrs;
                    var changeTrack = docModel.getChangeTrack();
                    if (changeTrack.isActiveChangeTracking()) {
                        if (!attrs) {
                            attrs = {};
                        } else {
                            attrs = _.copy(attrs, true);
                        }
                        attrs.changes = { inserted: changeTrack.getChangeTrackInfo() };
                    }

                    var undoOperations = [];
                    var redoOperations = [];

                    if (DOM.isImplicitParagraphNode(oldPara[0])) {
                        oldPara.data('implicit', false);
                        redoOperations.push({ name: Operations.PARA_INSERT, start: _.initial(wordFrom) });
                        undoOperations.push({ name: Operations.DELETE, start: _.initial(wordFrom) });
                    }

                    if (text.length - 1 === word.length && text.substring(0, text.length - 1) === word && !preAttrs) {
                        //optimized for forward typing
                        redoOperations.push({ name: Operations.TEXT_INSERT, start: _.clone(wordTo), text: text.substring(text.length - 1, text.length), attrs: attrs });
                        undoOperations.push({ name: Operations.DELETE, start: _.clone(wordTo), end: _.clone(wordTo) });
                    } else {
                        var insertOp = { name: Operations.TEXT_INSERT, start: _.clone(wordTo), text: text, attrs: attrs };
                        redoOperations.push(insertOp);

                        var deleteOp = null;

                        if (_.last(wordFrom) !== _.last(wordTo)) {
                            deleteOp = { name: Operations.DELETE, start: _.clone(wordFrom), end: Position.increaseLastIndex(wordTo, -1) };
                            redoOperations.push(deleteOp);
                        }

                        if (deleteOp) {
                            var insStart = Position.increaseLastIndex(wordFrom, text.length);
                            var undoDeleteOp = { name: Operations.TEXT_INSERT, start: insStart, text: word, attrs: preAttrs };
                            undoOperations.push(undoDeleteOp);
                        }

                        var undoInsertOp = { name: Operations.DELETE, start: _.clone(wordFrom), end: Position.increaseLastIndex(wordTo, -1) };
                        undoOperations.push(undoInsertOp);
                    }

                    sendOperations(docModel, changeTrack, undoOperations, redoOperations);

                    var leftNoIME = _.contains(ENDCHARS, text[cursorIndex - 1]);

                    applyOperations(docModel, redoOperations);

                    var pos = _.last(wordFrom) + cursorIndex;
                    if (cursorIndex > 0 && cursorIndex < text.length) {
                        var before = splitTextSpan(oldPara, [pos]);
                        oldPara.find('.' + CURSOR).removeClass(CURSOR);
                        before.classList.add(CURSOR);
                    }
                    oldStart[oldStart.length - 1] = pos;
                    oldEnd[oldEnd.length - 1] = pos;

                    if (leftNoIME) {
                        word = text.substring(cursorIndex);
                        wordFrom[wordFrom.length - 1] = pos;
                        wordTo[wordTo.length - 1] = _.last(wordFrom) + word.length;

                        oldPara.find('span.' + IMEMODE).removeClass(IMEMODE);

                        selectTextArea();
                    } else {
                        word = text;
                        wordTo[wordTo.length - 1] = _.last(wordFrom) + text.length;

                        var bulletDiv = oldPara.find('>div.' + CURSOR);
                        if (bulletDiv.length) {
                            bulletDiv.removeClass(CURSOR);
                            oldPara.find('span.' + IMEMODE).addClass(CURSOR);
                        }
                    }

                    handleTextAreaPos();

                    if (attrs) {
                        docModel.getStyleCollection('paragraph').updateElementFormatting(oldPara);
                    }

                }
            }
        }

        function sendOperations(docModel, changeTrack, undoOperations, redoOperations) {
            if (changeTrack.isActiveChangeTracking()) {
                var info = changeTrack.getChangeTrackInfo();
                _.each(undoOperations, function (op) {
                    addChangeTrackInfo(op, info);
                });
                _.each(redoOperations, function (op) {
                    addChangeTrackInfo(op, info);
                });
            }
            docModel.sendOperations(redoOperations);
            docModel.getUndoManager().addUndo(undoOperations, redoOperations);
        }

        function getDOMPosition(pos) {
            if (!pos) { return; }

            singleArray[0] = _.last(pos);
            var node = Position.getDOMPosition(oldPara, singleArray, true);
            if (node) {
//                node.node = node.node.parentElement;
                node.nodeIndex = $(node.node).index();
                return node;
            }
            throw 'nothing found';
        }

        function applyOperations(docModel, ops) {
            _.each(ops, function (op) {

                var start = getDOMPosition(op.start);
                var end = getDOMPosition(op.end);
                var text;

                if (op.name === Operations.DELETE) {

                    if (start.nodeIndex === end.nodeIndex) {
                        text = start.node.innerText;
                        text = text.substring(0, start.offset) + text.substring(end.offset + 1);
                        fillTextInSpan(start.node, text);
                    } else {
                        var startText = start.node.innerText;
                        fillTextInSpan(start.node, startText.substring(0, start.offset));
                        var endText = end.node.innerText;
                        fillTextInSpan(end.node, endText.substring(end.offset + 1));
                        var children = oldPara.children();
                        if ((end.nodeIndex - start.nodeIndex) > 1) {
                            for (var ni = start.nodeIndex + 1; ni < end.nodeIndex; ni++) {
                                oldPara[0].removeChild(children[ni]);
                            }
                        }
                    }

                } else if (op.name ===  Operations.TEXT_INSERT) {
                    var opText = op.text;
                    if (opText[opText.length - 1] === ' ') {
                        opText = opText.substring(0, opText.length - 1) + NBSP;
                    }

                    var attrs = op.attrs;
                    text = start.node.innerText;
                    text = text.substring(0, start.offset) + opText + text.substring(start.offset);
                    fillTextInSpan(start.node, text);

                    start.node.classList.add(IMEMODE);
                    var splitStart = _.last(op.start);
                    var splitEnd = splitStart - start.offset;

                    var counter = start.offset;
                    for (var i = splitStart; i >= splitEnd; i--) {
                        counter--;
                        if (_.contains(ENDCHARS, text[counter])) {

                            var before = splitTextSpan(oldPara, [i]);
                            before.classList.remove(IMEMODE, CURSOR);
                            var beforeText = before.innerText;

                            if (_.last(beforeText) === NBSP) {
                                fillTextInSpan(before, beforeText.substring(0, beforeText.length - 1) + ' ');
                            }
                            break;
                        }
                    }

                    handlePreselectedAttrs(attrs, start.node);
                }
            });

        }

        function addChangeTrackInfo(op, info) {
            var attrs = op.attrs;
            if (attrs) {
                if (attrs.changes) {
                    return;
                }
                attrs = _.copy(attrs, true);
            } else {
                attrs = {};
            }

            var name;
            if (op.name === Operations.TEXT_INSERT) {
                name = 'inserted';
            } else {
                name = 'deleted';
            }
            attrs.changes = {};
            attrs.changes[name] = info;

        }

        function handlePreselectedAttrs(attrs) {
            if (attrs) {
                attrs = { attributes: attrs };
                for (var i = 1; i < arguments.length; i++) {
                    $(arguments[i]).data(attrs);
                }
            }
        }

        textarea = $('<textarea>');

        function selectTextArea() {
            textarea.focus();
            textarea.val(INITTEXT + word);
            Forms.setInputSelection(textarea, DUMMYLEN + (_.last(oldStart) - _.last(wordFrom)));
        }

        function getCursorIndex() {
            return textarea[0].selectionStart - DUMMYLEN;
        }

        function getText() {
            return textarea.val().substring(DUMMYLEN);
        }

        function resetTextarea() {
            div.css('visibility', 'hidden');
        }

        function setReadOnly() {
            //console.error('setReadOnly');
            rootNode.css('-webkit-user-modify', 'read-only');
        }

        function setReadWrite() {
            //console.error('setReadWrite');
            rootNode.css('-webkit-user-modify', 'read-write');
        }

        var processSelectUpdate = (function () {
            var always = function () {
                var selecttype = window.getSelection().type;
                if (Utils.getActiveElement() === rootNode[0]) {
                    //listener hangs directly on document, but we only want the events of editdiv
                    if (selecttype !== 'None') {
                        self.updateSelectionAfterBrowserSelection({ preserveFocus: false, forceTrigger: false });
                        if (selecttype === 'Caret') {
                            setReadOnly();
                        }
                    }
                } else if (Utils.getActiveElement() === textarea[0]) {
                    if (window.getSelection().anchorNode === div[0]) {
                        //focus really in textarea
                        if (selecttype === 'Range') {
                            if (_.last(wordFrom) < _.last(wordTo)) {
                                restoreFN();
                                self.setTextSelection(wordFrom, wordTo);
                            }
                        }
                        setReadOnly();
                    } else {
                        //focus in document
                        if (selecttype !== 'Range') {
                            setReadOnly();
                        }
                    }
                }
            };
            var notAlways =  _.throttle(always, 250);

            return function (evt) {

                var selecttype = window.getSelection().type;
                if (evt.type === 'selectstart') {
                    if (selecttype === 'Range') {
                        ///User seems to call "select all", but android chrome does not really support that featurem when contenteditable=true and begins with scrolling :(
                        evt.preventDefault();
                        return;
                    }
                    always();
                } else {
                    notAlways();
                }

                if (selecttype === 'Range') {
                    if (isSoftKeyboardOpen()) {
                        setReadOnly();
                        _.defer(setReadWrite);
                    } else if (rootNode.css('-webkit-user-modify') === 'read-only') {
                        setReadWrite();
                    }
                }
            };
        }());

        //---------init---------

        textarea.on('keyup keydown change input', changedText);
        textarea.on('keypress', passThrough);

        self.on('changecursorinfo', changeSelection);

        blinkDeferred = app.repeatDelayed(function () {
            if (oldPara) {
                if (Utils.getActiveElement() && Utils.getActiveElement().tagName === 'INPUT') {
                    oldPara.removeClass(BLINK);
                } else {
                    oldPara.toggleClass(BLINK);
                }
            }
        }, { delay: 500 }, 'Text: Selection: blinkDeferred');

        this.waitForImportSuccess(function () {
            var docView = app.getView();

            restoreFN = _.bind(self.restoreBrowserSelection, self);

            self.restoreBrowserSelection = function () {
                //sideEffect of passThrough, with original restoreFN textarea-focus gets lost
                if (!_.isEqual(self.getStartPosition(), self.getEndPosition())) {
                    restoreFN();
                }
            };

            var docModel = app.getModel();
            textarea.on('cut copy paste', function (evt) {
                docModel[evt.type](evt);
            });

            div.css({
                position: 'absolute',
                zIndex: 10,
                top: '30px',
                width: '10px',
                height: '40px',
                overflow: 'hidden'
            });

            textarea.css({
                position: 'absolute',
                width: '10px',
                letterSpacing: -LETTER_WIDTH + 'px',
                fontFamily: MONO_FONT.family,
                fontSize: MONO_FONT.size + 'pt',
                lineHeight: MONO_FONT.size + 'pt',
                height: '30px',
                overflow: 'hidden',
                outline: 'none',
                opacitiy: 0,
                margin: 0,
                padding: 0,
                textIndent: '3px',
                left: 0,
                top: 0
            });

            textarea.addClass('transparent-selection');

            div.append(textarea);
            rootNode.parent().append(div);

            docModel.listenTo($(document), 'selectstart selectionchange select', processSelectUpdate);

            if (false) {
                //TODO
                app.registerWindowResizeHandler(function () {
                    if (isSoftKeyboardOpen() && window.getSelection().type !== 'Range') {
                        var boundRect = docModel.getSelection().getFocusPositionBoundRect();
                        if (boundRect) {
                            docView.scrollToPageRectangle(boundRect);
                        }
                    }
                });
            }

            app.registerGlobalEventHandler(window, 'orientationchange', resetTextarea);
            docModel.listenTo(docView, 'change:zoom', resetTextarea);

        });

        this.listenTo(app, 'docs:beforequit', function () {
            textarea.remove();
            div.remove();
            blinkDeferred.abort();
        });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            blinkDeferred = textarea = div = null;
        });

    } // class AndroidSelection

    // exports ================================================================

    return AndroidSelection;

});
