/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define('io.ox/office/textframework/model/attributeoperationmixin', [
    'io.ox/office/drawinglayer/view/drawingframe',
    'io.ox/office/editframework/utils/attributeutils',
    'io.ox/office/editframework/utils/border',
    'io.ox/office/textframework/components/table/table',
    'io.ox/office/textframework/format/tablestyles',
    'io.ox/office/textframework/utils/dom',
    'io.ox/office/textframework/utils/position',
    'io.ox/office/textframework/utils/operations',
    'io.ox/office/textframework/utils/snapshot',
    'io.ox/office/textframework/utils/textutils',
    'gettext!io.ox/office/textframework/main'
], function (DrawingFrame, AttributeUtils, Border, Table, TableStyles, DOM, Position, Operations, Snapshot, Utils, gt) {

    'use strict';

    // mix-in class AttributeOperationMixin ======================================

    /**
     * A mix-in class for the document model class providing the operation
     * handling for setting attributes used in a presentation and text document.
     *
     * @constructor
     *
     * @param {EditApplication} app
     *  The application instance.
     */
    function AttributeOperationMixin(app) {

        var // self reference for local functions
            self = this;

        // private methods ----------------------------------------------------

        /**
         * Returns the family of the attributes supported by the passed DOM
         * element.
         *
         * @param {HTMLElement|jQuery} element
         *  The DOM element whose associated attribute family will be returned.
         *  If this object is a jQuery collection, returns its first node.
         */
        function resolveElementFamily(element) {

            var // the element, as jQuery object
                $element = $(element),
                // the resulting style family
                family = null;

            if (DrawingFrame.isDrawingFrame($element)) {
                family = 'drawing'; // selecting drawing first (necessary after 39312)
            } else if (DOM.isPartOfParagraph($element)) {
                family = 'character';
            } else if (DOM.isSlideNode($element)) {
                family = 'slide';
            } else if (DOM.isParagraphNode($element)) {
                family = 'paragraph';
            } else if (DOM.isTableNode($element)) {
                family = 'table';
            } else if ($element.is('tr')) {
                family = 'row';
            } else if ($element.is('td')) {
                family = 'cell';
            } else {
                Utils.warn('Editor.resolveElementFamily(): unsupported element');
            }

            return family;
        }

        /**
         * Returning the list style id from specified paragraph attributes. The list
         * style can be set directly or via a style id at the paragraph.
         *
         * @param {Object} [attributes]
         *  An optional map of attribute maps (name/value pairs), keyed by attribute.
         *  It this parameter is defined, the parameter 'paragraph' can be omitted.
         *
         * @returns {String|null}
         *  The list style id or null, if no list style defined in specified attributes.
         */
        function getListStyleIdFromParaAttrs(attributes) {

            // shortcut, not handling style attributes
            if (attributes && attributes.paragraph && attributes.paragraph.listStyleId) { return attributes.paragraph.listStyleId; }

            // checking paragraph style
            if (attributes && attributes.styleId && self.isParagraphStyleWithListStyle(attributes.styleId)) { return getListStyleInfoFromStyleId(attributes.styleId, 'listStyleId'); }

            return null;
        }

        /**
         * Returns the value of a specified paragraph attribute of a given
         * paragraph style.
         *
         * @param {String} styleId
         *  The paragraph style id, that will be checked checked.
         *
         * @param {String} paraAttr
         *  The attribute of the paragraph attributes of the style, whose value
         *  shall be returned.
         *
         * @returns {String}
         *  The value of the specified paragraph attribute of the specified style.
         */
        function getListStyleInfoFromStyleId(styleId, paraAttr) {

            var // the style attributes
                styleAttrs = self.getParagraphStyles().getStyleSheetAttributeMap(styleId);

            return (styleAttrs && styleAttrs.paragraph && styleAttrs.paragraph[paraAttr]);
        }

        /**
         * Updating the neighbor paragraphs, if the list level of the specified paragraph
         * was modified and the paragraph has a visible top or bottom border (43897).
         *
         * @param {Node|jQuery} paragraph
         *  The paragraph node that gets new attributes assigned.
         *
         * @param {Object} attributes
         *  The container with the modified attributes of the paragraph.
         */
        function handleNeighborBorderUpdate(paragraph, attributes) {

            var // the previous and the next paragraph nodes
                prevParagraph = null, nextParagraph = null,
                // the full set of element attributes
                allAttrs = null;

            if (attributes && attributes.paragraph && _.isNumber(attributes.paragraph.listLevel)) {

                allAttrs = self.getParagraphStyles().getElementAttributes(paragraph);

                if (Border.isVisibleBorder(allAttrs.paragraph.borderTop)) { prevParagraph = $(paragraph).prev(); }
                if (Border.isVisibleBorder(allAttrs.paragraph.borderBottom)) { nextParagraph = $(paragraph).next(); }

                // also updating the neighboring paragraphs
                if (prevParagraph && prevParagraph.length > 0 && DOM.isParagraphNode(prevParagraph)) { self.getParagraphStyles().updateElementFormatting(prevParagraph); }
                if (nextParagraph && nextParagraph.length > 0 && DOM.isParagraphNode(nextParagraph)) { self.getParagraphStyles().updateElementFormatting(nextParagraph); }
            }

        }

        /**
         * Changes a specific formatting attribute of the specified element or
         * text range. The type of the attributes will be determined from the
         * specified range.
         *
         * @param {Number[]} start
         *  The logical start position of the element or text range to be
         *  formatted.
         *
         * @param {Number[]} [end]
         *  The logical end position of the element or text range to be
         *  formatted.
         *
         * @param {Object} attributes
         *  A map with formatting attribute values, mapped by the attribute
         *  names, and by attribute family names.
         */
        var implSetAttributes = (function () {

            // temporary array with only one entry, for use in Position functions
            var TMPPOSARRAY = [0];

            return function (start, end, attributes, target) {

                var // node info for start/end position
                    startInfo = null, endInfo = null,
                    // the main attribute family of the target components
                    styleFamily = null,
                    // the style sheet container for the target components
                    styleSheets = null,
                    // options for style collection method calls
                    options = null,
                    // the last text span visited by the character formatter
                    lastTextSpan = null,
                    // undo operations going into a single action
                    undoOperations = [],
                    // redo operation
                    redoOperation = null,
                    // Performance: Saving data for list updates
                    paraAttrs = null, listStyleId = null, listLevel = null, oldListStyleId = null,
                    // a helper attribute object with values set to null
                    nullAttributes = null,
                    // a helper iterator key
                    localkey = null,
                    //element which is passed to page breaks calculation
                    currentElement,
                    // the previous operation before this current operation
                    previousOperation = null,
                    // whether the paragraph cache can be reused (performance)
                    useCache = false,
                    // whether an update of formatting is required after assigning attributes (needed by spellcheck attribute)
                    forceUpdateFormatting = false,
                    // a data object, that is used for updating slides with master or layout target (presentation only)
                    forceLayoutFormattingData = null,
                    //flags for triggering page breaks
                    isCharOrDrawingOrRow,
                    isClearFormatting,
                    isAttributesInParagraph,
                    isPageBreakBeforeAttributeInParagraph,
                    isParSpacing,
                    // if target is present, root node is header or footer, otherwise editdiv
                    rootNode = self.getRootNode(target),
                    // the selection object
                    selection = self.getSelection(),
                    // a helper object for attribute modifications
                    modifiedAttributes = null,
                    // the unde manager object
                    undoManager = self.getUndoManager();

                // sets or clears the attributes using the current style sheet container
                // 'modifiedAttrs' is an optional parameter, that is only used, if the original attribute set was modified
                function setElementAttributes(element, modifiedAttrs) {
                    styleSheets.setElementAttributes(element, modifiedAttrs ? modifiedAttrs : attributes, options);
                }

                // change listener used to build the undo operations
                function changeListener(attributedNode, oldAttributes, newAttributes) {

                    var // the element node might be modified, if it is an empty text span (41250)
                        element = (!app.isODF() && DOM.isEmptySpan(attributedNode)) ? attributedNode.parentNode : attributedNode,
                        // selection object representing the passed element
                        range = Position.getPositionRangeForNode(rootNode, element),
                        // the attributes of the current family for the undo operation
                        undoAttributes = {},
                        // the operation used to undo the attribute changes
                        undoOperation = { name: Operations.SET_ATTRIBUTES, start: range.start, end: range.end, attrs: undoAttributes },
                        // last undo operation (used to merge character attributes of sibling text spans)
                        lastUndoOperation = ((undoOperations.length > 0) && (styleFamily === 'character')) ? _.last(undoOperations) : null;

                    function insertUndoAttribute(family, name, value) {
                        undoAttributes[family] = undoAttributes[family] || {};
                        undoAttributes[family][name] = value;
                    }

                    // merging the place holder drawings into the old attributes object (presentation only)
                    if (forceLayoutFormattingData && forceLayoutFormattingData.oldAttrs) { oldAttributes = self.mergePlaceHolderAttributes(element, startInfo.family, oldAttributes, forceLayoutFormattingData.oldAttrs); }

                    // extend undoOperation if target is present
                    self.extendPropertiesWithTarget(undoOperation, target);

                    // exceptional case for setting line type to 'none', to save the old color
                    if (_.has(attributes, 'line') && _.has(attributes.line, 'type') && attributes.line.type === 'none') {
                        if (_.has(oldAttributes, 'line') && _.has(oldAttributes.line, 'color') && !_.isUndefined(oldAttributes.line.color)) {
                            undoAttributes.line = {};
                            undoAttributes.line.color = oldAttributes.line.color;
                        }
                    }

                    // exceptional case for setting fill type to 'none', to save the old color
                    if (_.has(attributes, 'fill') && _.has(attributes.fill, 'type')) {

                        // swtiching from type 'solid' to 'none' or 'bitmap'
                        if (attributes.fill.type === 'none' || attributes.fill.type === 'bitmap') {

                            if (_.has(oldAttributes, 'fill') && _.has(oldAttributes.fill, 'color') && !_.isUndefined(oldAttributes.fill.color)) {
                                undoAttributes.fill = {};
                                undoAttributes.fill.color = oldAttributes.fill.color;
                            }

                        } else if (attributes.fill.type === 'none' || attributes.fill.type === 'solid') {

                            if (_.has(oldAttributes, 'fill') && _.has(oldAttributes.fill, 'imageUrl') && !_.isUndefined(oldAttributes.fill.imageUrl)) {
                                undoAttributes.fill = {};
                                undoAttributes.fill.imageUrl = oldAttributes.fill.imageUrl;
                            }
                        }
                    }

                    // find all old attributes that have been changed or cleared
                    _(oldAttributes).each(function (attributeValues, family) {
                        if (family === 'styleId') {
                            // style sheet identifier is a string property
                            if (attributeValues !== newAttributes.styleId) {
                                undoAttributes.styleId = attributeValues;
                            }
                        } else {
                            // process attribute map of the current family
                            _(attributeValues).each(function (value, name) {
                                if (!(family in newAttributes) || !_.isEqual(value, newAttributes[family][name])) {
                                    insertUndoAttribute(family, name, value);
                                }
                            });
                        }
                    });

                    // find all newly added attributes
                    _(newAttributes).each(function (attributeValues, family) {
                        if (family === 'styleId') {
                            // style sheet identifier is a string property
                            if (!_.isString(oldAttributes.styleId)) {
                                undoAttributes.styleId = null;
                            }
                        } else {
                            // process attribute map of the current family
                            _(attributeValues).each(function (value, name) {
                                if (!(family in oldAttributes) || !(name in oldAttributes[family])) {
                                    insertUndoAttribute(family, name, null);
                                }
                            });
                        }
                    });

                    // try to merge 'character' undo operation with last array entry, otherwise add operation to array
                    if (lastUndoOperation && (_.last(lastUndoOperation.end) + 1 === _.last(undoOperation.start)) && _.isEqual(lastUndoOperation.attrs, undoOperation.attrs)) {
                        lastUndoOperation.end = undoOperation.end;
                    } else {
                        undoOperations.push(undoOperation);
                    }

                    // invalidate spell result if language attribute changes
                    self.getSpellChecker().resetClosest(element);
                }

                // fail if attributes is not of type object
                if (!_.isObject(attributes)) {
                    return false;
                }

                // do nothing if an empty attributes object has been passed
                if (_.isEmpty(attributes)) {
                    return;
                }

                // resolve start and end position
                if (!_.isArray(start)) {
                    Utils.warn('Editor.implSetAttributes(): missing start position');
                    return false;
                }

                // whether the cache can be reused (performance)
                previousOperation = self.getPreviousOperation();
                useCache = (!target && previousOperation && self.getParagraphCacheOperation()[previousOperation.name] &&
                            selection.getParagraphCache() && _.isEqual(_.initial(start), selection.getParagraphCache().pos) && _.isUndefined(previousOperation.target));

                if (useCache) {
                    TMPPOSARRAY[0] = _.last(start);
                    startInfo = Position.getDOMPosition(selection.getParagraphCache().node, TMPPOSARRAY, true);
                    // setting paragraph cache is not required, it can simply be reused in next operation
                } else {
                    startInfo = Position.getDOMPosition(rootNode, start, true);

                    if (!startInfo) {
                        Utils.warn('Editor.implSetAttributes(): invalid start node!');
                    }

                    if (startInfo.family === 'character') {
                        selection.setParagraphCache(startInfo.node.parentElement, _.clone(_.initial(start)), _.last(start));
                    } else {
                        selection.setParagraphCache(null); // invalidating the cache (previous operation might not be in PARAGRAPH_CACHE_OPERATIONS)
                    }
                }

                if (!startInfo || !startInfo.node) {
                    Utils.warn('Editor.implSetAttributes(): invalid start position: ' + JSON.stringify(start));
                    return false;
                }
                // get attribute family of start and end node
                startInfo.family = resolveElementFamily(startInfo.node);

                if (!startInfo.family) { return; }

                if (_.isArray(end)) {
                    if (startInfo.family === 'character') {
                        //characters are children of paragraph, end is in the same paragraph, so we dont have to search the complete document for it
                        TMPPOSARRAY[0] = _.last(end);
                        endInfo = Position.getDOMPosition(startInfo.node.parentElement, TMPPOSARRAY, true);
                    } else {
                        endInfo = Position.getDOMPosition(rootNode, end, true);
                    }

                    endInfo.family = resolveElementFamily(endInfo.node);

                    if (!endInfo.family) { return; }

                } else {
                    end = start;
                    endInfo = startInfo;
                }

                // options for the style collection method calls (build undo operations while formatting)
                options = undoManager.isUndoEnabled() ? { changeListener: changeListener } : null;

                // Special handling for presentation in master/layout slides -> updating styles (before calling setElementAttributes)
                if (target && _.isFunction(self.isLayoutOrMasterId) && self.isLayoutOrMasterId(target)) { forceLayoutFormattingData = self.updateMasterLayoutStyles(styleFamily === 'character' ? startInfo.node.parentNode : startInfo.node, target, startInfo.family, attributes, { immediateUpdate: false, saveOldAttrs: true }); }

                // characters (start or end may point to a drawing node, ignore that but format as
                // characters if the start object is different from the end object)
                if ((startInfo.family === 'character') || (endInfo.family === 'character') ||
                        ((startInfo.node !== endInfo.node) && DrawingFrame.isDrawingFrame(startInfo.node) && DrawingFrame.isDrawingFrame(endInfo.node))) {

                    // check that start and end are located in the same paragraph (and handling absolute positioned drawings correctly)
                    if (startInfo.node.parentNode !== endInfo.node.parentNode) {

                        // replacing the absolute positioned drawings by its place holder nodes
                        if (DOM.isDrawingLayerNode(startInfo.node.parentNode)) { startInfo.node = DOM.getDrawingPlaceHolderNode(startInfo.node); }
                        if (DOM.isDrawingLayerNode(endInfo.node.parentNode)) { endInfo.node = DOM.getDrawingPlaceHolderNode(endInfo.node); }

                        if (startInfo.node.parentNode !== endInfo.node.parentNode) {
                            Utils.warn('Editor.implSetAttributes(): end position in different paragraph');
                            return;
                        }
                    }

                    // visit all text span elements covered by the passed range
                    // (not only the direct children of the paragraph, but also
                    // text spans embedded in component nodes such as fields and tabs)
                    styleFamily = 'character';
                    styleSheets = self.getCharacterStyles();
                    Position.iterateParagraphChildNodes(startInfo.node.parentNode, function (node) {

                        // visiting the span inside a hard break node
                        // -> this is necessary for change tracking attributes
                        if (DOM.isHardBreakNode(node)) {
                            setElementAttributes(node.firstChild);
                        }
                        // if element is field, set fixed attribute (for automatic updating)
                        if (DOM.isComplexFieldNode(node) || DOM.isFieldNode(node) || DOM.isRangeMarkerNode(node)) {
                            setElementAttributes(node);
                        }

                        // DOM.iterateTextSpans() visits the node itself if it is a
                        // text span, otherwise it visits all descendant text spans
                        // contained in the node except for drawings which will be
                        // skipped (they may contain their own paragraphs).
                        DOM.iterateTextSpans(node, function (span) {
                            // check for a spellchecked span (this needs to be checked, before attributes are applied)
                            if (!forceUpdateFormatting && self.isImportFinished() && DOM.isSpellerrorNode(span)) { forceUpdateFormatting = true; }
                            // assigning the new character attributes
                            setElementAttributes(span);
                            // try to merge with the preceding text span
                            Utils.mergeSiblingTextSpans(span, false);
                            // remember span (last visited span will be merged with its next sibling)
                            lastTextSpan = span;
                        });

                    }, undefined, {
                        // options for Position.iterateParagraphChildNodes()
                        allNodes: true,
                        start: _(start).last(),
                        end: _(end).last(),
                        split: true
                    });

                    // handling spell checking after modifying character attributes
                    if (forceUpdateFormatting) { self.implParagraphChanged(startInfo.node.parentNode); }

                    // try to merge last text span in the range with its next sibling
                    if (lastTextSpan) {
                        Utils.mergeSiblingTextSpans(lastTextSpan, true);
                    }

                // otherwise: only single components allowed at this time
                } else {

                    // check that start and end point to the same element
                    if (startInfo.node !== endInfo.node) {
                        Utils.warn('Editor.implSetAttributes(): no ranges supported for attribute family "' + startInfo.family + '"');
                        return;
                    }

                    // format the (single) element
                    styleFamily = startInfo.family;
                    styleSheets = self.getStyleCollection(styleFamily);

                    // Performance: Saving old list style id, before it is removed
                    if (self.isImportFinished() && (styleFamily === 'paragraph')) {
                        paraAttrs = AttributeUtils.getExplicitAttributes(startInfo.node);
                        if (paraAttrs) { oldListStyleId = getListStyleIdFromParaAttrs(paraAttrs); }
                    }

                    // assigning character attributes to text spans, not to paragraphs (41250)
                    if (!app.isODF() && styleFamily === 'paragraph' && 'character' in attributes) {
                        _.each($(startInfo.node).children('span'), function (node) {
                            // setting the character attributes (only those!) to the text span(s) inside the paragraph
                            if (DOM.isEmptySpan(node)) { setElementAttributes(node, { character: _.copy(attributes.character, true) }); }
                        });

                        self.implParagraphChanged(startInfo.node);

                        // removing the character attributes, so that they are not assigned to the paragraph
                        modifiedAttributes = _.copy(attributes, true);  // not modifying the original object
                        delete modifiedAttributes.character;

                        if (!_.isEmpty(modifiedAttributes)) { setElementAttributes(startInfo.node, modifiedAttributes); }

                    } else {

                        setElementAttributes(startInfo.node);

                        // update also the neighbors, if the list level was modified (43897)
                        handleNeighborBorderUpdate(startInfo.node, attributes);
                    }

                }

                // create the undo action
                if (undoManager.isUndoEnabled()) {
                    redoOperation = { name: Operations.SET_ATTRIBUTES, start: start, end: end, attrs: attributes };
                    // extend redoOperation if target is present
                    self.extendPropertiesWithTarget(redoOperation, target);
                    undoManager.addUndo(undoOperations, redoOperation);
                }

                // update numberings and bullets (but updateListsDebounced can only be called after successful document import)
                if (self.isImportFinished() && (styleFamily === 'paragraph')) {
                    if ((('styleId' in attributes) || _.isObject(attributes.paragraph) && (('listLevel' in attributes.paragraph) || ('listStyleId' in attributes.paragraph)))) {

                        // determining listStyleId und listLevel
                        if ((_.isObject(attributes.paragraph)) && ('listStyleId' in attributes.paragraph) && (attributes.paragraph.listStyleId !== null)) {
                            listStyleId = attributes.paragraph.listStyleId;  // list style assigned to paragraph
                            listLevel = ('listLevel' in attributes.paragraph) ? attributes.paragraph.listLevel : 0;
                        } else if (attributes.styleId && self.isParagraphStyleWithListStyle(attributes.styleId)) {
                            listStyleId = getListStyleInfoFromStyleId(attributes.styleId, 'listStyleId');
                            listLevel = getListStyleInfoFromStyleId(attributes.styleId, 'listLevel') || 0;
                        } else {
                            // list level modified -> checking paragraph attributes for list styles to receive list style id
                            paraAttrs = AttributeUtils.getExplicitAttributes(startInfo.node);
                            // updating lists, if required
                            if (self.isListStyleParagraph(null, paraAttrs)) {
                                if (paraAttrs && paraAttrs.paragraph) {
                                    listStyleId = paraAttrs.paragraph.listStyleId;
                                    listLevel = paraAttrs.paragraph.listLevel;
                                }
                            }
                        }

                        // handling the case, that the list style id is removed
                        if (!listStyleId && oldListStyleId) { listStyleId = oldListStyleId; }

                        // defining list style Id or listLevel, that need to be updated
                        if (listStyleId || listLevel) {
                            if (!listStyleId) {
                                listStyleId = self.getParagraphStyles().getElementAttributes(startInfo.node).paragraph.listStyleId;
                                //fix for Bug 37594 styleId is not at the para-attrs and not in the style it self. but by merging we get the correct stlyeId
                            }
                            // mark this paragraph for later list update. This is necessary, because it does not yet contain
                            // the list label node (DOM.LIST_LABEL_NODE_SELECTOR) and is therefore ignored in updateLists.
                            $(startInfo.node).data('updateList', 'true');
                            // registering this list style for update
                            // special handling for removal of listLevel, for example after 'Backspace'
                            if (listStyleId && (listLevel === null || listLevel === -1)) {
                                $(startInfo.node).data('removeLabel', 'true');
                            }

                            // not registering listStyle for updating, if this is a bullet list -> updating only specific paragraphs
                            if (self.getListCollection().isAllLevelsBulletsList(listStyleId)) { listStyleId = null; }

                            self.updateListsDebounced({ useSelectedListStyleIDs: true, paraInsert: true, listStyleId: listStyleId, listLevel: listLevel });
                        }
                    } else if (self.useSlideMode()) {

                        if (_.isObject(attributes.paragraph) && ('level' in attributes.paragraph || 'bullet' in attributes.paragraph)) {
                            self.updateListsDebounced($(startInfo.node));
                        }

                    } else if ('character' in attributes) {  // modified character attributes may lead to modified list labels

                        // checking paragraph attributes for list styles to receive list style id
                        paraAttrs = AttributeUtils.getExplicitAttributes(startInfo.node);
                        // updating lists, after character attributes were modified
                        if (self.isListStyleParagraph(null, paraAttrs)) {
                            if (paraAttrs && paraAttrs.paragraph) {
                                listStyleId = paraAttrs.paragraph.listStyleId;
                                listLevel = paraAttrs.paragraph.listLevel;
                                self.updateListsDebounced({ useSelectedListStyleIDs: true, paraInsert: true, listStyleId: listStyleId, listLevel: listLevel });
                            }
                        }
                    }
                }

                if (app.isODF() && (styleFamily === 'paragraph') && (attributes.character) && (Position.getParagraphNodeLength(startInfo.node) === 0)) {
                    // Task 28187: Setting character attributes at empty paragraphs
                    // -> removing character attributes from text span so that character styles at paragraph become visible (also 30927)
                    nullAttributes = _.copy(attributes, true);
                    for (localkey in nullAttributes.character) { nullAttributes.character[localkey] = null; }
                    self.getCharacterStyles().setElementAttributes(DOM.findFirstPortionSpan(startInfo.node), nullAttributes);
                }

                if (self.isImportFinished()) {
                    // adjust tabulators, if character or drawing attributes have been changed
                    // (changing paragraph or table attributes updates tabulators automatically)
                    if ((styleFamily === 'character') || (styleFamily === 'drawing')) {
                        self.getParagraphStyles().updateTabStops(startInfo.node.parentNode);

                        if (self.useSlideMode()) {
                            self.updateListsDebounced($(startInfo.node.parentNode), { singleParagraph: true }); // Updating list only the current paragraph
                        }
                    }

                    // if target comes with operation, take care of targeted header/footer, otherwise, run normal page break rendering
                    if (target) {
                        // trigger header/footer content update on other elements of same type, if change was made inside header/footer
                        if (DOM.isMarginalNode(startInfo.node)) {
                            self.updateEditingHeaderFooterDebounced(DOM.getMarginalTargetNode(self.getNode(), startInfo.node));
                        } else if (DOM.isMarginalNode(startInfo.node.parentNode)) {
                            self.updateEditingHeaderFooterDebounced(DOM.getMarginalTargetNode(self.getNode(), startInfo.node.parentNode));
                        }
                        self.setBlockOnInsertPageBreaks(true);
                    }

                    // after document is loaded, trigger pagebreaks reposition on certain events like font size change, height of drawing or row changed
                    isCharOrDrawingOrRow = 'character' in attributes || 'drawing' in attributes || 'row' in attributes || ('line' in attributes && 'width' in attributes.line);
                    isClearFormatting = 'styleId' in attributes && attributes.styleId === null;
                    if (_.isObject(attributes.paragraph)) {
                        isAttributesInParagraph = ('lineHeight' in attributes.paragraph) || ('listLevel' in attributes.paragraph) || ('listStyleId' in attributes.paragraph);
                        isPageBreakBeforeAttributeInParagraph = ('pageBreakBefore' in attributes.paragraph);
                        isParSpacing = 'marginBottom' in attributes.paragraph;
                    }
                    if ((isCharOrDrawingOrRow || isClearFormatting || isAttributesInParagraph || isPageBreakBeforeAttributeInParagraph || isParSpacing)) {
                        if (start.length > 1) {
                            // if its paragraph creation inside of table
                            currentElement = Position.getContentNodeElement(rootNode, start.slice(0, 1));
                        } else {
                            currentElement = Position.getContentNodeElement(rootNode, start);
                        }
                        if ($(currentElement).data('lineBreaksData')) {
                            $(currentElement).removeData('lineBreaksData'); //we changed paragraph layout, cached line breaks data needs to be invalidated
                        }
                        // attribute removed by user, needs to refresh page layout
                        if (isPageBreakBeforeAttributeInParagraph && !attributes.paragraph.pageBreakBefore) {
                            $(startInfo.node).removeClass('manual-page-break'); // remove class from div.p element
                            $(currentElement).removeClass('manual-page-break'); // remove class from parent table element also
                        }
                        self.insertPageBreaks(currentElement, DrawingFrame.getClosestTextFrameDrawingNode(startInfo.node));
                        if (!self.isPageBreakMode()) { app.getView().recalculateDocumentMargin(); }
                    }
                }

                // Special handling for presentation in master/layout slides -> updating styles
                if (target && _.isFunction(self.isLayoutOrMasterId) && self.isLayoutOrMasterId(target) && forceLayoutFormattingData) { self.forceMasterLayoutUpdate(forceLayoutFormattingData); }

                return true;
            };
        }());

        // public methods -----------------------------------------------------

        /**
         * Changes multiple attributes of the specified attribute family in the
         * current selection.
         *
         * @param {String} family
         *  The name of the attribute family containing the passed attributes.
         *
         * @param {Object} attributes
         *  A map of attribute value maps (name/value pairs), keyed by
         *  attribute families.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.clear=false]
         *      If set to true, all existing explicit attributes will be
         *      removed from the selection while applying the new attributes.
         */
        this.setAttributes = function (family, attributes, options) {

            var // the undo manager
                undoManager = self.getUndoManager();

            // Create an undo group that collects all undo operations generated
            // in the local setAttributes() method (it calls itself recursively
            // with smaller parts of the current selection).
            undoManager.enterUndoGroup(function () {

                var // table or drawing element contained by the selection
                    element = null,
                    // operations generator
                    generator = self.createOperationsGenerator(),
                    // the style sheet container
                    styleSheets = self.getStyleCollection(family),
                    // logical position
                    localPosition = null,
                    // another logical position
                    localDestPosition = null,
                    // the length of the paragraph
                    paragraphLength = null,
                    // whether after assigning cell attributes, it is still necessary to assign table attributes
                    createTableOperation = true,
                    // paragraph helper position and helper node
                    paraPos = null, paraNode = null,
                    // the old attributes, required for change tracking
                    oldAttrs = null,
                    // whether the modification can be change tracked
                    isTrackableChange = true,
                    // currently active root node
                    activeRootNode = self.getCurrentRootNode(),
                    // if we apply new style, do not remove manual pagebreaks
                    pageBreakBeforeAttr = false,
                    pageBreakAfterAttr = false,
                    // the old attributes, required for pageBreak paragraph style
                    oldParAttr = null,
                    // the promise for the asychronous execution of operations
                    operationsPromise = null,
                    // the promise for generating the operations
                    operationGeneratorPromise = null,
                    // the complete promise for generation and applying of operations
                    setAttributesPromise = null,
                    // the ratio of operation generation to applying of operation
                    operationRatio = 0.3,
                    // an optional array with selection ranges for large selections
                    splittedSelectionRange = null,
                    // a snapshot object
                    snapshot = null,
                    // the selection object
                    selection = self.getSelection();

                /**
                 * Helper function to generate operations. This is done synchronously. But for asynchronous usage,
                 * the selection range can be restricted. This is especially useful for large selections.
                 *
                 * @param {Number[][]} [selectionRange]
                 *  An array with two logical positions for the start and the end position of a selection range.
                 */
                function generateAllSetAttributeOperations(selectionRange) {

                    var // the logical start position, if specified
                        localStartPos = (selectionRange && selectionRange[0]) || null,
                        // the logical end position, if specified
                        localEndPos = (selectionRange && selectionRange[1]) || null,
                        // the change track object
                        changeTrack = self.getChangeTrack();

                    // generates a 'setAttributes' operation with the correct attributes
                    function generateSetAttributeOperation(startPosition, endPosition, oldAttrs) {

                        var // the options for the operation
                            operationOptions = { start: startPosition, attrs: _.clone(attributes) };

                        // adding the old attributes, author and date for change tracking
                        if (oldAttrs) {
                            oldAttrs = _.extend(oldAttrs, changeTrack.getChangeTrackInfo());
                            operationOptions.attrs.changes = { modified: oldAttrs };
                        }

                        // add end position if specified
                        if (_.isArray(endPosition)) {
                            operationOptions.end = endPosition;
                        }
                        // paragraph's style manual page break before
                        if (pageBreakBeforeAttr) {
                            operationOptions.attrs.paragraph.pageBreakBefore = true;
                            pageBreakBeforeAttr = false;
                        }
                        // paragraph's style manual page break after
                        if (pageBreakAfterAttr) {
                            operationOptions.attrs.paragraph.pageBreakAfter = true;
                            pageBreakAfterAttr = false;
                        }

                        // generate the 'setAttributes' operation
                        generator.generateOperation(Operations.SET_ATTRIBUTES, operationOptions);
                    }

                    // setting a specified selection, if defined
                    // if (selectionRange) { selection.setTextSelection.apply(selection, selectionRange); }

                    // generate 'setAttribute' operations
                    switch (family) {

                        case 'character':
                            if (selection.hasRange()) {
                                selection.iterateContentNodes(function (paragraph, position, startOffset, endOffset) {

                                    // validate start offset (iterator passes 'undefined' for fully covered paragraphs)
                                    if (!_.isNumber(startOffset)) {
                                        startOffset = 0;
                                    }
                                    // validate end offset (iterator passes 'undefined' for fully covered paragraphs)
                                    if (!_.isNumber(endOffset)) {
                                        endOffset = Position.getParagraphNodeLength(paragraph) - 1;
                                    }

                                    if (changeTrack.isActiveChangeTracking()) {

                                        // iterating over all children to send the old attributes in the operation
                                        // remove all empty text spans which have sibling text spans, and collect
                                        // sequences of sibling text spans (needed for white-space handling)
                                        Position.iterateParagraphChildNodes(paragraph, function (node, nodestart, nodelength, offsetstart, offsetlength) {

                                            // evaluate only text spans that are partly or completely covered by selection
                                            if (DOM.isTextSpan(node)) {
                                                // one operation for each span
                                                generateSetAttributeOperation(position.concat([nodestart + offsetstart]), position.concat([nodestart + offsetstart + offsetlength - 1]), changeTrack.getOldNodeAttributes(node));
                                            }

                                        }, undefined, { start: startOffset, end: endOffset });

                                    } else {

                                        // set the attributes at the covered text range
                                        if (startOffset === 0 && endOffset <= 0 && DOM.isImplicitParagraphNode(paragraph)) {
                                            generator.generateOperation(Operations.PARA_INSERT, { start: position });  // generating paragraph
                                            generateSetAttributeOperation(position);  // assigning attribute to new paragraph
                                        } else if (startOffset === 0 && endOffset <= 0 && Position.getParagraphNodeLength(paragraph) === 0) {
                                            generateSetAttributeOperation(position);  // assigning attribute to existing empty paragraph
                                        } else if (startOffset <= endOffset) {
                                            generateSetAttributeOperation(position.concat([startOffset]), position.concat([endOffset]));
                                        }

                                    }
                                    self.getSpellChecker().reset(attributes, paragraph);
                                }, null, { startPos: localStartPos, endPos: localEndPos });
                            } else {  // selection has no range
                                if (Position.getParagraphLength(activeRootNode, _.clone(selection.getStartPosition())) === 0) {
                                    // Task 28187: Setting character attributes at empty paragraphs
                                    // In implSetAttributes it is additionally necessary to remove the character attributes
                                    // from text span so that character styles at paragraph become visible

                                    paraPos = _.clone(selection.getStartPosition());
                                    paraPos.pop();
                                    paraNode = Position.getParagraphElement(activeRootNode, paraPos);

                                    // Defining the oldAttrs, if change tracking is active
                                    if (changeTrack.isActiveChangeTracking()) {
                                        // Creating one setAttribute operation for each span inside the paragraph, because of old attributes!
                                        oldAttrs = {};
                                        if (DOM.isParagraphNode(paraNode)) { oldAttrs = changeTrack.getOldNodeAttributes(paraNode); }
                                    }

                                    if (DOM.isImplicitParagraphNode(paraNode)) { generator.generateOperation(Operations.PARA_INSERT, { start: paraPos }); }
                                    // generating setAttributes operation
                                    generateSetAttributeOperation(_.initial(_.clone(selection.getStartPosition())), undefined, oldAttrs);
                                } else {
                                    // using preselected attributes for non-empty paragraphs
                                    self.addPreselectedAttributes(attributes);
                                }
                            }
                            break;

                        case 'paragraph':
                            // deleted all changes for bug #26454#, because it did the opposite of word behavior, and that is not what the user expects! @see Bug 37174

                            selection.iterateContentNodes(function (paragraph, position) {
                                // Defining the oldAttrs, if change tracking is active
                                if (changeTrack.isActiveChangeTracking()) {
                                    // Expanding operation for change tracking with old explicit attributes
                                    oldAttrs = changeTrack.getOldNodeAttributes(paragraph);
                                }
                                // Preserve pageBreakBefore and/or pageBreakAfter attribute after setting new paragraph style
                                if (DOM.isManualPageBreakNode(paragraph)) {
                                    oldParAttr = AttributeUtils.getExplicitAttributes(paragraph);
                                    if (_.isObject(oldParAttr) && oldParAttr.paragraph) {
                                        if (oldParAttr.paragraph.pageBreakBefore === true) {
                                            pageBreakBeforeAttr = true;
                                        }
                                        if (oldParAttr.paragraph.pageBreakAfter === true) {
                                            pageBreakAfterAttr = true;
                                        }
                                    }

                                }

                                // generating a new paragraph, if it is implicit
                                if (DOM.isImplicitParagraphNode(paragraph)) { generator.generateOperation(Operations.PARA_INSERT, { start: position }); }
                                // generating setAttributes operation
                                generateSetAttributeOperation(position, undefined, oldAttrs);
                            }, null, { startPos: localStartPos, endPos: localEndPos });
                            break;

                        case 'cell':
                            selection.iterateTableCells(function (cell, position) {
                                // Defining the oldAttrs, if change tracking is active
                                if (changeTrack.isActiveChangeTracking()) {
                                    // Expanding operation for change tracking with old explicit attributes
                                    oldAttrs = changeTrack.getOldNodeAttributes(cell);
                                }
                                generateSetAttributeOperation(position, undefined, oldAttrs);
                            });
                            break;

                        case 'table':
                            if ((element = selection.getEnclosingTable())) {

                                localPosition = Position.getOxoPosition(activeRootNode, element, 0);  // the logical position of the table

                                if (Utils.getBooleanOption(options, 'clear', false)) {
                                    // removing hard attributes at tables and cells, so that the table style will be visible
                                    Table.removeTableAttributes(self, activeRootNode, element, localPosition, generator);
                                }

                                if (Utils.getBooleanOption(options, 'onlyVisibleBorders', false)) {
                                    // setting border width directly at cells -> overwriting values of table style
                                    Table.setBorderWidthToVisibleCells(element, attributes, self.getTableCellStyles(), activeRootNode, generator, self);
                                    createTableOperation = false;
                                }

                                if (Utils.getBooleanOption(options, 'cellSpecificTableAttribute', false)) {
                                    // setting border mode directly at cells -> overwriting values of table style
                                    Table.setBorderModeToCells(element, attributes, TableStyles.getBorderStyleFromAttributes(self.getAttributes('cell').cell || {}), activeRootNode, generator, self);
                                }

                                // setting attributes to the table element
                                // -> also setting table attributes, if attributes are already assigned to table cells, to
                                // keep the getter functions simple and performant
                                if (createTableOperation) {
                                    // Defining the oldAttrs, if change tracking is active
                                    if (changeTrack.isActiveChangeTracking()) {
                                        // Expanding operation for change tracking with old explicit attributes
                                        oldAttrs = changeTrack.getOldNodeAttributes(element);
                                    }
                                    generateSetAttributeOperation(localPosition, undefined, oldAttrs);
                                }
                            }
                            break;

                        case 'drawing':
                            if (self.isDrawingSelected()) {
                                element = selection.getSelectedDrawing()[0];
                            } else if (self.getSelection().isAdditionalTextframeSelection()) {
                                element = selection.getSelectedTextFrameDrawing()[0];
                            } else {
                                break;
                            }
                            // TODO: needs change when multiple drawings can be selected
                            // TODO: this fails if a drawing style sheet changes the inline/floating mode instead of explicit attributes
                            if (DrawingFrame.isDrawingFrame(element)) {

                                localPosition = Position.getOxoPosition(activeRootNode, element, 0);

                                if (_.isObject(attributes.drawing)) {

                                    // when switching from inline to floated, saving current position in the drawing, so that it can
                                    // be set correctly when switching back to inline.
                                    // This is only necessary, if the drawing was moved in that way, that implMove needed to be called.
                                    if ((attributes.drawing.inline === false) && (DOM.isInlineDrawingNode(element))) {
                                        $(element).data('inlinePosition', localPosition[localPosition.length - 1]);
                                        // Fixing vertical offset, if drawing is set from inline to floated (31722)
                                        if ((attributes.drawing.anchorVertBase) && (attributes.drawing.anchorVertBase === 'paragraph') &&
                                            (_.isNumber(attributes.drawing.anchorVertOffset)) && (attributes.drawing.anchorVertOffset === 0) &&
                                            (($(element).offset().top - $(element).closest(DOM.PARAGRAPH_NODE_SELECTOR).offset().top > 0))) {
                                            attributes.drawing.anchorVertOffset = Utils.convertLengthToHmm($(element).offset().top - $(element).closest(DOM.PARAGRAPH_NODE_SELECTOR).offset().top, 'px');
                                        }
                                    }

                                    // when switching from floated to inline, a move of the drawing might be necessary
                                    if ((attributes.drawing.inline === true) && (DOM.isFloatingDrawingNode(element)) && ($(element).data('inlinePosition'))) {

                                        localDestPosition = _.clone(localPosition);
                                        paragraphLength = Position.getParagraphLength(activeRootNode, localDestPosition);
                                        if ((paragraphLength - 1) < $(element).data('inlinePosition')) {  // -> is this position still valid?
                                            localDestPosition[localDestPosition.length - 1] = paragraphLength - 1;
                                        } else {
                                            localDestPosition[localDestPosition.length - 1] = $(element).data('inlinePosition');
                                        }
                                        if (!_.isEqual(localPosition, localDestPosition)) {
                                            undoManager.enterUndoGroup(function () {

                                                var // the logical position of the paragraph
                                                    paraPos = _.clone(localDestPosition),
                                                    // the paragraph node
                                                    paraNode = null;

                                                paraPos.pop();
                                                paraNode = Position.getParagraphElement(activeRootNode, paraPos);

                                                if (DOM.isImplicitParagraphNode(paraNode)) { generator.generateOperation(Operations.PARA_INSERT, { start: paraPos }); }
                                                generator.generateOperation(Operations.MOVE, { start: localPosition, end: localPosition, to: localDestPosition });

                                                localPosition = _.clone(localDestPosition);
                                            }, this);
                                        }
                                    }

                                    // setting new position (inline, paragraph, ...) is not change tracked
                                    if (!changeTrack.ctSupportsDrawingPosition()) { isTrackableChange = false; }
                                }

                                // setting border and fillcolor of drawing is not change tracked
                                if (_.isObject(attributes.fill) || _.isObject(attributes.line)) { isTrackableChange = false; }

                                // Defining the oldAttrs, if change tracking is active
                                // -> and if change track supports changing of drawing position
                                if (changeTrack.isActiveChangeTracking() && isTrackableChange) {
                                    // Expanding operation for change tracking with old explicit attributes
                                    oldAttrs = changeTrack.getOldNodeAttributes(element);
                                }

                                // by setting fill- or line-attributes on a group
                                if (DrawingFrame.isGroupDrawingFrame(element) && (_.isObject(attributes.fill) || _.isObject(attributes.line))) {
                                    // generate operations for all children
                                    _.each(DrawingFrame.getAllGroupDrawingChildren(element), function (ele) {
                                        generateSetAttributeOperation(Position.getOxoPosition(activeRootNode, ele, 0));
                                    });
                                // otherwise
                                } else {
                                    // set attributes on the group
                                    generateSetAttributeOperation(localPosition, undefined, oldAttrs);
                                }

                                $(element).closest(DOM.PARAGRAPH_NODE_SELECTOR).removeData('lineBreaksData'); //we changed paragraph layout, cached line breaks data needs to be invalidated

                            }
                            break;

                        default:
                            Utils.error('Editor.setAttributes(): missing implementation for family "' + family + '"');
                    }
                }

                /**
                 * Helper function to apply the generated operations asynchronously.
                 *
                 * @returns {jQuery.Promise}
                 *  A promise that will be resolved when the operations have been applied.
                 */
                function doSetAttributesAsync() {

                    operationsPromise = self.applyTextOperationsAsync(generator, null, { showProgress: false, leaveOnSuccess: true, progressStart: operationRatio });

                    // restore the original selection after successful apply of operations
                    operationsPromise
                    .done(function () {
                        self.getSelection().restoreBrowserSelection();
                    });

                    return operationsPromise;
                }

                // add all attributes to be cleared
                if (Utils.getBooleanOption(options, 'clear', false)) {
                    if (family !== 'table') {  // special behaviour for tables follows below
                        attributes = self.extendAttributes(styleSheets.buildNullAttributes(), attributes);
                    }
                }

                // nothig to do if no attributes will be changed
                if (_.isEmpty(attributes)) { return; }

                // register pending style sheet via 'insertStyleSheet' operation
                if (_.isString(attributes.styleId) && styleSheets.isDirty(attributes.styleId)) {
                    self.generateInsertStyleOp(generator, family, attributes.styleId);
                }

                // Special handling for presentation in master/layout slides -> selecting only complete paragraphs
                if (self.getActiveTarget() && _.isFunction(self.isLayoutOrMasterId) && self.isLayoutOrMasterId(self.getActiveTarget())) { self.setMasterLayoutSelection(); }

                // checking selection size -> make array with splitted selection [0,100], [101, 200], ... (39061)
                splittedSelectionRange = Position.splitLargeSelection(self.getCurrentRootNode(), selection.getStartPosition(), selection.getEndPosition(), self.getMaxTopLevelNodes());

                if (splittedSelectionRange && splittedSelectionRange.length > 1) {

                    // make sure that only one asynchronous call is processed at the same time
                    if (self.checkSetClipboardPasteInProgress()) { return $.when(); }

                    // blocking keyboard input during generation and applying of operations
                    self.setBlockKeyboardEvent(true);

                    // creating a snapshot
                    snapshot = new Snapshot(app);

                    // show a message with cancel button
                    // -> immediately grabbing the focus, after calling enterBusy. This guarantees, that the
                    // keyboard blocker works. Otherwise the keyboard events will not be catched by the page.
                    app.getView().enterBusy({
                        cancelHandler: function () {
                            if (operationsPromise && operationsPromise.abort) { // order is important, the latter has to win
                                // restoring the old document state
                                snapshot.apply();
                                // calling abort function for operation promise
                                app.enterBlockOperationsMode(function () { operationsPromise.abort(); });
                            } else if (operationGeneratorPromise && operationGeneratorPromise.abort) {
                                operationGeneratorPromise.abort();  // no undo of changes required
                            }
                        },
                        immediate: true,
                        warningLabel: gt('Sorry, formatting content will take some time.')
                    }).grabFocus();

                    // generate operations asynchronously
                    operationGeneratorPromise = self.iterateArraySliced(splittedSelectionRange, function (oneRange) {
                        // calling function to generate operations synchronously with reduced selection range
                        generateAllSetAttributeOperations(oneRange);
                    }, { delay: 'immediate', infoString: 'Text: generateAllSetAttributeOperations' })
                        // add progress handling
                        .progress(function (progress) {
                            // update the progress bar according to progress of the operations promise
                            app.getView().updateBusyProgress(operationRatio * progress);
                        })
                        .fail(function () {
                            // leaving the busy mode during creation of operations -> no undo required
                            self.leaveAsyncBusy();
                        });

                    setAttributesPromise = operationGeneratorPromise.then(doSetAttributesAsync).always(function () {
                        if (snapshot) { snapshot.destroy(); }
                    });

                } else {
                    // synchronous handling for small selections

                    // calling function to generate operations synchronously
                    generateAllSetAttributeOperations();

                    // apply all collected operations
                    self.applyOperations(generator);

                    // applying operations synchronously
                    setAttributesPromise = $.when();
                }

                return setAttributesPromise;

            }, this); // enterUndoGroup();
        };

        /**
         * Changes a single attribute of the specified attribute family in the
         * current selection.
         *
         * @param {String} family
         *  The name of the attribute family containing the specified
         *  attribute.
         *
         * @param {String} name
         *  the key of the attribute in the assigned family
         *
         * @param {Object} value
         *  the new attributes
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.clear=false]
         *      If set to true, all existing explicit attributes will be
         *      removed from the selection, before applying the new attributes.
         */
        this.setAttribute = function (family, name, value, options) {
            self.setAttributes(family, Utils.makeSimpleObject(family, Utils.makeSimpleObject(name, value)), options);
        };

        /**
         * Changes multiple attributes of the specified attribute family for
         * the passed table node. Different from setAttributes in a way that uses
         * passed node instead of selected node with selection.
         *
         * @param {Node|jQuery} tableNode
         *  Table node to which attributes are applied
         *
         * @param {String} family
         *  The name of the attribute family containing the passed attributes.
         *
         * @param {Object} attributes
         *  A map of attribute value maps (name/value pairs), keyed by
         *  attribute families.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.clear=false]
         *      If set to true, all existing explicit attributes will be
         *      removed from the selection while applying the new attributes.
         *  @param {Boolean} [options.onlyVisibleBorders=false]
         *      Setting border width directly at cells -> overwriting values of table style.
         *  @param {Boolean} [options.cellSpecificTableAttribute=false]
         *      Setting border mode directly at cells -> overwriting values of table style.
         *
         */
        this.setAttributesToPassedTableNode = function (tableNode, family, attributes, options) {

            var // the undo manager
                undoManager = self.getUndoManager();

            // Create an undo group that collects all undo operations generated
            // in the local setAttributes() method (it calls itself recursively
            // with smaller parts of the current selection).
            undoManager.enterUndoGroup(function () {

                var // logical position
                    localPosition = null,
                    // operations generator
                    generator = self.createOperationsGenerator(),
                    // whether after assigning cell attributes, it is still necessary to assign table attributes
                    createTableOperation = true,
                    // the page node
                    editdiv = self.getNode(),
                    // the old attributes, required for change tracking
                    oldAttrs = null;

                // generates a 'setAttributes' operation with the correct attributes
                function generateSetAttributeOperation(startPosition, endPosition, oldAttrs) {

                    var // the options for the operation
                        operationOptions = { start: startPosition, attrs: _.clone(attributes) };

                    // adding the old attributes, author and date for change tracking
                    if (oldAttrs) {
                        oldAttrs = _.extend(oldAttrs, self.getChangeTrack().getChangeTrackInfo());
                        operationOptions.attrs.changes = { modified: oldAttrs };
                    }

                    // add end position if specified
                    if (_.isArray(endPosition)) {
                        operationOptions.end = endPosition;
                    }

                    // generate the 'setAttributes' operation
                    generator.generateOperation(Operations.SET_ATTRIBUTES, operationOptions);
                }

                // nothig to do if no attributes will be changed
                if (_.isEmpty(attributes)) { return; }

                if (tableNode) {

                    localPosition = Position.getOxoPosition(editdiv, tableNode, 0);  // the logical position of the table

                    if (Utils.getBooleanOption(options, 'clear', false)) {
                        // removing hard attributes at tables and cells, so that the table style will be visible
                        Table.removeTableAttributes(self, editdiv, tableNode, localPosition, generator);
                    }

                    if (Utils.getBooleanOption(options, 'onlyVisibleBorders', false)) {
                        // setting border width directly at cells -> overwriting values of table style
                        Table.setBorderWidthToVisibleCells(tableNode, attributes, self.getTableCellStyles(), editdiv, generator, self);
                        createTableOperation = false;
                    }

                    if (Utils.getBooleanOption(options, 'cellSpecificTableAttribute', false)) {
                        // setting border mode directly at cells -> overwriting values of table style
                        Table.setBorderModeToCells(tableNode, attributes, TableStyles.getBorderStyleFromAttributes(self.getAttributes('cell').cell || {}), editdiv, generator, self);
                    }

                    // setting attributes to the table element
                    // -> also setting table attributes, if attributes are already assigned to table cells, to
                    // keep the getter functions simple and performant
                    if (createTableOperation) {
                        // Defining the oldAttrs, if change tracking is active
                        if (self.getChangeTrack().isActiveChangeTracking()) {
                            // Expanding operation for change tracking with old explicit attributes
                            oldAttrs = self.getChangeTrack().getOldNodeAttributes(tableNode);
                        }
                        generateSetAttributeOperation(localPosition, undefined, oldAttrs);
                    }
                }

                // apply all collected operations
                self.applyOperations(generator);

            }, this); // end of enterUndoGroup
        };

        /**
         * Returns the values of all formatting attributes of the elements in
         * the current selection associated to the specified attribute family.
         *
         * @param {String} family
         *  The name of the attribute family used to select specific elements
         *  in the current selection:
         *  - 'character': all text spans (text portions, text components),
         *  - 'paragraph': all paragraph nodes,
         *  - 'table': all table nodes,
         *  - 'drawing': all drawing object nodes.
         *
         * @returns {Object}
         *  A map of attribute value maps (name/value pairs), keyed by
         *  attribute family.
         */
        this.getAttributes = function (family, options) {

            var // the selection object
                selection = self.getSelection(),
                // whether the selection is a simple cursor
                isCursor = selection.isTextCursor(),
                // table or drawing element contained by the selection
                element = null,
                // resulting merged attributes
                mergedAttributes = null,
                // max iterations
                maxIterations = Utils.getOption(options, 'maxIterations'),
                // get only attributes of the group
                groupOnly = Utils.getBooleanOption(options, 'groupOnly', false);

            // merges the passed element attributes into the resulting attributes
            function mergeElementAttributes(elementAttributes) {

                var // whether any attribute is still unambiguous
                    hasNonNull = false;

                // merges the passed attribute value into the attributes map
                function mergeAttribute(attributes, name, value) {
                    if (!(name in attributes)) {
                        // initial iteration: store value
                        attributes[name] = value;
                    } else if (!_.isEqual(value, attributes[name])) {
                        // value differs from previous value: ambiguous state
                        attributes[name] = null;
                    }
                    hasNonNull = hasNonNull || !_.isNull(attributes[name]);
                }

                // initial iteration: store attributes and return
                if (!mergedAttributes) {
                    mergedAttributes = elementAttributes;
                    return;
                }

                // process all passed element attributes
                _(elementAttributes).each(function (attributeValues, subFamily) {
                    if (subFamily === 'styleId') {
                        mergeAttribute(mergedAttributes, 'styleId', attributeValues);
                    } else {
                        var mergedAttributeValues = mergedAttributes[subFamily];
                        _(attributeValues).each(function (value, name) {
                            mergeAttribute(mergedAttributeValues, name, value);
                        });
                    }
                });

                // stop iteration, if all attributes are ambiguous
                return hasNonNull ? undefined : Utils.BREAK;
            }

            switch (family) {

                case 'character':
                    selection.iterateNodes(function (node) {
                        return DOM.iterateTextSpans(node, function (span) {
                            // ignore empty text spans (they cannot be formatted via operations),
                            // but get formatting of an empty span selected by a text cursor
                            if (isCursor || (span.firstChild.nodeValue.length > 0)) {
                                return mergeElementAttributes(self.getCharacterStyles().getElementAttributes(span));
                            }
                        });
                    }, null, { maxIterations: maxIterations });
                    if (isCursor && self.getPreselectedAttributes()) {
                        // add preselected attributes (text cursor selection cannot result in ambiguous attributes)
                        self.extendAttributes(mergedAttributes, self.getPreselectedAttributes());
                    }
                    break;

                case 'paragraph':
                    selection.iterateContentNodes(function (paragraph) {
                        return mergeElementAttributes(self.getParagraphStyles().getElementAttributes(paragraph));
                    }, null, { maxIterations: maxIterations });
                    break;

                case 'cell':
                    selection.iterateTableCells(function (cell) {
                        return mergeElementAttributes(self.getTableCellStyles().getElementAttributes(cell));
                    });
                    break;

                case 'table':
                    if ((element = selection.getEnclosingTable())) {
                        mergeElementAttributes(self.getTableStyles().getElementAttributes(element));
                    }
                    break;

                case 'drawing':
                    var doMergeElementAttributes = false;
                    // TODO: needs change when multiple drawings can be selected
                    if ((element = selection.getSelectedDrawing()[0]) && DrawingFrame.isDrawingFrame(element)) {
                        doMergeElementAttributes = true;
                    } else if ((element = selection.getSelectedTextFrameDrawing()[0]) && DrawingFrame.isDrawingFrame(element)) {
                        doMergeElementAttributes = true;
                    }

                    if (doMergeElementAttributes) {
                        if (DrawingFrame.isGroupDrawingFrame(element) && groupOnly !== true) {
                            _.each(DrawingFrame.getAllGroupDrawingChildren(element), function (drawing) {
                                mergeElementAttributes(self.getDrawingStyles().getElementAttributes(drawing));
                            });
                        } else {
                            mergeElementAttributes(self.getDrawingStyles().getElementAttributes(element));
                        }
                    }
                    break;

                default:
                    Utils.error('Editor.getAttributes(): missing implementation for family "' + family + '"');
            }

            return mergedAttributes || {};
        };

        /**
         * Removes all hard-set attributes depending on the current selection.
         * A selection range clears only hard-set character attributes and without
         * a range the hard-set paragraph attributes are cleared.
         */
        this.resetAttributes = function () {

            var // the undo manager
                undoManager = self.getUndoManager(),
                // the selection object
                selection = self.getSelection();

            function clearParagraphAttributes() {
                var paraAttributes = self.getParagraphStyles().buildNullAttributes();
                delete paraAttributes.styleId;
                self.setAttributes('paragraph', paraAttributes);
            }

            function clearCharacterAttributes() {
                var charAttributes = self.getCharacterStyles().buildNullAttributes();
                // don't reset hyperlink attribute
                delete charAttributes.styleId;
                delete charAttributes.character.url;
                self.setAttributes('character', charAttributes);
            }

            if (!selection.hasRange()) {
                undoManager.enterUndoGroup(function () {

                    var start = selection.getStartPosition(),
                        end = selection.getEndPosition(),
                        activeRootNode = self.getCurrentRootNode(),
                        firstParagraph = Position.getLastNodeFromPositionByNodeName(activeRootNode, start, DOM.PARAGRAPH_NODE_SELECTOR),
                        lastParagraph = Position.getLastNodeFromPositionByNodeName(activeRootNode, end, DOM.PARAGRAPH_NODE_SELECTOR),
                        prevPara,
                        nextPara;

                    clearParagraphAttributes();
                    prevPara = Utils.findPreviousNode(activeRootNode, firstParagraph, DOM.PARAGRAPH_NODE_SELECTOR);
                    if (prevPara) {
                        self.getParagraphStyles().updateElementFormatting(prevPara);
                    }
                    nextPara = Utils.findNextNode(activeRootNode, lastParagraph, DOM.PARAGRAPH_NODE_SELECTOR);
                    if (nextPara) {
                        self.getParagraphStyles().updateElementFormatting(nextPara);
                    }
                    self.addPreselectedAttributes(self.getCharacterStyles().buildNullAttributes());
                }, this);
            } else {
                clearCharacterAttributes();
            }
        };

        // operation handler --------------------------------------------------

        /**
         * The handler for the setAttributes operation.
         *
         * Info:undo/redo generation is done inside 'implSetAttributes'.
         *
         * @param {Object} operation
         *  The operation object.
         *
         * @param {Boolean} external
         *  Whether the operation was triggered by an external client.
         *
         * @returns {Boolean}
         *  Whether the setting of attributes was successful.
         */
        this.setAttributesHandler = function (operation, external) {
            return implSetAttributes(operation.start, operation.end, operation.attrs, operation.target, external);
        };

        // initialization -----------------------------------------------------

        // destroy all class members on destruction
        this.registerDestructor(function () {
            self = null;
        });

    } // class AttributeOperationMixin

    // constants --------------------------------------------------------------

    // exports ================================================================

    return AttributeOperationMixin;
});
