/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define('io.ox/office/textframework/format/tablerowstyles', [
    'io.ox/office/tk/utils',
    'io.ox/office/editframework/model/stylecollection'
], function (Utils, StyleCollection) {

    'use strict';

    var // definitions for table row attributes
        DEFINITIONS = {

            /**
             * The height of the table row. If 'auto' then the row is auto sized.
             */
            height: { def: 'auto' }

        };

    // class TableRowStyles ===================================================

    /**
     * Contains the style sheets for table row formatting attributes. The CSS
     * formatting will be read from and written to <tr> elements.
     *
     * @constructor
     *
     * @extends StyleCollection
     *
     * @param {TextModel} docModel
     *  The text document model containing instance.
     */
    function TableRowStyles(docModel) {

        // base constructor ---------------------------------------------------

        StyleCollection.call(this, docModel, 'row', {
            families: 'changes',
            styleSheetSupport: false,
            formatHandler: updateTableRowFormatting
        });

        // private methods ----------------------------------------------------

        /**
         * Will be called for every table row element whose attributes have been
         * changed. Repositions and reformats the table row according to the passed
         * attributes.
         *
         * @param {jQuery} row
         *  The <tr> element whose table row attributes have been changed,
         *  as jQuery object.
         *
         * @param {Object} mergedAttributes
         *  A map of attribute maps (name/value pairs), keyed by attribute family,
         *  containing the effective attribute values merged from style sheets and
         *  explicit attributes.
         */
        function updateTableRowFormatting(row, mergedAttributes) {

            var // the row attributes of the passed attribute map
                rowAttributes = mergedAttributes.row,
                rowHeight = null,
                cellHeight = null,
                maxHeight = 0,
                paddingHeight = 0;

            if (rowAttributes.height !== 'auto') {
                if (_.browser.WebKit || _.browser.IE) {

                    // IE requires setting of height of div.cell inside the <td>
                    if (_.browser.IE) {
                        // min-height of div.cell must not influence the height of the <td> element -> resetting before resizing
                        row.children('td').children('div.cell').css('min-height', '0px');
                    }

                    // Chrome requires row height at the cells, setting height at <tr> is ignored.
                    rowHeight = Utils.convertHmmToLength(rowAttributes.height, 'px', 1);
                    row.children('td').each(function () {
                        paddingHeight =  Utils.getElementCssLength(this, 'padding-top') +
                                         Utils.getElementCssLength(this, 'padding-bottom') +
                                         Utils.getElementCssLength(this, 'border-bottom-width');

                        cellHeight = (rowHeight - paddingHeight) + 'px';
                        if (_.browser.IE) {
                            maxHeight = Math.max(rowHeight - paddingHeight, maxHeight);
                            $(this).css('min-height', cellHeight);
                        } else {
                            $(this).css('height', cellHeight);
                        }
                    });

                    // IE requires setting of height of div.cell inside the <td>
                    if (_.browser.IE) {
                        // div.cell needs the height of the maximum height of all cells
                        row.children('td').children('div.cell').css('min-height', maxHeight + 'px');
                    }

                } else {
                    // FireFox requires row height at the rows. Setting height at cells, makes
                    // height of cells unchanged, even if text leaves the cell.
                    row.css('height', Utils.convertHmmToCssLength(rowAttributes.height, 'px', 1));
                }
            } else {
                // required for undo, if the table cell was not set before
                if (_.browser.WebKit) {
                    // Chrome requires row height at the cells.
                    row.children('td').css('height', 0);
                } else if (_.browser.IE) {
                    row.children('td').css('min-height', '').children('div.cell').css('min-height', '');
                } else {
                    // FireFox requires row height at the rows.
                    row.css('height', '');
                }
            }

            // change track attribute handling
            docModel.getChangeTrack().updateChangeTrackAttributes(row, mergedAttributes);

        }

        // initialization -----------------------------------------------------

        // register the attribute definitions for the style family
        docModel.registerAttributeDefinitions('row', DEFINITIONS);

        // destroy all class members on destruction
        this.registerDestructor(function () {
            docModel = null;
        });

    } // class TableRowStyles

    // exports ================================================================

    // derive this class from class StyleCollection
    return StyleCollection.extend({ constructor: TableRowStyles });

});
