/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define('io.ox/office/textframework/components/comment/commentlayer', [
    'io.ox/contacts/api',
    'io.ox/office/tk/forms',
    'io.ox/office/tk/utils/dateutils',
    'io.ox/office/tk/utils/driveutils',
    'io.ox/office/tk/object/triggerobject',
    'io.ox/office/tk/object/timermixin',
    'io.ox/office/baseframework/app/appobjectmixin',
    'io.ox/office/drawinglayer/view/drawingframe',
    'io.ox/office/editframework/utils/attributeutils',
    'io.ox/office/textframework/utils/textutils',
    'io.ox/office/textframework/utils/operations',
    'io.ox/office/textframework/utils/dom',
    'io.ox/office/textframework/format/characterstyles',
    'io.ox/office/textframework/utils/position',
    'gettext!io.ox/office/textframework/main'
], function (ContactsAPI, Forms, DateUtils, DriveUtils, TriggerObject, TimerMixin, AppObjectMixin, DrawingFrame, AttributeUtils, Utils, Operations, DOM, CharacterStyles, Position, gt) {

    'use strict';

    // class CommentLayer =====================================================

    /**
     * An instance of this class represents the model for all comments in the
     * edited document.
     *
     * Triggers the following events:
     * - 'commentlayer:created': When the comment layer on the right side of the
     *      document becomes visible.
     * - 'commentlayer:removed': When the comment layer on the right side of the
     *      document is no longer visible.
     * - 'update:authorfilter': When the filter list for the authors is modified.
     * - 'update:commentauthorlist': When the list of author comments is modified.
     *
     * @constructor
     *
     * @extends TriggerObject
     * @extends TimerMixin
     * @extends AppObjectMixin
     *
     * @param {TextApplication} app
     *  The application instance.
     */
    function CommentLayer(app) {

        var // self reference
            self = this,
            // a list of all comments (jQuery), the model
            comments = [],
            // an object with target string as key and target node as value, the model
            allComments = {},
            // temporary list for collecting comments authors during document loading.
            commentsAuthorList = [],
            // whether there is an active filter list for the authors
            isActiveAuthorFilterList = false,
            // a list of authors that can be used for filtering
            authorFilter = null,
            // a sorted list of all comment place holders (jQuery), from start of
            // document to end of document
            commentPlaceHolders = $(),
            // a place holder ID for absolute positioned drawings
            placeHolderCommentID = 0,
            // a counter for marginal comments
            marginalCommentCounter = 0,
            // a marker class to identify the comment connection lines
            commentLineClass = 'isCommentConnectionLine',
            // a selector to identify the comment connection lines
            commentLineSelector = '.' + commentLineClass,
            // the paragraph style name for comments
            commentParagraphStyleName = 'annotation text',
            // the text model object
            model = null,
            // the undo manager
            undoManager = null,
            // the models page layout object
            pageLayout = null,
            // a container for jQueryfied comments inserted during undo
            undoComments = [],
            // the layer node for all comments
            commentLayerNode = null,
            // the layer node for all comment bubbles
            bubbleLayerNode = null,
            // the application content root node
            appContentRootNode = null,
            // the children of the page that contain can contain comment place holder nodes as sorted jQuery list
            // -> in OOXML this is only the page content node, in ODF also the first header and last footer
            contentChildren = null,
            // the comment default width in pixel
            commentDefaultWidth = 300,
            // the character and paragraph attributes for paragraphs in comments
            commentDefaultAttributes = { paragraph: { marginBottom: 0 } },
            // the current mode for highlighting comments (setting default to 'selected')
            displayMode = CommentLayer.DISPLAY_MODE.SELECTED,
            // the default mode for highlighting comments (this can be modified dependent from resolution)
            defaultDisplayMode = Utils.TOUCHDEVICE ? CommentLayer.DISPLAY_MODE.BUBBLES : CommentLayer.DISPLAY_MODE.SELECTED,
            // the view mode for the comments
            viewMode = CommentLayer.VIEW_MODE.CLASSIC,
            // the default view mode for the comments
            defaultViewMode = CommentLayer.VIEW_MODE.CLASSIC,
            // the horizontal comment layer offset relative to the page
            commentLayerOffset = 0,
            // whether missing comment nodes can be ignored (only during removing marginal comments allowed)
            ignoreMissingCommentNode = false,
            // whether the bubble mode is activated
            isBubbleMode = false,
            // whether the document can be edited
            isEditable = false;

        // base constructors --------------------------------------------------

        TriggerObject.call(this);
        TimerMixin.call(this);
        AppObjectMixin.call(this, app);

        // private methods ----------------------------------------------------

        /**
         * Registering one comment in the comment layer model.
         *
         * @param {HTMLElement|jQuery} comment
         *  One comment node that is positioned absolutely.
         *
         * @param {String} id
         *  The unique id of the comment node.
         *
         * @returns {Number}
         *  The number of comments in the model.
         */
        function addIntoCommentModel(comment, id) {

            var // the comment node in jQuery format
                $comment = $(comment);

            // saving in the model
            comments.push($comment);
            allComments[id] = $comment;

            // updating the list of authors
            handleListOfAuthors(comment, { insert: true });

            // counting marginal comments
            if (app.isODF() && DOM.isTargetCommentNode($comment.parent())) { marginalCommentCounter++; }

            return comments.length;
        }

        /**
         * Empty the complete model. This is for example necessary after a reload of the
         * document.
         */
        function emptyModel() {
            // clearing all parts of the model
            comments = [];
            allComments = {};
            marginalCommentCounter = 0;
        }

        /**
         * Removing one comment with the specified id from the comments model
         *
         * @param {String} id
         *  The id of the comment node.
         *
         * @param {Boolean} isMarginalComment
         *  Whether the comment node is in header or footer.
         *
         * @returns {Number}
         *  The number of comments in the model.
         */
        function removeFromCommentModel(id, isMarginalComment) {

            var // the comment node in the collectors
                commentNode = Utils.getDomNode(allComments[id]);

            if (commentNode) {

                comments = _.filter(comments, function (oneComment) {
                    return Utils.getDomNode(oneComment) !== commentNode;
                });

                delete allComments[id];

                // counting marginal comments
                if (isMarginalComment) { marginalCommentCounter--; }

                // updating the list of authors
                handleListOfAuthors(commentNode, { insert: false });
            }

            return comments.length;
        }

        /**
         * Handling the model for the list of authors. This function is called, if
         * a model is inserted into the model or removed from the model.
         *
         * @param {HTMLElement|jQuery} [comment]
         *  One comment node. This node is not used, if the option 'insert' is set
         *  to false. If 'insert' is true (the default), this comment node is
         *  mandatory.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.insert=true]
         *      If set to true, the comment was inserted. Otherwise the comment was
         *      deleted.
         */
        function handleListOfAuthors(comment, options) {

            var // the number of comment authors
                oldNumber = commentsAuthorList.length;

            if (Utils.getBooleanOption(options, 'insert', true)) {
                addIntoAuthorList(comment);
            } else {
                // generating completely new list after deleting a comment
                commentsAuthorList = createListOfAuthors();
            }

            // inform listeners, that the comment author list is modified
            if (self.isImportFinished() && (oldNumber !== commentsAuthorList.length)) {
                self.trigger('update:commentauthorlist');
            }
        }

        /**
         * Collecting all authors, but only while the document is loading. This is
         * triggered during an insertComment operation or after refreshing all
         * comments after loading with fast load or local storage.
         *
         * @param {HTMLElement|jQuery} comment
         *  One comment node.
         */
        function addIntoAuthorList(comment) {

            var // comment's author name
                author = DOM.getCommentAuthor(comment);

            // add author to list if not there already
            if (author && _.indexOf(commentsAuthorList, author) < 0) {
                commentsAuthorList.push(author);
            }
        }

        /**
         * Whether the document contains at least one comment in header or footer.
         *
         * @returns {Boolean}
         *  Returns, whether the document contains at least one marginal comment.
         */
        function containsMarginalComments() {
            return marginalCommentCounter > 0;
        }

        /**
         * Refreshing the links between a comment from the comment layer and its
         * place holder node.
         *
         * @param {Node|jQuery} comment
         *  The absolute positioned comment in the comment layer node.
         */
        function refreshLinkForComment(comment) {

            var // the drawing id of the drawing in the drawing layer
                commentID = DOM.getTargetContainerId(comment),
                // the selector string for the search for place holder nodes and space maker nodes
                selectorString = '[data-container-id=' + commentID + ']',
                // finding the place holder and optionally the space maker node, that are located inside the page content node
                foundNodes = null,
                // the marginal root for a comment place holder node (odt only)
                marginalRoot = null;

            // helper function to create data link between the comment and the place holder
            function createDataConnections(allNodes) {

                var // whether the corresponding place holder node was found (this is required)
                    placeHolderFound = false,
                    // whether the comment is located in header or footer (the comment thread is marked with class 'targetcomment')
                    isMarginalComment = app.isODF() && DOM.isTargetCommentNode($(comment).parent());

                _.each(allNodes, function (oneNode) {

                    if (DOM.isCommentPlaceHolderNode(oneNode)) {

                        if (isMarginalComment) {
                            // saving at least the target information at the comment node
                            marginalRoot = DOM.getClosestMarginalTargetNode(oneNode);
                            comment.data(DOM.COMMENTPLACEHOLDER_LINK, DOM.getTargetContainerId(marginalRoot));

                        } else {
                            // creating links in both directions
                            comment.data(DOM.COMMENTPLACEHOLDER_LINK, oneNode);
                            $(oneNode).data(DOM.COMMENTPLACEHOLDER_LINK, Utils.getDomNode(comment));
                        }

                        placeHolderFound = true;

                        return; // leaving after finding the first place holder node (might be more in header or footer)
                    }
                });

                if (!placeHolderFound) {
                    Utils.error('CommentLayer.refreshLinkForComment(): failed to find place holder node with comment ID: ' + commentID);
                }

            }

            // the children of the page node that can contain comment place holder nodes
            contentChildren = contentChildren || (app.isODF() ? DOM.getContentChildrenOfPage(model.getNode()) : DOM.getPageContentNode(model.getNode()));

            // finding the place holder and optionally the space maker node, that are located inside the page content node
            foundNodes = $(contentChildren).find(selectorString);

            // updating the value for the global comment id, so that new comments in the comment layer get an increased number.
            updatePlaceHolderCommentID(commentID);

            if (foundNodes.length > 0) {
                createDataConnections(foundNodes);
            } else {
                Utils.error('CommentLayer.refreshLinkForComment(): failed to find place holder node with commentID: ' + commentID);
            }

        }

        /**
         * Refreshing the list of comment place holder nodes, that always need to have the correct order
         * of comments in the document. This order is needed for the correct ordering of comments in the
         * side pane.
         *
         * @param {Node|jQuery} [contentNode]
         *  Optional node, that can be used, if available.
         */
        function updatePlaceHolderCollection() {

            var // the collector for all start ranges and all comments
                startCollector = null,
                // a collector for all saved comment IDs
                commentIDs = [],
                // the current ID
                currentID = 0,
                // a comment node and its place holder node
                commentNode = null, placeHolderNode = null,
                // a collector for the comment place holders
                placeHolderCollector = [];

            // the children of the page node, in which comment place holders can be located
            contentChildren = contentChildren || (app.isODF() ? DOM.getContentChildrenOfPage(model.getNode()) : DOM.getPageContentNode(model.getNode()));

            // restoring the sorted list of all place holders
            // -> after inserting comment, deleting comment, splitting paragraph or loading document from local storage)

            if (app.isODF()) {
                // inside the document, the order is determined by the position of the place holder node,
                // that always marks the beginning of the comment
                startCollector = contentChildren.find(DOM.COMMENTPLACEHOLDER_NODE_SELECTOR);

                if (containsMarginalComments()) {

                    // in header and footer the place holder can occur more often
                    // -> they need to be filtered

                    _.each(startCollector, function (node) {

                        currentID = DOM.getTargetContainerId(node);

                        if (!_.contains(commentIDs, currentID)) {
                            placeHolderCollector.push(node);
                            commentIDs.push(currentID);
                        }
                    });
                } else {
                    placeHolderCollector = startCollector; // no comments in header or footer
                }

                // improving the position of comments inside text frames that are in the text drawing layer of the main document
                if (DOM.getTextDrawingLayerNode(model.getNode()).find(DOM.COMMENTPLACEHOLDER_NODE_SELECTOR).length > 0) {
                    // these pageComments place holder nodes are at the end of the place holder collector
                    // -> iterating over all place holder nodes and resorting them
                    placeHolderCollector = _.sortBy(placeHolderCollector, function (placeHolder) {
                        return $(placeHolder).offset().top;
                    });
                }

            } else {
                // inside the document, the order of the comments is determined by the start of the comment.
                // In OOXML this can be the start range node or the place holder node itself, if the comment
                // has no start range node.
                startCollector = contentChildren.find(DOM.RANGEMARKER_STARTTYPE_SELECTOR + ', ' + DOM.COMMENTPLACEHOLDER_NODE_SELECTOR);

                _.each(startCollector, function (node) {

                    if (DOM.isCommentPlaceHolderNode(node)) {

                        currentID = DOM.getTargetContainerId(node);

                        if (!_.contains(commentIDs, currentID)) {
                            placeHolderCollector.push(node);
                            commentIDs.push(currentID);
                        }

                    } else if (DOM.isRangeMarkerStartNode(node) && DOM.getRangeMarkerType(node) === 'comment') {

                        currentID = DOM.getRangeMarkerId(node);

                        if (!_.contains(commentIDs, currentID)) {
                            commentNode = self.getCommentRootNode(currentID);
                            placeHolderNode = DOM.getCommentPlaceHolderNode(commentNode);

                            if (placeHolderNode) {
                                placeHolderCollector.push(placeHolderNode);
                                commentIDs.push(currentID);
                            }
                        }
                    }
                });

            }

            // setting collection for global container
            commentPlaceHolders = $(placeHolderCollector);
        }

        /**
         * Helper function to keep the global placeHolderCommentID up-to-date. After inserting
         * a new comment or after loading from local storage, this number needs to be updated.
         * Then a new comment can be inserted from the client with a valid and unique id.
         *
         * @param {String} id
         *  The id of the comment node.
         */
        function updatePlaceHolderCommentID(id) {

            // does the id end with a number?
            // -> then the placeHolderCommentID needs to be checked

            var // resulting array of the regular expression
                matches = /(\d+)$/.exec(id),
                // the number value at the end of the id
                number = 0;

            if (_.isArray(matches)) {
                number = parseInt(matches[1], 10);

                if (number >= placeHolderCommentID) {
                    placeHolderCommentID = number + 1;
                }
            }
        }

        /**
         * Leaving the mode with an active target. This includes, that the selection receives
         * the standard root node and the model has no longer an active target set. For
         * performance reasons the selection can be used as parameter. If it is not provided,
         * it is received from the model.
         *
         * @param {Object} [sel]
         *  The current selection, optional. Can be given to this function for performance
         *  reasons.
         */
        function leaveTargetNode(sel) {

            var // the selection object
                selection = sel || model.getSelection();

            selection.setNewRootNode(model.getNode());
            selection.setTextSelection(selection.getFirstDocumentPosition()); // really necessary? (see 40913)
            model.setActiveTarget('');
        }

        /**
         * Caching the currently active target, so that it can be restored later.
         *
         * @returns {Object}
         *  An object with the property 'activeTarget' for the currently active target string
         *  and the property 'rootNode' for the currently used active root node.
         */
        function cacheTargetNode() {

            return { activeTarget: model.getActiveTarget(), rootNode: model.getSelection().getRootNode() };
        }

        /**
         * Setting the left offset of the comment layer node. This needs to be done, after creating
         * a new comment layer node, or after the page width has changed. This happens for example
         * after change of page orientation, or on small devices.
         *
         * @param {HTMLElement|jQuery} [node]
         *  The node, that represents the comment layer of the document. This can be used as
         *  parameter, if it is new created. Otherwise the globally saved 'commentLayerNode'
         *  is used.
         */
        function updateHorizontalCommentLayerPosition(node) {

            var // the container node for the comments. This can come in as parameter (after creating
                // a new container node, or the global 'commentLayerNode' is used.
                layerNode = node || commentLayerNode,
                // the new calculated pixel position for the left offset
                leftPos = 0,
                // the right padding of the page node
                rightPadding = 0,
                // the horizontal offset for the bubble layer node relative to the right border of the page
                bubbleLayerOffset = 0;

            if (layerNode) {

                leftPos = (Utils.round(model.getNode().innerWidth(), 1));

                // the left position ignores the margin, so that the right margin can be used for
                // centering the page including the comment layer

                $(layerNode).css('left', (leftPos + commentLayerOffset) + 'px');

                // also adapting the bubble layer node
                if (bubbleLayerNode) {

                    // receiving the width of the right margin
                    rightPadding = Utils.convertCssLength(model.getNode().css('padding-right'), 'px', 1);

                    if (rightPadding > CommentLayer.BUBBLE_LAYER_WIDTH) {
                        bubbleLayerOffset = Utils.round((rightPadding + CommentLayer.BUBBLE_LAYER_WIDTH) / 2, 1);
                    } else {
                        // bubble must be complete on the page
                        bubbleLayerOffset = CommentLayer.BUBBLE_LAYER_WIDTH + 1;
                    }

                    $(bubbleLayerNode).css('left', (leftPos - bubbleLayerOffset) + 'px');
                }
            }
        }

        /**
         * Updating the comments layer once after the page breaks are all inserted.
         */
        function updateAfterPageBreak() {

            // first header and last footer are now available (those can contain comment place holder in ODF)
            if (app.isODF()) {
                // the children of the page node, in which comment place holders can be located
                contentChildren = DOM.getContentChildrenOfPage(model.getNode());
            }

            // updating the comments layer for ODF and OOXML
            updateCommentsLayer();
        }

        /**
         * Updating the complete comment layer. This includes the positioning of the comment container
         * node and the positioning of all comments inside the comment container.
         */
        function updateCommentsLayer() {
            if (!self.isEmpty()) {
                additionalUpdatePlaceHolderCollection(); // refreshing marginal place holders before calling update comments (only odf)
                updateHorizontalCommentLayerPosition();
                updateComments();
            }
        }

        /**
         * Updating the comments in vertical order using 'updateComments' after activating or deactivating
         * a comment node. This can cause a change of height of the comment. Typically an updateComments
         * is sufficient for the new vertical ordering of the comments inside the comment layer. But in the
         * 'all' mode the comment selections and the comment lines need to be updated, too.
         */
        function updateCommentsAndRangesAfterActivation() {

            var // the ranges overlay node
                overlayNode = null;

            if (CommentLayer.ALWAYS_VISIBLE_COMMENT_LINES_MODE[displayMode]) {
                overlayNode = model.getRangeMarker().getRangeOverlayNode();
                if (overlayNode && overlayNode.children().length > 0) {
                    model.trigger('update:absoluteElements');  // updating the comments vertically and also the visible ranges
                } else {
                    updateComments(); // only updating the vertical comment positions
                }
            } else {
                updateComments(); // only updating the vertical comment positions
            }

        }

        /**
         * Updating the comments in vertical order using 'updateComments'. But in some special cases it it
         * necessary to trigger 'update:absoluteElements' that additionally repaints the comment ranges. This
         * function should only be used rarely, because 'update:absoluteElements' it typically triggered in
         * the model, not here.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.ignoreMissing=false]
         *      Whether it is allowed, that the place holder list contains (invalid) place holder nodes from
         *      non-first marginal nodes.
         */
        function updateCommentsAndRanges(options) {

            var // the ranges overlay node
                overlayNode = null;

            // during deletion of marginal comments it can happen, that the place holder list
            // contains (invalid) place holder nodes from non-first marginal nodes.
            ignoreMissingCommentNode = Utils.getBooleanOption(options, 'ignoreMissing', false);

            if (CommentLayer.VISIBLE_COMMENT_THREAD_MODES[displayMode]) {
                overlayNode = model.getRangeMarker().getRangeOverlayNode();
                if (overlayNode && overlayNode.children().length > 0) {
                    model.trigger('update:absoluteElements');  // updating the comments vertically and also the visible ranges
                } else {
                    updateComments(); // only updating the vertical comment positions
                }
            } else {
                updateComments(); // only updating the vertical comment positions
            }

            // must be set to false again
            ignoreMissingCommentNode = false;
        }

        /**
         * Special function for updating the place holder collection and the comment threading more often,
         * if the document contains comments in header or footer. This requires a more often update of
         * the place holder collection, because they are more often exchanged, if they are located inside
         * header or footer.
         *
         * Info: This function is typically called before the vertical update of comments is done in
         *       function 'updateComments'.
         * Info: Minimize usage of this function.
         */
        function additionalUpdatePlaceHolderCollection() {

            // Updating place holder collection can be expensive
            // -> reducing calls of updatePlaceHolderCollection()
            // -> keeping track of marginal comments
            if (app.isODF() && containsMarginalComments()) {
                updatePlaceHolderCollection();
            }
        }

        /**
         * Updating the vertical position of the comment nodes in the comment layer.
         * This function is called very often and is therefore performance critical.
         */
        var updateComments = Utils.profileMethod('CommentLayer.updateComments(): ', function () {

            // Performance: Only display the comments in the visible region of the document
            // -> handling like for change track markers
            // Performance: Keeping sorted order of comment place holders, that is used
            // for display order of comments in the side pane

            var // the vertical offset of the page node in pixel
                pageNodeOffset = model.getNode().offset().top,
                // the next available vertical pixel position -> used to avoid overlap of comments
                minVerticalPos = 0, minVerticalBubblePos = 0,
                // the vertical distance between two comments in pixel
                minDistance = 8,
                // the zoom factor
                zoomFactor = app.getView().getZoomFactor() / 100,
                // the range marker object
                rangeMarker = model.getRangeMarker(),
                // the previous thread node
                previousThreadNode = null,
                // the top pixel position of the previous thread node
                previousTopPosition = 0;

            _.each(commentPlaceHolders, function (placeHolderNode) {

                var // the id of the comment place holder node
                    id = DOM.getTargetContainerId(placeHolderNode),
                    // the comment node in the comment layer
                    commentNode = DOM.getCommentPlaceHolderNode(placeHolderNode, { ignoreMissing: ignoreMissingCommentNode }),
                    // whether this is the first node in the thread
                    isFirstInThread = false,
                    // the comment thread node
                    commentThreadNode = null,
                    // the comment bubble node belonging to the thread
                    bubbleNode = null,
                    // the node that determines the top distance of the comment in the comment layer
                    // 1. setting top position according to the beginning of the range
                    // 2. setting top position according to the place holder node
                    refNode = null,
                    // the vertical offset of the comment node relative to the page node
                    topPosition = 0, topBubblePosition = 0,
                    // the text span following the start node
                    textSpan = null,
                    // a correction value that handles the font-size
                    correction = null;

                // checking, if the current comment node is the first in its thread
                if (commentNode && ($(commentNode).prev().length === 0)) { isFirstInThread = true; }

                // only iterating over the comment threads
                if (isFirstInThread) {

                    // the comment thread node
                    commentThreadNode = $(commentNode).parent();

                    // the node that determines the top distance of the comment in the comment layer
                    refNode = (rangeMarker.getStartMarker(id)) ? rangeMarker.getStartMarker(id) : $(placeHolderNode);

                    textSpan = Utils.findNextSiblingNode(refNode, 'span');
                    correction = textSpan ? Utils.convertCssLength($(textSpan).css('font-size'), 'px', 1) - Utils.convertCssLength($(refNode).css('font-size'), 'px', 1) : 0;

                    // setting upper position of the comment
                    topPosition = Utils.round((refNode.offset().top - correction - pageNodeOffset) / zoomFactor, 1);

                    // if this is the bubble mode, the vertical bubble positions need to be updated
                    if (isBubbleMode) {

                        // getting the bubble node belonging to the thread
                        bubbleNode = $(commentThreadNode.data(CommentLayer.BUBBLE_CONNECT));

                        // calculating the vertical position for the bubble node
                        topBubblePosition = Math.max(topPosition, minVerticalBubblePos);
                        minVerticalBubblePos = topBubblePosition + bubbleNode.height() + minDistance;
                        topBubblePosition += 'px';

                        // setting the vertical offset for the comment bubble node
                        bubbleNode.css('top', topBubblePosition);

                    } else {

                        // workaround for abs pos element, which get the "moz-dragger" but we dont want it
                        if (_.browser.Firefox && Utils.convertCssLength(commentLayerNode.css('left'), 'px', 1) > 0) { commentThreadNode.css('left', 0); }

                        // check if the previous node can be shrinked
                        if ((minVerticalPos > topPosition) && previousThreadNode && previousThreadNode.hasClass(CommentLayer.INACTIVE_THREAD_CLASS)) {
                            // Shrinking the previous comment thread, so that the current thread goes upwards
                            handleDecreasedCommentThread(previousThreadNode, { decrease: true });

                            // setting new minimum value for the current comment
                            minVerticalPos = previousTopPosition + previousThreadNode.height() + minDistance;
                        }

                        // modifying the top position for the comment threads
                        topPosition = Math.max(topPosition, minVerticalPos);

                        // removing the information for decreased comments from the current layer.
                        // -> a following comment thread can decrease the size again
                        handleDecreasedCommentThread(commentThreadNode, { decrease: false });

                        // setting minimum value for the following comment
                        minVerticalPos = topPosition + commentThreadNode.height() + minDistance;

                        // saving value for following comment
                        previousTopPosition = topPosition;
                        previousThreadNode = commentThreadNode;

                        // setting the vertical offset for the comment thread node
                        commentThreadNode.css('top', topPosition + 'px');
                    }
                }
            });
        });

        /**
         * Drawing a connection line between the comment node and the upper left corner of the
         * range to that the comment belongs.
         *
         * @param {HTMLElement|jQuery} commentNode
         *  The comment node, that will be connected to its range.
         *
         * @param {Object} startPos
         *  An object containing the properties x and y. These represent the pixel position of
         *  the upper left corner of the range, that will be connected to the comment. The pixel
         *  positions are received using the jQuery 'offset' function and relative to the page.
         *
         * @param {String} id
         *  The id of the comment node.
         *
         * @param {String} highlightClassNames
         *  One or more space-separated classes to be added to the class attribute of the
         *  generated 'div'-elements, that are used to draw the connection line.
         */
        function drawCommentLineConnection(commentNode, startPos, id, highlightClassNames) {

            var // drawing the line into the range marker overlay
                overlayNode = model.getRangeMarker().getRangeOverlayNode(),
                // the horizontal pixel position, where the vertical line will be drawn
                breakPosX = 0,
                // the current zoom factor
                zoomFactor = app.getView().getZoomFactor(),
                // the comment thread node
                commentThreadNode = $(commentNode).parent(),
                // the destination position at the comment node
                destPos = Position.getPixelPositionToRootNodeOffset(model.getNode(), commentThreadNode, zoomFactor),
                // whether one line is sufficient to connect comment and range
                oneLine = (startPos.y === destPos.y),
                // the classes for the new highlight elements (adding marker for connection line)
                classNames = highlightClassNames ? highlightClassNames + ' ' + commentLineClass : commentLineClass,
                // the first vertical line, that will always be drawn
                line1 = $('<div>').addClass(classNames).attr('data-range-id', id),
                // optional second and third line
                line2 = null, line3 = null,
                // the right padding of the page in pixel
                rightPadding = 0,
                // a minimum thickness, that is used for zoom factors smaller than 100 (overwrites css)
                minThickNess = zoomFactor < 100 ? Math.ceil(100 / zoomFactor) : 1;

            if (oneLine) {
                line1.css({ top: startPos.y, left: startPos.x, width: destPos.x - startPos.x });
                if (zoomFactor < 100) { line1.css({ height: minThickNess }); }
                overlayNode.append(line1);
            } else {
                rightPadding = Utils.getElementCssLength(model.getNode(), 'paddingRight');
                breakPosX = overlayNode.width() - Math.round(rightPadding / 2);

                line2 = $('<div>').addClass(classNames).attr('data-range-id', id);
                line3 = $('<div>').addClass(classNames).attr('data-range-id', id);

                line1.css({ top: startPos.y, left: startPos.x, width: breakPosX - startPos.x });
                line2.css({ top: Math.min(startPos.y, destPos.y), left: breakPosX, height: Math.abs(startPos.y - destPos.y) });
                line3.css({ top: destPos.y, left: breakPosX, width: destPos.x - breakPosX });

                if (zoomFactor < 100) {
                    line1.css({ height: minThickNess });
                    line2.css({ width: minThickNess });
                    line3.css({ height: minThickNess });
                }

                overlayNode.append(line1, line2, line3);
            }
        }

        /**
         * Visualizing the selection range of a comment thread. This handler is assigned
         * to the comment thread node. If bubble mode is active, 'this' is the currently
         * hovered bubble node.
         */
        function drawCommentRangeHover() {

            var // the parent comment of the comment thread
                commentNode = isBubbleMode ? $($(this).data(CommentLayer.BUBBLE_CONNECT)).children().first() : $(this).children().first(),
                // the id of the hovered comment
                id = DOM.getTargetContainerId(commentNode),
                // the author specific color class name
                colorClassName = getCommentColorClassName(commentNode),
                // the range marker handler
                rangeMarker = model.getRangeMarker(),
                // the position of the upper left corner of the highlighted range
                startPos = null;

            // highlighting the complete comment thread
            $(this).addClass(getHoveredCommentThreadColorClassName(commentNode));

            // highlighting the range with the specific id, but not if this is
            // the active target (cursor is positioned inside the comment).
            if (id && model.getActiveTarget() !== id) {

                startPos = rangeMarker.highLightRange(id, 'visualizedcommenthover fillcolor ' + colorClassName);

                // Drawing a line connection between comment and range (not necessary for bubble mode)
                if (startPos && !isBubbleMode) { drawCommentLineConnection(commentNode, startPos, id, 'highlightcommentlinehover fillcolor ' + colorClassName); }
            }
        }

        /**
         *  Removing the visualization of one range, if the visualization was
         *  triggered by hover.
         */
        function removeCommentRangeHover() {

            var // the comment node for this comment thread
                allComments = isBubbleMode ? $($(this).data(CommentLayer.BUBBLE_CONNECT)).children() : $(this).children(),
                // the first comment node in the comment thread
                commentNodeParent = allComments.first(),
                // the id of the parent comment
                id = DOM.getTargetContainerId(commentNodeParent),
                // the id of the hovered comment and its children
                ids = getAllIdsInCommentThread(allComments);

            // a helper function to collect all IDs from a list of comments
            function getAllIdsInCommentThread(comments) {

                var // a collector for all comment ids
                    allIds = [];

                _.each(comments, function (oneComment) {
                    allIds.push(DOM.getTargetContainerId(oneComment));
                });

                return allIds;
            }

            // Removing the hover-highlighting of the complete comment thread
            $(this).removeClass(getHoveredCommentThreadColorClassName(commentNodeParent));

            // Removing the highlighting of a highlighted range, but not if this node or one
            // of its children is the active target (cursor is positioned inside the comment).
            if (!_.contains(ids, model.getActiveTarget())) {
                model.getRangeMarker().removeHighLightRange(id);
            }
        }

        /**
         * Handler for hover over comments. This handler can separate the different comments
         * inside a comment thread. Therefore comment specific modifications can be done
         * within this function.
         *
         * @param {jQuery.Event} event
         *  The jQuery event object.
         */
        function handleCommentHover(event) {

            if (event.type === 'mouseenter') {
                // making delete and reply buttons visible after 'mouseenter' event
                $(event.currentTarget).find('.commentbutton').addClass('commentbuttonhover');
            } else {
                // making delete and reply buttons invisible after 'mouseleave' event
                $(event.currentTarget).find('.commentbutton').removeClass('commentbuttonhover');
            }
        }

        /**
         * Handler for 'mousedown' and 'touchstart' events on the comment layer itself. This
         * shall simply be ignored.
         *
         * @param {jQuery.Event} event
         *  The jQuery event object.
         */
        function handleMouseDownOnCommentLayer(event) {
            if (DOM.isCommentLayerNode(event.target)) {
                event.preventDefault();
                event.stopPropagation();
                return;
            }
        }

        /**
         * In ODF documents, that comment start node (the comment node itself) and the
         * comment end node (the range marker) must have the same parent. Therefore
         * the current selection needs to be adapted, so that the end position is
         * inside a paragraph, that has the same parent as the start paragraph node.
         *
         * @param {Number[]} start
         *  The logical start position for the new comment.
         *
         * @returns {Number[]}
         *  The logical end position for the new comment.
         */
        function getBestODFRangeEndPosition(startPos) {

            var // the best end position for the current selection
                endPos = null,
                // the currently active root node of the document
                rootNode = model.getCurrentRootNode(),
                // the paragraph node containing the start position
                startParagraph = Utils.getDomNode(Position.getParagraphElement(rootNode, _.initial(startPos))),
                // the end paragraph
                endParagraph = null;

            if (startParagraph) {

                endParagraph = startParagraph;

                // iterating over all following paragraphs, not traversing into table
                while (endParagraph && endParagraph.nextSibling && DOM.isParagraphNode(endParagraph.nextSibling)) {
                    endParagraph = endParagraph.nextSibling;
                }

                endPos = Position.getOxoPosition(rootNode, endParagraph);
                endPos.push(Position.getParagraphNodeLength(endParagraph));  // setting to last position inside this paragraph
            }

            if (!endPos) {
                endPos = Position.increaseLastIndex(startPos);
            }

            return endPos;
        }

        /**
         * Determining the class name for the color of the author for a specified comment
         * node. This class name is used in comment nodes (color of the name) and in the
         * comment thread (color of left border).
         *
         * @param {HTMLElement|jQuery} commentNode
         *  The comment node.
         *
         * @returns {String|null}
         *  The class name for the color of the author of the specified comment node.
         */
        function getCommentColorClassName(commentNode) {
            var colorIndex = getAuthorColorIndex(commentNode);
            return _.isNumber(colorIndex) ? ('comment-author-' + colorIndex) : null;
        }

        /**
         * Determining the class name for the color of the author to highlight the active
         * comment thread.
         *
         * @param {HTMLElement|jQuery} commentNode
         *  The comment node.
         *
         * @returns {String|null}
         *  The active comment thread class name for the color of the author of the
         *  specified comment node.
         */
        function getActiveCommentThreadColorClassName(commentNode) {
            var colorIndex = getAuthorColorIndex(commentNode);
            return _.isNumber(colorIndex) ? ('comment-thread-active-author-' + colorIndex) : null;
        }

        /**
         * Determining the class name for the color of the author to highlight the hovered
         * comment thread.
         *
         * @param {HTMLElement|jQuery} commentNode
         *  The comment node.
         *
         * @returns {String|null}
         *  The hovered comment thread class name for the color of the author of the
         *  specified comment node.
         */
        function getHoveredCommentThreadColorClassName(commentNode) {
            var colorIndex = getAuthorColorIndex(commentNode);
            return _.isNumber(colorIndex) ? ('comment-thread-hovered-author-' + colorIndex) : null;
        }

        /**
         * Determining the author index for author of a specified comment node.
         *
         * @param {HTMLElement|jQuery} commentNode
         *  The comment node.
         *
         * @returns {Number|null}
         *  The index of the author of the specified comment node, or null, if it
         *  could not be determined.
         */
        function getAuthorColorIndex(commentNode) {
            var author = DOM.getCommentAuthor(commentNode);
            return author ? app.getAuthorColorIndex(author) : null;
        }

        /**
         * For author of passed comment, assign appropriate color, from authors color list set.
         * The generated author class 'comment-author-X' is used for the text color of the
         * comment author and additionally for the color of the left border of the comment thread.
         * The latter only, if the specified comment node is the first in the thread.
         *
         * @param {HTMLElement|jQuery} node
         *  The comment node.
         */
        function assignColorForComment(node) {
            var // the comment node (jQuery)
                $node = $(node),
                // the class name for the color of the author name
                authorClass = getCommentColorClassName(node);

            $node.find('.commentauthor').addClass(authorClass);
            // assigning color of top most comment to comment thread
            if ($node.prev().length === 0) { $node.parent().addClass(authorClass); }
        }

        /**
         * Removing empty text spans between the comment placeholder node and a preceding
         * range end marker node (only OOXML).
         *
         * @param {HTMLElement|jQuery} node
         *  The comment node in the comment layer.
         */
        function removePreviousEmptyTextSpan(comment) {

            var // the place holder node for the comment
                placeHolderNode = $(DOM.getCommentPlaceHolderNode(comment)),
                // the node preceding the comment place holder node
                precedingNode = placeHolderNode.length > 0 && placeHolderNode.prev();

            if (precedingNode.length > 0 && precedingNode.prev().length > 0 && DOM.isEmptySpan(precedingNode) && DOM.isRangeMarkerEndNode(precedingNode.prev())) {
                precedingNode.remove();
            }
        }

        /**
         * guest users are saved in document as "User xxx",
         * this function fetches the original display name out of the user id
         */
        function repairDisplayName(commentNode) {
            var uid = DOM.getCommentAuthorUid(commentNode);
            if (!uid) {
                return;
            }
            Utils.getUserInfo(Utils.resolveUserId(parseInt(uid, 10))).done(function (info) {
                var authorName = info.displayName;
                var authorNode = commentNode.find('.commentauthor');
                authorNode.text(authorName);
                if (info.guest) {
                    authorNode.css({ pointerEvents: 'none', textDecoration: 'none!important' });
                }
            });
        }

        /**
         * After loading is finished, push collected comment authors list to global document authors list,
         * and after that, for each comment that is loaded with document, assign appropriate color for author.
         *
         * @param {Boolean} usedLocalStorage
         *  Whether the document was loaded from the local storage.
         */
        function finalizeComments(usedLocalStorage, usedFastLoad) {

            // merge collected comments authors during loading, with global document authors list
            app.addAuthors(commentsAuthorList);

            _.each(comments, function (comment) {
                assignColorForComment(comment);
                // additionally set the correct time stamp in the comment node
                setValidTimeString(comment);
                // and additionally register the event handler on the meta info node
                activateHandlersAtMetaInfoNode(DOM.getCommentMetaInfoNode(comment)); // registering a click handler for the complete meta info node
                // setting the author pictures at the comment meta info nodes
                if (!usedLocalStorage) { setAuthorPictureAndLink(comment); }
                // removing empty text spans between comment node and range marker end node, if fast load was used
                if (usedFastLoad && !app.isODF()) { removePreviousEmptyTextSpan(comment); }
                // docuemnt saves guest with "user xxx" but we want to see their display names
                if (usedFastLoad) { repairDisplayName(comment); }
                // adding class touch for buttons on touch devices
                if (usedFastLoad && Utils.TOUCHDEVICE) { prepareCommentOnTouch(comment); }
            });
        }

        /**
         * After loading the document it is necessary, that the size of the comments is decreased, if it
         * is too high. This requires the classes 'inactivethread' at the comment threads and 'decreased'
         * at the comment node.
         */
        function markCommentsInactiveAfterLoad() {

            if (!commentLayerNode) { return; } // nothing to do

            _.each(commentLayerNode.children(), function (commentThread) {
                // marking comment thread as inactive
                $(commentThread).addClass(CommentLayer.INACTIVE_THREAD_CLASS);
            });
        }

        /**
         * Checking the height of the comments inside a specified comment thread, if the thread
         * is no longer active.
         *
         * @param {HTMLElement|jQuery} commentThread
         *  The comment thread node in the comment layer, whose comments height will be checked.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.decrease=true]
         *      Whether the comments in the specified comment thread shall be increased or
         *      decreased. Decreasing is the default.
         */
        function handleDecreasedCommentThread(commentThread, options) {

            var // the jQueryfied comment thread
                $commentThread = $(commentThread);

            if (Utils.getBooleanOption(options, 'decrease', true)) {
                // making arrows in lower right corner visible, if the comment content node size is decreased
                _.each($commentThread.children(), function (comment) {
                    if ($(comment).children('div.content').children('div.textframe').outerHeight(true) > CommentLayer.COMMENT_CONTENT_MAXHEIGHT) {
                        $(comment).addClass('decreased');
                    }
                });
            } else {
                _.each($commentThread.children(), function (comment) {
                    $(comment).removeClass('decreased');
                });
            }
        }

        /**
         * After an undo or redo, in that comments were inserted, it is necessary to set the 'decreased' class to the
         * comments, if required. Because the comment thread already has the class 'CommentLayer.INACTIVE_THREAD_CLASS',
         * the height of the comment is already reduced to 'CommentLayer.COMMENT_CONTENT_MAXHEIGHT'.
         */
        function handleUndoComments() {

            // making arrows in lower right corner visible, if the comment was modified during undo (it might have been inserted during undo)
            _.each(undoComments, function (comment) {
                if (comment.children('div.content').children('div.textframe').outerHeight(true) === CommentLayer.COMMENT_CONTENT_MAXHEIGHT) {
                    comment.addClass('decreased');
                }
            });

            updateCommentsLayer();
            if (app.isODF() && containsMarginalComments()) { updateCommentThreads(); } // this is necessary for undo of deleting a full header or footer (never during loading document)
        }

        /**
         * After loading is finished, the buttons in the comment info marker need
         * to be increased on touch devices.
         *
         * @param {jQuery} commentNode
         *  The comment node in the comment layer, whose icon size will be updated.
         */
        function prepareCommentOnTouch(commentNode) {
            commentNode.find('.commentbutton').addClass('touch'); // increase distance between buttons
            commentNode.find(DOM.COMMENTCLOSERNODE_SELECTOR).addClass('touch'); // bigger icons on touch devices
        }

        /**
         * After loading is finished, the comment creation time can be formatted,
         * so that the user can read it easily.
         *
         * @param {HTMLElement|jQuery} commentNode
         *  The comment node in the comment layer, whose date will be formatted
         *  to be displayed correctly.
         */
        function setValidTimeString(commentNode) {

            var // the date saved in the comment node
                date = DOM.getCommentDate(commentNode),
                // the formatted date string
                dateString = date ? app.getView().getDisplayDateString(Date.parse(date), { toLocal: false }) : '';

            commentNode.find('.commentdate').text(dateString);
        }

        /**
         * After loading is finished (not loading from local storage), the picture
         * of the comment author can be loaded.
         *
         * @param {jQuery} commentNode
         *  The comment node in the comment layer, whose date will be formatted
         *  to be displayed correctly.
         */
        function setAuthorPictureAndLink(commentNode) {

            var // the element containing the photo (as CSS background)
                authorPictureNode = commentNode.find('.commentpicture'),
                // the node containing the author's name
                authorNode = commentNode.find('.commentauthor'),
                // the size of the author image container
                nodeSize = 24,
                // size of the photo bitmap (double on retina displays)
                bitmapSize = nodeSize * (Utils.RETINA ? 2 : 1),
                // the authors user id
                uid = DOM.getCommentAuthorUid(commentNode),
                // the user id as converted number
                localuid = uid ? Utils.resolveUserId(parseInt(uid, 10)) : null;

            // Check, if the comment author has the name as the current user (if user id was not sent from the server)
            // -> TODO: This can be removed, after user id is sent from the server correctly. But it can also stay.
            if (!localuid && (DOM.getCommentAuthor(commentNode) === self.getCommentAuthor())) { localuid = ox.user_id; }

            // setting a picture into the picture node
            if (authorPictureNode.length > 0) {
                authorPictureNode = authorPictureNode.height(nodeSize).css('min-width', (nodeSize + 4) + 'px');
                ContactsAPI.pictureHalo(authorPictureNode, _.isNumber(localuid) ? { internal_userid: localuid } : {}, { width: bitmapSize, height: bitmapSize, scaleType: 'cover' });
            }

            // creating a link for the author to open popup with author information
            if (authorNode.length > 0) {
                if (_.isNumber(localuid) && !DriveUtils.isGuest()) {
                    authorNode.addClass('halo-link');
                    authorNode.data({ internal_userid: localuid });
                }
            }
        }

        /**
         * After the document is loaded, this function searches for all comment threads in the document (only once).
         * Because this function compares two following comments, it is important that the order in the document
         * is valid (the comment place holder nodes are relevant). Therefore the iteration happens over the global
         * 'commentPlaceHolders' that always needs to be up to date.
         *
         * The behavior for the comment threads is different in OOXML and ODF:
         *
         * ODF: There is a comment start range node and an end range node. The reply to this comment, is a comment
         * node directly following the end range node. The reply itself contains no end range node.
         *
         * OOXML: Comment that have the same range positions (except the positions of the range markers itself) are
         * interpreted as a thread.
         *
         * The comment nodes will be used to find the order of the comments.
         *
         * @param {Number[]} [commentPosition]
         *  An optional parameter, that contains the logical position of a comment place holder node. This can be
         *  used to reduce the search for threads to previous comments only in the same paragraph (performance).
         *  -> Assumption: All placeholder nodes of one thread are in the same paragraph
         */
        function updateCommentThreads(commentPosition) {

            var // the range marker object
                rangeMarker = model.getRangeMarker(),
                // objects for the current comment
                currentComment = null;

            // iterating over the array of comments, if there is more than one comment.
            // -> a comment thread requires at least two comments
            if (commentPlaceHolders.length > 1) {

                _.each(commentPlaceHolders, function (oneCommentPlaceHolder) {

                    var // the comment id
                        id = DOM.getTargetContainerId(oneCommentPlaceHolder),
                        // the comment node belonging to the place holder node
                        oneComment = $(DOM.getCommentPlaceHolderNode(oneCommentPlaceHolder, { ignoreMissing: app.isODF() })),
                        // whether the comment was really found (see 36816 for odf: comments in textframes at page in header or footer)
                        foundCommentNode = oneComment.length > 0,
                        // the range start marker
                        startNode = rangeMarker.getStartMarker(id),
                        // the range end marker
                        endNode = rangeMarker.getEndMarker(id),
                        // whether the comment is inside header or footer
                        isMarginalComment = app.isODF() && DOM.isTargetCommentNode(oneComment.parent()),
                        // the root node of the comment place holder node
                        rootNode = isMarginalComment ? DOM.getClosestMarginalTargetNode(oneCommentPlaceHolder) : model.getNode(),
                        // the logical position of the start marker (OOXML) or the comment place holder node (ODF)
                        startPos = app.isODF() ? (foundCommentNode ? Position.getOxoPosition(rootNode, oneCommentPlaceHolder) : null) : (startNode ? Position.getOxoPosition(rootNode, startNode) : null),
                        // the start paragraph position
                        startParaPos = startPos ? _.initial(startPos) : null,
                        // the logical end marker position
                        endPos = endNode ? Position.getOxoPosition(rootNode, endNode) : null,
                        // the end paragraph position (if endPos is not defined, the start position must be used in ODF)
                        endParaPos = endPos ? _.initial(endPos) : (app.isODF() ? startParaPos : null),
                        // object for the previous comment node
                        prevComment = currentComment,
                        // the thread nodes for the current and the previous comment
                        prevThreadNode = null, currentThreadNode = null,
                        // Performance: Whether the search can be continued
                        doContinue = true;

                    // helper function, that shifts the comment from one thread to another thread
                    function shiftComment() {

                        // -> marking the comment nodes as thread
                        oneComment.addClass(DOM.CHILDCOMMENTNODE_CLASS);

                        // deleting the reply button from the comment node
                        oneComment.find(DOM.COMMENTREPLYNODE_SELECTOR).remove();

                        if (!DOM.isChildCommentNode(prevComment.commentNode)) {
                            prevComment.commentNode.addClass(DOM.PARENTCOMMENTNODE_CLASS);
                        }

                        // shifting the child comment node into the correct comment thread
                        prevThreadNode = prevComment.commentNode.parent();
                        currentThreadNode = oneComment.parent();

                        if (Utils.getDomNode(prevThreadNode) !== Utils.getDomNode(currentThreadNode)) {
                            // inserting behind the previous comment (not at the end of the thread!)
                            oneComment.insertAfter(prevComment.commentNode);
                            // ... and removing the current thread node
                            deactivateAllHandlersAtCommentThread(currentThreadNode);
                            currentThreadNode.remove();
                        }

                    }

                    // shortcut, if the parameter commentPosition is defined
                    if (commentPosition && endParaPos && !_.isEqual(_.initial(commentPosition), endParaPos)) { doContinue = false; }

                    if (doContinue) {

                        if (app.isODF()) {

                            // only the last number can be different
                            // -> only compare one comment with the previous comment node in the sortedComments array
                            currentComment = { id: id, commentNode: oneComment, endNode: endNode, startPos: startPos, endPos: endPos, startParaPos: startParaPos, endParaPos: endParaPos, isChildComment: false };

                            // comparing the range of previous comment and current comment
                            // -> further check, if the start nodes are neighbors and the end nodes are nearly neighbors (because of the comment node itself)
                            if (prevComment) {

                                if (!endPos && _.isEqual(startParaPos, prevComment.endParaPos)) {

                                    if ((prevComment.isChildComment && (_.last(startPos) - _.last(prevComment.startPos) === 1)) ||
                                        (!prevComment.isChildComment && (_.last(startPos) - _.last(prevComment.endPos) === 1))) {

                                        currentComment.isChildComment = true;  // mark the current node as child comment
                                        shiftComment(); // and shift the comment into a new thread, if required

                                    } else if (!prevComment.endNode && startPos && _.isEqual(Position.increaseLastIndex(prevComment.startPos), startPos)) {
                                        // if the previous comment has no end node, than this comment is also a child comment, if it follows
                                        // directly the previous place holder node.
                                        currentComment.isChildComment = true;  // mark the current node as child comment
                                        shiftComment(); // and shift the comment into a new thread, if required
                                    }
                                }
                            }

                        } else {  // OOXML

                            // only the last number can be different
                            // -> only compare one comment with the previous comment node in the sortedComments array
                            currentComment = { id: id, commentNode: oneComment, startNode: startNode, endNode: endNode, startPos: startPos, endPos: endPos, startParaPos: startParaPos, endParaPos: endParaPos };

                            // compare valid ranges that contain start and end range marker
                            if (startNode && endNode) {

                                // comparing the range of previous comment and current comment
                                // check that start and end range marker of the comments are in the same paragraph
                                if (prevComment && _.isEqual(startParaPos, prevComment.startParaPos) && _.isEqual(endParaPos, prevComment.endParaPos)) {

                                    // further check, if the start nodes are neighbors and the end nodes are nearly neighbors (because of the comment node itself)
                                    if ((_.last(startPos) - _.last(prevComment.startPos) === 1) && (_.last(endPos) - _.last(prevComment.endPos) === 2)) {
                                        shiftComment(); // shift the comment into a new thread, if required
                                    }
                                }

                            } else {

                                // start node or end node or both are not valid
                                // -> this is a comment without valid range
                                // -> grouping should be valid, if the previous node also has no valid

                                currentComment.commentPos = Position.getOxoPosition(rootNode, oneCommentPlaceHolder);

                                if (prevComment && (!prevComment.startNode || !prevComment.endNode) && _.isArray(prevComment.commentPos) && _.isEqual(_.initial(currentComment.commentPos), _.initial(prevComment.commentPos)) && (_.last(currentComment.commentPos) - _.last(prevComment.commentPos) === 1)) {
                                    shiftComment(); // shift the comment into a new thread, if required
                                }
                            }
                        }
                    }
                });
            }

        }

        /**
         * After load from local storage the links between comments and comment place holders
         * need to restored.
         * Additionally the global 'comments' and 'allComments' collectors are filled with the
         * comments.
         */
        function refreshCommentLayer() {

            // creating all links between comments in comment layer and the corresponding
            // comment place holder -> for this the data-drawingid is required

            var // the page content node
                commentLayerNode = DOM.getCommentLayerNode(model.getNode()),
                // collecting all drawings in the drawing layer
                allCommentsInLayer = commentLayerNode.children(DOM.COMMENTTHREADNODE_SELECTOR).children(DOM.COMMENTNODE_SELECTOR),
                // whether document was loaded in IE
                isIE = _.browser.IE,
                // the comment thread node
                commentThread = null;

            // reset model
            emptyModel();

            // filling the model with the comments
            _.each(allCommentsInLayer, function (oneComment) {
                if (isIE) { $(oneComment).attr('contenteditable', false); }
                addIntoCommentModel(oneComment, DOM.getTargetContainerId(oneComment));
                if (!self.isImportFinished()) { addIntoAuthorList(oneComment); }
            });

            // restoring the links between comments and their placeholders
            if (comments.length > 0) {
                _.each(comments, function (oneComment) {
                    refreshLinkForComment(oneComment);

                    // registering the hover handler for the comment thread (but only for the first comment in its thread)
                    if (oneComment.prev().length === 0) {
                        commentThread = oneComment.parent();
                        commentThread.hover(drawCommentRangeHover, removeCommentRangeHover);
                        if (isEditable) { activateHandlersAtComments(commentThread); }
                    }
                });

                // restoring the sorted list of all place holders
                updatePlaceHolderCollection();
            }

        }

        /**
         * After loading the document with fast load or local storage, it is necessary
         * to fill the comments model.
         * This function must be called in fast load process, after all fast load strings
         * have been applied to the DOM, but before the operations are handled (38562).
         * This is important for drawings inside comments, because insertDrawing and
         * setAttributes operations for drawings are still sent from the server, even
         * if fast load is used.
         * In odt documents on the other hand, comment can be included in header and
         * footer. But in odt drawings are not allowed in comments. Therefore it is
         * important that the comments model is filled, after the headers and footers
         * are created. So even when fast load is used, 'updateModel' must not be
         * triggered by 'fastload:done', because this would be too early.
         *
         * So this function is called differently corresponding to the loading process
         * and the file format:
         *
         * 1. Loading with operations: This function is not called.
         * 2. Fast load (ODT): Late, in importSuccessHandler
         * 3. Fast load (OOXML): Very early, after 'fastload:done' (38562)
         * 4. Local storage: Late, in importSuccessHandler
         */
        function updateModel() {
            refreshCommentLayer();
            // refreshing also the range marker model, if it was not done before
            model.getRangeMarker().refreshModel({ onlyOnce: true });
        }

        /**
         * The click handler for the comment meta info node on top of the comment
         *
         * @param {jQuery.Event} event
         *  The 'click' event in the comment meta info node.
         */
        function metaInfoClickHandler(event) {
            // Add 'pointerdown' only to preventDefault the right click event.
            // This is necessary to prevent rendering the resize-rectangle in IE.
            if (event.type === 'pointerdown') {
                if (event.button === 2) { event.preventDefault(); }
                return;
            }

            var // the click target
                target = $(event.target),
                // the target parent
                targetParent = target.parent();

            if (DOM.isCommentCloserNode(target) || DOM.isCommentCloserNode(targetParent)) {
                self.deleteComment(target.closest(DOM.COMMENTNODE_SELECTOR));

                // fix for bug 42448: stop bubbling to prevent firing from listeners on the deleted comment.
                // Also preventDefault the event, because when the 'trash icon/area' is over the text area
                // (e.g. on nexus7), clicking on it would set the cursor to the clicked position in the text
                // (ghost click).
                event.preventDefault();
                event.stopPropagation();

            } else if (DOM.isCommentReplyNode(target) || DOM.isCommentReplyNode(targetParent)) {
                self.replyToComment(target.closest(DOM.COMMENTNODE_SELECTOR));
            }

        }

        /**
         * The click handler for the comment bubble nodes in bubble mode. The 'click' event cannot be used here,
         * because the node is inside the div page node. Therefore the mouse handlers (for example mousedown) will
         * also be executed (mousedown comes before 'click'). This leads to unwanted effects.
         *
         * @param {jQuery.Event} event
         *  The 'click' event in the comment bubble node.
         */
        function bubbleNodeClickHandler(event) {

            var // the click target
                target = $(event.currentTarget),
                // the thread node belonging to the bubble node
                threadNode = target.data(CommentLayer.BUBBLE_CONNECT),
                // the selection object
                selection = null;

            if (threadNode) {
                // receiving the selection object from the model
                selection = model.getSelection();
                // setting position for the thread node
                setCommentThreadNextToBubble(target, threadNode);
                // making thread node visible ...
                $(threadNode).css('display', 'block');
                // ... and activating the first comment inside (and optionally deactivate another comment node)
                self.activateCommentNode($(threadNode).children().first(), selection, { deactivate: true });
                // ... and setting cursor into the activated comment at the first position (not always [0,0])
                selection.setTextSelection(Position.getFirstPositionInParagraph(selection.getRootNode(), [0]));
            }

            // no handling of this event in page node (has negative side effects)
            // -> also closing another comment must be handled in this handler
            event.preventDefault();
            event.stopPropagation();
        }

        /**
         * Helper function to make a comment thread visible next to its corresponding
         * bubble node. The comment thread is specified by one of its comments.
         *
         * @param {HTMLElement|jQuery} commentNode
         *  The comment node, whose comment thread will be made visible next to
         *  its bubble node.
         */
        function showCommentThreadInBubbleMode(commentNode) {

            var // the thread node of the specified comment node
                threadNode = $(commentNode).parent(),
                // the bubble node belonging to the thread node
                bubbleNode = null;

            if (threadNode && threadNode.length > 0 && threadNode.css('display') !== 'block') {

                bubbleNode = threadNode.data(CommentLayer.BUBBLE_CONNECT);

                if (bubbleNode) {
                    // setting position for the thread node
                    setCommentThreadNextToBubble(bubbleNode, threadNode);
                    // making thread node visible ...
                    threadNode.css('display', 'block');
                }
            }
        }

        /**
         * Activating the handlers at a meta info node of a comment node.
         *
         * @param {HTMLElement|jQuery} metaInfoNode
         *  The meta info node at the top of a comment node, whose handlers at the comments will
         *  be activated.
         */
        function activateHandlersAtMetaInfoNode(metaInfoNode) {
            $(metaInfoNode).on('mousedown touchstart pointerdown', metaInfoClickHandler); // registering a click handler for the complete meta info node
        }

        /**
         * Activating the handlers for all comments at one specified comment thread.
         *
         * @param {HTMLElement|jQuery} commentThread
         *  The comment thread node in the comment layer, whose handlers at the comments will
         *  be activated.
         */
        function activateHandlersAtComments(commentThread) {
            $(commentThread).on('mouseenter mouseleave', '> .comment', handleCommentHover);
        }

        /**
         * Activating the handlers at a specified comment bubble node.
         *
         * @param {HTMLElement|jQuery} bubbleNode
         *  The comment bubble node, whose handlers for hover and 'click' will be
         *  be activated.
         */
        function activateHandlersAtBubbleNode(bubbleNode) {

            var // the bubble node jQueryfied
                $bubbleNode = $(bubbleNode);

            $bubbleNode.hover(drawCommentRangeHover, removeCommentRangeHover);
            $bubbleNode.on('mousedown touchstart', bubbleNodeClickHandler);
        }

        /**
         * Activating the handlers for the comment layer node.
         */
        function activateHandlersAtCommentLayer() {
            commentLayerNode.on('mousedown touchstart', handleMouseDownOnCommentLayer);
        }

        /**
         * Deactivating the handlers at a specified meta info node inside a comment node.
         *
         * @param {HTMLElement|jQuery} metaInfoNode
         *  The comment meta info node, whose handlers at the comments will be deactivated.
         */
        function deactivateHandlersAtMetaInfoNode(metaInfoNode) {
            $(metaInfoNode).off('mousedown touchstart pointerdown', metaInfoClickHandler); // registering a click handler for the complete meta info node
        }

        /**
         * Deactivating the handlers for all comments at one specified comment thread.
         *
         * @param {HTMLElement|jQuery} commentThread
         *  The comment thread node in the comment layer, whose handlers at the comments will
         *  be deactivated.
         */
        function deactivateHandlersAtComments(commentThread) {
            // deactivating handler for hover over comments
            $(commentThread).off('mouseenter mouseleave', '> .comment', handleCommentHover);
        }

        /**
         * Deactivating all handlers for one specified comment thread.
         *
         * @param {HTMLElement|jQuery} commentThread
         *  The comment thread node in the comment layer, whose handlers will
         *  be deactivated.
         */
        function deactivateAllHandlersAtCommentThread(commentThread) {

            var // the comment thread node jQueryfied
                $commentThread = $(commentThread);

            // deactivating handler for hover over comments
            deactivateHandlersAtComments(commentThread);
            // deactivating handler for hover over comment threads
            $commentThread.off('mouseenter', drawCommentRangeHover);
            $commentThread.off('mouseleave', removeCommentRangeHover);
        }

        /**
         * Deactivating the handlers at a specified comment bubble node.
         *
         * @param {HTMLElement|jQuery} bubbleNode
         *  The comment bubble node, whose handlers for hover and 'click' will be
         *  be deactivated.
         */
        function deactivateAllHandlersAtBubbleNode(bubbleNode) {

            var // the bubble node jQueryfied
                $bubbleNode = $(bubbleNode);

            $bubbleNode.hover('mouseenter mouseleave', removeCommentRangeHover);
            $bubbleNode.off('mousedown touchstart', bubbleNodeClickHandler);
        }

        /**
         * Deactivating the handlers at the comment layer node.
         */
        function deactivateHandlersAtCommentLayer() {
            commentLayerNode.off('mousedown touchstart', handleMouseDownOnCommentLayer);
        }

        /**
         * Setting the pixel position of a comment thread in bubble mode. Standard behavior is,
         * that the comment thread is placed directly next to the bubble. If there is not
         * enough space, the comment is displayed left of the bubble.
         *
         * @param {HTMLElement|jQuery} bubbleNode
         *  The comment bubble node.
         *
         * @param {HTMLElement|jQuery} commentThread
         *  The comment thread node that will be positioned next to the specified comment bubble
         *  node.
         */
        function setCommentThreadNextToBubble(bubbleNode, commentThread) {

            var // the jQueryfied bubble node
                $bubbleNode = $(bubbleNode),
                // the jQuerified comment thread node
                $commentThread = $(commentThread),
                // the left pixel position of the comment layer
                commentLayerLeft = Utils.convertCssLength(commentLayerNode.css('left'), 'px', 1),
                // the left pixel position of the bubble layer
                bubbleLayerLeft = Utils.convertCssLength(bubbleLayerNode.css('left'), 'px', 1),
                // calculating left offset of the comment thread node in the comment layer
                left = bubbleLayerLeft - commentLayerLeft + $bubbleNode.width() + 2,
                // the top offset of the bubble node
                top = Utils.convertCssLength($bubbleNode.css('top'), 'px', 1),
                // the available width inside the application content root node
                availableWidth = appContentRootNode.width(),
                // the page offset in the application content root node
                pageOffset = model.getNode().offset().left,
                // the most right pixel postion of the comment thread node, if it is positioned right of the bubble
                rightPos = pageOffset + Utils.convertCssLength(commentLayerNode.css('left'), 'px', 1) + commentLayerNode.width();

            // it might be necessary to move the content thread left of the bubble node
            if (rightPos > availableWidth) {
                // showing comment thread below the bubble node as right as possible
                top += $bubbleNode.height();
                left -= (rightPos - availableWidth + 2);
            }

            // setting position of thread node
            $commentThread.css({ left: left, top: top });
        }

        /**
         * Creating a bubble node that is displayed for a comment thread in the comments bubble mode.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.multiComment=false]
         *      Whether the comment thread contains more than one comment.
         *  @param {String} [options.colorClassName='']
         *      The class name that is used to set the author specific color for the bubble node.
         *
         * @returns {jQuery}
         *  A new created bubble node for a comment thread. The required handlers are already
         *  registered.
         */
        function createBubbleNode(options) {

            var // the new bubble node
                bubbleNode = $('<div>').addClass(DOM.COMMENTBUBBLENODE_CLASS),
                // whether the corresponding comment thread has more than one child
                isMultiComment = Utils.getBooleanOption(options, 'multiComment', false),
                // an optional class name for the color of the bubble
                colorClassName = Utils.getStringOption(options, 'colorClassName', ''),
                // the link with icon in the bubble node
                linkNode = $('<a tabindex="1">' + Forms.createIconMarkup(isMultiComment ? 'fa-comments-o' : 'fa-comment-o') + '</a>');

            if (colorClassName) {
                // setting the color to the icon in the bubble node
                linkNode.children('.fa').addClass(colorClassName);
                // also setting the color for the background color of the bubble node (for hover and activating)
                if (colorClassName) { bubbleNode.addClass(colorClassName); }
            }

            // appending the icon node
            bubbleNode.append(linkNode);

            // setting marker to recognize that more than one comment is in the thread
            if (isMultiComment) { bubbleNode.addClass(CommentLayer.MULTI_COMMENT_THREAD); }

            // assigning the hover handler and the click to the bubble node
            activateHandlersAtBubbleNode(bubbleNode);

            return bubbleNode;
        }

        /**
         * Updating the icon inside the bubble node of a specified comment thread. If the thread
         * contains more than one comment another icon is displayed compared to a thread with only
         * one comment.
         *
         * @param {HTMLElement|jQuery} commentThread
         *  The comment thread node that will be positioned next to the specified comment bubble
         *  node.
         */
        function updateBubbleNode(commentThread) {

            var // the jQuerified comment thread node
                $commentThread = $(commentThread),
                // the bubble node belonging to the specified comment thread
                bubbleNode = $commentThread.data(CommentLayer.BUBBLE_CONNECT),
                // whehter the thread contains more than one comment
                isMultiCommentThread = $commentThread.children().length > 1,
                // the color for the author of the first comment in the comment thread
                colorClassName = null;

            // switching the icon at the bubble node
            function setBubbleIcon(iconName) {
                var linkNode = $('<a tabindex="1">' + Forms.createIconMarkup(iconName) + '</a>');
                linkNode.children('.fa').addClass(colorClassName);
                bubbleNode.children('a').remove().end().append(linkNode);
            }

            if (bubbleNode) {

                bubbleNode = $(bubbleNode);

                // the author specific color class name
                colorClassName = getCommentColorClassName($commentThread.children().first());

                if (isMultiCommentThread) {
                    setBubbleIcon('fa-comments-o');
                    bubbleNode.addClass(CommentLayer.MULTI_COMMENT_THREAD);
                } else {
                    setBubbleIcon('fa-comment-o');
                    bubbleNode.removeClass(CommentLayer.MULTI_COMMENT_THREAD);
                }
            }

        }

        /**
         * Deleting a specified bubble node of a comment thread.
         *
         * @param {HTMLElement|jQuery} bubbleNode
         *  The comment bubble node, that will be removed from the DOM, after the data
         *  object is cleaned and the handlers are removed.
         */
        function removeBubbleNode(bubbleNode) {

            // removing data object
            $(bubbleNode).removeData(CommentLayer.BUBBLE_CONNECT);

            // removing the click and hover handlers
            deactivateAllHandlersAtBubbleNode(bubbleNode);

            // ... and finally remove the node itself
            $(bubbleNode).remove();
        }

        /**
         * Deactivating the comment bubble mode. This function can be used for clean up of the
         * bubbles and the bubble layer.
         */
        function leaveBubbleMode() {

            var // all bubble nodes inside the bubble node layer
                allBubbles = bubbleLayerNode && bubbleLayerNode.children(DOM.COMMENTBUBBLENODE_SELECTOR);

            _.each(allBubbles, function (bubbleNode) {

                var // the comment thread node for the bubble layer node
                    commentThread = $(bubbleNode).data(CommentLayer.BUBBLE_CONNECT);

                if (commentThread) {
                    // removing data object from comment thread
                    $(commentThread).removeData(CommentLayer.BUBBLE_CONNECT);

                    // making each comment thread visible again and remove modified left alignment
                    $(commentThread).css({ display: 'block', left: '' });
                }

                removeBubbleNode(bubbleNode);
            });

            // setting global marker for bubble mode
            isBubbleMode = false;

            // switching to default view
            // -> this also updates the horizontal and vertical comment layer positions
            self.switchCommentDisplayView(defaultViewMode, { forceSwitch: true });
        }

        /**
         * Creating a comment node that is inserted into a comment thread.
         *
         * @param {String} author
         *  The author of the comment.
         *
         * @param {String} uid
         *  The unique id for the author of the comment.
         *
         * @param {String} date
         *  The date of the comment.
         *
         * @returns {jQuery}
         *  A new created comment node.
         */
        function createCommentNode(author, uid, date) {

            var // the new comment node, handled like a drawing frame
                commentNode = DrawingFrame.createDrawingFrame(DOM.COMMENTNODE_CLASS),
                // the content node inside the comment node
                contentNode = DrawingFrame.getContentNode(commentNode),
                // the author and date node inside the comment node
                metaInfoNode = $('<div>').addClass(DOM.COMMENTMETAINFO_CLASS),
                // the common node for author and date inside the comment meta info node
                authorDateNode = $('<div>').addClass('commentauthordate'),
                // the author node inside the author date node
                authorNode = $('<a>').addClass('commentauthor').text(author),
                // the date node inside the author date node
                dateNode = $('<div>').addClass('commentdate'),
                // the node containing the buttons inside the comment meta info node
                buttonNode = $('<div>').addClass('commentbutton'),
                // the reply button inside the comment button node
                replyButtonNode = $('<a class="reply" tabindex="1">' + Forms.createIconMarkup('fa-reply') + '</a>').addClass(DOM.COMMENTREPLYNODE_CLASS),
                // the delete button inside the comment button node
                deleteButtonNode = $('<a class="closer" tabindex="1">' + Forms.createIconMarkup('fa-trash-o') + '</a>').addClass(DOM.COMMENTCLOSERNODE_CLASS),
                // a text frame node, that will be added into the content node
                textFrameNode = $('<div>').addClass('textframe').attr('contenteditable', true),
                // whether the import is finished
                isImportFinished = self.isImportFinished(),
                // the date format string displayed in the comment node
                dateString = isImportFinished ? app.getView().getDisplayDateString(Date.parse(date), { toLocal: false }) : date,
                // the element containing the photo (as CSS background)
                authorPictureNode = $('<div>').addClass('commentpicture'),
                // the bottom node of a comment, that contains an arrow, if the comments height is decreased
                bottomNode = $('<div>').addClass('commentbottom').append($('<a class="shrinkarrow" tabindex="1">' + Forms.createIconMarkup('fa-sort-desc') + '</a>'));

            // setting a valid date string
            dateNode.text(dateString);

            // preparing node for author and date
            authorDateNode.append(authorNode);
            authorDateNode.append(dateNode);

            // adding tooltips to buttons and picture
            Forms.setToolTip(replyButtonNode, { tooltip: gt('Reply to comment') });
            Forms.setToolTip(deleteButtonNode, { tooltip: gt('Delete comment') });
            Forms.setToolTip(authorPictureNode, { tooltip: author });

            // preparing the node for the buttons
            buttonNode.append(replyButtonNode);
            buttonNode.append(deleteButtonNode);

            if (Utils.TOUCHDEVICE) { // bigger icons and distances on touch devices
                deleteButtonNode.addClass('touch'); // increase distance between buttons
                buttonNode.addClass('touch'); // bigger icons on touch devices
            }

            // adding picture, author, date, reply and close button to meta info node
            metaInfoNode.append(authorPictureNode);
            metaInfoNode.append(authorDateNode);
            metaInfoNode.append(buttonNode);

            // adding author and date to the comment
            commentNode.prepend(metaInfoNode);

            // adding the bottom node to the comment
            commentNode.append(bottomNode);

            // inserting an implicit paragraph into the comment
            textFrameNode.append(model.getValidImplicitParagraphNode());

            // adding classes to the content node
            contentNode.addClass(DrawingFrame.TEXTFRAMECONTENT_NODE_CLASS + ' autoresizeheight').append(textFrameNode);
            // adding classes to the comment node
            commentNode.addClass(DOM.COMMENTNODE_CLASS);

            // saving also author and date at the comment node
            commentNode.attr('data-container-author', author);
            commentNode.attr('data-container-uid', uid || '');
            commentNode.attr('data-container-date', date);

            if (!_.browser.IE) { $(commentNode).attr('contenteditable', true); }

            // add author color at the meta info node, assign picture, ...
            if (isImportFinished) {
                setAuthorPictureAndLink(commentNode);
                activateHandlersAtMetaInfoNode(metaInfoNode);
            }

            return commentNode;
        }

        /**
         * Making a comment thread invisible and also removing its visible range.
         */
        function hideCommentThread(commentThread) {

            // making each comment thread invisible (so it can be made visible individually)
            $(commentThread).css('display', 'none');

            // remove hover highlighting at comment thread without hover event
            removeCommentRangeHover.call(commentThread);
        }

        /**
         * Making a comment thread visible and also drawing its visible range if necessary.
         *
         * @param {jQuery} commentThread
         *  The comment thread, that will be displayed.
         */
        function displayCommentThread(commentThread) {

            // make the comment thread visible
            commentThread.css('display', '');

            // drawing hover highlighting at comment thread without hover event, if display mode 'all' is set
            if (displayMode === CommentLayer.DISPLAY_MODE.ALL) { drawCommentRangeHover.call(commentThread); }
        }

        /**
         * Making in bubble mode the bubble of a specified comment thread visible.
         *
         * @param {jQuery} commentThread
         *  The comment thread, whose bubble will be displayed.
         */
        function displayCommentThreadBubble(commentThread) {

            var // the bubble node belonging to the specified comment thread
                bubbleNode = commentThread.data(CommentLayer.BUBBLE_CONNECT);

            // making bubble node visible
            if (bubbleNode) { $(bubbleNode).css('display', ''); }
        }

        /**
         * Check, whether the commens of the specified author are not visible because of an active filter.
         *
         * @param {String} author
         *  The name of the author that will be checked
         *
         * @returns {Boolean}
         *  Whether the commens of the specified author are not visible because of an active filter.
         */
        function isFilteredAuthor(author) {
            return _.contains(commentsAuthorList, author) && authorFilter && !_.contains(authorFilter, author);
        }

        /**
         * Updating the list of authors that are not filtered. This is especially useful, if an author,
         * whose comments are not visible because of an active filter, inserts a new comment.
         *
         * @param {String} author
         *  The name of the author, who shall be added to or removed from the list of not filtered authors.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.addAuthor=false]
         *      If set to true, the author will be added to the list of not filtered authors. Otherwise
         *      the author will be removed from this list.
         */
        function updateAuthorFilter(author, options) {

            var // whether the specified author shall be removed or added to the author filter list
                addAuthor = Utils.getBooleanOption(options, 'addAuthor', false),
                // whether the author list was modified
                modified = false;

            if (addAuthor) {
                if (!_.contains(authorFilter, author)) {
                    authorFilter.push(author);
                    modified = true;
                }
            } else {
                if (_.contains(authorFilter, author)) {
                    authorFilter = _.without(authorFilter, author);
                    modified = true;
                }
            }

            if (modified) {
                // setting new list of not filtered authors
                self.setAuthorFilter(authorFilter);
            }
        }

        /**
         * Enabling the bubble mode for the comment layer. The comment layer itself is no longer visible,
         * but the bubble layer is filled with comment bubbles. Those bubbles have a hover handler to show
         * the comment range and a click handler to show the comment thread.
         */
        function activateBubbleMode() {

            var // collector for all comment threads inside the comment layer
                allCommentThreads = null,
                // the bubble node for each comment thread
                bubbleNode = null,
                // the currently active target
                activeTarget = '';

            // setting global marker for bubble mode
            isBubbleMode = true;

            // nothing to do (no comments)
            if (!commentLayerNode) { return; }

            // switching to default view (setting predefined view state)
            if (viewMode !== defaultViewMode) {
                self.switchCommentDisplayView(defaultViewMode, { forceSwitch: true });
            }

            // setting height of comment layer node to 0
            $(commentLayerNode).height(0);

            // making comment layer visible
            commentLayerNode.css('display', 'block');

            // deactivating an optionally activated comment node
            activeTarget = model.getActiveTarget();

            if (activeTarget && self.isCommentTarget(activeTarget)) {
                self.deActivateCommentNode(activeTarget);
            }

            // activating the handler at comments and comment threads and draw ranges
            allCommentThreads = commentLayerNode.children(DOM.COMMENTTHREADNODE_SELECTOR);

            _.each(allCommentThreads, function (commentThread) {

                var // the comment threads children
                    allComments = $(commentThread).children();

                // making each comment thread invisible (so it can be made visible individually)
                hideCommentThread(commentThread);

                // deactivating all handlers to avoid multiple registrations
                deactivateAllHandlersAtCommentThread(commentThread);

                // activating only handler for hover over comments (not comment threads)
                if (isEditable) { activateHandlersAtComments(commentThread); }

                // receiving a bubble node
                bubbleNode = createBubbleNode({ multiComment: (allComments.length > 1), colorClassName: getCommentColorClassName(allComments.first()) });

                // assigning the bubble node to the thread and vice versa
                $(commentThread).data(CommentLayer.BUBBLE_CONNECT, Utils.getDomNode(bubbleNode));
                bubbleNode.data(CommentLayer.BUBBLE_CONNECT, commentThread);

                // appending bubble node into bubble layer
                bubbleLayerNode.append(bubbleNode);
            });

            // updating of horizontal and vertical positions in comment layer required
            updateCommentsLayer();
        }

        /**
         * Enable the highlighting for all comments. The ranges are displayed and the
         * connection lines are drawn from the ranges to the comments for all comments
         * in the document.
         */
        function highlightAllComments() {

            var // collector for all comment threads inside the comment layer
                allCommentThreads = null,
                // the currently active target
                activeTarget = null;

            // nothing to do (no comments)
            if (!commentLayerNode) { return; }

            // making comment layer visible
            commentLayerNode.css('display', 'block');

            // activating the handler at comments and comment threads and draw ranges
            allCommentThreads = commentLayerNode.children(DOM.COMMENTTHREADNODE_SELECTOR);

            _.each(allCommentThreads, function (commentThread) {

                // deactivating all handlers to avoid multiple registrations
                deactivateAllHandlersAtCommentThread(commentThread);

                // activating only handler for hover over comments (not comment threads)
                if (isEditable) { activateHandlersAtComments(commentThread); }

                // drawing the 'hover'-highlight for each comment thread without hover event
                // -> but not if the thread is not visible because of filters
                if ($(commentThread).css('display') !== 'none') {
                    drawCommentRangeHover.call(commentThread);
                }
            });

            // activate an active comment node
            activeTarget = model.getActiveTarget();

            if (activeTarget && self.isCommentTarget(activeTarget)) {
                self.activateCommentNode(allComments[activeTarget]);
            }

        }

        /**
         * Enable the highlighting for the selected and the hovered comments. The ranges are displayed
         * and the connection lines are drawn from the ranges to the comments only for those specific
         * comments.
         */
        function highlightSelectedComments() {

            var // collector for all comment threads inside the comment layer
                allCommentThreads = null,
                // the currently active target
                activeTarget = null;

            // nothing to do (no comments)
            if (!commentLayerNode) { return; }

            // making comment layer visible
            commentLayerNode.css('display', 'block');

            // activating the handler at comments and comment threads and draw ranges
            allCommentThreads = commentLayerNode.children(DOM.COMMENTTHREADNODE_SELECTOR);

            _.each(allCommentThreads, function (commentThread) {

                // deactivating all handlers to avoid multiple registrations
                deactivateAllHandlersAtCommentThread(commentThread);

                // remove hover highlighting at comment thread without hover event
                removeCommentRangeHover.call(commentThread);

                // activating handler for hover over comments
                if (isEditable) { activateHandlersAtComments(commentThread); }

                // activating handler for hover over comment threads
                $(commentThread).hover(drawCommentRangeHover, removeCommentRangeHover);
            });

            // activate an active comment node
            activeTarget = model.getActiveTarget();

            if (activeTarget && self.isCommentTarget(activeTarget)) {
                self.activateCommentNode(allComments[activeTarget]);
            }
        }

        /**
         * Disabling the highlighting for the comments. The ranges are not displayed and the
         * connection lines are not drawn from the ranges to the comments. If the option
         * 'hidden' is set to true, the comment layer becomes invisible.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.hidden=false]
         *      If set to true, the comment layer will no longer be displayed.
         */
        function highlightNoComments(options) {

            var // collector for all comment threads inside the comment layer
                allCommentThreads = null,
                // the currently active target
                activeTarget = null,
                // the place holder node for the active comment
                placeHolderNode = null,
                // the logical position of the place holder node of the first comment in the comment thread
                placeHolderPosition = null,
                // the root node for the comment place holder
                rootNode = null,
                // the target in which the comment place holder is located (only set for header and footer)
                marginalTarget = '',
                // whether the comment layer shall be hidden or not
                hidden = Utils.getBooleanOption(options, 'hidden', false);

            // nothing to do (no comments)
            if (!commentLayerNode) { return; }

            // making comment layer visible
            if (!hidden) { commentLayerNode.css('display', 'block'); }

            // deactivate an active comment node
            activeTarget = model.getActiveTarget();

            if (activeTarget && self.isCommentTarget(activeTarget)) {
                self.deActivateCommentNode(activeTarget);

                // ... but activate it again, if hidden is false (without highlighting range)
                if (!hidden) {
                    self.activateCommentNode(allComments[activeTarget]);
                } else {
                    // setting the cursor to the place holder position of the active target comment
                    placeHolderNode = DOM.getCommentPlaceHolderNode(allComments[activeTarget]);

                    marginalTarget = $(placeHolderNode).data(DOM.DATA_TARGETSTRING_NAME) || '';
                    // the root node of the comment place holder (this can be a marginal node)
                    rootNode = marginalTarget ? model.getRootNode(marginalTarget) : model.getNode();

                    placeHolderPosition = Position.getOxoPosition(rootNode, placeHolderNode, 0);
                    if (marginalTarget) {
                        pageLayout.enterHeaderFooterEditMode(rootNode);
                        model.getSelection().setNewRootNode(rootNode);
                    }
                    model.getSelection().setTextSelection(placeHolderPosition);
                }
            }

            // deactivating the handler at comments and comment threads and remove existing highlighting
            allCommentThreads = commentLayerNode.children(DOM.COMMENTTHREADNODE_SELECTOR);

            _.each(allCommentThreads, function (commentThread) {

                // deactivating all handlers at comment thread
                deactivateAllHandlersAtCommentThread(commentThread);

                // remove hover highlighting at comment thread without hover event
                removeCommentRangeHover.call(commentThread);
            });

            // hiding complete comment layer
            if (hidden) { commentLayerNode.css('display', 'none'); }
        }

        /**
         * Modifications that are necessary, if the edit mode of the document changes.
         * In read-only mode the handler that makes the buttons for reply and delete
         * visible can be removed. Then these buttons will never become visible.
         *
         * @param {jQuery.Event} event
         *  The jQuery event object.
         *
         * @param {Boolean} state
         *  The new edit mode state. If set to true, the user can modify the
         *  document, otherwise the user can view the document content, but
         *  cannot edit it.
         */
        function updateEditMode(event, state) {

            var // collector for all comment threads inside the comment layer
                allCommentThreads = null,
                // the currently active target
                activeTarget = null;

            // saving value
            isEditable = state;

            // nothing to do (no comments)
            if (!commentLayerNode) { return; }

            // deactivating the comment handler at comments
            allCommentThreads = commentLayerNode.children(DOM.COMMENTTHREADNODE_SELECTOR);

            // iterating over all comment threads
            _.each(allCommentThreads, function (commentThread) {

                if (state) {
                    // activating the handlers at all comments at a comment thread
                    activateHandlersAtComments(commentThread);
                    // check, if currently a comment node is active
                    // activate an active comment node
                    activeTarget = model.getActiveTarget();

                    if (activeTarget && self.isCommentTarget(activeTarget)) {
                        allComments[activeTarget].find('.commentbutton').addClass('commentbuttonactive');
                    }

                } else {
                    // deactivating the handlers at all comments at a comment thread
                    deactivateHandlersAtComments(commentThread);
                    // making buttons for reply and remove invisible
                    $(commentThread).find('.commentbutton').removeClass('commentbuttonactive commentbuttonhover');
                }
            });

        }

        /**
         * After deleting comment(s) it might be necessary to remove the complete
         * comment layer node.
         * If a bubble layer node exists, it might be removed, too.
         *
         * @returns {Boolean}
         *  Whether the comment layer was removed.
         */
        function removeCommentLayerIfEmpty() {

            var // whether the comment layer was removed
                removedCommentLayer = false;

            if (commentLayerNode && commentLayerNode.children().length === 0) {

                // removing the comment layer node, if it is empty now
                deactivateHandlersAtCommentLayer();
                commentLayerNode.remove();
                commentLayerNode = null;
                removedCommentLayer = true;

                // removing a bubble layer node, too
                if (bubbleLayerNode && bubbleLayerNode.children().length === 0) {
                    bubbleLayerNode.remove();
                    bubbleLayerNode = null;
                }
            }

            return removedCommentLayer;
        }

        /**
         * After deleting comment(s) it might be necessary ot remove the corresponding
         * comment thread.
         * Additionally it might be necessary to remove a corresponding comment bubble,
         * too.
         *
         * @param {jQuery} commentThread
         *  The comment thread, that will be removed, if it is empty.
         *
         * @returns {Boolean}
         *  Whether the specified comment thread was removed.
         */
        function removeCommentThreadIfEmpty(commentThread) {

            var // the comment bubble node belonging to the comment thread node
                bubbleNode = null,
                // whether the comment thread was removed
                removedThread = false;

            if (commentThread.children().length === 0) {

                // removing the data links between the comment thread and its bubble
                bubbleNode = commentThread.data(CommentLayer.BUBBLE_CONNECT);

                if (bubbleNode) {
                    commentThread.removeData(CommentLayer.BUBBLE_CONNECT);
                    removeBubbleNode(bubbleNode);
                }

                // removing handlers from comment thread node
                deactivateAllHandlersAtCommentThread(commentThread);

                // and removing the nodes
                commentThread.remove();

                removedThread = true;
            }

            return removedThread;
        }

        /**
         * Removing one comment node. Before removing the node from the DOM it might
         * be necessary to remove all registered handlers.
         *
         * @param {jQuery} commentNode
         *  The comment node, that will be removed.
         */
        function removeCommentNode(commentNode) {

            // removing all handlers at the comment meta info node.
            deactivateHandlersAtMetaInfoNode(DOM.getCommentMetaInfoNode(commentNode));

            // ... and removing the comment node itself from the DOM.
            $(commentNode).remove();
        }

        /**
         * Handler for the events 'commentlayer:created' and 'commentlayer:removed'. In this
         * function all necessary work can be done, that is needed, if the comment layer on
         * the right side of the document is created or is removed.
         * This includes for example setting a right margin to the page node, so that it is
         * always centered.
         *
         * @param {jQuery.Event} event
         *  The jQuery event object.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Number} [options.width=0]
         *      The width of the comment layer, that will be inserted or was removed (in pixel).
         */
        function updatePagePosition(event, options) {

            var // the width of the created or removed comment layer
                width = Utils.getIntegerOption(options, 'width', 0),
                // the page node
                pageNode = null,
                // the right margin in pixel of the page node
                marginWidth = 0,
                // whether a new value for the margin was calculated
                marginModified = false;

            if (width > 0) {

                pageNode = model.getNode();

                // receiving the old value of the width of the right margin
                marginWidth = Utils.getElementCssLength(pageNode, 'margin-right');

                // expanding or shrinking the right margin of the page node
                if (event.type === 'commentlayer:created') {
                    if (pageNode.hasClass(DOM.COMMENTMARGIN_CLASS)) {
                        Utils.log('Warning: Page node already has margin for comments!'); // this should never happen
                    } else {
                        marginWidth += width;
                        pageNode.addClass(DOM.COMMENTMARGIN_CLASS);
                        pageNode.data(DOM.COMMENTMARGIN_CLASS, width);  // using class name as key to save the width
                        marginModified = true;
                    }
                } else if (event.type === 'commentlayer:removed') {
                    if (!pageNode.hasClass(DOM.COMMENTMARGIN_CLASS)) {
                        Utils.log('Warning: Page node does not have a margin for comments!'); // this should never happen
                    } else {
                        marginWidth -= width;
                        pageNode.removeClass(DOM.COMMENTMARGIN_CLASS);
                        pageNode.removeData(DOM.COMMENTMARGIN_CLASS);
                        marginModified = true;
                    }
                }

                if (marginModified) {
                    pageNode.css('margin-right', marginWidth);

                    // and updating the horizontal position of the comment layer node
                    updateHorizontalCommentLayerPosition();
                }
            }
        }

        /**
         * Generating a list of all comment authors. This is necessary after
         * deleting a comment.
         *
         * @returns {String[]}
         *  The list of all comment authors in the document.
         */
        function createListOfAuthors() {

            var // the list of all comment authors in the document
                authorList = {};

            _.each(comments, function (oneComment) {

                var // comment's author name
                    author = DOM.getCommentAuthor(oneComment);

                // add author to list if not there already
                if (author) {
                    authorList[author] = 1;
                }
            });

            return _.keys(authorList);
        }

        /**
         * After a modification of the filter for the author of the comments, it
         * is necessary that the new filter is applied.
         * This function is triggered by the event ''update:authorfilter'.
         * Comments are made visible using the css style display directly at the
         * comment nodes. If all comments inside a comment thread are made invisible,
         * the comment thread also needs display 'none'. In case of bubble mode, the
         * bubble then also needs display 'none'.
         *
         * Info: This settings must be set in any display mode, even if the complete
         * comment layer is hidden. This has the advantage, that the comment nodes
         * itself are up to date, when the display mode is switched.
         */
        function applyAuthorFilter() {

            var // the comment thread node
                commentThread = null,
                // the bubble node belonging to the thread
                bubbleNode = null;

            if (isActiveAuthorFilterList) {

                // author filter active -> make all comments from non-listed authors invisible
                _.each(comments, function (oneComment) {

                    var // the comment's author name
                        author = DOM.getCommentAuthor(oneComment);

                    commentThread = oneComment.parent();
                    bubbleNode = isBubbleMode ? $(commentThread).data(CommentLayer.BUBBLE_CONNECT) : null;

                    if (_.contains(authorFilter, author)) {

                        oneComment.css('display', '');  // show all listed authors

                        if (bubbleNode) {
                            $(bubbleNode).css('display', '');  // show the corresponding bubble, too
                        } else {
                            // making the comment thread visible (again)
                            displayCommentThread(commentThread);
                        }

                    } else {

                        oneComment.css('display', 'none'); // hide comment of not-listed author -> this is the filter

                        // check, whether all comments inside the comment thread are invisible
                        if (allCommentsInThreadHidden(commentThread)) {

                            if (bubbleNode) {
                                $(bubbleNode).css('display', 'none'); // hide the corresponding bubble, too
                            }

                            // hide the comment thread (also in bubble mode it might be visible in this moment)
                            hideCommentThread(commentThread);
                        }
                    }
                });

            } else {

                // no filter active -> make all comments visible
                _.each(comments, function (oneComment) {

                    commentThread = oneComment.parent();
                    bubbleNode = isBubbleMode ? $(commentThread).data(CommentLayer.BUBBLE_CONNECT) : null;

                    oneComment.css('display', '');

                    if (bubbleNode) {
                        $(bubbleNode).css('display', '');  // hide the corresponding bubble, too
                    } else {
                        // making the comment thread visible (again)
                        displayCommentThread(commentThread);
                    }
                });
            }

            // updating the vertical position of the comments and optionally the visible ranges
            updateCommentsAndRanges();
        }

        /**
         * Helper function to find the next or previous (not filtered) comment node.
         * Iterating through the sorted list of comment place holders, so that there is
         * a clear order during stepping through the comments.
         *
         * @param {Boolean} next
         *  Whether the next or the previous comment shall be found.
         *
         * @param {Number} index
         *  The start index for the search. The index describes the position of the comment
         *  place holder inside the sorted collection 'commentPlaceHolders'.
         *
         * @returns {HTMLElement|Null}
         *  The comment node, or null, if it could not be determined.
         */
        function getNextVisibleCommentNode(next, index) {

            var // whether the search need to be continued
                doContinue = true,
                // the visible comment node
                commentNode = null,
                // the comment place holder node for the comment node
                commentPlaceHolderNode = null,
                // the maximum number of runs
                maxRun = isActiveAuthorFilterList ? commentPlaceHolders.length : 0,
                // the current repeating run
                currentRun = 0;

            while (doContinue) {

                commentNode = null;

                if (next) {
                    index = (index === commentPlaceHolders.length - 1) ? 0 : (index + 1);
                } else {
                    index = (index === 0) ? (commentPlaceHolders.length - 1) : (index - 1);
                }

                commentPlaceHolderNode = commentPlaceHolders[index];

                commentNode = DOM.getCommentPlaceHolderNode(commentPlaceHolderNode);

                currentRun++;

                doContinue = isActiveAuthorFilterList && isFilteredCommentNode(commentNode) && currentRun < maxRun;
            }

            return commentNode;
        }

        /**
         * Checking, whether the specified comment node is not displayed because of an
         * active filter.
         *
         * Additional possible checks:
         * 1. 'isActiveAuthorFilterList' must be true (then 'authorFilter' is defined.
         * 2. The list 'authorFilter' is defined and author of the comment is not in the list.
         *
         * @param {HTMLElement|jQuery} commentNode
         *  The comment node that will be checked.
         *
         * @returns {Boolean}
         *  Whether the specified comment node is not displayed because of an active filter.
         */
        function isFilteredCommentNode(commentNode) {
            return $(commentNode).css('display') === 'none';
        }

        /**
         * Whether all comments in a comment thread are hidden (because of an applied filter)
         * Check, whether all comments inside the comment thread are invisible
         *
         * @param {HTMLElement|jQuery} commentThread
         *  The comment thread node whose child comments will be checked.
         *
         * @returns {Boolean}
         *  Whether all comments inside a specified comment thread are hidden.
         */
        function allCommentsInThreadHidden(commentThread) {
            return _.every($(commentThread).children(), function (comment) {
                return isFilteredCommentNode(comment);
            });
        }

        /**
         * Creating a list that contains all comments that are not filtered. This list can
         * be used for iteration through comments. Filtering can happen with an active
         * author filter.
         *
         * @returns {jQuery[]}
         *  A list of all jQuerified comments, that have the css style 'display' not set to
         *  'none'. This means, that these comments are not filtered.
         */
        function getListOfNonFilteredComments() {

            var // the list of non filtered comments
                visibleComments = [];

            _.each(comments, function (oneComment) {

                if (!isFilteredCommentNode(oneComment)) {
                    visibleComments.push(oneComment);
                }

            });

            return visibleComments;
        }

        /**
         * Comment specific handling after an undo operation.
         */
        function handleUndoAfter() {
            if (undoComments.length > 0) {
                handleUndoComments();
                undoComments = [];
            }
        }

        /**
         * Comment specific handling after reloading the document (for example after
         * canceling a long running action).
         *
         * @param {jQuery.Event} event
         *  The jQuery event object.
         *
         * @param {Object} [state]
         *  The object describing the current document state.
         */
        function handleDocumentReload(event, state) {

            var // the page node
                pageNode = model.getNode(),
                // the new comment layer node
                newCommentLayerNode = DOM.getCommentLayerNode(pageNode),
                // the new comment bubble layer node
                newCommentBubbleLayerNode = DOM.getCommentBubbleLayerNode(pageNode);

            // refreshing the children of the page node that can contain comment place holder nodes
            contentChildren = app.isODF() ? DOM.getContentChildrenOfPage(pageNode) : DOM.getPageContentNode(pageNode);
            // updating the comment layer node
            commentLayerNode = newCommentLayerNode.length > 0 ? newCommentLayerNode : null;
            // updating the comment bubble layer node
            bubbleLayerNode = newCommentBubbleLayerNode.length > 0 ? newCommentBubbleLayerNode : null;

            // restoring data saved at the page node. This might have been removed during deleting comments
            // before the long running action was cancelled or it might have been inserted by a long
            // paste action, that was aborted.
            if (state) {
                if (state.getCommentMargin()) {
                    pageNode.addClass(DOM.COMMENTMARGIN_CLASS);
                    pageNode.data(DOM.COMMENTMARGIN_CLASS, state.commentMargin);  // using class name as key to save the width
                } else {
                    pageNode.removeClass(DOM.COMMENTMARGIN_CLASS);
                    pageNode.removeData(DOM.COMMENTMARGIN_CLASS);
                }
            }

            // refreshing the model
            updateModel();

            // do nothing, if there is no comment layer. If a comment layer was added before (for example during pasting),
            // it was removed before the event 'document:reloaded' was triggered.
            // -> it is necessary to update the model, before leaving
            if (!commentLayerNode) {
                return;
            }

            // reactivating the current display mode
            self.switchCommentDisplayMode(displayMode, { forceSwitch: true });

            // updating the horizontal comment layer position
            updateHorizontalCommentLayerPosition();
        }

        /**
         * Function that is executed after the 'import success' event of the application.
         */
        function importSuccessHandler() {

            // updating the comments model after load from local storage or (but only for ODF) after load with fast load
            if (model.isLocalStorageImport() || (app.isODF() && model.isFastLoadImport())) { updateModel(); }
            // updating the comment threads (not necessary for load from local storage)
            if (!model.isLocalStorageImport()) { updateCommentThreads(); }
            // setting color, time, ... at the comments
            finalizeComments(model.isLocalStorageImport(), model.isFastLoadImport());
            // setting decreased size for large comments
            markCommentsInactiveAfterLoad();
            // switching to the default view mode
            self.switchCommentDisplayView(defaultViewMode, { forceSwitch: true });
            // bubble mode is default on touch devices
            if (Utils.TOUCHDEVICE) { self.switchCommentDisplayMode(defaultDisplayMode); }
            // refresh horizontal and vertical position of comments and the comment layer
            updateCommentsLayer();
            // making the comment layer visible (can be invisible until yet)
            if (self.getCommentLayer()) { self.getCommentLayer().css('visibility', ''); }
            // inform listeners, that a comment layer was created
            if (self.isCommentLayerVisible()) { self.trigger('commentlayer:created', { width: commentLayerNode.width() }); }
            // triggering an update of the author list of all comments
            self.trigger('update:commentauthorlist');

        }

        // methods ------------------------------------------------------------

        /**
         * Whether the document contains comments in the document.
         *
         * @returns {Boolean}
         *  Whether the document contains at least one comment.
         */
        this.isEmpty = function () {
            return comments.length === 0;
        };

        /**
         * Check, whether the specified target string is a valid target for a comment.
         *
         * @param {String} target
         *  The target string.
         *
         * @returns {Boolean}
         *  Whether the specified target string is a valid target for a comment node.
         */
        this.isCommentTarget = function (target) {
            return self.getCommentRootNode(target) !== null;
        };

        /**
         * Provides the array with all comments.
         *
         * @returns {jQuery[]}
         *  The list with all comments.
         */
        this.getComments = function () {
            return comments;
        };

        /**
         * Provides the comment node in the comment layer corresponding to the
         * specified target string.
         *
         * @param {String} target
         *  The target string used to receive the corresponding comment node.
         *
         * @returns {jQuery|null}
         *  The comment node inside the comment layer specified by the target
         *  or null, if target is not defined or the comment node cannot be determined.
         */
        this.getCommentRootNode = function (target) {
            return allComments[target] || null;
        };

        /**
         * Returns the active comment display mode.
         *
         * @returns {String}
         *  The currently active comment display mode.
         */
        this.getDisplayMode = function () {
            return displayMode;
        };

        /**
         * Returns whether there is a visible comment layer right of the page node. This means
         * a comment layer in the 'classic' view. If this is visible, the centering of the page
         * needs to be readjusted.
         *
         * @returns {Boolean}
         *  Whether the comment layer right of the page is visible.
         */
        this.isCommentLayerVisible = function () {
            return comments.length > 0 && viewMode === CommentLayer.VIEW_MODE.CLASSIC && CommentLayer.VISIBLE_COMMENT_THREAD_MODES[displayMode];
        };

        /**
         * Returning a list of all authors of comments, whose comments are visible.
         * This list contains all authors, if no filter is active. If a filter is
         * active, then this filter represents the list of comment authors.
         *
         * @returns {String[]}
         *  The list of all comment authors in the document that are not filtered.
         */
        this.getListOfUnfilteredAuthors = function () {
            return authorFilter ? authorFilter : commentsAuthorList;
        };

        /**
         * Returning a list of all authors of comments.
         *
         * @returns {String[]}
         *  The list of all comment authors in the document.
         */
        this.getListOfCommentAuthors = function () {
            return commentsAuthorList;
        };

        /**
         * Returning the number of authors of comments in the document.
         *
         * @returns {Number}
         *  The number of authors of comments in the document.
         */
        this.getCommentAuthorCount = function () {
            return commentsAuthorList.length;
        };

        /**
         * Setting a new filter for the comment authors.
         *
         * @param {String[]} nameList
         *  The list of those authors, whose comments shall be displayed.
         */
        this.setAuthorFilter = function (nameList) {

            var // the number of comment authors in the document
                maxNumber = commentsAuthorList.length;

            if (nameList && nameList.length < maxNumber) {
                isActiveAuthorFilterList = true;
                authorFilter = nameList;
            } else {
                isActiveAuthorFilterList = false;
                authorFilter = null;
            }

            // trigger that the new filter need to be applied
            self.trigger('update:authorfilter');
        };

        /**
         * Removing an optionally set author filter.
         */
        this.removeAuthorFilter = function () {
            if (isActiveAuthorFilterList) { self.setAuthorFilter(); }
        };

        /**
         * Activating the active mode for a specified comment node.
         *
         * @param {HTMLElement|jQuery} commentNode
         *  The comment node inside the comment layer, that will become the
         *  active target.
         *
         * @param {Object} [selection]
         *  The current selection object. Can be given into this function
         *  for performance reasons.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.deactivate=false]
         *      If set to true, it is checked before the activation of the specified comment,
         *      if another comment needs to be deactivated. This is usually not necessary,
         *      but during handling in 'model.processMouseDown' this simplifies the process
         *      of deactivating a comment node, but allowing clicks into the same comment node
         *      without deactivating and following immediate activating.
         *
         * @returns {String}
         *  The id of the activated comment.
         */
        this.activateCommentNode = function (commentNode, selection, options) {

            var // the unique id of the comment
                id = DOM.getTargetContainerId(commentNode),
                // the author specific color class name
                colorClassName = getCommentColorClassName(commentNode),
                // the position of the upper left corner of the highlighted range
                startPos = null,
                // the range marker object
                rangeMarker = model.getRangeMarker(),
                // the comment thread node
                commentThread = $(commentNode).parent(),
                // the bubble node belonging to the comment thread node
                bubbleNode = null,
                // the optional parent comment node
                parentComment = commentNode,
                // whether another comment need to be deactivated before
                deactivate = Utils.getBooleanOption(options, 'deactivate', false),
                // whether the old and the new target are inside the same thread
                sameThread = false,
                // the old active target
                oldTarget = model.getActiveTarget();

            // do not activate comment, if it is already activated
            if (id === oldTarget) { return id; }

            if (!selection) { selection = model.getSelection(); }

            // also handling deactivating of another comment, if required
            if (deactivate && oldTarget) {
                if (model.isHeaderFooterEditState()) {
                    pageLayout.leaveHeaderFooterEditMode();
                } else {
                    if (self.isCommentTarget(oldTarget) && Utils.getDomNode(commentThread) === Utils.getDomNode(self.getCommentRootNode(oldTarget)).parentNode) { sameThread = true; }
                    self.deActivateCommentNode(oldTarget, selection, { sameThread: sameThread });
                    // this might reduce the height of the comment, so that the current hover line might be wrong (38814)
                    removeCommentRangeHover.call(commentThread);
                }
            }

            model.setActiveTarget(id);
            selection.setNewRootNode(commentNode);

            // deactivating changeTracking temporarily, as long as this comment node is active
            model.getChangeTrack().setChangeTrackSupported(false);

            // removing click highlights of other comment nodes (that have the class 'activerange')
            // -> this is useful during undo, that restores more than one comment
            rangeMarker.removeHighLightRangeByClass('activerange');

            // highlighting of comments is not possible for display type CommentLayer.DISPLAY_MODE.NONE
            if (displayMode === CommentLayer.DISPLAY_MODE.NONE) { return id; }

            // highlighting the commented range (but for comment threads in the color of the parent comment)
            if (DOM.isChildCommentNode(commentNode)) {
                parentComment = commentThread.children().first();
                id = DOM.getTargetContainerId(parentComment);
                colorClassName = getCommentColorClassName(parentComment);
            }

            // making buttons for reply and remove visible
            if (isEditable) { $(commentNode).find('.commentbutton').addClass('commentbuttonactive'); }

            // activating the comment thread
            commentThread.addClass(getActiveCommentThreadColorClassName(parentComment));

            // removing class 'decreased' from all comment nodes inside the thread
            _.each(commentThread.children(), function (comment) { $(comment).removeClass('decreased'); });

            // ... and removing the marker, that it is inactive (increases the height)
            commentThread.removeClass(CommentLayer.INACTIVE_THREAD_CLASS);

            // -> it is not necessary to remove the highlighting with class 'visualizedcommenthover'
            // -> both stay next to each other, until 'deActivateCommentNode' is called.
            startPos = rangeMarker.highLightRange(id, 'visualizedcommentclick fillcolor activerange ' + colorClassName);

            // Drawing a line connection between comment and range
            if (startPos && !isBubbleMode) { drawCommentLineConnection(parentComment, startPos, id, 'highlightcommentlineclick fillcolor activerange ' + colorClassName); }

            if (isBubbleMode) {
                // setting background to bubble
                bubbleNode = commentThread.data(CommentLayer.BUBBLE_CONNECT);
                if (bubbleNode) { $(bubbleNode).addClass('activated'); }
            }

            // the height of the nodes might have been changed -> updateComments must be executed
            // new ordering of comments in side pane
            // -> also the updating the lines, if they are visible in the 'all' mode
            if (!self.isEmpty() && self.isCommentLayerVisible()) { updateCommentsAndRangesAfterActivation(); }

            // returns the activated id
            return id;
        };

        /**
         * Deactivating the active mode, if a comment node is the currently active target.
         *
         * @param {String} target
         *  The target string used to receive the corresponding comment node. If this is
         *  the target of a comment node, this comment node will no longer be the active
         *  node.
         *
         * @param {Object} selection
         *  The current selection object.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.sameThread=false]
         *      If set to true, another comment in the same thread will be activated
         *      immediately after this deactivation. This is only important for the bubble
         *      mode, in which the comment thread must not be made invisible in this case.
         */
        this.deActivateCommentNode = function (target, selection, options) {

            var // the comment thread node
                commentThread = null,
                // the comment node
                commentNode = allComments[target],
                // the parent comment inside the thread
                parentComment = null,
                // the bubble node belonging to the thread
                bubbleNode = null,
                // the id of the parent comment
                parentTarget = null,
                // whether there is a new activated comment in the same thread
                sameThread = Utils.getBooleanOption(options, 'sameThread', false);

            if (commentNode) {

                // Leaving the mode with an active comment target.
                leaveTargetNode(selection);

                // activating changeTracking again, because this comment node is no longer active
                model.getChangeTrack().setChangeTrackSupported(true);

                // making buttons for reply and remove invisible
                commentNode.find('.commentbutton').removeClass('commentbuttonactive');

                commentThread = commentNode.parent();

                // making arrows in lower right corner visible, if comment content node size is decreased
                commentThread.addClass(CommentLayer.INACTIVE_THREAD_CLASS);

                // deactivating the comment thread
                _.each(_.range(1, 8), function (i) { commentThread.removeClass('comment-thread-active-author-' + i); });

                parentComment = commentThread.children().first();
                parentTarget = DOM.getTargetContainerId(parentComment);

                // removing the highlighting of the commented range
                model.getRangeMarker().removeHighLightRange(parentTarget);

                // if all comment ranges shall be visible, the lines need to be redrawn
                if (displayMode === CommentLayer.DISPLAY_MODE.ALL) { drawCommentRangeHover.call(commentThread); }

                // in bubble mode the comment thread can be made invisible again
                // -> but not, if the click happens in another comment in the same thread!
                if (isBubbleMode && !sameThread) { hideCommentThread(commentThread); }

                if (isBubbleMode) {
                    // removing the background from the bubble
                    bubbleNode = commentThread.data(CommentLayer.BUBBLE_CONNECT);
                    if (bubbleNode) { $(bubbleNode).removeClass('activated'); }
                }

                // the height of the nodes might have been changed -> updateComments must be executed
                // new ordering of comments in side pane
                // -> also the updating the lines, if they are visible in the 'all' mode
                if (!self.isEmpty() && self.isCommentLayerVisible()) { updateCommentsAndRangesAfterActivation(); }
            }

        };

        /**
         * Provides a unique ID for the comment in the comment layer and
         * its placeholder element and the comment ranges.
         * Also handling 'broken' range markers, that can exist without
         * an existing comment. In this case the id needs to be further
         * increased.
         *
         * @returns {String}
         *  A unique id.
         */
        this.getNextCommentID = function () {

            var // the next free comment id
                commentId = 'cmt' + placeHolderCommentID++,
                // the range marker object
                rangeMarker = model.getRangeMarker();

            // checking, if there also exist no range markers with this id
            while (rangeMarker.isUsedRangeMarkerId(commentId)) {
                commentId = 'cmt' + placeHolderCommentID++;
            }

            return commentId;
        };

        /**
         * Returning the current author of the application as string
         * with full display name.
         *
         * @returns {String}
         *  The author string for the comment meta info object.
         */
        this.getCommentAuthor = function () {
            return app.getClientOperationName();
        };

        /**
         * Returning the user id of the current author of the application.
         * The value for the uid is already transformed so that it can be
         * used in the operation.
         *
         * @returns {String}
         *  The uid of the author for the comment meta info object.
         */
        this.getCommentAuthorUid = function () {
            return Utils.calculateUserId(ox.user_id);
        };

        /**
         * Returning the current date as string in ISO date format.
         * This function calls DateUtils.getMSFormatIsoDateString() to simulate
         * the MS behavior, that the local time is send a UTC time. The
         * ending 'Z' in the string represents UTC time, but MS always
         * uses local time with ending 'Z'.
         *
         * @returns {String}
         *  The date string for the change track info object.
         */
        this.getCommentDate = function () {
            return DateUtils.getMSFormatIsoDateString({ useSeconds: false });
        };

        /**
         * Returning the comment layer node, if it is already set. It is not
         * created within this function. If it does not exist, null is returned.
         *
         * @returns {jQuery}
         *  The comment layer node
         */
        this.getCommentLayer = function () {
            return commentLayerNode;
        };

        /**
         * Receiving the comment layer node. If it does not exist, it is created. The node is
         * searched as child of div.page. It can happen, that the global commentLayerNode is
         * not defined, for example after loading document from local storage.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.visible=true]
         *      If set to true, the comment layer will be created with visibility value set
         *      to 'hidden'. This is useful for loading from local storage or using fast load,
         *      because then the comments will displayed unordered, until the document is
         *      loaded completely.
         *
         * @returns {jQuery}
         *  The comment layer node
         */
        this.getOrCreateCommentLayerNode = function (options) {

            var // the comment layer node
                commentLayer = DOM.getCommentLayerNode(model.getNode()),
                // the bubble layer node
                bubbleLayer = null,
                // the page content node
                pageContentNode = null,
                // whether the commentlayer shall be visible
                visible = Utils.getBooleanOption(options, 'visible', true);

            // is there already a comment layer node
            if (commentLayer.length > 0) { return commentLayer; }

            // create a new comment layer node
            pageContentNode = DOM.getPageContentNode(model.getNode());

            commentLayer = $('<div>').addClass(DOM.COMMENTLAYER_CLASS);

            bubbleLayer = $('<div>').addClass(DOM.COMMENTBUBBLELAYER_CLASS);

            if (_.browser.Firefox) {
                // workaround for abs pos element, which get the "moz-dragger" but we dont want it
                commentLayer.attr('contenteditable', 'false');
            }

            // comment layer will not be made visible, if specified in options
            if (!visible) { commentLayer.css('visibility', 'hidden'); }

            // Adding the text drawing layer behind an optional footer wrapper
            if (DOM.isFooterWrapper(pageContentNode.next())) { pageContentNode = pageContentNode.next(); }

            // inserting the comment layer node in the page
            pageContentNode.after(commentLayer);

            // inserting the comment bubble layer node in the page
            commentLayer.after(bubbleLayer);

            // saving global variable (this function might is triggered during fast load or local storage load)
            commentLayerNode = commentLayer;

            // saving also the global bubble layer node
            bubbleLayerNode = bubbleLayer;

            // setting the selected view mode for the comment layer
            self.switchCommentDisplayView(viewMode, { forceSwitch: true });

            // setting also the selected display mode for the comment layer
            self.switchCommentDisplayMode(displayMode, { forceSwitch: true });

            // setting the left distance in the div.page node
            updateHorizontalCommentLayerPosition(commentLayerNode);

            // registering handler for mousedown and touchstart at the commentlayer
            activateHandlersAtCommentLayer();

            return commentLayer;
        };

        /**
         * Handler for insertComment operations. Inserts a comment node into the document DOM.
         *
         * @param {Number[]} start
         *  The logical start position for the new comment.
         *
         * @param {String} id
         *  The unique id of the comment.
         *
         * @param {String} author
         *  The author of the comment.
         *
         * @param {String} uid
         *  The unique id for the author of the comment.
         *
         * @param {String} date
         *  The date of the comment.
         *
         * @returns {Boolean}
         *  Whether the comment has been inserted successfully.
         */
        this.insertCommentHandler = function (start, id, author, uid, date, target) {

            var // the new comment node, handled like a drawing frame
                commentNode = null,
                // the thread node for the new comment
                commentThreadNode = $('<div>').addClass(DOM.COMMENTTHREADNODE_CLASS),
                // the bubble node belonging to the thread node
                bubbleNode = null,
                // the place holder element for the drawing in the page content
                commentPlaceHolder = $('<div>', { contenteditable: false }).addClass('inline ' + DOM.COMMENTPLACEHOLDER_CLASS),
                // an object with all range end markers in the document
                endMarker = model.getRangeMarker().getEndMarker(id),
                // the text span for inserting the comment place holder node
                insertSpan = model.prepareTextSpanForInsertion(start, null, target),
                // whether comments of this author are not shown because of a filter -> removing the filter for this author
                isAuthorFiltered = isActiveAuthorFilterList && author && isFilteredAuthor(author),
                // whether this document is completely loaded
                isImportFinished = self.isImportFinished();

            if (!insertSpan) { return false; }

            // inserting the place holder node into the DOM
            if (endMarker) {
                if (DOM.isEmptySpan(insertSpan) && Utils.getDomNode($(insertSpan).prev()) === Utils.getDomNode(endMarker)) { $(insertSpan).remove(); }
                commentPlaceHolder.insertAfter(endMarker);  // no empty span between end marker and comment placeholder node
            } else {
                commentPlaceHolder.insertAfter(insertSpan);
            }

            // marking comment thread in header or footer
            if (target) { commentThreadNode.addClass(DOM.TARGETCOMMENTNODE_CLASS); }

            // generating the comment node
            commentNode = createCommentNode(author, uid, date);

            // adding to its own comment thread
            commentThreadNode.append(commentNode);

            // create comments layer node, if necessary
            if (!commentLayerNode) { commentLayerNode = self.getOrCreateCommentLayerNode(); }

            // inserting the comment thread into the comment layer
            commentLayerNode.append(commentThreadNode);

            // assigning the target id to the comment node and its place holder node
            DOM.setTargetContainerId(commentNode, id);
            DOM.setTargetContainerId(commentPlaceHolder, id);

            // in bubble mode, a new bubble node must be assigned to the comment thread
            if (isBubbleMode) {
                bubbleNode = createBubbleNode({ multiComment: false });
                // assigning the bubble node to the thread and vice versa
                commentThreadNode.data(CommentLayer.BUBBLE_CONNECT, Utils.getDomNode(bubbleNode));
                bubbleNode.data(CommentLayer.BUBBLE_CONNECT, Utils.getDomNode(commentThreadNode));

                // appending bubble node into bubble layer
                bubbleLayerNode.append(bubbleNode);
            }

            // creating links between comment node and comment placeholder node (need to be restored after loading from local storage)
            // -> this must not be done for header or footer (although this is anyhow not supported yet)
            // -> but at least for 'delete all comments' the place holder must be found
            if (target && pageLayout.isIdOfMarginalNode(target)) {
                commentNode.data(DOM.COMMENTPLACEHOLDER_LINK, target);
            } else {
                commentNode.data(DOM.COMMENTPLACEHOLDER_LINK, Utils.getDomNode(commentPlaceHolder));
                commentPlaceHolder.data(DOM.COMMENTPLACEHOLDER_LINK, Utils.getDomNode(commentNode));
            }

            // saving the comments in the model (this is the access to the model, it can cause modification of author list)
            addIntoCommentModel(commentNode, id);

            // restoring the sorted list of all place holders
            updatePlaceHolderCollection();

            // handling the global counter
            updatePlaceHolderCommentID(id);

            // add author color at the meta info node, assign picture, ...
            if (isImportFinished) {
                commentThreadNode.addClass(CommentLayer.INACTIVE_THREAD_CLASS); // marking comment thread as inactive
                updateCommentThreads(); // check, if the new comment is inside the thread of an existing comment
                // updateCommentThreads(start); // check, if the new comment is inside the thread of an existing comment
                assignColorForComment(commentNode);  // assigning color to thread
            }

            // Info: After updateCommentThreads() it is possible, that the comment node is inside a new comment thread!

            // registering the hover handler for the comment thread field
            // -> this needs to be done also after loading from local storage or using fast load
            // -> this is not necessary, if the comment thread node was already removed in updateCommentThreads()
            if (commentThreadNode.parent().length > 0) {
                commentThreadNode.hover(drawCommentRangeHover, removeCommentRangeHover);
                if (isEditable) { activateHandlersAtComments(commentThreadNode); }
            }

            // switching to the default display mode, if 'none' is currently active
            if (displayMode === CommentLayer.DISPLAY_MODE.HIDDEN) {
                self.switchCommentDisplayMode(defaultDisplayMode);
            }

            // updating the comments side pane, setting the vertical offsets
            // -> doing this before calling 'activateCommentNode', so that correct connection lines are drawn
            updateComments();

            // special handling, if the bubble mode is active
            if (isBubbleMode) {
                // adjust the height of the new inserted comment thread, if it is still in the dom (not for new child comments)
                if (Utils.containsNode(model.getNode(), commentThreadNode)) {
                    setCommentThreadNextToBubble(bubbleNode, commentThreadNode);
                } else {
                    // then the bubble node can be removed
                    removeBubbleNode(bubbleNode);
                    bubbleNode = null;
                }

                // handling for undo/redo
                if (model.isUndoRedoRunning()) {
                    // in bubble mode in undo/redo comments next to the bubble must not be visible,
                    // or at maximum one comments should become active and visible
                    // TODO: This needs to be checked for external clients in bubble mode.
                    hideCommentThread(commentNode.parent()); // hide the comment thread
                }

                // updating the bubble node after the comment was inserted
                updateBubbleNode(commentNode.parent());
            }

            // The new comment must be visible, also if a filter is active and the thread is invisible (can happen with undo)
            if (isActiveAuthorFilterList && !isAuthorFiltered && commentNode.parent().css('display') === 'none') {
                if (isBubbleMode) {
                    displayCommentThreadBubble(commentNode.parent()); // in bubble mode at least the bubble needs to be visible again
                } else {
                    displayCommentThread(commentNode.parent());
                }
            }

            // activating the author, if he was filtered out
            if (isAuthorFiltered) { updateAuthorFilter(author, { addAuthor: true }); }

            // collecting all comments that are inserted during undo or redo operation
            if (model.isUndoRedoRunning()) { undoComments.push(commentNode); }

            // triggering, that comment layer became visible, if this is the first comment in classic view mode
            if (isImportFinished && comments.length === 1 && self.isCommentLayerVisible()) { self.trigger('commentlayer:created', { width: commentLayerNode.width() }); }

            repairDisplayName(commentNode);

            return true;
        };

        /**
         * Inserting a comment. This function generates the operations for the ranges,
         * the comment node and the first paragraph. It does not modify the DOM. This
         * function is only executed in the client with edit privileges, not in remote
         * clients.
         */
        this.insertComment = function () {

            return model.getUndoManager().enterUndoGroup(function () {

                var // the operations generator
                    generator = model.createOperationsGenerator(),
                    // the selection object date of comment creation
                    selection = model.getSelection(),
                    // the logical start position of the selection
                    start = selection.getStartPosition(),
                    // the logical end position of the selection
                    end = selection.getEndPosition(),
                    // whether start and end position are in the same paragraph
                    sameParent = Position.hasSameParentComponent(start, end),
                    // created operation
                    newOperation = null,
                    // the new id of the comment
                    commentId = self.getNextCommentID(),
                    // the logical position for the end range
                    endRangePos = sameParent ? Position.increaseLastIndex(end) : end,
                    // the logical position for the comment
                    commentPos = app.isODF() ? start : Position.increaseLastIndex(endRangePos),
                    // the author of the comment
                    commentAuthor = self.getCommentAuthor(),
                    // the authors user id
                    commentUserId = self.getCommentAuthorUid(),
                    // the date of comment creation
                    commentDate = self.getCommentDate(),
                    // the new inserted comment node
                    commentNode = null,
                    // the paragraph styles
                    paragraphStyles = null,
                    // the style id for the comments paragraph
                    commentStyleId = null;

                // checking, that start and end of selection are in non-implicit paragraphs
                model.doCheckImplicitParagraph(start);
                if (!sameParent) { model.doCheckImplicitParagraph(end); }

                // TODO: Handling change tracking correctly

                if (app.isODF()) {

                    // A. inserting the comment itself, the first paragraph and the end range marker
                    // B. in ODF comment node and marker must have the same parent (no comment from paragraph into table)

                    newOperation = { start: commentPos, id: commentId, author: commentAuthor, uid: commentUserId, date: commentDate };
                    generator.generateOperation(Operations.COMMENT_INSERT, newOperation);

                    newOperation = { start: [0], target: commentId };
                    generator.generateOperation(Operations.PARA_INSERT, newOperation);

                    if (selection.hasRange()) {
                        if (selection.getSelectionType() !== 'cell' && !Position.hasSameParentComponent(start, endRangePos, 2)) {
                            // Limit the end range position, if the paragraphs do not have the same parent (using number 2)
                            endRangePos = getBestODFRangeEndPosition(start);
                            // if the new endRangePos is in the same paragraph as the comment node, it need to be increased by 1
                            if (_.isEqual(_.initial(commentPos), _.initial(endRangePos))) { endRangePos = Position.increaseLastIndex(endRangePos); }
                        }

                        newOperation = { start: endRangePos, position: 'end', type: 'comment', id: commentId };
                        generator.generateOperation(Operations.RANGE_INSERT, newOperation);
                    }

                } else {

                    // inserting two comment ranges, the comment itself and the first paragraph
                    // no target for all operations required
                    // -> inserting comment is only possible in main document

                    newOperation = { start: start, position: 'start', type: 'comment', id: commentId };
                    generator.generateOperation(Operations.RANGE_INSERT, newOperation);

                    newOperation = { start: endRangePos, position: 'end', type: 'comment', id: commentId };
                    generator.generateOperation(Operations.RANGE_INSERT, newOperation);

                    newOperation = { start: commentPos, id: commentId, author: commentAuthor, date: commentDate, uid: commentUserId };
                    generator.generateOperation(Operations.COMMENT_INSERT, newOperation);

                    newOperation = { start: [0], target: commentId };
                    generator.generateOperation(Operations.PARA_INSERT, newOperation);

                    // checking, if the paragraph style inside comments is already available
                    paragraphStyles = model.getStyleCollection('paragraph');
                    commentStyleId = paragraphStyles.getStyleIdByName(commentParagraphStyleName);

                    if (paragraphStyles && paragraphStyles.isDirty(commentStyleId)) {
                        // inserting parent of text frame style to document
                        generator.generateOperation(Operations.INSERT_STYLESHEET, {
                            attrs: paragraphStyles.getStyleSheetAttributeMap(commentStyleId),
                            type: 'paragraph',
                            styleId: commentStyleId,
                            styleName: paragraphStyles.getName(commentStyleId),
                            parent: paragraphStyles.getParentId(commentStyleId),
                            uiPriority: paragraphStyles.getUIPriority(commentStyleId),
                            default: false
                        });
                        paragraphStyles.setDirty(commentStyleId, false);
                    }

                    // reducing font size and line height inside the comment
                    newOperation = { start: [0], target: commentId, attrs: { styleId: commentStyleId } };
                    generator.generateOperation(Operations.SET_ATTRIBUTES, newOperation);
                }

                // reducing the distance of paragraphs inside the comment
                newOperation = { start: [0], target: commentId, attrs: commentDefaultAttributes };
                generator.generateOperation(Operations.SET_ATTRIBUTES, newOperation);

                // apply all collected operations
                model.applyOperations(generator);

                // activating the new inserted comment
                commentNode = allComments[commentId];
                self.activateCommentNode(commentNode, model.getSelection());
                // setting cursor into text frame
                selection.setTextSelection([0, 0]);
            });

        };

        /**
         * Inserting a comment as a reply to an already existing comment. This function generates the
         * operations for the ranges, the comment node and the first paragraph. It does not modify
         * the DOM. This function is only executed in the client with edit privileges, not in remote
         * clients.
         *
         * @param {jQuery} node
         *  The comment node, for that the reply node will be generated.
         */
        this.replyToComment = function (node) {

            return model.getUndoManager().enterUndoGroup(function () {

                var // the operations generator
                    generator = model.createOperationsGenerator(),
                    // the handler for the range markers
                    rangeMarker = model.getRangeMarker(),
                    // created operation
                    newOperation = null,
                    // the new id of the comment
                    commentId = self.getNextCommentID(),
                    // the author of the comment
                    commentAuthor = self.getCommentAuthor(),
                    // the authors user id
                    commentUserId = self.getCommentAuthorUid(),
                    // the date of comment creation
                    commentDate = self.getCommentDate(),
                    // the new inserted comment node
                    commentNode = null,
                    // the place holder node for the comment
                    placeHolderNode = null,
                    // the comment thread
                    commentThread = node.parent(),
                    // whether the comment is inside header or footer
                    isMarginalComment = app.isODF() && DOM.isTargetCommentNode(commentThread),
                    // the last comment in the comment thread
                    lastCommentInThread = (node.next().length > 0) ? commentThread.children().last() : node,
                    // the target of the last comment node in the thread
                    lastCommentTarget = DOM.getTargetContainerId(lastCommentInThread),
                    // the range start marker node
                    startMarkerNode = rangeMarker.getStartMarker(lastCommentTarget),
                    // the range end marker node
                    endMarkerNode = rangeMarker.getEndMarker(lastCommentTarget),
                    // whether a marginal end marker node is available
                    isMarginalEndMarker = isMarginalComment && endMarkerNode,
                    // the target node for the start marker and end marker
                    rootNode = isMarginalEndMarker ? DOM.getClosestMarginalTargetNode(endMarkerNode) : model.getNode(),
                    // the target in which the comment is located (only set for header and footer)
                    marginalTarget = isMarginalEndMarker ? DOM.getTargetContainerId(rootNode) : '',
                    // the logical range start marker position
                    startMarkerPos = startMarkerNode ? Position.getOxoPosition(rootNode, startMarkerNode) : null,
                    // the logical range end marker position
                    endMarkerPos = endMarkerNode ? Position.getOxoPosition(rootNode, endMarkerNode) : null,
                    // the logical position for the end range
                    startRangePos = null,
                    // the logical position for the end range
                    endRangePos = null,
                    // the logical position for the comment
                    commentPos = null,
                    // the paragraph styles
                    paragraphStyles = null,
                    // the style id for comment paragraphs.
                    commentStyleId = null;

                // leaving the target node (so that model.finalizeOperations() does not modify the operation:
                // -> restoring the target node, if it was not the deleted node
                leaveTargetNode();

                // TODO: Handling change tracking correctly

                if (app.isODF()) {

                    // B. in ODF only the comment node needs to be inserted, directly behind the end range node
                    // of the parent comment or directly behind the previous child comment
                    if (endMarkerPos) { endRangePos = endMarkerPos; }

                    // the logical position for the comment
                    if (endRangePos) {
                        commentPos = Position.increaseLastIndex(endRangePos); // the first child comment -> first position behind the end range marker
                    } else {
                        placeHolderNode = DOM.getCommentPlaceHolderNode(lastCommentInThread); // -> not first child -> position behind preceeding comment
                        marginalTarget = $(placeHolderNode).data(DOM.DATA_TARGETSTRING_NAME) || '';
                        // the root node of the comment place holder (this can be a marginal node)
                        if (marginalTarget) { rootNode = model.getRootNode(marginalTarget); }

                        commentPos = Position.increaseLastIndex(Position.getOxoPosition(rootNode, placeHolderNode));
                    }

                    newOperation = { start: commentPos, id: commentId, author: commentAuthor, uid: commentUserId, date: commentDate };
                    if (marginalTarget) { newOperation.target = marginalTarget; }
                    generator.generateOperation(Operations.COMMENT_INSERT, newOperation);

                    newOperation = { start: [0], target: commentId };
                    generator.generateOperation(Operations.PARA_INSERT, newOperation);

                } else {

                    if (startMarkerPos && endMarkerPos) {

                        startRangePos = Position.increaseLastIndex(startMarkerPos);

                        // the new end positions for the end range marker and the comment node itself are dependent
                        // from a check, if the start range marker is inside the same paragraph
                        if (_.isEqual(_.initial(startMarkerPos), _.initial(endMarkerPos))) {
                            endRangePos = Position.increaseLastIndex(endMarkerPos, 3); // increasing position by 3: one for startmarker, one for endmarker, one for comment
                        } else {
                            endRangePos = Position.increaseLastIndex(endMarkerPos, 2); // increasing position by 2: one for endmarker, one for comment
                        }

                        commentPos = Position.increaseLastIndex(endRangePos);

                        // no target for all operations required
                        // -> inserting comment is only possible in main document

                        newOperation = { start: startRangePos, position: 'start', type: 'comment', id: commentId };
                        generator.generateOperation(Operations.RANGE_INSERT, newOperation);

                        newOperation = { start: endRangePos, position: 'end', type: 'comment', id: commentId };
                        generator.generateOperation(Operations.RANGE_INSERT, newOperation);

                        newOperation = { start: commentPos, id: commentId, author: commentAuthor, uid: commentUserId, date: commentDate };
                        generator.generateOperation(Operations.COMMENT_INSERT, newOperation);

                        newOperation = { start: [0], target: commentId };
                        generator.generateOperation(Operations.PARA_INSERT, newOperation);

                    } else {

                        // this is not a valid OOXML range. start or end range node are missing
                        // -> insert only a new comment without range

                        placeHolderNode = DOM.getCommentPlaceHolderNode(lastCommentInThread); // -> position behind preceeding comment
                        commentPos = Position.increaseLastIndex(Position.getOxoPosition(rootNode, placeHolderNode));

                        newOperation = { start: commentPos, id: commentId, author: commentAuthor, uid: commentUserId, date: commentDate };
                        generator.generateOperation(Operations.COMMENT_INSERT, newOperation);

                        newOperation = { start: [0], target: commentId };
                        generator.generateOperation(Operations.PARA_INSERT, newOperation);
                    }

                    // the paragraph styles
                    paragraphStyles = model.getStyleCollection('paragraph');
                    // setting comment style (in reply to it cannot be dirty, because there is alreay a valid comment in the document)
                    commentStyleId = paragraphStyles.getStyleIdByName(commentParagraphStyleName);
                    // reducing font size and line height inside the comment
                    newOperation = { start: [0], target: commentId, attrs: { styleId: commentStyleId } };
                    generator.generateOperation(Operations.SET_ATTRIBUTES, newOperation);
                }

                // reducing distance of paragraphs inside the comment
                newOperation = { start: [0], target: commentId, attrs: commentDefaultAttributes };
                generator.generateOperation(Operations.SET_ATTRIBUTES, newOperation);

                // apply all collected operations
                model.applyOperations(generator);

                // activating the new inserted comment
                commentNode = allComments[commentId];
                self.activateCommentNode(commentNode, model.getSelection());
                // setting cursor into text frame
                model.getSelection().setTextSelection([0, 0]);
            });

        };

        /**
         * Deleting a specific comment node. This function generates the delete operation
         * and does not change the DOM. It is only executed in the client with edit
         * privileges.
         *
         * @param {HTMLElement|jQuery} node
         *  The comment node inside the comment layer for that a delete operation
         *  will be generated.
         *  If the option 'all' is set, this node is ignored.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.all=false]
         *      If set to true, all comments in the document are removed.
         */
        this.deleteComment = function (node, options) {

            var // the operations generator
                generator = model.createOperationsGenerator(),
                // the handler for the range markers
                rangeMarker = model.getRangeMarker(),
                // whether all comments shall be removed
                removeAll = Utils.getBooleanOption(options, 'all', false),
                // whether the specified node is the first node in a thread
                isParentComment = removeAll ? false : DOM.isParentCommentNode(node),
                // a collector for all comments, that need to be deleted
                allComments = removeAll ? $(comments) : (isParentComment ? $(node).parent().children() : $(node)),
                // a collector for all positions (and targets) to be deleted
                allPositions = {},
                // the place holder position of the comment node
                placeHolderPosition = null,
                // the target node of the saved place holder node
                placeHolderRootNode = null,
                // the target of a saved place holder position
                placeHolderTarget = '',
                // a collector for all deleted targets
                deletedIds = {},
                // the number of previous removed elements in the paragraph containing the place holder node for the active comment
                correction = 0,
                // whether the user needs to be asked, because filtered comments are removed, too
                askUser = isParentComment && isActiveAuthorFilterList && _.find(allComments, isFilteredCommentNode),
                // caching the current target node (might be the deleted comment, but can also be any other target)
                targetCache = cacheTargetNode(),
                // the resulting promise
                promise = null;

            // collecting all positions that need to be removed, if one comment is removed
            function collectAllPositions(oneComment) {

                var // the id of the comment node
                    localCommentId = DOM.getTargetContainerId(oneComment),
                    // the place holder node for the comment
                    placeHolderNode = DOM.getCommentPlaceHolderNode(oneComment),
                    // the target in which the comment is located (only set for header and footer)
                    target = $(placeHolderNode).data(DOM.DATA_TARGETSTRING_NAME) || '',
                    // the root node of the comment place holder (this can be a marginal node)
                    rootNode = model.getRootNode(target),
                    // the logical position of the place holder node (only searching in main document, ignoring target)
                    commentPosition = Position.getOxoPosition(rootNode, placeHolderNode, 0),
                    // the range start marker node
                    startMarkerNode = rangeMarker.getStartMarker(localCommentId),
                    // the range end marker node
                    endMarkerNode = rangeMarker.getEndMarker(localCommentId),
                    // the logical range start marker position
                    startMarkerPos = startMarkerNode ? Position.getOxoPosition(rootNode, startMarkerNode) : null,
                    // the logical range end marker position
                    endMarkerPos = endMarkerNode ? Position.getOxoPosition(rootNode, endMarkerNode) : null;

                // saving the place holder for later setting of cursor
                if (!placeHolderPosition) {
                    placeHolderPosition = _.clone(commentPosition);
                    if (app.isODF()) {
                        placeHolderTarget = target;
                        placeHolderRootNode = rootNode;
                    }
                }

                deletedIds[localCommentId] = 1;  // collecting all deleted targets for later usage

                if (!allPositions[target]) { allPositions[target] = []; }  // using empty string as key

                allPositions[target].push(commentPosition);
                // deleting the comment and also the start range marker and the end range marker, if they exist
                if (endMarkerPos) { allPositions[target].push(endMarkerPos); }
                if (startMarkerPos) { allPositions[target].push(startMarkerPos); }
            }

            // counting the removed elements inside the paragraph that contains the place holder node for
            // an active comment node. This is necessary for setting the cursor to the position of the place
            // holder node after removal.
            function countPreviousRemovedNodesInParagraph(placeHolderPosition, allPositions) {

                var // the number of removed nodes in the paragraph before the place holder position
                    previousNodes = 0,
                    // the logical position of the paragraph
                    paraPos = _.initial(placeHolderPosition),
                    // the text position of the place holder node inside its paragraph
                    textPos = _.last(placeHolderPosition);

                _.each(allPositions, function (onePos) {
                    if (_.isEqual(paraPos, _.initial(onePos)) && _.last(onePos) < textPos) {
                        previousNodes++;
                    }
                });

                return previousNodes;
            }

            // applying the operations for deleting the comments. I necessary, the user was
            // asked before.
            function doDeleteComment() {

                // leaving the target node (so that model.finalizeOperations() does not modify the operation:
                // -> restoring the target node, if it was not the deleted node
                leaveTargetNode();

                _.each(allComments, function (commentNode) {
                    collectAllPositions(commentNode);
                });

                // sort all logical positions -> comments and ranges might be in arbitrary order
                _.each(allPositions, function (allPosInOneTarget, oneTarget) {
                    // for (var oneTarget in allPositions) {
                    allPosInOneTarget.sort(Utils.compareNumberArrays);

                    // generating operations
                    _.each(allPosInOneTarget, function (onePos) {
                        generator.generateOperation(Operations.DELETE, oneTarget ? { start: onePos, target: oneTarget }  : { start: onePos });
                    });
                });

                // reverting operation order -> delete from back to front
                generator.reverseOperations();

                // sending delete operation for the comment
                model.applyOperations(generator);

                // Setting cursor to the position of the placeholder (39945).
                // It might be necessary to decrease the last index, because some nodes in the paragraph were removed.
                if (app.isODF() && placeHolderTarget) {  // the deleted comment is inside header or footer
                    model.getSelection().setNewRootNode(placeHolderRootNode);
                    model.setActiveTarget(placeHolderTarget);
                }

                correction = countPreviousRemovedNodesInParagraph(placeHolderPosition, allPositions['']); // not required in marginal node
                if (correction > 0) {
                    placeHolderPosition[placeHolderPosition.length - 1] -= correction;
                    if (app.isODF() && _.last(placeHolderPosition) < 0) { placeHolderPosition[placeHolderPosition.length - 1] = 0; } // TODO handle positions in header/footer
                }
                model.getSelection().setTextSelection(placeHolderPosition);

                if (targetCache.activeTarget) {
                    // activating changeTracking again, because this comment node is no longer active
                    model.getChangeTrack().setChangeTrackSupported(true);
                }

            }

            // asking the user, if some filtered comments will be removed, too.
            // Or if the user want to delete all comments.
            if (askUser || removeAll) {
                promise = model.showDeleteWarningDialog(
                  gt('Delete Comments'),
                  // choose the appropriate dialog text for askUser or removeAll
                  ((askUser) ? gt('At least one filtered comment will be removed, too. Do you want to continue?') : gt('All comments in the document will be removed. Do you want to continue?'))
                );
                // return focus to editor after 'No' button
                promise.fail(function () { app.getView().grabFocus(); });
            } else {
                promise = $.when();
            }

            // delete contents on resolved promise
            return promise.done(doDeleteComment);
        };

        /**
         * Removing one comment element from the comment layer node. The parameter
         * is the place holder node in the page content.
         *
         * @param {jQuery} placeHolderNode
         *  The place holder node in the page content
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.keepDrawingLayer=false]
         *      If set to true, the comment in the comment layer corresponding to the
         *      comment place holder is NOT removed. This is for example necessary after
         *      a split of paragraph. The default is 'false' so that place holder node
         *      and comment in comment layer are removed together.
         */
        this.removeFromCommentLayer = function (placeHolderNode, options) {

            var // the comment node in the comment layer
                commentNode = DOM.getCommentPlaceHolderNode(placeHolderNode),
                // the comment thread node
                commentThread = null,
                // the bubble node belonging to the thread in bubble mode
                bubbleNode = null,
                // whether the comment thread was removed
                removedCommentThread = false,
                // whether the drawings in the drawing layer or comments in the comment layer shall not be removed
                // -> this is necessary after splitting a paragraph
                keepDrawingLayer = Utils.getBooleanOption(options, 'keepDrawingLayer', false),
                // the target of the comment
                commentTarget = DOM.getTargetContainerId(commentNode),
                // whether the comment layer node was visible before removing the comment
                visibleBefore = false,
                // the old width of the comment layer
                oldWidth = visibleBefore ? commentLayerNode.width() : 0,
                // whether the comment is in header or footer
                isMarginalComment = false;

            if (keepDrawingLayer) {

                // removing only the place holder node
                $(placeHolderNode).removeData(DOM.COMMENTPLACEHOLDER_LINK);
                $(placeHolderNode).remove();
            } else {

                commentThread = $(commentNode).parent();

                isMarginalComment = app.isODF() && DOM.isTargetCommentNode(commentThread);

                // removing the data links between the comment and its placeholder
                $(commentNode).removeData(DOM.COMMENTPLACEHOLDER_LINK);
                $(placeHolderNode).removeData(DOM.COMMENTPLACEHOLDER_LINK);

                // in bubble mode, the complete thread needs to be deactivated, also
                // if only a child comment is removed. Therefore it is necessary to
                // deactivate the thread, before the (child) comment node is removed.
                if (isBubbleMode) { self.deActivateCommentNode(commentTarget); }

                // deleting the comment node and the placeholder node
                removeCommentNode(commentNode);
                $(placeHolderNode).remove();

                // deleting the complete comment thread, if it is empty now
                removedCommentThread = removeCommentThreadIfEmpty(commentThread);

                // maybe the bubble needs to be updated after deleting the comment
                if (isBubbleMode && !removedCommentThread) {
                    updateBubbleNode(commentThread);
                }

                // making comment thread invisible, if filters are active and all other comments are
                // invisible. This can happen, if a child comment is removed.
                if (!removedCommentThread && isActiveAuthorFilterList && allCommentsInThreadHidden(commentThread)) {
                    hideCommentThread(commentThread); // hide the complete comment thread

                    // in bubble mode, the bubble also needs to be hidden
                    if (isBubbleMode) {
                        bubbleNode = commentThread.data(CommentLayer.BUBBLE_CONNECT);
                        if (bubbleNode) { $(bubbleNode).css('display', 'none'); } // hide the corresponding bubble, too
                    }
                }

                // deleting the highlighting of the range and the connection line
                model.getRangeMarker().removeHighLightRange(commentTarget);

                // deactivating an active comment node (necessary after undoing an
                // insertion of a comment, but before removing node from model, 37868)
                if (model.getActiveTarget() && model.getActiveTarget() === commentTarget) {
                    self.deActivateCommentNode(commentTarget);
                }

                // check if the comment layer is visible
                visibleBefore = self.isCommentLayerVisible();
                oldWidth = visibleBefore ? commentLayerNode.width() : 0;

                // ... updating the model collectors (accessing the model)
                if (!_.isEmpty(comments)) {
                    removeFromCommentModel(commentTarget, isMarginalComment);
                }

                // removing comments layer, if it is empty now
                removeCommentLayerIfEmpty();

                // inform others, that a visible comment layer was removed
                if (visibleBefore && comments.length === 0) {
                    self.trigger('commentlayer:removed', { width: oldWidth });
                }

                // new ordering of comments in side pane
                if (!self.isEmpty()) { updateCommentsAndRanges({ ignoreMissing: isMarginalComment }); }
            }
        };

        /**
         * Removing all comments from the comment layer, that have place holder nodes
         * located inside the specified node. The node can be a paragraph that will be
         * removed. In this case it is necessary, that the comments in the comment layer
         * are removed, too.
         * Info: The ranges corresponding to the comments are not handled inside this
         * function.
         *
         * @param {HTMLElement|jQuery} node
         *  The node, for which all comments in the comment layer will be removed. This
         *  assumes that the node contains comment place holder nodes. Only the comments
         *  in the comment layer will be removed. This function does not take care of the
         *  comment place holder nodes itself.
         */
        this.removeAllInsertedCommentsFromCommentLayer = function (node) {

            var // the collection of all comment place holder nodes
                allPlaceHolder = $(node).find(DOM.COMMENTPLACEHOLDER_NODE_SELECTOR),
                // whether the comment layer node is visible before removal
                visibleBefore = false,
                // the old width of the comment layer node
                oldWidth = 0;

            // starting to remove the comments, if there are place holder nodes
            if (allPlaceHolder.length > 0) {

                // check if the comment layer is visible
                visibleBefore = self.isCommentLayerVisible();
                oldWidth = visibleBefore ? commentLayerNode.width() : 0;

                _.each(allPlaceHolder, function (placeHolderNode) {

                    var // the comment node corresponding to the place holder node
                        commentNode = DOM.getCommentPlaceHolderNode(placeHolderNode),
                        // the comment thread node
                        commentThread = $(commentNode).parent(),
                        // the id of the comment node
                        id = DOM.getTargetContainerId(placeHolderNode),
                        // whether this comment is in header or footer
                        isMarginalComment = app.isODF() && DOM.isTargetCommentNode(commentThread);

                    if (commentNode) {
                        removeCommentNode(commentNode);
                        // deleting the complete comment thread, if it is empty now
                        removeCommentThreadIfEmpty(commentThread);
                    } else {
                        Utils.warn('removeAllInsertedCommentsFromCommentLayer(): failed to find comment node for place holder node!');
                    }

                    // updating the model collectors ...
                    removeFromCommentModel(id, isMarginalComment);
                });

                // removing comments layer, if it is empty now
                removeCommentLayerIfEmpty();

                // inform others, that a visible comment layer was removed
                if (visibleBefore && comments.length === 0) {
                    self.trigger('commentlayer:removed', { width: oldWidth });
                }

                // ...  and restoring the sorted list of all place holders
                updatePlaceHolderCollection();
            }

        };

        /**
         * An existing connection line between comment and a highlighted comment context needs
         * to be updated, if for example an external operation, triggers a repaint of the range
         * markers. In this case it is necessary, that also the connection line is repainted.
         * This function is therefore triggered from the range marker class.
         *
         * @param {String} id
         *  The unique id of the comment node (and range marker). This can be used to identify the
         *  corresponding comment node.
         *
         * @param {Object} startPos
         *  An object containing the properties x and y. These represent the pixel position of
         *  the upper left corner of the range, that will be connected to the comment. The pixel
         *  positions are received using the jQuery 'offset' function and relative to the page.
         *
         * @param {jQuery} allOverlayChildren
         *  A collection with all children of the range overlay node. This is only required for
         *  restoring the connection line nodes with the correct class names.
         */
        this.updateCommentLineConnection = function (id, startPos, allOverlayChildren) {

            var // the comment node corresponding to the id
                commentNode = self.getCommentRootNode(id),
                // a string containing all classes of the connection lines
                allClassString = '';

            if (commentNode) {
                // the classes need to be restored
                if (allOverlayChildren && allOverlayChildren.length > 0) {

                    Utils.iterateArray(allOverlayChildren, function (child) {
                        if (DOM.getRangeMarkerId(child) === id && $(child).is(commentLineSelector)) {
                            allClassString = $(child).attr('class');  // creating string with all classes
                            drawCommentLineConnection(commentNode, startPos, id, allClassString);
                            return Utils.BREAK;
                        }
                    }, { reverse: true }); // reverse order -> click nodes behind hover nodes
                }
            }

        };

        /**
         * After splitting a paragraph with comment place holder in the new paragraph (that
         * was cloned before), it is necessary that the comments in the comment layer
         * updates its link to the place holder comment in the new paragraph.
         *
         * @param {Node|jQuery} paragraph
         *  The paragraph node.
         */
        this.repairLinksToPageContent = function (paragraph) {

            var // whether the paragraph is inside header or footer (only supported for ODF)
                isMarginalNode = app.isODF() && DOM.isMarginalNode(paragraph),
                // the target of the marginal
                target = isMarginalNode ? DOM.getTargetContainerId(DOM.getClosestMarginalTargetNode(paragraph)) : '',
                // the function that is used to find the place holder nodes. If only a paragraph is updated
                // it is sufficient to search for children. In the case of a page content node, 'find' needs to be used
                searchFunction = DOM.isParagraphNode(paragraph) ? 'children' : 'find',
                // whether each single node needs to be checked
                checkMarginal = app.isODF() && (searchFunction === 'find');

            _.each($(paragraph)[searchFunction](DOM.COMMENTPLACEHOLDER_NODE_SELECTOR), function (placeHolder) {

                var comment = $(DOM.getCommentPlaceHolderNode(placeHolder));

                if (checkMarginal && DOM.isMarginalNode(placeHolder)) {
                    target = DOM.getTargetContainerId(DOM.getClosestMarginalTargetNode(placeHolder));
                }

                if (target) {
                    comment.data(DOM.COMMENTPLACEHOLDER_LINK, target);
                } else {
                    comment.data(DOM.COMMENTPLACEHOLDER_LINK, placeHolder);
                }
            });

            // restoring the sorted list of all place holders
            updatePlaceHolderCollection();
        };

        /**
         * For zoom levels smaller than 100, the app content node (the parent of the page) gets the
         * property 'overflow' set to hidden. Without an additional padding, the comment will become
         * invisible. Therefore it is necessary to check for an additional padding, if the zoom
         * level is changed.
         *
         * @returns {String}
         *  The width of the additional padding, that is necessary, so that the comment stay visible.
         */
        this.getAppContentPaddingWidth = function () {

            return comments.length > 0 ? commentDefaultWidth + 'px' : '0px';
        };

        /**
         * When the document is saved in the local storage, it is necessary to remove all classes that
         * handle a temporary highlighting of a comment node. This is not neccessary for the visible
         * comment range or the line connection to the comment, because these nodes are in an overlay
         * node, that is not stored in the local storage.
         */
        this.clearCommentThreadHighlighting = function () {

            // helper function for iteration
            function removeColorClasses(commentThread) {
                // removing all classes 'comment-thread-active-author-X' and 'comment-thread-hovered-author-X'
                // where X goes from 1 to 8
                _.each(_.range(1, 8), function (i) {
                    $(commentThread).removeClass('comment-thread-active-author-' + i + ' comment-thread-hovered-author-' + i + ' comment-author-' + i);
                });
            }

            // only necessary, if a comment layer node exists
            if (commentLayerNode) {
                // iterating over all comments
                _.each(commentLayerNode.children(), function (oneCommentThread) {
                    removeColorClasses(oneCommentThread);
                    // removing also highlight classes at comment nodes
                    $(oneCommentThread).find('.commentbuttonactive').removeClass('commentbuttonactive');
                });
            }
        };

        /**
         * When pasting content, that contains comments, it is necessary, that the comments get a new
         * unique target (and all operations for content inside the comment).
         *
         * @param {Object[]} operations
         *  The collection with all paste operations.
         */
        this.handlePasteOperationTarget = function (operations) {

            var // a container for all comment ids
                allTargets = [],
                // a container for all range marker ids of comments
                allRangeMarkerTargets = {};

            // replacing the comment ids
            function replaceTarget(ops, target1, target2) {

                _.each(ops, function (op) {
                    if (op) {
                        if (op.id === target1) { op.id = target2; }
                        if (op.target === target1) { op.target = target2; }
                    }
                });
            }

            // collecting all 'id's of insertComment operations and assign new target values
            _.each(operations, function (operation) {
                if (operation && operation.name === Operations.COMMENT_INSERT) {
                    allTargets.push(operation.id);
                } else if (operation && operation.name === Operations.RANGE_INSERT && operation.type === 'comment') {
                    allRangeMarkerTargets[operation.id] = 1;
                }
            });

            // iterating over all registered comment ids
            if (allTargets.length > 0) {
                _.each(allTargets, function (oldTarget) {
                    var newTarget = self.getNextCommentID();
                    // iterating over all operations, so that also the range markers will be updated
                    replaceTarget(operations, oldTarget, newTarget);
                    delete allRangeMarkerTargets[oldTarget];
                });
            }

            // are there still unmodified range marker targets (maybe only the start marker was pasted)
            _.each(_.keys(allRangeMarkerTargets), function (oldTarget) {
                var newTarget = self.getNextCommentID();
                // iterating over all operations
                replaceTarget(operations, oldTarget, newTarget);
            });

        };

        /**
         * Activating or deactivating a comment node on a small device. This function
         * is necessary, because of the 'early exit' of processMouseDown on small devices.
         *
         * @param {jQuery.Event} event
         *  The jQuery event object.
         *
         * @param {String} activeTarget
         *  The target string used to receive the corresponding comment node.
         */
        this.handleCommentSelectionOnSmallDevice = function (event, activeTarget) {

            var // a selected drawing node
                drawingNode = $(event.target).closest(DrawingFrame.NODE_SELECTOR);

            if (drawingNode.length > 0) {
                if (DOM.isCommentNode(drawingNode)) {
                    // checking, if this was a click inside a comment
                    self.activateCommentNode(drawingNode, model.getSelection(), { deactivate: true });
                }
            } else {
                //  leaving comment, restore original rootNode
                self.deActivateCommentNode(activeTarget, model.getSelection());
            }

        };

        /**
         * Typically comments can only be pasted into the main document. In this case the
         * comment node together with its range node(s) and the content inside the comment
         * will be pasted. But if the paste target is not the main document, but for example
         * a header or footer or another comment, some information need to be removed. In this
         * case the comment information will not be pasted.
         *
         * This means that the operations that have the target property set to a comment id
         * will be removed. Additionally the insertComment and insertRange operations that
         * have the specified id(s) need to be replaced by insertText operations of length 1.
         * This is the secure version, that avoids to remove insertComment or insertRange,
         * because otherwise the logical positions of all following operations need to be
         * recalculated. This functionality is based on 40089.
         *
         * Info: The inserted array of operations will be returned modified from this function.
         *
         * @param {Object[]} operations
         *  A list of all paste operations
         *
         * @returns {Object[]}
         *  A (modified) list of all paste operations
         */
        this.handleCommentOperationsForPasting = function (operations) {

            var // a list of all comment IDs
                allIDs = collectAllCommentIDs(operations),
                // the new inserted text instead of comment node or range node
                text = ' ',
                // the collector for the new operations
                newOps = {},
                // the collector for the new delete operations (to avoid position recalculation)
                newDeleteOps = [];

            // helper function to collect all IDs from the insertComment operations
            function collectAllCommentIDs(operations) {

                var allInsertCommentOps = _.filter(operations, function (operation) {
                    return operation.name && operation.name === Operations.COMMENT_INSERT;
                });

                return _.map(allInsertCommentOps, function (operation) {
                    return operation.id;
                });
            }

            // iterating over all collected comment IDs
            if (allIDs && allIDs.length > 0) {

                // removing all operations that have the specified ids as target property
                _.each(allIDs, function (id) {
                    operations = _.reject(operations, function (operation) {
                        return (operation.target && operation.target === id);
                    });
                });

                // ... collecting the insertComment and insertRange operations with the specified ids
                _.each(allIDs, function (id) {
                    _.each(operations, function (operation, index) {
                        if ((operation.name === Operations.COMMENT_INSERT || operation.name === Operations.RANGE_INSERT) && operation.id === id) {
                            var newOp = { name: Operations.TEXT_INSERT, text: text, start: _.clone(operation.start) },
                                newDeleteOp = { name: Operations.DELETE, start: _.clone(operation.start) };

                            if (operation.target) {
                                newOp.target = operation.target;
                                newDeleteOp.target = operation.target;
                            }

                            newOps[index] = newOp;
                            newDeleteOps.push(newDeleteOp);
                        }
                    });

                });

                // ... and finally replacing the operations in the array
                _.each(newOps, function (newOp, index) {
                    operations[index] = newOp;
                });

                // and adding the delete operations in reverse order
                if (newDeleteOps && !_.isEmpty(newDeleteOps)) { operations = operations.concat(newDeleteOps.reverse()); }
            }

            return operations;
        };

        /**
         * Switching the display type for the comments. Allowed parameters for the
         * specified mode are the values defined in 'CommentLayer.DISPLAY_MODE'.
         *
         * @param {String} mode
         *  The mode that defines the highlighting and the visibility of the comments.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.forceSwitch=false]
         *      If set to true, the display mode of the comment layer is set, even if
         *      it is not modified. This is especially required, when a new comment
         *      layer is created, after a previous comment layer was removed with
         *      deleting the last comment.
         */
        this.switchCommentDisplayMode = function (mode, options) {

            var // whether the comment layer is visible before the switch
                isVisibleBefore = false,
                // whether the comment layer is visible after the switch
                isVisibleAfter = false,
                // whether it is necessary to force a switch of the view mode
                forceSwitch = Utils.getBooleanOption(options, 'forceSwitch', false),
                // the width of a visible comment layer node
                oldWidth = 0;

            if (displayMode === mode && !forceSwitch) { return; } // nothing to do

            // registering, if commentlayer is visible before switch
            isVisibleBefore = self.isCommentLayerVisible();

            // saving the width of the comment layer, if it is visible
            if (isVisibleBefore) { oldWidth = commentLayerNode.width(); }

            // leaving bubble mode, if it is currently active
            // -> this can be used to clean up bubbles and connections between threads and bubbles
            if (displayMode === CommentLayer.DISPLAY_MODE.BUBBLES) { leaveBubbleMode(); }

            // setting the new display mode
            displayMode = mode;

            // after setting display mode, it can be checked if the comment layer is visible after switch
            isVisibleAfter = self.isCommentLayerVisible();

            // ... doing mode specific things
            switch (displayMode) {

                // no comment highlighting
                case CommentLayer.DISPLAY_MODE.NONE:
                    highlightNoComments();
                    break;

                    // only selected and hovered comment is highlighted
                case CommentLayer.DISPLAY_MODE.BUBBLES:
                    activateBubbleMode();
                    break;

                // only selected and hovered comment is highlighted
                case CommentLayer.DISPLAY_MODE.SELECTED:
                    highlightSelectedComments();
                    break;

                // all comments are highlighted
                case CommentLayer.DISPLAY_MODE.ALL:
                    highlightAllComments();
                    break;

                // the comment layer is hidden
                case CommentLayer.DISPLAY_MODE.HIDDEN:
                    highlightNoComments({ hidden: true });
                    break;

                default:
                    Utils.warn('CommentLayer.switchCommentDisplayMode(): unknown comment display mode');
            }

            // if a author filter is active, it need to be applied
            // -> in bubble mode not all bubbles will be visible
            // -> if comment layer is visible, empty threads are not displayed
            if (isActiveAuthorFilterList) { applyAuthorFilter(); }

            // ... inform others about changed visibility of comment layer
            if (isVisibleBefore && !isVisibleAfter) {
                self.trigger('commentlayer:removed', { width: oldWidth });
            } else if (isVisibleAfter && !isVisibleBefore) {
                self.trigger('commentlayer:created', { width: commentLayerNode.width() });
            }

        };

        /**
         * Returns the active comment view mode.
         *
         * Info: The comment view is only used for testing reasons. Therefore this function
         * can be removed later.
         *
         * @returns {String}
         *  The currently active comment view mode.
         */
        this.getCommentDisplayView = function () {
            return viewMode;
        };

        /**
         * Switching the display view for the comments. Allowed parameters for the
         * specified view are 'classic', 'modern' and 'avantgard'.
         *
         * Info: The comment view is only used for testing reasons. Therefore this function
         * can be removed later.
         *
         * @param {String} mode
         *  The mode that defines the view of the comments.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.forceSwitch=false]
         *      If set to true, the view of the comment layer is set, even if it
         *      is not modified. This is especially required, when a new comment
         *      layer is created, after a previous comment layer was removed with
         *      deleting the last comment.
         */
        this.switchCommentDisplayView = function (mode, options) {

            var // whether it is necessary to force a switch of the view mode
                forceSwitch = Utils.getBooleanOption(options, 'forceSwitch', false);

            if (((viewMode === mode) && !forceSwitch) || !commentLayerNode) { return; } // nothing to do

            viewMode = mode;

            switch (viewMode) {

                case CommentLayer.VIEW_MODE.MODERN:

                    commentLayerNode.removeClass('classicview avantgardview').addClass('modernview');
                    commentLayerNode.children().removeClass('classicview avantgardview').addClass('modernview');

                    $(commentLayerNode).height(0);

                    break;

                case CommentLayer.VIEW_MODE.CLASSIC:
                    // make comment layer visible and setting height and background-color
                    commentLayerNode.removeClass('modernview avantgardview').addClass('classicview');
                    commentLayerNode.children().removeClass('modernview avantgardview').addClass('classicview');

                    $(commentLayerNode).css('height', '100%');  // using height of the page

                    break;

                case CommentLayer.VIEW_MODE.AVANTGARD:
                    // shifting comment layer to the left and setting increased box shadow to all comment threads
                    commentLayerNode.removeClass('classicview modernview').addClass('avantgardview');
                    commentLayerNode.children().removeClass('classicview modernview').addClass('avantgardview');

                    $(commentLayerNode).height(0);

                    break;

                default:
                    Utils.warn('CommentLayer.switchCommentDisplayView(): unknown comment display view');
            }

            // setting the horizontal offset of the comment layer node
            commentLayerOffset = CommentLayer.HORIZONTAL_COMMENTLAYER_OFFSET[viewMode];

            // updating the comments and horizontal position of comment layer
            updateCommentsLayer();
        };

        /**
         * Selecting and activating the next or the previous comment from the comment layer.
         * If there is no comment, nothing happens.
         * If there is only one comment and this is not active, it will be activated.
         * If there is only one comment and this is already active, nothing happens.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.next=true]
         *      If set to true, the following comment will be activated, otherwise the previous
         *      comment.
         *      If set to true and no comment is active, the first comment will be activated.
         *      Otherwise the last comment will be activated.
         */
        this.selectNextComment = function (options) {

            var // whether the next or the previous comment shall be selected
                next = Utils.getBooleanOption(options, 'next', true),
                // whether a comment node is the active target
                isActiveComment = false,
                // the active target
                activeTarget = '',
                // the comment node to be selected
                commentNode = null,
                // the comment place holder node for the comment to be selected
                commentPlaceHolderNode = null,
                // whehter it is necessary to deactivate another comment node
                deactivate = false,
                // the selection object
                selection = null,
                // the position of the currently active comment in the jQuery list 'commentPlaceHolders'
                index = 0,
                // the list of comments that can be used for iteration (only visible comments, not filtered comments)
                visibleComments = isActiveAuthorFilterList ? getListOfNonFilteredComments() : comments;

            // nothing to do (no visible comments)
            if (visibleComments.length === 0) { return; }

            // checking, if a comment is the active node
            activeTarget = model.getActiveTarget();
            isActiveComment = activeTarget && self.isCommentTarget(activeTarget);

            if (visibleComments.length === 1) {

                // exit, if this one comment is already active
                if (isActiveComment) { return; }

                // activate the one and only not filtered comment
                commentNode = visibleComments[0];

            } else {

                // using the ordered list 'commentPlaceHolders', that is also used for the order
                // of the comments in the comment layer.
                // Is currently a comment active? Then activate next or previous (not filtered) comment.
                // Is no comment active? Then activate first or last (not filtered) comment

                if (isActiveComment) {

                    // activate next or previous comment and deactivate the current node
                    commentNode = self.getCommentRootNode(activeTarget);
                    commentPlaceHolderNode = DOM.getCommentPlaceHolderNode(commentNode);

                    // searching this place holder node in the sorted list of all place holder nodes
                    index = commentPlaceHolders.index(commentPlaceHolderNode);

                    deactivate = true;

                } else {

                    // setting the index for the following search of comment
                    index = next ? -1 : commentPlaceHolders.length;
                }

                // searching the next or previous not filtered comment node
                commentNode = getNextVisibleCommentNode(next, index);

                // could not find valid comment node -> do nothing
                if (!commentNode) { return; }
            }

            // receiving selection object ...
            selection = model.getSelection();

            // switching to the default display mode, if 'none' is currently active
            if (displayMode === CommentLayer.DISPLAY_MODE.HIDDEN) {
                self.switchCommentDisplayMode(defaultDisplayMode);
            }

            // activating the comment thread in bubble mode
            if (isBubbleMode) { showCommentThreadInBubbleMode(commentNode); }

            // .. activating the new comment node ...
            self.activateCommentNode(commentNode, selection, { deactivate: deactivate });

            // ... and setting cursor into the activated comment at the first position (not always [0,0])
            selection.setTextSelection(Position.getFirstPositionInParagraph(selection.getRootNode(), [0]));
        };

        // initialization -----------------------------------------------------

        app.onInit(function () {

            // receiving the global model object
            model = app.getModel();

            // setting the global content root node
            appContentRootNode = app.getView().getContentRootNode();

            // setting the global undo manager
            undoManager = model.getUndoManager();

            // setting the global page layout object
            pageLayout = model.getPageLayout();

            // registering handler for preparations after successful document loading
            self.waitForImportSuccess(importSuccessHandler);

            // registering handler for loading document with fastload (early for drawings in comments, not allowed in ODF)
            if (!app.isODF()) { model.one('fastload:done', updateModel); }

            // registering handler for first inserting of page breaks
            model.one('pageBreak:after', updateAfterPageBreak);

            // update comments when document edit mode changes
            self.listenTo(model, 'change:editmode', updateEditMode);

            // updating the model after the document was reloaded (after cancelling long running actions)
            self.listenTo(model, 'document:reloaded', handleDocumentReload);

            // register handler for document modifications (triggered very often)
            self.listenTo(model, 'update:absoluteElements removed:marginal', updateCommentsLayer);

            // register handler for changes of page settings
            self.listenTo(model, 'change:pageSettings', updateHorizontalCommentLayerPosition);

            // register handler for tasks, when the header or footer are no longer active and all marginal nodes are exchanged
            self.listenTo(pageLayout, 'updateMarginal:leave', additionalUpdatePlaceHolderCollection);

            // register handler for tasks after undo operations
            self.listenTo(undoManager, 'undo:after redo:after', handleUndoAfter);

            // listening the creation or removal of comment layer node
            self.on('commentlayer:created commentlayer:removed', updatePagePosition);

            // listening to modifications of the author filter list
            self.on('update:authorfilter', applyAuthorFilter);

            // whether the document can be edited
            isEditable = model.getEditMode();
        });

    } // class CommentLayer

    // constants --------------------------------------------------------------

    /**
     * Display types for the comments:
     *     'ALL':      All ranges of the comments are always visible
     *     'SELECTED': Only the selected and the hovered comment range are visible
     *     'BUBBLES':  Shows small bubbles on right document margin instead of comment layer
     *     'NONE':     No ranges of any comment is visible
     *     'HIDDEN':   The comments layer is not visible
     *
     * @constant
     */
    CommentLayer.DISPLAY_MODE = {
        HIDDEN: 'hidden',
        NONE: 'none',
        BUBBLES: 'bubbles',
        SELECTED: 'selected',
        ALL: 'all'
    };

    /**
     * A collector containing only those display modes, in which the comment thread are visible.
     * This is not the case for 'hidden', where no comments are visible and for 'bubbles', where
     * bubbles represent the comment thread.
     */
    CommentLayer.VISIBLE_COMMENT_THREAD_MODES = {
        none: 1,
        selected: 1,
        all: 1
    };

    /**
     * A collector containing only those display modes, in which the comment lines that connect
     * the comments and the comment selections are always visible.
     */
    CommentLayer.ALWAYS_VISIBLE_COMMENT_LINES_MODE = {
        all: 1
    };

    /**
     * View modes for the comments:
     *     'MODERN':    All comment threads on right side of the page, no comment layer visible
     *     'CLASSIC':   All comment threads on right side of the page in a visible comment layer
     *     'AVANTGARD': All comment threads overlapping with page on right side, no comment layer visible
     *
     * @constant
     */
    CommentLayer.VIEW_MODE = {
        MODERN: 'modern',
        CLASSIC: 'classic',
        AVANTGARD: 'avantgard'
    };

    /**
     * The horizontal offset of the comment layer relativ to the right border of the page
     * in pixel for the different view modes.
     */
    CommentLayer.HORIZONTAL_COMMENTLAYER_OFFSET = {
        modern: 18,
        classic: 1,
        avantgard: -20
    };

    /**
     * The name of the jQuery data key that is used as connection between a comment thread
     * and its bubble node.
     */
    CommentLayer.BUBBLE_CONNECT = 'bubbleConnect';

    /**
     * Marker class to differentiate comment thread bubbles with only one comment from those
     * with more than one comment.
     */

    CommentLayer.MULTI_COMMENT_THREAD = 'multicomment';

    /**
     * The width of the bubble layer node in pixel.
     */
    CommentLayer.BUBBLE_LAYER_WIDTH = 25;

    /**
     * The maximum height of the comment content node in pixel, if the comment thread is not active.
     * This value is also needs to be adapted in less file as 'max-height' for class 'inactivethread'.
     */
    CommentLayer.COMMENT_CONTENT_MAXHEIGHT = 36;

    /**
     * The class name used to marked comment threads that are not active.
     */
    CommentLayer.INACTIVE_THREAD_CLASS = 'inactivethread';

    // export =================================================================

    // derive this class from class TriggerObject
    return TriggerObject.extend({ constructor: CommentLayer });

});
