/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Stefan Eckert <stefan.eckert@open-xchange.com>
 */

define('io.ox/office/text/format/stylesheetmixin', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/locale/localedata',
    'io.ox/office/textframework/view/textdialogs',
    'io.ox/office/textframework/utils/operations',
    'io.ox/office/textframework/utils/position',
    'io.ox/office/textframework/utils/dom',
    'gettext!io.ox/office/text/main'
], function (Utils, LocaleData, Dialogs, Operations, Position, DOM, gt) {

    'use strict';

    // mix-in class StylesheetMixin ============================================

    /**
     *
     */
    function StylesheetMixin(app) {

        var self = this;

        var paraStyles = this.getStyleCollection('paragraph');
        var charStyles = this.getStyleCollection('character');

        var paraClearAttributes = null;
        var charClearAttributes = null;

        var paragraphEngToLocale = {};
        var paragraphLocaleToEng = {};

        // private methods ----------------------------------------------------

        function getParaClearAttributes() {
            if (!paraClearAttributes) {
                paraClearAttributes = paraStyles.buildNullAttributes();
            }
            return paraClearAttributes;
        }

        function getCharClearAttributes() {
            if (!charClearAttributes) {
                charClearAttributes = charStyles.buildNullAttributes();
                delete charClearAttributes.character.url;
            }
            return charClearAttributes;
        }

        function getCurrentParaAttributes() {
            var endSpan = Position.getDOMPosition(self.getCurrentRootNode(), self.getSelection().getEndPosition()).node.parentElement;

            var character = charStyles.getElementAttributes(endSpan);
            if (character.character.fillColor && character.character.fillColor.type === 'auto') {
                //workaround for Bug 38656 (libre office cant show strike and underline when character fill color is set)
                delete character.character.fillColor;
            }
            var paragraph =  paraStyles.getElementAttributes(endSpan.parentElement);

            var baseAttributes = {
                character: character.character,
                paragraph: paragraph.paragraph,
                changes: { inserted: '', modified: '', removed: '' }
            };

            return baseAttributes;
        }

        function generateOverwriteStyleIdOp(generator, paraPos, paraNode, styleId) {

            var paraLength = Position.getParagraphLength(paraNode, paraPos);
            if (paraLength) {
                var startChar = _.clone(paraPos);
                var endChar = _.clone(paraPos);
                startChar.push(0);
                endChar.push(paraLength);

                generator.generateOperation(Operations.SET_ATTRIBUTES, {
                    attrs: getCharClearAttributes(),
                    start: startChar,
                    end: endChar
                });
            }

            generator.generateOperation(Operations.SET_ATTRIBUTES, {
                attrs: getParaClearAttributes(),
                start: paraPos,
                end: paraPos
            });

            generator.generateOperation(Operations.SET_ATTRIBUTES, {
                attrs: { styleId: styleId },
                start: paraPos,
                end: paraPos
            });
        }

        function getEnglishName(name) {
            return Utils.translateWordFromDatabase(name, paragraphLocaleToEng);
        }

        function getTranslatedNameByStyleId(styleId) {
            return Utils.translateWordFromDatabase(paraStyles.getName(styleId), paragraphEngToLocale);
        }

        function trim(string) {
            return string.toLowerCase().replace(/\s/g, '');
        }

        function getFuzzyStyleId() {
            var names = paraStyles.getStyleSheetNames();
            var normalIds = _.keys(names);
            var normalNames = _.values(names);
            var trimedIds = [];
            var trimedNames = [];

            _.each(normalIds, function (normalId) {
                trimedIds.push(trim(normalId));
            });
            _.each(normalNames, function (normalValue) {
                trimedNames.push(trim(normalValue));
            });

            for (var i = 0; i < arguments.length; i++) {
                var arg = arguments[i];
                var trimmedArg = trim(arg);
                var index = trimedIds.indexOf(trimmedArg);
                if (index >= 0) {
                    return normalIds[index];
                }
                index = trimedNames.indexOf(trimmedArg);
                if (index >= 0) {
                    return normalIds[index];
                }
            }
        }

        /**
         * Returns the warning text for a paragraph style that exists already,
         * to be used in the text input dialog asking for the name of a new or
         * renamed paragraph style.
         */
        function getExistsAlreadyWarningText(styleId) {
            var styleName = getTranslatedNameByStyleId(styleId);
            var warningText =
                //#. warning text shown in the text input dialog to create new paragraph style
                //#. "%1$s" is the name of the existing paragraph style
                //#, c-format
                gt('Paragraph style "%1$s" exists already.', _.noI18n(styleName));
            return warningText;
        }

        // methods ------------------------------------------------------------

        this.showInsertNewStyleDialog = function (name) {
            var baseAttributes = getCurrentParaAttributes();
            var para = $('<div>');
            var span = $('<span>');

            para.data({ attributes: { paragraph: baseAttributes.paragraph } });
            span.data({ attributes: { paragraph: baseAttributes.character } });

            var counter = 1;

            var validator = function (newName) {
                var engNewName = getEnglishName(newName);
                var oldStyleId = getFuzzyStyleId(newName, engNewName);
                return oldStyleId ? getExistsAlreadyWarningText(oldStyleId) : null;
            };

            para.addClass('caption');
            para.css({
                display: 'block',
                cursor: 'default',
                overflow: 'hidden',
                margin: 0,
                paddingTop: 10
            });

            span.css({
                textOverflow: 'ellipsis',
                whiteSpace: 'nowrap',
                display: 'block',
                overflow: 'hidden'
            });

            paraStyles.updateElementFormatting(para, { baseAttributes: baseAttributes });
            charStyles.updateElementFormatting(span, { baseAttributes: baseAttributes });

            span.appendTo(para);

            if (!name) {
                name = gt('style %1$d', counter);
                while (paraStyles.containsStyleSheetByName(name)) {
                    counter++;
                    name = gt('style %1$d', counter);
                }
            }

            var selection = self.getSelection();
            var activeRootNode = selection.getRootNode();
            var startPos = selection.getStartPosition();
            var endPos = selection.getEndPosition();
            var paraPos = _.initial(startPos);
            var cursorIndex = Math.round((_.last(startPos) + _.last(endPos)) / 2);
            var exampleText = $(Position.getDOMPosition(activeRootNode, paraPos).node).text();

            if (exampleText.length > 20) {
                exampleText = exampleText.substring(cursorIndex - 20, cursorIndex + 40);

                if (exampleText.length > 3) {
                    span.text('...' + exampleText);
                } else {
                    span.text(name);
                }
            } else if (exampleText.length > 3) {
                span.text(exampleText);
            } else {
                span.text(name);
            }

            var dialog = new Dialogs.InputValidateDialog(app.getView(), {
                title: gt('Create paragraph style'),
                value: name,
                validator: validator,
                maxLength: 100
            });

            dialog.on('show', function () {
                dialog.append(para);
            });

            var def = dialog.show();

            def = def.then(function () {
                var styleId = dialog.getText();
                var selection = self.getSelection();

                var startPos = selection.getStartPosition();
                var paraPos = _.initial(startPos);
                var paraNode = self.getCurrentRootNode();
                var localGenerator = self.createOperationsGenerator();

                self.getUndoManager().enterUndoGroup(function () {
                    self.doCheckImplicitParagraph(startPos);

                    localGenerator.generateOperation(Operations.INSERT_STYLESHEET, {
                        attrs: baseAttributes,
                        type: 'paragraph',
                        styleId: styleId,
                        styleName: styleId,
                        custom: true,
                        uiPriority: 0
                    });

                    generateOverwriteStyleIdOp(localGenerator, paraPos, paraNode, styleId);

                    self.applyOperations(localGenerator);
                });
            }, function (result) {
                // reopen the dialog if the style name entered in this dialog is invalid
                return (result === 'invalid') ? self.showInsertNewStyleDialog(dialog.getText()) : result;
            });

            return def;
        };

        this.showRenameStyleDialog = function (styleId, fieldName) {

            var styleName = paraStyles.getName(styleId);
            var styleLabel = Utils.translateWordFromDatabase(styleName, paragraphEngToLocale);

            function validator(newName) {
                var engNewName = getEnglishName(newName);
                var oldStyleId = getFuzzyStyleId(newName, engNewName);
                if (newName === styleName || newName === styleLabel) { return null; }
                return oldStyleId ? getExistsAlreadyWarningText(oldStyleId) : null;
            }

            var dialog = new Dialogs.InputValidateDialog(app.getView(), {
                title: gt('Rename paragraph style "%1$s"', styleLabel),
                value: fieldName || styleLabel,
                validator: validator,
                maxLength: 100
            });

            var def = dialog.show();

            def = def.then(function () {
                var newName = dialog.getText();
                var localGenerator = self.createOperationsGenerator();
                localGenerator.generateOperation(Operations.CHANGE_STYLESHEET, {
                    type: 'paragraph',
                    styleId: styleId,
                    styleName: newName,
                    uiPriority: paraStyles.getUIPriority(styleId),
                    hidden: paraStyles.isHidden(styleId),
                    custom: paraStyles.isCustom(styleId)
                });
                self.applyOperations(localGenerator);
            }, function (result) {
                // reopen the dialog if the style name entered in this dialog is invalid
                return (result === 'invalid') ? self.showRenameStyleDialog(styleId, dialog.getText()) : null;
            });

            return def;
        };

        this.changeStylesheet = function (styleId) {
            var baseAttributes = getCurrentParaAttributes();
            var styleName = paraStyles.getName(styleId);

            var selection = self.getSelection();
            var startPos = selection.getStartPosition();
            var paraPos = _.initial(startPos);
            var paraNode = self.getCurrentRootNode();
            var localGenerator = self.createOperationsGenerator();

            self.getUndoManager().enterUndoGroup(function () {

                self.doCheckImplicitParagraph(startPos);

                localGenerator.generateOperation(Operations.CHANGE_STYLESHEET, {
                    attrs: baseAttributes,
                    type: 'paragraph',
                    styleId: styleId,
                    styleName: styleName,
                    uiPriority: paraStyles.getUIPriority(styleId)
                });

                generateOverwriteStyleIdOp(localGenerator, paraPos, paraNode, styleId);

                self.applyOperations(localGenerator);

            });
        };

        this.deleteStylesheet = function (styleId) {
            var localGenerator = self.createOperationsGenerator();
            var defaultParaStyle = paraStyles.getDefaultStyleId();
            var defaultAtts = { styleId: defaultParaStyle };
            var allP = self.getNode().find('.p').not(DOM.MARGINAL_NODE_SELECTOR);
            var marginalP = self.getPageLayout().getHeaderFooterPlaceHolder().find('.p');
            var emptyAttr = {};

            _.each(allP, function (para) {
                para = $(para);
                if (DOM.isImplicitParagraphNode(para)) {
                    return;
                }
                var data = para.data();
                var attrs = Utils.getOption(data, 'attributes', emptyAttr);
                var paraStyleId = Utils.getStringOption(attrs, 'styleId', defaultParaStyle);
                if (paraStyles.isChildOfOther(paraStyleId, styleId)) {
                    var paraPos = Position.getOxoPosition(self.getNode(), para);

                    localGenerator.generateOperation(Operations.SET_ATTRIBUTES, {
                        attrs: defaultAtts,
                        start: paraPos,
                        end: paraPos
                    });
                }
            });

            _.each(marginalP, function (para) {
                para = $(para);
                if (DOM.isImplicitParagraphNode(para)) {
                    return;
                }
                var data = para.data();
                var attrs = Utils.getOption(data, 'attributes', emptyAttr);
                var paraStyleId = Utils.getStringOption(attrs, 'styleId', defaultParaStyle);
                var rootNode = DOM.getClosestMarginalTargetNode(para);
                if (paraStyles.isChildOfOther(paraStyleId, styleId) && rootNode.length) {
                    var paraPos = Position.getOxoPosition(rootNode, para);

                    localGenerator.generateOperation(Operations.SET_ATTRIBUTES, {
                        attrs: defaultAtts,
                        start: paraPos,
                        end: paraPos,
                        target: DOM.getTargetContainerId(rootNode)
                    });
                }
            });

            localGenerator.generateOperation(Operations.DELETE_STYLESHEET, {
                type: 'paragraph',
                styleId: styleId
            });

            self.applyOperations(localGenerator);
        };

        this.getTranslatedNameByStyleId = getTranslatedNameByStyleId;

        // initialization -----------------------------------------------------

        this.waitForSuccess(LocaleData.loadResource('io.ox/office/textframework/resource/paragraphstylenames', { merge: true }), function (data) {
            paragraphEngToLocale = _.clone(data);
            paragraphLocaleToEng = {};

            for (var key in data) {
                var entry = data[key];
                var newEntry = key.replace('(\\d+)', '$1');
                var newKey = entry.replace('$1', '(\\d+)');

                paragraphLocaleToEng[newKey] = newEntry;
                paragraphLocaleToEng[trim(newKey)] = trim(newEntry);
            }
        });

        paraStyles.on('change:stylesheet', function (evt, styleId) {
            app.getView().enterBusy({
                warningLabel: gt('Please wait while updating document.')
            });
            var emptyAttr = {};
            var defaultParaStyle = paraStyles.getDefaultStyleId();
            var allP = self.getNode().find('.p');

            self.iterateArraySliced(allP, function (para) {
                para = $(para);
                var data = para.data();
                var attrs = Utils.getOption(data, 'attributes', emptyAttr);
                var paraStyleId = Utils.getStringOption(attrs, 'styleId', defaultParaStyle);
                if (paraStyles.isChildOfOther(paraStyleId, styleId)) {
                    paraStyles.updateElementFormatting(para);
                }
            }, { infoString: 'Text: Change:styleSheet' }).always(function () {
                app.getView().leaveBusy();
                self.insertPageBreaks();
            });
        });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            app = self = null;
        });

    } // class StylesheetMixin

    // exports ================================================================

    return StylesheetMixin;

});
