/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define('io.ox/office/text/format/drawingstyles', [
    'io.ox/office/textframework/utils/textutils',
    'io.ox/office/editframework/utils/attributeutils',
    'io.ox/office/editframework/utils/border',
    'io.ox/office/drawinglayer/model/drawingstylecollection',
    'io.ox/office/drawinglayer/view/drawingframe',
    'io.ox/office/text/components/drawing/drawingresize',
    'io.ox/office/textframework/utils/dom',
    'io.ox/office/textframework/utils/position'
], function (Utils, AttributeUtils, Border, DrawingStyleCollection, DrawingFrame, DrawingResize, DOM, Position) {

    'use strict';

    var // definitions for text-specific drawing attributes
        DEFINITIONS = {

            /**
             * Margin from top border of the drawing to text contents, in 1/100
             * of millimeters.
             */
            marginTop: { def: 0 },

            /**
             * Margin from bottom border of the drawing to text contents, in
             * 1/100 of millimeters.
             */
            marginBottom: { def: 0 },

            /**
             * Margin from left border of the drawing to text contents, in 1/100
             * of millimeters.
             */
            marginLeft: { def: 0 },

            /**
             * Margin from right border of the drawing to text contents, in
             * 1/100 of millimeters.
             */
            marginRight: { def: 0 },

            /**
             * Style, width and color of the left drawing border.
             */
            borderLeft: {
                def: Border.NONE,
                format: function (element, border) {
                    element.css('border-left', this.getCssBorder(border));
                }
            },

            /**
             * Style, width and color of the right drawing border.
             */
            borderRight: {
                def: Border.NONE,
                format: function (element, border) {
                    element.css('border-right', this.getCssBorder(border));
                }
            },

            /**
             * Style, width and color of the top drawing border.
             */
            borderTop: {
                def: Border.NONE,
                format: function (element, border) {
                    element.css('border-top', this.getCssBorder(border));
                }
            },

            /**
             * Style, width and color of the bottom drawing border.
             */
            borderBottom: {
                def: Border.NONE,
                format: function (element, border) {
                    element.css('border-bottom', this.getCssBorder(border));
                }
            },

            // object anchor --------------------------------------------------

            /**
             * Width of the drawing, as number in 1/100 of millimeters. The
             * value 0 will be interpreted as 100% width of the parent element.
             */
            width: {
                def: 0,
                format: function (element, width) {
                    if (!DrawingFrame.isGroupedDrawingFrame(element)) {
                        if (width === 0) {
                            element.width('100%');
                        } else {
                            element.width(Utils.convertHmmToLength(width, 'px', 1));
                        }
                    }
                }
            },

            /**
             * Height of the drawing, as number in 1/100 of millimeters.
             */
            height: {
                def: 0
            },

            /**
             * If set to true, the drawing is rendered as inline element ('as
             * character'), otherwise it is anchored relative to another
             * element (page, paragraph, table cell, ...).
             */
            inline: { def: true },

            anchorHorBase: { def: 'margin' },

            anchorHorAlign: { def: 'left' },

            anchorHorOffset: { def: 0 },

            anchorVertBase: { def: 'margin' },

            anchorVertAlign: { def: 'top' },

            anchorVertOffset: { def: 0 },

            anchorLayerOrder: { def: 0 },

            anchorBehindDoc: { def: false },

            /**
             * Specifies how text floats around the drawing.
             * - 'none': Text does not float around the drawing.
             * - 'square': Text floats around the bounding box of the drawing.
             * - 'tight': Text aligns to the left/right outline of the drawing.
             * - 'through': Text aligns to the entire outline of the drawing.
             * - 'topAndBottom': Text floats above and below the drawing only.
             */
            textWrapMode: { def: 'topAndBottom' },

            /**
             * Specifies on which side text floats around the drawing. Effective
             * only if the attribute 'textWrapMode' is either 'square',
             * 'tight', or 'through'.
             * - 'both': Text floats at the left and right side.
             * - 'left': Text floats at the left side of the drawing only.
             * - 'right': Text floats at the right side of the drawing only.
             * - 'largest': Text floats at the larger side of the drawing only.
             */
            textWrapSide: { def: 'both' }

        },

        // values for the 'textWrapMode' attribute allowing to wrap the text around the drawing
        WRAPPED_TEXT_VALUES = /^(square|tight|through)$/;

    // private global functions ===============================================

    /**
     * Returns whether the passed 'textWrapMode' attribute allows to wrap the
     * text around the drawing.
     */
    function isTextWrapped(textWrapMode) {
        return WRAPPED_TEXT_VALUES.test(textWrapMode);
    }

    // class TextDrawingStyles ================================================

    /**
     * Contains the style sheets for drawing formatting attributes. The CSS
     * formatting will be read from and written to drawing elements of any type.
     *
     * @constructor
     *
     * @extends DrawingStyleCollection
     *
     * @param {TextModel} docModel
     *  The text document model containing instance.
     */
    function TextDrawingStyles(docModel) {

        // base constructor ---------------------------------------------------

        DrawingStyleCollection.call(this, docModel, {
            families: 'line fill image shape geometry changes',
            formatHandler: updateDrawingFormatting
        });

        // private methods ----------------------------------------------------

        /**
         * Helper function that calculates the horizontal offset for absolute positioned drawings.
         * This function is not used for in line drawings.
         *
         * @param {Object} pageLayout
         *  The object handling the layout of the pages.
         *
         * @param {Object} drawingAttributes
         *  A map with drawing attributes, as name/value pairs.
         *
         * @param {Boolean} isDrawingInDrawingLayer
         *  Whether the drawing is a node inside the drawing layer. This is the case for all drawings
         *  that are anchored to the page or the paragraph.
         *
         * @param {Number} pageWidth
         *  The width of the page in 1/100 mm.
         *
         * @param {Number} paraWidth
         *  The width of the paragraph in 1/100 mm.
         *
         * @param {Number} drawingWidth
         *  The width of the drawing in 1/100 mm.
         *
         * @param {jQuery} paragraph
         *  The paragraph node that is the parent of the drawing, as jQuery object.
         *
         * @param {jQuery} pageNode
         *  The page node node as jQuery object.
         *
         * @param {Number} zoomFactor
         *  The current zoom factor in percent.
         *
         * @returns {Number}
         *  The horizontal offset in 1/100mm relative to the page or margin or paragraph
         */
        function calculateHorizontalOffset(pageLayout, drawingAttributes, isDrawingInDrawingLayer, pageWidth, paraWidth, drawingWidth, paragraph, pageNode, zoomFactor) {

            var // the calculated left offset of the drawing in 1/100 mm
                leftOffset = 0,
                // the width of the left margin in 1/100 mm
                leftMarginWidth = pageLayout.getPageAttribute('marginLeft'),
                // the width of the right margin in 1/100 mm
                rightMarginWidth = 0;

            // drawing horizontally aligned to column or margin
            if ((drawingAttributes.anchorHorBase === 'column') || (drawingAttributes.anchorHorBase === 'margin')) {

                if (isDrawingInDrawingLayer) {  // drawing is located in the drawing layer

                    switch (drawingAttributes.anchorHorAlign) {
                        case 'center':
                            rightMarginWidth = pageLayout.getPageAttribute('marginRight');
                            leftOffset = leftMarginWidth + (pageWidth - leftMarginWidth - rightMarginWidth - drawingWidth) / 2;
                            break;
                        case 'right':
                            rightMarginWidth = pageLayout.getPageAttribute('marginRight');
                            leftOffset = pageWidth - rightMarginWidth - drawingWidth;
                            break;
                        case 'left':
                            leftOffset = leftMarginWidth;
                            break;
                        case 'offset':
                            leftOffset = drawingAttributes.anchorHorOffset;
                            // increasing the left offset by the distance between page and paragraph
                            // Info: The paragraph can be located anywhere, for example in a text frame positioned at the page (35146)
                            // TODO: Check for: leftMarginWidth = Math.round(Utils.convertLengthToHmm(Position.getPixelPositionToRootNodeOffset(pageNode, paragraph, zoomFactor * 100).x, 'px') / zoomFactor);
                            // if the paragraph has an additional left margin set as css attribute, this need to be handled, too (might come from a style ID (35146)
                            if (paragraph) { leftMarginWidth -= Utils.convertCssLengthToHmm(paragraph.css('margin-left')); }
                            leftOffset += leftMarginWidth;
                            if (leftOffset + drawingWidth > pageWidth) { leftOffset = pageWidth - drawingWidth; }
                            break;
                        default:
                            leftOffset = 0;
                    }

                } else {
                    switch (drawingAttributes.anchorHorAlign) {
                        case 'center':
                            leftOffset = (paraWidth - drawingWidth) / 2;
                            break;
                        case 'right':
                            leftOffset = paraWidth - drawingWidth;
                            break;
                        case 'offset':
                            leftOffset = drawingAttributes.anchorHorOffset;
                            if (drawingAttributes.anchorHorOffset + drawingWidth > paraWidth) {
                                leftOffset = paraWidth - drawingWidth;
                            }
                            break;
                        default:
                            leftOffset = 0;
                    }
                }

            // drawing horizontally aligned to page
            } else if (drawingAttributes.anchorHorBase === 'page') {

                if (isDrawingInDrawingLayer) {  // drawing is located in the drawing layer (and not in header or footer)

                    switch (drawingAttributes.anchorHorAlign) {
                        case 'center':
                            leftOffset = (pageWidth - drawingWidth) / 2;
                            break;
                        case 'right':
                            leftOffset = pageWidth - drawingWidth;
                            break;
                        case 'offset':
                            leftOffset = drawingAttributes.anchorHorOffset;
                            break;
                        default:
                            leftOffset = 0;
                    }

                } else {  // the drawing will be positioned inside its paragraph

                    switch (drawingAttributes.anchorHorAlign) {
                        case 'center':
                            leftOffset = (paraWidth - drawingWidth) / 2;
                            break;
                        case 'right':
                            leftOffset = pageWidth - leftMarginWidth - drawingWidth;
                            break;
                        case 'left':
                            leftOffset = -leftMarginWidth;
                            break;
                        case 'offset':
                            // reducing the left offset by the distance between page and paragraph
                            leftOffset = drawingAttributes.anchorHorOffset;
                            if (drawingAttributes.anchorHorOffset + drawingWidth > pageWidth) { leftOffset = pageWidth - drawingWidth; }
                            // decreasing the left offset by the distance between page and paragraph (TODO: zoomFactor?)
                            leftMarginWidth = Utils.round((Utils.convertCssLengthToHmm(pageNode.css('padding-left')) * zoomFactor), 1);
                            leftOffset -= leftMarginWidth;  // reducing offset by the left page margin
                            break;
                        default:
                            leftOffset = 0;
                    }

                }

            // drawing horizontally aligned to character
            } else if (drawingAttributes.anchorHorBase === 'character') {
                // Also 'character' looks better, if it is handled like 'column' (33276). The value for 'anchorHorOffset'
                // is not relative to the character, but to the paragraph.
                switch (drawingAttributes.anchorHorAlign) {
                    case 'center':
                        leftOffset = (paraWidth - drawingWidth) / 2;
                        break;
                    case 'right':
                        leftOffset = paraWidth - drawingWidth;
                        break;
                    case 'offset':
                        leftOffset = drawingAttributes.anchorHorOffset;
                        break;
                    default:
                        leftOffset = 0;
                }
            } else {
                // Other yet unsupported anchor bases
                leftOffset = 0;
            }

            return Utils.roundUp(leftOffset, 1);
        }

        /**
         * Calculating the vertical offset for a drawing that is anchored to the paragraph.
         *
         * @param {Object} drawingAttributes
         *  A map with drawing attributes, as name/value pairs.
         *
         * @param {jQuery} paragraph
         *  The paragraph node that is the parent of the drawing, as jQuery object.
         *
         * @param {jQuery} drawing
         *  The drawing node as jQuery object.
         *
         * @param {Number} zoomFactor
         *  The current zoom factor in percent.
         *
         * @returns {Number}
         *  The vertical offset in 1/100mm relative to the paragraph.
         */
        function calculateParagraphTopOffset(drawingAttributes, paragraph, drawing, zoomFactor) {

            var // the calculated top offset of the drawing (in 1/100 mm relative to the paragraph)
                topOffset = 0,
                // the neighboring text span behind (preferred) or before the drawing
                textSpan = Utils.findNextSiblingNode(drawing, 'span'),
                // a position object with keys x and y in pixel
                pos = null;

            if (drawingAttributes.anchorVertAlign === 'offset') {

                if (drawingAttributes.anchorVertBase === 'line') {
                    topOffset = drawingAttributes.anchorVertOffset < 0 ? 0 : drawingAttributes.anchorVertOffset; // no negative values for 'line' (see Exercise03 document)
                } else {
                    topOffset = drawingAttributes.anchorVertOffset; // allowing negative values
                }

            } else if (drawingAttributes.anchorVertBase === 'line') {

                // automatic alignment (top/bottom/center) is not possible for the vertical anchor
                // base 'paragraph', but for 'line'. In this case the drawing is aligned to the upper
                // border of the text line.

                textSpan = Utils.findNextSiblingNode(drawing, 'span');

                if (textSpan) {
                    // receiving the pixel position of the upper left corner of the text span relative to the paragraph
                    pos = Position.getPixelPositionToRootNodeOffset(paragraph, textSpan, zoomFactor * 100);
                } else {
                    // trying to get the previous text span, but that might be a multi line span. Therefore the lower right
                    // corner need to be used and then the font-size can be added
                    textSpan = Utils.findPreviousSiblingNode(drawing, 'span');

                    if (textSpan) {

                        pos = Position.getPixelPositionToRootNodeOffset(paragraph, textSpan, zoomFactor * 100);

                        if (pos && pos.y && DOM.isMultiLineTextSpan(textSpan)) {
                            pos.y += ($(textSpan).height() - Utils.convertCssLength($(textSpan).css('font-size'), 'px', 1));
                        }
                    }
                }

                // ... and finally the offset can be calculated
                // -> TODO: The space maker node shifts the text span downwards, so that the pos.y is no longer precise.
                if (pos && pos.y) {
                    if (drawingAttributes.anchorVertAlign === 'top') {
                        topOffset = Utils.convertLengthToHmm(pos.y, 'px', 1);
                    } else if (drawingAttributes.anchorVertAlign === 'bottom') {
                        topOffset = Utils.convertLengthToHmm(pos.y - drawing.height(), 'px', 1);
                    } else {
                        topOffset = Utils.convertLengthToHmm(pos.y - 0.5 * drawing.height(), 'px');
                    }
                }

                // no support of drawings above the paragraph
                if (topOffset < 0) { topOffset = 0; }
            }

            return Utils.roundUp(topOffset, 1);
        }

        /**
         * Helper function to save the drawing attributes for the margin directly at the
         * drawing node. This is useful for performance reasons. It needs to be fast to
         * position drawings absolutely and to find intersecting space maker nodes.
         *
         * Info: As long as it is not possible to modify the drawing margin in OX Text,
         * it is sufficient to set the margin at the node once.
         *
         * @param {jQuery} drawing
         *  The drawing node as jQuery object.
         *
         * @param {Object} drawingAttributes
         *  A map with drawing attributes, as name/value pairs.
         */
        function saveMarginsAtDrawing(drawing, drawingAttributes) {

            if (!drawing.data(DOM.DRAWING_MARGINS_HMM)) {
                drawing.data(DOM.DRAWING_MARGINS_HMM, {
                    marginTop: drawingAttributes.marginTop,
                    marginBottom: drawingAttributes.marginBottom,
                    marginLeft: drawingAttributes.marginLeft,
                    marginRight: drawingAttributes.marginRight
                });
            }

            if (!drawing.data(DOM.DRAWING_MARGINS_PIXEL)) {
                drawing.data(DOM.DRAWING_MARGINS_PIXEL, {
                    marginTop: Utils.convertHmmToLength(drawingAttributes.marginTop, 'px', 1),
                    marginBottom: Utils.convertHmmToLength(drawingAttributes.marginBottom, 'px', 1),
                    marginLeft: Utils.convertHmmToLength(drawingAttributes.marginLeft, 'px', 1),
                    marginRight: Utils.convertHmmToLength(drawingAttributes.marginRight, 'px', 1)
                });
            }
        }

        /**
         * Calculating the vertical offset for a drawing that is anchored to the page or margin.
         *
         * @param {jQuery} drawing
         *  The drawing node as jQuery object.
         *
         * @param {Object} drawingAttributes
         *  A map with drawing attributes, as name/value pairs.
         *
         * @param {Object} pageLayout
         *  The object handling the layout of the pages.
         *
         * @param {jQuery} pageNode
         *  The root node containing the document model and representation, as jQuery object.
         *
         * @param {Boolean} inFooter
         *  Whether the drawing is located inside the footer node.
         *
         * @returns {Object}
         *  An object with two properties:
         *    topOffset: The vertical offset in 1/100 mm relative to the page or margin.
         *    isBottomOffset: Whether the value of the property 'topOffset' is related to
         *                    the bottom of the page
         *                    -> this is only used in footers to increase the precision,
         *                       because page height is not required if a drawing in the
         *                       footer is aligned to the page.
         */
        function calculatePageTopOffset(drawing, drawingAttributes, pageLayout, pageNode, inFooter) {

            var // the calculated top offset of the drawing (in pixel relative to the page)
                topOffset = 0,
                // whether the calculated top offset is related to the bottom (of the footer)
                isBottomOffset = false;

            if (drawingAttributes.anchorVertAlign === 'offset') {

                // offset to 'margin' or 'bottomMargin' & 'page' can be negative
                topOffset = drawingAttributes.anchorVertOffset;

                // -> special handling for drawings in footers aligned to the page
                if (inFooter && drawingAttributes.anchorVertBase === 'page') {
                    // this is a drawing in the footer
                    // -> the offset from the bottom of the page is the height of the page minus
                    //    the specified vertical offset minus the height of the drawing.
                    topOffset = pageLayout.getPageAttribute('height') - topOffset - Utils.convertLengthToHmm(drawing.height(), 'px');
                    // the calculated offset relates to the bottom of the page (this works only in footer nodes)
                    isBottomOffset = true;
                    // no negative top offsets in footer
                    if (topOffset < 0) { topOffset = 0; }
                }

                // Info: For the type 'margin' the margin height is not handled here, because it is dynamic.
                // Word uses the height of the header, that is a dynamic value. Therefore only the value
                // of 'anchorVertOffset' is saved at the node. The precise calculation happens inside
                // drawingLayer.calculateVerticalPageOffset().

            } else {
                // automatic alignment (top/bottom/center/...)
                if (drawingAttributes.anchorVertBase === 'page') {
                    if (drawingAttributes.anchorVertAlign === 'top') {
                        topOffset = 0;
                    } else if (drawingAttributes.anchorVertAlign === 'bottom') {
                        topOffset = pageLayout.getPageAttribute('height') - Utils.convertLengthToHmm(drawing.height(), 'px');
                        if (inFooter) {
                            topOffset = 0;
                            isBottomOffset = true;
                        }
                    } else if (drawingAttributes.anchorVertAlign === 'center') {
                        topOffset = (pageLayout.getPageAttribute('height') - Utils.convertLengthToHmm(drawing.height(), 'px')) / 2;
                    } else {
                        topOffset = 0;
                    }
                }

                // alignment at 'margin' (top/center/bottom) will be handled dynamically in 'drawingLayer.setAbsoluteDrawingPosition'
                // -> this is necessary, because height of header or footer changes.
            }

            return { topOffset: Utils.roundUp(topOffset, 1), isBottomOffset: isBottomOffset };
        }

        /**
         * Setting the maximum width for a drawing. This needs to be adapted, if the drawing gets
         * a new anchor. If it was anchored to page before and is now inline or anchored to paragraph,
         * the size needs to be reduced.
         *
         * @param {jQuery} drawing
         *  The drawing node as jQuery object.
         *
         * @param {Object} mergedAttributes
         *  A map with the merged drawing attributes, as name/value pairs.
         *
         * @param {Number} drawingWidth
         *  The width of the drawing in 1/100 mm.
         *
         * @param {Number} borderWidth
         *  The width of the drawing of the border in 1/100 mm.
         *
         * @param {Number} paraWidth
         *  The width of the paragraph in 1/100 mm.
         *
         * @param {Number} pageWidth
         *  The width of the page in 1/100 mm.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.useParagraphWidth=true]
         *      If set to true, the width of the paragraph is used as maximal width. Otherwise the
         *      width of the page is used.
         *
         * @returns {Number}
         *  The new width of the drawing in 1/100mm
         */
        function setMaximumDrawingWidth(drawing, mergedAttributes, drawingWidth, borderWidth, paraWidth, pageWidth, options) {

            var // whether the paragraph or the page width is used
                useParagraphWidth = Utils.getBooleanOption(options, 'useParagraphWidth', true),
                // the maximum allowed width
                maxWidth = useParagraphWidth ? paraWidth : pageWidth,
                // the image attributes of the passed attribute map
                imageAttributes = mergedAttributes.image,
                // the new drawing width, handling width of paragraph or page
                newDrawingWidth = drawingWidth;

            // reducing width of drawing to width of paragraph or page, if no left or right cropping is enabled (Task 30982)
            if ((drawingWidth > maxWidth) && (imageAttributes.cropLeft === 0) && (imageAttributes.cropRight === 0)) {
                newDrawingWidth = maxWidth;
                drawing.width(Utils.convertHmmToCssLength(newDrawingWidth - borderWidth, 'px', 1));
            }

            return newDrawingWidth;
        }

        /**
         * Setting the height of the drawing. This needs to be done, because it might be
         * dependent from other attribute values.
         *
         * @param {jQuery} drawing
         *  The drawing node as jQuery object.
         *
         * @param {Object} attributes
         *  A map with the merged drawing attributes, as name/value pairs.
         */
        function setDrawingHeight(drawing, attributes) {

            var // whether the height of the drawing shall be set
                // -> no setting of height for grouped drawings and for shapes with property autoResizeHeight
                setHeight = !(DrawingFrame.isGroupedDrawingFrame(drawing) || attributes.shape.autoResizeHeight);

            if (setHeight) {
                drawing.height(Utils.convertHmmToLength(attributes.drawing.height, 'px', 1));
            } else if (attributes.shape.autoResizeHeight) {
                drawing.height(''); // if 'autoResizeHeight' is set, an explicit height must be removed
            }
        }

        /**
         * Helper function to update attributes of children of drawings. This is only used for text frames yet.
         *
         * @param {jQuery|Node} drawing
         *  The drawing node as jQuery object, whose children need an update of element formatting.
         */
        function handleChildAttributes(drawing) {

            var // the explicit drawing attributes
                explicitDrawingAttrs = AttributeUtils.getExplicitAttributes(drawing),
                // the paragraph style sheet container
                paragraphStyles = null,
                // the container for the paragraphs and tables inside the drawing
                textFrameNode = null;

            // checking the fill color (task 36385)
            if (explicitDrawingAttrs && explicitDrawingAttrs.fill) {

                paragraphStyles = docModel.getStyleCollection('paragraph');
                textFrameNode = DrawingFrame.getTextFrameNode(drawing);

                Utils.iterateSelectedDescendantNodes(textFrameNode, DOM.PARAGRAPH_NODE_SELECTOR, function (paragraph) {
                    paragraphStyles.updateElementFormatting(paragraph);
                }, undefined, { children: true });
            }
        }

        /**
         * Handling drawing that are in-line with text. This is especially important, if the drawing
         * was until yet no in-line drawing. In this case some clean up at the node is required.
         *
         * @param {jQuery} drawing
         *  The drawing node as jQuery object.
         *
         * @param {Object} mergedAttributes
         *  A map with the merged drawing attributes, as name/value pairs.
         *
         * @param {Number} drawingWidth
         *  The width of the drawing in 1/100 mm.
         *
         * @param {Number} borderWidth
         *  The width of the drawing of the border in 1/100 mm.
         *
         * @param {Number} paraWidth
         *  The width of the paragraph in 1/100 mm.
         *
         * @param {Number} pageWidth
         *  The width of the page in 1/100 mm.
         *
         * @param {Number} halfBorderWidth
         *  The half width of the drawings border in 1/100 mm.
         */
        function handleInlineDrawing(drawing, mergedAttributes, drawingWidth, borderWidth, paraWidth, pageWidth, halfBorderWidth) {

            var // the drawing layer for absolutely positioned drawings
                drawingLayer = docModel.getDrawingLayer();

            // switch from floating to inline mode
            if (!drawing.hasClass('inline')) {

                // reducing drawing width if required
                setMaximumDrawingWidth(drawing, mergedAttributes, drawingWidth, borderWidth, paraWidth, pageWidth, { useParagraphWidth: true });

                // removing an optional space maker node (for header/footer this must happen, before 'absolute' is removed)
                // -> this is at this position only required for drawings anchored to the paragraph.
                drawingLayer.removeSpaceMakerNode(drawing);

                // removing the drawing from the model containers ('paragraphDrawingCollectorUpdateAlways' and 'paragraphDrawingCollector', 43851)
                drawingLayer.removeDrawingFromModelContainer(drawing);

                // Word uses fixed predefined margins in in line mode, we too?
                drawing.removeClass('float left right absolute').addClass('inline');
                // ignore other attributes in in line mode

                // whether the drawing needs to be removed from the drawing layer
                if (DOM.isDrawingLayerNode(drawing.parent())) {
                    drawingLayer.setAbsoluteDrawingPosition(drawing, { doClean: true });
                    drawingLayer.shiftDrawingAwayFromDrawingLayer(drawing);

                    // the selection needs to be repainted. Otherwise it is possible
                    // to move the drawing to the page borders.
                    DrawingFrame.clearSelection(drawing);
                    // BUG 36372: Text frame gets blue resize/selection-border wrongly (via undo)
                    //  previously check if the drawing was selected before pressing undo
                    if (docModel.getSelection().isDrawingSelected(drawing)) {
                        DrawingResize.drawDrawingSelection(docModel.getApp(), drawing);
                    }
                } else {
                    // Handling of absolute drawings anchored at paragraph
                    drawingLayer.setAbsoluteDrawingParagraphPosition(drawing, { doClean: true });
                }
            }

            // setting margin to in-line drawings. This is necessary, because a thick border drawing with
            // canvas might overlap the surrounding text.
            drawing.css({ margin: Utils.convertHmmToCssLength(halfBorderWidth, 'px', 1) });
        }

        /**
         * Handling drawing that are absolute positioned. This are all drawings that are not in-line
         * with text. This is especially important, if the drawing anchor was modified. In this case
         * some clean up at the node is required.
         *
         * @param {jQuery} drawing
         *  The drawing node as jQuery object.
         *
         * @param {Object} mergedAttributes
         *  A map with the merged drawing attributes, as name/value pairs.
         *
         * @param {Number} drawingWidth
         *  The width of the drawing in 1/100 mm.
         *
         * @param {Number} borderWidth
         *  The width of the drawing of the border in 1/100 mm.
         *
         * @param {Number} paraWidth
         *  The width of the paragraph in 1/100 mm.
         *
         * @param {Number} pageWidth
         *  The width of the page in 1/100 mm.
         *
         * @param {Number} halfBorderWidth
         *  The half width of the drawings border in 1/100 mm.
         *
         * @param {Boolean} inHeaderFooter
         *  Whether the drawing is located inside a header or footer node.
         *
         * @param {Number} zoomFactor
         *  The current zoom factor in percent.
         */
        function handleAbsoluteDrawing(drawing, mergedAttributes, drawingWidth, borderWidth, paraWidth, pageWidth, halfBorderWidth, inHeaderFooter, zoomFactor) {

            var // the page layout object
                pageLayout = docModel.getPageLayout(),
                // the page node
                pageNode = docModel.getNode(),
                // the drawing layer for absolutely positioned drawings
                drawingLayer = docModel.getDrawingLayer(),
                // the paragraph element containing the drawing node
                // -> this can also be a drawing group without paragraphs
                paragraph = drawing.parent(),
                // the drawing attributes of the passed attribute map
                drawingAttributes = mergedAttributes.drawing,
                // offset from top/left/right margin of paragraph element, in 1/100 mm
                topOffset = 0, leftOffset = 0,
                // whether the drawing is positioned absolutely to the page and is shifted into the drawing layer
                isDrawingInDrawingLayer = false,
                // the horizontal and vertical offsets in pixel relative to the page
                topValue = 0, leftValue = 0,
                // the header or footer root node for the drawing
                targetRootNode = null,
                // whether the drawing is vertically aligned to the margin
                isVerticalMarginAligned = false,
                // whether the drawing is vertically aligned to the bottom margin
                useBottomMargin = false,
                // a helper object containing information about the vertical offset
                topOffsetObject = null,
                // whether the drawing is inside the footer node
                isFooterNode = false;

            // calculate top offset (only if drawing is anchored to paragraph or to line).
            // Anchor to 'line' means the base line of the character position. The drawings with these anchor
            // types are absolutely positioned inside a paragraph.
            if ((drawingAttributes.anchorVertBase === 'paragraph') || (drawingAttributes.anchorVertBase === 'line')) {

                drawing.removeClass('inline float left right').addClass('absolute');  // absolute positioned!

                // whether the drawing needs to be removed from the drawing layer
                // -> for the following calculations, the drawing must be located inside the paragraph
                if (DOM.isDrawingLayerNode(paragraph)) {
                    drawingLayer.setAbsoluteDrawingPosition(drawing, { doClean: true });
                    drawingLayer.shiftDrawingAwayFromDrawingLayer(drawing);
                }

                // calculating the top offset of the drawing in relation to the page or margin
                topOffset = calculateParagraphTopOffset(drawingAttributes, paragraph, drawing, zoomFactor);

            } else if (drawingAttributes.anchorVertBase === 'page' || drawingAttributes.anchorVertBase === 'margin' || drawingAttributes.anchorVertBase === 'bottomMargin') {

                // setting position 'absolute' to all drawings that have a vertical offset corresponding to the page
                isDrawingInDrawingLayer = true;

                // setting the header or footer target node for the drawing
                if (inHeaderFooter) { targetRootNode = DOM.getMarginalTargetNode(pageNode, drawing); }

                // if inHeaderFooter is true and targetRootNode is found, it might be better to shift only
                // those drawings into the drawing layer, that are children of the 'div.header-footer-placeholder'.
                // All other drawing layers will be removed soon and replaced by these header and footer
                // templates, that are located inside 'div.header-footer-placeholder'.

                // whether the drawing needs to be moved to the drawing layer
                if (DOM.isParagraphNode(drawing.parent())) { drawingLayer.shiftDrawingIntoDrawingLayer(drawing, targetRootNode); }

                drawing.removeClass('inline float left right').addClass('absolute');  // absolute positioned!

                // marking this drawing as vertically aligned to the margin
                if (drawingAttributes.anchorVertBase === 'margin' || drawingAttributes.anchorVertBase === 'bottomMargin') { isVerticalMarginAligned = true; }
                if (drawingAttributes.anchorVertBase === 'bottomMargin') { useBottomMargin = true; }

                if (targetRootNode) { isFooterNode = DOM.isFooterNode(targetRootNode); }

                // calculating the top offset of the drawing in relation to the page or margin
                topOffsetObject = calculatePageTopOffset(drawing, drawingAttributes, pageLayout, pageNode, isFooterNode);
                topOffset = topOffsetObject.topOffset;
            }

            // reducing drawing width if required (without margin)
            drawingWidth = setMaximumDrawingWidth(drawing, mergedAttributes, drawingWidth, borderWidth, paraWidth, pageWidth, { useParagraphWidth: false });

            // calculate left/right offset (required before setting wrap mode)
            leftOffset = calculateHorizontalOffset(pageLayout, drawingAttributes, isDrawingInDrawingLayer, pageWidth,  paraWidth, drawingWidth, paragraph, pageNode, zoomFactor);

            leftValue = Utils.convertHmmToLength(leftOffset, 'px', 1);
            topValue = Utils.convertHmmToLength(topOffset, 'px', 1);

            // setting calculated values at the drawing element
            if (isDrawingInDrawingLayer) {
                // the topValue is the distance in pixel to the top page border
                drawingLayer.setAbsoluteDrawingPosition(drawing, { leftValue: leftValue, topValue: topValue, newValues: true, isBottomOffset: topOffsetObject.isBottomOffset, targetNode: targetRootNode, isMarginalNode: inHeaderFooter, isFooterNode: isFooterNode, verticalMargin: (isVerticalMarginAligned ? drawingAttributes.anchorVertAlign : ''), useBottomMargin: useBottomMargin });

                updateCropDrawing(drawing, drawingAttributes);
            } else {
                drawingLayer.setAbsoluteDrawingParagraphPosition(drawing, { leftValue: leftValue, topValue: topValue, textWrapMode: drawingAttributes.textWrapMode, textWrapSide: drawingAttributes.textWrapSide, newValues: true, targetNode: targetRootNode });
                // cache for drawings (and paragraphs) to update only selected elements
                drawingLayer.registerUpdateElement(drawing);

                resetCropDrawing(drawing);
            }

            // removing a margin, no margin at absolutely positioned drawings.
            // A margin must be handled by space maker node. It might have been set at inline drawing before.
            drawing.css('margin', '');

            // saving margins directly at the drawing (for performance reasons)
            saveMarginsAtDrawing(drawing, drawingAttributes);

            handleLayerOrder(drawing, inHeaderFooter, drawingAttributes);
        }

        /**
         * Will be called for every drawing node whose attributes have been changed. Repositions
         * and reformats the drawing according to the passed attributes.
         *
         * @param {jQuery} drawing
         *  The drawing node whose attributes have been changed, as jQuery object.
         *
         * @param {Object} mergedAttributes
         *  A map of attribute maps (name/value pairs), keyed by attribute family, containing the
         *  effective attribute values merged from style sheets and explicit attributes.
         */
        function updateDrawingFormatting(drawing, mergedAttributes) {

            var // the drawing attributes of the passed attribute map
                drawingAttributes = mergedAttributes.drawing,
                // the paragraph element containing the drawing node
                // -> this can also be a drawing group without paragraphs
                paragraph = drawing.parent(),
                // total width of the paragraph, in 1/100 mm
                paraWidth = Utils.convertLengthToHmm(paragraph.width(), 'px'),
                // the page node
                pageNode = docModel.getNode(),
                // the current zoom factor in percent
                zoomFactor = docModel.getApp().getView().getZoomFactor() / 100,
                // the width of the page in 1/100 mm
                pageWidth = Math.round(Utils.convertLengthToHmm(pageNode.outerWidth(true), 'px') / zoomFactor),
                // whether the drawing is located in header or footer
                inHeaderFooter = (drawing.hasClass(DOM.TARGET_NODE_CLASSNAME) || DOM.getMarginalTargetNode(pageNode, drawing).length > 0),
                // the width of the border of the drawing
                borderWidth = mergedAttributes.line.width,
                // the half width of the border of the drawing
                // half border width goes only for text frames, full width for images; cannot be less than 1px (or 26.4 hmm), to be compatible with canvas border, #36576
                halfBorderWidth = DrawingFrame.isTextFrameShapeDrawingFrame(drawing) ? ((0.5 * borderWidth) > 26.5 ? (0.5 * borderWidth) : 26.5) : borderWidth,
                // current drawing width, in 1/100 mm, without margin
                // -> since the border is drawn with a canvas, the drawing width (including border) needs to be adjusted
                // -> the canvas draws half of the border width outside and half of it inside the drawing.
                drawingWidth = Utils.convertLengthToHmm(drawing.width(), 'px') + borderWidth;

            // Setting the height of the drawing
            setDrawingHeight(drawing, mergedAttributes);

            // nothing to do, if paragraph is a groupcontent object
            // -> in this case the group is responsible for the correct attributes
            if (DrawingFrame.isGroupContentNode(paragraph)) {
                drawing.removeClass('inline float left right').addClass('absolute grouped');  // absolute positioned drawing inside its group!
                // set generic formatting to the drawing frame
                DrawingFrame.updateFormatting(docModel.getApp(), drawing, mergedAttributes);
                // simply do nothing, let the group make further attribute settings
                return;
            }

            // setting marker for drawings in header or footer
            if (inHeaderFooter) { drawing.addClass(DOM.TARGET_NODE_CLASSNAME); }

            // setting position to 'inline'
            if (drawingAttributes.inline) {
                handleInlineDrawing(drawing, mergedAttributes, drawingWidth, borderWidth, paraWidth, pageWidth, halfBorderWidth);
            } else {
                handleAbsoluteDrawing(drawing, mergedAttributes, drawingWidth, borderWidth, paraWidth, pageWidth, halfBorderWidth, inHeaderFooter, zoomFactor);
            }

            // change track attribute handling
            docModel.getChangeTrack().updateChangeTrackAttributes(drawing, mergedAttributes);

            // set generic formatting to the drawing frame
            DrawingFrame.updateFormatting(docModel.getApp(), drawing, mergedAttributes);

            // update formatting of all paragraphs inside the text frame content node (36385)
            if (DrawingFrame.isTextFrameShapeDrawingFrame(drawing)) { handleChildAttributes(drawing); }

            // set visibility of anchor
            if (docModel.getSelection().isDrawingSelected(drawing)) {
                docModel.getApp().getView().setVisibleDrawingAnchor(drawing, mergedAttributes);
            }
        }

        /**
         * drawings outside of the page-bounds are hidden with css overflow
         * drawings outside of the page-break are hidden here
         *
         * with Utils.getIntersectionRectangle, a drawing rect and page rect
         * cropping on top and on bottom of a drawing will be calculated
         * and set via css "clip"
         */
        function updateCropDrawing(drawing, drawingAttributes) {

            if (Utils.SMALL_DEVICE) { return; }

            var pageLayout = docModel.getPageLayout();
            var pageHeight = pageLayout.getPageAttribute('height');

            var pageRect = { left: 0, top: 0, width: 1e+25, height: pageHeight };
            var drawRect = { left: drawingAttributes.anchorHorOffset, top: drawingAttributes.anchorVertOffset, width: drawingAttributes.width, height: drawingAttributes.height };

            if (drawingAttributes.anchorVertBase === 'margin') {
                drawRect.top += pageLayout.getPageAttribute('marginTop');
            }

            var interRect = Utils.getIntersectionRectangle(pageRect, drawRect);

            if (interRect && (drawRect.top !== interRect.top || drawRect.height !== interRect.height)) {
                var cropTop = Math.round(Utils.convertHmmToLength(interRect.top - drawRect.top, 'px'));

                if (cropTop === 0) {
                    var cropHeight = Math.round(Utils.convertHmmToLength(interRect.height, 'px'));
                    drawing.css('clip', 'rect(-20000px 20000px ' + (cropHeight + cropTop) + 'px -20000px)');
                } else {
                    drawing.css('clip', 'rect(' + cropTop + 'px 20000px 20000px -20000px)');
                }
            } else {
                resetCropDrawing(drawing);
            }
        }

        /**
         * resets css "clip"
         */
        function resetCropDrawing(drawing) {
            if (Utils.SMALL_DEVICE) { return; }

            drawing.css('clip', '');
        }

        /**
         * Will be called for every absolute position drawing node whose attributes have been changed
         * and detects If "anchorLayerOrder" or "anchorBehindDoc" have changed.
         * If it is changed, z-index for all drawings on current page will be updated.
         *
         * @param {jQuery} drawing
         *  The drawing node whose attributes have been changed, as jQuery object.
         *
         * @param {Boolean} inHeaderFooter
         *  Whether the drawing is located inside a header or footer node.
         *
         * @param {Object} drawingAttributes
         *  A map of attribute maps (name/value pairs), keyed by attribute family, containing the
         *  effective attribute values merged from style sheets and explicit attributes.
         */
        var handleLayerOrder = Utils.profileMethod('DrawingStyles.handleLayerOrder(): ', function (drawing, inHeaderFooter, drawingAttributes) {

            var drawingEl = drawing[0];

            var order = drawingAttributes.anchorLayerOrder;
            if (drawingAttributes.anchorBehindDoc) { order -= 0xFFFFFFFFF; }

            var oldOrder = parseInt(drawingEl.getAttribute('layer-order'), 10);
            if (oldOrder !== order) {
                drawingEl.setAttribute('layer-order', order);

                //we return here, because after importFinish sorting is called in DrawingLayer on event 'update:absoluteElements'
                if (!docModel.getDrawingLayer().isImportFinished()) { return; }

                var startIndex = 0;
                var allDrawings = null;
                if (inHeaderFooter) {
                    startIndex = 11;
                    allDrawings = Utils.getAllAbsoluteDrawingsOnNode(DOM.getMarginalTargetNode(docModel.getNode(), drawing));
                } else {
                    startIndex = 41;
                    //begin at 41 because page-break has zIndex 40
                    var pageNumber = Utils.getPageNumber(drawingEl, docModel);
                    allDrawings = Utils.getAllAbsoluteDrawingsOnPage(docModel.getNode(), pageNumber);
                }
                Utils.sortDrawingsOrder(allDrawings, startIndex);
            }
        });

        // initialization -----------------------------------------------------

        // register the attribute definitions for the style family
        docModel.registerAttributeDefinitions('drawing', DEFINITIONS);

        // destroy all class members on destruction
        this.registerDestructor(function () {
            docModel = null;
        });

    } // class TextDrawingStyles

    // static methods ---------------------------------------------------------

    /**
     * Returns the drawing position identifier representing the passed drawing
     * attributes.
     *
     * @param {Object} attributes
     *  A map with drawing attributes, as name/value pairs.
     *
     * @returns {String|Null}
     *  The GUI drawing position identifier; or null, if any of the passed
     *  attributes is ambiguous.
     */
    TextDrawingStyles.getPositionFromAttributes = function (attributes) {

        // inline mode overrules floating attributes
        if (_.isNull(attributes.inline)) { return null; }
        if (attributes.inline === true) { return 'inline'; }

        // filter unsupported position anchor modes
        if (!/^(page|margin|column)$/.test(attributes.anchorHorBase) || !/^(page|margin|paragraph|line)$/.test(attributes.anchorVertBase)) { return null; }

        // filter ambiguous or unsupported horizontal alignment types
        if (!/^(left|center|right)$/.test(attributes.anchorHorAlign)) { return null; }

        // filter ambiguous text wrapping modes
        if (_.isNull(attributes.textWrapMode) || _.isNull(attributes.textWrapSide)) { return null; }

        // build the resulting drawing position
        return attributes.anchorHorAlign + ':' + (isTextWrapped(attributes.textWrapMode) ? attributes.textWrapSide : 'none');
    };

    /**
     * Returns the drawing attributes that are needed to represent the passed
     * GUI drawing position.
     *
     * @param {String} position
     *  The GUI drawing position.
     *
     * @param {Object} [origAttributes]
     *  The original attributes of the target drawing object.
     *
     * @param {Object}
     *  A map with drawing attributes, as name/value pairs.
     */
    TextDrawingStyles.getAttributesFromPosition = function (position, origAttributes) {

        var // extract alignment and text wrap mode
            matches = position.match(/^([a-z]+):([a-z]+)$/),
            // resulting drawing attributes
            drawingAttributes = {},
            // the horizontal anchor for the operation
            newAnchorHorBase = null;

        // inline flag overrules all other attributes, no need to return any other (unfortunately it is still necessary for undo reasons)
        if (position === 'inline') {
            return { inline: true,
                     anchorHorBase: null,
                     anchorHorAlign: null,
                     anchorHorOffset: null,
                     anchorVertBase: null,
                     anchorVertAlign: null,
                     anchorVertOffset: null,
                     textWrapMode: null,
                     textWrapSide: null };
        }

        // check that passed position contains alignment and text wrapping mode
        if (!_.isArray(matches) || (matches.length < 3)) {
            Utils.warn('TextDrawingStyles.getAttributesFromPosition(): invalid drawing position: "' + position + '"');
            return {};
        }

        // set horizontal alignment
        newAnchorHorBase = (origAttributes.anchorHorBase === 'page' || origAttributes.anchorHorBase === 'margin') ? origAttributes.anchorHorBase : 'column';
        _(drawingAttributes).extend({ inline: false, anchorHorBase: newAnchorHorBase, anchorHorAlign: matches[1], anchorHorOffset: 0 });

        // set text wrapping mode
        if (matches[2] === 'none') {
            _(drawingAttributes).extend({ textWrapMode: 'topAndBottom', textWrapSide: null });
        } else {
            _(drawingAttributes).extend({ textWrapMode: 'square', textWrapSide: matches[2] });
        }

        return drawingAttributes;
    };

    // exports ================================================================

    // derive this class from class DrawingStyleCollection
    return DrawingStyleCollection.extend({ constructor: TextDrawingStyles });

});
