/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/popup/validationlistmenu', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/keycodes',
    'io.ox/office/tk/forms',
    'io.ox/office/tk/popup/listmenu',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'gettext!io.ox/office/spreadsheet/main'
], function (Utils, KeyCodes, Forms, ListMenu, SheetUtils, gt) {

    'use strict';

    // class ValidationListMenu ===============================================

    /**
     * A drop-down list for data validation of type 'source' or 'list' in a
     * spreadsheet cell.
     *
     * @constructor
     *
     * @extends ListMenu
     *
     * @param {SpreadsheetView} docView
     *  The spreadsheet view that contains this instance.
     */
    function ValidationListMenu(docView) {

        var // self reference
            self = this;

        // base constructor ---------------------------------------------------

        ListMenu.call(this, { autoFocus: false, anchorAlign: 'trailing', expandWidth: true });

        // private methods ----------------------------------------------------

        /**
         * Inserts the value represented by the passed list item button of the
         * validation list menu into the active cell.
         */
        function applyListValue(buttonNode) {

            var // the value of the selected list item
                itemValue = (buttonNode.length > 0) ? Forms.getButtonValue(buttonNode) : null;

            // always hide the pop-up menu
            self.hide();

            // insert the value into the cell (via controller for busy screen etc.)
            if (_.isString(itemValue)) {
                docView.executeControllerItem('cell/active/value', itemValue);
            }
        }

        // public methods -----------------------------------------------------

        /**
         * Collects and inserts the list entries for this validation list.
         * Called by grid pane instances from the click handler of cell
         * drop-down buttons.
         *
         * @param {Address} address
         *  The address of the cell containing the validation data.
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved when this
         *  validation list has been initialized successfully.
         */
        this.initialize = function (address) {

            var // the validation collection of the active sheet
                validationCollection = docView.getValidationCollection(),
                // the resulting promise
                promise = validationCollection.queryListValues(address);

            // receive source data from document, fill the menu with the values
            promise = promise.then(function (values) {

                // close the menu, if no data is available for the list
                if (values.length === 0) { return $.Deferred().reject(); }

                // insert the data into the list
                self.clearContents();
                values.forEach(function (value) {
                    var display = _.isString(value) ? value : '#######'; // TODO: Excel uses real cell results internally
                    self.createItemNode(value || '', { label: _.noI18n(display) });
                });
            });

            // reject with a specific error code causing a warning alert box
            return promise.then(null, _.constant(SheetUtils.makeRejected('validation:source')));
        };

        /**
         * Processes the passed 'keydown' event for this list menu, if it is
         * currently visible. Moves the selected list item, if cursor keys or
         * similar keys have been pressed, or applies the selected list item,
         * if the TAB or ENTER keys have been pressed.
         *
         * @param {jQuery.Event} event
         *  The 'keydown' event object to be processed.
         *
         * @returns {Boolean}
         *  Whether the key event represents a supported key and has been
         *  processed successfully.
         */
        this.handleKeyDownEvent = function (event) {

            // do nothing in hidden pop-up menu
            if (!this.isVisible()) { return false; }

            var // the selected item button node
                itemNode = this.getSelectedItemNodes().first();

            // apply the selected value on TAB or ENTER (optionally with SHIFT)
            if (KeyCodes.matchKeyCode(event, 'TAB', { shift: null }) || KeyCodes.matchKeyCode(event, 'ENTER', { shift: null })) {
                // let event bubble up for selection change
                applyListValue(itemNode);
                return true;
            }

            // change selection in the list
            itemNode = this.getItemNodeForKeyEvent(event, itemNode);
            if (itemNode.length > 0) {
                this.selectItemNode(itemNode, { scroll: true });
                // suppress propagation and default action of the key event
                event.stopPropagation();
                event.preventDefault();
                return true;
            }

            // unsupported key
            return false;
        };

        // initialization -----------------------------------------------------

        // handle click events on the list items
        Forms.touchAwareListener({ delegate: Forms.BUTTON_SELECTOR, node: this.getNode() }, function (event) {
            applyListValue($(event.currentTarget));
        });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            self = docView = null;
        });

    } // class ValidationListMenu

    // constants --------------------------------------------------------------

    /**
     * Tool tip for cell drop-down buttons attached to an instance of the class
     * ValidationListMenu.
     *
     * @constant
     */
    ValidationListMenu.BUTTON_TOOLTIP = gt('Select a value for this cell');

    // exports ================================================================

    // derive this class from class ListMenu
    return ListMenu.extend({ constructor: ValidationListMenu });

});
