/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/popup/sheetcontextmenu', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/forms',
    'io.ox/office/tk/control/button',
    'io.ox/office/baseframework/view/popup/contextmenu',
    'io.ox/office/spreadsheet/view/labels'
], function (Utils, Forms, Button, ContextMenu, Labels) {

    'use strict';

    // class SheetContextMenu =================================================

    /**
     * A context menu for sheet tab buttons. Provides menu actions to
     * manipulate the active sheet, or the sheet collection of the document.
     *
     * @constructor
     *
     * @extends ContextMenu
     *
     * @param {SpreadsheetView} docView
     *  The spreadsheet view containing this instance.
     */
    function SheetContextMenu(docView, rootNode) {

        var // self reference
            self = this,
            // app reference
            app = docView.getApp();

        // base constructor ---------------------------------------------------

        ContextMenu.call(this, docView, rootNode, {
            selector: Forms.OPTION_BUTTON_SELECTOR,
            // do not show context menu in read-only mode, or during cell edit mode
            enableKey: ['document/editable', 'view/cell/editmode/off'],
            delay: 200,
            lazyInitHandler: initMenu
        });

        // private methods ----------------------------------------------------
        /**
         * Activate the clicked sheet, before the context menu will be shown.
         */
        function contextMenuPrepareHandler(event) {
            var // the clicked button element
                sheetButton = $(event.sourceEvent.target);

            // click the button if it is not selected (activate the sheet)
            if (!Forms.isCheckedButtonNode(sheetButton)) {
                sheetButton.click();
            }
        }

        function initMenu() {
            self.addGroup('sheet/insert', new Button({ label: Labels.INSERT_SHEET_LABEL }))
                .addSeparator()
                .addGroup('sheet/rename/dialog', new Button({ label: Labels.RENAME_SHEET_LABEL }));

            if (app.isOOXML()) {
                self.addGroup('sheet/copy/dialog', new Button({ label: Labels.COPY_SHEET_LABEL }));
            }

            self.addGroup('sheet/delete', new Button({ label: Labels.DELETE_SHEET_LABEL }))
                .addGroup('sheet/visible', new Button({ label: Labels.HIDE_SHEET_LABEL, value: false }))
                .addSeparator()
                .addGroup('sheet/reorder/dialog', new Button({ label: Labels.REORDER_SHEETS_LABEL }))
                .addGroup('sheet/showall', new Button({ label: Labels.SHOW_SHEETS_LABEL }));
        }

        // public methods -----------------------------------------------------
        this.getXY = function (event) {
            return {
                pageX: event.sourceEvent.pageX,
                pageY: event.sourceEvent.pageY
            };
        };

        // initialization -----------------------------------------------------

        // preprocessing before the context menu will be shown
        this.on('contextmenu:prepare', contextMenuPrepareHandler);

        // hide contextmenu if the user start scrolling
        this.listenTo(docView, 'change:scrollpos', function () { self.hide(); });

        // destroy all class members
        this.registerDestructor(function () {
            self = docView = rootNode = null;
        });

    } // class SheetContextMenu

    // exports ================================================================

    // derive this class from class ContextMenu
    return ContextMenu.extend({ constructor: SheetContextMenu });

});
