/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 * @author Edy Haryono <edy.haryono@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/mixin/viewfuncmixin', [
    'io.ox/office/tk/utils',
    'io.ox/office/editframework/utils/color',
    'io.ox/office/editframework/utils/border',
    'io.ox/office/editframework/utils/mixedborder',
    'io.ox/office/editframework/utils/hyperlinkutils',
    'io.ox/office/drawinglayer/view/imageutil',
    'io.ox/office/spreadsheet/utils/config',
    'io.ox/office/spreadsheet/utils/operations',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/utils/paneutils',
    'io.ox/office/spreadsheet/model/cellcollection',
    'io.ox/office/spreadsheet/model/formula/tokenarray',
    'io.ox/office/spreadsheet/view/labels',
    'io.ox/office/spreadsheet/view/dialogs',
    'gettext!io.ox/office/spreadsheet/main'
], function (Utils, Color, Border, MixedBorder, HyperlinkUtils, ImageUtils, Config, Operations, SheetUtils, PaneUtils, CellCollection, TokenArray, Labels, Dialogs, gt) {

    'use strict';

    var // convenience shortcuts
        Address = SheetUtils.Address,
        Interval = SheetUtils.Interval,
        Range = SheetUtils.Range,
        IntervalArray = SheetUtils.IntervalArray,
        RangeArray = SheetUtils.RangeArray,
        Range3DArray = SheetUtils.Range3DArray,

        // tolerance for optimal column width, in 1/100 mm
        SET_COLUMN_WIDTH_TOLERANCE = 50,

        // tolerance for optimal row height (set explicitly), in 1/100 mm
        SET_ROW_HEIGHT_TOLERANCE = 15,

        // tolerance for optimal row height (updated implicitly), in 1/100 mm
        UPDATE_ROW_HEIGHT_TOLERANCE = 50,

        // default value for a visible border
        DEFAULT_SINGLE_BORDER = { style: 'single', width: Utils.convertLengthToHmm(1, 'px'), color: Color.AUTO },

        // default settings for mixed border attributes
        DEFAULT_BORDER_DATA = {
            borderTop: {},
            borderBottom: {},
            borderLeft: {},
            borderRight: {}
        },

        // default settings for subtotal results
        DEFAULT_SUBTOTAL_DATA = {
            cells: 0,
            numbers: 0,
            sum: 0,
            min: Number.POSITIVE_INFINITY,
            max: Number.NEGATIVE_INFINITY,
            average: Number.NaN

        },

        // default settings for search/replace
        DEFAULT_SEARCH_SETTINGS = {
            query: '', // active search query
            sheets: null, // search data received from server, mapped by sheet index
            index: -1, // current index into the search results
            replace: false // whether search results are replaced currently
        },

        // maps border names to the names of the opposite borders and relative cell position
        OPPOSITE_BORDERS = {
            borderTop:    { borderName: 'borderBottom', cols:  0, rows: -1 },
            borderBottom: { borderName: 'borderTop',    cols:  0, rows:  1 },
            borderLeft:   { borderName: 'borderRight',  cols: -1, rows:  0 },
            borderRight:  { borderName: 'borderLeft',   cols:  1, rows:  0 }
        },

        // maximum size of an image in any direction, in 1/100 mm
        MAX_IMAGE_SIZE = 21000;

    // mix-in class ViewFuncMixin =============================================

    /**
     * Mix-in class for the class SpreadsheetView that provides implementations
     * of complex document functionality and status requests, called from the
     * application controller.
     *
     * @constructor
     */
    function ViewFuncMixin() {

        var // self reference (spreadsheet view instance)
            self = this,

            // the application instance
            app = this.getApp(),

            // the spreadsheet model, and other model objects
            docModel = this.getDocModel(),
            undoManager = docModel.getUndoManager(),
            numberFormatter = docModel.getNumberFormatter(),
            cellStyles = docModel.getCellStyles(),
            grammarConfig = null,

            // the active sheet model and index
            activeSheetModel = null,
            activeSheet = -1,

            // collections of the active sheet
            colCollection = null,
            rowCollection = null,
            mergeCollection = null,
            cellCollection = null,
            tableCollection = null,
            drawingCollection = null,

            // the contents and formatting of the active cell (instance of CellDescriptor)
            activeCellDesc = null,

            // additional information for selected ranges (subtotals, mixed borders)
            selectionSettings = null,

            // settings for search/replace
            searchSettings = _.copy(DEFAULT_SEARCH_SETTINGS, true);

        // private methods ----------------------------------------------------

        /**
         * Creates a localized sheet name that is not yet used in the document.
         *
         * @returns {String}
         *  A sheet name not yet used in this document.
         */
        function generateUnusedSheetName() {

            var // the new sheet name
                sheetName = '',
                // zero-based sheet index for the new sheet name
                nameIndex = docModel.getSheetCount() - 1;

            // generate a valid name
            while ((sheetName.length === 0) || docModel.hasSheet(sheetName)) {
                sheetName = SheetUtils.getSheetName(nameIndex);
                nameIndex += 1;
            }

            return sheetName;
        }

        /**
         * Imports the passed settings for the borders of the selection (in the
         * property 'borders'), and the subtotal results (in the property
         * 'subtotals').
         *
         * @param {Object} settings
         *  Settings for the selected ranges, as received from the respective
         *  server request.
         */
        function importSelectionSettings(settings) {

            // clear the old selection settings
            selectionSettings = {};

            // import mixed border attributes
            selectionSettings.borders = _.copy(DEFAULT_BORDER_DATA, true);
            if (_.isObject(settings.borders)) {
                _.extend(selectionSettings.borders, settings.borders);
                // delete inner borders for specific selections
                if (!self.hasMultipleColumnsSelected()) { delete selectionSettings.borders.borderInsideVert; }
                if (!self.hasMultipleRowsSelected()) { delete selectionSettings.borders.borderInsideHor; }
            }

            // import subtotal results
            selectionSettings.subtotals = _.copy(DEFAULT_SUBTOTAL_DATA, true);
            if (_.isObject(settings.subtotals)) {
                _.extend(selectionSettings.subtotals, settings.subtotals);
                selectionSettings.subtotals.average = selectionSettings.subtotals.sum / selectionSettings.subtotals.numbers;
            }

            // import selection lock data
            selectionSettings.locked = settings.locked === true;
        }

        /**
         * Collects several update requests, and sends a single deferred server
         * request.
         *
         * @param {Object} requestProps
         *  The properties to be inserted into the current request data object.
         */
        var requestViewUpdate = (function () {

            var // the data object collecting all grid pane requests
                requestData = null,
                // the last running request
                runningDef = null,
                // the start time of the initial request (without other requests already running)
                startTime = 0,
                // counter for running requests (next free index)
                nextRequestIndex = 0,
                // minimum request index valid for evaluation (ignore requests after switching to another sheet)
                requiredRequestIndex = 0;

            // direct callback: register a new sheet rectangle to be updated in a grid pane
            function registerUpdateRequest(requestProps) {

                // do not request anything in error mode
                if (app.isInQuit() || app.isInternalError()) { return; }

                // initialize request data, if missing or for another sheet
                if (!requestData || (requestData.sheet !== activeSheet)) {
                    requestData = { sheet: activeSheet, locale: Config.LOCALE };
                }

                // insert the passed request data
                _.extend(requestData, requestProps);
            }

            // deferred callback: send single server request and notify all waiting grid panes
            function executeUpdateRequest() {

                var // local reference to the current request data
                    localRequestData = requestData,
                    // current request index
                    localRequestIndex = nextRequestIndex,
                    // start time of this request
                    localStartTime = _.now(),
                    // the new server request
                    def = null;

                if (!localRequestData || app.isInQuit() || app.isInternalError()) { return; }
                requestData = null;

                // take start time, if no other request is running already
                if (!runningDef) {
                    startTime = _.now();
                }

                // do not query for cell contents, exclusively done in CellCollection class
                if ('cells' in localRequestData) {
                    Utils.error('ViewFuncMixin.executeUpdateRequest(): unexpected query for cell contents');
                    delete localRequestData.cells;
                }

                // immediately delete the request data (new updates may be requested in the meantime)
                localRequestIndex = nextRequestIndex;
                nextRequestIndex += 1;
                Utils.info('ViewFuncMixin.executeUpdateRequest(): sending update request #' + (localRequestIndex + 1) + ':', localRequestData);

                // wait for pending actions (bug 30044, must wait for the
                // 'insertSheet' operation before requesting data for the sheet)
                def = runningDef = app.saveChanges().then(function () {
                    return app.sendActionRequest('updateview', localRequestData);
                });

                def.fail(function () {
                    Utils.error('ViewFuncMixin.executeUpdateRequest(): update request #' + (localRequestIndex + 1) + ' failed');
                });

                def.done(function (layoutData) {

                    Utils.info('ViewFuncMixin.executeUpdateRequest(): update response #' + (localRequestIndex + 1) + ' received (' + (_.now() - localStartTime) + 'ms):', layoutData);
                    // check validity of response object
                    if (!_.isObject(layoutData) || !_.isObject(layoutData.sheet)) {
                        Utils.error('ViewFuncMixin.executeUpdateRequest(): missing required layout data');
                        return;
                    }

                    // ignore this response if the active sheet has changed in the meantime
                    if (localRequestIndex < requiredRequestIndex) { return; }

                    // if this update request is triggered by the search function
                    if (_.isObject(localRequestData.search) && !_.isObject(layoutData.found)) {
                        self.yell(Labels.NO_SEARCHRESULT_OPTIONS);
                    }

                    // store search results
                    if (_.isObject(layoutData.found)) {
                        var foundAddresses = Utils.getObjectOption(layoutData.found, 'sheets', {});
                        // convert JSON addresses to address arrays
                        searchSettings.sheets = Utils.mapProperties(foundAddresses, docModel.createAddressArray, docModel);
                        // jump to first search result only if this is a new search
                        if (!searchSettings.replace) {
                            self.searchNext('next', searchSettings.query);
                        }
                    }

                    // copy layout data for the current selection
                    if (_.isObject(layoutData.selection)) {
                        importSelectionSettings(layoutData.selection);
                    }

                    // Check if this is the last request. Do not notify listeners about response
                    // data of previous requests, but collect the response data and all change
                    // flags of multiple requests, and notify listeners after last response.
                    if (def === runningDef) {
                        runningDef = null;
                        self.trigger('change:layoutdata');
                    } else if (_.now() - startTime >= 3000) {
                        startTime = _.now();
                        self.trigger('change:layoutdata');
                    }
                });
            }

            // delete the current cached request data when the active sheet changes
            self.on('before:activesheet', function () {
                requestData = null;
                requiredRequestIndex = nextRequestIndex;
            });

            // create and return the debounced method requestViewUpdate()
            return self.createDebouncedMethod(registerUpdateRequest, executeUpdateRequest, { delay: 100, maxDelay: 250 });
        }()); // ViewFuncMixin.requestViewUpdate()

        /**
         * Requests an update of the selection part of the view layout data.
         */
        function requestSelectionUpdate(selection) {
            requestViewUpdate({ selection: { ranges: selection.ranges.toJSON(), active: selection.address.toJSON() } });
        }

        /**
         * Updates the cached data of the active cell (contents, formatting),
         * and the entire selection (border settings, subtotals). To improve
         * performance, updating settings of the entire selection (which is a
         * potentially expensive operation) will be done debounced.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.direct=false]
         *      If set to true, all selection settings will be updated
         *      immediately (without additional debounced update).
         */
        var updateSelectionSettings = (function () {

            var // whether selection update is pending (for option 'direct')
                pendingRangeUpdate = false;

            // update settings of the active cell synchronously for fast GUI feedback
            function updateActiveCell(options) {

                // receive settings of active cell from cell collection
                activeCellDesc = cellCollection.getCellEntry(self.getActiveCell());
                pendingRangeUpdate = true;

                // invoke deferred callback immediately if specified
                if (Utils.getBooleanOption(options, 'direct', false)) {
                    updateSelectedRanges();
                }
            }

            // update settings of the entire selection deferred (expensive cell iterations)
            function updateSelectedRanges() {

                var // the current cell selection
                    selection = self.getSelection(),
                    // all merged ranges covering the selection
                    mergedRanges = null,
                    // object to collect and sort all border types
                    allBorders = {},
                    // whether current selection contains a locked cell
                    selectionLocked = false;

                // selection settings may have been updated immediately (option 'direct')
                if (!pendingRangeUpdate) { return; }
                pendingRangeUpdate = false;

                // If any selected range is not completely covered by the cell collection, or if the
                // selection contains too many cells, request new data from server. Keep old selection
                // settings until that data has been received to prevent flickering icons in the side pane.
                if ((selection.ranges.cells() > SheetUtils.MAX_SELECTION_CELL_COUNT) || !cellCollection.containsRanges(selection.ranges)) {
                    selectionSettings.locked = true;
                    requestSelectionUpdate(selection);
                    return;
                }

                // initialize the selection settings object with default data
                importSelectionSettings({});

                // get all merged ranges covering the selection
                mergedRanges = mergeCollection.getMergedRanges(selection.ranges);

                // iterate all ranges
                cellCollection.iterateCellsInRanges(selection.ranges, function (cellData, originalRange) {

                    var // address of the current cell
                        address = cellData.address,
                        // clone the global OPPOSITE_BORDERS for local modification
                        oppositeBorders = _.copy(OPPOSITE_BORDERS, true),
                        // merged range covering the current cell
                        mergedRange = mergedRanges.findByAddress(address),
                        // additional columns/rows of a merged cell
                        additionalCols = 0,
                        additionalRows = 0;

                    // special handling for cells that are part of a merged range
                    if (mergedRange) {

                        // skip hidden cells inside merged range
                        if (!mergedRange.startsAt(address)) { return; }

                        // modify opposite borders for the reference cell of a merged range
                        oppositeBorders.borderRight.cols = mergedRange.cols();
                        oppositeBorders.borderBottom.rows = mergedRange.rows();
                        additionalCols = oppositeBorders.borderRight.cols - 1;
                        additionalRows = oppositeBorders.borderBottom.rows - 1;
                    }

                    // process all outer borders of the cell
                    _.each(oppositeBorders, function (OPPOSITE_BORDER, borderName) {

                        var // cell data of the preceding/next cell (taking care of merged cells)
                            nextCellData = cellCollection.getCellEntry(new Address(address[0] + OPPOSITE_BORDER.cols, address[1] + OPPOSITE_BORDER.rows)),
                            // the border value of the current and adjacent cell
                            thisBorder = cellData.attributes.cell[borderName],
                            nextBorder = nextCellData.attributes.cell[OPPOSITE_BORDER.borderName],
                            // whether the borders are visible
                            thisBorderVisible = Border.isVisibleBorder(thisBorder),
                            nextBorderVisible = Border.isVisibleBorder(nextBorder),
                            // the target border name for the border collection (inner or outer borders)
                            targetBorderName = null,
                            targetBorders = null;

                        // get the target border name for the border collection
                        switch (borderName) {
                            case 'borderLeft':
                                targetBorderName = (address[0] === originalRange.start[0]) ? 'borderLeft' : null;
                                break;
                            case 'borderRight':
                                targetBorderName = (address[0] + additionalCols === originalRange.end[0]) ? 'borderRight' : 'borderInsideVert';
                                break;
                            case 'borderTop':
                                targetBorderName = (address[1] === originalRange.start[1]) ? 'borderTop' : null;
                                break;
                            case 'borderBottom':
                                targetBorderName = (address[1] + additionalRows === originalRange.end[1]) ? 'borderBottom' : 'borderInsideHor';
                                break;
                        }

                        // skip inner border, if already processed by preceding cell
                        if (!_.isString(targetBorderName)) { return; }
                        targetBorders = allBorders[targetBorderName] || (allBorders[targetBorderName] = []);

                        // insert all visible borders into the border collection
                        if (thisBorderVisible) { targetBorders.push(thisBorder); }
                        if (nextBorderVisible) { targetBorders.push(nextBorder); }

                        // ignore invisible border, if the other border is visible
                        if (!thisBorderVisible && !nextBorderVisible) { targetBorders.push(Border.NONE); }
                    });

                    // selection is editable only if all its cells are editable
                    if (!cellData.attributes.cell.unlocked) { selectionLocked = true; }
                });

                // all border information collected and sorted in allBorders
                _.each(allBorders, function (borders, borderName) {
                    selectionSettings.borders[borderName] = MixedBorder.mixBorders(borders);
                });

                // iterate unified ranges to calculate subtotals (no check for result needed, all cells are available)
                cellCollection.iterateCellsInRanges(selection.ranges.merge(), function (cellEntry) {
                    selectionSettings.subtotals.cells += 1;
                    if (CellCollection.isNumber(cellEntry) && _.isFinite(cellEntry.result)) {
                        selectionSettings.subtotals.numbers += 1;
                        selectionSettings.subtotals.sum += cellEntry.result;
                        selectionSettings.subtotals.min = Math.min(selectionSettings.subtotals.min, cellEntry.result);
                        selectionSettings.subtotals.max = Math.max(selectionSettings.subtotals.max, cellEntry.result);
                    }
                }, { type: 'content' });
                selectionSettings.subtotals.average = selectionSettings.subtotals.sum / selectionSettings.subtotals.numbers;

                // whether selection contains locked cells
                selectionSettings.locked = selectionLocked;

                // notify listeners
                self.trigger('change:layoutdata');
            }

            // create the actual debounced updateSelectionSettings() method
            return self.createDebouncedMethod(updateActiveCell, updateSelectedRanges, { delay: 200 });
        }()); // end of local scope of updateSelectionSettings()

        /**
         * Returns the column/row intervals of the selected ranges, optionally
         * depending on a specific target column/row.
         *
         * @param {Boolean} columns
         *  Whether to return column intervals (true) or row intervals (false).
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Number} [options.target]
         *      If specified, the zero-based index of a specific target column
         *      or row. If the current selection contains entire column/row
         *      ranges covering this column/row, the respective intervals of
         *      these ranges will be returned instead; otherwise a single
         *      interval containing the target column/row will be returned.
         *  @param {Boolean} [options.visible=false]
         *      If set to true, the returned intervals will be reduced to the
         *      column or row intervals covered by the visible panes.
         *
         * @returns {IntervalArray}
         *  An array of column/row intervals for the current selection, already
         *  merged and sorted.
         */
        function getSelectedIntervals(columns, options) {

            var // the selected ranges
                ranges = self.getSelectedRanges(),
                // the target column/row
                target = Utils.getIntegerOption(options, 'target'),
                // the resulting intervals
                intervals = null;

            // use entire column/row ranges in selection, if target index has been passed
            if (_.isNumber(target)) {
                // filter full column/row ranges in selection
                ranges = ranges.filter(function (range) { return docModel.isFullRange(range, columns); });
                // if the target column/row is not contained in the selection, make an interval for it
                if (!ranges.containsIndex(target, columns)) {
                    intervals = new IntervalArray(new Interval(target));
                }
            }

            // convert selected ranges to column/row intervals
            if (!intervals) {
                intervals = ranges.intervals(columns);
            }

            // reduce the intervals to the header pane intervals
            if (Utils.getBooleanOption(options, 'visible', false)) {
                intervals = self.getVisibleIntervals(intervals, columns);
            }

            // merge and sort the resulting intervals
            return intervals.merge();
        }

        /**
         * Returns the column/row intervals from the current selection, if it
         * contains hidden columns/rows. If the selected columns/rows are
         * completely visible, the selection consists of a single column/row
         * interval, and that interval is located next to a hidden column/row
         * interval (at either side, preferring leading), that hidden interval
         * will be returned instead.
         *
         * @returns {IntervalArray}
         *  The column/row intervals calculated from the current selection,
         *  containing hidden columns/rows. May be an empty array.
         */
        function getSelectedHiddenIntervals(columns) {

            // the entire selected intervals
            var intervals = getSelectedIntervals(columns);

            // safety check, should not happen
            if (intervals.empty()) { return intervals; }

            var // the column/row collection
                collection = columns ? colCollection : rowCollection,
                // the mixed column/row attributes
                attributes = collection.getMixedAttributes(intervals);

            // if attribute 'visible' is false or null, the intervals contain hidden entries
            if (attributes.visible !== true) { return intervals; }

            // the resulting intervals
            var resultIntervals = new IntervalArray();

            // no special behavior for multiple visible intervals
            if (intervals.length > 1) { return resultIntervals; }

            // try to find preceding hidden interval
            var lastIndex = intervals.first().first - 1;
            if ((lastIndex >= 0) && !collection.isEntryVisible(lastIndex)) {
                var prevEntry = collection.getPrevVisibleEntry(lastIndex);
                resultIntervals.push(new Interval(prevEntry ? (prevEntry.index + 1) : 0, lastIndex));
                return resultIntervals;
            }

            // try to find following hidden interval
            var firstIndex = intervals[0].last + 1;
            if ((firstIndex <= collection.getMaxIndex()) && !collection.isEntryVisible(firstIndex)) {
                var nextEntry = collection.getNextVisibleEntry(firstIndex);
                resultIntervals.push(new Interval(firstIndex, nextEntry ? (nextEntry.index - 1) : collection.getMaxIndex()));
                return resultIntervals;
            }

            // no hidden intervals found
            return resultIntervals;
        }

        /**
         * Returns whether additional columns or rows can be inserted into the
         * active sheet.
         *
         * @returns {Boolean}
         *  Whether additional columns or rows can be inserted into the active
         *  sheet, according to the current selection and the number of used
         *  columns/rows in the sheet.
         */
        function canInsertIntervals(intervals, columns) {

            var // the number of inserted columns/rows
                insertCount = intervals.size(),
                // the number of used columns/rows
                usedCount = columns ? cellCollection.getUsedCols() : cellCollection.getUsedRows(),
                // the available number of columns/rows in the sheet
                maxCount = docModel.getMaxIndex(columns) + 1;

            // the number of used columns/rows, and the number of inserted columns/rows
            // must not exceed the total number of columns/rows in the sheet
            return (insertCount > 0) && ((usedCount + insertCount) <= maxCount);
        }

        /**
         * Returns whether columns or rows can be deleted from the sheet.
         *
         * @returns {Boolean}
         *  Whether the columns or rows can be deleted in the active sheet,
         *  according to the current selection.
         */
        function canDeleteIntervals(intervals, columns) {

            var // the number of deleted columns/rows
                deleteCount = intervals.size(),
                // the available number of columns/rows in the sheet
                maxCount = docModel.getMaxIndex(columns) + 1;

            // prevent deleting all columns/rows (entire sheet) at once
            return (deleteCount > 0) && (deleteCount < maxCount);
        }

        /**
         * Sets the optimal column width in the specified column intervals,
         * based on the content of cells.
         *
         * @param {IntervalArray|Interval} colIntervals
         *  An array of column intervals, or a single column interval.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected on any error.
         */
        function setOptimalColumnWidth(colIntervals) {

            var // resulting intervals with the new column widths
                resultIntervals = new IntervalArray(),
                // default width for empty columns
                defWidth = docModel.getDefaultAttributes('column').column.width,
                // the rendering cache of the active sheet
                renderCache = self.getRenderCache(),
                // left and right cell padding, plus one pixel grid line
                totalPadding = 2 * activeSheetModel.getEffectiveTextPadding() + 1;

            // process all visible columns
            colCollection.iterateEntries(colIntervals, function (colDesc) {

                var // the new column width
                    colWidth = 0;

                renderCache.iterateCellsInCol(colDesc.index, function (element) {
                    if (!element.hasMergedColumns() && (element.contentWidth > 0)) {
                        colWidth = Math.max(colWidth, element.contentWidth + totalPadding);
                    }
                });

                // convert to 1/100 mm; or use default column width, if no cells are available
                colWidth = (colWidth === 0) ? defWidth : activeSheetModel.convertPixelToHmm(colWidth);

                // bug 40737: restrict to maximum column width allowed in the UI
                // TODO: use real maximum according to file format: 255 digits in OOXML, 1000mm in ODF
                colWidth = Math.min(colWidth, SheetUtils.MAX_COLUMN_WIDTH);

                // do not generate operations, if the custom height was not modified before, and the new width is inside the tolerance
                if (!colDesc.attributes.column.customWidth && (Math.abs(colWidth - colDesc.attributes.column.width) < SET_COLUMN_WIDTH_TOLERANCE)) {
                    return;
                }

                // insert the optimal column width into the cache
                var lastInterval = resultIntervals.last();
                if (lastInterval && (lastInterval.last + 1 === colDesc.index) && (lastInterval.attrs.width === colWidth)) {
                    lastInterval.last += 1;
                } else {
                    lastInterval = new Interval(colDesc.index);
                    lastInterval.attrs = { width: colWidth, customWidth: false };
                    resultIntervals.push(lastInterval);
                }
            });

            // create operations for all columns
            return activeSheetModel.changeColumns(resultIntervals);
        }

        /**
         * Sets the optimal row height in the specified row intervals, based on
         * the content of cells.
         *
         * @param {IntervalArray|Interval} rowIntervals
         *  An array of row intervals, or a single row interval.
         *
         * @param {Boolean} [update=false]
         *  If set to true, an operation has caused updating the automatic row
         *  heights implicitly. Rows with user-defined height (row attribute
         *  'customHeight' set to true) will not be changed in this case. By
         *  default, the optimal height of all rows in the passed intervals
         *  will be calculated (manual row heights will be reset).
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected on any error.
         */
        function setOptimalRowHeight(rowIntervals, update) {

            var // tolerance for automatic row height
                heightTolerance = update ? UPDATE_ROW_HEIGHT_TOLERANCE : SET_ROW_HEIGHT_TOLERANCE,
                // resulting intervals with the new row heights
                resultIntervals = new IntervalArray(),
                // the rendering cache of the active sheet
                renderCache = self.getRenderCache();

            // process all visible rows
            rowCollection.iterateEntries(rowIntervals, function (rowDesc) {

                var // resulting optimal height of the current row
                    rowHeight = docModel.getRowHeightHmm(rowDesc.attributes.character);

                renderCache.iterateCellsInRow(rowDesc.index, function (element) {
                    if (!element.hasMergedRows() && (element.contentHeight > 0)) {
                        rowHeight = Math.max(rowHeight, activeSheetModel.convertPixelToHmm(element.contentHeight));
                    }
                });

                // bug 40737: restrict to maximum row height allowed in the UI
                // TODO: use real maximum according to file format: 409pt in OOXML, 1000mm in ODF
                rowHeight = Math.min(rowHeight, SheetUtils.MAX_ROW_HEIGHT);

                // do not generate operations, if the row height was optimal before, and the new height is inside the tolerance
                if (!rowDesc.attributes.row.customHeight && (Math.abs(rowHeight - rowDesc.attributes.row.height) < heightTolerance)) {
                    return;
                }

                // insert the optimal row height into the cache
                var lastInterval = resultIntervals.last();
                if (lastInterval && (lastInterval.last + 1 === rowDesc.index) && (lastInterval.attrs.height === rowHeight)) {
                    lastInterval.last += 1;
                } else {
                    lastInterval = new Interval(rowDesc.index);
                    lastInterval.attrs = { height: rowHeight, customHeight: false };
                    resultIntervals.push(lastInterval);
                }
            }, { customSize: update ? false : null }); // do not change custom row height in update mode

            // create operations for all rows
            return activeSheetModel.changeRows(resultIntervals);
        }

        /**
         * Updates the optimal row height in the specified ranges, based on the
         * content of cells.
         *
         * @param {RangeArray|Range} ranges
         *  An array of range addresses, or a single range address.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected on any error.
         */
        function updateOptimalRowHeight(ranges) {

            var // the merged and sorted row intervals
                rowIntervals = RangeArray.get(ranges).rowIntervals().merge();

            // restrict to visible area (performance)
            // TODO: iterate all row intervals, not single rows, important when updating entire column
            rowIntervals = self.getVisibleIntervals(rowIntervals, false);

            // set update mode (do not modify rows with custom height)
            return setOptimalRowHeight(rowIntervals, true);
        }

        /**
         * Performs additional checks whether the contents of the passed cell
         * ranges can be edited, regardless of their 'unlocked' attribute, and
         * the lock state of the active sheet. Used as helper method by various
         * public methods of this class that all support the same options.
         *
         * @param {RangeArray|Range} ranges
         *  An array of cell range addresses, or a single cell range address.
         *
         * @options {Object} [options]
         *  Optional parameters:
         *  @param {String} [options.lockTables='none']
         *      Specifies how to check table ranges covering the passed cell
         *      range addresses (an auto-filter range, however, is never
         *      considered to be locked):
         *      - 'none' (default): Table ranges are not locked per se.
         *      - 'full': The entire table range is locked.
         *      - 'header': Only the header cells of the table are locked.
         *
         * @returns {Boolean}
         *  Whether the contents of the passed cell ranges can be edited.
         */
        function checkForEditableContents(ranges, options) {

            // check the table ranges according to the passed mode
            switch (Utils.getStringOption(options, 'lockTables', 'none')) {
                case 'full':
                    return tableCollection.getAllTables().every(function (tableModel) {
                        return tableModel.isAutoFilter() || !ranges.overlaps(tableModel.getRange());
                    });
                case 'header':
                    return tableCollection.getAllTables().every(function (tableModel) {
                        return tableModel.isAutoFilter() || !ranges.overlaps(tableModel.getRange().header());
                    });
            }

            // any other table mode: assume that tables are not locked
            return true;
        }

        /**
         * Fills a cell with the passed value and formatting, and updates the
         * optimal row height of the cell. See public method
         * SheetModel.setSingleCellContents() for details.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved after the cell has been changed, or
         *  that will be rejected on any error.
         */
        function setSingleCellContents(address, value, attributes, options) {

            // bug 36773: collect all changes into an undo group
            return undoManager.enterUndoGroup(function () {

                // try to fill the cell
                var promise = activeSheetModel.setSingleCellContents(address, value, attributes, options);

                // set optimal row height on success
                return promise.then(function () {
                    return updateOptimalRowHeight(new Range(address));
                });
            });
        }

        /**
         * Fills a cell range with the passed values and formatting, and
         * updates the optimal row height of the affected cells. See public
         * method SheetModel.setCellContents() for details.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved after the cell has been changed, or
         *  that will be rejected on any error.
         */
        function setCellContents(start, contents, options) {

            // bug 36773: collect all changes into an undo group
            return undoManager.enterUndoGroup(function () {

                // try to fill the cells
                var promise = activeSheetModel.setCellContents(start, contents, options);

                // set optimal row height on success
                return promise.then(function () {
                    return updateOptimalRowHeight(Range.create(start[0], start[1], start[0], start[1] + contents.length - 1));
                });
            });
        }

        /**
         * Fills all cell ranges with the same value and formatting, and
         * updates the optimal row height of the affected cells. See public
         * method SheetModel.fillCellRanges() for details.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved after the cell has been changed, or
         *  that will be rejected on any error.
         */
        function fillCellRanges(ranges, value, attributes, options) {

            // bug 36773: collect all changes into an undo group
            return undoManager.enterUndoGroup(function () {

                // try to fill the cell ranges
                var promise = activeSheetModel.fillCellRanges(ranges, value, attributes, options);

                // set optimal row height on success
                return promise.then(function () {
                    return updateOptimalRowHeight(ranges);
                });
            });
        }

        /**
         * Clears all cell ranges, and updates the optimal row height of the
         * affected cells. See public method SheetModel.clearCellRanges() for
         * details.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved after the cell has been cleared, or
         *  that will be rejected on any error.
         */
        function clearCellRanges(ranges) {

            // bug 36773: collect all changes into an undo group
            return undoManager.enterUndoGroup(function () {

                // try to clear the cell ranges
                var promise = activeSheetModel.clearCellRanges(ranges);

                // set optimal row height on success
                return promise.then(function () {
                    return updateOptimalRowHeight(ranges);
                });
            });
        }

        /**
         * Clears all cached search results.
         */
        function clearSearchResults() {
            searchSettings = _.copy(DEFAULT_SEARCH_SETTINGS, true);
        }

        /**
         * Selects the specified search result cell, and scrolls to that cell.
         *
         * @param {Number} index
         *  The zero-based index into the array of search result addresses.
         */
        function selectSearchResult(index) {

            var // the addresses of all cells containing the search text
                addresses = searchSettings.sheets ? searchSettings.sheets[activeSheet] : null,
                // the specified cell address
                address = addresses ? addresses[index] : null;

            searchSettings.index = index;
            if (address) {
                self.selectCell(address);
                self.scrollToCell(address);
            }
        }

        /**
         * Checks whether the passed search query string matches the cached
         * query string of an ongoing search operation, and returns the cached
         * cell addresses of the search result on success. If the passed string
         * is different from the cached query string, a new search request will
         * be sent to the server automatically.
         *
         * @param {String} query
         *  The search query string to be checked.
         *
         * @returns {AddressArray|Null}
         *  The addresses of all cells containing the query text in the active
         *  sheet; or null, if no search result is available.
         */
        function getSearchResults(query) {

            // if there are no results or if user wants to search a new query, do a search and quit
            if ((searchSettings.query !== query) || !_.isObject(searchSettings.sheets)) {

                // reset current search results before a new search request
                clearSearchResults();

                // do not request results for an empty search string
                // do nothing while sending operations (TODO: really?)
                if ((query.length > 0) && (app.getState() !== 'sending')) {

                    // store search query string for later usage
                    searchSettings.query = query;

                    // the search starts from the active cell (this way, user will always get the closest result)
                    requestViewUpdate({ search: {
                        searchText: query,
                        start: self.getActiveCell(),
                        limit: SheetUtils.MAX_SEARCH_RESULT_COUNT
                    } });
                }
                return null;
            }

            var // the addresses of all cells containing the search text
                addresses = searchSettings.sheets[activeSheet];

            // user may have switched to other sheets after search
            return (addresses && (addresses.length > 0)) ? addresses : null;
        }

        /**
         * Replaces the text in the active cell, if it is part of the search
         * results, otherwise of the first search result.
         *
         * @param {String} query
         *  The text to be replaced.
         *
         * @param {String} replace
         *  The replacement text.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved after the search string has been
         *  replaced, or that will be rejected on any error.
         */
        function replaceSingle(query, replace) {

            var // the addresses of all cells containing the searched text
                addresses = getSearchResults(query);

            // nothing to do without search results
            if (!addresses) { return $.when(); }

            var // address of the active cell
                activeCell = self.getActiveCell(),
                // the array index of the active cell in the search results
                index = Utils.findFirstIndex(addresses, function (elem) { return elem.equals(activeCell); });

            // select the first search result, if active cell is not a result cell (this
            // updates the 'activeCellDesc' object with the contents of the new cell!)
            index = Math.max(0, index);
            selectSearchResult(index);

            var // the old text before replacement
                oldText = _.isString(activeCellDesc.formula) ? activeCellDesc.formula : activeCellDesc.display,
                // the new text after replacement
                // TODO handle number format special replacements like Date, Currency etc.
                newText = _.isString(oldText) ? oldText.replace(new RegExp(_.escapeRegExp(query), 'gi'), replace) : null;

            // nothing to do, if the text does not change
            if (oldText === newText) { return $.when(); }

            // send a document operation to change the cell
            searchSettings.replace = true;
            var promise = self.setCellContents(newText, undefined, { parse: true });

            promise.done(function () {

                // remove the modified cell from the result array
                addresses.splice(index, 1);

                // show info alert banner, if all cells have been replaced
                if (addresses.length === 0) {
                    searchSettings.sheets = null;
                    self.yell({ type: 'info', message: gt('Nothing else to replace.') });
                } else {
                    // select the next search result (wrap to first result, if the last has been replaced)
                    selectSearchResult(index % addresses.length);
                }
            });

            return promise;
        }

        /**
         * Sends a server request to replace all occurrences of the specified
         * text in the active sheet.
         *
         * @param {String} query
         *  The text to be replaced.
         *
         * @param {String} replace
         *  The replacement text.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved after the search string has been
         *  replaced, or that will be rejected on any error.
         */
        function replaceAll(query, replace) {

            var // the index of the active sheet where to replace (cached for deferred done handler)
                sheet = activeSheet,
                // the server request for the 'replace all' action
                request = app.sendActionRequest('replaceAll', {
                    sheet: sheet,
                    searchText: query,
                    replaceText: replace
                });

            // ignore failed requests
            request.fail(function (response) {
                Utils.error('ViewFuncMixin.replaceAll(): replace request failed:', response);
            });

            // show an alert banner containing the number of replaced cells
            request.done(function (result) {

                var // the sheet map containing addresses of all changed cells
                    sheetMap = Utils.getObjectOption(result.changed, 'sheets'),
                    // the addresses of the changed ranges in the active sheet
                    ranges = sheetMap ? docModel.createRangeArray(Utils.getArrayOption(sheetMap[sheet], 'cells')) : null;

                // show special label, if no cell has changed
                if (!ranges || ranges.empty()) {
                    self.yell({ type: 'info', message: gt('Nothing to replace.') });
                    return;
                }

                // generate a message that contains the number of replaced cells
                var cellCount = ranges.cells();
                self.yell({
                    type: 'success',
                    message: gt.format(
                        //#. %1$d is the number of cells replaced with a 'replace all' document action
                        //#, c-format
                        gt.ngettext('Successfully replaced %1$d cell.', 'Successfully replaced %1$d cells.', cellCount),
                        _.noI18n(cellCount)
                    )
                });
            });

            return request;
        }

        /**
         * Initializes all sheet-dependent class members according to the
         * current active sheet.
         */
        function changeActiveSheetHandler(event, sheet, sheetModel) {

            // store model instance and index of the new active sheet
            activeSheetModel = sheetModel;
            activeSheet = sheet;

            // get collections of the new active sheet
            colCollection = sheetModel.getColCollection();
            rowCollection = sheetModel.getRowCollection();
            mergeCollection = sheetModel.getMergeCollection();
            cellCollection = sheetModel.getCellCollection();
            tableCollection = sheetModel.getTableCollection();
            drawingCollection = sheetModel.getDrawingCollection();
        }

        /**
         * Handles 'docs:update:cells' notifications triggered for any changed
         * cell in the entire document.
         */
        function updateNotificationHandler(changedData) {

            // do nothing, if the active sheet did not change
            if (!(activeSheet in changedData)) { return; }

            // throw away search results if the active sheet is altered in any way
            if (self.isSearchActive() && !searchSettings.replace) {
                clearSearchResults();
            }
        }

        /**
         * Initializes additional document functionality for debug mode.
         */
        function initDebugOperations() {

            // additional controller items for the debug controls
            app.onInit(function () {

                function triggerUpdateCellsEvent(ranges) {
                    var allRanges = Range3DArray.createFromRanges(ranges, activeSheet);
                    app.trigger('docs:update:cells', Utils.makeSimpleObject(activeSheet, { sheet: activeSheet, dirtyRanges: ranges, changedRanges: ranges }), allRanges);
                }

                app.getController().registerDefinitions({
                    'debug/view/update': {
                        parent: 'debug/enabled',
                        set: function (type) {
                            switch (type) {
                                case 'view':
                                    triggerUpdateCellsEvent(new RangeArray(docModel.getSheetRange()));
                                    break;
                                case 'cells':
                                    triggerUpdateCellsEvent(self.getSelectedRanges());
                                    break;
                                case 'selection':
                                    requestSelectionUpdate(self.getSelection());
                                    break;
                            }
                        }
                    }
                });
            });
        }

        // public methods -----------------------------------------------------

        /**
         * Returns the subtotal results in the current selection.
         *
         * @returns {Object}
         *  The current subtotal results in the current selection, in the
         *  properties 'sum', 'min', 'max', 'numbers', 'cells', and 'average'.
         */
        this.getSubtotals = function () {
            return _.clone(selectionSettings.subtotals);
        };

        /**
         * Returns whether the active sheet is locked. In locked sheets it is
         * not possible to insert, delete, or modify columns and rows, to edit
         * protected cells, or to manipulate drawing objects.
         *
         * @returns {Boolean}
         *  Whether the active sheet is locked.
         */
        this.isSheetLocked = function () {
            return activeSheetModel.isLocked();
        };

        /**
         * Returns whether the active cell is currently locked. The cell is in
         * locked state, if its 'unlocked' attribute is not set, AND if the
         * sheet is locked.
         *
         * @returns {Boolean}
         *  Whether the active cell is locked.
         */
        this.isActiveCellLocked = function () {
            return !activeCellDesc.attributes.cell.unlocked && this.isSheetLocked();
        };

        /**
         * Returns whether the passed ranges are editable, whereas editable
         * means that no locked cells are found in the ranges, and the sheet
         * where these ranges are, is not protected.
         *
         * @param {RangeArray|Range} ranges
         *  An array of cell range addresses, or a single cell range address.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {String} [options.lockTables='none']
         *      Specifies how to check a table range covering the passed cell
         *      ranges (an auto-filter range, however, is never considered to
         *      be locked):
         *      - 'none' (default): Table ranges are not locked per se.
         *      - 'full': The entire table range is locked.
         *      - 'header': Only the header cells of the table are locked.
         *  @param {String} [options.error='cells:locked']
         *      The error code to be passed to the rejected promise.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved if the passed ranges are editable;
         *  or that will be rejected with an object with property 'cause' set
         *  to an appropriate error code (see option 'error' above), if they
         *  are not.
         */
        this.areRangesEditable = function (ranges, options) {

            var // the error code for rejecting the promise
                errorCode = Utils.getStringOption(options, 'error', 'cells:locked'),
                // the promise representing the AJAX request
                request = null,
                // whether all cells in all ranges are editable (not locked)
                rangesEditable = true,
                // the number of cells visited locally
                visitedCells = 0;

            // prevent processing any cell twice
            ranges = RangeArray.get(ranges).merge();

            // bug 39869: do not allow to change the contents of table ranges
            if (!checkForEditableContents(ranges, options)) {
                return SheetUtils.makeRejected(errorCode);
            }

            // return resolved promise if sheet is not locked
            if (!activeSheetModel.isLocked()) { return $.when(); }

            // visit all cells locally, if they are covered by the cell collection
            if (cellCollection.containsRanges(ranges)) {

                // visit only the existing cells in the collection (performance),
                // but include all cells in all hidden columns and rows
                cellCollection.iterateCellsInRanges(ranges, function (cellData) {
                    visitedCells += 1;
                    if (!cellData.attributes.cell.unlocked) {
                        rangesEditable = false;
                        return Utils.BREAK; // locked cell found: exit loop
                    }
                }, { type: 'existing', hidden: 'all' });

                // if 'rangesEditable' is false, a locked cell has been found already
                if (!rangesEditable) { return SheetUtils.makeRejected(errorCode); }

                // if the ranges contain undefined cells not visited in the iterator above,
                // check the default attributes of empty cells
                if (visitedCells < ranges.cells()) {
                    if (!cellStyles.getDefaultStyleAttributeSet().cell.unlocked) {
                        return SheetUtils.makeRejected(errorCode);
                    }
                }

                // all cells are editable
                return $.when();
            }

            // ranges are not covered by the cell collections, send server request
            // to detect the locked state of the specified cells
            request = app.sendActionRequest('updateview', {
                sheet: activeSheet,
                locale: Config.LOCALE,
                selection: { ranges: ranges.toJSON(), active: ranges.first().start.toJSON() }
            });

            // reject the promise if selection contains a locked cell
            return request.then(function (response) {
                var selection = Utils.getObjectOption(response, 'selection'),
                    locked = Utils.getBooleanOption(selection, 'locked', false);
                return locked ? SheetUtils.makeRejected(errorCode) : null;
            });
        };

        /**
         * Tries to initiate modifying the active sheet. If the document is in
         * read-only mode, or the active sheet is locked, a notification will
         * be shown to the user.
         *
         * @param {String} [errorCode='sheet:locked']
         *  The error code specifying the type of the alert notification.
         *
         * @returns {Boolean}
         *  Whether modifying the active sheet is possible.
         */
        this.requireUnlockedActiveSheet = function (errorCode) {

            // prerequisite is global document edit mode (shows its own warning alert)
            if (!this.requireEditMode()) { return false; }

            // success if the sheet is not locked
            if (!this.isSheetLocked()) { return true; }

            // show warning alert banner for locked sheets
            return this.yellOnResult(errorCode || 'sheet:locked');
        };

        /**
         * Tries to initiate modifying the active cell in the active sheet. If
         * the document is in read-only mode, or the active sheet and the
         * active cell are locked, a notification will be shown to the user.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {String} [options.lockTable='none']
         *      Specifies how to check a table range covering the active cell
         *      (an auto-filter range, however, is never considered to be
         *      locked):
         *      - 'none' (default): Table ranges are not locked per se.
         *      - 'full': The entire table range is locked.
         *      - 'header': Only the header cells of the table are locked.
         *
         * @returns {Boolean}
         *  Whether modifying the current active cell is possible.
         */
        this.requireEditableActiveCell = function (options) {

            // prerequisite is global document edit mode (shows its own warning alert)
            if (!this.requireEditMode()) { return false; }

            // bug 39869: do not allow to change table ranges (according to passed options)
            var locked = this.isActiveCellLocked() || !checkForEditableContents(new Range(this.getActiveCell()), options);

            // show warning alert banner for locked active cell
            return locked ? this.yellOnResult('cells:locked') : true;
        };

        /**
         * Tries to initiate modifying all cells in the current selection of
         * the active sheet. If the document is in read-only mode, or the
         * active sheet is locked and the selection contains locked cells, a
         * notification will be shown to the user.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {String} [options.lockTable='none']
         *      Specifies how to check a table range covering the selected cell
         *      ranges:
         *      - 'none' (default): Table ranges are not locked per se.
         *      - 'full': The entire table range is locked.
         *      - 'header': Only the header cells of the table are locked.
         *
         * @returns {Boolean}
         *  Whether modifying the current selection is possible.
         */
        this.requireEditableSelection = function (options) {

            // prerequisite is global document edit mode (shows its own warning alert)
            if (!this.requireEditMode()) { return false; }

            // the selection is locked, if it contains a locked cell, AND if the sheet is locked
            var locked = selectionSettings.locked && this.isSheetLocked();

            // bug 39869: do not allow to change table ranges (according to passed options)
            locked = locked || !checkForEditableContents(this.getSelectedRanges(), options);

            // show warning alert banner for locked selection
            return locked ? this.yellOnResult('cells:locked') : true;
        };

        // sheet operations ---------------------------------------------------

        /**
         * Returns whether the passed sheet type is supported by this
         * application.
         *
         * @param {String} sheetType
         *  A sheet type identifier, as used in the 'insertSheet' operation.
         *
         * @returns {Boolean}
         *  Whether the passed sheet type is supported by this application.
         */
        this.isSheetTypeSupported = function (sheetType) {
            return (/^(worksheet)$/).test(sheetType);
        };

        /**
         * Inserts a new sheet in the spreadsheet document, and activates it in
         * the view. In case of an error, shows a notification message.
         *
         * @returns {Boolean}
         *  Whether the new sheet has been inserted successfully.
         */
        this.insertSheet = function () {

            var // create an unused sheet name
                sheetName = generateUnusedSheetName(),
                // try to insert a new sheet
                result = docModel.insertSheet(activeSheet + 1, sheetName);

            // show error messages, convert result to boolean
            result = this.yellOnResult(result);

            // activate the new sheet
            if (result) { this.activateNextSheet(); }
            return result;
        };

        /**
         * Deletes the active sheet from the spreadsheet document, and
         * activates another sheet in the view. Does nothing, if the document
         * contains a single sheet only or if the user doesn't really want
         * to delete the sheet without undo after being asked for.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved, if the active sheet has been
         *  deleted, or that will be rejected otherwise.
         */
        this.deleteSheet = function () {

            // ask to proceed (deleting sheets cannot be undone)
            var promise = this.showQueryDialog(gt('Delete Sheet'), gt('The following delete sheet operation cannot be undone. Proceed operation?'));

            // delete the sheet after confirmation
            promise = promise.then(function () {
                if (!docModel.deleteSheet(activeSheet)) {
                    return $.Deferred().reject();
                }
            });

            return promise;
        };

        /**
         * Moves the active sheet in the document to a new position.
         *
         * @param {Number} toSheet
         *  The new zero-based index of the sheet.
         *
         * @returns {Boolean}
         *  Whether moving the sheet was successful.
         */
        this.moveSheet = function (toSheet) {
            return docModel.moveSheet(activeSheet, toSheet);
        };

        /**
         * Shows the 'Reorder Sheets' dialog and creates and applies the
         * respective 'moveSheet' operations.
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved after the
         *  dialog has been closed, and the sheets have been reordered.
         */
        this.showReorderSheetsDialog = function () {

            // leave cell edit mode before showing the dialog
            if (!this.leaveCellEditMode('auto', { validate: true })) {
                return $.Deferred().reject();
            }

            return new Dialogs.SheetOrderDialog(this).show().done(function (collectOps) {
                undoManager.enterUndoGroup(function () {
                    collectOps.forEach(function (op) {
                        docModel.moveSheet(op.from, op.to);
                    });
                });
            });
        };

        /**
         * Copies the active sheet and inserts the copy with the provided sheet
         * name behind the active sheet. In case of an error, shows a warning
         * message.
         *
         * @param {String} sheetName
         *  The name of the new sheet.
         *
         * @returns {Boolean}
         *  Whether the sheet has been copied successfully.
         */
        this.copySheet = function (sheetName) {

            var // try to copy the sheet
                result = docModel.copySheet(activeSheet, activeSheet + 1, sheetName);

            // show error messages, convert result to boolean
            result = this.yellOnResult(result);

            // activate the new sheet
            if (result) { this.activateNextSheet(); }
            return result;
        };

        /**
         * Shows a dialog that allows to enter a name of a new sheet copied
         * from the active sheet. The dialog will be kept open until a valid
         * sheet name has been entered.
         *
         * @returns {jQuery.Promise}
         *  A promise representing the dialog. Will be resolved with the new
         *  sheet name, after the sheet has been copied successfully; or will
         *  be rejected, if the dialog has been canceled, or if the document
         *  has switched to read-only mode.
         */
        this.showCopySheetDialog = function () {

            // leave cell edit mode before copying the sheet
            if (!this.leaveCellEditMode('auto', { validate: true })) {
                return $.Deferred().reject();
            }

            // create the dialog, initialize with an unused sheet name
            var dialog = new Dialogs.SheetNameDialog(this, generateUnusedSheetName(), this.copySheet.bind(this), {
                title: gt('Copy Sheet'),
                okLabel: gt('Copy')
            });

            // show the dialog, copying the sheet will be done in the callback function
            return dialog.show();
        };

        /**
         * Returns the name of the active sheet.
         *
         * @returns {String}
         *  The name of the active sheet.
         */
        this.getSheetName = function () {
            return docModel.getSheetName(activeSheet);
        };

        /**
         * Renames the active sheet in the spreadsheet document, if possible.
         * In case of an error, shows a notification message.
         *
         * @param {String} sheetName
         *  The new name of the active sheet.
         *
         * @returns {Boolean}
         *  Whether the sheet has been renamed successfully.
         */
        this.renameSheet = function (sheetName) {

            // leave cell edit mode before renaming the sheet (sheet name may be used in formulas)
            if (!this.leaveCellEditMode('auto', { validate: true })) {
                return false;
            }

            // try to rename the sheet, show error messages, convert result to boolean
            var result = docModel.renameSheet(activeSheet, sheetName);
            return this.yellOnResult(result);
        };

        /**
         * Shows a dialog that allows to enter a new name for the active sheet.
         * The dialog will be kept open until a valid sheet name has been
         * entered.
         *
         * @returns {jQuery.Promise}
         *  A promise representing the dialog. Will be resolved with the new
         *  sheet name, after the sheet has been renamed successfully; or will
         *  rejected, if the dialog has been canceled, or if the document has
         *  switched to read-only mode.
         */
        this.showRenameSheetDialog = function () {

            // leave cell edit mode before renaming the sheet (sheet name may be used in formulas)
            if (!this.leaveCellEditMode('auto', { validate: true })) {
                return $.Deferred().reject();
            }

            // create the dialog, initialize with the name of the active sheet
            var dialog = new Dialogs.SheetNameDialog(this, this.getSheetName(), this.renameSheet.bind(this), {
                title: gt('Rename Sheet'),
                okLabel: gt('Rename')
            });

            // show the dialog, renaming the sheet will be done in the callback function
            return dialog.show();
        };

        /**
         * Returns the merged formatting attributes of the active sheet.
         *
         * @returns {Object}
         *  The merged attribute set of the active sheet.
         */
        this.getSheetAttributes = function () {
            return activeSheetModel.getMergedAttributes();
        };

        /**
         * Changes the formatting attributes of the active sheet.
         *
         * @param {Object} attributes
         *  An (incomplete) attribute set with the new attributes for the
         *  active sheet.
         *
         * @returns {ViewFuncMixin}
         *  A reference to this instance.
         */
        this.setSheetAttributes = function (attributes) {

            // leave cell edit mode before hiding the sheet
            if (_.isObject(attributes.sheet) && (attributes.sheet.visible === false)) {
                if (!this.leaveCellEditMode('auto', { validate: true })) {
                    return this;
                }
            }

            // generate the 'setSheetAttributes' operation
            activeSheetModel.createAndApplyOperations(function (generator) {
                generator.generateSheetOperation(Operations.CHANGE_SHEET, { attrs: attributes });
            });
            return this;
        };

        /**
         * Makes all hidden sheets in the view visible.
         *
         * @returns {ViewFuncMixin}
         *  A reference to this instance.
         */
        this.showAllSheets = function () {

            var // the attribute set to be inserted into the operations
                properties = { attrs: { sheet: { visible: true } } };

            // generate the 'setSheetAttributes' operations for all hidden sheets
            docModel.createAndApplyOperations(function (generator) {
                docModel.iterateSheetModels(function (sheetModel, sheet) {
                    if (!sheetModel.getMergedAttributes().sheet.visible) {
                        generator.generateOperation(Operations.CHANGE_SHEET, _.extend({ sheet: sheet }, properties));
                    }
                });
            });
            return this;
        };

        // sheet view operations ----------------------------------------------

        /**
         * Returns the value of a specific view attribute of the active sheet.
         *
         * @param {String} name
         *  The name of the view attribute.
         *
         * @returns {Any}
         *  A clone of the attribute value.
         */
        this.getSheetViewAttribute = function (name) {
            return activeSheetModel.getViewAttribute(name);
        };

        /**
         * Changes specific view attributes of the active sheet.
         *
         * @param {Object} attributes
         *  A map with all view attributes to be changed.
         *
         * @returns {ViewFuncMixin}
         *  A reference to this instance.
         */
        this.setSheetViewAttributes = function (attributes) {
            activeSheetModel.setViewAttributes(attributes);
            return this;
        };

        /**
         * Changes a specific view attribute of the active sheet.
         *
         * @param {String} name
         *  The name of the view attribute.
         *
         * @param {Any} value
         *  The new value for the view attribute.
         *
         * @returns {ViewFuncMixin}
         *  A reference to this instance.
         */
        this.setSheetViewAttribute = function (name, value) {
            activeSheetModel.setViewAttribute(name, value);
            return this;
        };

        /**
         * Returns the current zoom factor of the active sheet.
         *
         * @returns {Number}
         *  The current zoom factor, as floating-point number. The value 1
         *  represents a zoom of 100%.
         */
        this.getZoom = function () {
            return activeSheetModel.getViewAttribute('zoom');
        };

        /**
         * Changes the current zoom factor of all sheets in the document.
         *
         * @param {Number} zoom
         *  The new zoom factor, as floating-point number. The value 1
         *  represents a zoom of 100%.
         *
         * @returns {ViewFuncMixin}
         *  A reference to this instance.
         */
        this.setZoom = function (zoom) {
            activeSheetModel.setViewAttribute('zoom', zoom);
            return this;
        };

        /**
         * Decreases the zoom factor of the active sheet in the document.
         *
         * @returns {ViewFuncMixin}
         *  A reference to this instance.
         */
        this.decZoom = function () {

            var // current zoom factor
                currZoom = this.getZoom(),
                // find last entry in ZOOM_FACTORS with a factor less than current zoom
                prevZoom = Utils.findLast(PaneUtils.ZOOM_FACTORS, function (zoom) { return zoom < currZoom; });

            // set new zoom at active sheet
            if (_.isNumber(prevZoom)) { this.setZoom(prevZoom); }
            return this;
        };

        /**
         * Increases the zoom factor of the active sheet in the document.
         *
         * @returns {ViewFuncMixin}
         *  A reference to this instance.
         */
        this.incZoom = function () {

            var // current zoom factor
                currZoom = this.getZoom(),
                // find first entry in ZOOM_FACTORS with a factor greater than current zoom
                nextZoom = _.find(PaneUtils.ZOOM_FACTORS, function (zoom) { return zoom > currZoom; });

            // set new zoom at active sheet
            if (_.isNumber(nextZoom)) { this.setZoom(nextZoom); }
            return this;
        };

        /**
         * Generates and sends operations for all view attributes that have
         * been changed while the document was open.
         *
         * @returns {ViewFuncMixin}
         *  A reference to this instance.
         */
        this.sendChangedViewAttributes = function () {

            // Only send changed view attributes when user has edit rights, and has modified
            // the document locally, or if 'important' sheet view settings have been changed.
            // This is needed to prevent creating new versions of the document after selecting
            // cells or scrolling around, but without editing anything.
            if (!docModel.getEditMode()) { return this; }

            var // whether the document is modified, and view settings can be sent
                modified = app.isLocallyModified();

            // detect if any sheet contains 'important' changed view attributes
            if (!modified) {
                docModel.iterateSheetModels(function (sheetModel) {
                    if (sheetModel.hasChangedViewAttributes()) {
                        modified = true;
                        return Utils.BREAK;
                    }
                });
            }

            // nothing to do (e.g. only selection has been changed, which will not be saved then)
            if (!modified) { return this; }

            // generate and apply the operations for the document, and all affected sheets
            docModel.createAndApplyOperations(function (generator) {

                // create attribute operation for the active sheet index
                if (activeSheet !== docModel.getDocumentAttributes().activeSheet) {
                    generator.generateOperation(Operations.SET_DOCUMENT_ATTRIBUTES, { attrs: { document: { activeSheet: activeSheet } } });
                }

                // create operations for the changed view attributes
                docModel.iterateSheetModels(function (sheetModel, sheet) {

                    var // the new sheet attributes according to the view settings
                        sheetAttributes = sheetModel.getChangedViewAttributes();

                    // create an operation, if any attributes have been changed
                    if (!_.isEmpty(sheetAttributes)) {
                        generator.generateOperation(Operations.CHANGE_SHEET, { sheet: sheet, attrs: { sheet: sheetAttributes } });
                    }
                });
            });
            return this;
        };

        // split settings -----------------------------------------------------

        /**
         * Returns whether the view split mode is enabled for the active sheet,
         * regardless of the specific split mode (dynamic or frozen mode).
         *
         * @returns {Boolean}
         *  Whether the view split mode is enabled for the active sheet.
         */
        this.hasSplit = function () {
            return activeSheetModel.hasSplit();
        };

        /**
         * Returns the current view split mode of the active sheet.
         *
         * @returns {String}
         *  The current view split mode. The empty string indicates that the
         *  view is not split at all. Otherwise, one of the following values
         *  will be returned:
         *  - 'split': Dynamic split mode (movable split lines).
         *  - 'frozen': Frozen split mode (a fixed number of columns and/or
         *      rows are frozen, split lines are not movable). When leaving
         *      frozen split mode, the split lines will be hidden.
         *  - 'frozenSplit': Frozen split mode (same behavior as 'frozen'
         *      mode). When leaving frozen split mode, the split lines remain
         *      visible and become movable (dynamic 'split' mode).
         */
        this.getSplitMode = function () {
            return activeSheetModel.getSplitMode();
        };

        /**
         * Returns whether the view is currently split, and dynamic split mode
         * is activated. Returns also true, if the split view is currently
         * frozen, but will thaw to dynamic split mode.
         *
         * @returns {Boolean}
         *  Whether dynamic split mode is activated (split modes 'split' or
         *  'frozenSplit').
         */
        this.hasDynamicSplit = function () {
            return activeSheetModel.hasDynamicSplit();
        };

        /**
         * Returns whether the view is currently split, and the frozen split
         * mode is activated.
         *
         * @returns {Boolean}
         *  Whether frozen split mode is activated (split modes 'frozen' or
         *  'frozenSplit').
         */
        this.hasFrozenSplit = function () {
            return activeSheetModel.hasFrozenSplit();
        };

        /**
         * Toggles the dynamic view split mode according to the current view
         * split mode, and the current selection. Enabling dynamic split mode
         * in a frozen split view will thaw the frozen panes. Otherwise,
         * enabling the dynamic split mode will split the view at the position
         * left of and above the active cell. Disabling the dynamic split mode
         * will remove the split also if the view is in frozen split mode.
         *
         * @param {Boolean} state
         *  The new state of the dynamic view split mode.
         *
         * @returns {ViewFuncMixin}
         *  A reference to this instance.
         */
        this.setDynamicSplit = function (state) {

            var // whether the view is currently split
                hasSplit = this.hasSplit(),
                // address of the active cell
                address = this.getActiveCell(),
                // the position of the active cell in the sheet
                cellRectangle = null,
                // the visible area of the active grid pane that will be split
                visibleRectangle = null,
                // the new split positions
                splitWidth = 0, splitHeight = 0,
                // additional scroll anchor attributes
                attributes = null;

            // disable dynamic or frozen split mode
            if (!state && hasSplit) {
                activeSheetModel.clearSplit();

            // convert frozen split to dynamic split
            } else if (state && this.hasFrozenSplit()) {
                activeSheetModel.setDynamicSplit(activeSheetModel.getSplitWidthHmm(), activeSheetModel.getSplitHeightHmm());

            // set dynamic split at active cell
            } else if (state && !hasSplit) {

                // additional scroll anchor attributes
                attributes = {};

                // get position of active cell and visible area of the grid pane
                cellRectangle = this.getCellRectangle(address, { expandMerged: true });
                visibleRectangle = this.getActiveGridPane().getVisibleRectangle();

                // do not split vertically, if the active cell is shown at the left border
                splitWidth = cellRectangle.left - visibleRectangle.left;
                if ((-cellRectangle.width < splitWidth) && (splitWidth < PaneUtils.MIN_PANE_SIZE)) {
                    splitWidth = 0;
                }

                // do not split horizontally, if the active cell is shown at the top border
                splitHeight = cellRectangle.top - visibleRectangle.top;
                if ((-cellRectangle.height < splitHeight) && (splitHeight < PaneUtils.MIN_PANE_SIZE)) {
                    splitHeight = 0;
                }

                // split in the middle of the grid pane, if active cell is outside the visible area
                if (((splitWidth === 0) && (splitHeight === 0)) ||
                    (splitWidth < 0) || (splitWidth > visibleRectangle.width - PaneUtils.MIN_PANE_SIZE) ||
                    (splitHeight < 0) || (splitHeight > visibleRectangle.height - PaneUtils.MIN_PANE_SIZE)
                ) {
                    splitWidth = Math.floor(visibleRectangle.width / 2);
                    splitHeight = Math.floor(visibleRectangle.height / 2);
                    attributes.anchorRight = colCollection.getScrollAnchorByOffset(visibleRectangle.left + splitWidth, { pixel: true });
                    attributes.anchorBottom = rowCollection.getScrollAnchorByOffset(visibleRectangle.top + splitHeight, { pixel: true });
                    attributes.activePane = 'topLeft';
                } else {
                    if (splitWidth > 0) { attributes.anchorRight = address[0]; }
                    if (splitHeight > 0) { attributes.anchorBottom = address[1]; }
                }

                // activate the dynamic split view
                splitWidth = activeSheetModel.convertPixelToHmm(splitWidth);
                splitHeight = activeSheetModel.convertPixelToHmm(splitHeight);
                activeSheetModel.setDynamicSplit(splitWidth, splitHeight, attributes);
            }

            return this;
        };

        /**
         * Toggles the frozen view split mode according to the current view
         * split mode, and the current selection. Enabling frozen split mode
         * in a dynamic split view will freeze the panes with their current
         * size (split mode 'frozenSplit'). Otherwise, enabling the frozen
         * split mode will split the view at the position left of and above the
         * active cell. Disabling the frozen split mode will either return to
         * dynamic split mode (if split mode was 'frozenSplit'), or will remove
         * the frozen split at all (if split mode was 'frozen').
         *
         * @param {Boolean} state
         *  The new state of the frozen view split mode.
         *
         * @returns {ViewFuncMixin}
         *  A reference to this instance.
         */
        this.setFrozenSplit = function (state) {

            var // whether the view is currently split
                hasSplit = this.hasSplit(),
                // address of the active cell
                address = this.getActiveCell();

            // convert frozen split to dynamic split
            if (!state && hasSplit && (this.getSplitMode() === 'frozenSplit')) {
                activeSheetModel.setDynamicSplit(activeSheetModel.getSplitWidthHmm(), activeSheetModel.getSplitHeightHmm());

            // disable split mode completely
            } else if (!state && hasSplit) {
                activeSheetModel.clearSplit();

            // convert dynamic split to frozen split
            } else if (state && hasSplit && !this.hasFrozenSplit()) {
                activeSheetModel.setFrozenSplit(activeSheetModel.getSplitColInterval(), activeSheetModel.getSplitRowInterval());

            // enable frozen split mode
            } else if (state && !hasSplit) {

                var // calculate frozen column interval (must result in at least one frozen column)
                    colAnchor = activeSheetModel.getViewAttribute('anchorRight'),
                    colInterval = new Interval(Math.round(colAnchor), address[0] - 1);
                if (colInterval.first > colInterval.last) { colInterval = null; }

                var // calculate frozen row interval (must result in at least one frozen row)
                    rowAnchor = activeSheetModel.getViewAttribute('anchorBottom'),
                    rowInterval = new Interval(Math.round(rowAnchor), address[1] - 1);
                if (rowInterval.first > rowInterval.last) { rowInterval = null; }

                // activate the frozen split view
                activeSheetModel.setFrozenSplit(colInterval, rowInterval);
            }

            return this;
        };

        /**
         * Freezes the passed number of columns and rows in the active sheet,
         * regardless of the current split settings.
         *
         * @param {Number} cols
         *  The number of columns to freeze. The value zero will not freeze any
         *  column.
         *
         * @param {Number} rows
         *  The number of rows to freeze. The value zero will not freeze any
         *  row.
         *
         * @returns {ViewFuncMixin}
         *  A reference to this instance.
         */
        this.setFixedFrozenSplit = function (cols, rows) {

            var // the scroll anchors for the first visible column/row
                colAnchor = activeSheetModel.getViewAttribute(activeSheetModel.hasColSplit() ? 'anchorLeft' : 'anchorRight'),
                rowAnchor = activeSheetModel.getViewAttribute(activeSheetModel.hasRowSplit() ? 'anchorTop' : 'anchorBottom'),
                // the first visible column and row in the split view (reduce to be able to freeze the passed number of columns/rows)
                col = Math.max(0, Math.min(Math.round(colAnchor), docModel.getMaxCol() - cols + 1)),
                row = Math.max(0, Math.min(Math.round(rowAnchor), docModel.getMaxRow() - rows + 1));

            // activate the frozen split view
            activeSheetModel.setFrozenSplit((cols > 0) ? new Interval(col, col + cols - 1) : null, (rows > 0) ? new Interval(row, row + rows - 1) : null);

            return this;
        };

        // column operations --------------------------------------------------

        /**
         * Returns the mixed formatting attributes of all columns in the active
         * sheet covered by the current selection.
         *
         * @returns {Object}
         *  A complete attribute set containing the mixed attributes of all
         *  selected columns. All attributes that could not be resolved
         *  unambiguously will be set to the value null.
         */
        this.getColumnAttributes = function () {
            var intervals = getSelectedIntervals(true);
            return colCollection.getMixedAttributes(intervals);
        };

        /**
         * Returns the column attributes of the column the active cell is
         * currently located in.
         *
         * @returns {Object}
         *  The column attribute map of the active cell.
         */
        this.getActiveColumnAttributes = function () {
            var activeCol = this.getActiveCell()[0];
            // return the column attributes only
            return colCollection.getEntry(activeCol).attributes.column;
        };

        /**
         * Returns whether additional columns can be inserted into the sheet,
         * based on the current selection.
         *
         * @returns {Boolean}
         *  Whether additional columns can be inserted, according to the
         *  current selection and the number of used columns in the sheet.
         */
        this.canInsertColumns = function () {
            var intervals = getSelectedIntervals(true);
            return canInsertIntervals(intervals, true);
        };

        /**
         * Returns whether the selected columns can be deleted from the sheet.
         *
         * @returns {Boolean}
         *  Whether the selected columns can be deleted.
         */
        this.canDeleteColumns = function () {
            var intervals = getSelectedIntervals(true);
            return canDeleteIntervals(intervals, true);
        };

        /**
         * Returns whether the current selection contains hidden columns, or is
         * located next to hidden columns, that can be made visible.
         *
         * @returns {Boolean}
         *  Whether the current selection contains hidden columns.
         */
        this.canShowColumns = function () {
            return getSelectedHiddenIntervals(true).length > 0;
        };

        /**
         * Inserts new columns into the active sheet, according to the current
         * selection.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected on any error.
         */
        this.insertColumns = function () {
            var intervals = getSelectedIntervals(true);
            return this.yellOnPromise(activeSheetModel.insertColumns(intervals));
        };

        /**
         * Deletes existing columns from the active sheet, according to the
         * current selection.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected on any error.
         */
        this.deleteColumns = function () {
            var intervals = getSelectedIntervals(true);
            return this.yellOnPromise(activeSheetModel.deleteColumns(intervals));
        };

        /**
         * Changes the specified column attributes in the active sheet,
         * according to the current selection.
         *
         * @param {Object} attributes
         *  The (incomplete) column attributes, as simple object (NOT mapped as
         *  a 'column' sub-object).
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected on any error.
         */
        this.changeColumns = function (attributes) {
            var intervals = getSelectedIntervals(true);
            intervals.forEach(function (interval) { interval.attrs = attributes; });
            return this.yellOnPromise(activeSheetModel.changeColumns(intervals));
        };

        /**
         * Shows all hidden columns in the current selection.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected on any error.
         */
        this.showColumns = function () {
            var intervals = getSelectedHiddenIntervals(true);
            intervals.forEach(function (interval) { interval.attrs = { visible: true }; });
            return this.yellOnPromise(activeSheetModel.changeColumns(intervals));
        };

        /**
         * Changes the width of columns in the active sheet, according to the
         * current selection.
         *
         * @param {Number} width
         *  The column width in 1/100 mm. If this value is less than 1, the
         *  columns will be hidden, and their original width will not be
         *  changed.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Number} [options.target]
         *      The target column to be changed. If specified, the column width
         *      will be set to that column only. If the current selection
         *      contains any ranges covering this column completely, the column
         *      width will be set to all columns contained in these ranges.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected on any error.
         */
        this.setColumnWidth = function (width, options) {

            var // the target column to be modified
                targetCol = Utils.getIntegerOption(options, 'target'),
                // hide columns if passed column width is zero; show hidden columns when width is changed
                attributes = (width === 0) ? { visible: false } : { visible: true, width: width, customWidth: true },
                // the column intervals to be modified
                intervals = getSelectedIntervals(true, { target: targetCol });

            intervals.forEach(function (interval) { interval.attrs = attributes; });
            return this.yellOnPromise(activeSheetModel.changeColumns(intervals));
        };

        /**
         * Sets optimal column width based on the content of cells.
         *
         * @param {Number} [targetCol]
         *  The target column to be changed. If the current selection contains
         *  any ranges covering this column completely, the optimal column
         *  width will be set to all columns contained in these ranges.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected on any error.
         */
        this.setOptimalColumnWidth = function (targetCol) {
            var intervals = getSelectedIntervals(true, { target: targetCol, visible: true });
            return this.yellOnPromise(setOptimalColumnWidth(intervals));
        };

        // row operations -----------------------------------------------------

        /**
         * Returns the mixed formatting attributes of all rows in the active
         * sheet covered by the current selection.
         *
         * @returns {Object}
         *  A complete attribute set containing the mixed attributes of all
         *  selected rows. All attributes that could not be resolved
         *  unambiguously will be set to the value null.
         */
        this.getRowAttributes = function () {
            var intervals = getSelectedIntervals(false);
            return rowCollection.getMixedAttributes(intervals);
        };

        /**
         * Returns the row attributes of the row the active cell is currently
         * located in.
         *
         * @returns {Object}
         *  The row attribute map of the active cell.
         */
        this.getActiveRowAttributes = function () {
            var activeRow = this.getActiveCell()[1];
            // return the row attributes only
            return rowCollection.getEntry(activeRow).attributes.row;
        };

        /**
         * Returns whether additional rows scan be inserted into the sheet,
         * based on the current selection.
         *
         * @returns {Boolean}
         *  Whether additional rows can be inserted, according to the current
         *  selection and the number of used rows in the sheet.
         */
        this.canInsertRows = function () {
            var intervals = getSelectedIntervals(false);
            return canInsertIntervals(intervals, false);
        };

        /**
         * Returns whether the selected rows can be deleted from the sheet.
         *
         * @returns {Boolean}
         *  Whether the selected rows can be deleted.
         */
        this.canDeleteRows = function () {

            // bug 39869: prevent deleting header row, or all of the data rows, of a table range
            var intervals = getSelectedIntervals(false);
            return canDeleteIntervals(intervals, false) && tableCollection.getAllTables().every(function (tableModel) {

                // any row in an auto filter can be deleted
                if (tableModel.isAutoFilter()) { return true; }

                // header rows cannot be deleted
                var tableRange = tableModel.getRange();
                if (tableModel.hasHeaderRow() && intervals.containsIndex(tableRange.start[1])) {
                    return false;
                }

                // at least one data row must remain in the table
                var dataRange = tableModel.getDataRange();
                if (dataRange && IntervalArray.get(dataRange.rowInterval()).difference(intervals).empty()) {
                    return false;
                }

                return true;
            });
        };

        /**
         * Returns whether the current selection contains hidden rows, or is
         * located next to hidden rows, that can be made visible.
         *
         * @returns {Boolean}
         *  Whether the current selection contains hidden rows.
         */
        this.canShowRows = function () {
            return getSelectedHiddenIntervals(false).length > 0;
        };

        /**
         * Inserts new rows into the active sheet, according to the current
         * selection.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected on any error.
         */
        this.insertRows = function () {
            var intervals = getSelectedIntervals(false);
            return this.yellOnPromise(activeSheetModel.insertRows(intervals));
        };

        /**
         * Deletes existing rows from the active sheet, according to the
         * current selection.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected on any error.
         */
        this.deleteRows = function () {
            var intervals = getSelectedIntervals(false);
            return this.yellOnPromise(activeSheetModel.deleteRows(intervals));
        };

        /**
         * Changes the specified row attributes in the active sheet, according
         * to the current selection..
         *
         * @param {Object} attributes
         *  The (incomplete) row attributes, as simple object (NOT mapped as a
         *  'row' sub-object).
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected on any error.
         */
        this.changeRows = function (attributes) {
            var intervals = getSelectedIntervals(false);
            intervals.forEach(function (interval) { interval.attrs = attributes; });
            return this.yellOnPromise(activeSheetModel.changeRows(intervals));
        };

        /**
         * Shows all hidden rows in the current selection.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected on any error.
         */
        this.showRows = function () {
            var intervals = getSelectedHiddenIntervals(false);
            intervals.forEach(function (interval) { interval.attrs = { visible: true }; });
            return this.yellOnPromise(activeSheetModel.changeRows(intervals));
        };

        /**
         * Changes the height of rows in the active sheet, according to the
         * current selection.
         *
         * @param {Number} height
         *  The row height in 1/100 mm. If this value is less than 1, the rows
         *  will be hidden, and their original height will not be changed.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Number} [options.target]
         *      The target row to be changed. If specified, the row height will
         *      be set to that row only. If the current selection contains any
         *      ranges covering this row completely, the row height will be set
         *      to all rows contained in these ranges.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected on any error.
         */
        this.setRowHeight = function (height, options) {

            var // the target row to be modified
                targetRow = Utils.getIntegerOption(options, 'target'),
                // hide rows if passed row height is zero; show hidden rows when height is changed
                attributes = (height === 0) ? { visible: false } : { visible: true, height: height, customHeight: true },
                // the row intervals to be modified
                intervals = getSelectedIntervals(false, { target: targetRow });

            intervals.forEach(function (interval) { interval.attrs = attributes; });
            return this.yellOnPromise(activeSheetModel.changeRows(intervals));
        };

        /**
         * Sets optimal row height based on the content of cells.
         *
         * @param {Number} [targetRow]
         *  The target row to be changed. If the current selection contains any
         *  ranges covering this row completely, the optimal row height will be
         *  set to all rows contained in these ranges.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected on any error.
         */
        this.setOptimalRowHeight = function (targetRow) {
            var intervals = getSelectedIntervals(false, { target: targetRow, visible: true });
            return this.yellOnPromise(setOptimalRowHeight(intervals));
        };

        // cell operations ----------------------------------------------------

        /**
         * Returns the contents and formatting of the current active cell.
         *
         * @returns {CellDescriptor}
         *  The contents and formatting of the current active cell.
         */
        this.getActiveCellEntry = function () {
            return activeCellDesc;
        };

        /**
         * Returns the merged formatting attributes of the current active cell.
         *
         * @returns {Object}
         *  All formatting attributes of the active cell, mapped by cell
         *  attribute family ('cell' and 'character'), and the identifier of
         *  the cell style sheet in the 'styleId' string property. If the cell
         *  in-place edit mode is currently active, the edit attributes added
         *  there will be included too.
         */
        this.getCellAttributes = function () {

            var // the merged attributes of the active cell
                cellAttributes = _.copy(activeCellDesc.attributes, true),
                // additional formatting attributes from cell edit mode
                editAttributes = this.getCellEditAttributes();

            if (_.isObject(editAttributes)) {
                _.extend(cellAttributes.cell, editAttributes.cell);
                _.extend(cellAttributes.character, editAttributes.character);
            }

            return cellAttributes;
        };

        /**
         * Returns the URL of a hyperlink at the current active cell.
         *
         * @returns {String|Null}
         *  The URL of a hyperlink attached to the active cell; or null, if the
         *  active cell does not contain a hyperlink.
         */
        this.getCellURL = function () {
            return cellCollection.getCellURL(this.getActiveCell());
        };

        /**
         * Sets the contents of the active cell, and updates the view according
         * to the new cell value and formatting.
         *
         * @param {String} [value]
         *  The new value for the active cell. If omitted, only the cell
         *  formatting will be changed.
         *
         * @param {Object} [attributes]
         *  A cell attribute set, optionally containing the string property
         *  'styleId', and attribute value maps (name/value pairs), keyed by
         *  the supported attribute families 'cell' and 'character'. Omitted
         *  attributes will not be changed. Attributes set to the value null
         *  will be removed from the cell.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.parse=false]
         *      If set to true, the 'parse' property will be inserted into the
         *      operation, set to the name of the current UI language.
         *  @param {Boolean} [options.clear=false]
         *      If set to true, all existing explicit attributes will be
         *      removed from the cell while applying the new attributes.
         *  @param {Any} [options.result]
         *      The result of a formula, as calculated by the local formula
         *      interpreter.
         *  @param {String} [options.url]
         *      The URL of a hyperlink to be set at the cell. If set to the
         *      empty string, an existing hyperlink will be removed.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected on any error.
         */
        this.setCellContents = function (value, attributes, options) {

            // store character attributes while in-place cell edit mode is active
            if (this.isCellEditMode()) {
                this.setCellEditAttributes(attributes);
                return $.when();
            }

            // do nothing if active cell is locked
            if (!this.requireEditableActiveCell({ lockTables: _.isUndefined(value) ? 'none' : 'header' })) {
                return $.Deferred().reject();
            }

            // apply operations, show error alert on failure
            var promise = setSingleCellContents(this.getActiveCell(), value, attributes, options);
            return this.yellOnPromise(promise);
        };

        /**
         * Fills all cell ranges in the current selection with the same value
         * and formatting, and updates the view according to the new cell value
         * and formatting.
         *
         * @param {String} [value]
         *  The new value for all cells in the current selection. If omitted,
         *  only the cell formatting will be changed.
         *
         * @param {Object} [attributes]
         *  A cell attribute set, optionally containing the string property
         *  'styleId', and attribute value maps (name/value pairs), keyed by
         *  the supported attribute families 'cell' and 'character'. Omitted
         *  attributes will not be changed. Attributes set to the value null
         *  will be removed from the cell.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.parse=false]
         *      If set to true, the 'parse' property will be inserted into the
         *      operations, set to the name of the current UI language.
         *  @param {Boolean} [options.clear=false]
         *      If set to true, all existing explicit attributes will be
         *      removed from the selection while applying the new attributes.
         *  @param {String} [options.url]
         *      The URL of a hyperlink to be set at the cell. If set to the
         *      empty string, an existing hyperlink will be removed.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected on any error.
         */
        this.fillCellRanges = function (value, attributes, options) {

            // store character attributes while in-place cell edit mode is active
            if (this.isCellEditMode()) {
                this.setCellEditAttributes(attributes);
                return $.when();
            }

            // do nothing if current selection contains locked cells
            if (!this.requireEditableSelection({ lockTables: _.isUndefined(value) ? 'none' : 'header' })) {
                return $.Deferred().reject();
            }

            // apply operations, show error alert on failure
            options = _.extend({}, options, { ref: this.getActiveCell() });
            var promise = fillCellRanges(this.getSelectedRanges(), value, attributes, options);
            return this.yellOnPromise(promise);
        };

        /**
         * Deletes the values and formatting attributes from all cell ranges in
         * the current selection.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected on any error.
         */
        this.clearCellRanges = function () {

            // do nothing if current selection contains locked cells
            if (!this.requireEditableSelection({ lockTables: 'header' })) {
                return $.Deferred().reject();
            }

            // apply operations, show error alert on failure
            var promise = clearCellRanges(this.getSelectedRanges());
            return this.yellOnPromise(promise);
        };

        /**
         * Changes a single attribute of type 'cell' for all cells in the
         * current selection.
         *
         * @param {String} name
         *  The name of the cell attribute.
         *
         * @param {Any} value
         *  The new value of the cell attribute. If set to the value null, all
         *  attributes set explicitly will be removed from the cells.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected on any error.
         */
        this.setCellAttribute = function (name, value) {
            return this.fillCellRanges(undefined, { cell: Utils.makeSimpleObject(name, value) });
        };

        /**
         * Changes a single attribute of type 'character' for all cells in the
         * current selection.
         *
         * @param {String} name
         *  The name of the character attribute.
         *
         * @param {Any} value
         *  The new value of the character attribute. If set to the value null,
         *  the current explicit attribute will be removed from the cells.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected on any error.
         */
        this.setCharacterAttribute = function (name, value) {
            return this.fillCellRanges(undefined, { character: Utils.makeSimpleObject(name, value) });
        };

        /**
         * removes all explicit formatting attributes from all cells in the
         * current selection.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected on any error.
         */
        this.clearAttributes = function () {
            return this.fillCellRanges(undefined, undefined, { clear: true });
        };

        /**
         * Returns the number format category of the active cell.
         *
         * @returns {String}
         *  The current number format category.
         */
        this.getNumberFormatCategory = function () {
            return activeCellDesc.format.category;
        };

        /**
         * Sets selected number format category for the current active cell in
         * the current selection, and implicitly set the first format code of
         * the predefined template of this category to the cell.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected on any error.
         */
        this.setNumberFormatCategory = function (category) {

            var formatCode = numberFormatter.getCategoryDefaultCode(category);
            if ((category === 'custom') || !_.isString(formatCode)) { return $.when(); }

            return this.fillCellRanges(undefined, { cell: { numberFormat: numberFormatter.createNumberFormat(formatCode) } });
        };

        /**
         * Returns the format code of the active cell.
         *
         * @returns {String}
         *  The current number format code.
         */
        this.getNumberFormatCode = function () {
            return numberFormatter.resolveFormatCode(activeCellDesc.attributes.cell.numberFormat);
        };

        /**
         * Returns the format code of the active cell.
         *
         * @param {String} formatCode
         *  The number format code to be set to the current selection.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected on any error.
         */
        this.setNumberFormatCode = function (formatCode) {

            // do nothing if custom fall-back button is clicked
            if (formatCode === Labels.CUSTOM_FORMAT_VALUE) { return $.when(); }

            return this.setCellAttribute('numberFormat', numberFormatter.createNumberFormat(formatCode));
        };

        /**
         * Returns the border mode of the current selection.
         *
         * @returns {Object}
         *  The border mode representing the visibility of the borders in all
         *  ranges of the current selection. See 'MixedBorder.getBorderMode()'
         *  for more details.
         */
        this.getBorderMode = function () {
            return MixedBorder.getBorderMode(selectionSettings.borders);
        };

        /**
         * Changes the visibility of the borders in all ranges of the current
         * selection.
         *
         * @param {Object} borderMode
         *  The border mode representing the visibility of the borders in all
         *  ranges of the current selection. See 'MixedBorder.getBorderMode()'
         *  for more details.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected on any error.
         */
        this.setBorderMode = function (borderMode) {

            // quit if current selection contains locked cells
            if (!this.requireEditableSelection()) { return $.Deferred().reject(); }

            var // the border attributes to be sent
                borderAttributes = MixedBorder.getBorderAttributes(borderMode, selectionSettings.borders, DEFAULT_SINGLE_BORDER),
                // create and apply the operations
                promise = activeSheetModel.setBorderAttributes(this.getSelectedRanges(), borderAttributes);

            // bug 34021: immediately update all selection settings to prevent flickering UI elements
            promise.always(function () { updateSelectionSettings({ direct: true }); });

            // show warning messages
            return this.yellOnPromise(promise);
        };

        /**
         * Returns the merged border attributes of the current selection.
         *
         * @returns {Object}
         *  An attribute map containing the merged borders of all available
         *  border attributes ('borderTop', 'borderLeft', etc.) in the current
         *  selection. Each border attribute value contains the properties
         *  'style', 'width', and 'color'. If the border property is equal in
         *  all affected cells, it will be set as property value in the border
         *  object, otherwise the property will be set to the value null.
         */
        this.getBorderAttributes = function () {
            return _.copy(selectionSettings.borders, true);
        };

        /**
         * Changes the specified border properties (line style, line color, or
         * line width) of all visible cell borders in the current selection.
         *
         * @param {Border} border
         *  A border attribute, may be incomplete. All properties contained in
         *  this object will be set for all visible borders in the current
         *  selection. Omitted border properties will not be changed.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected on any error.
         */
        this.changeVisibleBorders = function (border) {

            // quit if current selection contains locked cells
            if (!this.requireEditableSelection()) { return $.Deferred().reject(); }

            // create and apply the operations
            var promise = activeSheetModel.changeVisibleBorders(this.getSelectedRanges(), border);

            // bug 34021: immediately update all selection settings to prevent flickering UI elements
            promise.always(function () { updateSelectionSettings({ direct: true }); });

            // show warning messages
            return this.yellOnPromise(promise);
        };

        /**
         * Returns whether the format painter is currently active.
         *
         * @returns {Boolean}
         *  Whether the format painter is currently active.
         */
        this.isFormatPainterActive = function () {
            return this.isCustomSelectionMode('painter');
        };

        /**
         * Activates or deactivates the format painter.
         *
         * @param {Boolean} state
         *  Whether to activate or deactivate the format painter.
         *
         * @returns {Boolean}
         *  Whether the format painter has been activated.
         */
        this.activateFormatPainter = function (state) {

            // deactivate format painter: cancel custom selection mode
            if (!state) {
                this.cancelCustomSelectionMode();
                return false;
            }

            var // paste only the auto style into the range that will be selected
                attrs = { styleId: activeCellDesc.style },
                // the promise representing custom selection mode
                promise = null;

            // do not lock editable cells by copying the 'unlocked' attribute (in locked sheets only, as in Excel)
            if (this.isSheetLocked()) {
                attrs.cell = { unlocked: true };
            }

            // start custom selection mode (wait for range selection)
            promise = this.enterCustomSelectionMode('painter', {
                selection: this.getActiveCellAsSelection(),
                statusLabel: Labels.FORMAT_PAINTER_LABEL
            });

            // select the target range, bug 35295: check that the cells are not locked
            promise = promise.then(function (selection) {
                self.setCellSelection(selection);
                return self.areRangesEditable(selection.ranges);
            });

            // copy the formatting attributes to the target range
            promise.done(function () {
                self.executeControllerItem('cell/attributes', attrs);
            });

            // show error alert if copying attributes has failed
            return true;
        };

        /**
         * Fills or clears a cell range according to the position and contents
         * of the cell range currently selected (the 'auto-fill' feature).
         *
         * @param {String} border
         *  Which border of the selected range will be modified by the
         *  operation. Allowed values are 'left', 'right', 'top', or 'bottom'.
         *
         * @param {Number} count
         *  The number of columns/rows to extend the selected range with
         *  (positive values), or to shrink the selected range (negative
         *  values).
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the auto-fill target range is
         *  editable, and the auto-fill operation has been applied. Otherwise,
         *  the promise will be rejected.
         */
        this.autoFill = function (border, count) {

            var // all cell ranges selected in the sheet
                ranges = this.getSelectedRanges(),
                // whether to expand/shrink the leading or trailing border
                leading = /^(left|top)$/.test(border),
                // whether to expand/delete columns or rows
                columns = /^(left|right)$/.test(border),
                // the target range for auto-fill
                targetRange = ranges.first().clone(),
                // the promise of the entire auto-fill operations
                promise = null;

            // for safety: do nothing if called in a multi-selection
            if (ranges.length !== 1) {
                // no error code, this should not happen as the GUI does not offer auto-fill
                promise = $.Deferred().reject();

            // fill cells outside the selected range
            } else if (count > 0) {

                // build the target range (adjacent to the selected range)
                if (leading) {
                    targetRange.setStart(ranges.first().getStart(columns) - count, columns);
                    targetRange.setEnd(ranges.first().getStart(columns) - 1, columns);
                } else {
                    targetRange.setStart(ranges.first().getEnd(columns) + 1, columns);
                    targetRange.setEnd(ranges.first().getEnd(columns) + count, columns);
                }

                // only the target range must be editable (not the source cells)
                promise = this.areRangesEditable(targetRange, { lockTables: 'header' });

                // apply auto-fill, and update optimal row heights
                promise = promise.then(function () {
                    return undoManager.enterUndoGroup(function () { // bug 36773: collect all changes into an undo group
                        return activeSheetModel.autoFill(ranges.first(), border, count).then(function () {
                            return updateOptimalRowHeight(targetRange);
                        });
                    });
                });

                // expand the selected range
                promise.always(function () {
                    self.changeActiveRange(ranges.first().boundary(targetRange));
                });

            // delete cells inside the selected range
            } else if (count < 0) {

                // build the target range (part of the selected range, may be the entire selected range)
                if (leading) {
                    targetRange.setEnd(targetRange.getStart(columns) - count - 1, columns);
                } else {
                    targetRange.setStart(targetRange.getEnd(columns) + count + 1, columns);
                }

                // the target range must be editable (not the entire selected range)
                promise = this.areRangesEditable(targetRange, { lockTables: 'header' });

                // clear the cell range, and update optimal row heights
                promise = promise.then(function () {
                    return clearCellRanges(targetRange);
                });

                // update selection
                promise.always(function () {
                    // do not change selection after deleting the entire range
                    if (ranges.first().equals(targetRange)) { return; }
                    // select the unmodified part of the original range
                    targetRange = ranges.first().clone();
                    if (leading) {
                        targetRange.start.move(-count, columns);
                    } else {
                        targetRange.end.move(count, columns);
                    }
                    self.changeActiveRange(targetRange);
                });

            // do nothing if no cells will be filled or cleared
            } else {
                promise = $.when();
            }

            return this.yellOnPromise(promise);
        };

        /**
         * Inserts one or more formulas calculating subtotal results into or
         * next to the current selection.
         *
         * @param {String} funcKey
         *  The resource key of the subtotal function to be inserted into the
         *  formulas.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the server request used to
         *  query the subtotal formulas has returned.
         */
        this.insertAutoFormula = function (funcKey) {

            var // the current selection
                selection = this.getSelection(),
                // the selected range
                range = selection.activeRange(),
                // whether the range consists of a single column
                singleCol = range.singleCol(),
                // whether the range consists of a single row
                singleRow = range.singleRow(),
                // whether the range consists of a single cell (also single merged range)
                singleCell = this.isSingleCellSelection(),
                // the translated upper-case function name
                funcName = grammarConfig.getFunctionName(funcKey);

            // returns the correct formula expression for the specified cell range
            function buildFormula(sourceRange) {
                return '=' + funcName + '(' + sourceRange + ')';
            }

            // returns a promise that resolves if all cells in the passed range are editable
            function ensureEditable(editRange) {
                return self.yellOnPromise(self.areRangesEditable(editRange, { lockTables: 'header' }));
            }

            // sends a server request and returns the promise that resolves with the cell content array
            function sendFormulaRequest() {
                Utils.info('ViewFuncMixin.insertAutoFormula(): requesting formulas for range ' + range);

                var // the request data sent to the server
                    requestData = {
                        func: funcName,
                        sheet: activeSheet,
                        start: range.start.toJSON(),
                        end: range.end.toJSON(),
                        ref: selection.address.toJSON()
                    };

                return app.sendQueryRequest('autoFormula', requestData, {
                    resultFilter: function (data) {
                        return Utils.getArrayOption(data, 'contents', undefined, true);
                    }
                })
                .done(function (response) {
                    Utils.info('ViewFuncMixin.insertAutoFormula(): request succeeded:', response);
                })
                .fail(function (response) {
                    Utils.error('ViewFuncMixin.insertAutoFormula(): request failed:', response);
                });
            }

            // selects the specified range, and keeps the current active cell
            function selectRange(start, end) {
                self.selectRange(new Range(start, end), { active: selection.address });
            }

            // passed value must be the native name of a sheet function
            if (!funcName) {
                Utils.error('ViewFuncMixin.insertAutoFormula(): invalid function key: "' + funcKey + '"');
                return $.Deferred().reject();
            }

            // multiple selection not supported
            if (selection.ranges.length !== 1) {
                return $.Deferred().reject();
            }

            // single cell is selected (also a single merged range)
            if (singleCell) {

                // query CalcEngine for a formula string and start in-place edit mode
                return sendFormulaRequest().done(function (contents) {
                    self.enterCellEditMode({ text: contents[0].formula, pos: -1 });
                });
            }

            // multiple columns and rows are selected: fill formulas into the row below the range
            if (!singleCol && !singleRow) {

                // there must be space below the range
                if (range.end[1] >= docModel.getMaxRow()) { return $.Deferred().reject(); }

                var // fill formulas into the cells below the selected range
                    targetRange = Range.create(range.start[0], range.end[1] + 1, range.end[0], range.end[1] + 1);

                // check that the range is completely unlocked, fill it with the formula
                return ensureEditable(targetRange).then(function () {

                    var sourceRange = range.clone();
                    sourceRange.end[0] = range.start[0];

                    // apply the operation, change selection, show error alerts on failure
                    var promise = fillCellRanges(targetRange, buildFormula(sourceRange), undefined, { ref: targetRange.start, parse: true });
                    promise.always(function () { selectRange(range.start, targetRange.end); });
                    return self.yellOnPromise(promise);
                });
            }

            // single column or single row (with multiple cells): send server request if range is empty
            if (cellCollection.areRangesBlank(range)) {

                // check cell count first
                if (range.cells() > SheetUtils.MAX_FILL_CELL_COUNT) {
                    return this.yellOnPromise(SheetUtils.makeRejected('cells:overflow'));
                }

                // send server request, fill cells with one operation
                return sendFormulaRequest().then(function (contents) {

                    // convert the contents array according to the 'setCellContents' operation
                    contents = contents.map(function (json) {
                        var result = { value: json.formula };
                        if ('attrs' in json) { result.attrs = json.attrs; }
                        // put every array element into a single-element array for column selection
                        return singleCol ? [result] : result;
                    });

                    // apply the operation (changing selection not necessary), show error alerts on failure
                    var promise = setCellContents(range.start, singleRow ? [contents] : contents, { parse: true });
                    return self.yellOnPromise(promise);
                });
            }

            // range contains any values: insert single formula into first empty cell
            return (function () {

                var // the cell address used to find the first empty cell
                    address = range.end.clone(),
                    // whether to operate on different columns in a single row
                    columns = singleRow,
                    // maximum index in the cell address (restrict loop to 1000 iterations)
                    maxIndex = Math.min(docModel.getMaxIndex(columns), address.get(columns) + 1000);

                // JSHint requires to define the function outside the while loop
                function createFormulaCell() {

                    var sourceRange = range.clone();
                    sourceRange.setEnd(Math.min(address.get(columns) - 1, range.getEnd(columns)), columns);

                    // apply the operation, show error alerts on failure
                    var promise = setSingleCellContents(address, buildFormula(sourceRange), undefined, { parse: true }).always(function () {
                        selectRange(range.start, address);
                    });
                    return self.yellOnPromise(promise);
                }

                // find the first empty cell, starting at the last cell in the selected range
                while (address.get(columns) <= maxIndex) {
                    if (cellCollection.isBlankCell(address)) {
                        return ensureEditable(new Range(address)).then(createFormulaCell);
                    }
                    address.move(1, columns);
                }

                return $.Deferred().reject();
            }());
        };

        /**
         * Shows the 'Insert Function' dialog, and starts the cell in-place
         * edit mode with a formula containing the function selected in the
         * dialog.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved after a function has been selected
         *  in the dialog, and the cell edit mode has been started; or that
         *  will be rejected after canceling the dialog.
         */
        this.insertFunction = function () {

            // active cell must be editable
            if (!this.requireEditableActiveCell({ lockTables: 'header' })) {
                return $.Deferred().reject();
            }

            var cellDesc = this.getActiveCellEntry(),
                value = cellDesc.formula || cellDesc.value;

            if (_.isString(value)) {
                var result = /\=?\s*([^(]+)/.exec(value);
                value = result ? result[1] : null;
            }

            // show the dialog, wait for the result
            var promise = new Dialogs.FunctionDialog(this, value).show();

            // start cell edit mode, insert the selected function as formula into the active cell
            return promise.done(function (funcKey) {
                var formula = '=' + grammarConfig.getFunctionName(funcKey) + '()';
                self.enterCellEditMode({ text: formula, pos: -1 });
            });
        };

        /**
         * Merges or unmerges all cells of the current selection.
         *
         * @param {String} type
         *  The merge type. Must be one of:
         *  - 'merge': Merges entire ranges.
         *  - 'horizontal': Merges the single rows in all ranges.
         *  - 'vertical': Merges the single columns in all ranges.
         *  - 'unmerge': Removes all merged ranges covered by the selection.
         *  - 'toggle': Will be mapped to 'merge', if the selection does not
         *      cover any merged ranges, otherwise will be mapped to 'unmerge'.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved after the ranges have been merged
         *  or unmerged successfully, or that will be rejected on any error.
         */
        this.mergeRanges = function (type) {

            // the selected ranges
            var ranges = this.getSelectedRanges();

            // quit if current selection contains locked cells
            // bug 39869: do not allow to merge any cells in a table range (except auto-filter)
            if (!this.requireEditableSelection({ lockTables: 'full' })) {
                return $.Deferred().reject();
            }

            // toggle merge state if specified
            if (type === 'toggle') {
                type = mergeCollection.rangesOverlapAnyMergedRange(ranges) ? 'unmerge' : 'merge';
            }

            // send the 'mergeCells' operations
            return this.yellOnPromise(mergeCollection.mergeRanges(ranges, type));
        };

        // search operations --------------------------------------------------

        /**
         * Implementation for search/replace operations as triggered by the
         * base class EditView. See the description of the constructor
         * parameter 'searchHandler' in the class EditView for more details.
         *
         * @param {String} command
         *  The search command to be executed.
         *
         * @param {Object} settings
         *  The current search settings.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved after executing the specified
         *  command successfully, or rejected on any error.
         */
        this.executeSearchCommand = function (command, settings) {
            switch (command) {
                case 'search:start':
                    return self.searchNext('next', settings.query);
                case 'search:next':
                    return self.searchNext('next', settings.query);
                case 'search:prev':
                    return self.searchNext('prev', settings.query);
                case 'search:end':
                    //no special highlighting, so we have no special behavior here
                    return $.when();
                case 'replace:next':
                    return self.searchReplace(settings.query, settings.replace);
                case 'replace:all':
                    return self.searchReplace(settings.query, settings.replace, { all: true });
                default:
                    Utils.error('ViewFuncMixin.executeSearchCommand(): unsupported search command "' + command + '"');
                    return $.Deferred().reject('internal');
            }
        };

        /**
         * Iterates through the current search results.
         *
         * @param {String} direction
         *  The search direction, either 'next', or 'prev'.
         *
         * @param {String} [query]
         *  The current string to be searched in the active sheet, as contained
         *  in the respective GUI element. If this value is different from the
         *  initial query string, a new search request will be sent to the
         *  server. If omitted, the current query string will be used again.
         *
         * @returns {ViewFuncMixin}
         *  A reference to this instance.
         */
        this.searchNext = function (direction, query) {

            if (!_.isString(query)) {
                Utils.error('ViewFuncMixin.searchNext(): no query assigned');
                return this;
            }
            var // the addresses of all cells containing the current search text
                addresses = getSearchResults(query);

            // scroll to and select the next cell of the search results
            if (addresses) {
                switch (direction) {
                    case 'prev':
                        selectSearchResult((searchSettings.index === 0) ? (addresses.length - 1) : (searchSettings.index - 1));
                        break;
                    case 'next':
                        selectSearchResult((searchSettings.index + 1) % addresses.length);
                        break;
                }
            }

            return this;
        };

        /**
         * Replaces the value of the current (single) selection with the
         * replacement value.
         *
         * @param {String} query
         *  The current string to be searched in the active sheet, as contained
         *  in the respective GUI element. If this value is different from the
         *  initial query string, a new search request will be sent to the
         *  server.
         *
         * @param {String} replace
         *  The replacement text.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.all=false]
         *      If set to true, all occurrences of the search text in the
         *      active sheet will be replaced.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved after the search string has been
         *  replaced, or that will be rejected on any error.
         */
        this.searchReplace = function (query, replace, options) {

            // no replace functionality in locked cells (TODO: replace in unprotected cells?)
            if (!this.requireUnlockedActiveSheet()) {
                return $.Deferred().reject();
            }

            // do nothing and return immediately if search query is empty
            if (query === '') {
                this.yell({ type: 'info', message: gt('Nothing to replace.') });
                return $.when();
            }

            // send server request to replace all occurrences in the active sheet
            var all = Utils.getBooleanOption(options, 'all', false);
            return all ? replaceAll(query, replace) : replaceSingle(query, replace);
        };

        // hyperlink operations -----------------------------------------------

        /**
         * Opens the 'Edit Hyperlink' dialog, and applies the settings made in
         * the dialog to the current cell selection.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved after the dialog has been closed,
         *  and the hyperlink settings have been applied successfully; or that
         *  will be rejected after canceling the dialog, or on any error.
         */
        this.editHyperlink = function () {

            // sheet must be unlocked, active cell must be editable (perform additional check for tables)
            if (!this.requireUnlockedActiveSheet() || !this.requireEditableActiveCell({ lockTables: 'header' })) {
                return $.Deferred().reject();
            }

            var // the descriptor of the active cell
                cellDesc = this.getActiveCellEntry(),
                // the literal string value of the cell, used as initial display string
                display = (CellCollection.isText(cellDesc) && !cellDesc.formula) ? cellDesc.result : null,
                // the current hyperlink of the active cell
                currUrl = this.getCellURL(),
                // whether the cell contained a hyperlink before
                hasHyperlink = _.isString(currUrl);

            // if there is no real hyperlink, try the literal cell string
            if (!currUrl && display) {
                currUrl = HyperlinkUtils.checkForHyperlink(display);
            }

            // do not duplicate equal URL and display string in the dialog
            if (currUrl === display) { display = null; }

            // try to leave cell edit mode, show the 'Edit Hyperlink' dialog
            var promise = this.leaveCellEditMode('auto', { validate: true }) ?
                new Dialogs.HyperlinkDialog(this, currUrl, display).show() :
                $.Deferred().reject();

            // apply hyperlink settings on success
            promise = promise.then(function (result) {

                // missing result.url indicates to delete a hyperlink ('Remove' button)
                if (!result.url) {
                    return self.deleteHyperlink();
                }

                // add character formatting for a new hyperlink, and generate the operation
                var attributes = hasHyperlink ? null : { character: { underline: true, color: docModel.createJSONColor(Color.HYPERLINK, 'text') } };
                var operationPromise = self.fillCellRanges(undefined, attributes, { url: result.url });

                // create a text cell, if user has entered a display string in the dialog;
                // or if the cell was blank, and a valid URL has been entered (do not change
                // existing cells, e.g. numbers, formulas, etc.)
                operationPromise = operationPromise.then(function () {
                    var newDisplay = result.text ? result.text : CellCollection.isBlank(cellDesc) ? result.url : null;
                    return newDisplay ? self.setCellContents(newDisplay, null, { parse: true }) : null;
                });

                return operationPromise;
            });

            return promise;
        };

        /**
         * Deletes the hyperlink range covering the active cell.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved after the hyperlink ranges have
         *  been deleted successfully, ot that will be rejected on any error.
         */
        this.deleteHyperlink = function () {
            return this.getCellURL() ? this.fillCellRanges(undefined, { character: { underline: null, color: null } }, { url: '' }) : $.when();
        };

        // defined name operations --------------------------------------------

        /**
         * Inserts a new defined name into the document.
         *
         * @param {String} label
         *  The label of the new defined name.
         *
         * @param {String} formula
         *  The formula expression to be bound to the new defined name.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the defined name has been
         *  created successfully, or that will be rejected on any error.
         */
        this.insertName = function (label, formula) {

            // the descriptor for the formula expression
            var formulaDesc = { grammar: 'ui', formula: formula, refSheet: activeSheet, refAddress: this.getActiveCell() };

            // create and apply the operations to create the defined name
            // TODO: support for sheet-local names
            var promise = docModel.getNameCollection().insertName(label, formulaDesc);
            return this.yellOnPromise(promise);
        };

        /**
         * Shows a dialog that allows to create a new defined name for the
         * document. The dialog will be kept open until a valid label and
         * formula expression has been entered for the defined name.
         *
         * @returns {jQuery.Promise}
         *  A promise representing the dialog. Will be resolved, after the
         *  defined name has been created successfully; or will be rejected, if
         *  the dialog has been canceled, or if the document has switched to
         *  read-only mode.
         */
        this.showInsertNameDialog = function () {

            // leave cell edit mode before creating a name
            if (!this.leaveCellEditMode('auto', { validate: true })) {
                return $.Deferred().reject();
            }

            // create a formula from the current selection (bug 40477: use start address of single merged range)
            var tokenArray = new TokenArray(docModel, { type: 'name', temp: true });
            if (this.isSingleCellSelection()) {
                tokenArray.appendRange(new Range(this.getActiveCell()), { sheet: activeSheet, abs: true });
            } else {
                tokenArray.appendRangeList(this.getSelectedRanges(), { sheet: activeSheet, abs: true });
            }
            var formula = tokenArray.getFormula('ui');
            tokenArray.destroy();

            // create the dialog with empty input fields
            var dialog = new Dialogs.DefinedNameDialog(this, '', formula, this.insertName.bind(this), {
                title: gt('Insert name'),
                okLabel: gt('Insert')
            });

            // show the dialog (inserting the defined name will be done by the callback passed to the dialog)
            return dialog.show();
        };

        /**
         * Changes the label or formula definition of an existing defined name.
         *
         * @param {String} label
         *  The label of the defined name to be changed.
         *
         * @param {String|Null} newLabel
         *  The new label for the defined name. If set to null, the label of
         *  the defined name will not be changed.
         *
         * @param {String|Null} newFormula
         *  The new formula expression to be bound to the defined name. If set
         *  to null, the formula expression of the defined name will not be
         *  changed (an invalid formula expression in the defined name will be
         *  retained).
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the defined name has been
         *  changed successfully, or that will be rejected on any error.
         */
        this.changeName = function (label, newLabel, newFormula) {

            // the descriptor for the new formula expression
            var formulaDesc = _.isString(newFormula) ? { grammar: 'ui', formula: newFormula, refSheet: activeSheet, refAddress: this.getActiveCell() } : null;

            // create and apply the operations to change the defined name
            // TODO: support for sheet-local names
            var promise = docModel.getNameCollection().changeName(label, newLabel, formulaDesc);
            return this.yellOnPromise(promise);
        };

        /**
         * Shows a dialog that allows to change an existing defined name in the
         * document. The dialog will be kept open until a valid label and
         * formula expression has been entered.
         *
         * @param {String} label
         *  The label of the defined name to be changed.
         *
         * @returns {jQuery.Promise}
         *  A promise representing the dialog. Will be resolved, after the
         *  defined name has been changed successfully; or will be rejected, if
         *  the dialog has been canceled, or if the document has switched to
         *  read-only mode.
         */
        this.showChangeNameDialog = function (label) {

            // leave cell edit mode before changing a defined name
            if (!this.leaveCellEditMode('auto', { validate: true })) {
                return $.Deferred().reject();
            }

            // get the name model from the document collection (TODO: shet-local names)
            var nameModel = docModel.getNameCollection().getNameModel(label);
            if (!nameModel) { return $.Deferred().reject(); }

            // resolve the formula expression relative to the active cell
            var formula = nameModel.getFormula('ui', this.getActiveCell());

            // the callback function invoked from the dialog after pressing the OK button
            function actionHandler(newLabel, newFormula) {
                return self.changeName(label, (label !== newLabel) ? newLabel : null, (formula !== newFormula) ? newFormula : null);
            }

            // create the dialog with empty input fields
            var dialog = new Dialogs.DefinedNameDialog(this, label, formula, actionHandler, {
                title: gt('Change name'),
                okLabel: gt('Change')
            });

            // show the dialog (changing the defined name will be done by the callback passed to the dialog)
            return dialog.show();
        };

        /**
         * Deletes a defined name from the document.
         *
         * @param {String} label
         *  The label of the defined name to be deleted.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the defined name has been
         *  deleted successfully, or that will be rejected on any error.
         */
        this.deleteName = function (label) {
            // TODO: support for sheet-local names
            var promise = docModel.getNameCollection().deleteName(label);
            return this.yellOnPromise(promise);
        };

        // table/filter operations --------------------------------------------

        /**
         * Returns the model of the table range located at the active cell of
         * the current selection. If the active cell is not located inside a
         * table range, tries to return the model of the anonymous table range
         * representing the auto filter of the active sheet.
         *
         * @returns {TableModel|Null}
         *  The model of the active table range; or null, if no table range has
         *  been found.
         */
        this.getActiveTable = function () {
            return tableCollection.findTable(this.getActiveCell()) || tableCollection.getTable('');
        };

        /**
         * Returns whether the active cell is located in a real table range
         * (any table but the auto-filter).
         *
         * @returns {Boolean}
         *  Whether the active cell is located in a real table range.
         */
        this.isRealTableSelected = function () {
            var tableModel = this.getActiveTable();
            return _.isObject(tableModel) && !tableModel.isAutoFilter();
        };

        /**
         * Returns whether the active cell of the current selection is located
         * in a table range with activated filter/sorting. If the active cell
         * is not located in a table range, returns whether the active sheet
         * contains an activated auto filter (regardless of the selection).
         *
         * @returns {Boolean}
         *  Whether a table range with activated filter is selected.
         */
        this.hasTableFilter = function () {
            var tableModel = this.getActiveTable();
            return _.isObject(tableModel) && tableModel.isFilterActive();
        };

        /**
         * Enables or disables the filter in the selected table range of the
         * active sheet.
         *
         * @param {Boolean} state
         *  Whether to activate or deactivate the table filter. If the active
         *  cell of the current selection is not located inside a table range,
         *  the auto filter of the active sheet will be toggled.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the table filter has been
         *  toggled successfully, or that will be rejected on any error.
         */
        this.toggleTableFilter = function (state) {

            // no filter manipulation in locked sheets
            if (!this.requireUnlockedActiveSheet()) { return $.Deferred().reject(); }

            // the active table range
            var tableModel = self.getActiveTable();

            // toggle filter of a real table range (not auto filter)
            if (tableModel && (tableModel.getName() !== '')) {
                return this.yellOnPromise(tableCollection.changeTable(tableModel.getName(), { table: { filtered: state } }));
            }

            // toggle the auto filter (insert table for content range of selected cell)
            return undoManager.enterUndoGroup(function () {

                // remove existing auto filter (regardless of passed state)
                var promise = tableModel ? tableCollection.deleteTable('') : $.when();

                // create a new table for the auto filter
                if (state) {
                    promise = promise.then(function () {

                        var // the current selection ranges
                            ranges = self.getSelectedRanges(),
                            // the expanded first selected range
                            range = null,
                            // fixed attributes for the auto filter
                            attributes = { table: { headerRow: true, filtered: true } };

                        // auto filter cannot be created on a multi selection
                        if (ranges.length !== 1) {
                            return SheetUtils.makeRejected('autofilter:multiselect');
                        }

                        // expand single cell to content range
                        range = activeSheetModel.getContentRangeForCell(ranges.first());

                        // bug 36606: restrict range to used area of the sheet (range may become null)
                        range = range.intersect(cellCollection.getUsedRange());

                        // bug 36227: selection must not be entirely blank
                        if (!range || cellCollection.areRangesBlank(range)) {
                            return SheetUtils.makeRejected('autofilter:blank');
                        }

                        // auto filter cannot be created on a multi selection
                        return tableCollection.insertTable('', range, attributes);
                    });
                }

                // translate error codes for auto filter
                promise = promise.then(null, function (result) {
                    result.cause = result.cause.replace(/^table:/, 'autofilter:');
                    return result;
                });

                // show warning messages if needed
                return self.yellOnPromise(promise);
            });
        };

        /**
         * Returns whether the active table range can be refreshed (the active
         * sheet must contain an active table, and that table must contain any
         * active filter rules or sort options).
         *
         * @returns {Boolean}
         *  Whether the active table range can be refreshed.
         */
        this.canRefreshTable = function () {
            var tableModel = this.getActiveTable();
            return _.isObject(tableModel) && tableModel.isRefreshable();
        };

        /**
         * Refreshes the active table ranges (reapplies all filter rules and
         * sort options to the current cell contents in the table range).
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the table has been refreshed
         *  successfully, or that will be rejected on any error.
         */
        this.refreshTable = function () {

            var // the active table
                tableModel = this.getActiveTable(),
                // the result promise
                promise = null;

            // no filter manipulation in locked sheets
            if (tableModel && this.requireUnlockedActiveSheet()) {
                promise = tableCollection.refreshTable(tableModel.getName());
            } else {
                promise = $.Deferred().reject();
            }

            // show warning messages if needed
            return this.yellOnPromise(promise);
        };

        /**
         * Changes the formatting attributes of the specified column in a table
         * range of the active sheet.
         *
         * @param {String} tableName
         *  The name of the table to be modified. The empty string addresses
         *  the anonymous table range used to store filter settings for the
         *  standard auto filter of the sheet.
         *
         * @param {Number} tableCol
         *  The zero-based index of the table column to be modified, relative
         *  to the cell range covered by the table.
         *
         * @param {Object} attributes
         *  The attribute set with filter and sorting attributes to be set for
         *  the table column.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the filter has been applied
         *  successfully, or that will be rejected on any error.
         */
        this.changeTableColumn = function (tableName, tableCol, attributes) {

            // no filter manipulation in locked sheets
            if (!this.requireUnlockedActiveSheet()) { return $.Deferred().reject(); }

            // generate and apply the operations (including row and drawing operations)
            var promise = tableCollection.changeTableColumn(tableName, tableCol, attributes);

            // show warning messages if needed
            return this.yellOnPromise(promise);
        };

        // drawing operations -------------------------------------------------

        /**
         * Inserts new drawing sobject into the active sheet, and selects them
         * afterwards.
         *
         * @param {Array<Object>} drawingData
         *  The type and formatting attributes of the drawing objects to be
         *  inserted. Each array element MUST be an object with the following
         *  properties:
         *  @param {String} element.type
         *      The type identifier of the drawing object.
         *  @param {Object} element.attrs
         *      Initial attributes for the drawing object, especially the
         *      anchor attributes specifying the physical position of the
         *      drawing object in the sheet.
         *
         * @param {Function} [callback]
         *  A callback function that will be invoked everytime after an
         *  'insertDrawing' operation has been generated for a drawing object.
         *  Allows to create additional operations for the drawing objects,
         *  before all these operations will be applied at once. Receives the
         *  following parameters:
         *  (1) {SheetOperationsGenerator} generator
         *      The operations generator (already containing the initial
         *      'insertDrawing' operation of the current drawing).
         *  (2) {Number} sheet
         *      The zero-based index of this sheet.
         *  (3) {Array<Number>} position
         *      The effective document position of the current drawing object
         *      in the sheet, as inserted into the 'insertDrawing' operation.
         *  (4) {Object} data
         *      The data element from the array parameter 'drawingData' that is
         *      currently processed.
         *  The callback function may return a promise to defer processing of
         *  subsequent drawing objects.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when all drawing objects have been
         *  created successfully, or that will be rejected on any error.
         */
        this.insertDrawings = function (drawingData, callback) {

            // check that the active sheet is not locked
            if (this.isSheetLocked()) {
                return this.yellOnPromise(SheetUtils.makeRejected('drawing:insert:locked'));
            }

            var // the position of the first new drawing object
                firstPosition = [drawingCollection.getModelCount()],
                // the positions of all drawing objects to be selected later
                allPositions = [];

            // create the new drawing object in the active sheet, collect all positions for selection
            var promise = drawingCollection.insertDrawings(firstPosition, drawingData, function (generator, sheet, position, data) {
                allPositions.push(position);
                if (_.isFunction(callback)) {
                    callback.call(self, generator, sheet, position, data);
                }
            });

            // select the drawing objects on success
            return this.yellOnPromise(promise).done(function () {
                self.setDrawingSelection(allPositions);
            });
        };

        /**
         * Inserts a new drawing object into the active sheet, and selects it
         * afterwards.
         *
         * @param {String} type
         *  The type identifier of the drawing object.
         *
         * @param {Object} attributes
         *  Initial attributes for the drawing object, especially the anchor
         *  attributes specifying the position of the drawing object in the
         *  active sheet.
         *
         * @param {Function} [callback]
         *  A callback function that will be invoked after the 'insertDrawing'
         *  operation has been generated for the drawing object. Allows to
         *  create additional operations for the drawing object, before all
         *  these operations will be applied at once. Receives the following
         *  parameters:
         *  (1) {SheetOperationsGenerator} generator
         *      The operations generator (already containing the initial
         *      'insertDrawing' operation).
         *  (2) {Number} sheet
         *      The zero-based index of this sheet.
         *  (3) {Array<Number>} position
         *      The document position of the drawing object in the sheet, as
         *      inserted into the 'insertDrawing' operation.
         *  The callback function may return a promise to be able to run
         *  asynchronous code.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the drawing object has been
         *  created successfully, or that will be rejected on any error.
         */
        this.insertDrawing = function (type, attributes, callback) {
            return this.insertDrawings([{ type: type, attrs: attributes }], function (generator, sheet, position) {
                if (_.isFunction(callback)) {
                    return callback.call(self, generator, sheet, position);
                }
            });
        };

        /**
         * Inserts new image objects into the active sheet, and selects them
         * afterwards.
         *
         * @param {Object|Array<Object>} imageDescriptors
         *  The descriptor of the image object, containing the properties 'url'
         *  and 'name', or an array of such image descriptors to be able to
         *  insert multiple images at once.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the image object has been
         *  created and inserted into the active sheet, or that will be
         *  rejected on any error.
         */
        this.insertImages = function (imageDescriptors) {

            // create the image DOM nodes for all images
            var promises = _.getArray(imageDescriptors).map(function (imageDesc) {

                // create the image DOM node, the promise waits for it to load the image data
                var promise = app.createImageNode(ImageUtils.getFileUrl(app, imageDesc.url));

                // fail with empty resolved promise to be able to use $.when() which would
                // otherwise abort immediately if ANY of the promises fails
                return promise.then(function (imgNode) {
                    return { node: imgNode[0], url: imageDesc.url, name: imageDesc.name };
                }, _.constant($.when()));
            });

            // wait for all images nodes, create and apply the 'insertDrawing' operations
            return $.when.apply($, promises).then(function () {

                // $.when() passes the results as function arguments, filter out invalid images
                var results = _.filter(arguments, _.identity);
                if (results.length === 0) { return $.Deferred().reject(); }

                var // active cell as target position for the images
                    activeCell = self.getActiveCell(),
                    // all image nodes collected in a jQuery object
                    imgNodes = $();

                // the drawing attributes to be passed to insertDrawings()
                var drawingData = results.map(function (result) {

                    var // current width of the image, in 1/100 mm
                        width = Utils.convertLengthToHmm(result.node.width, 'px'),
                        // current height of the image, in 1/100 mm
                        height = Utils.convertLengthToHmm(result.node.height, 'px'),
                        // the generic drawing attributes (anchor position)
                        drawingAttrs = {
                            anchorType: 'oneCell',
                            startCol: activeCell[0],
                            startRow: activeCell[1],
                            name: result.name
                        },
                        // additional specific image attributes
                        imageAttrs = { imageUrl: result.url };

                    // restrict size to fixed limit
                    if (width > height) {
                        if (width > MAX_IMAGE_SIZE) {
                            height = Math.round(height * MAX_IMAGE_SIZE / width);
                            width = MAX_IMAGE_SIZE;
                        }
                    } else {
                        if (height > MAX_IMAGE_SIZE) {
                            width = Math.round(width * MAX_IMAGE_SIZE / height);
                            height = MAX_IMAGE_SIZE;
                        }
                    }
                    drawingAttrs.width = width;
                    drawingAttrs.height = height;

                    // collect all image nodes for clean-up
                    imgNodes = imgNodes.add(result.node);

                    // create the drawing descriptor for insertDrawings()
                    return { type: 'image', attrs: { drawing: drawingAttrs, image: imageAttrs } };
                });

                // destroy the temporary image nodes
                app.destroyImageNodes(imgNodes);

                // insert the image objects into the active sheet
                return self.insertDrawings(drawingData);
            });
        };

        /**
        * Shows an insert image dialog from drive, local or URL dialog
        *
        * @param {String} dialogType
        *  The dialog type. Types: 'drive', 'local' or 'url'. Default is 'drive'.
        *
        *
        * @returns {jQuery.Promise}
        *  A promise that will be resolved or rejected after the dialog has
        *  been closed and the image has been inserted into the document.
        */
        this.showInsertImageDialog = function (dialogType) {

            // leave cell edit mode before showing the dialog
            if (!this.leaveCellEditMode('auto', { validate: true })) {
                return $.Deferred().reject();
            }

            // show the dialog, handle the dialog result
            return ImageUtils.showInsertImageDialogByType(app, dialogType).then(function (imageDescriptor) {
                return self.insertImages(imageDescriptor).fail(function () {
                    app.rejectEditAttempt('image');
                });
            });
        };

        /**
         * Deletes all drawings currently selected in the active sheet.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when all drawing objects have been
         *  deleted successfully, or that will be rejected on any error.
         */
        this.deleteDrawings = function () {

            // check that the active sheet is not locked
            if (this.isSheetLocked()) {
                return this.yellOnPromise(SheetUtils.makeRejected('drawing:delete:locked'));
            }

            // rescue drawing selection, back to cell selection
            var positions = this.getSelectedDrawings();
            this.removeDrawingSelection();

            // delete all drawing objects previously selected
            return this.yellOnPromise(drawingCollection.deleteDrawings(positions));
        };

        /**
         * Changes the formatting attributes of all selected drawing objects,
         * or of the specified drawing objects.
         *
         * @param {Object} attributes
         *  An (incomplete) attribute set to be applied at the drawing objects.
         *
         * @param {Array<Array<Number>>} [positions]
         *  An array with positions of the drawing objects to be modified. If
         *  omitted, all selected drawing objects will be modified.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when all drawing objects have been
         *  modified successfully, or that will be rejected on any error.
         */
        this.setDrawingAttributes = function (attributes, positions) {

            // fall-back to the selected drawing objects
            positions = positions || this.getSelectedDrawings();

            // modify the drawing objects in the active sheet
            return this.yellOnPromise(drawingCollection.setDrawingAttributes(positions, attributes));
        };

        // initialization -----------------------------------------------------

        // get reference to localized formula resources
        docModel.onInitFormulaResource(function () {
            grammarConfig = docModel.getGrammarConfig('ui');
        });

        // initialize selection settings, used until first real update
        importSelectionSettings({});
        activeCellDesc = _.copy(CellCollection.DEFAULT_CELL_DATA, true);
        activeCellDesc.explicit = {};
        activeCellDesc.attributes = cellStyles.getDefaultStyleAttributeSet();
        activeCellDesc.format = numberFormatter.parseNumberFormat(activeCellDesc.attributes.cell.numberFormat);

        // initialize sheet-dependent class members according to the active sheet
        this.on('change:activesheet', changeActiveSheetHandler);

        // refresh selection settings after changing the selection
        this.on('change:selection:prepare', function () { updateSelectionSettings(); });

        // update selection settings after any changes in the active sheet (debounced in case of mass operations, e.g. undo)
        this.on('sheet:triggered', self.createDebouncedMethod(_.noop, updateSelectionSettings, { delay: 20, maxDelay: 500 }));

        // application notifies changed contents/results of cells
        this.listenTo(app, 'docs:update:cells', updateNotificationHandler);

        if (Config.DEBUG) { initDebugOperations(); }

        // destroy all class members on destruction
        this.registerDestructor(function () {
            self = app = docModel = undoManager = numberFormatter = cellStyles = grammarConfig = null;
            activeSheetModel = colCollection = rowCollection = mergeCollection = cellCollection = null;
            tableCollection = drawingCollection = null;
        });

    } // class ViewFuncMixin

    // exports ================================================================

    return ViewFuncMixin;

});
