/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/mixin/gridtrackingmixin', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/utils/tracking',
    'io.ox/office/drawinglayer/view/drawingframe',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/utils/paneutils'
], function (Utils, Tracking, DrawingFrame, SheetUtils, PaneUtils) {

    'use strict';

    var // convenience shortcuts
        Address = SheetUtils.Address,
        Range = SheetUtils.Range;

    // mix-in class GridTrackingMixin =========================================

    /**
     * Mix-in class for the class GridPane that implements all kinds of mouse
     * and touch tracking, for example selection, or manipulation of drawing
     * objects.
     *
     * @constructor
     */
    function GridTrackingMixin() {

        var // self reference (the GridPane instance)
            self = this,

            // the spreadsheet model and view
            docView = this.getDocView(),
            docModel = docView.getDocModel(),

            // the root node of all rendering layers
            layerRootNode = this.getLayerRootNode(),

            // type of the last finished tracking cycle
            lastTrackingType = null,

            // whether current tracking can be continued in read-only mode
            readOnlyTracking = false;

        // private methods ----------------------------------------------------

        /**
         * Handles a changed document edit mode. If switched to read-only mode,
         * cancels in-place cell edit mode, or a running tracking action to
         * move or resize drawing objects.
         */
        function editModeHandler(event, editMode) {
            // stop tracking if edit rights have been lost
            if (!editMode && !readOnlyTracking) {
                Tracking.cancelTracking();
            }
        }

        /**
         * Returns all data for the cell covering the page coordinates in the
         * passed tracking event.
         *
         * @param {jQuery.Event} event
         *  The tracking event to be evaluated, expected to contain the numeric
         *  properties 'pageX' and 'pageY'.
         *
         * @param {Object} [options]
         *  Optional parameters that will be forwarded to the method
         *  ColRowCollection.getEntryByOffset().
         *
         * @returns {Object}
         *  A result descriptor for the cell containing the tracking position.
         *  See method GridPane.getCellDataByOffset() for details.
         */
        function getCellData(event, options) {
            return self.getCellDataByOffset(event.pageX, event.pageY, options);
        }

        /**
         * Returns the address of the anchor cell for a tracking cell range,
         * according to the passed tracking position.
         *
         * @param {jQuery.Event} event
         *  The initial 'tracking:start' event.
         *
         * @param {Range} origRange
         *  The address of the cell range to be tracked.
         *
         * @param {String} trackerPos
         *  The position identifier of the tracker node.
         *
         * @param {Boolean} resize
         *  Whether resize tracking (true), or border tracking (false) is
         *  active. For border tracking, the tracker position passed in the
         *  parameter 'trackerPos' must be 't', 'b', 'l', or 'r'.
         *
         * @returns {Address}
         *  The address of the anchor cell in the passed range.
         */
        function getAnchorCell(event, origRange, trackerPos, resize) {
            switch (trackerPos) {
                case 'l':
                    return resize ? origRange.end.clone() : new Address(origRange.start[0], getCellData(event).address[1]);
                case 'r':
                    return resize ? origRange.start.clone() : new Address(origRange.end[0], getCellData(event).address[1]);
                case 't':
                    return resize ? origRange.end.clone() : new Address(getCellData(event).address[0], origRange.start[1]);
                case 'b':
                    return resize ? origRange.start.clone() : new Address(getCellData(event).address[0], origRange.end[1]);
                case 'tl':
                    return origRange.end.clone();
                case 'tr':
                    return new Address(origRange.start[0], origRange.end[1]);
                case 'bl':
                    return new Address(origRange.end[0], origRange.start[1]);
                case 'br':
                    return origRange.start.clone();
            }
        }

        /**
         * Recalculates the address of the tracked range according to the
         * passed tracking position.
         *
         * @param {jQuery.Event} event
         *  The tracking event.
         *
         * @param {Range} origRange
         *  The address of the original cell range currently tracked.
         *
         * @param {Address} anchorCell
         *  The anchor cell inside the passed original range, as has been
         *  returned by the method GridTrackingMixin.getAnchorCell().
         *
         * @param {String} trackerPos
         *  The position identifier of the tracker node.
         *
         * @param {Boolean} resize
         *  Whether resize tracking (true), or border tracking (false) is
         *  active. For border tracking, the tracker position passed in the
         *  parameter 'trackerPos' must be 't', 'b', 'l', or 'r'.
         *
         * @returns {Range}
         *  The address of the cell range currently tracked.
         */
        function recalcTrackedRange(event, origRange, anchorCell, trackerPos, resize) {

            var // current column/row entries
                trackingData = getCellData(event, { outerHidden: true }),
                // the cell address currently tracked
                address = trackingData.address,
                // the resulting range
                range = null;

            // adjusts column/row index according to the offset inside the covered cell
            function adjustTrackedBorder(columns) {

                var // the maximum column/row index
                    maxIndex = docModel.getMaxIndex(columns),
                    // the column/row descriptor
                    entryData = columns ? trackingData.colDesc : trackingData.rowDesc,
                    // whether the leading border is tracked
                    leadingBorder = resize ? (address.get(columns) < anchorCell.get(columns)) : (trackerPos === (columns ? 'l' : 't')),
                    // whether the trailing border is tracked
                    trailingBorder = resize ? (address.get(columns) > anchorCell.get(columns)) : (trackerPos === (columns ? 'r' : 'b')),
                    // the offset ratio in the tracked column/row
                    ratio = (entryData.size > 0) ? (entryData.relOffset / entryData.size) : Number.NaN;

                if (leadingBorder && (address.get(columns) < maxIndex) && (ratio > 0.5)) {
                    // jump to next column/row if tracking leading border, and covering second half of the tracked column/row ...
                    address.move(1, columns);
                } else if (trailingBorder && (address.get(columns) > 0) && (ratio < 0.5)) {
                    // ... or to previous column/row if tracking trailing border, and covering first half of the tracked column/row
                    address.move(-1, columns);
                }
            }

            // adjust column/row index according to the offset inside the tracked cell
            adjustTrackedBorder(false);
            adjustTrackedBorder(true);

            // build and return the new range according to the current position of the tracked cell
            if (resize) {
                // resize mode: build range between anchor cell and tracked cell
                range = Range.createFromAddresses(anchorCell, address);
                if ((trackerPos === 't') || (trackerPos === 'b')) {
                    // resize vertically: restore row indexes to original range
                    range.start[0] = origRange.start[0];
                    range.end[0] = origRange.end[0];
                } else if ((trackerPos === 'l') || (trackerPos === 'r')) {
                    // resize horizontally: restore row indexes to original range
                    range.start[1] = origRange.start[1];
                    range.end[1] = origRange.end[1];
                }
            } else {
                // border mode: move entire range according to distance between tracked cell and anchor cell
                range = docModel.getBoundedMovedRange(origRange, address[0] - anchorCell[0], address[1] - anchorCell[1]);
            }

            return range;
        }

        // cell selection (mouse) ---------------------------------------------

        /**
         * Handles all tracking events for cell selection originating from
         * mouse events.
         */
        var mouseSelectionTrackingHandler = (function () {

            var // current address (prevent updates while tracking over the same cell)
                currentAddress = null;

            // updates the current address according to the passed tracking event
            function updateCurrentAddress(event) {
                var address = getCellData(event, { outerHidden: true }).address;
                if (address.differs(currentAddress)) {
                    currentAddress = address;
                    self.trigger('select:move', currentAddress);
                }
            }

            // return the actual mouseSelectionTrackingHandler() method
            return function (event) {
                switch (event.type) {
                    case 'tracking:start':
                        currentAddress = getCellData(event).address;
                        self.trigger('select:start', currentAddress, PaneUtils.getSelectionMode(event));
                        self.scrollToCell(currentAddress);
                        break;
                    case 'tracking:move':
                        // prevents that Chrome modifies the text area selection while moving mouse
                        event.preventDefault();
                        updateCurrentAddress(event);
                        break;
                    case 'tracking:scroll':
                        self.scrollRelative(event.scrollX, event.scrollY);
                        updateCurrentAddress(event);
                        break;
                    case 'tracking:end':
                        self.trigger('select:end', currentAddress, true);
                        self.scrollToCell(currentAddress);
                        break;
                    case 'tracking:cancel':
                        self.trigger('select:end', currentAddress, false);
                        self.scrollToCell(currentAddress);
                        break;
                }
            };
        }()); // end of mouseSelectionTrackingHandler() local scope

        // cell selection (touch) ---------------------------------------------

        /**
         * Handles all tracking events for cell selection originating from
         * touch events.
         */
        var touchSelectionTrackingHandler = (function () {

            var // cell address where tracking has started
                startAddress = null,
                // whether the touch point has been moved
                moved = false;

            function finalizeTracking(event) {

                var // current selection
                    selection = docView.getSelection(),
                    // array index of the range that contains a tapped cell
                    rangeIndex = 0;

                // do nothing, if the touch point has been moved
                if (moved) { return; }

                if (event.duration <= 750) {
                    // short-tap: select a single cell, or start in-place edit mode
                    if (!docView.hasDrawingSelection() && selection.isActive(startAddress)) {
                        docView.enterCellEditMode();

                    } else if (!docView.hasDrawingSelection() && selection.containsAddress(startAddress)) {
                        rangeIndex = Utils.findFirstIndex(selection.ranges, function (range) { return range.containsAddress(startAddress); });
                        self.trigger('select:active', rangeIndex, startAddress);

                    } else {
                        self.trigger('select:start', startAddress, 'select');
                        self.trigger('select:end', startAddress, true);
                        self.scrollToCell(startAddress);
                    }
                } else {
                    if (!selection.isActive(startAddress) && selection.containsAddress(startAddress)) {
                        rangeIndex = Utils.findFirstIndex(selection.ranges, function (range) { return range.containsAddress(startAddress); });
                        self.trigger('select:active', rangeIndex, startAddress);

                    } else if (!selection.containsAddress(startAddress)) {
                        self.trigger('select:start', startAddress, 'select');
                        self.trigger('select:end', startAddress, true);
                        self.scrollToCell(startAddress);
                    }

                    // ios fires no native 'contextmenu'-events
                    if (_.device('ios')) {
                        $('html').trigger('contextmenu');
                    }
                }
            }

            // return the actual touchSelectionTrackingHandler() method
            return function (event) {
                switch (event.type) {
                    case 'tracking:start':
                        startAddress = getCellData(event).address;
                        moved = false;
                        break;
                    case 'tracking:move':
                        self.scrollRelative(-event.moveX, -event.moveY);
                        if (Math.abs(event.offsetX) >= 6 || Math.abs(event.offsetY) >= 6) { moved = true; }
                        event.preventDefault(); // prevent native scrolling
                        break;
                    case 'tracking:end':
                        finalizeTracking(event);
                        break;
                    case 'tracking:cancel':
                        break;
                }
            };
        }()); // end of touchSelectionTrackingHandler() local scope

        // resize selection range (touch) -------------------------------------

        /**
         * Handles tracking events to resize selected ranges on touch devices.
         */
        var resizeSelectionTrackingHandler = (function () {

            var // the position identifier of the tracked node
                trackerPos = null,
                // the array index of the tracked range
                rangeIndex = 0,
                // the original address of the tracked range
                origRange = null,
                // the anchor cell in the original range
                anchorCell = null,
                // the current address of the tracked range
                currRange = null;

            // initializes the highlighted range according to the passed tracking event
            function initializeTracking(event) {

                var // the handle node currently tracked
                    trackerNode = $(event.target),
                    // the root node of the range that will be tracked
                    rangeNode = trackerNode.closest('.range');

                // extract range address
                rangeIndex = Utils.getElementAttributeAsInteger(rangeNode, 'data-index', -1);
                origRange = docView.getSelectedRanges()[rangeIndex];

                // initialize additional data needed while tracking
                trackerPos = trackerNode.attr('data-pos');
                currRange = origRange.clone();
                anchorCell = getAnchorCell(event, origRange, trackerPos, true);
            }

            // updates the highlighted range according to the passed tracking event
            function updateTracking(event) {

                var // the new range address
                    newRange = recalcTrackedRange(event, origRange, anchorCell, trackerPos, true);

                // update the tracked range in the respective token array, this will
                // trigger change events which will cause refreshing all grid panes
                if (currRange.differs(newRange)) {
                    currRange = newRange;
                    self.trigger('select:range', rangeIndex, currRange);
                }
            }

            // return the actual resizeSelectionTrackingHandler() method
            return function (event) {
                switch (event.type) {
                    case 'tracking:start':
                        initializeTracking(event);
                        event.preventDefault(); // prevent native scrolling
                        break;
                    case 'tracking:move':
                        updateTracking(event);
                        event.preventDefault(); // prevent native scrolling
                        break;
                    case 'tracking:scroll':
                        self.scrollRelative(event.scrollX, event.scrollY);
                        updateTracking(event);
                        break;
                }
            };
        }()); // end of resizeSelectionTrackingHandler() local scope

        // move/resize highlighted ranges -------------------------------------

        /**
         * Handles all tracking events for highlighted ranges.
         */
        var highlightTrackingHandler = (function () {

            var // the position identifier of the tracked node (border or corner handle)
                trackerPos = null,
                // whether resize tracking is active
                resize = false,
                // the array index of the tracked highlighted range
                rangeIndex = null,
                // the original range address
                origRange = null,
                // the anchor cell in the original range
                anchorCell = null,
                // the current address of the tracked range
                currRange = null;

            // initializes the highlighted range according to the passed tracking event
            function initializeTracking(event) {

                var // the border/handle node currently tracked
                    trackerNode = $(event.target),
                    // the root node of the highlighted range that will be tracked
                    rangeNode = trackerNode.closest('.range');

                // the array index and range address of the tracked range
                rangeIndex = Utils.getElementAttributeAsInteger(rangeNode, 'data-index', -1);
                origRange = docView.getHighlightedRange(rangeIndex);
                if (!origRange) {
                    Utils.warn('GridTrackingMixin.initializeTracking(): cannot find tracking range');
                    Tracking.cancelTracking();
                    return;
                }

                // initialize additional data needed while tracking
                trackerPos = trackerNode.attr('data-pos');
                resize = trackerNode.parent().hasClass('resizers');
                origRange = origRange.toRange(); // convert to simple range without sheet indexes
                currRange = origRange.clone();
                anchorCell = getAnchorCell(event, origRange, trackerPos, resize);

                // render the tracked range with special glow effect
                docView.trackHighlightedRange(rangeIndex);
            }

            // updates the highlighted range according to the passed tracking event
            function updateTracking(event) {

                var // the new range address
                    newRange = recalcTrackedRange(event, origRange, anchorCell, trackerPos, resize);

                // update the tracked range in the respective token array, this will
                // trigger change events which will cause refreshing all grid panes
                if (currRange.differs(newRange)) {
                    currRange = newRange;
                    docView.changeHighlightedRange(rangeIndex, currRange);
                }
            }

            // finalizes the highlighted range
            function finalizeTracking() {
                docView.trackHighlightedRange(-1);
            }

            // return the actual highlightTrackingHandler() method
            return function (event) {
                switch (event.type) {
                    case 'tracking:start':
                        initializeTracking(event);
                        event.preventDefault();
                        break;
                    case 'tracking:move':
                        updateTracking(event);
                        // prevent Chrome from modifying the text area selection while moving mouse
                        event.preventDefault();
                        break;
                    case 'tracking:scroll':
                        self.scrollRelative(event.scrollX, event.scrollY);
                        updateTracking(event);
                        break;
                    case 'tracking:end':
                        finalizeTracking(event);
                        break;
                    case 'tracking:cancel':
                        finalizeTracking(event);
                        if (origRange) { docView.changeHighlightedRange(rangeIndex, origRange); }
                        break;
                }
            };
        }()); // end of highlightTrackingHandler() local scope

        // auto fill ----------------------------------------------------------

        /**
         * Handles all tracking events for the auto-fill handle.
         */
        var autoFillTrackingHandler = (function () {

            var // the sheet range without leading/trailing hidden columns/rows
                visibleRange = null,
                // the initial selection range
                origRange = null,
                // the sheet rectangle covered by the initial selection range
                origRectangle = null,
                // start position of tracking (bottom-right corner of selection range)
                startPos = null,
                // whether to restrict tracking direction (column/row ranges)
                fixedColumns = null,
                // resulting maximum number of columns that can be filled at the leading/trailing border
                maxLeadingCols = 0,
                maxTrailingCols = 0,
                // resulting maximum number of rows that can be filled at the leading/trailing border
                maxLeadingRows = 0,
                maxTrailingRows = 0;

            // initializes the auto fill handling according to the passed tracking event
            function initializeTracking() {

                // cancel custom selection when starting auto-fill
                docView.cancelCustomSelectionMode();

                // do not expand auto-fill range into leading/trailing hidden columns/rows
                visibleRange = docView.getSheetModel().shrinkRangeToVisible(docModel.getSheetRange());
                if (!visibleRange) { Tracking.cancelTracking(); return; }

                // store address and position of the original selection range
                origRange = docView.getSelectedRanges().first();
                origRectangle = docView.getRangeRectangle(origRange);
                startPos = { left: origRectangle.left + origRectangle.width, top: origRectangle.top + origRectangle.height };

                // special behavior for column/row ranges
                fixedColumns = docModel.isColRange(origRange) ? true : docModel.isRowRange(origRange) ? false : null;

                // restrict number of columns/rows according to selection mode
                var maxCols = 0, maxRows = 0;
                if (fixedColumns === true) {
                    maxCols = SheetUtils.MAX_AUTOFILL_COL_ROW_COUNT;
                } else if (fixedColumns === false) {
                    maxRows = SheetUtils.MAX_AUTOFILL_COL_ROW_COUNT;
                } else {
                    maxCols = Math.floor(SheetUtils.MAX_FILL_CELL_COUNT / origRange.rows());
                    maxRows = Math.floor(SheetUtils.MAX_FILL_CELL_COUNT / origRange.cols());
                }

                // calculate maximum number of columns/rows that can be filled at once
                maxLeadingCols = Math.min(maxCols, Math.max(0, origRange.start[0] - visibleRange.start[0]));
                maxTrailingCols = Math.min(maxCols, Math.max(0, visibleRange.end[0] - origRange.end[0]));
                maxLeadingRows = Math.min(maxRows, Math.max(0, origRange.start[1] - visibleRange.start[1]));
                maxTrailingRows = Math.min(maxRows, Math.max(0, visibleRange.end[1] - origRange.end[1]));

                // render selected range with modified auto-fill style
                docView.setSheetViewAttribute('autoFillData', { border: 'bottom', count: 0 });
            }

            // updates the auto fill area according to the passed tracking event
            function updateTracking(event) {

                var // current tracking position
                    trackingData = getCellData(event, { outerHidden: true }),
                    // whether to delete parts of the selected range
                    deleteMode = Utils.rectangleContainsPixel(origRectangle, trackingData),
                    // the adjusted tracking position to determine the direction
                    left = (deleteMode || (trackingData.left >= startPos.left)) ? trackingData.left : Math.min(trackingData.left + origRectangle.width, startPos.left),
                    top = (deleteMode || (trackingData.top >= startPos.top)) ? trackingData.top : Math.min(trackingData.top + origRectangle.height, startPos.top),
                    // whether to expand horizontally or vertically
                    columns = _.isBoolean(fixedColumns) ? fixedColumns : (Math.abs(top - startPos.top) < Math.abs(left - startPos.left)),
                    // whether to expand the leading or trailing border
                    leading = columns ? (left < startPos.left) : (top < startPos.top),
                    // the tracker position to calculate the target rectangle
                    trackerPos = columns ? (leading ? 'l' : 'r') : (leading ? 't' : 'b'),
                    // simulate tracking at a specific border of the selected range
                    range = recalcTrackedRange(event, origRange, leading ? origRange.start : origRange.end, trackerPos, false),
                    // the new value for the 'autoFillData' attribute
                    autoFillData = {};

                // calculate the auto-fill data from target range
                if (deleteMode) {
                    // delete mode: always delete from trailing border
                    autoFillData.border = columns ? 'right' : 'bottom';
                    autoFillData.count = range.getStart(columns) - origRange.getEnd(columns) - 1;
                } else if (leading) {
                    // expand leading range border
                    autoFillData.border = columns ? 'left' : 'top';
                    autoFillData.count = Math.min(origRange.getStart(columns) - range.getStart(columns), columns ? maxLeadingCols : maxLeadingRows);
                } else {
                    // expand trailing range border
                    autoFillData.border = columns ? 'right' : 'bottom';
                    autoFillData.count = Math.min(range.getEnd(columns) - origRange.getEnd(columns), columns ? maxTrailingCols : maxTrailingRows);
                }
                docView.setSheetViewAttribute('autoFillData', autoFillData);

                // set correct mouse pointer according to direction
                event.cursor = (_.isBoolean(fixedColumns) || (autoFillData.count === 0)) ? null : columns ? 'ew-resize' : 'ns-resize';
            }

            // finalizes the selection tracking
            function finalizeTracking(event, apply) {

                // apply auto-fill operation
                if (apply) {
                    var autoFillData = docView.getSheetViewAttribute('autoFillData');
                    // bug 39533: via controller for busy screen etc.
                    docView.executeControllerItem('cell/autofill', autoFillData);
                }

                // reset the view attribute
                docView.setSheetViewAttribute('autoFillData', null);
            }

            // return the actual autoFillTrackingHandler() method
            return function (event) {

                switch (event.type) {
                    case 'tracking:start':
                        initializeTracking(event);
                        break;
                    case 'tracking:move':
                        updateTracking(event);
                        break;
                    case 'tracking:scroll':
                        self.scrollRelative(event.scrollX, event.scrollY);
                        updateTracking(event);
                        break;
                    case 'tracking:end':
                        finalizeTracking(event, true);
                        break;
                    case 'tracking:cancel':
                        finalizeTracking(event, false);
                        break;
                }
            };
        }()); // end of autoFillTrackingHandler() local scope

        // select/move/resize drawings ----------------------------------------

        /**
         * Handles all tracking events for drawing frames.
         */
        var drawingTrackingHandler = (function () {

            var // the current drawing frame being tracked
                drawingFrame = null,
                // the logical position of the tracked drawing frame
                drawingPos = null,
                // the container element used to visualize the movement and resizing
                moveBox = null,
                // the position change in the sheet (in px)
                shiftX = 0, shiftY = 0,
                // the start position of the event in the sheet
                startX = 0, startY = 0,
                // the current width and height of the move box
                currentWidth = null, currentHeight = null,
                // the current left and top position of the move box
                currentLeft = null, currentTop = null,
                // whether the mouse was really moved
                mouseMoved = false,
                // whether the drawing shall be moved or resized
                doResize = false,
                // whether resizing is available in horizontal/vertical direction
                useX = false, useY = false, leftResize = false, topResize = false,
                // the maximum allowed values for move and resize
                maxLeft = 0, maxTop = 0, maxRight = 0, maxBottom = 0,
                // the width-to-height ratio of the drawing
                ratio = 1,
                // the width and height of the 'active' drawing before resizing it, in px
                activeStartWidth = 0, activeStartHeight = 0,
                // whether more than one drawing is selected
                isMultiSelection = false,
                // multi selection: container for all positions of selected drawings
                allDrawingPositions = [],
                // multi selection: container for all drawing frames of selected drawings
                allDrawingFrames = [],
                // multi selection: container for all drawing move boxes of selected drawings
                allDrawingMoveBoxes = [],
                // multi selection: container for all drawing geometries
                allStartWidths = [], allStartHeights = [];

            // initializes the drawing according to the passed tracking event
            function initializeTracking(event) {

                var // the column/row collections of the active sheet
                    colCollection = docView.getColCollection(),
                    rowCollection = docView.getRowCollection(),
                    // the grid pane layer rectangle
                    layerRectangle = self.getLayerRectangle(),
                    // whether to toggle the selection of the clicked drawing frame
                    toggle = event.shiftKey || event.ctrlKey || event.metaKey,
                    // whether the clicked drawing frame is already selected
                    selected = false,
                    // taking care of the distance to the border and of the geometry of each drawing during resize
                    widthRatio = 1, heightRatio = 1,
                    // the resize handle position
                    pos = null,
                    // the current selection
                    selection = null,
                    // the selected drawing objects
                    drawings = null,
                    // the drawing attributes
                    oneDrawingAttributes = null, drawingAttributes = null,
                    // the width and height of the 'active' drawing before resizing it, in hmm
                    activeStartWidthHmm = 0, activeStartHeightHmm = 0;

                // ensure that a drawing frame has been hit
                if (!(drawingFrame = Utils.findFarthest(layerRootNode, event.target, DrawingFrame.NODE_SELECTOR))) {
                    Tracking.cancelTracking();
                    return;
                }

                // convert to single jQuery object
                drawingFrame = $(drawingFrame).first();

                // do not modify the selection, if clicked on a selected drawing without modifier key
                selected = DrawingFrame.isSelected(drawingFrame);
                if (toggle || !selected) {
                    drawingPos = [$(drawingFrame).index()];
                    docView.selectDrawing(drawingPos, { toggle: toggle });
                    docView.scrollToDrawingFrame(drawingPos);
                }

                // check edit mode and sheet protection after selecting the drawing
                if (!docModel.getEditMode() || docView.isSheetLocked()) {
                    Tracking.cancelTracking();
                    return;
                }

                // finding moveBox, if not yet found
                if (!moveBox) {
                    moveBox = drawingFrame.find('.tracker');
                    if (moveBox.length === 0) {
                        moveBox = null;
                    }
                }

                // cancel tracking, if moveBox cannot be found
                if (!moveBox) {
                    Tracking.cancelTracking();
                    return;
                }

                // receiving the current selection
                selection = docView.getSelection();

                isMultiSelection = (selection.drawings.length > 1);

                // Handling multi selection as special case for performance reasons
                if (isMultiSelection) {
                    drawings = selection.drawings;

                    // resetting all drawing collectors
                    allDrawingFrames = [];
                    allDrawingMoveBoxes = [];
                    allDrawingPositions = [];
                    allStartWidths = [];
                    allStartHeights = [];

                    _.each(drawings, function (position) {

                        var localDrawingFrame = self.getDrawingFrame(position),
                            localMoveBox = localDrawingFrame.find('.tracker');

                        // collecting drawings for multi selection handling
                        if (localDrawingFrame && localMoveBox) {
                            allDrawingFrames.push(localDrawingFrame);
                            allDrawingMoveBoxes.push(localMoveBox);
                            allDrawingPositions.push(_.clone(position));
                        }
                    });
                }

                // setting correct size of moveBox (required after undo of drawing attributes on selected drawing
                if (isMultiSelection) {
                    _.each(allDrawingMoveBoxes, function (localMoveBox, index) {
                        localMoveBox.css({
                            left: 0,
                            top: 0,
                            width: allDrawingFrames[index].width(),
                            height: allDrawingFrames[index].height()
                        });
                    });
                } else {
                    moveBox.css({
                        left: 0,
                        top: 0,
                        width: drawingFrame.width(),
                        height: drawingFrame.height()
                    });
                }

                // check, whether the drawing shall be moved or resized
                pos = DrawingFrame.getResizerHandleType(event.target);
                if (_.isString(pos)) {
                    doResize = true;

                    // collecting information about the handle node
                    useX = /[lr]/.test(pos);
                    useY = /[tb]/.test(pos);
                    leftResize = /[l]/.test(pos);
                    topResize = /[t]/.test(pos);

                    if (isMultiSelection) {
                        _.each(allDrawingFrames, function (oneDrawingFrame) {
                            allStartWidths.push(oneDrawingFrame.width());
                            allStartHeights.push(oneDrawingFrame.height());
                        });
                    }

                    // geometry of the 'active' drawing
                    activeStartWidth = drawingFrame.width();
                    activeStartHeight = drawingFrame.height();
                    // receiving explicit attributes of the drawing
                    drawingAttributes = DrawingFrame.getModel(drawingFrame).getMergedAttributes().drawing;
                    activeStartWidthHmm = drawingAttributes.width;
                    activeStartHeightHmm = drawingAttributes.height;
                    // using hmm to calculate ratio instead of px because of increased precision
                    ratio = Utils.round(activeStartWidthHmm / activeStartHeightHmm, 0.01);

                } else {
                    doResize = false;
                    useX = false;
                    useY = false;
                }

                startX = event.pageX - layerRootNode.offset().left + layerRectangle.left;
                startY = event.pageY - layerRootNode.offset().top + layerRectangle.top;

                // the maximum allowed change to the left, top, right and bottom
                if (isMultiSelection) {
                    _.each(allDrawingFrames, function (oneDrawingFrame, index) {
                        if (doResize) {
                            // receiving explicit attributes of the drawing
                            oneDrawingAttributes = DrawingFrame.getModel(oneDrawingFrame).getMergedAttributes().drawing;
                            // taking care of the distance to the border and of the geometry of each drawing
                            widthRatio = Utils.round(activeStartWidthHmm / oneDrawingAttributes.width, 0.01);
                            heightRatio = Utils.round(activeStartHeightHmm / oneDrawingAttributes.height, 0.01);
                        } else {  // no change of image size during move of drawing
                            widthRatio = 1;
                            heightRatio = 1;
                        }
                        if (index === 0) {
                            maxLeft = Math.round((oneDrawingFrame.offset().left - layerRootNode.offset().left + layerRectangle.left) * widthRatio);
                            maxTop = Math.round((oneDrawingFrame.offset().top - layerRootNode.offset().top + layerRectangle.top) * heightRatio);
                            maxRight = Math.round((colCollection.getTotalSize() - layerRectangle.left - oneDrawingFrame.offset().left + layerRootNode.offset().left - oneDrawingFrame.width()) * widthRatio);
                            maxBottom = Math.round((rowCollection.getTotalSize() - layerRectangle.top - oneDrawingFrame.offset().top + layerRootNode.offset().top - oneDrawingFrame.height()) * heightRatio);
                        } else {
                            maxLeft = Math.min(maxLeft, Math.round((oneDrawingFrame.offset().left - layerRootNode.offset().left + layerRectangle.left) * widthRatio));
                            maxTop = Math.min(maxTop, Math.round((oneDrawingFrame.offset().top - layerRootNode.offset().top + layerRectangle.top) * heightRatio));
                            maxRight = Math.min(maxRight, Math.round((colCollection.getTotalSize() - layerRectangle.left - oneDrawingFrame.offset().left + layerRootNode.offset().left - oneDrawingFrame.width()) * widthRatio));
                            maxBottom = Math.min(maxBottom, Math.round((rowCollection.getTotalSize() - layerRectangle.top - oneDrawingFrame.offset().top + layerRootNode.offset().top - oneDrawingFrame.height()) * heightRatio));
                        }
                    });
                } else {
                    maxLeft = Math.round(drawingFrame.offset().left - layerRootNode.offset().left + layerRectangle.left);
                    maxTop = Math.round(drawingFrame.offset().top - layerRootNode.offset().top + layerRectangle.top);
                    maxRight = Math.round(colCollection.getTotalSize() - layerRectangle.left - drawingFrame.offset().left + layerRootNode.offset().left - drawingFrame.width());
                    maxBottom = Math.round(rowCollection.getTotalSize() - layerRectangle.top - drawingFrame.offset().top + layerRootNode.offset().top - drawingFrame.height());
                }

                // keeping the width-to-height ratio constant in resize mode
                if (doResize && useX && useY) {
                    if (leftResize && topResize) {
                        if ((maxLeft / maxTop) > ratio) { maxLeft = Math.round(maxTop * ratio); } else { maxTop = Math.round(maxLeft / ratio); }
                    } else if (leftResize && !topResize) {
                        if ((maxLeft / maxBottom) > ratio) { maxLeft = Math.round(maxBottom * ratio); } else { maxBottom = Math.round(maxLeft / ratio); }
                    } else if (!leftResize && topResize) {
                        if ((maxRight / maxTop) > ratio) { maxRight = Math.round(maxTop * ratio); } else { maxTop = Math.round(maxRight / ratio); }
                    } else if (!leftResize && !topResize) {
                        if ((maxRight / maxBottom) > ratio) { maxRight = Math.round(maxBottom * ratio); } else { maxBottom = Math.round(maxRight / ratio); }
                    }
                }
            }

            // calculating the attributes for a resized drawing during the resize action
            function getCurrentDrawingResizeAttributes(localMoveBox, startWidth, startHeight, options) {

                var // the horizontal and vertical scaling factors of the drawing
                    scaleWidth = 1, scaleHeight = 1,
                    // attributes for setting the position of the movebox
                    attrs = {},
                    // local variables for the shift in horizontal and vertical direction
                    localShiftX = shiftX, localShiftY = shiftY,
                    // whether an object with attributes shall be returned
                    getAttributes = Utils.getBooleanOption(options, 'getAttributes', true);

                currentWidth = null;
                currentHeight = null;
                currentLeft = null;
                currentTop = null;

                if (useX && !useY) {
                    localShiftY = 0;
                    currentHeight = startHeight;

                    if (isMultiSelection) {
                        // Modifying the shift for each drawing
                        localShiftX = Math.round(shiftX * startWidth / activeStartWidth);
                    }

                    if (leftResize) {
                        currentLeft = localShiftX;
                        currentWidth = startWidth - localShiftX;
                    } else {
                        currentWidth = startWidth + localShiftX;
                    }
                } else if (useY && !useX) {
                    localShiftX = 0;
                    currentWidth = startWidth;

                    if (isMultiSelection) {
                        // Modifying the shift for each drawing
                        localShiftY = Math.round(shiftY * startHeight / activeStartHeight);
                    }

                    if (topResize) {
                        currentTop = localShiftY;
                        currentHeight = startHeight - localShiftY;
                    } else {
                        currentHeight = startHeight + localShiftY;
                    }
                } else if (useX && useY) {

                    if (isMultiSelection) {
                        // Modifying the shift for each drawing
                        // -> localShiftX and localShiftY are yet adapted to the 'active' drawing
                        localShiftX = Math.round(shiftX * startWidth / activeStartWidth);
                        localShiftY = Math.round(shiftY * startHeight / activeStartHeight);
                    }

                    currentWidth = leftResize ? (startWidth - localShiftX) : (startWidth + localShiftX);
                    currentHeight = topResize ? (startHeight - localShiftY) : (startHeight + localShiftY);

                    // use the same scaling factor for vertical and horizontal resizing, if both are enabled -> the larger factor wins
                    scaleWidth = Utils.round(currentWidth / startWidth, 0.01);
                    scaleHeight = Utils.round(currentHeight / startHeight, 0.01);

                    if (scaleWidth > scaleHeight) {
                        currentHeight = Math.round(scaleWidth * startHeight);
                        localShiftY = topResize ? (startHeight - currentHeight) : (currentHeight - startHeight);
                    } else {
                        currentWidth = Math.round(scaleHeight * startWidth);
                        localShiftX = leftResize ? (startWidth - currentWidth) : (currentWidth - startWidth);
                    }

                    if (leftResize) { currentLeft = localShiftX; }
                    if (topResize) { currentTop = localShiftY; }
                }

                // Modifying the global values of shiftX, if this is not a multi selection. Otherwise shiftX and shiftY
                // need to be calculated for each drawing individually
                if (!isMultiSelection) {
                    shiftX = localShiftX;
                    shiftY = localShiftY;
                }

                // defining resize attributes
                if (_.isNumber(currentLeft)) { attrs.left = currentLeft; }
                if (_.isNumber(currentTop)) { attrs.top = currentTop; }
                if (_.isNumber(currentWidth)) { attrs.width = currentWidth; }
                if (_.isNumber(currentHeight)) { attrs.height = currentHeight; }

                return getAttributes ? attrs : { shiftX: localShiftX, shiftY: localShiftY };
            }

            // updates the position of the drawing frame according to the passed tracking event
            function updateDrawingPosition(event) {

                var // the grid pane layer rectangle
                    layerRectangle = self.getLayerRectangle(),
                    // the current position of the event in the sheet
                    currentX = 0, currentY = 0,
                    // the minimal size for a drawing side in pixel
                    minimumSize = 8;

                if (moveBox) {

                    currentX = event.pageX - layerRootNode.offset().left + layerRectangle.left;
                    currentY = event.pageY - layerRootNode.offset().top + layerRectangle.top;

                    shiftX = currentX - startX;
                    shiftY = currentY - startY;

                    // restricting values of shiftX and shiftY
                    // -> comparing shifts with 'maxTop', 'maxRight', 'maxBottom', 'maxLeft'
                    if ((leftResize || !doResize) && (-shiftX >= maxLeft)) {
                        shiftX = -maxLeft;
                        if (useX && useY) { shiftY = Math.round(shiftX / ratio); }
                    }

                    if ((topResize || !doResize) && (-shiftY >= maxTop)) {
                        shiftY = -maxTop;
                        if (useX && useY) { shiftX = Math.round(shiftY * ratio); }
                    }

                    if (((useX && !leftResize) || !doResize) && (shiftX >= maxRight)) {
                        shiftX = maxRight;
                        if (useX && useY) { shiftY = Math.round(shiftX / ratio); }
                    }

                    if (((useY && !topResize) || !doResize) && (shiftY >= maxBottom)) {
                        shiftY = maxBottom;
                        if (useX && useY) { shiftX = Math.round(shiftY * ratio); }
                    }

                    // ignoring specific shifts  for stretching of drawings
                    if (doResize && useX && !useY) { shiftY = 0; }
                    if (doResize && !useX && useY) { shiftX = 0; }

                    // avoid also decreasing the drawing too much
                    // -> comparing shifts with 'activeStartWidth' and 'activeStartHeight'
                    if (doResize) {
                        if (useX && useY) {
                            if (leftResize && (shiftX >= (activeStartWidth - minimumSize))) {
                                if (topResize && (shiftY >= (activeStartHeight - minimumSize))) {
                                    shiftX = activeStartWidth - minimumSize;
                                    shiftY = Math.round(shiftX / ratio);
                                } else if (!topResize && (-shiftY >= (activeStartHeight - minimumSize))) {
                                    shiftX = activeStartWidth - minimumSize;
                                    shiftY = -Math.round(shiftX / ratio);
                                }
                            } else if (!leftResize && (-shiftX >= (activeStartWidth - minimumSize))) {
                                if (topResize && (shiftY >= (activeStartHeight - minimumSize))) {
                                    shiftX = -activeStartWidth + minimumSize;
                                    shiftY = -Math.round(shiftX / ratio);
                                } else if (!topResize && (-shiftY >= (activeStartHeight - minimumSize))) {
                                    shiftX = -activeStartWidth + minimumSize;
                                    shiftY = Math.round(shiftX / ratio);
                                }
                            }
                        } else {
                            if (leftResize && (shiftX >= (activeStartWidth - minimumSize))) {
                                shiftX = activeStartWidth - minimumSize;
                            } else if (!leftResize && useX && (-shiftX >= (activeStartWidth - minimumSize))) {
                                shiftX = -activeStartWidth + minimumSize;
                            }
                            if (topResize && (shiftY >= (activeStartHeight - minimumSize))) {
                                shiftY = activeStartHeight - minimumSize;
                            } else if (!topResize && useY && (-shiftY >= (activeStartHeight - minimumSize))) {
                                shiftY = -activeStartHeight + minimumSize;
                            }
                        }
                    }

                    // only move the moveBox, if the mouse was really moved
                    mouseMoved = (shiftX !== 0) || (shiftY !== 0);

                    if (mouseMoved) {

                        // make move box visible when tracking position has moved

                        if (isMultiSelection) {
                            _.each(allDrawingFrames, function (drawingFrame) {
                                DrawingFrame.toggleTracking(drawingFrame, true);
                            });
                        } else {
                            DrawingFrame.toggleTracking(drawingFrame, true);
                        }

                        if (_.isNumber(shiftX) && _.isNumber(shiftY) && (shiftX !== 0) || (shiftY !== 0)) {
                            if (!doResize) {
                                // moving the drawing(s)
                                if (isMultiSelection) {
                                    _.each(allDrawingMoveBoxes, function (localMoveBox) {
                                        localMoveBox.css({ left: shiftX, top: shiftY });
                                    });
                                } else {
                                    moveBox.css({ left: shiftX, top: shiftY });
                                }
                            } else {
                                // resizing the drawing(s)
                                if (isMultiSelection) {
                                    _.each(allDrawingMoveBoxes, function (localMoveBox, index) {
                                        localMoveBox.css(getCurrentDrawingResizeAttributes(localMoveBox, allStartWidths[index], allStartHeights[index]));
                                    });
                                } else {
                                    moveBox.css(getCurrentDrawingResizeAttributes(moveBox, activeStartWidth, activeStartHeight));
                                }
                            }
                        }
                    }
                }
            }

            // sets the drawing attributes to the drawing using view.setDrawingAttributes
            function setDrawingAttributes(localMoveBox, localDrawingFrame, index) {

                var // the grid pane layer rectangle
                    layerRectangle = self.getLayerRectangle(),
                    // the new drawing corner positions in pixels
                    newTop = 0, newLeft = 0, newWidth = 0, newHeight = 0,
                    // the shift in each direction specific for each drawing
                    localShiftX = shiftX, localShiftY = shiftY,
                    // the calculated drawing attributes
                    attributes = {},
                    // a helper object to contain drawing specific shifts in multi drawing selections
                    shiftObject = {};

                if (!doResize) {
                    // moving the drawings
                    newLeft = localDrawingFrame.position().left + localShiftX + layerRectangle.left;
                    newTop = localDrawingFrame.position().top + localShiftY + layerRectangle.top;
                    newWidth = localDrawingFrame.width();
                    newHeight = localDrawingFrame.height();
                } else {
                    // resizing the drawings
                    if (isMultiSelection) {
                        // Modifying the shift for each drawing using function from 'updateDrawingPosition'
                        shiftObject = getCurrentDrawingResizeAttributes(localMoveBox, allStartWidths[index], allStartHeights[index], { getAttributes: false });
                        localShiftX = shiftObject.shiftX;
                        localShiftY = shiftObject.shiftY;
                    }

                    newLeft = leftResize ? localDrawingFrame.position().left + localShiftX : localDrawingFrame.position().left;
                    newTop = topResize ? localDrawingFrame.position().top + localShiftY : localDrawingFrame.position().top;
                    newLeft += layerRectangle.left;
                    newTop += layerRectangle.top;
                    newWidth = localMoveBox.outerWidth();
                    newHeight = localMoveBox.outerHeight();
                }

                // modifying attributes
                attributes = docView.getDrawingCollection().getAttributeSetForRectangle({ left: newLeft, top: newTop, width: newWidth, height: newHeight });

                // setting moveBox back into the drawing frame
                localMoveBox.css({ left: 0, top: 0 });

                // generating drawing attributes operation(s)
                if (isMultiSelection) {
                    docView.setDrawingAttributes(attributes, [allDrawingPositions[index]]);
                } else {
                    docView.setDrawingAttributes(attributes);
                }
            }

            // finalizes the drawing tracking
            function finalizeDrawingTracking(event, applyPosition) {

                // generate an operation only, if the drawing was really moved
                if (mouseMoved && moveBox && applyPosition) {
                    if (isMultiSelection) {
                        docModel.getUndoManager().enterUndoGroup(function () {
                            _.each(allDrawingMoveBoxes, function (oneMoveBox, index) {
                                // index can be used for drawingFrames and drawingPositions, too
                                setDrawingAttributes(oneMoveBox, allDrawingFrames[index], index);
                            });
                        });
                    } else {
                        setDrawingAttributes(moveBox, drawingFrame);
                    }
                }

                // Clarify: setting moveBox to null is important for further selections of drawing
                if (isMultiSelection) {
                    _.each(allDrawingFrames, function (drawingFrame) {
                        // make move box invisible after tracking has ended
                        DrawingFrame.toggleTracking(drawingFrame, false);
                    });
                    allDrawingMoveBoxes = [];
                } else {
                    // make move box invisible after tracking has ended
                    DrawingFrame.toggleTracking(drawingFrame, false);
                    moveBox = null;
                }

                if (Utils.TOUCHDEVICE && event.duration > 750 && !mouseMoved) {
                    var selectedDrawings = docView.getSelectedDrawings(),
                        triggerNode = self.getDrawingFrame(selectedDrawings[0]);
                    triggerNode.trigger('contextmenu');
                }
            }

            // return the actual drawingTrackingHandler() method
            return function (event) {

                switch (event.type) {
                    case 'tracking:start':
                        initializeTracking(event);
                        event.preventDefault(); // prevent native scrolling on touch devices
                        break;
                    case 'tracking:move':
                        updateDrawingPosition(event);
                        event.preventDefault(); // prevent native scrolling on touch devices
                        break;
                    case 'tracking:scroll':
                        self.scrollRelative(event.scrollX, event.scrollY);
                        updateDrawingPosition(event);
                        break;
                    case 'tracking:end':
                        updateDrawingPosition(event);
                        finalizeDrawingTracking(event, true);
                        break;
                    case 'tracking:cancel':
                        finalizeDrawingTracking(event, false);
                        break;
                }
            };
        }()); // end of drawingTrackingHandler() local scope

        // --------------------------------------------------------------------

        /**
         * Returns the tracking type according to the target node of the passed
         * 'tracking:start' event.
         */
        function getTrackingType(event) {

            var // target node
                targetNode = $(event.target);

            // ignore inline pop-up menus, and other nodes that handle clicks internally
            if (targetNode.closest('.inline-popup,.skip-tracking').length > 0) {
                return null;
            }

            // drawing layer: select, move, and resize drawing objects
            if (targetNode.closest('.drawing-layer ' + DrawingFrame.NODE_SELECTOR).length > 0) {
                return 'drawing';
            }

            // highlight layer: move and resize highlighted cell ranges
            if (targetNode.closest('.highlight-layer [data-pos]').length > 0) {
                return 'highlight';
            }

            // auto-fill: resize active selection range for auto-fill and auto-delete
            if (targetNode.closest('.selection-layer .autofill [data-pos]').length > 0) {
                return 'autofill';
            }

            // change current selection range with resizer handles on touch devices
            if (targetNode.closest('.selection-layer .select [data-pos]').length > 0) {
                return 'resizeselect';
            }

            // touch devices: scroll, or set new selection
            if (event.trackingType === 'touch') {
                return 'touchselect';
            }

            // set new selection, or extend current selection with mouse and modifier keys
            // (will be ignored on touch devices where a simple tap generates touch and mouse events)
            if (lastTrackingType !== 'touchselect') {
                return 'mouseselect';
            }

            return null;
        }

        /**
         * Handles all tracking events in this grid pane.
         */
        function trackingStartHandler(event) {

            var // type of tracking according to target node
                trackingType = getTrackingType(event),
                // the callback function for the tracking events
                trackingHandler = null;

            // select tracking mode according to event target node
            switch (trackingType) {
                case 'mouseselect':
                    // set new selection, or extend current selection with mouse and modifier keys
                    trackingHandler = mouseSelectionTrackingHandler;
                    readOnlyTracking = true;
                    break;
                case 'touchselect':
                    // touch devices: scroll, or set new selection
                    trackingHandler = touchSelectionTrackingHandler;
                    readOnlyTracking = true;
                    break;
                case 'resizeselect':
                    // change current selection range with resizer handles on touch devices
                    trackingHandler = resizeSelectionTrackingHandler;
                    readOnlyTracking = true;
                    break;
                case 'autofill':
                    // auto-fill: resize active selection range for auto-fill and auto-delete
                    trackingHandler = autoFillTrackingHandler;
                    readOnlyTracking = false;
                    break;
                case 'highlight':
                    // highlight layer: move and resize highlighted cell ranges
                    trackingHandler = highlightTrackingHandler;
                    readOnlyTracking = false;
                    break;
                case 'drawing':
                    // drawing layer: select, move, and resize drawing objects
                    trackingHandler = drawingTrackingHandler;
                    readOnlyTracking = false;
                    break;
                default:
                    // ignore inline pop-up menus, and other nodes that handle clicks internally
                    Tracking.cancelTracking();
                    return;
            }

            // handle all tracking events for this tracking cycle
            PaneUtils.processTrackingCycle(event, trackingHandler, function () {
                lastTrackingType = trackingType;
                docView.grabFocus();
            });

            // Workaround for Firefox, it sends 'focusout' events after the old
            // selection DOM node that has been clicked is removed (the 'focusout'
            // event causes canceling the tracking sometimes).
            self.executeDelayed(function () { self.grabFocus(); });
        }

        /**
         * Handles double clicks triggered by the tracking framework.
         */
        function doubleClickHandler(event) {
            if (getTrackingType(event) === 'mouseselect') {
                self.trigger('select:dblclick');
            }
        }

        // initialization -----------------------------------------------------

        // leave in-place edit mode or cancel tracking, if document goes into read-only mode
        this.listenTo(docModel, 'change:editmode', editModeHandler);

        // tracking for cell selection with mouse/touch
        this.listenToWhenVisible(this, layerRootNode, 'tracking:start', trackingStartHandler);
        this.listenToWhenVisible(this, layerRootNode, 'tracking:dblclick', doubleClickHandler);

        layerRootNode.on('dblclick dragenter dragexit dragover dragleave', false);

        // destroy all class members on destruction
        this.registerDestructor(function () {
            self = docModel = docView = layerRootNode = null;
        });

    } // class GridTrackingMixin

    // exports ================================================================

    return GridTrackingMixin;

});
