/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/control/discretefiltergroup', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/control/checkgroup',
    'io.ox/office/spreadsheet/model/formula/formulautils',
    'io.ox/office/spreadsheet/model/cellcollection',
    'gettext!io.ox/office/spreadsheet/main'
], function (Utils, CheckGroup, FormulaUtils, CellCollection, gt) {

    'use strict';

    var // GUI label of the filter entry representing blank cells
        BLANK_CELLS_LABEL =
            //#. Represents a list entry in spreadsheet data filters that selects all blank (empty) cells in a column
            gt.pgettext('filter', 'Blank cells');

    // class DiscreteFilterGroup ==============================================

    /**
     * A check-list group control that shows all entries of a discrete filter
     * (all different values in a table column).
     *
     * @constructor
     *
     * @extends CheckGroup
     *
     * @param {SpreadsheetView} docView
     *  The spreadsheet view that contains this instance.
     */
    function DiscreteFilterGroup(docView) {

        var // self reference
            self = this;

        // base constructor ---------------------------------------------------

        CheckGroup.call(this, { matcher: matcher, sorted: sortComparator, boxed: true, checkAll: true });

        // private methods ----------------------------------------------------

        /**
         * A case-insensitive list item matcher.
         */
        function matcher(value, entry) {
            return (docView.getApp().isODF() && value === entry) || (_.isString(value) && _.isString(entry) && value.toLowerCase() === entry.toLowerCase());
        }

        /**
         * Compares the passed sort order descriptors stored at the list
         * entries of this control.
         */
        function sortComparator(sortIndex1, sortIndex2) {
            var result = FormulaUtils.compareValues(sortIndex1.result, sortIndex2.result);
            return (result === 0) ? FormulaUtils.compareStrings(sortIndex1.display, sortIndex2.display) : result;
        }

        /**
         * Return a key for the cellData. If it is OOXML it's allways the label String to lower-case.
         * If it's ODF it return the cellData, if the cellData is a String convert it to lower-case.
         * @param {type} cellData to get the ODF key
         * @param {String} label return as the OOXML key to lower-case
         * @param {Boolean} isODF
         * @returns {Object} the key
         */
        function getEntryKey(cellData, label, isODF) {
            var key;
            if (isODF) {
                key = cellData.result + '_';
                if (CellCollection.isBlank(cellData)) {
                    key += 'b';
                } else if (CellCollection.isNumber(cellData)) {
                    key += 'n';
                } else if (CellCollection.isBoolean(cellData)) {
                    key += 'boo';
                } else if (CellCollection.isText(cellData)) {
                    key = key.toLowerCase();
                    key += 's';
                } else {
                    key += 'e';
                }
            } else {
                key = label.toLowerCase();
            }
            return key;
        }

        // public methods -----------------------------------------------------

        /**
         * Collects and inserts the cell entries of a table column into this
         * control.
         *
         * @param {String} tableName
         *  The name of the table model in the active sheet to be filtered.
         *
         * @param {Number} tableCol
         *  The zero-based column index, relative to the table range.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the control has been
         *  initialized.
         */
        this.initialize = function (tableName, tableCol) {

            this.clearOptionButtons();

            // resolve the table model from the document
            var tableModel = docView.getTableCollection().getTable(tableName),
                isODF = docView.getApp().isODF();

            if (!tableModel) {
                Utils.warn('DiscreteFilterGroup.initialize(): table model not found');
                return $.Deferred().reject();
            }

            // unfiltered table column: show entries of all visible rows; filtered table column: show
            // entries for all data rows that are not filtered by other columns (but that may be hidden)
            var promise = tableModel.isColumnFiltered(tableCol) ?
                    tableModel.queryFilterResult({ excludeTableCol: tableCol }) :
                    $.when(tableModel.getVisibleDataRows());

            // receive source data from document
            promise = promise.then(function (rowIntervals) {
                return tableModel.queryColumnData(tableCol, rowIntervals);
            });

            // fill the menu with the values
            promise = promise.then(function (cellArray) {

                var // each display string will be inserted once only ooxml = (case-insensitive, always use first occurrence);
                    // odf = value + ObjectType Suffix e.g. "otto" : key = "otto_s"
                    usedEntries = {},
                    // the filter attributes of the table column
                    attributes = tableModel.getColumnAttributes(tableCol).filter;

                // insert the new entries into the list
                _.each(cellArray, function (cellData) {

                    var // the label displayed in the list entry (preserve case)
                        // bug 36264: trim space characters but no other whitespace
                        label = _.isString(cellData.display) ? cellData.display.replace(/^ +| +$/g, '') : '#######',
                        // the key used to detect duplicates
                        key = getEntryKey(cellData, label, isODF),
                        // options for the list entry
                        options = null;
                        //result && type

                    if (!(key in usedEntries)) {
                        usedEntries[key] = isODF ? cellData.result : key;
                        options = (label.length === 0) ?
                            options = { label: BLANK_CELLS_LABEL, labelStyle: 'font-style:italic;', sortIndex: { result: null, display: '' } } :
                            options = { label: _.noI18n(label), sortIndex: { result: cellData.result, display: label } };
                        self.createOptionButton(isODF ? cellData.result : label.toLowerCase(), options);
                    }
                });

                // initialize list entries according to current filter rules
                self.setValue((attributes.type === 'discrete') ? attributes.entries : _.values(usedEntries));
            });

            return promise;
        };

        // initialization -----------------------------------------------------

        // destroy all class members on destruction
        this.registerDestructor(function () {
            self = docView = null;
        });

    } // class DiscreteFilterGroup

    // exports ================================================================

    // derive this class from class CheckGroup
    return CheckGroup.extend({ constructor: DiscreteFilterGroup });

});
