/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Mario Schroeder <mario.schroeder@open-xchange.com>
 */
define('io.ox/office/spreadsheet/utils/clipboard', [
    'io.ox/office/tk/io',
    'io.ox/office/tk/utils',
    'io.ox/office/drawinglayer/view/drawingframe',
    'io.ox/office/drawinglayer/view/imageutil',
    'io.ox/office/editframework/utils/color',
    'io.ox/office/editframework/utils/hyperlinkutils',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/model/cellcollection',
    'io.ox/office/spreadsheet/model/formula/tokenarray'
], function (IO, Utils, DrawingFrame, ImageUtil, Color, HyperlinkUtils, SheetUtils, CellCollection, TokenArray) {

    'use strict';

    var // convenience shortcuts
        Address = SheetUtils.Address,
        AddressArray = SheetUtils.AddressArray,
        Range = SheetUtils.Range,
        RangeArray = SheetUtils.RangeArray,

        // XML name spaces for Office and Excel custom tags
        EXCEL_NAMESPACES = 'xmlns:o="urn:schemas-microsoft-com:office:office" xmlns:x="urn:schemas-microsoft-com:office:excel" xmlns="http://www.w3.org/TR/REC-html40"',

        // the HTML header
        HTML_HEADER = '<head><meta charset="utf-8"></head>',

        // the localized names of the standard number format
        STANDARD_NUMBER_FORMATS = [
            'Standard',
            'General',
            'Estandarra',
            'Skoueriek',
            'Standaard',
            'Yleinen',
            'Bendras',
            '\u041E\u043F\u0448\u0442\u043E',
            '\u0639\u0627\u062F\u06CC',
            'Geral',
            'Padr\xe3o',
            'Estandar'
        ],

        // regex to check for error codes
        ERROR_CODES_REGEX = /#(DIV\/0|N\/A|NAME|NULL|NUM|REF|VALUE)(\!|\?)/i,

        // regex to check for invalid XML characters; for UTF-16 the high surrogate and the low surrogate need to match.
        INVALID_XML_CHARS_PATTERN = // UTF-8:
                                    '[\\u0000-\\u0008\\u000B-\\u000C\\u000E-\\u001F\\u007F-\\u0084\\u0086-\\u009F\\uFDD0-\\uFDDF]|' +
                                    // UTF-16:
                                    // [#x1FFFE-#x1FFFF], [#x2FFFE-#x2FFFF], [#x3FFFE-#x3FFFF],
                                    '\\uD83F[\\uDFFE-\\uDFFF]|\\uD87F[\\uDFFE-\\uDFFF]|\\uD8BF[\\uDFFE-\\uDFFF]|' +
                                    // [#x4FFFE-#x4FFFF], [#x5FFFE-#x5FFFF], [#x6FFFE-#x6FFFF],
                                    '\\uD8FF[\\uDFFE-\\uDFFF]|\\uD93F[\\uDFFE-\\uDFFF]|\\uD97F[\\uDFFE-\\uDFFF]|' +
                                    // [#x7FFFE-#x7FFFF], [#x8FFFE-#x8FFFF], [#x9FFFE-#x9FFFF],
                                    '\\uD9BF[\\uDFFE-\\uDFFF]|\\uD9FF[\\uDFFE-\\uDFFF]|\\uDA3F[\\uDFFE-\\uDFFF]|' +
                                    // [#xAFFFE-#xAFFFF], [#xBFFFE-#xBFFFF], [#xCFFFE-#xCFFFF],
                                    '\\uDA7F[\\uDFFE-\\uDFFF]|\\uDABF[\\uDFFE-\\uDFFF]|\\uDAFF[\\uDFFE-\\uDFFF]|' +
                                    // [#xDFFFE-#xDFFFF], [#xEFFFE-#xEFFFF], [#xFFFFE-#xFFFFF],
                                    '\\uDB3F[\\uDFFE-\\uDFFF]|\\uDB7F[\\uDFFE-\\uDFFF]|\\uDBBF[\\uDFFE-\\uDFFF]|' +
                                    // [#x10FFFE-#x10FFFF]
                                    '\\uDBFF[\\uDFFE-\\uDFFF]',

        INVALID_XML_CHARS_REGEX = new RegExp(INVALID_XML_CHARS_PATTERN, 'g');

    // private static functions ===============================================

    /**
     * Returns all descendants of the passed root node matching the passed CSS
     * selector. Additionally, the root node itself will be included in the
     * result set, if it matches the selector.
     *
     * @param {HTMLElement|jQuery|String} rootNode
     *  The root node to be searched for matching elements, or HTML mark-up
     *  text to be searched on-the-fly.
     *
     * @returns {jQuery}
     *  A jQuery collection containing all matching elements.
     */
    function collectMatchingNodes(rootNode, selector) {
        rootNode = $(rootNode);
        return rootNode.filter(selector).add(rootNode.find(selector));
    }

    /**
     * Converts a number format code to be used with the mso-number-format style of Excel.
     *
     * @param {String} format
     *  The generic number format code.
     *
     * @returns {String}
     *  The number format code prepared for Excel.
     */
    function convertNumberFormatToExcel(format) {

        // TODO: the token 'standard' may be part of the format code, e.g.: Standard;-Standard;0
        if (format.toLowerCase() === 'standard') {
            return 'General';
        }

        format = Utils.cleanString(format);

        // replace the ampersand with the text &amp; (must be done first!)
        format = format.replace(/&/g, '&amp;');

        if (_.browser.IE) {
            // use ISO Latin-1 code
            format = format
                // replace space or backslash space with the text &#32;
                .replace(/(\\\s|\s)/g, '&#32;')
                // replace backslash double quote with the text &#92;&#34;
                .replace(/\\"/g, '&#92;&#34;')
                // replace backslash apostrophe with the text &#92;&#39;
                .replace(/\\'/g, '&#92;&#39;')
                // replace backslash percent with the text &#92;&#37;
                .replace(/\\%/g, '&#92;&#37;')
                // replace backslash exclamation mark with the text &#92;&#33;
                .replace(/\\!/g, '&#92;&#33;')
                // replace double backslash the text &#92;&#92;
                .replace(/\\\\/g, '&#92;&#92;')
                // replace double quote, Euro symbol, double quote with the text &#92;&#8364; as work around for IE problems with the double quote
                .replace(/"€"/g, '&#92;&#8364;');

        } else {
            // use UTF-8 representation
            format = format
                // replace space or backslash space with the text \\0020;
                .replace(/(\\\s|\s)/g, '\\0020')
                // replace backslash double quote with the text \\005c&#34;
                .replace(/\\"/g, '\\005c&#34;')
                // replace backslash percent with the text \\005c&#37;
                .replace(/\\%/g, '\\005c&#37;')
                // replace backslash exclamation mark with the text \\005c&#33;
                .replace(/\\!/g, '\\005c&#33;')
                // replace double backslash the text \\005c;&#92;
                .replace(/\\\\/g, '\\005c\\005c');
        }

        return format
            // replace the left angle bracket with the text &lt;
            .replace(/</g, '&lt;')
            // replace the right angle bracket with the text &gt;
            .replace(/>/g, '&gt;')
            // replace the double quote character with the text &#34;
            .replace(/"/g, '&#34;')
            // replace the apostrophe with the text &#39; (&apos; is not an HTML entity!)
            .replace(/'/g, '&#39;');
    }

    /**
     * Converts a number format code to be used with the sdnum attribute of Calc.
     *
     * @param {String} format
     *  The generic number format code.
     *
     * @returns {String}
     *  The number format code prepared for Calc.
     */
    function convertNumberFormatToCalc(format) {
        // TODO: the token 'standard' may be part of the format code, e.g.: Standard;-Standard;0
        return (format.toLowerCase() === 'standard') ? 'General' : Utils.escapeHTML(format);
    }

    /**
     * Returns the number format code for the given cell data.
     *
     * @param {Object} cellData
     *  The contents (display string, result, formula) and formatting of the cell.
     *
     * @returns {String|Null}
     *  The number format code.
     */
    function getNumberFormat(numberFormatter, cellData) {
        var numberFormat = cellData && cellData.attributes && cellData.attributes.cell && cellData.attributes.cell.numberFormat;
        return _.isObject(numberFormat) ? numberFormatter.resolveFormatCode(numberFormat) : null;
    }

    /**
     * Returns the number format code for the given cell data to be exported to Excel.
     *
     * @param {Object} cellData
     *  The contents (display string, result, formula) and formatting of the cell.
     *
     * @returns {String|Null}
     *  The number format code.
     */
    function getExcelNumberFormat(numberFormatter, cellData) {
        var formatCode = getNumberFormat(numberFormatter, cellData);
        return formatCode ? convertNumberFormatToExcel(formatCode) : null;
    }

    /**
     * Returns the number format code for the given cell data to be exported to Calc.
     *
     * @param {Object} cellData
     *  The contents (display string, result, formula) and formatting of the cell.
     *
     * @returns {String|undefined}
     *  The number format code.
     */
    function getCalcNumberFormat(numberFormatter, cellData) {
        var formatCode = getNumberFormat(numberFormatter, cellData);
        return formatCode ? convertNumberFormatToCalc(formatCode) : null;
    }

    /**
     * Returns true for the standard number format.
     *
     * @param {String} format
     *  The number format code as String.
     *
     * @returns {Boolean}
     *  Returns true for the standard number format.
     */
    function isStandardNumberFormat(format) {
        return _.any(STANDARD_NUMBER_FORMATS, function (item) {
            return (_.isString(this) && this.indexOf(item.toLowerCase()) > -1);
        }, format.toLowerCase());
    }

    /**
     * Creates a number format cell style object from a Calc number format
     * string.
     *
     * @param {String|Null} format
     *  The number format code as String.
     *
     *  @returns {Object|Null}
     *   The number format object
     */
    function parseCalcNumberFormat(format) {

        if (!_.isString(format)) { return null; }

        if (isStandardNumberFormat(format)) {
            return { id: 0 };
        }

        var numberFormat = format.match(/(?:(\d{4,5}|0);){0,1}(?:(\d{4,5}|0);){0,1}(.*)/),
            language;

        if (_.isArray(numberFormat) && numberFormat[3] && numberFormat[3].length > 0) {
            numberFormat[2] = parseInt(numberFormat[2], 10);
            if (numberFormat[2] > 0) {
                language = numberFormat[2];
            } else {
                numberFormat[1] = parseInt(numberFormat[1], 10);
                if (numberFormat[1] > 0) {
                    language = numberFormat[1];
                } else {
                    language = 1033;
                }
            }

            return { code: numberFormat[3], language: language };
        }

        return null;
    }

    /**
     * Returns the direct child element that matches the given jQuery selector.
     * Is limited to work inside HTML tables, e.g. find a 'strong' as child of a 'td' element.
     *
     * @param {jQuery} element
     *  The jQuery root element to start looking for the child.
     *
     * @returns {jQuery}
     *  The matching child element.
     */
    function findMatchingChild(element, selector) {
        return element.find(selector).filter(function () {
            var parent = $(this).parents('span,a,th,td,tr,tfoot,thead,tbody,colgroup,table').first();
            return parent[0] === element[0];
        });
    }

    /**
     * Returns true if the given jQuery selector matches a direct child element.
     * Is limited to work inside HTML tables, e.g. find a 'strong' as child of a 'td' element.
     *
     * @param {jQuery} element
     *  The jQuery root element to start looking for the child.
     *
     * @returns {Boolean}
     *  Returns true if a child of the given 'element' matches the given 'selector'.
     */
    function hasMatchingChild(element, selector) {
        return findMatchingChild(element, selector).length > 0;
    }

    /**
     * Converts a CSS font size with measurement unit into a HTML font size.
     *
     * @param {String} cssSize
     *  The CSS font size value with its measurement unit to be converted, as String.
     *
     * @returns {Number}
     *  The converted CSS size.
     */
    function convertToFontSize(cssSize) {
        var size = Utils.convertCssLength(cssSize, 'pt');
        if (size > 30) { return 7; }    // 7 -> 36pt
        if (size > 21) { return 6; }    // 6 -> 24pt
        if (size > 16) { return 5; }    // 5 -> 18pt
        if (size > 13) { return 4; }    // 4 -> 14pt
        if (size > 11) { return 3; }    // 3 -> 12pt
        if (size > 9) { return 2; }     // 2 -> 10pt
        if (size > 0) { return 1; }     // 1 -> 8pt
        return 0;
    }

    /**
     * Converts a HTML font size String (used as size attribute inside a font element)
     * into a size in point.
     *
     * @param {String} size
     *  The font size to be converted, as String.
     *
     * @returns {Number}
     *  The converted size, in point.
     */
    function convertFontSizeToPoint(size) {
        switch (size) {
            case '1':
                return 8;
            case '2':
                return 10;
            case '3':
                return 12;
            case '4':
                return 14;
            case '5':
                return 18;
            case '6':
                return 24;
            case '7':
                return 36;
            default:
                return 0;
        }
    }

    /**
     * Returns the character styles as CSS style String.
     *
     * @param {SpreadsheetModel} docModel
     *  The document model.
     *
     * @param {Object} cellData
     *  The contents (display string, result, formula) and formatting of the cell.
     *
     * @returns {String}
     *  The CSS style String.
     */
    function getCssCharacterStyles(docModel, cellData) {

        var // attribute map for the 'character' attribute family
            charAttributes = cellData && cellData.attributes && cellData.attributes.character,
            // the fill color cell attribute
            fillColor = cellData && cellData.attributes && cellData.attributes.cell && cellData.attributes.cell.fillColor,
            // the CSS text decoration
            textDecoration = 'none',
            // the character style attribute data
            characterStyle = '';

        if (!charAttributes) { return ''; }

        if (charAttributes.fontName) { characterStyle += 'font-family:' + docModel.getCssFontFamily(charAttributes.fontName) + ';'; }
        if (charAttributes.fontSize) { characterStyle += 'font-size:' + charAttributes.fontSize + 'pt;'; }

        characterStyle += 'font-weight:' + ((charAttributes.bold) ? 'bold;' : 'normal;');
        characterStyle += 'font-style:' + ((charAttributes.italic) ? 'italic;' : 'normal;');

        if (charAttributes.underline) { textDecoration = Utils.addToken(textDecoration, 'underline', 'none'); }
        if (charAttributes.strike !== 'none') { textDecoration = Utils.addToken(textDecoration, 'line-through', 'none'); }
        characterStyle += 'text-decoration:' + textDecoration + ';';

        if (charAttributes.color) { characterStyle += 'color:' + docModel.getCssTextColor(charAttributes.color, [fillColor]) + ';'; }

        return characterStyle;
    }

    /**
     * Returns the cell styles as CSS style String.
     *
     * @param {SpreadsheetModel} docModel
     *  The document model.
     *
     * @param {Object} cellData
     *  The contents (display string, result, formula) and formatting of the cell.
     *
     * @returns {String}
     *  The CSS style String.
     */
    function getCssCellStyles(docModel, cellData) {

        var // attribute map for the 'cell' attribute family
            cellAttributes = cellData && cellData.attributes && cellData.attributes.cell,
            // the cell style attribute data
            cellStyle = '';

        if (!cellAttributes) { return ''; }

        if (_.contains(['left', 'center', 'right', 'justify'], cellAttributes.alignHor)) { cellStyle += 'text-align:' + cellAttributes.alignHor + ';'; }
        if (_.contains(['top', 'middle', 'bottom', 'justify'], cellAttributes.alignVert)) { cellStyle += 'vertical-align:' + cellAttributes.alignVert + ';'; }

        cellStyle += 'white-space:' + (cellAttributes.wrapText ? 'normal;' : 'nowrap;');
        if (cellAttributes.fillColor) { cellStyle += 'background-color:' + docModel.getCssColor(cellAttributes.fillColor, 'fill') + ';'; }

        ['Top', 'Bottom', 'Left', 'Right'].forEach(function (attrPart) {

            var // the cell style attribute name
                oxAttrName = 'border' + attrPart,
                // the CSS attribute name
                cssAttrName = 'border-' + attrPart.toLowerCase() + ':',
                // the border cell style attribute value
                border = cellAttributes[oxAttrName],
                // the effective CSS attributes
                cssAttrs = docModel.getCssBorderAttributes(border);

            cellStyle += cssAttrName;
            cellStyle += ((cssAttrs && cssAttrs.width > 0) ? cssAttrs.style + ' ' + cssAttrs.width + 'px ' + cssAttrs.color : 'none');
            cellStyle += ';';
        });

        return cellStyle;
    }

    /**
     * Converts the passed formatted cell display text to HTML mark-up.
     *
     * @param {String|Null} display
     *  The formatted cell display text. The value null represents a cell that
     *  cannot be formatted to a valid display string with its current number
     *  format.
     *
     * @returns {String}
     *  The encoded display text ready to be inserted into an HTML mark-up
     *  string. The null value will be converted to an empty string.
     */
    function escapeDisplayString(display) {
        // nothing to do for empty strings or null values
        return display ? Utils.escapeHTML(display) : '';
    }

    /**
     * Returns the character styles as a String of the HTML elements tree.
     *
     * @param {SpreadsheetModel} docModel
     *  The document model.
     *
     * @param {CellDescriptor} cellDesc
     *  The contents (display string, result, formula) and formatting of the
     *  cell.
     *
     * @param {String} [url]
     *  The URL of the hyperlink attached to the cell.
     *
     * @returns {String}
     *  The HTML elements String.
     */
    function getHtmlCharacterStyles(docModel, cellDesc, url) {

        var // attribute map for the 'character' attribute family
            charAttrs = cellDesc.attributes.character,
            // the fill color cell attribute
            fillColor = cellDesc.attributes.cell.fillColor,
            // the character style HTML elements
            characterStyle = '';

        // create opening tags
        if (charAttrs.bold) { characterStyle += '<b>'; }
        if (charAttrs.italic) { characterStyle += '<i>'; }
        if (charAttrs.underline) { characterStyle += '<u>'; }
        if (charAttrs.strike !== 'none') { characterStyle += '<s>'; }

        if (url) { url = HyperlinkUtils.checkForHyperlink(url); }
        if (url) { characterStyle += '<a href="' + url + '">'; }

        characterStyle += '<font ';
        // To make it work in Calc we need to use just the font name instead of the entire CSS font family
        if (charAttrs.fontName) { characterStyle += 'face="' + Utils.escapeHTML(charAttrs.fontName) + '" '; }
        // We need to use rgb colors, like #RRGGBB
        if (charAttrs.color) { characterStyle += 'color="' + docModel.getCssTextColor(charAttrs.color, [fillColor]) + '" '; }
        // Add the font-size CSS also here to support precise font sizes in Excel
        if (charAttrs.fontSize) { characterStyle += 'size="' + convertToFontSize(charAttrs.fontSize) + '" style="font-size:' + charAttrs.fontSize + 'pt;"'; }
        characterStyle += '>';

        // add cell value (nothing to do for empty strings or null
        characterStyle += escapeDisplayString(cellDesc.display);

        // create closing tags
        characterStyle += '</font>';
        if (url) { characterStyle += '</a>'; }
        if (charAttrs.strike !== 'none') { characterStyle += '</s>'; }
        if (charAttrs.underline) { characterStyle += '</u>'; }
        if (charAttrs.italic) { characterStyle += '</i>'; }
        if (charAttrs.bold) { characterStyle += '</b>'; }

        return characterStyle;
    }

    /**
     * Returns the cell styles as HTML attributes String.
     *
     * @param {SpreadsheetModel} docModel
     *  The document model.
     *
     * @param {CellDescriptor} cellDesc
     *  The contents (display string, result, formula) and formatting of the
     *  cell.
     *
     *  @returns {String}
     *   The HTML attributes String.
     */
    function getHtmlCellStyles(docModel, cellDesc) {

        var // attribute map for the 'cell' attribute family
            cellAttrs = cellDesc.attributes.cell,
            // the cell style attribute data
            cellStyle = '';

        if (/^(left|center|right|justify)$/.test(cellAttrs.alignHor)) { cellStyle += ' align="' + cellAttrs.alignHor + '"'; }
        if (/^(top|middle|bottom)$/.test(cellAttrs.alignVert)) { cellStyle += ' valign="' + cellAttrs.alignVert + '"'; }
        if (!Color.parseJSON(cellAttrs.fillColor).isAuto()) { cellStyle += ' bgcolor="' + docModel.getCssColor(cellAttrs.fillColor, 'fill') + '"'; }

        return cellStyle;
    }

    /**
     * Returns the attribute map for the 'character' attribute family of the HTML cell.
     *
     * @param {jQuery} element
     *  The HTML element.
     *
     *  @returns {Object}
     *   The character attribute map.
     */
    function getCharacterAttributes(element) {

        var // attribute map for the 'character' attribute family
            charAttrs = {},
            // the <font> element, if HTML elements are used instead of CSS
            fontElement = findMatchingChild(element, 'font');

        var fontName = fontElement.attr('face') || element.css('font-family');
        if (_.isString(fontName) && fontName.length > 0) {
            fontName = (fontName.split(',')[0]).replace(/["']/g, '');
            charAttrs.fontName = fontName;
        }

        var fontSize = convertFontSizeToPoint(fontElement.attr('size')) || Utils.convertCssLength(element.css('font-size'), 'pt');
        if (fontSize > 0) {
            charAttrs.fontSize = fontSize;
        }

        if (hasMatchingChild(element, 'em,i')) {
            charAttrs.italic = true;
        } else {
            var fontStyle = element.css('font-style');
            if (_.isString(fontStyle) && fontStyle.length > 0) {
                charAttrs.italic = fontStyle === 'italic';
            }
        }

        if (hasMatchingChild(element, 'strong,b')) {
            charAttrs.bold = true;
        } else {
            var fontWeight = element.css('font-weight');
            if (_.isString(fontWeight) && fontWeight.length > 0) {
                charAttrs.bold = /^(bold|bolder)$/.test(fontWeight) || (parseInt(fontWeight, 10) >= 600);
            }
        }

        var textDecoration = element.css('text-decoration');
        if (hasMatchingChild(element, 'u')) {
            charAttrs.underline = true;
        } else if (_.isString(textDecoration) && textDecoration.length > 0) {
            charAttrs.underline = textDecoration.indexOf('underline') >= 0;
        }
        if (hasMatchingChild(element, 's')) {
            charAttrs.strike = 'single';
        } else if (_.isString(textDecoration) && textDecoration.length > 0) {
            charAttrs.strike = (textDecoration.indexOf('line-through') >= 0) ? 'single' : 'none';
        }

        var color = Color.parseCSS(fontElement.attr('color') || element.css('color'));
        if (color) {
            charAttrs.color = color.toJSON();
        }

        return { character: charAttrs };
    }

    /**
     * Returns the attribute map for the 'cell' attribute family of the HTML element.
     *
     * @param {jQuery} element
     *  The HTML element.
     *
     *  @returns {Object}
     *   The cell attribute map.
     */
    function getCellAttributes(element) {

        var // attribute map for the 'cell' attribute family
            cellAttrs = {};

        // map CSS border style to operation border line style
        function mapCssBorderStyle(style) {
            switch (style) {
                case 'solid':
                case 'groove':
                case 'ridge':
                case 'inset':
                case 'outset':
                    return 'single';
                case 'double':
                    return 'double';
                case 'dotted':
                    return 'dotted';
                case 'dashed':
                    return 'dashed';
                default:
                    return 'none';
            }
        }

        // check value and filter prefix (webkit-, moz-, ms-)
        var alignHor = element.attr('align') || element.css('text-align');
        alignHor = _.isString(alignHor) ? /(?:^|-)(left|center|right|justify)$/i.exec(alignHor) : null;
        if (alignHor) { cellAttrs.alignHor = alignHor[1].toLowerCase(); }

        // check value and filter prefix (webkit-, moz-, ms-)
        var alignVert = element.attr('valign') || element.css('vertical-align');
        alignVert = _.isString(alignVert) ? /(?:^|-)(top|middle|bottom|justify)$/i.exec(alignVert) : null;
        if (alignVert) { cellAttrs.alignVert = alignVert[1].toLowerCase(); }

        var wrapText = element.css('white-space');
        if (_.isString(wrapText) && wrapText.length > 0) {
            cellAttrs.wrapText = wrapText !== 'nowrap';
        }

        var fillColor = Color.parseCSS(element.attr('bgcolor') || element.css('background-color'), true);
        if (fillColor) {
            cellAttrs.fillColor = fillColor.toJSON();
        }

        _.each(['Top', 'Bottom', 'Left', 'Right'], function (attrPart) {

            var // the element style attribute name
                oxAttrName = 'border' + attrPart,
                // the CSS attribute name
                cssAttrName = 'border-' + attrPart.toLowerCase(),
                // the border style
                borderStyle = mapCssBorderStyle(element.css(cssAttrName + '-style')),
                // the border width
                borderWidth,
                // the border color
                borderColor;

            if (borderStyle !== 'none') {

                if (_.browser.Firefox && element[0]) {
                    // On FF element.css(cssAttrName + '-width') is always 0px, bug #35125
                    borderWidth = element[0].style[oxAttrName + 'Width'] || element.css(cssAttrName + '-width');
                } else {
                    borderWidth = element.css(cssAttrName + '-width');
                }
                borderColor = Color.parseCSS(element.css(cssAttrName + '-color'));

                cellAttrs[oxAttrName] = {
                    style: borderStyle,
                    width: Utils.convertCssLengthToHmm(borderWidth),
                    color: borderColor ? borderColor.toJSON() : Color.AUTO
                };
            }

        });

        var numberFormat = parseCalcNumberFormat(element.attr('sdnum'));
        if (numberFormat) {
            cellAttrs.numberFormat = numberFormat;
        }

        return { cell: cellAttrs };
    }

    /**
     * Creates a DOM node for a drawing object, and adds data attributes
     * describing the position of the drawing object relative to the bounding
     * rectangle containing all drawing objects copied to the clipboard.
     */
    function createDataNodeFor(drawingModel, rectangle, boundRectangle, modelData) {

        // the data node to be returned
        var dataNode = $('<div data-type="' + drawingModel.getType() + '">');

        // adjust rectangle relative to bounding rectangle, and the explicit attributes
        dataNode.attr({
            'data-left': rectangle.left - boundRectangle.left,
            'data-top': rectangle.top - boundRectangle.top,
            'data-width': rectangle.width,
            'data-height': rectangle.height,
            'data-attrs': JSON.stringify(drawingModel.getExplicitAttributes())
        });

        // add passed custom JSON data
        if (modelData) { dataNode.attr('data-json', JSON.stringify(modelData)); }

        return dataNode;
    }

    /**
     * Returns the BASE64 encoded image data URL, whose data URL is for the
     * given (image) drawing attributes.
     *
     * @param {Object} mergedAttr
     *  The merged image drawing attributes, containing the image width, height
     *  and source data or URL.
     *
     * @returns {String}
     *  The image data URL, containing the image data.
     */
    function getImageDataUrl(app, mergedAttr) {

        var imageAttr = mergedAttr.image,
            imageUrl = null;

        if (imageAttr.imageData) {
            imageUrl = imageAttr.imageData;
        } else if (imageAttr.imageUrl) {

            if (/^(http:\/\/|ftp:)/.test(imageAttr.imageUrl)) {
                imageUrl = imageAttr.imageUrl;
            } else {
                var canvas = document.createElement('canvas'),
                    jqImageNode =  $('<img>', { src: app.getServerModuleUrl(IO.FILTER_MODULE_NAME, {
                        action: 'getfile',
                        get_filename: imageAttr.imageUrl
                    }) });

                canvas.width = jqImageNode[0].naturalWidth || jqImageNode.width() || Math.floor(mergedAttr.drawing.width * 96 / 2540.0);
                canvas.height = jqImageNode[0].naturalHeight || jqImageNode.height() || Math.floor(mergedAttr.drawing.height * 96 / 2540.0);

                // retrieve image data as image/png to avoid
                // quality losses with lossy, compressed formats
                try {
                    var canvasCtx = canvas.getContext('2d');
                    canvasCtx.drawImage(jqImageNode[0], 0, 0);
                    imageUrl = canvas.toDataURL('image/png');
                } catch (ex) {
                    Utils.warn('cannot convert image to data URL');
                }
            }
        }

        return imageUrl || 'data:,';
    }

    /**
     * Returns the CSS style Object for the line/border
     * attributes of the image.
     *
     * @param {Object} mergedAttr
     *  The merged image attributes, containing the line attrs
     *
     * @returns {Object}
     *  The CSS style object, containing line/border attributes.
     */
    function getImageLineStyle(app, mergedAttr) {
        var lineAttr = mergedAttr.line;

        return (lineAttr.type === 'solid' ? {
            borderStyle: lineAttr.style,
            borderWidth: (lineAttr.width * 0.001).toString() + 'cm',
            borderColor: app.getModel().getCssColor(lineAttr.color, 'line')
        } : null);
    }

    /**
     * Creates a HTML node, containing the data, necessary to
     * reconstruct an image object later in time (e.g. at paste)
     */
    function createDataNodeForImage(app, imageModel, rectangle, boundRectangle) {

        // create the DOM node representing the image object
        var dataNode = createDataNodeFor(imageModel, rectangle, boundRectangle),
            imageNode = $('<img>'),
            mergedAttr = imageModel.getMergedAttributes(),
            lineStyle = getImageLineStyle(app, mergedAttr);

        // add image attributes
        imageNode.attr({
            width: rectangle.width,
            height: rectangle.height,
            src: getImageDataUrl(app, mergedAttr)
        });

        // add image border style, if set
        if (lineStyle) {
            imageNode.css(lineStyle);
        }

        return dataNode.append(imageNode);
    }

    /**
     *
     */
    function createDataNodeForChart(chartModel, rectangle, boundRectangle) {

        // the JSON data to be transported in the DOM node
        var chartData = {
            modelData: chartModel.getCloneData()
        };

        // create the DOM node representing the chart object
        var dataNode = createDataNodeFor(chartModel, rectangle, boundRectangle, chartData);

        // try to create a replacement image that will be used when pasting into other documents
        dataNode.append(chartModel.getReplacementNode({ mimeType: 'image/png' }));

        return dataNode;
    }

    // static class Clipboard =================================================

    var Clipboard = {};

    // static methods ---------------------------------------------------------

    /**
     * Creates a unique client clipboard id.
     *
     * @returns {String}
     *  The client clipboard id.
     */
    Clipboard.createClientClipboardId = function () {
        return ox.session + ':' + _.uniqueId();
    };

    /**
     * Returns whether the passed value is an error code.
     *
     * @param {String} value
     *  Any literal value that can be used in formulas.
     *
     * @returns {Boolean}
     *  Whether the passed value is an error code.
     */
    Clipboard.isErrorCode = function (value) {
        return ERROR_CODES_REGEX.test(value);
    };

    /**
     * Creates a HTML text from the selected drawings.
     *
     * @param {Array<DrawingModel>} drawingModels
     *  The array of selected drawing models
     *
     * @returns {String|Null}
     *  The HTML mark-up of a DOM container node with data nodes for the passed
     *  drawing objects. If none of the drawing objects could be converted to
     *  DOM nodes, null will be returned instead.
     */
    Clipboard.convertDrawingModels = function (app, drawingModels) {

        var // the positions of all drawing objects
            rectangles = _.invoke(drawingModels, 'getRectangle'),
            // the bounding rectangle of all drawings for the top-left position
            boundRectangle = Utils.getBoundingRectangle(rectangles),
            // the resulting container up for all drawing obejcts
            containerNode = $('<div>');

        // process all drawing models
        drawingModels.forEach(function (drawingModel, index) {

            var // the location of the drawing object (already calculated above)
                rectangle = rectangles[index],
                // the resulting data node
                dataNode = null;

            // generate the DOM representation of the drawing model
            switch (drawingModel.getType()) {
                case 'image':
                    dataNode = createDataNodeForImage(app, drawingModel, rectangle, boundRectangle);
                    break;
                case 'chart':
                    dataNode = createDataNodeForChart(drawingModel, rectangle, boundRectangle);
                    break;
            }

            // insert the data node into the container
            if (dataNode) { containerNode.append(dataNode); }
        });

        // nothing valid to copy
        if (containerNode.children().length === 0) { return ''; }

        // initialize the container node
        containerNode.attr({
            id: 'ox-clipboard-data',
            'data-app-guid': app.getGlobalUid(),
            'data-ox-drawings': true
        });

        return '<html ' + EXCEL_NAMESPACES + '>' + HTML_HEADER + '<body>' + containerNode[0].outerHTML + '</body></html>';
    };

    /**
     * Creates a HTML table from the active range of the current selection.
     *
     * @param {SheetModel} sheetModel
     *  The sheet model instance containing the cells.
     *
     * @param {Range} range
     *  The range address.
     *
     * @param {String} clientClipboardId
     *  The client clipboard id to identify the copy data when pasting.
     *
     * @returns {String}
     *  The resulting HTML mark-up of the <table> element.
     */
    Clipboard.getHtmlMarkupFromRange = function (sheetModel, range, clientClipboardId) {

        var // the document model
            docModel = sheetModel.getDocModel(),
            // the number formatter of the document
            numberFormatter = docModel.getNumberFormatter(),
            // the cell collection of the passed sheet
            cellCollection = sheetModel.getCellCollection(),
            // the formula parser
            tokenArray = new TokenArray(sheetModel, { temp: true }),
            // the current table row
            row = range.start[1],
            // the tableMarkup string
            tableMarkup = '';

        tableMarkup += '<html ' + EXCEL_NAMESPACES + '>' + HTML_HEADER;
        tableMarkup += '<body><table id="ox-clipboard-data"';

        if (clientClipboardId) {
            tableMarkup += ' data-ox-clipboard-id="' + clientClipboardId + '"';
        }

        tableMarkup += ' data-app-guid="' + docModel.getApp().getGlobalUid() + '"';
        tableMarkup += ' data-range-start="' + range.start[0] +  ';' + range.start[1] + '"';
        tableMarkup += '>';
        tableMarkup += '<tbody><tr>';

        cellCollection.iterateCellsInRanges(range, function (cellDesc, origRange, mergedRange) {

            var address = cellDesc.address,
                isReferenceCell,
                colspan,
                rowspan,
                style,
                excelNumberFormat,
                calcNumberFormat,
                formula;

            // check if the cell is the reference cell of a merged collection
            isReferenceCell = mergedRange && mergedRange.startsAt(address);

            // check if we need to add a new row to the tableMarkup
            if (row !== address[1]) {
                tableMarkup += '</tr><tr>';
                row = address[1];
            }

            // render cell if it's the reference cell or if it's not a merged cell
            if (isReferenceCell || !mergedRange) {

                tableMarkup += '<td ';

                if (isReferenceCell) {
                    // add cell with colspan and rowspan
                    colspan = mergedRange.end[0] - mergedRange.start[0] + 1;
                    rowspan = mergedRange.end[1] - mergedRange.start[1] + 1;

                    tableMarkup += (colspan > 1) ? 'colspan="' + colspan + '" ' : '';
                    tableMarkup += (rowspan > 1) ? 'rowspan="' + rowspan + '" ' : '';
                }

                // add cell and character style attributes
                style = Utils.escapeHTML(getCssCharacterStyles(docModel, cellDesc) + getCssCellStyles(docModel, cellDesc));

                // handle Excel number format
                excelNumberFormat = getExcelNumberFormat(numberFormatter, cellDesc);
                if (excelNumberFormat) {
                    style += 'mso-number-format:\'' + excelNumberFormat + '\';';
                }

                if (style.length > 0) {
                    tableMarkup += ' style="' + style + '"';
                }

                // set the cell value type
                if (CellCollection.isNumber(cellDesc)) {
                    tableMarkup += ' x:num="' + cellDesc.result + '"';
                    tableMarkup += ' sdval="' + cellDesc.result + '"';
                } else if (CellCollection.isBoolean(cellDesc)) {
                    tableMarkup += ' x:bool="' + cellDesc.result + '"';
                } else if (CellCollection.isError(cellDesc)) {
                    tableMarkup += ' x:err="' + escapeDisplayString(cellDesc.display) + '"';
                } else if (CellCollection.isText(cellDesc)) {
                    tableMarkup += ' x:str="' + escapeDisplayString(cellDesc.result) + '"';
                }

                // handle Calc number format
                calcNumberFormat = getCalcNumberFormat(numberFormatter, cellDesc);
                if (calcNumberFormat) {
                    tableMarkup += ' sdnum="1033;1033;' + calcNumberFormat + '"';
                }

                var oxCellData = {};

                // handle formula
                if (cellDesc.formula) {
                    oxCellData.formula = cellDesc.formula;

                    // relocate all formulas relative to A1
                    tokenArray.parseFormula('ui', cellDesc.formula);
                    formula = tokenArray.getFormula('ui', { refAddress: range.start, targetAddress: Address.A1 });
                    Utils.log('external clipboard - formula - range:', range, ', original: ', cellDesc.formula, ', relocated: ', formula);

                    if (!Clipboard.isErrorCode(formula)) {
                        tableMarkup += ' x:fmla="' + Utils.escapeHTML(formula) + '"';
                        tableMarkup += ' formula="' + Utils.escapeHTML(formula) + '"';
                    }
                }

                oxCellData.attributes = {
                    apply: cellDesc.attributes.apply,
                    cell: cellDesc.attributes.cell,
                    character: cellDesc.attributes.character
                    //FIXME: no styleId, there is a bug in calc!!!
                    //styleId: cellDesc.attributes.styleId
                };
                oxCellData.explicit = cellDesc.explicit;
                oxCellData.result = cellDesc.result;

                tableMarkup += ' ox-cell-data="' + Utils.escapeHTML(JSON.stringify(oxCellData)) + '"';

                // add HTML attributes
                tableMarkup += getHtmlCellStyles(docModel, cellDesc);

                tableMarkup += '>';

                // add HTML elements and cell content
                var url = cellCollection.getCellURL(address);
                tableMarkup += getHtmlCharacterStyles(docModel, cellDesc, url);

                tableMarkup += '</td>';
            }
        }, { merged: true, ordered: true });

        tokenArray.destroy();
        tableMarkup += '</tr></tbody></table></body></html>';

        return tableMarkup;
    };

    /**
     * Creates a plain text representation from the active range of the current
     * selection.
     *
     * @param {SheetModel} sheetModel
     *  The sheet model instance containing the cells.
     *
     * @param {Range} range
     *  The range address.
     *
     * @returns {String}
     *  The resulting plain text string.
     */
    Clipboard.getPlainTextFromRange = function (sheetModel, range) {

        var // the resulting TAB-separated display strings
            resultText = '';

        sheetModel.getCellCollection().iterateCellsInRanges(range, function (cellDesc) {
            if (cellDesc.address[0] > range.start[0]) {
                resultText += '\t';
            } else if (cellDesc.address[1] > range.start[1]) {
                resultText += '\r\n';
            }
            resultText += cellDesc.display || '';
        }, { ordered: true });

        return resultText;
    };

    /**
     * Retrieves the client clipboard ID from the HTML data.
     *
     * @param {HTMLElement|jQuery|String} rootNode
     *  The root node to be searched for the clipboard identifier, or HTML
     *  mark-up text to be searched on-the-fly.
     *
     * @returns {String|Null}
     *  The client clipboard ID or undefined.
     */
    Clipboard.getClientClipboardId = function (rootNode) {
        var clipboardId = collectMatchingNodes(rootNode, '#ox-clipboard-data').attr('data-ox-clipboard-id');
        return clipboardId ? clipboardId : null;
    };

    /**
     * Retrieves the GUID of the source application from the passed HTML data.
     *
     * @param {HTMLElement|jQuery|String} rootNode
     *  The root node to be searched for the application identifier, or HTML
     *  mark-up text to be searched on-the-fly.
     *
     * @returns {String|Null}
     *  The GUID of the source application from the HTML data; or null, if no
     *  identifier has been found.
     */
    Clipboard.getApplicationGuid = function (rootNode) {
        var appGuid = collectMatchingNodes(rootNode, '#ox-clipboard-data').attr('data-app-guid');
        return appGuid ? appGuid : null;
    };

    /**
     * Returns true if the HTML contains exactly one element of the given type.
     *
     * @param {String|jQuery} html
     *  The HTML data to check.
     *
     * @param {String} query
     *  The element to search the hml node for,
     *  but only one arbitrary level deep
     *
     * @returns {jQuery|Null}
     *  Returns the matched jQuery object if such an object is found.
     *  or null otherwise
     */
    Clipboard.getContainedHtmlElements = function (html, query) {
        var jqResult = html.filter(query);
        if (!jqResult.length) {
            jqResult = html.find(query);
        }
        return (jqResult.length > 0) ? jqResult : null;
    };

    /**
     * Removes characters that are invalid in XML from the given string
     * and returns a new valid string.
     *
     * @param {String} inputStr
     *  The string to clean up.
     *
     * @returns {String}
     *  The result string containing only valid XML characters.
     */
    Clipboard.removeInvalidXmlChars = function (inputStr) {
        if (!_.isString(inputStr) || inputStr.length === 0) {
            return '';
        }
        return inputStr.replace(INVALID_XML_CHARS_REGEX, '');
    };

    /**
     * Parses the HTML from the clipboard data for a table and
     * creates the cell contents and merged cell collection.
     *
     * @param {SpreadsheetView} docView
     *  The document view that processes the paste event.
     *
     * @param {String|jQuery} html
     *  The HTML input to parse.
     *
     * @returns {Object}
     *  An object with the following attributes:
     *  - {Array} contents
     *      A two-dimensional array containing the values and attribute sets
     *      for the cells.
     *  - {RangeArray} mergedRanges
     *      An array of merged cell ranges.
     *  - {Object} urlMap
     *      A map with URLs as map keys, and range arrays (instances of the
     *      class RangeArray) as values.
     *
     */
    Clipboard.parseHTMLData = function (docView, html) {

        // convert HTML mark-up string to DOM tree
        html = $(html);

        var // the document model
            docModel = docView.getDocModel(),
            // a two dimensional array containing the cell contents
            contents = [],
            // an array of ranges of merged cells
            mergedRanges = new RangeArray(),
            // the table node
            table,
            // the row nodes
            rows,
            // the number of columns
            colCount,
            // the default character and cell attributes
            defaultAttrs = docModel.getDefaultAttributes(['character', 'cell']),
            // a token array used to parse formula expressions
            tokenArray = new TokenArray(docView.getSheetModel(), { temp: true }),
            // start address of the active range in the selection
            targetStart = docView.getSelection().activeRange().start,
            // start address of the clipboard range
            sourceStart = null,
            //
            intern = false, lastFormula = null,
            // a map for all URLs found in the pasted contents, values are address arrays
            urlMap = {};

        // Bug 35126: Excel sends CSS rules containing fractional border widths, e.g.
        // 'border-top: .5pt solid ...' which will be translated to 0px borders in Chrome.
        // Only solution is to manually parse and modify the <style> element contained in
        // the pasted HTML data.
        html.find('>style').each(function () {

            // the first and only child node of the <style> element must be a text node
            if ((this.childNodes.length !== 1) || (this.firstChild.nodeType !== 3)) { return; }

            var // the text contents of the <style> node
                cssText = this.firstChild.nodeValue;

            // text must be a (multi-line) HTML comment
            if (!/^\s*<!--(.|[\r\n])+-->\s*$/.test(cssText)) { return; }

            // replace fractional line width in all border specifications with 1px
            cssText = cssText.replace(/([\r\n]\s*\{?(?:border(?:-top|-bottom|-left|-right)?|mso-diagonal-(?:down|up)):\s*)\.\d+pt /g, '$11px ');

            // replace non-standard line style 'hairline' with solid lines
            cssText = cssText.replace(/([\r\n]\s*\{?(?:border(?:-top|-bottom|-left|-right)?|mso-diagonal-(?:down|up)):.*?) hairline /g, '$1 solid ');

            // write the modified CSS rules back to the <style> element
            this.firstChild.nodeValue = cssText;
        });

        //TODO: use of Clipboard.getContainedHtmlElements(html, query)
        table = html.find('table');
        if (table.length === 0) {
            table = html.filter('table');
        }

        // merge in the character and cell attributes of the HTML table
        delete defaultAttrs.cell.numberFormat;
        docModel.extendAttributes(defaultAttrs, getCharacterAttributes(table));
        docModel.extendAttributes(defaultAttrs, getCellAttributes(table));

        if (table.attr('data-app-guid') === docModel.getApp().getGlobalUid()) {
            //internal clipboard
            intern = true;
            sourceStart = table.attr('data-range-start');
        }

        if (sourceStart) {
            sourceStart = sourceStart.split(';');
        } else {
            sourceStart = [0, 0];
        }
        sourceStart = new Address(sourceStart[0], sourceStart[1]);

        // make sure that table header, body and footer are in the correct order
        rows = _.union(table.find('thead tr').get(), table.find('tbody tr').get(), table.find('tfoot tr').get());

        colCount = _.max(_.map(rows, function (row) { return $(row).find('td,th').length; }));

        _.each(rows, function (row, rowId) {

            var colId,
                rowData = [],
                cells,
                cell,
                cellId = 0,
                cellData,
                cellValue,
                colspan,
                rowspan,
                isMergedCell,
                rowParent,
                rowAttrs,
                linkChild;

            row = $(row);
            rowParent = row.parents('thead, tbody, tfoot, table').first();

            rowAttrs = getCharacterAttributes(rowParent);
            docModel.extendAttributes(rowAttrs, getCellAttributes(rowParent));
            docModel.extendAttributes(rowAttrs, getCharacterAttributes(row));
            docModel.extendAttributes(rowAttrs, getCellAttributes(row));

            cells = row.find('td,th');

            for (colId = 0; colId < colCount; colId++) {

                var address = new Address(colId, rowId);

                isMergedCell = mergedRanges.containsAddress(address);

                if (isMergedCell) {
                    // the cell is part of a merged collection, but not the reference cell
                    cellData = {};

                } else {
                    // the cell is not merged or it is the reference cell
                    cell = $(cells.get(cellId));
                    colspan = cell.attr('colspan');
                    rowspan = cell.attr('rowspan');

                    if (colspan || rowspan) {
                        // the cell is merged and it's a reference cell
                        colspan = (colspan && colspan.length > 0) ? parseInt(colspan, 10) : 1;
                        rowspan = (rowspan && rowspan.length > 0) ? parseInt(rowspan, 10) : 1;

                        mergedRanges.push(Range.create(colId, rowId, colId + colspan - 1, rowId + rowspan - 1));
                    }

                    // look for Calc cell value
                    cellValue = cell.attr('sdval');
                    if (_.isString(cellValue) && cellValue.length > 0) {
                        cellValue = parseFloat(cellValue, 10);
                    }
                    if (!_.isNumber(cellValue) || _.isNaN(cellValue)) {
                        cellValue = cell.text();
                        // if the cell value contains '\n' we need to make sure that we keep only the line feeds
                        // the user wants and not those Excel added by surprise.
                        if ((cellValue.indexOf('\n') > -1) || (cell.find('br').length > 0)) {
                            // remove line feeds from HTML and leave only the <br> elements
                            cell.html(cell.html().replace(/<br\s*\/?>([^\n]*)\n +/gi, '<br>$1').replace(/\n /g, ''));
                            // make '\n' out of the <br> elements
                            cell.find('br').replaceWith('\n');
                            // finally the cell value contains only the user line feeds
                            cellValue = cell.text();
                        }
                    }

                    cellData = {
                        value: Clipboard.removeInvalidXmlChars(cellValue),
                        attrs: _.copy(defaultAttrs, true)
                    };

                    // merge in the character and cell attributes of the HTML row and cell
                    docModel.extendAttributes(cellData.attrs, rowAttrs);
                    docModel.extendAttributes(cellData.attrs, getCharacterAttributes(cell));
                    docModel.extendAttributes(cellData.attrs, getCellAttributes(cell));

                    // insert hyperlinks
                    var url = findMatchingChild(cell, 'a').attr('href');
                    if (_.isString(url) && url.length > 0) {
                        if ((url = HyperlinkUtils.checkForHyperlink(url))) {
                            (urlMap[url] || (urlMap[url] = new AddressArray())).push(address);
                        }
                    }

                    var oxCellData = cell.attr('ox-cell-data');
                    if (oxCellData) {
                        oxCellData = JSON.parse(oxCellData);
                        if (oxCellData.formula) {
                            if (intern) {
                                //internal clipboard
                                tokenArray.parseFormula('ui', oxCellData.formula);
                                cellData.value = tokenArray.getFormula('ui', { refAddress: sourceStart, targetAddress: targetStart });
                                lastFormula = cellData.value;
                                Utils.log('clipboard - formula - range.start: ', sourceStart, targetStart, ', original: ', oxCellData.formula, ', relocated: ', cellData.value);
                            } else {
                                //external clipboard but also spreadsheet
                                var relocated = cell.attr('formula');
                                if (relocated) {
                                    tokenArray.parseFormula('ui', relocated);
                                    cellData.value = tokenArray.getFormula('ui', { refAddress: sourceStart, targetAddress: targetStart });
                                    lastFormula = cellData.value;
                                } else {
                                    cellData.value = oxCellData.result;
                                }
                            }

                        } else {
                            cellData.value = oxCellData.result;
                        }
                        docModel.extendAttributes(cellData.attrs, oxCellData.attributes);
                        cellData.explicit = oxCellData.explicit;
                    }

                    // merge in the hyperlink's character attributes that are attached to it's child span
                    linkChild = findMatchingChild(cell, 'a').children('span').first();
                    if (linkChild.length > 0) {
                        docModel.extendAttributes(cellData.attrs, getCharacterAttributes(linkChild));
                    }

                    cellId++;
                }

                rowData.push(cellData);
            }

            contents.push(rowData);
        });

        if (lastFormula && contents.length === 1) {
            var firstRow = contents[0];
            if (firstRow.length === 1) {
                tokenArray.parseFormula('ui', lastFormula.substr(1));
                var result = tokenArray.interpretFormula('val', { refAddress: targetStart, targetAddress: targetStart });
                if (result.type === 'result') {
                    firstRow[0].result = result.value;
                }
            }
        }

        tokenArray.destroy();

        // merge URL address arrays to range arrays
        urlMap = Utils.mapProperties(urlMap, RangeArray.mergeAddresses);

        return { contents: contents, mergedRanges: mergedRanges, urlMap: urlMap };
    };

    /**
     * Parses the plain text from the clipboard data and creates the cell contents.
     * Rows should be separated by line endings ('\r\n', '\r' or '\n')
     * and cells should be separated by tab or semicolon.
     *
     * @param {String} text
     *  The text input to parse.
     *
     * @returns {Object}
     *  An object with the following attributes:
     *  - {Array} contents
     *      A two-dimensional array containing the values and attribute sets
     *      for the cells.
     */
    Clipboard.parseTextData = function (text) {

        var // a two dimensional array containing the cell contents
            contents = [],
            // the row nodes
            rows,
            // the cell separator
            sep;

        if (!_.isString(text)) { return { contents: [] }; }

        // remove \r,\n, \t and ; to remove empty rows at the end of the text
        text = text.replace(/[;\s]+$/, '');

        // parse rows according to the line ending type
        rows = (text.indexOf('\r\n') > -1) ? text.split(/\r\n/) : text.split(/[\r\n]/);
        // look for the cell separator
        sep = (text.indexOf('\t') > -1) ? '\t' : ';';

        _.each(rows, function (row) {
            var rowData = [],
                cells;

            cells = row.split(sep);
            _.each(cells, function (cell) {
                rowData.push({ value: Utils.cleanString(cell) });
            });
            contents.push(rowData);
        });

        return { contents: contents };
    };

    /**
     * Returns the container node with DOM data nodes for drawing objects that
     * have been copied into the clipboard by an OX editor application.
     */
    Clipboard.getDrawingContainerNode = function (clipboardNodes) {
        var containerNode = collectMatchingNodes(clipboardNodes, '#ox-clipboard-data[data-ox-drawings="true"]');
        return (containerNode.length > 0) ? containerNode.first() : null;
    };

    /**
     * Returns the position and size stored in the passed data node. This
     * method is tolerant against missing data attributes, and falls back to
     * reasonable default values.
     *
     * @param {jQuery} dataNode
     *  The DOM node containing a rectangle specification.
     *
     * @returns {Object}
     *  The rectangle representing the location of the object.
     */
    Clipboard.getRectangleFromNode = function (dataNode, offsetX, offsetY) {
        return {
            left: offsetX + Utils.getElementAttributeAsInteger(dataNode, 'data-left', 0),
            top: offsetY + Utils.getElementAttributeAsInteger(dataNode, 'data-top', 0),
            width: Utils.getElementAttributeAsInteger(dataNode, 'data-width', 0) || 256,
            height: Utils.getElementAttributeAsInteger(dataNode, 'data-height', 0) || 192
        };
    };

    /**
     * Returns the explicit attributes stored in the passed data node..
     *
     * @param {jQuery} dataNode
     *  The DOM node containing a rectangle specification.
     *
     * @returns {Object|Null}
     *  The explicit attributes of the object.
     */
    Clipboard.getAttributesFromNode = function (dataNode) {

        // get the stringified JSON data from the 'data-attrs' element attribute
        var jsonStr = dataNode.attr('data-attrs');
        if (!jsonStr) { return null; }

        try {
            var attrs = JSON.parse(jsonStr);
            return _.isObject(attrs) ? attrs : null;
        } catch (ex) {
            Utils.warn('Clipboard.getAttributesFromNode(): invalid attributes');
        }
    };

    /**
     * Extracts the JSON data from the passed DOM node representing some
     * document contents in the clipboard.
     *
     * @param {jQuery} dataNode
     *  The DOM node containing JSON data.
     *
     * @returns {Any}
     *  The reconstructed JSON data; or null on error.
     */
    Clipboard.getJSONDataFromNode = function (dataNode) {

        // get the stringified JSON data from the 'data-json' element attribute
        var jsonStr = dataNode.attr('data-json');
        if (!jsonStr) { return null; }

        // parsing JSON may throw
        try {
            return JSON.parse(jsonStr);
        } catch (ex) {
            Utils.warn('Clipboard.getJSONDataFromNode(): invalid JSON data');
            return null;
        }
    };

    // exports ================================================================

    return Clipboard;

});
