/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/viewsettingsmixin', [
    'io.ox/office/tk/utils'
], function (Utils) {

    'use strict';

    // class ViewSettingsMixin ================================================

    /**
     * A mix-in class for spreadsheet model classes that stores specific view
     * attributes. Used by the document model class to store global view
     * settings, and by the sheet model to store view settings per sheet.
     *
     * Triggers the following events:
     * - 'change:viewattributes'
     *      After at least one view attribute has been changed. Event handlers
     *      receive an incomplete attribute map containing all changed view
     *      attributes with their new values.
     *
     * @constructor
     *
     * @param {Object} definitions
     *  A map with definition objects for all supported view attributes,
     *  containing their default values, and other settings. Each definition
     *  object is keyed by the attribute name, and contains the following
     *  properties:
     *  - {Any} definition.def
     *      The default value for the view attribute.
     *  - {Function} [definition.equals=_.isEqual]
     *      A predicate that returns whether two values of the view attribute
     *      are equal.
     *  - {Function|RegExp} [definition.validate=_.identity]
     *      A callback function used to validate a new value for the attribute,
     *      or a regular expression that can be used to validate enumeration
     *      attributes. The callback function may return any value to be used
     *      as new attribute value, or may return Utils.BREAK to reject the new
     *      attribute value and keep the current value. Defaults to the
     *      identity function (accepts all attribute values). Will be called in
     *      the context of the object instance containing the view attributes.
     */
    function ViewSettingsMixin(definitions) {

        var // self reference
            self = this,

            // the current view attributes of this object
            viewAttributes = Utils.mapProperties(definitions, function (definition) { return definition.def; });

        // public methods -----------------------------------------------------

        /**
         * Returns the values of all view attribute.
         *
         * @returns {Object}
         *  A map containing all view attribute values. For performance, this
         *  is a reference to the original attribute map. The values in this
         *  map MUST NOT be changed (use the setter methods)!
         */
        this.getViewAttributes = function () {
            return viewAttributes;
        };

        /**
         * Returns the value of a specific view attribute.
         *
         * @param {String} name
         *  The name of the view attribute.
         *
         * @returns {Any}
         *  The attribute value if available, otherwise null. For performance,
         *  this is the original attribute value. Arrays or objects MUST NOT be
         *  changed (use the setter methods)!
         */
        this.getViewAttribute = function (name) {
            return (name in viewAttributes) ? viewAttributes[name] : null;
        };

        /**
         * Changes specific view attributes of this object, and triggers a
         * 'change:viewattributes' event.
         *
         * @param {Object} attributes
         *  A map with all view attributes to be changed.
         *
         * @returns {ViewSettingsMixin}
         *  A reference to this instance.
         */
        this.setViewAttributes = function (attributes) {

            var // all changed attributes, for listeners
                changedAttributes = {};

            _.each(attributes, function (value, name) {
                if (name in definitions) {

                    // execute validator
                    var validator = definitions[name].validate;
                    if (_.isFunction(validator)) {
                        value = validator.call(self, value);
                    } else if (_.isRegExp(validator)) {
                        if (!validator.test(value)) { value = Utils.BREAK; }
                    }

                    // change attribute value
                    var equality = definitions[name].equals || _.isEqual;
                    if (!_.isUndefined(value) && (value !== Utils.BREAK) && !equality(viewAttributes[name], value)) {
                        viewAttributes[name] = changedAttributes[name] = value;
                    }
                }
            });

            if (!_.isEmpty(changedAttributes)) {
                this.trigger('change:viewattributes', changedAttributes);
            }
            return this;
        };

        /**
         * Changes a specific view attribute of this object, and triggers a
         * 'change:viewattributes' event.
         *
         * @param {String} name
         *  The name of the view attribute.
         *
         * @param {Any} value
         *  The new value for the view attribute.
         *
         * @returns {ViewSettingsMixin}
         *  A reference to this instance.
         */
        this.setViewAttribute = function (name, value) {
            return this.setViewAttributes(Utils.makeSimpleObject(name, value));
        };

        // initialization -----------------------------------------------------

        // destroy all class members on destruction
        this.registerDestructor(function () {
            definitions = self = viewAttributes = null;
        });

    } // class ViewSettingsMixin

    // exports ================================================================

    return ViewSettingsMixin;

});
