/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/formula/tokenizer', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/locale/parser',
    'io.ox/office/baseframework/model/modelobject',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/model/formula/formulautils',
    'io.ox/office/spreadsheet/model/formula/grammarconfig',
    'io.ox/office/spreadsheet/model/formula/sheetref',
    'io.ox/office/spreadsheet/model/formula/cellref',
    'io.ox/office/spreadsheet/model/formula/tokens'
], function (Utils, Parser, ModelObject, SheetUtils, FormulaUtils, GrammarConfig, SheetRef, CellRef, Tokens) {

    'use strict';

    // convenience shortcuts
    var Address = SheetUtils.Address;

    // class TokenDescriptor ==================================================

    /**
     * A descriptor for a single formula token parsed from a formula expression
     * string, together with additional information such as the original
     * display text of the token.
     *
     * @property {BaseToken} token
     *  The formula token.
     *
     * @property {Number} index
     *  The array index of the formula token.
     *
     * @property {String} text
     *  The display text of the token from the original formula expression.
     *
     * @property {Number} start
     *  The start position of the text in the original formula expression.
     *
     * @property {Number} end
     *  The end position of the text in the original formula expression.
     */
    function TokenDescriptor(token, text) {

        this.token = token;
        this.index = -1;
        this.text = text;
        this.start = 0;
        this.end = 0;

    } // class TokenDescriptor

    // public methods ---------------------------------------------------------

    /**
     * Returns a text description of the token for debugging (!) purposes.
     */
    TokenDescriptor.prototype.toString = function () {
        return this.token.toString();
    };

    // class ExpressionParser =================================================

    /**
     * An instance of this class parses a single formula expression for a
     * specific formula grammar. This is the base class of several class
     * specializations for the different grammars and reference syntaxes.
     *
     * @constructor
     */
    var ExpressionParser = _.makeExtendable(function (docModel, grammarConfig, initOptions) {

        // the document model
        this.docModel = docModel;

        // maximum column/row index in a sheet
        this.MAXCOL = docModel.getMaxCol();
        this.MAXROW = docModel.getMaxRow();

        // the formula grammar configuration
        this.config = grammarConfig;
        this.RE = grammarConfig.RE;

        // index of the sheet to extend local references to
        this.extendSheet = Utils.getIntegerOption(initOptions, 'extendSheet', -1);

        // the sheet model of the specified extension sheet
        this.extendSheetModel = docModel.getSheetModel(this.extendSheet);

        // whether a matrix literal is currently being parsed
        this.matrix = false;

    }); // class ExpressionParser

    // private methods --------------------------------------------------------

    /**
     * Extracts the leading token from the passed formula subexpression.
     *
     * @param {String} expr
     *  The formula subexpression to be parsed.
     *
     * @returns {TokenDescriptor|Null}
     *  A descriptor for the parsed formula token, or null on error.
     */
    ExpressionParser.prototype._parseNextGlobalToken = function (expr) {

        // the matches of the regular expressions
        var matches = null;
        // a token descriptor built by another helper method of this class
        var tokenDesc = null;

        // string literal
        if ((matches = this.RE.STRING_LIT.exec(expr))) {
            return this.createLiteralToken(matches[0].slice(1, matches[0].length - 1).replace(/""/g, '"'), matches[0]);
        }

        // boolean literal (before defined names)
        if ((matches = this.RE.BOOLEAN_LIT.exec(expr))) {
            return this.createLiteralToken(this.config.getBooleanValue(matches[0]), matches[0]);
        }

        // error code literal
        if ((matches = this.RE.ERROR_LIT.exec(expr))) {
            return this.createLiteralToken(this.config.getErrorCode(matches[0]), matches[0]);
        }

        // whitespace (MUST be parsed before operators)
        if ((matches = this.RE.WHITESPACE.exec(expr))) {
            return this.createFixedToken('ws', matches[0]);
        }

        // list/parameter separator (MUST be parsed before operators)
        if ((matches = this.RE.SEPARATOR.exec(expr))) {
            return this.createToken(new Tokens.SeparatorToken(), matches[0]);
        }

        // unary/binary operator
        if ((matches = this.RE.OPERATOR.exec(expr))) {
            return this.createToken(new Tokens.OperatorToken(this.config.getOperatorKey(matches[0])), matches[0]);
        }

        // opening parenthesis
        if ((matches = this.RE.OPEN.exec(expr))) {
            return this.createToken(new Tokens.ParenthesisToken(true), matches[0]);
        }

        // closing parenthesis
        if ((matches = this.RE.CLOSE.exec(expr))) {
            return this.createToken(new Tokens.ParenthesisToken(false), matches[0]);
        }

        // opening parenthesis of a matrix literal
        if ((matches = this.RE.MATRIX_OPEN.exec(expr))) {
            this.matrix = true;
            return this.createToken(new Tokens.MatrixDelimiterToken('mat_open'), matches[0]);
        }

        // any type of cell reference without and with sheets
        if ((tokenDesc = this._parseNextRefToken(expr))) {
            return tokenDesc;
        }

        // number literal (must be parsed after row ranges, e.g. 1:3 is a row range instead of two numbers)
        if ((matches = Parser.parseLeadingNumber(expr, { dec: this.config.DEC }))) {
            return this.createLiteralToken(matches.number, matches.text);
        }

        // defined names and function calls
        if ((tokenDesc = this._parseNextNameToken(expr))) {
            return tokenDesc;
        }
    };

    /**
     * Extracts the leading token in a matrix literal from the passed formula
     * subexpression.
     *
     * @param {String} expr
     *  The formula subexpression to be parsed.
     *
     * @returns {TokenDescriptor|Null}
     *  A descriptor for the parsed formula token, or null on error.
     */
    ExpressionParser.prototype._parseNextMatrixToken = function (expr) {

        // the matches of the regular expressions
        var matches = null;

        // number literal
        if ((matches = Parser.parseLeadingNumber(expr, { dec: this.config.DEC, sign: true }))) {
            return this.createLiteralToken(matches.number, matches.text);
        }

        // string literal
        if ((matches = this.RE.STRING_LIT.exec(expr))) {
            return this.createLiteralToken(matches[0].slice(1, matches[0].length - 1).replace(/""/g, '"'), matches[0]);
        }

        // boolean literal
        if ((matches = this.RE.BOOLEAN_LIT.exec(expr))) {
            return this.createLiteralToken(this.config.getBooleanValue(matches[0]), matches[0]);
        }

        // error code literal
        if ((matches = this.RE.ERROR_LIT.exec(expr))) {
            return this.createLiteralToken(this.config.getErrorCode(matches[0]), matches[0]);
        }

        // whitespace
        if ((matches = this.RE.WHITESPACE.exec(expr))) {
            return this.createFixedToken('ws', matches[0]);
        }

        // row separator
        if ((matches = this.RE.MATRIX_ROW_SEPARATOR.exec(expr))) {
            return this.createToken(new Tokens.MatrixDelimiterToken('mat_row'), matches[0]);
        }

        // column separator
        if ((matches = this.RE.MATRIX_COL_SEPARATOR.exec(expr))) {
            return this.createToken(new Tokens.MatrixDelimiterToken('mat_col'), matches[0]);
        }

        // closing parenthesis
        if ((matches = this.RE.MATRIX_CLOSE.exec(expr))) {
            this.matrix = false;
            return this.createToken(new Tokens.MatrixDelimiterToken('mat_close'), matches[0]);
        }
    };

    // protected methods ------------------------------------------------------

    /**
     * Returns a sheet reference structure with either a fixed (implicit), or a
     * variable absolute flag, for the passed matches of a regular expression.
     *
     * @param {Array<String>} matches
     *  The matches of a regular expression.
     *
     * @param {Number} index
     *  The start array index of the sheet reference in the passed matches.
     *
     * @param {Boolean} [abs=false]
     *  If set to true, the sheet reference is assumed to be implicitly
     *  absolute. The matches MUST NOT contain an optional absolute marker, but
     *  MUST contain three elements for the simple sheet name, for the complex
     *  sheet name, and for the reference error, and only one of the elements
     *  must be set to a string. Otherwise, the sheet reference contains an
     *  optional absolute marker. The matches MUST contain elements for the
     *  absolute marker, followed by the three sheet name elements mentioned
     *  above.
     *
     * @returns {SheetRef}
     *  A sheet reference structure for the passed matches.
     */
    ExpressionParser.prototype._createSheetRef = function (matches, index, abs) {
        if (!abs) { abs = !!matches[index]; index += 1; }
        return SheetRef.create(this.docModel, matches[index], matches[index + 1], matches[index + 2], abs);
    };

    /**
     * Returns an object with a sheet reference structure for a single sheet
     * name from the passed matches of a regular expression, as expected by the
     * public method ExpressionParser.createReferenceToken() of this class.
     *
     * @param {Array<String>} matches
     *  The matches of a regular expression.
     *
     * @param {Number} index
     *  The start array index of the sheet reference in the passed matches.
     *
     * @param {Boolean} [abs=false]
     *  If set to true, the sheet reference is assumed to be implicitly
     *  absolute. See method ExpressionParser._createSheetRef() for details.
     *
     * @returns {Object}
     *  An object with 'r1' property set to a sheet reference structure for the
     *  passed matches and 'r2' property set to null.
     */
    ExpressionParser.prototype._createSheetRefData = function (matches, index, abs) {
        return { r1: this._createSheetRef(matches, index, abs), r2: null };
    };

    /**
     * Returns a cell reference structure for the passed matches of a regular
     * expression.
     *
     * @param {Array<String>} matches
     *  The matches of a regular expression.
     *
     * @param {Number} index
     *  The start array index of the cell address in the passed matches. The
     *  matches are expected to contain four elements for the absolute column
     *  marker, the column index, the absolute row marker, and the row index,
     *  in this order.
     *
     * @returns {CellRef|Null}
     *  A cell reference structure for the passed matches, if the column and
     *  row indexes are valid; otherwise null.
     */
    ExpressionParser.prototype._createA1CellRef = function (matches, index) {
        var col = Address.parseCol(matches[index + 1]);
        var row = Address.parseRow(matches[index + 3]);
        return ((col >= 0) && (col <= this.MAXCOL) && (row >= 0) && (row <= this.MAXROW)) ? new CellRef(col, row, !!matches[index], !!matches[index + 2]) : null;
    };

    /**
     * Returns a cell reference structure for a column from the passed matches
     * of a regular expression.
     *
     * @param {Array<String>} matches
     *  The matches of a regular expression.
     *
     * @param {Number} index
     *  The start array index of the column index in the passed matches. The
     *  matches are expected to contain two elements for the absolute column
     *  marker, and the column index, in this order.
     *
     * @param {Number} row
     *  The fixed row index to be inserted into the created cell reference.
     *
     * @returns {CellRef|Null}
     *  A cell reference structure for the passed matches, if the column index
     *  is valid; otherwise null.
     */
    ExpressionParser.prototype._createA1ColRef = function (matches, index, row) {
        var col = Address.parseCol(matches[index + 1]);
        return ((col >= 0) && (col <= this.MAXCOL)) ? new CellRef(col, row, !!matches[index], true) : null;
    };

    /**
     * Returns a cell reference structure for a row from the passed matches of
     * a regular expression.
     *
     * @param {Array<String>} matches
     *  The matches of a regular expression.
     *
     * @param {Number} index
     *  The start array index of the row index in the passed matches. The
     *  matches are expected to contain two elements for the absolute row
     *  marker, and the row index, in this order.
     *
     * @param {Number} col
     *  The fixed column index to be inserted into the created cell reference.
     *
     * @returns {CellRef|Null}
     *  A cell reference structure for the passed matches, if the row index is
     *  valid; otherwise null.
     */
    ExpressionParser.prototype._createA1RowRef = function (matches, index, col) {
        var row = Address.parseRow(matches[index + 1]);
        return ((row >= 0) && (row <= this.MAXROW)) ? new CellRef(col, row, true, !!matches[index]) : null;
    };

    /**
     * Returns an object with a cell reference structure for a cell address
     * from the passed matches of a regular expression, as expected by the
     * public method ExpressionParser.createReferenceToken() of this class.
     *
     * @param {Array<String>} matches
     *  The matches of a regular expression.
     *
     * @param {Number} index
     *  The start array index of the cell address in the passed matches. See
     *  method ExpressionParser._createA1CellRef() for details.
     *
     * @returns {Object|Null}
     *  An object with 'r1' property set to a cell reference structure for the
     *  passed matches and 'r2' property set to null, if the column and row
     *  indexes are valid; otherwise null.
     */
    ExpressionParser.prototype._createA1CellData = function (matches, index) {
        var cellRef = this._createA1CellRef(matches, index);
        return cellRef ? { r1: cellRef, r2: null } : null;
    };

    /**
     * Returns an object with two cell reference structures for a cell range
     * address from the passed matches of a regular expression, as expected by
     * the public method ExpressionParser.createReferenceToken() of this class.
     *
     * @param {Array<String>} matches
     *  The matches of a regular expression.
     *
     * @param {Number} index1
     *  The start array index of the first cell address in the passed matches.
     *  See method ExpressionParser._createA1CellRef() for details.
     *
     * @param {Number} index2
     *  The start array index of the second cell address in the passed matches.
     *  See method ExpressionParser._createA1CellRef() for details.
     *
     * @returns {Object|Null}
     *  An object with 'r1' property set to a cell reference structure for the
     *  first cell address, and 'r2' property set to a cell reference structure
     *  for the second cell address, if the column and row indexes are all
     *  valid; otherwise null.
     */
    ExpressionParser.prototype._createA1CellRangeData = function (matches, index1, index2) {
        var cell1Ref = this._createA1CellRef(matches, index1);
        var cell2Ref = cell1Ref ? this._createA1CellRef(matches, index2) : null;
        return cell2Ref ? { r1: cell1Ref, r2: cell2Ref } : null;
    };

    /**
     * Returns an object with two cell reference structures for a column range
     * address from the passed matches of a regular expression, as expected by
     * the public method ExpressionParser.createReferenceToken() of this class.
     *
     * @param {Array<String>} matches
     *  The matches of a regular expression.
     *
     * @param {Number} index1
     *  The start array index of the first column index in the passed matches.
     *  See method ExpressionParser._createA1ColRef() for details.
     *
     * @param {Number} index2
     *  The start array index of the second column index in the passed matches.
     *  See method ExpressionParser._createA1ColRef() for details.
     *
     * @returns {Object|Null}
     *  An object with 'r1' property set to a cell reference structure for the
     *  first cell address, and 'r2' property set to a cell reference structure
     *  for the second cell address, if the column indexes are all valid;
     *  otherwise null.
     */
    ExpressionParser.prototype._createA1ColRangeData = function (matches, index1, index2) {
        var cell1Ref = this._createA1ColRef(matches, index1, 0);
        var cell2Ref = cell1Ref ? this._createA1ColRef(matches, index2, this.MAXROW) : null;
        return cell2Ref ? { r1: cell1Ref, r2: cell2Ref } : null;
    };

    /**
     * Returns an object with two cell reference structures for a row range
     * address from the passed matches of a regular expression, as expected by
     * the public method ExpressionParser.createReferenceToken() of this class.
     *
     * @param {Array<String>} matches
     *  The matches of a regular expression.
     *
     * @param {Number} index1
     *  The start array index of the first row index in the passed matches. See
     *  method ExpressionParser._createA1RowRef() for details.
     *
     * @param {Number} index2
     *  The start array index of the second row index in the passed matches.
     *  See method ExpressionParser._createA1RowRef() for details.
     *
     * @returns {Object|Null}
     *  An object with 'r1' property set to a cell reference structure for the
     *  first cell address, and 'r2' property set to a cell reference structure
     *  for the second cell address, if the row indexes are all valid;
     *  otherwise null.
     */
    ExpressionParser.prototype._createA1RowRangeData = function (matches, index1, index2) {
        var cell1Ref = this._createA1RowRef(matches, index1, 0);
        var cell2Ref = cell1Ref ? this._createA1RowRef(matches, index2, this.MAXCOL) : null;
        return cell2Ref ? { r1: cell1Ref, r2: cell2Ref } : null;
    };

    /**
     * Sub classes MUST provide an implementation to parse a reference token
     * from the passed formula subexpression.
     */
    ExpressionParser.prototype._parseNextRefToken = function (/*expr*/) {
        Utils.error('ExpressionParser._parseNextRefToken(): missing implementation');
        return null;
    };

    /**
     * Sub classes MUST provide an implementation to parse a name or function
     * token from the passed formula subexpression.
     */
    ExpressionParser.prototype._parseNextNameToken = function (/*expr*/) {
        Utils.error('ExpressionParser._parseNextNameToken(): missing implementation');
        return null;
    };

    // public methods ---------------------------------------------------------

    /**
     * Creates a token descriptor from the passed token and original string
     * representation.
     *
     * @param {BaseToken} token
     *  The new token instance to be inserted into the token descriptor
     *  returned from this method.
     *
     * @param {String} text
     *  The original text representation of the passed token in the parsed
     *  formula expression.
     *
     * @returns {TokenDescriptor}
     *  The token descriptor containing the passed token and text.
     */
    ExpressionParser.prototype.createToken = function (token, text) {
        return new TokenDescriptor(token, text);
    };

    /**
     * Creates a token descriptor containing a new fixed text token.
     *
     * @param {String} type
     *  The type of the fixed token to be inserted into the new token
     *  descriptor returned from this method.
     *
     * @param {String} text
     *  The text representation of the fixed token in the parsed formula
     *  expression.
     *
     * @returns {TokenDescriptor}
     *  The token descriptor containing a new fixed token, and the passed text.
     */
    ExpressionParser.prototype.createFixedToken = function (type, text) {
        return this.createToken(new Tokens.FixedToken(type, text), text);
    };

    /**
     * Creates a token descriptor containing a new literal token.
     *
     * @param {Any} value
     *  The value for the literal token to be inserted into the new token
     *  descriptor returned from this method.
     *
     * @param {String} text
     *  The original text representation of the value in the parsed formula
     *  expression.
     *
     * @returns {TokenDescriptor}
     *  The token descriptor containing a new literal token, and the passed
     *  text.
     */
    ExpressionParser.prototype.createLiteralToken = function (value, text) {
        return this.createToken(new Tokens.LiteralToken(value), text);
    };

    /**
     * Creates a token descriptor containing a new reference token.
     *
     * @param {Object|Null} sheetData
     *  An object containing zero, one, or two sheet references (instances of
     *  the class SheetRef), in the properties 'r1' and 'r2'. If null is passed
     *  as parameter value, or if property 'r1' of the object is null, a local
     *  reference token without sheets will be created.
     *
     * @param {Object|Null} cellData
     *  An object containing zero, one, or two cell references (instances of
     *  the class CellRef), in the properties 'r1' and 'r2'. If null is passed
     *  as parameter value, or if property 'r1' of the object is null, a
     *  reference token representing a reference error will be created.
     *
     * @param {String} text
     *  The original text representation of the reference in the parsed formula
     *  expression.
     *
     * @returns {TokenDescriptor}
     *  The token descriptor containing a new reference token, and the passed
     *  text.
     */
    ExpressionParser.prototype.createReferenceToken = function (sheetData, cellData, text) {

        // bug 40293: add sheet reference to sheet-local cell references if specified
        if (!sheetData && this.extendSheetModel) {
            sheetData = { r1: new SheetRef(this.extendSheet, true), r2: null };
        }

        return this.createToken(new Tokens.ReferenceToken(
            this.docModel,
            cellData ? cellData.r1 : null,
            cellData ? cellData.r2 : null,
            sheetData ? sheetData.r1 : null,
            sheetData ? sheetData.r2 : null
        ), text);
    };

    /**
     * Creates a token descriptor containing a new name token.
     *
     * @param {SheetRef|Null} sheetRef
     *  A sheet reference structure for sheet-local names; or null for a global
     *  name token without sheet.
     *
     * @param {String} label
     *  The label of the defined name.
     *
     * @param {String} text
     *  The original text representation of the defined name in the parsed
     *  formula expression.
     *
     * @returns {TokenDescriptor}
     *  The token descriptor containing a new name token, and the passed text.
     */
    ExpressionParser.prototype.createNameToken = function (sheetRef, label, text) {

        // bug 40293: add sheet reference to existing sheet-local names if specified
        if (!sheetRef && this.extendSheetModel && this.extendSheetModel.getNameCollection().hasNameModel(label)) {
            sheetRef = new SheetRef(this.extendSheet, true);
        }

        return this.createToken(new Tokens.NameToken(this.docModel, label, sheetRef), text);
    };

    /**
     * Creates a token descriptor containing a new function/macro token.
     *
     * @param {SheetRef|Null} sheetRef
     *  A sheet reference structure for macro calls with sheet/module name; or
     *  null for a built-in function, or a global macro call without sheet.
     *
     * @param {String} label
     *  The label of the function/macro.
     *
     * @param {String} text
     *  The original text representation of the function in the parsed formula
     *  expression.
     *
     * @returns {TokenDescriptor}
     *  The token descriptor containing a new function token, and the passed
     *  text.
     */
    ExpressionParser.prototype.createFunctionToken = function (sheetRef, label, text) {

        // the resource key of a built-in sheet function (method getFunctionKey() returns null for unknown functions)
        var funcKey = sheetRef ? null : this.config.getFunctionKey(label);

        return this.createToken(funcKey ? new Tokens.FunctionToken(funcKey) : new Tokens.MacroToken(this.docModel, label, sheetRef), text);
    };

    /**
     * Parses and tokenizes the passed complete formula expression.
     *
     * @param {String} formula
     *  The formula expression to be parsed and tokenized.
     *
     * @returns {Array<TokenDescriptor>}
     *  An array of token descriptors containing the parsed formula tokens, and
     *  the original text representations of the tokens.
     */
    ExpressionParser.prototype.parseFormula = function (formula) {

        // the descriptors of the resulting formula tokens
        var tokenDescs = [];

        // extract all supported tokens from start of formula string
        while (formula.length > 0) {

            // try to parse the next formula token from the beginning of the (remaining) formula expression
            var tokenDesc = this.matrix ? this._parseNextMatrixToken(formula) : this._parseNextGlobalToken(formula);

            // no token returned: push entire remaining formula text as 'bad' token
            if (!tokenDesc) { tokenDesc = this.createFixedToken('bad', formula); }

            // push the token descriptor, and remove the text from the formula expression
            tokenDescs.push(tokenDesc);
            formula = formula.substr(tokenDesc.text.length);
        }

        return tokenDescs;
    };

    // class OOXMLExpressionParser ============================================

    /**
     * A formula expression parser for the OOXML reference syntax.
     *
     * @constructor
     *
     * @extends ExpressionParser
     */
    var OOXMLExpressionParser = ExpressionParser.extend({ constructor: function (docModel, grammarConfig, initOptions) {

        // base constructor ---------------------------------------------------

        ExpressionParser.call(this, docModel, grammarConfig, initOptions);

    } }); // class OOXMLExpressionParser

    // protected methods ------------------------------------------------------

    /**
     * Returns an object with two SheetRef instances for the passed RE matches
     * of a sheet range.
     */
    OOXMLExpressionParser.prototype._createSheetRangeData = function (matches, index) {
        var sheet1Ref = SheetRef.create(this.docModel, matches[index], matches[index + 2], null, true);
        var sheet2Ref = SheetRef.create(this.docModel, matches[index + 1], matches[index + 3], null, true);
        return { r1: sheet1Ref, r2: sheet2Ref };
    };

    /**
     * Extracts a reference token from the passed formula subexpression.
     */
    OOXMLExpressionParser.prototype._parseNextRefToken = function (expr) {

        // the matches of the regular expressions
        var matches = null;
        // cell references built from the matches of a regular expression
        var cellRefs = null;

        // local cell range reference (before cell references!), e.g. $A$1:$C$3
        if ((matches = this.RE.CELL_RANGE_REF.exec(expr)) && (cellRefs = this._createA1CellRangeData(matches, 1, 5))) {
            return this.createReferenceToken(null, cellRefs, matches[0]);
            // otherwise, continue with other REs, it may be a local name such as 'A1:name42'
        }

        // cell range reference with sheet (before cell references!), e.g. Sheet1!$A$1:$C$3
        if ((matches = this.RE.CELL_RANGE_ABS3D_REF.exec(expr)) && (cellRefs = this._createA1CellRangeData(matches, 4, 8))) {
            return this.createReferenceToken(this._createSheetRefData(matches, 1, true), cellRefs, matches[0]);
            // otherwise, continue with other REs, it may be a local name such as 'Sheet1!A1:name42'
        }

        // cell range reference in multiple sheets (before cell references!), e.g. Sheet1:Sheet3!$A$1:$C$3
        if ((matches = this.RE.CELL_RANGE_ABSCUBE_REF.exec(expr)) && (cellRefs = this._createA1CellRangeData(matches, 5, 9))) {
            return this.createReferenceToken(this._createSheetRangeData(matches, 1), cellRefs, matches[0]);
            // otherwise, continue with other REs, it may be a local name such as 'Sheet1:Sheet2!A1:name42'
        }

        // local cell reference, e.g. $A$1
        if ((matches = this.RE.CELL_REF.exec(expr)) && (cellRefs = this._createA1CellData(matches, 1))) {
            return this.createReferenceToken(null, cellRefs, matches[0]);
            // otherwise, continue with other REs, it may be a local name such as 'name42'
        }

        // cell reference with sheet, e.g. Sheet1!$A$1
        if ((matches = this.RE.CELL_ABS3D_REF.exec(expr)) && (cellRefs = this._createA1CellData(matches, 4))) {
            return this.createReferenceToken(this._createSheetRefData(matches, 1, true), cellRefs, matches[0]);
            // otherwise, continue with other REs, it may be a local name such as 'name42'
        }

        // cell reference in multiple sheets, e.g. Sheet1:Sheet3!$A$1
        if ((matches = this.RE.CELL_ABSCUBE_REF.exec(expr)) && (cellRefs = this._createA1CellData(matches, 5))) {
            return this.createReferenceToken(this._createSheetRangeData(matches, 1), cellRefs, matches[0]);
            // otherwise, continue with other REs, it may be a local name such as 'Sheet1:Sheet2!name42'
        }

        // local column range reference, e.g. $A:$C
        if ((matches = this.RE.COL_RANGE_REF.exec(expr)) && (cellRefs = this._createA1ColRangeData(matches, 1, 3))) {
            return this.createReferenceToken(null, cellRefs, matches[0]);
            // otherwise, continue with other REs, it may be a range of names such as 'name:name'
        }

        // column range reference with sheet, e.g. Sheet1!$A:$C
        if ((matches = this.RE.COL_RANGE_ABS3D_REF.exec(expr)) && (cellRefs = this._createA1ColRangeData(matches, 4, 6))) {
            return this.createReferenceToken(this._createSheetRefData(matches, 1, true), cellRefs, matches[0]);
            // otherwise, continue with other REs, it may be a range of names such as 'name:name'
        }

        // column range reference in multiple sheets, e.g. Sheet1:Sheet3!$A:$C
        if ((matches = this.RE.COL_RANGE_ABSCUBE_REF.exec(expr)) && (cellRefs = this._createA1ColRangeData(matches, 5, 7))) {
            return this.createReferenceToken(this._createSheetRangeData(matches, 1), cellRefs, matches[0]);
            // otherwise, continue with other REs, it may be a range of names such as 'Sheet1:Sheet2!name:name'
        }

        // local row range reference, e.g. $1:$3
        if ((matches = this.RE.ROW_RANGE_REF.exec(expr)) && (cellRefs = this._createA1RowRangeData(matches, 1, 3))) {
            return this.createReferenceToken(null, cellRefs, matches[0]);
            // otherwise, continue with other REs
        }

        // row range reference with sheet, e.g. Sheet1!$1:$3
        if ((matches = this.RE.ROW_RANGE_ABS3D_REF.exec(expr)) && (cellRefs = this._createA1RowRangeData(matches, 4, 6))) {
            return this.createReferenceToken(this._createSheetRefData(matches, 1, true), cellRefs, matches[0]);
            // otherwise, continue with other REs
        }

        // row range reference in multiple sheets, e.g. Sheet1:Sheet3!$1:$3
        if ((matches = this.RE.ROW_RANGE_ABSCUBE_REF.exec(expr)) && (cellRefs = this._createA1RowRangeData(matches, 5, 7))) {
            return this.createReferenceToken(this._createSheetRangeData(matches, 1), cellRefs, matches[0]);
            // otherwise, continue with other REs
        }

        // reference error with sheet, e.g. Sheet1!#REF!
        if ((matches = this.RE.ERROR_ABS3D_REF.exec(expr))) {
            return this.createReferenceToken(this._createSheetRefData(matches, 1, true), null, matches[0]);
        }

        // reference error in multiple sheets, e.g. Sheet1:Sheet3!#REF!
        if ((matches = this.RE.ERROR_ABSCUBE_REF.exec(expr))) {
            return this.createReferenceToken(this._createSheetRangeData(matches, 1), null, matches[0]);
        }
    };

    /**
     * Extracts a name or function token from the passed formula subexpression.
     */
    OOXMLExpressionParser.prototype._parseNextNameToken = function (expr) {

        // the matches of the regular expressions
        var matches = null;

        // local function call, or global macro call, e.g. SUM() or my_macro()
        if ((matches = this.RE.FUNCTION_REF.exec(expr))) {
            return this.createFunctionToken(null, matches[1], matches[0]);
        }

        // macro call with sheet, e.g. Module1!my_macro()
        if ((matches = this.RE.FUNCTION_ABS3D_REF.exec(expr))) {
            return this.createFunctionToken(this._createSheetRef(matches, 1, true), matches[4], matches[0]);
        }

        // defined name with sheet, e.g. Sheet1!my_name
        if ((matches = this.RE.NAME_ABS3D_REF.exec(expr))) {
            return this.createNameToken(this._createSheetRef(matches, 1, true), matches[4], matches[0]);
        }

        // defined name without sheet reference
        if ((matches = this.RE.NAME_REF.exec(expr))) {
            return this.createNameToken(null, matches[1], matches[0]);
        }
    };

    // class OOXMLExpressionParser ============================================

    /**
     * A formula expression parser for the ODF UI reference syntax.
     *
     * @constructor
     *
     * @extends ExpressionParser
     */
    var ODFUIExpressionParser = ExpressionParser.extend({ constructor: function (docModel, grammarConfig, initOptions) {

        // base constructor ---------------------------------------------------

        ExpressionParser.call(this, docModel, grammarConfig, initOptions);

    } }); // class ODFUIExpressionParser

    // protected methods ------------------------------------------------------

    /**
     * Returns an object with two SheetRef instances for the passed RE matches
     * of a sheet range.
     */
    ODFUIExpressionParser.prototype._createSheetRangeData = function (matches, index) {
        var sheet1Ref = SheetRef.create(this.docModel, matches[index + 1], matches[index + 2], null, !!matches[index]);
        var sheet2Ref = SheetRef.create(this.docModel, matches[index + 4], matches[index + 5], null, !!matches[index + 3]);
        return { r1: sheet1Ref, r2: sheet2Ref };
    };

    /**
     * Extracts a reference token from the passed formula subexpression.
     */
    ODFUIExpressionParser.prototype._parseNextRefToken = function (expr) {

        // the matches of the regular expressions
        var matches = null;
        // cell references built from the matches of a regular expression
        var cellRefs = null;

        // local cell range reference (before cell references!), e.g. $A$1:$C$3
        if ((matches = this.RE.CELL_RANGE_REF.exec(expr)) && (cellRefs = this._createA1CellRangeData(matches, 1, 5))) {
            return this.createReferenceToken(null, cellRefs, matches[0]);
            // otherwise, continue with other REs, it may be a local name such as 'A1:name42'
        }

        // cell range reference with sheet (before cell references!), e.g. Sheet1!$A$1:$C$3
        if ((matches = this.RE.CELL_RANGE_REL3D_REF.exec(expr)) && (cellRefs = this._createA1CellRangeData(matches, 5, 9))) {
            return this.createReferenceToken(this._createSheetRefData(matches, 1), cellRefs, matches[0]);
            // otherwise, continue with other REs, it may be a local name such as 'Sheet1!A1:name42'
        }

        // cell range reference in multiple sheets (before cell references!), e.g. Sheet1:Sheet3!$A$1:$C$3
        if ((matches = this.RE.CELL_RANGE_RELCUBE_REF.exec(expr)) && (cellRefs = this._createA1CellRangeData(matches, 7, 11))) {
            return this.createReferenceToken(this._createSheetRangeData(matches, 1), cellRefs, matches[0]);
            // otherwise, continue with other REs, it may be a local name such as 'Sheet1:Sheet2!A1:name42'
        }

        // local cell reference, e.g. $A$1
        if ((matches = this.RE.CELL_REF.exec(expr)) && (cellRefs = this._createA1CellData(matches, 1))) {
            return this.createReferenceToken(null, cellRefs, matches[0]);
            // otherwise, continue with other REs, it may be a local name such as 'name42'
        }

        // cell reference with sheet, e.g. Sheet1!$A$1
        if ((matches = this.RE.CELL_REL3D_REF.exec(expr)) && (cellRefs = this._createA1CellData(matches, 5))) {
            return this.createReferenceToken(this._createSheetRefData(matches, 1), cellRefs, matches[0]);
            // otherwise, continue with other REs, it may be a local name such as 'name42'
        }

        // cell reference in multiple sheets, e.g. Sheet1:Sheet3!$A$1
        if ((matches = this.RE.CELL_RELCUBE_REF.exec(expr)) && (cellRefs = this._createA1CellData(matches, 7))) {
            return this.createReferenceToken(this._createSheetRangeData(matches, 1), cellRefs, matches[0]);
            // otherwise, continue with other REs, it may be a local name such as 'Sheet1:Sheet2!name42'
        }

        // local column range reference, e.g. $A:$C
        if ((matches = this.RE.COL_RANGE_REF.exec(expr)) && (cellRefs = this._createA1ColRangeData(matches, 1, 3))) {
            return this.createReferenceToken(null, cellRefs, matches[0]);
            // otherwise, continue with other REs, it may be a range of names such as 'name:name'
        }

        // column range reference with sheet, e.g. Sheet1!$A:$C
        if ((matches = this.RE.COL_RANGE_REL3D_REF.exec(expr)) && (cellRefs = this._createA1ColRangeData(matches, 5, 7))) {
            return this.createReferenceToken(this._createSheetRefData(matches, 1), cellRefs, matches[0]);
            // otherwise, continue with other REs, it may be a range of names such as 'name:name'
        }

        // column range reference in multiple sheets, e.g. Sheet1:Sheet3!$A:$C
        if ((matches = this.RE.COL_RANGE_RELCUBE_REF.exec(expr)) && (cellRefs = this._createA1ColRangeData(matches, 7, 9))) {
            return this.createReferenceToken(this._createSheetRangeData(matches, 1), cellRefs, matches[0]);
            // otherwise, continue with other REs, it may be a range of names such as 'Sheet1:Sheet2!name:name'
        }

        // local row range reference, e.g. $1:$3
        if ((matches = this.RE.ROW_RANGE_REF.exec(expr)) && (cellRefs = this._createA1RowRangeData(matches, 1, 3))) {
            return this.createReferenceToken(null, cellRefs, matches[0]);
            // otherwise, continue with other REs
        }

        // row range reference with sheet, e.g. Sheet1!$1:$3
        if ((matches = this.RE.ROW_RANGE_REL3D_REF.exec(expr)) && (cellRefs = this._createA1RowRangeData(matches, 5, 7))) {
            return this.createReferenceToken(this._createSheetRefData(matches, 1), cellRefs, matches[0]);
            // otherwise, continue with other REs
        }

        // row range reference in multiple sheets, e.g. Sheet1:Sheet3!$1:$3
        if ((matches = this.RE.ROW_RANGE_RELCUBE_REF.exec(expr)) && (cellRefs = this._createA1RowRangeData(matches, 7, 9))) {
            return this.createReferenceToken(this._createSheetRangeData(matches, 1), cellRefs, matches[0]);
            // otherwise, continue with other REs
        }

        // reference error with sheet, e.g. Sheet1!#REF!
        if ((matches = this.RE.ERROR_REL3D_REF.exec(expr))) {
            return this.createReferenceToken(this._createSheetRefData(matches, 1), null, matches[0]);
        }

        // reference error in multiple sheets, e.g. Sheet1:Sheet3!#REF!
        if ((matches = this.RE.ERROR_RELCUBE_REF.exec(expr))) {
            return this.createReferenceToken(this._createSheetRangeData(matches, 1), null, matches[0]);
        }
    };

    /**
     * Extracts a name or function token from the passed formula subexpression.
     */
    ODFUIExpressionParser.prototype._parseNextNameToken = function (expr) {

        // the matches of the regular expressions
        var matches = null;

        // local function call, or global macro call, e.g. SUM() or my_macro()
        if ((matches = this.RE.FUNCTION_REF.exec(expr))) {
            return this.createFunctionToken(null, matches[1], matches[0]);
        }

        // defined name without sheet reference
        if ((matches = this.RE.NAME_REF.exec(expr))) {
            return this.createNameToken(null, matches[1], matches[0]);
        }
    };

    // class OFExpressionParser ===============================================

    /**
     * A formula expression parser for the OpenFormula reference syntax.
     *
     * @constructor
     *
     * @extends ExpressionParser
     */
    var OFExpressionParser = ExpressionParser.extend({ constructor: function (docModel, grammarConfig, initOptions) {

        // base constructor ---------------------------------------------------

        ExpressionParser.call(this, docModel, grammarConfig, initOptions);

    } }); // class OFExpressionParser

    // protected methods ------------------------------------------------------

    /**
     * Returns an object with two SheetRef instances for the passed RE matches
     * of a sheet range.
     */
    OFExpressionParser.prototype._createSheetRangeData = function (matches, index1, index2) {
        return { r1: this._createSheetRef(matches, index1), r2: this._createSheetRef(matches, index2) };
    };

    /**
     * Extracts a reference token from the passed formula subexpression.
     */
    OFExpressionParser.prototype._parseNextRefToken = function (expr) {

        // the matches of the regular expressions
        var matches = null;

        // local cell reference, e.g. [.$A$1]
        if ((matches = this.RE.OF_CELL_REF.exec(expr))) {
            return this.createReferenceToken(null, this._createA1CellData(matches, 1), matches[0]);
        }

        // cell reference with sheet, e.g. [$Sheet1.$A$1]
        if ((matches = this.RE.OF_CELL_3D_REF.exec(expr))) {
            return this.createReferenceToken(this._createSheetRefData(matches, 1), this._createA1CellData(matches, 5), matches[0]);
        }

        // local cell range reference, e.g. [.$A$1:.$C$3]
        if ((matches = this.RE.OF_CELL_RANGE_REF.exec(expr))) {
            return this.createReferenceToken(null, this._createA1CellRangeData(matches, 1, 5), matches[0]);
        }

        // cell range reference with sheet, e.g. [$Sheet1.$A$1:.$C$3]
        if ((matches = this.RE.OF_CELL_RANGE_3D_REF.exec(expr))) {
            return this.createReferenceToken(this._createSheetRefData(matches, 1), this._createA1CellRangeData(matches, 5, 9), matches[0]);
        }

        // cell range reference in multiple sheets, e.g. [$Sheet1.$A$1:$Sheet3.$C$3]
        if ((matches = this.RE.OF_CELL_RANGE_CUBE_REF.exec(expr))) {
            return this.createReferenceToken(this._createSheetRangeData(matches, 1, 9), this._createA1CellRangeData(matches, 5, 13), matches[0]);
        }
    };

    /**
     * Extracts a name or function token from the passed formula subexpression.
     */
    OFExpressionParser.prototype._parseNextNameToken = function (expr) {

        // the matches of the regular expressions
        var matches = null;

        // local function call, or global macro call, with optional whitespace, e.g. SUM () or my_macro()
        if ((matches = this.RE.OF_FUNCTION_REF.exec(expr))) {
            return this.createFunctionToken(null, matches[1], matches[0]);
        }

        // defined name without sheet reference
        if ((matches = this.RE.NAME_REF.exec(expr))) {
            return this.createNameToken(null, matches[1], matches[0]);
        }
    };

    // class Tokenizer ========================================================

    /**
     * Parses formula expressions to arrays of formula tokens.
     *
     * @constructor
     *
     * @extends ModelObject
     *
     * @param {SpreadsheetModel} docModel
     *  The spreadsheet document model containing this instance.
     */
    function Tokenizer(docModel) {

        // base constructor ---------------------------------------------------

        ModelObject.call(this, docModel);

        // public methods -----------------------------------------------------

        /**
         * Converts the passed formula expression to an array of formula
         * tokens.
         *
         * @param {String} grammar
         *  The identifier of the formula grammar to be used to parse the
         *  formula expression. See class GrammarConfig for details.
         *
         * @param {String} formula
         *  The formula string, without the leading equality sign.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Number} [options.extendSheet]
         *      If specified, all cell references without a sheet reference,
         *      and all references to existing sheet-local names will be
         *      extended with a sheet reference pointing to the sheet contained
         *      in this option.
         *      Example: The formula expression 'A1+global+local' extended to
         *      the first sheet may become 'Sheet1!A1+global+Sheet1!local'.
         *
         * @returns {Array<TokenDescriptor>}
         *  An array of descriptors for all parsed formula tokens.
         */
        this.parseFormula = function (grammar, formula, options) {
            FormulaUtils.info('Tokenizer.parseFormula(): grammar="' + grammar + '", formula="' + formula + '"');

            // the grammar configuration containing resources and regular expressions for the parser
            var grammarConfig = docModel.getGrammarConfig(grammar);

            // the parser instance representing state and result of this method call
            var exprParser = (function () {
                switch (grammarConfig.REF_SYNTAX) {
                    case 'ooxml':
                        return new OOXMLExpressionParser(docModel, grammarConfig, options);
                    case 'odfui':
                        return new ODFUIExpressionParser(docModel, grammarConfig, options);
                    case 'of':
                        return new OFExpressionParser(docModel, grammarConfig, options);
                }
            }());

            // parse the raw formula tokens
            var tokenDescs = exprParser.parseFormula(formula);
            // whether the intersection operator is a space and needs post-processing
            var spaceAsIsect = grammarConfig.getOperatorName('isect') === ' ';

            // post-process the token array
            FormulaUtils.logTokens('\xa0 tokenize', tokenDescs);
            Utils.iterateArray(tokenDescs, function (tokenDesc, index) {

                var // the current token
                    token = tokenDesc.token,
                    // the token preceding and following the current token
                    prevTokenDesc = tokenDescs[index - 1],
                    nextTokenDesc = tokenDescs[index + 1],
                    // the next non-whitespace token preceding the current token
                    prevNonWsTokenDesc = (prevTokenDesc && prevTokenDesc.token.isType('ws')) ? tokenDescs[index - 2] : prevTokenDesc;

                // replace combination of 'op[sub] lit[number]' with a negative number, if the
                // minus is the leading token or preceded by an operator or opening parenthesis
                var nextValue = null;
                if (token.isType('op') && (token.getValue() === 'sub') &&
                    nextTokenDesc && nextTokenDesc.token.isType('lit') && _.isNumber(nextValue = nextTokenDesc.token.getValue()) && (nextValue > 0) &&
                    (!prevNonWsTokenDesc || prevNonWsTokenDesc.token.matchesType(/^(op|sep|open)$/))
                ) {
                    tokenDesc.token = new Tokens.LiteralToken(-nextValue);
                    tokenDesc.text += nextTokenDesc.text;
                    tokenDescs.splice(index + 1, 1);
                }

                // if the range intersection operator is a space character, insert an operator
                // token for a white-space token containing a SPACE character and surrounded
                // by reference/name tokens or other subexpressions in parentheses
                var spaceIndex = null;
                if (spaceAsIsect && token.isType('ws') && ((spaceIndex = tokenDesc.text.indexOf(' ')) >= 0) &&
                    prevTokenDesc && prevTokenDesc.token.matchesType(/^(ref|name|close)$/) &&
                    nextTokenDesc && nextTokenDesc.token.matchesType(/^(ref|name|open|func)$/)
                ) {

                    // insert the remaining white-space following the first SPACE character
                    if (spaceIndex + 1 < tokenDesc.text.length) {
                        var trailingSpace = tokenDesc.text.slice(spaceIndex + 1);
                        tokenDescs.splice(index + 1, 0, exprParser.createFixedToken('ws', trailingSpace));
                    }

                    // insert an intersection operator for the SPACE character after the current token
                    tokenDescs.splice(index + 1, 0, exprParser.createToken(new Tokens.OperatorToken('isect'), ' '));

                    // shorten the current white-space token to the text preceding the first SPACE character, or remove it
                    if (spaceIndex > 0) {
                        token.setValue(tokenDesc.text.slice(0, spaceIndex));
                        tokenDesc.text = token.getValue();
                    } else {
                        tokenDescs.splice(index, 1);
                    }
                }

            }, { reverse: true, context: this });
            FormulaUtils.logTokens('\xa0 postprocess', tokenDescs);

            // add text positions to the token descriptors
            tokenDescs.forEach(function (tokenDesc, index) {
                tokenDesc.index = index;
                tokenDesc.start = (index === 0) ? 0 : tokenDescs[index - 1].end;
                tokenDesc.end = tokenDesc.start + tokenDesc.text.length;
            });

            return tokenDescs;
        };

        // initialization -----------------------------------------------------

        // destroy all class members on destruction
        this.registerDestructor(function () {
            docModel = null;
        });

    } // class Tokenizer

    // exports ================================================================

    // derive this class from class ModelObject
    return ModelObject.extend({ constructor: Tokenizer });

});
