/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/formula/impl/statisticalfuncs', [
    'io.ox/office/tk/utils',
    'io.ox/office/spreadsheet/utils/errorcode',
    'io.ox/office/spreadsheet/model/formula/formulautils'
], function (Utils, ErrorCode, FormulaUtils) {

    'use strict';

    /**************************************************************************
     *
     * This module implements all statistical spreadsheet functions.
     *
     * See the README file in this directory for a detailed documentation about
     * the format of function descriptor objects.
     *
     *************************************************************************/

    // standard options for numeric parameter iterators (method FormulaContext.iterateNumbers())
    var SKIP_MAT_SKIP_REF = { // id: CSS5
        valMode: 'convert', // value operands: convert strings and booleans to numbers
        matMode: 'skip', // matrix operands: skip strings and booleans
        refMode: 'skip', // reference operands: skip strings and booleans
        emptyParam: true, // empty parameters count as zero
        complexRef: true // accept multi-range and multi-sheet references
    };

    // standard options for numeric parameter iterators (method FormulaContext.iterateNumbers())
    var SKIP_MAT_ZERO_REF = { // id: CSZ5
        valMode: 'convert', // value operands: convert strings and booleans to numbers
        matMode: 'skip', // matrix operands: skip strings and booleans
        refMode: 'zero', // reference operands: replace all strings with zeros
        emptyParam: true, // empty parameters count as zero
        complexRef: true // accept multi-range and multi-sheet references
    };

    // standard options for numeric parameter iterators (method FormulaContext.iterateNumbers())
    var ZERO_MAT_ZERO_REF = { // id: CZZ5
        valMode: 'convert', // value operands: convert strings and booleans to numbers
        matMode: 'zero', // matrix operands: replace all strings with zeros
        refMode: 'zero', // reference operands: replace all strings with zeros
        emptyParam: true, // empty parameters count as zero
        complexRef: true // accept multi-range and multi-sheet references
    };

    // private global functions ===============================================

    // numeric aggregation ----------------------------------------------------

    /**
     * Creates and returns a resolver function that reduces all numbers of all
     * function parameters to their minimum or maximum.
     *
     * @param {String} method
     *  The aggregation method to be implemented, either 'min' or 'max'.
     *
     * @param {Object} iteratorOptions
     *  Parameters passed to the number iterator used internally.
     *
     * @returns {Function}
     *  The resulting function implementation to be assigned to the 'resolve'
     *  property of a function descriptor. The spreadsheet function will pass
     *  all numbers of all its operands to the aggregation callback function.
     */
    function implementMinMaxAggregation(method, iteratorOptions) {

        var // start with signed infinity according to the function type
            initial = (method === 'max') ? Number.NEGATIVE_INFINITY : Number.POSITIVE_INFINITY,
            // the aggregation function (Math.min or Math.max)
            aggregate = _.bind(Math[method], Math);

        // default result (no numbers found in parameters) is zero (no #NUM! error code from infinity)
        function finalize(result) { return isFinite(result) ? result : 0; }

        // create and return the resolver function to be assigned to the 'resolve' property of a function descriptor
        return FormulaUtils.implementNumericAggregation(initial, aggregate, finalize, iteratorOptions);
    }

    /**
     * Creates and returns a resolver function that reduces all numbers of all
     * function parameters to their standard deviation, or their variance.
     *
     * @param {String} method
     *  The aggregation method to be implemented, either 'dev' for the standard
     *  deviation, or 'var' for the variance.
     *
     * @param {Boolean} population
     *  If set to true, calculates the standard deviation or variance based on
     *  the entire population. If set to false, calculates the result based on
     *  a sample.
     *
     * @param {Object} iteratorOptions
     *  Parameters passed to the number iterator used internally.
     *
     * @returns {Function}
     *  The resulting function implementation.
     */
    function implementStdDevAggregation(method, population, iteratorOptions) {

        // default result (no numbers found in parameters) is the #DIV/0! error code thrown by divide()
        function finalize(numbers, sum) {

            var mean = FormulaUtils.divide(sum, numbers.length),
                result = Utils.getSum(numbers, function (number) {
                    var diff = number - mean;
                    return diff * diff;
                }),
                size = population ? numbers.length : (numbers.length - 1);

            result = FormulaUtils.divide(result, size);
            return (method === 'dev') ? Math.sqrt(result) : result;
        }

        // create and return the resolver function to be assigned to the 'resolve' property of a function descriptor
        return FormulaUtils.implementNumericAggregationWithArray(finalize, iteratorOptions);
    }

    // exports ================================================================

    return {

        AVEDEV: {
            category: 'statistical',
            minParams: 1,
            type: 'val',
            signature: 'any',
            resolve: (function () {

                // default result (no numbers found in parameters) is the #NUM! error code (not #DIV/0!)
                function finalize(numbers, sum) {
                    if (numbers.length === 0) { throw ErrorCode.NUM; }
                    var mean = sum / numbers.length;
                    return Utils.getSum(numbers, function (number) { return Math.abs(number - mean); }) / numbers.length;
                }

                // empty parameters count as zero: AVEDEV(1,) = 0.5
                return FormulaUtils.implementNumericAggregationWithArray(finalize, SKIP_MAT_SKIP_REF);
            }())
        },

        AVERAGE: {
            category: 'statistical',
            minParams: 1,
            type: 'val',
            signature: 'any',
            // empty parameters count as zero: AVERAGE(1,) = 0.5
            // default result (no numbers found in parameters) is the #DIV/0! error code thrown by FormulaUtils.divide()
            resolve: FormulaUtils.implementNumericAggregation(0, FormulaUtils.add, FormulaUtils.divide, SKIP_MAT_SKIP_REF)
        },

        AVERAGEA: {
            category: 'statistical',
            minParams: 1,
            type: 'val',
            signature: 'any',
            // empty parameters count as zero: AVERAGEA(1,) = 0.5
            // default result (no numbers found in parameters) is the #DIV/0! error code thrown by FormulaUtils.divide()
            resolve: FormulaUtils.implementNumericAggregation(0, FormulaUtils.add, FormulaUtils.divide, ZERO_MAT_ZERO_REF)
        },

        AVERAGEIF: {
            category: 'statistical',
            ceName: null,
            minParams: 2,
            maxParams: 3,
            type: 'val',
            signature: 'ref val:any ref',
            resolve: FormulaUtils.implementFilterAggregation(0, FormulaUtils.add, FormulaUtils.divide)
        },

        AVERAGEIFS: {
            category: 'statistical',
            ceName: null,
            minParams: 3,
            repeatParams: 2,
            type: 'val',
            signature: 'ref ref val:any'
        },

        'BETA.DIST': {
            category: 'statistical',
            name: { ooxml: '_xlfn.BETA.DIST', odf: 'COM.MICROSOFT.BETA.DIST' },
            ceName: null,
            minParams: 4,
            maxParams: 6,
            type: 'val'
        },

        'BETA.INV': {
            category: 'statistical',
            name: { ooxml: '_xlfn.BETA.INV', odf: 'COM.MICROSOFT.BETA.INV' },
            ceName: null,
            minParams: 3,
            maxParams: 5,
            type: 'val'
        },

        BETADIST: {
            category: 'statistical',
            minParams: 3,
            maxParams: 5,
            type: 'val'
        },

        BETAINV: {
            category: 'statistical',
            minParams: 3,
            maxParams: 5,
            type: 'val'
        },

        'BINOM.DIST': {
            category: 'statistical',
            name: { ooxml: '_xlfn.BINOM.DIST', odf: 'COM.MICRSOFT.BINOM.DIST' },
            ceName: null,
            minParams: 4,
            maxParams: 4,
            type: 'val'
        },

        'BINOM.DIST.RANGE': {
            category: 'statistical',
            // AOO 4.4 writes 'B' (https://bz.apache.org/ooo/show_bug.cgi?id=126519)
            name: { ooxml: '_xlfn.BINOM.DIST.RANGE', odf: ['BINOM.DIST.RANGE', 'B'] },
            ceName: { ooxml: null, odf: 'B' }, // TODO: CalcEngine should allow to use this function in ODF
            minParams: 3,
            maxParams: 4,
            type: 'val'
        },

        'BINOM.INV': {
            category: 'statistical',
            name: { ooxml: '_xlfn.BINOM.INV', odf: 'COM.MICRSOFT.BINOM.INV' },
            ceName: null,
            minParams: 3,
            maxParams: 3,
            type: 'val'
        },

        BINOMDIST: {
            category: 'statistical',
            minParams: 4,
            maxParams: 4,
            type: 'val'
        },

        CHIDIST: {
            category: 'statistical',
            name: { odf: 'LEGACY.CHIDIST' },
            minParams: 2,
            maxParams: 2,
            type: 'val'
        },

        CHIINV: {
            category: 'statistical',
            name: { odf: 'LEGACY.CHIINV' },
            minParams: 2,
            maxParams: 2,
            type: 'val'
        },

        'CHISQ.DIST': {
            category: 'statistical',
            name: { ooxml: '_xlfn.CHISQ.DIST', odf: 'COM.MICROSOFT.CHISQ.DIST' },
            ceName: { odf: null },
            minParams: 3,
            maxParams: 3,
            type: 'val'
        },

        'CHISQ.DIST.RT': {
            category: 'statistical',
            name: { ooxml: '_xlfn.CHISQ.DIST.RT', odf: 'COM.MICROSOFT.CHISQ.DIST.RT' },
            ceName: null,
            minParams: 2,
            maxParams: 2,
            type: 'val'
        },

        'CHISQ.INV': {
            category: 'statistical',
            name: { ooxml: '_xlfn.CHISQ.INV', odf: 'COM.MICROSOFT.CHISQ.INV' },
            ceName: { odf: null },
            minParams: 2,
            maxParams: 2,
            type: 'val'
        },

        'CHISQ.INV.RT': {
            category: 'statistical',
            name: { ooxml: '_xlfn.CHISQ.INV.RT', odf: 'COM.MICROSOFT.CHISQ.INV.RT' },
            ceName: null,
            minParams: 2,
            maxParams: 2,
            type: 'val'
        },

        'CHISQ.TEST': {
            category: 'statistical',
            name: { ooxml: '_xlfn.CHISQ.TEST', odf: 'COM.MICROSOFT.CHISQ.TEST' },
            ceName: null,
            minParams: 2,
            maxParams: 2,
            type: 'val'
        },

        CHISQDIST: {
            category: 'statistical',
            name: { ooxml: null },
            ceName: { ooxml: null },
            minParams: 2,
            maxParams: 3,
            type: 'val'
        },

        CHISQINV: {
            category: 'statistical',
            name: { ooxml: null },
            ceName: { ooxml: null },
            minParams: 2,
            maxParams: 2,
            type: 'val'
        },

        CHITEST: {
            category: 'statistical',
            name: { odf: 'LEGACY.CHITEST' },
            minParams: 2,
            maxParams: 2,
            type: 'val'
        },

        CONFIDENCE: {
            category: 'statistical',
            minParams: 3,
            maxParams: 3,
            type: 'val'
        },

        'CONFIDENCE.NORM': {
            category: 'statistical',
            name: { ooxml: '_xlfn.CONFIDENCE.NORM', odf: 'COM.MICROSOFT.CONFIDENCE.NORM' },
            ceName: null,
            minParams: 3,
            maxParams: 3,
            type: 'val'
        },

        'CONFIDENCE.T': {
            category: 'statistical',
            name: { ooxml: '_xlfn.CONFIDENCE.T', odf: 'COM.MICROSOFT.CONFIDENCE.T' },
            ceName: null,
            minParams: 3,
            maxParams: 3,
            type: 'val'
        },

        CORREL: {
            category: 'statistical',
            minParams: 2,
            maxParams: 2,
            type: 'val'
        },

        COUNT: {
            category: 'statistical',
            minParams: 1,
            type: 'val',
            signature: 'any',
            resolve: function () {
                var count = 0;
                this.iterateOperands(0, function (operand) {
                    if (operand.isValue()) {
                        // count booleans and strings that are convertible to numbers
                        // (simple values only, not counted in matrixes and references)
                        try {
                            this.convertToNumber(operand.getValue());
                            count += 1;
                        } catch (ex) {}
                    } else {
                        // matrixes and references: count real numbers only (no strings, no booleans)
                        this.iterateValues(operand, function (value) {
                            if (_.isNumber(value)) { count += 1; }
                        }, {
                            acceptErrors: true, // do not fail on error codes: =COUNT({1,#VALUE!}) = 1
                            complexRef: true // accept multi-range and multi-sheet references
                        });
                    }
                });
                return count;
            }
        },

        COUNTA: {
            category: 'statistical',
            minParams: 1,
            type: 'val',
            signature: 'any',
            resolve: function () {
                var count = 0;
                this.iterateOperands(0, function (operand) {
                    switch (operand.getType()) {
                        case 'val':
                            // all values are counted (also zeros, empty strings, FALSE,
                            // error codes, empty parameters): COUNTA(1,,#VALUE!) = 3
                            count += 1;
                            break;
                        case 'mat':
                            // all matrix elements are counted (also zeros, empty strings, FALSE, error codes)
                            count += operand.getMatrix().size();
                            break;
                        case 'ref':
                            // all filled cells are counted (also error codes)
                            this.iterateValues(operand, function () { count += 1; }, {
                                acceptErrors: true, // count all error codes
                                complexRef: true // accept multi-range and multi-sheet references
                            });
                    }
                });
                return count;
            }
        },

        COUNTBLANK: {
            category: 'statistical',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'ref',
            resolve: function (ranges) {

                var // number of non-blank cells
                    count = 0,
                    // whether empty strings (formula results) are blank (Excel only)
                    emptyStr = this.getFileFormat() === 'ooxml';

                // count all non-blank cells in the reference
                this.iterateValues(ranges, function (value) {
                    if (!emptyStr || (value !== '')) { count += 1; }
                }, {
                    acceptErrors: true, // count all error codes
                    complexRef: false // do not accept multi-range and multi-sheet references
                });

                // return number of remaining blank cells
                return ranges.cells() - count;
            }
        },

        COUNTIF: {
            category: 'statistical',
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: 'ref val:any',
            // COUNTIF returns the count of all matching cells in the source range (last parameter of the finalizer)
            resolve: FormulaUtils.implementFilterAggregation(0, _.identity, function (result, count1, count2) { return count2; })
        },

        COUNTIFS: {
            category: 'statistical',
            ceName: null,
            minParams: 2,
            repeatParams: 2,
            type: 'val'
        },

        COVAR: {
            category: 'statistical',
            minParams: 2,
            maxParams: 2,
            type: 'val'
        },

        'COVARIANCE.P': {
            category: 'statistical',
            name: { ooxml: '_xlfn.COVARIANCE.P', odf: 'COM.MICROSOFT.COVARIANCE.P' },
            ceName: null,
            minParams: 2,
            maxParams: 2,
            type: 'val'
        },

        'COVARIANCE.S': {
            category: 'statistical',
            name: { ooxml: '_xlfn.COVARIANCE.S', odf: 'COM.MICROSOFT.COVARIANCE.S' },
            ceName: null,
            minParams: 2,
            maxParams: 2,
            type: 'val'
        },

        CRITBINOM: {
            category: 'statistical',
            minParams: 3,
            maxParams: 3,
            type: 'val'
        },

        DEVSQ: {
            category: 'statistical',
            minParams: 1,
            type: 'val',
            signature: 'any',
            resolve: (function () {

                // default result (no numbers found in parameters) is the #NUM! error code (not #DIV/0!)
                function finalize(numbers, sum) {
                    if (numbers.length === 0) { throw ErrorCode.NUM; }
                    var mean = sum / numbers.length;
                    return Utils.getSum(numbers, function (number) {
                        var diff = number - mean;
                        return diff * diff;
                    });
                }

                // empty parameters count as zero: DEVSQ(1,) = 0.5
                return FormulaUtils.implementNumericAggregationWithArray(finalize, SKIP_MAT_SKIP_REF);
            }())
        },

        'EXPON.DIST': {
            category: 'statistical',
            name: { ooxml: '_xlfn.EXPON.DIST', odf: 'COM.MICROSOFT.EXPON.DIST' },
            ceName: null,
            minParams: 3,
            maxParams: 3,
            type: 'val'
        },

        EXPONDIST: {
            category: 'statistical',
            minParams: 3,
            maxParams: 3,
            type: 'val'
        },

        'F.DIST': {
            category: 'statistical',
            // accept 'COM.MICROSOFT.F.DIST' in ODF (http://opengrok.libreoffice.org/xref/core/sc/source/core/tool/compiler.cxx#2628)
            name: { ooxml: '_xlfn.F.DIST', odf: ['FDIST', 'COM.MICROSOFT.F.DIST'] },
            ceName: null,
            minParams: 4,
            maxParams: 4,
            type: 'val'
        },

        'F.DIST.RT': {
            category: 'statistical',
            name: { ooxml: '_xlfn.F.DIST.RT', odf: 'COM.MICROSOFT.F.DIST.RT' },
            ceName: null,
            minParams: 4,
            maxParams: 4,
            type: 'val'
        },

        'F.INV': {
            category: 'statistical',
            // LO 5.0 writes 'COM.MICROSOFT.F.INV' (https://bugs.documentfoundation.org/show_bug.cgi?id=94214)
            name: { ooxml: '_xlfn.F.INV', odf: ['FINV', 'COM.MICROSOFT.F.INV'] },
            ceName: null,
            minParams: 3,
            maxParams: 3,
            type: 'val'
        },

        'F.INV.RT': {
            category: 'statistical',
            name: { ooxml: '_xlfn.F.INV.RT', odf: 'COM.MICROSOFT.F.INV.RT' },
            ceName: null,
            minParams: 3,
            maxParams: 3,
            type: 'val'
        },

        'F.TEST': {
            category: 'statistical',
            name: { ooxml: '_xlfn.F.TEST', odf: 'COM.MICROSOFT.F.TEST' },
            ceName: null,
            minParams: 3,
            maxParams: 3,
            type: 'val'
        },

        FDIST: {
            category: 'statistical',
            name: { odf: 'LEGACY.FDIST' },
            minParams: 3,
            maxParams: 3,
            type: 'val'
        },

        FINV: {
            category: 'statistical',
            name: { odf: 'LEGACY.FINV' },
            minParams: 3,
            maxParams: 3,
            type: 'val'
        },

        FISHER: {
            category: 'statistical',
            minParams: 1,
            maxParams: 1,
            type: 'val'
        },

        FISHERINV: {
            category: 'statistical',
            minParams: 1,
            maxParams: 1,
            type: 'val'
        },

        FORECAST: {
            category: 'statistical',
            minParams: 3,
            maxParams: 3,
            type: 'val'
        },

        FREQUENCY: {
            category: 'statistical',
            minParams: 2,
            maxParams: 2,
            type: 'mat'
        },

        FTEST: {
            category: 'statistical',
            minParams: 2,
            maxParams: 2,
            type: 'val'
        },

        GAMMA: {
            category: 'statistical',
            name: { ooxml: '_xlfn.GAMMA' },
            minParams: 1,
            maxParams: 1,
            type: 'val'
        },

        'GAMMA.DIST': {
            category: 'statistical',
            name: { ooxml: '_xlfn.GAMMA.DIST', odf: 'COM.MICROSOFT.GAMMA.DIST' },
            ceName: null,
            minParams: 4,
            maxParams: 4,
            type: 'val'
        },

        'GAMMA.INV': {
            category: 'statistical',
            name: { ooxml: '_xlfn.GAMMA.INV', odf: 'COM.MICROSOFT.GAMMA.INV' },
            ceName: null,
            minParams: 3,
            maxParams: 3,
            type: 'val'
        },

        GAMMADIST: {
            category: 'statistical',
            minParams: 4,
            maxParams: 4,
            type: 'val'
        },

        GAMMAINV: {
            category: 'statistical',
            minParams: 3,
            maxParams: 3,
            type: 'val'
        },

        GAMMALN: {
            category: 'statistical',
            minParams: 1,
            maxParams: 1,
            type: 'val'
        },

        'GAMMALN.PRECISE': {
            category: 'statistical',
            name: { ooxml: '_xlfn.GAMMALN.PRECISE', odf: 'COM.MICROSOFT.GAMMALN.PRECISE' },
            ceName: null,
            minParams: 1,
            maxParams: 1,
            type: 'val'
        },

        GAUSS: {
            category: 'statistical',
            name: { ooxml: '_xlfn.GAUSS' },
            minParams: 1,
            maxParams: 1,
            type: 'val'
        },

        GEOMEAN: {
            category: 'statistical',
            minParams: 1,
            type: 'val',
            signature: 'any',
            resolve: (function () {

                // default result (no numbers found in parameters) is the #NUM! error code
                // a single 0 in the parameters results in the #NUM! error code
                function finalize(result, count) {
                    if ((result === 0) || (count === 0)) { throw ErrorCode.NUM; }
                    return Math.pow(result, 1 / count);
                }

                // empty parameters count as zero: GEOMEAN(1,) = #NUM!
                return FormulaUtils.implementNumericAggregation(1, FormulaUtils.multiply, finalize, SKIP_MAT_SKIP_REF);
            }())
        },

        GROWTH: {
            category: 'statistical',
            minParams: 1,
            maxParams: 4,
            type: 'mat'
        },

        HARMEAN: {
            category: 'statistical',
            minParams: 1,
            type: 'val',
            signature: 'any',
            resolve: (function () {

                // adds reciprocal of the passed number to the intermediate result
                function aggregate(result, number) {
                    // a single 0 in the parameters results in the #NUM! error code
                    if (number === 0) { throw ErrorCode.NUM; }
                    return result + 1 / number;
                }

                // default result (no numbers found in parameters) is the #N/A error code
                function finalize(result, count) {
                    if (count === 0) { throw ErrorCode.NA; }
                    return count / result;
                }

                // empty parameters count as zero: GEOMEAN(1,) = #NUM!
                return FormulaUtils.implementNumericAggregation(0, aggregate, finalize, SKIP_MAT_SKIP_REF);
            }())
        },

        'HYPGEOM.DIST': {
            category: 'statistical',
            name: { ooxml: '_xlfn.HYPGEOM.DIST', odf: 'COM.MICROSOFT.HYPGEOM.DIST' },
            ceName: null,
            minParams: 5,
            maxParams: 5,
            type: 'val'
        },

        HYPGEOMDIST: {
            category: 'statistical',
            minParams: 4,
            maxParams: 4,
            type: 'val'
        },

        INTERCEPT: {
            category: 'statistical',
            minParams: 2,
            maxParams: 2,
            type: 'val'
        },

        KURT: {
            category: 'statistical',
            minParams: 1,
            type: 'val'
        },

        LARGE: {
            category: 'statistical',
            minParams: 2,
            maxParams: 2,
            type: 'val'
        },

        LINEST: {
            category: 'statistical',
            minParams: 1,
            maxParams: 4,
            type: 'mat'
        },

        LOGEST: {
            category: 'statistical',
            minParams: 1,
            maxParams: 4,
            type: 'mat'
        },

        LOGINV: {
            category: 'statistical',
            minParams: 3,
            maxParams: 3,
            type: 'val'
        },

        'LOGNORM.DIST': {
            category: 'statistical',
            name: { ooxml: '_xlfn.LOGNORM.DIST', odf: 'COM.MICROSOFT.LOGNORM.DIST' },
            ceName: null,
            minParams: 4,
            maxParams: 4,
            type: 'val'
        },

        'LOGNORM.INV': {
            category: 'statistical',
            name: { ooxml: '_xlfn.LOGNORM.INV', odf: 'COM.MICROSOFT.LOGNORM.INV' },
            ceName: null,
            minParams: 3,
            maxParams: 3,
            type: 'val'
        },

        LOGNORMDIST: {
            category: 'statistical',
            minParams: 3,
            maxParams: 3,
            type: 'val'
        },

        MAX: {
            category: 'statistical',
            minParams: 1,
            type: 'val',
            signature: 'any',
            // empty parameters count as zero: MAX(-1,) = 0
            resolve: implementMinMaxAggregation('max', SKIP_MAT_SKIP_REF)
        },

        MAXA: {
            category: 'statistical',
            minParams: 1,
            type: 'val',
            signature: 'any',
            // empty parameters count as zero: MAXA(-1,) = 0
            resolve: implementMinMaxAggregation('max', SKIP_MAT_ZERO_REF)
        },

        MEDIAN: {
            category: 'statistical',
            minParams: 1,
            type: 'val',
            signature: 'any',
            resolve: (function () {

                // default result (no numbers found in parameters) is the #NUM! error code
                function finalize(numbers) {
                    var count = numbers.length;
                    if (count === 0) { throw ErrorCode.NUM; }
                    numbers = _.sortBy(numbers);
                    // even array length: return arithmetic mean of both numbers in the middle of the array
                    return (count % 2 === 0) ? ((numbers[count / 2 - 1] + numbers[count / 2]) / 2) : numbers[(count - 1) / 2];
                }

                // create and return the resolver function to be assigned to the 'resolve' option of a function descriptor
                return FormulaUtils.implementNumericAggregationWithArray(finalize, SKIP_MAT_SKIP_REF);
            }())
        },

        MIN: {
            category: 'statistical',
            minParams: 1,
            type: 'val',
            signature: 'any',
            // empty parameters count as zero: MIN(1,) = 0
            resolve: implementMinMaxAggregation('min', SKIP_MAT_SKIP_REF)
        },

        MINA: {
            category: 'statistical',
            minParams: 1,
            type: 'val',
            signature: 'any',
            // empty parameters count as zero: MINA(1,) = 0
            resolve: implementMinMaxAggregation('min', SKIP_MAT_ZERO_REF)
        },

        MODE: {
            category: 'statistical',
            minParams: 1,
            type: 'val',
            signature: 'any',
            resolve: (function () {

                // finds the most used number in all collected numbers
                function finalize(numbers) {

                    // default result (all parameters skipped) is the #N/A error code
                    if (numbers.length === 0) { throw ErrorCode.NA; }

                    var // reduce numbers to their counts (maps each distinct number to its count)
                        counts = _.countBy(numbers),
                        // find the highest count
                        maxCount = _.reduce(counts, function (max, num) { return Math.max(max, num); }, 0),
                        // a map that contains all numbers occuring the most as key
                        map = {};

                    // no duplicate numbers: throw the #N/A error code
                    if (maxCount === 1) { throw ErrorCode.NA; }

                    // insert each number into the map that occurs the most
                    _.each(counts, function (count, number) {
                        if (count === maxCount) { map[number] = true; }
                    });

                    // find the first number in the original array that occurs the most
                    // e.g.: =MODE(3,3,4,4) results in 3; but: =MODE(4,4,3,3) results in 4
                    return _.find(numbers, function (number) { return number in map; });
                }

                // create and return the resolver function to be assigned to the 'resolve' option of a function descriptor
                return FormulaUtils.implementNumericAggregationWithArray(finalize, {
                    valMode: 'exact', // value operands: exact match for numbers (no strings, no booleans)
                    matMode: 'skip', // matrix operands: skip strings and booleans
                    refMode: 'exact', // reference operands: exact match for numbers (no strings, no booleans)
                    emptyParam: true, // empty parameters result in #VALUE! error
                    complexRef: false // do not accept multi-range and multi-sheet references
                });
            }())
        },

        'MODE.SNGL': {
            category: 'statistical',
            name: { ooxml: '_xlfn.MODE.SNGL', odf: 'COM.MICROSOFT.MODE.SNGL' },
            minParams: 1,
            type: 'val',
            signature: 'any',
            resolve: 'MODE'
        },

        'MODE.MULT': {
            category: 'statistical',
            name: { ooxml: '_xlfn.MODE.MULT', odf: 'COM.MICROSOFT.MODE.MULT' },
            ceName: null,
            minParams: 1,
            type: 'mat'
        },

        'NEGBINOM.DIST': {
            category: 'statistical',
            name: { ooxml: '_xlfn.NEGBINOM.DIST', odf: 'COM.MICROSOFT.NEGBINOM.DIST' },
            ceName: null,
            minParams: 4,
            maxParams: 4,
            type: 'val'
        },

        NEGBINOMDIST: {
            category: 'statistical',
            minParams: 3,
            maxParams: 3,
            type: 'val'
        },

        'NORM.DIST': {
            category: 'statistical',
            name: { ooxml: '_xlfn.NORM.DIST', odf: 'COM.MICROSOFT.NORM.DIST' },
            ceName: null,
            minParams: 4,
            maxParams: 4,
            type: 'val'
        },

        'NORM.INV': {
            category: 'statistical',
            name: { ooxml: '_xlfn.NORM.INV', odf: 'COM.MICROSOFT.NORM.INV' },
            ceName: null,
            minParams: 3,
            maxParams: 3,
            type: 'val'
        },

        'NORM.S.DIST': {
            category: 'statistical',
            name: { ooxml: '_xlfn.NORM.S.DIST', odf: 'COM.MICROSOFT.NORM.S.DIST' },
            ceName: null,
            minParams: 2,
            maxParams: 2,
            type: 'val'
        },

        'NORM.S.INV': {
            category: 'statistical',
            name: { ooxml: '_xlfn.NORM.S.INV', odf: 'COM.MICROSOFT.NORM.S.INV' },
            ceName: null,
            minParams: 1,
            maxParams: 1,
            type: 'val'
        },

        NORMDIST: {
            category: 'statistical',
            minParams: 4,
            maxParams: 4,
            type: 'val'
        },

        NORMINV: {
            category: 'statistical',
            minParams: 3,
            maxParams: 3,
            type: 'val'
        },

        NORMSDIST: {
            category: 'statistical',
            minParams: 1,
            maxParams: 1,
            type: 'val'
        },

        NORMSINV: {
            category: 'statistical',
            minParams: 1,
            maxParams: 1,
            type: 'val'
        },

        PEARSON: {
            category: 'statistical',
            minParams: 2,
            maxParams: 2,
            type: 'val'
        },

        PERCENTILE: {
            category: 'statistical',
            minParams: 2,
            maxParams: 2,
            type: 'val'
        },

        'PERCENTILE.EXC': {
            category: 'statistical',
            name: { ooxml: '_xlfn.PERCENTILE.EXC', odf: 'COM.MICROSOFT.PERCENTILE.EXC' },
            ceName: null,
            minParams: 2,
            maxParams: 2,
            type: 'val'
        },

        'PERCENTILE.INC': {
            category: 'statistical',
            name: { ooxml: '_xlfn.PERCENTILE.INC', odf: 'COM.MICROSOFT.PERCENTILE.INC' },
            ceName: null,
            minParams: 2,
            maxParams: 2,
            type: 'val'
        },

        PERCENTRANK: {
            category: 'statistical',
            minParams: 2,
            maxParams: { ooxml: 3, odf: 2 },
            type: 'val'
        },

        'PERCENTRANK.EXC': {
            category: 'statistical',
            name: { ooxml: '_xlfn.PERCENTRANK.EXC', odf: 'COM.MICROSOFT.PERCENTRANK.EXC' },
            ceName: null,
            minParams: 2,
            maxParams: 3,
            type: 'val'
        },

        'PERCENTRANK.INC': {
            category: 'statistical',
            name: { ooxml: '_xlfn.PERCENTRANK.INC', odf: 'COM.MICROSOFT.PERCENTRANK.INC' },
            ceName: null,
            minParams: 2,
            maxParams: 3,
            type: 'val'
        },

        PERMUT: {
            category: 'statistical',
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: 'val:int val:int',
            resolve: function (number, chosen) {
                if (number <= 0) { throw ErrorCode.NUM; }
                if (chosen <= 0) { throw ErrorCode.NUM; }
                if (number < chosen) { throw ErrorCode.NUM; }

                return FormulaUtils.factorial(number) / FormulaUtils.factorial(number - chosen);
            }
        },

        PERMUTATIONA: {
            category: 'statistical',
            name: { ooxml: '_xlfn.PERMUTATIONA' },
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: 'val:int val:int',
            resolve: function (base, exp) {
                if (base < 0) { throw ErrorCode.NUM; }
                if (exp < 0) { throw ErrorCode.NUM; }

                return Math.pow(base, exp);
            }
        },

        PHI: {
            category: 'statistical',
            name: { ooxml: '_xlfn.PHI' },
            minParams: 1,
            maxParams: 1,
            type: 'val'
        },

        POISSON: {
            category: 'statistical',
            minParams: 3,
            maxParams: 3,
            type: 'val'
        },

        'POISSON.DIST': {
            category: 'statistical',
            name: { ooxml: '_xlfn.POISSON.DIST', odf: 'COM.MICROSOFT.POISSON.DIST' },
            ceName: null,
            minParams: 3,
            maxParams: 3,
            type: 'val'
        },

        PROB: {
            category: 'statistical',
            minParams: 3,
            maxParams: 4,
            type: 'val'
        },

        QUARTILE: {
            category: 'statistical',
            minParams: 2,
            maxParams: 2,
            type: 'val'
        },

        'QUARTILE.EXC': {
            category: 'statistical',
            name: { ooxml: '_xlfn.QUARTILE.EXC', odf: 'COM.MICROSOFT.QUARTILE.EXC' },
            ceName: null,
            minParams: 2,
            maxParams: 2,
            type: 'val'
        },

        'QUARTILE.INC': {
            category: 'statistical',
            name: { ooxml: '_xlfn.QUARTILE.INC', odf: 'COM.MICROSOFT.QUARTILE.INC' },
            ceName: null,
            minParams: 2,
            maxParams: 2,
            type: 'val'
        },

        RANK: {
            category: 'statistical',
            minParams: 2,
            maxParams: 3,
            type: 'val'
        },

        'RANK.AVG': {
            category: 'statistical',
            name: { ooxml: '_xlfn.RANK.AVG', odf: 'COM.MICROSOFT.RANK.AVG' },
            ceName: null,
            minParams: 2,
            maxParams: 3,
            type: 'val'
        },

        'RANK.EQ': {
            category: 'statistical',
            name: { ooxml: '_xlfn.RANK.EQ', odf: 'COM.MICROSOFT.RANK.EQ' },
            ceName: null,
            minParams: 2,
            maxParams: 3,
            type: 'val'
        },

        RSQ: {
            category: 'statistical',
            minParams: 2,
            maxParams: 2,
            type: 'val'
        },

        SKEW: {
            category: 'statistical',
            minParams: 1,
            type: 'val'
        },

        'SKEW.P': {
            category: 'statistical',
            name: { ooxml: '_xlfn.SKEW.P', odf: 'SKEWP' },
            ceName: null,
            minParams: 1,
            type: 'val'
        },

        SLOPE: {
            category: 'statistical',
            minParams: 2,
            maxParams: 2,
            type: 'val'
        },

        SMALL: {
            category: 'statistical',
            minParams: 2,
            maxParams: 2,
            type: 'val'
        },

        STANDARDIZE: {
            category: 'statistical',
            minParams: 3,
            maxParams: 3,
            type: 'val'
        },

        STDEV: {
            category: 'statistical',
            minParams: 1,
            type: 'val',
            signature: 'any',
            // empty parameters count as 0: STDEV(1,) = 0.707
            resolve: implementStdDevAggregation('dev', false, SKIP_MAT_SKIP_REF)
        },

        'STDEV.P': {
            category: 'statistical',
            name: { ooxml: '_xlfn.STDEV.P', odf: 'COM.MICROSOFT.STDEV.P' },
            ceName: { odf: null }, // TODO: CalcEngine should allow to use this function in ODF
            minParams: 1,
            type: 'val',
            signature: 'any',
            resolve: 'STDEVP'
        },

        'STDEV.S': {
            category: 'statistical',
            name: { ooxml: '_xlfn.STDEV.S', odf: 'COM.MICROSOFT.STDEV.S' },
            ceName: { odf: null }, // TODO: CalcEngine should allow to use this function in ODF
            minParams: 1,
            type: 'val',
            signature: 'any',
            resolve: 'STDEV'
        },

        STDEVA: {
            category: 'statistical',
            minParams: 1,
            type: 'val',
            signature: 'any',
            // empty parameters count as 0: STDEVA(1,) = 0.707
            resolve: implementStdDevAggregation('dev', false, SKIP_MAT_ZERO_REF)
        },

        STDEVP: {
            category: 'statistical',
            minParams: 1,
            type: 'val',
            signature: 'any',
            // empty parameters count as 0: STDEVP(1,) = 0.5
            resolve: implementStdDevAggregation('dev', true, SKIP_MAT_SKIP_REF)
        },

        STDEVPA: {
            category: 'statistical',
            minParams: 1,
            type: 'val',
            signature: 'any',
            // empty parameters count as 0: STDEVPA(1,) = 0.5
            resolve: implementStdDevAggregation('dev', true, SKIP_MAT_ZERO_REF)
        },

        STEYX: {
            category: 'statistical',
            minParams: 2,
            maxParams: 2,
            type: 'val'
        },

        'T.DIST': {
            category: 'statistical',
            name: { ooxml: '_xlfn.T.DIST', odf: 'COM.MICROSOFT.T.DIST' },
            ceName: null,
            minParams: 3,
            maxParams: 3,
            type: 'val'
        },

        'T.DIST.2T': {
            category: 'statistical',
            name: { ooxml: '_xlfn.T.DIST.2T', odf: 'COM.MICROSOFT.T.DIST.2T' },
            ceName: null,
            minParams: 2,
            maxParams: 2,
            type: 'val'
        },

        'T.DIST.RT': {
            category: 'statistical',
            name: { ooxml: '_xlfn.T.DIST.RT', odf: 'COM.MICROSOFT.T.DIST.RT' },
            ceName: null,
            minParams: 2,
            maxParams: 2,
            type: 'val'
        },

        'T.INV': {
            category: 'statistical',
            name: { ooxml: '_xlfn.T.INV', odf: 'COM.MICROSOFT.T.INV' },
            ceName: null,
            minParams: 2,
            maxParams: 2,
            type: 'val'
        },

        'T.INV.2T': {
            category: 'statistical',
            name: { ooxml: '_xlfn.T.INV.2T', odf: 'COM.MICROSOFT.T.INV.2T' },
            ceName: null,
            minParams: 2,
            maxParams: 2,
            type: 'val'
        },

        'T.TEST': {
            category: 'statistical',
            name: { ooxml: '_xlfn.T.TEST', odf: 'COM.MICROSOFT.T.TEST' },
            ceName: null,
            minParams: 4,
            maxParams: 4,
            type: 'val'
        },

        TDIST: {
            category: 'statistical',
            // AOO 4.4 writes 'TDIST' (https://bz.apache.org/ooo/show_bug.cgi?id=126519)
            name: { odf: ['LEGACY.TDIST', 'TDIST'] },
            minParams: 3,
            maxParams: 3,
            type: 'val'
        },

        TINV: {
            category: 'statistical',
            minParams: 2,
            maxParams: 2,
            type: 'val'
        },

        TREND: {
            category: 'statistical',
            minParams: 1,
            maxParams: 4,
            type: 'mat'
        },

        TRIMMEAN: {
            category: 'statistical',
            minParams: 2,
            maxParams: 2,
            type: 'val'
        },

        TTEST: {
            category: 'statistical',
            minParams: 4,
            maxParams: 4,
            type: 'val'
        },

        VAR: {
            category: 'statistical',
            minParams: 1,
            type: 'val',
            signature: 'any',
            // empty parameters count as 0: VAR(1,) = 0.5
            resolve: implementStdDevAggregation('var', false, SKIP_MAT_SKIP_REF)
        },

        'VAR.P': {
            category: 'statistical',
            name: { ooxml: '_xlfn.VAR.P', odf: 'COM.MICROSOFT.VAR.P' },
            ceName: { odf: null }, // TODO: CalcEngine should allow to use this function in ODF
            minParams: 1,
            type: 'val',
            signature: 'any',
            resolve: 'VARP'
        },

        'VAR.S': {
            category: 'statistical',
            name: { ooxml: '_xlfn.VAR.S', odf: 'COM.MICROSOFT.VAR.S' },
            ceName: { odf: null }, // TODO: CalcEngine should allow to use this function in ODF
            minParams: 1,
            type: 'val',
            signature: 'any',
            resolve: 'VAR'
        },

        VARA: {
            category: 'statistical',
            minParams: 1,
            type: 'val',
            signature: 'any',
            // empty parameters count as 0: VARA(1,) = 0.5
            resolve: implementStdDevAggregation('var', false, SKIP_MAT_ZERO_REF)
        },

        VARP: {
            category: 'statistical',
            minParams: 1,
            type: 'val',
            signature: 'any',
            // empty parameters count as 0: VARP(1,) = 0.25
            resolve: implementStdDevAggregation('var', true, SKIP_MAT_SKIP_REF)
        },

        VARPA: {
            category: 'statistical',
            minParams: 1,
            type: 'val',
            signature: 'any',
            // empty parameters count as 0: VARPA(1,) = 0.25
            resolve: implementStdDevAggregation('var', true, SKIP_MAT_ZERO_REF)
        },

        WEIBULL: {
            category: 'statistical',
            minParams: 4,
            maxParams: 4,
            type: 'val'
        },

        'WEIBULL.DIST': {
            category: 'statistical',
            name: { ooxml: '_xlfn.WEIBULL.DIST', odf: 'COM.MICROSOFT.WEIBULL.DIST' },
            ceName: null,
            minParams: 4,
            maxParams: 4,
            type: 'val'
        },

        'Z.TEST': {
            category: 'statistical',
            name: { ooxml: '_xlfn.Z.TEST', odf: 'COM.MICROSOFT.Z.TEST' },
            ceName: null,
            minParams: 3,
            maxParams: 3,
            type: 'val'
        },

        ZTEST: {
            category: 'statistical',
            minParams: 2,
            maxParams: 3,
            type: 'val'
        }
    };

});
