/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/formula/impl/operators', [
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/model/formula/formulautils'
], function (SheetUtils, FormulaUtils) {

    'use strict';

    /**************************************************************************
     *
     * This module implements all unary and binary operators used in
     * spreadsheet formulas.
     *
     * See the README file in this directory for a detailed documentation about
     * the format of operator descriptor objects.
     *
     *************************************************************************/

    // convenience shortcuts
    var ErrorCode = SheetUtils.ErrorCode,
        Range3DArray = SheetUtils.Range3DArray;

    // exports ================================================================

    return {

        // arithmetic operators -----------------------------------------------

        // binary addition operator, or unary plus operator
        add: {
            name: '+',
            ceName: '+',
            minParams: 1,
            maxParams: 2,
            type: 'val',
            signature: 'val val:num', // unary plus operator passes all values (no conversion to number)
            resolve: function (value1, value2) {
                return _.isUndefined(value2) ? value1 : (this.convertToNumber(value1) + value2);
            }
        },

        // binary subtraction operator, or unary minus operator
        sub: {
            name: '-',
            ceName: '-',
            minParams: 1,
            maxParams: 2,
            type: 'val',
            signature: 'val:num val:num', // unary minus operator always converts to numbers
            resolve: function (value1, value2) {
                return _.isUndefined(value2) ? (-value1) : (value1 - value2);
            }
        },

        // binary multiplication operator
        mul: {
            name: '*',
            ceName: '*',
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: 'val:num val:num',
            resolve: FormulaUtils.multiply
        },

        // binary division operator
        div: {
            name: '/',
            ceName: '/',
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: 'val:num val:num',
            resolve: FormulaUtils.divide
        },

        // binary power operator
        pow: {
            name: '^',
            ceName: '^',
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: 'val:num val:num',
            resolve: FormulaUtils.POWER
        },

        // unary percent operator
        pct: {
            name: '%',
            ceName: '%',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:num',
            resolve: function (number) { return number / 100; }
        },

        // string operators ---------------------------------------------------

        // binary string concatenation operator
        con: {
            name: '&',
            ceName: '&',
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: 'val:str val:str',
            resolve: FormulaUtils.add
        },

        // comparison operators -----------------------------------------------

        // binary less-than operator
        lt: {
            name: '<',
            ceName: '<',
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: 'val val',
            resolve: function (value1, value2) {
                return this.compareValues(value1, value2) < 0;
            }
        },

        // binary less-than-or-equal operator
        le: {
            name: '<=',
            ceName: '<=',
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: 'val val',
            resolve: function (value1, value2) {
                return this.compareValues(value1, value2) <= 0;
            }
        },

        // binary greater-than operator
        gt: {
            name: '>',
            ceName: '>',
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: 'val val',
            resolve: function (value1, value2) {
                return this.compareValues(value1, value2) > 0;
            }
        },

        // binary greater-than-or-equal operator
        ge: {
            name: '>=',
            ceName: '>=',
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: 'val val',
            resolve: function (value1, value2) {
                return this.compareValues(value1, value2) >= 0;
            }
        },

        // binary equals operator
        eq: {
            name: '=',
            ceName: '=',
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: 'val val',
            resolve: function (value1, value2) {
                return this.compareValues(value1, value2) === 0;
            }
        },

        // binary equals-not operator
        ne: {
            name: '<>',
            ceName: '<>',
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: 'val val',
            resolve: function (value1, value2) {
                return this.compareValues(value1, value2) !== 0;
            }
        },

        // reference operators ------------------------------------------------

        // binary list reference operator
        list: {
            name: { ooxml: ',', odf: '~' },
            ceName: ',',
            minParams: 2,
            maxParams: 2,
            type: 'ref',
            signature: 'ref ref',
            resolve: function (ranges1, ranges2) {
                return ranges1.concat(ranges2);
            }
        },

        // binary intersection reference operator
        isect: {
            name: { ooxml: ' ', odf: '!' },
            ceName: ' ',
            minParams: 2,
            maxParams: 2,
            type: 'ref',
            signature: 'ref:sheet ref:sheet',
            resolve: function (ranges1, ranges2) {

                // ranges in both operands must refer to the same sheet
                if (ranges1.first().sheet1 !== ranges2.first().sheet1) {
                    throw ErrorCode.VALUE;
                }

                // calculate the intersection of each range pair from ranges1 and ranges2,
                // process all outer ranges separately to be able to check resulting list
                // size inside the loop, otherwise this loop may create a MAX^2 list
                var resultRanges = new Range3DArray();
                ranges1.forEach(function (range1) {
                    var ranges = ranges2.intersect(range1);
                    if (resultRanges.length + ranges.length > FormulaUtils.MAX_REF_LIST_SIZE) {
                        throw FormulaUtils.UNSUPPORTED_ERROR;
                    }
                    resultRanges.append(ranges);
                });
                return resultRanges;
            }
        },

        // binary range reference operator
        range: {
            name: ':',
            ceName: ':',
            minParams: 2,
            maxParams: 2,
            type: 'ref',
            signature: 'ref:sheet ref:sheet',
            resolve: function (ranges1, ranges2) {

                // ranges in both operands must refer to the same sheet
                if (ranges1.first().sheet1 !== ranges2.first().sheet1) {
                    throw ErrorCode.VALUE;
                }

                // build the bounding ranges from all ranges
                return ranges1.concat(ranges2).boundary();
            }
        }
    };

});
